<?php
use NewSite\Auth\SiteAuth;
use NewSite\Settings\SettingsService;
use NewSite\Shop\CurrencyService;

$sectionTitle = trim($sectionSettings['section_title'] ?? '');
$titleAlign = $sectionSettings['title_align'] ?? 'center';
$titleAlign = in_array($titleAlign, ['left', 'center', 'right'], true) ? $titleAlign : 'center';
$contentAlign = $sectionSettings['content_align'] ?? 'left';
$contentAlign = in_array($contentAlign, ['left', 'center', 'right'], true) ? $contentAlign : 'left';
$productId = (int)($sectionSettings['product_id'] ?? 0);
$requestedProductId = isset($_GET['product']) ? (int)$_GET['product'] : 0;
$showPrice = isset($sectionSettings['show_price']) ? (bool)$sectionSettings['show_price'] : true;
$showDescription = isset($sectionSettings['show_description']) ? (bool)$sectionSettings['show_description'] : true;
$showMedia = isset($sectionSettings['show_media']) ? (bool)$sectionSettings['show_media'] : true;
$accordionStyle = $sectionSettings['accordion_style'] ?? 'separated';
$accordionStyle = $accordionStyle === 'bar' ? 'bar' : 'separated';
$variantStyle = $sectionSettings['variant_style'] ?? 'select';
$variantStyle = $variantStyle === 'buttons' ? 'buttons' : 'select';
$currencySymbol = CurrencyService::getSymbol();

if (!function_exists('extractYouTubeIdFromUrl')) {
    function extractYouTubeIdFromUrl(string $url): string
    {
        $url = trim($url);
        if ($url === '') {
            return '';
        }

        $patterns = [
            '~youtu\.be/([A-Za-z0-9_-]{11})~',
            '~youtube\.com/(?:watch\?v=|embed/|shorts/)([A-Za-z0-9_-]{11})~',
            '~[?&]v=([A-Za-z0-9_-]{11})~',
            '~^([A-Za-z0-9_-]{11})$~'
        ];

        $videoId = '';
        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $url, $match)) {
                $videoId = $match[1] ?? '';
                break;
            }
        }

        return $videoId;
    }
}

function formatProductPrice(string $price): string
{
    return CurrencyService::formatValue($price, CurrencyService::getCurrent());
}

if (!function_exists('productDownloadUrlToPublic')) {
    function productDownloadUrlToPublic(string $url): string
    {
        $url = trim($url);
        if ($url === '') {
            return '';
        }

        if (strpos($url, '/admin-file/') !== false && function_exists('toPublicFilePath')) {
            return (string)toPublicFilePath($url);
        }

        return $url;
    }
}

if (!function_exists('productFreeDownloadTrackingUrl')) {
    function productFreeDownloadTrackingUrl(int $productId, int $variantId = 0): string
    {
        $params = ['product' => max(0, $productId)];
        if ($variantId > 0) {
            $params['variant'] = $variantId;
        }

        return '/free-download?' . http_build_query($params);
    }
}

$pageId = $page['id'] ?? null;
$product = null;
$isPreview = isset($_GET['preview']) && $_GET['preview'] === '1';
if ($pageId) {
    $targetProductId = $productId ?: $requestedProductId;
    if ($targetProductId) {
        $stmt = $db->prepare("SELECT * FROM products WHERE is_active = 1 AND page_id = ? AND id = ?");
        $stmt->execute([$pageId, $targetProductId]);
        $product = $stmt->fetch();
    } elseif ($isPreview) {
        $stmt = $db->prepare("SELECT * FROM products WHERE is_active = 1 AND page_id = ? ORDER BY RANDOM() LIMIT 1");
        $stmt->execute([$pageId]);
        $product = $stmt->fetch();
    }
}

if (!$product) {
    return;
}

$mediaItems = [];
$stmt = $db->prepare("SELECT media_url FROM product_media WHERE product_id = ? ORDER BY sort_order ASC, id ASC");
$stmt->execute([$product['id']]);
$mediaItems = $stmt->fetchAll();
if (empty($mediaItems) && !empty($product['media_url'])) {
    $mediaItems = [['media_url' => $product['media_url']]];
}
$mediaItems = array_values(array_filter($mediaItems, function ($item) {
    return !empty(trim($item['media_url'] ?? ''));
}));

$stmt = $db->prepare("SELECT title, content FROM product_blocks WHERE product_id = ? ORDER BY sort_order ASC, id ASC");
$stmt->execute([$product['id']]);
$productBlocks = $stmt->fetchAll();

$stmt = $db->prepare("SELECT id, label, price, is_default, stock, action_type, download_url, external_link_label FROM product_variants WHERE product_id = ? ORDER BY sort_order ASC, id ASC");
$stmt->execute([$product['id']]);
$productVariants = $stmt->fetchAll();

$allowComments = (int)($product['allow_comments'] ?? 1) === 1;
$productComments = [];
if ($allowComments) {
    $stmt = $db->prepare("SELECT pc.id, pc.body, pc.created_at, su.display_name, su.nickname, su.profile_photo
                          FROM product_comments pc
                          LEFT JOIN site_users su ON su.id = pc.user_id
                          WHERE pc.product_id = ?
                          ORDER BY pc.created_at DESC
                          LIMIT 100");
    $stmt->execute([(int)$product['id']]);
    $productComments = $stmt->fetchAll();
}

$productCommentFlashType = '';
$productCommentFlashMessage = '';
if (isset($productCommentFlash) && is_array($productCommentFlash) && (int)($productCommentFlash['product_id'] ?? 0) === (int)$product['id']) {
    $flashType = trim((string)($productCommentFlash['type'] ?? ''));
    $productCommentFlashType = $flashType === 'success' ? 'success' : 'error';
    $productCommentFlashMessage = trim((string)($productCommentFlash['message'] ?? ''));
}

$canPostProductComment = SiteAuth::isLoggedIn();
?>
<section class="section-products" itemscope itemtype="https://schema.org/Product">
    <div class="container">
        <?php
            $displayTitle = $sectionTitle !== '' ? $sectionTitle : trim($product['name'] ?? '');
?>
        <?php if ($displayTitle !== ''): ?>
            <div class="products-header text-align-<?php echo e($titleAlign); ?>">
                <h2><?php echo e($displayTitle); ?></h2>
            </div>
        <?php endif; ?>
        <?php
    $name = trim($product['name'] ?? '');
$description = trim($product['description'] ?? '');
$price = trim($product['price'] ?? '');
$displayPrice = formatProductPrice($price);
$defaultVariant = null;
if (!empty($productVariants)) {
    foreach ($productVariants as $variant) {
        if (!empty($variant['is_default'])) {
            $defaultVariant = $variant;
            break;
        }
    }
    if (!$defaultVariant) {
        $defaultVariant = $productVariants[0];
    }
}
$defaultVariantPrice = $defaultVariant ? formatProductPrice((string)($defaultVariant['price'] ?? '')) : '';
$effectivePrice = $defaultVariantPrice !== '' ? $defaultVariantPrice : $displayPrice;
$effectivePriceRaw = $defaultVariant ? (string)($defaultVariant['price'] ?? '') : $price;
if ($effectivePriceRaw === '') {
    $effectivePriceRaw = $price;
}
$productStock = $product['stock'] ?? null;
$productUnlimited = $productStock === null;
$productStockValue = $productUnlimited ? '' : (int)$productStock;
$defaultVariantStock = $defaultVariant['stock'] ?? null;
$defaultVariantUnlimited = $defaultVariantStock === null;
$defaultVariantStockValue = $defaultVariantUnlimited ? '' : (int)$defaultVariantStock;
$actionType = $product['action_type'] ?? 'cart';
$actionType = in_array($actionType, ['cart', 'download', 'external_link'], true) ? $actionType : 'cart';
$downloadUrl = productDownloadUrlToPublic((string)($product['download_url'] ?? ''));
$externalLinkLabel = trim($product['external_link_label'] ?? '');
$baseDigital = $actionType === 'download';
$baseExternal = $actionType === 'external_link';
$effectiveAction = $actionType;
$effectiveDownload = $downloadUrl;
if ($defaultVariant) {
    $variantAction = $defaultVariant['action_type'] ?? $actionType;
    $variantAction = in_array($variantAction, ['cart', 'download', 'external_link'], true) ? $variantAction : $actionType;
    $effectiveAction = $variantAction;
    $effectiveDownload = productDownloadUrlToPublic((string)($defaultVariant['download_url'] ?? '')) ?: $downloadUrl;
    $baseDigital = $variantAction === 'download';
    $baseExternal = $variantAction === 'external_link';
}
$effectivePriceValue = CurrencyService::parsePrice($effectivePriceRaw);
if ($effectiveAction === 'download' && $effectivePriceValue > 0) {
    $effectiveAction = 'cart';
}
if ($effectiveAction === 'download' && $effectiveDownload !== '') {
    $effectiveVariantId = $defaultVariant ? (int)($defaultVariant['id'] ?? 0) : 0;
    $effectiveDownload = productFreeDownloadTrackingUrl((int)$product['id'], $effectiveVariantId);
}
$displayedPrice = $effectiveAction === 'download' ? 'Free' : $effectivePrice;
$isExternalLink = $effectiveAction === 'external_link';
$likeCount = (int)($product['likes_count'] ?? 0);
$dislikeCount = (int)($product['dislikes_count'] ?? 0);
$netLikeCount = $likeCount - $dislikeCount;
$userLikeState = 'none';
$classActive = ' active';
$classIsHidden = ' is-hidden';
$likeWrapStateClass = '';
$likeUserId = !empty($_SESSION['site_user_id']) ? (int)$_SESSION['site_user_id'] : 0;
$likeSessionId = session_id();
$hasDigitalPurchase = false;
$purchasedVariantIds = [];
if ($likeUserId > 0) {
    $stmt = $db->prepare("SELECT oi.variant_id FROM order_items oi JOIN orders o ON o.id = oi.order_id WHERE o.user_id = ? AND oi.product_id = ? AND oi.is_digital = 1 AND oi.unit_price > 0 AND o.status = 'paid'");
    $stmt->execute([$likeUserId, (int)$product['id']]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    if (!empty($rows)) {
        foreach ($rows as $row) {
            if (!empty($row['variant_id'])) {
                $purchasedVariantIds[] = (int)$row['variant_id'];
            } else {
                $hasDigitalPurchase = true;
            }
        }
        $purchasedVariantIds = array_values(array_unique($purchasedVariantIds));
    }
}
if ($likeUserId > 0) {
    $stmt = $db->prepare("SELECT is_liked FROM product_likes WHERE product_id = ? AND user_id = ? LIMIT 1");
    $stmt->execute([(int)$product['id'], $likeUserId]);
    $row = $stmt->fetch();
    if ($row) {
        $userLikeState = $row['is_liked'] ? 'like' : 'dislike';
    }
} elseif ($likeSessionId !== '') {
    $stmt = $db->prepare("SELECT is_liked FROM product_likes WHERE product_id = ? AND session_id = ? LIMIT 1");
    $stmt->execute([(int)$product['id'], $likeSessionId]);
    $row = $stmt->fetch();
    if ($row) {
        $userLikeState = $row['is_liked'] ? 'like' : 'dislike';
    }
}
if ($userLikeState === 'like') {
    $likeWrapStateClass = ' is-liked';
} elseif ($userLikeState === 'dislike') {
    $likeWrapStateClass = ' is-disliked';
}
$likeButtonStateClass = $userLikeState !== 'none' ? ' is-active' : '';
$quantityInputId = 'product-qty-input-' . (int)$product['id'];
$externalLinkButtonText = 'Link unavailable';
if ($downloadUrl !== '') {
    $externalLinkButtonText = $externalLinkLabel !== '' ? $externalLinkLabel : 'Open link';
}
?>

        <?php if ($showMedia && !empty($mediaItems)): ?>
            <div class="product-media-wrapper">
                <div class="product-media-slider" data-auto="1" data-interval="4" data-hover-pause="0">
                    <div class="product-media-track">
                        <?php foreach ($mediaItems as $index => $mediaItem):
                            $rawMedia = trim($mediaItem['media_url'] ?? '');
                            $youtubeId = extractYouTubeIdFromUrl($rawMedia);
                            $isYouTube = $youtubeId !== '';
                            if ($isYouTube) {
                                $mediaUrl = 'https://www.youtube-nocookie.com/embed/' . $youtubeId
                                    . '?autoplay=1&mute=1&loop=1&playlist=' . $youtubeId
                                    . '&controls=0&disablekb=1&modestbranding=1&rel=0&playsinline=1'
                                    . '&showinfo=0&iv_load_policy=3&fs=0&cc_load_policy=0';
                            } else {
                                $mediaUrl = toPublicFilePath($rawMedia);
                            }
                            $extension = strtolower(pathinfo(parse_url($mediaUrl, PHP_URL_PATH) ?? '', PATHINFO_EXTENSION));
                            $isVideo = !$isYouTube && in_array($extension, ['mp4', 'webm', 'ogg'], true);
                            ?>
                            <div class="product-media-slide" data-slide-index="<?php echo $index; ?>">
                                <?php if ($isYouTube): ?>
                                    <iframe src="<?php echo e($mediaUrl); ?>" allow="autoplay; encrypted-media; picture-in-picture" allowfullscreen loading="lazy" title="<?php echo e($name !== '' ? ($name . ' video') : 'Product video'); ?>" data-youtube-url="<?php echo e('https://www.youtube.com/watch?v=' . $youtubeId); ?>"></iframe>
                                <?php elseif ($isVideo): ?>
                                    <video autoplay muted loop playsinline preload="metadata">
                                        <source src="<?php echo e($mediaUrl); ?>" type="video/<?php echo e($extension ?: 'mp4'); ?>">
                                        <track kind="captions" srclang="en" label="English captions" src="data:text/vtt,WEBVTT%0A%0A">
                                        <track kind="descriptions" srclang="en" label="English descriptions" src="data:text/vtt,WEBVTT%0A%0A">
                                    </video>
                                <?php else: ?>
                                    <img src="<?php echo e($mediaUrl); ?>" alt="<?php echo e($name); ?>" itemprop="image"<?php echo $index > 0 ? ' loading="lazy" decoding="async"' : ' fetchpriority="high"'; ?>>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php if (count($mediaItems) > 1): ?>
                    <div class="product-media-dots">
                        <?php foreach ($mediaItems as $index => $mediaItem): ?>
                            <button type="button" class="product-media-dot" data-slide="<?php echo $index; ?>"></button>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        <?php endif; ?>

        <div class="product-detail-layout">
            <div class="product-detail-left product-accordion-style-<?php echo e($accordionStyle); ?>">
                <?php if ($accordionStyle === 'bar'): ?>
                    <?php
                    $tabSections = [];
                    if ($showDescription && $description !== '') {
                        $tabSections[] = [
                            'title' => 'Description',
                            'content' => nl2br(e($description)),
                            'is_description' => true
                        ];
                    }
                    foreach ($productBlocks as $block) {
                        $blockTitle = trim($block['title'] ?? '');
                        $blockContent = trim($block['content'] ?? '');
                        if ($blockTitle === '' && $blockContent === '') {
                            continue;
                        }
                        $tabSections[] = [
                            'title' => $blockTitle !== '' ? $blockTitle : 'Details',
                            'content' => $blockContent !== '' ? nl2br(e($blockContent)) : ''
                        ];
                    }
                    ?>
                    <?php if (!empty($tabSections)): ?>
                        <div class="product-tabs" data-product-tabs>
                            <div class="product-tab-list" role="tablist">
                                <?php foreach ($tabSections as $index => $tab): ?>
                                    <button
                                        type="button"
                                        class="product-tab-btn<?php echo $index === 0 ? $classActive : ''; ?>"
                                        role="tab"
                                        data-tab-target="tab-<?php echo $index; ?>">
                                        <?php echo e($tab['title']); ?>
                                    </button>
                                <?php endforeach; ?>
                            </div>
                            <div class="product-tab-panels">
                                <?php foreach ($tabSections as $index => $tab): ?>
                                    <div class="product-tab-panel<?php echo $index === 0 ? $classActive : ''; ?>" data-tab-panel="tab-<?php echo $index; ?>">
                                        <?php if ($tab['content'] !== ''): ?>
                                            <div class="product-detail-description text-align-<?php echo e($contentAlign); ?>"<?php echo !empty($tab['is_description']) ? ' itemprop="description"' : ''; ?>><?php echo $tab['content']; ?></div>
                                        <?php endif; ?>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    <?php endif; ?>
                <?php else: ?>
                    <?php if ($showDescription && $description !== ''): ?>
                        <details class="product-detail-section product-accordion" open>
                            <summary class="product-accordion-summary">Description</summary>
                            <div class="product-detail-description text-align-<?php echo e($contentAlign); ?>" itemprop="description"><?php echo nl2br(e($description)); ?></div>
                        </details>
                    <?php endif; ?>

                    <?php foreach ($productBlocks as $block):
                        $blockTitle = trim($block['title'] ?? '');
                        $blockContent = trim($block['content'] ?? '');
                        if ($blockTitle === '' && $blockContent === '') {
                            continue;
                        }
                        ?>
                        <details class="product-detail-section product-accordion">
                            <?php if ($blockTitle !== ''): ?>
                                <summary class="product-accordion-summary"><?php echo e($blockTitle); ?></summary>
                            <?php else: ?>
                                <summary class="product-accordion-summary">Details</summary>
                            <?php endif; ?>
                            <?php if ($blockContent !== ''): ?>
                                <div class="product-detail-description text-align-<?php echo e($contentAlign); ?>"><?php echo nl2br(e($blockContent)); ?></div>
                            <?php endif; ?>
                        </details>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>

            <div class="product-detail-right">
                <?php
                        $offerCurrency = CurrencyService::getCurrent();
$offerPriceValue = CurrencyService::convert($effectivePriceValue, SettingsService::get('store_currency', 'USD'), $offerCurrency);
if ($effectiveAction === 'download') {
    $offerPriceValue = 0.0;
}
$offerPriceValue = number_format($offerPriceValue, 2, '.', '');
$hasStock = $productUnlimited || $productStockValue > 0;
if (!$hasStock && !empty($productVariants)) {
    foreach ($productVariants as $variant) {
        $variantStock = $variant['stock'] ?? null;
        if ($variantStock === null || (int)$variantStock > 0) {
            $hasStock = true;
            break;
        }
    }
}
$availabilityUrl = $hasStock ? 'https://schema.org/InStock' : 'https://schema.org/OutOfStock';
?>
                <div class="product-meta-card" data-base-action="<?php echo e($effectiveAction); ?>" data-base-download="<?php echo e($effectiveDownload); ?>" data-base-link="<?php echo e($downloadUrl); ?>" data-base-link-label="<?php echo e($externalLinkLabel); ?>" data-base-digital="<?php echo $baseDigital ? '1' : '0'; ?>" data-product-owned="<?php echo $hasDigitalPurchase ? '1' : '0'; ?>" data-owned-message="You already have this." itemprop="offers" itemscope itemtype="https://schema.org/Offer">
                    <meta itemprop="priceCurrency" content="<?php echo e($offerCurrency); ?>">
                    <meta itemprop="price" content="<?php echo e($offerPriceValue); ?>">
                    <link itemprop="availability" href="<?php echo e($availabilityUrl); ?>">
                    <?php if ($name !== ''): ?>
                        <h3 class="product-detail-name" itemprop="name"><?php echo e($name); ?></h3>
                    <?php endif; ?>
                    <div class="product-share-row">
                        <button type="button" class="btn btn-secondary product-share-btn" data-share-title="<?php echo e($name); ?>">Share</button>
                        <div class="product-like-wrap<?php echo $likeWrapStateClass; ?>" data-product-like data-product-id="<?php echo (int)$product['id']; ?>" data-like-state="<?php echo e($userLikeState); ?>" data-like-count="<?php echo $likeCount; ?>" data-dislike-count="<?php echo $dislikeCount; ?>">
                            <button type="button" class="btn product-like-btn<?php echo $likeButtonStateClass; ?>" aria-pressed="<?php echo $userLikeState !== 'none' ? 'true' : 'false'; ?>" title="Like (right-click or long-press to dislike)">
                                <svg viewBox="0 0 24 24" aria-hidden="true" focusable="false">
                                    <path d="M12 20.8s-6.7-4.4-9.2-8.1C1.1 9 2.4 5.7 5.6 5.1c2-.4 3.8.5 4.8 1.9 1-1.4 2.8-2.3 4.8-1.9 3.2.6 4.5 3.9 2.8 7.6-2.5 3.7-9.2 8.1-9.2 8.1z" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path>
                                </svg>
                                <span class="product-like-count"><?php echo $netLikeCount; ?></span>
                            </button>
                        </div>
                    </div>
                    <?php if ($showPrice && !$isExternalLink && ($effectivePrice !== '' || $effectiveAction === 'download')): ?>
                        <div class="product-detail-price" data-base-price="<?php echo e($displayPrice); ?>">
                            <span class="product-price-value"><?php echo e($displayedPrice); ?></span>
                        </div>
                    <?php endif; ?>
                    <?php
    // Show digital product badge for paid downloadable products or external link badge
    $isPaidDigital = $baseDigital && $effectivePriceValue > 0;
$showBadge = $isPaidDigital || $isExternalLink;
$purchaseLocked = $hasDigitalPurchase && $isPaidDigital;
$badgeLabel = $isExternalLink ? 'External Link' : 'Digital Download';
?>
                    <div class="product-digital-badge<?php echo $showBadge ? '' : $classIsHidden; ?>" data-digital-badge>
                        <svg class="badge-icon-download<?php echo $isExternalLink ? $classIsHidden : ''; ?>" xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/><polyline points="7 10 12 15 17 10"/><line x1="12" y1="15" x2="12" y2="3"/></svg>
                        <svg class="badge-icon-external<?php echo $isExternalLink ? '' : $classIsHidden; ?>" xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M18 13v6a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h6"/><polyline points="15 3 21 3 21 9"/><line x1="10" y1="14" x2="21" y2="3"/></svg>
                        <span data-badge-label><?php echo e($badgeLabel); ?></span>
                    </div>
                    <?php if ($product['quantity_max'] === null && !$isExternalLink): ?>
                        <div class="product-qty<?php echo $baseDigital ? $classIsHidden : ''; ?>" data-product-qty>
                            <label for="<?php echo e($quantityInputId); ?>">Quantity</label>
                            <div class="product-qty-controls">
                                <button type="button" class="product-qty-btn" data-qty="-1">-</button>
                                <input id="<?php echo e($quantityInputId); ?>" type="number" class="product-qty-input" min="1" value="1">
                                <button type="button" class="product-qty-btn" data-qty="1">+</button>
                            </div>
                        </div>
                    <?php endif; ?>
                    <div class="product-action-row">
                        <button type="button" class="btn btn-primary product-cart-btn<?php echo $effectiveAction === 'cart' ? '' : $classIsHidden; ?>" data-product-id="<?php echo $product['id']; ?>" data-product-name="<?php echo e($name); ?>" data-stock="<?php echo e($productStockValue); ?>" data-unlimited="<?php echo $productUnlimited ? '1' : '0'; ?>" data-quantity-max="<?php echo $product['quantity_max'] === null ? '' : (int)$product['quantity_max']; ?>"<?php echo $purchaseLocked ? ' disabled aria-disabled="true" title="You already have this."' : ''; ?>>
                            <?php echo $purchaseLocked ? 'Already Owned' : 'Add to Cart'; ?>
                        </button>
                        <a class="btn btn-primary product-download-btn<?php echo $effectiveAction === 'download' ? '' : $classIsHidden; ?><?php echo $effectiveDownload === '' ? ' is-disabled' : ''; ?>" data-download-btn href="<?php echo $effectiveDownload !== '' ? e($effectiveDownload) : '#'; ?>"<?php echo $effectiveDownload !== '' ? ' download' : ' aria-disabled="true"'; ?>>
                            <?php echo $effectiveDownload !== '' ? 'Download' : 'Download Unavailable'; ?>
                        </a>
                        <a class="btn btn-primary product-link-btn<?php echo $isExternalLink ? '' : $classIsHidden; ?><?php echo $downloadUrl === '' ? ' is-disabled' : ''; ?>" href="<?php echo $downloadUrl !== '' ? e($downloadUrl) : '#'; ?>"<?php echo $downloadUrl !== '' ? ' target="_blank" rel="noopener"' : ' aria-disabled="true"'; ?>>
                            <?php echo e($externalLinkButtonText); ?>
                        </a>
                    </div>
                    <?php if (!empty($productVariants)): ?>
                        <div class="product-variant product-variant-<?php echo e($variantStyle); ?>">
                            <div class="product-variant-label">Select Version</div>
                            <?php if ($variantStyle === 'buttons'): ?>
                                <div class="product-variant-buttons">
                                    <?php foreach ($productVariants as $variant):
                                        $variantLabel = trim($variant['label'] ?? '');
                                        if ($variantLabel === '') {
                                            continue;
                                        }
                                        $variantId = (int)($variant['id'] ?? 0);
                                        $variantPrice = trim($variant['price'] ?? '');
                                        $variantDisplayPrice = formatProductPrice($variantPrice);
                                        $variantAction = $variant['action_type'] ?? $actionType;
                                        $variantAction = in_array($variantAction, ['cart', 'download', 'external_link'], true) ? $variantAction : $actionType;
                                        $variantDownload = productDownloadUrlToPublic((string)($variant['download_url'] ?? '')) ?: $downloadUrl;
                                        $variantLinkLabel = trim($variant['external_link_label'] ?? '');
                                        $variantActionEffective = $variantAction;
                                        $variantPriceValue = CurrencyService::parsePrice($variantPrice);
                                        if ($variantActionEffective === 'download' && $variantPriceValue > 0) {
                                            $variantActionEffective = 'cart';
                                        }
                                        if ($variantActionEffective === 'external_link') {
                                            $variantDisplayPrice = '';
                                        }
                                        if ($variantActionEffective === 'download') {
                                            $variantDisplayPrice = 'Free';
                                            if ($variantDownload !== '') {
                                                $variantDownload = productFreeDownloadTrackingUrl((int)$product['id'], $variantId);
                                            }
                                        }
                                        $isDefault = !empty($variant['is_default']);
                                        if (!$isDefault && $defaultVariant && $variantLabel === ($defaultVariant['label'] ?? '')) {
                                            $isDefault = true;
                                        }
                                        $variantStock = $variant['stock'] ?? null;
                                        $variantUnlimited = $variantStock === null;
                                        $variantStockValue = $variantUnlimited ? '' : (int)$variantStock;
                                        $variantOutOfStock = !$variantUnlimited && $variantStockValue === 0;
                                        ?>
                                        <?php $variantOwned = $hasDigitalPurchase || ($variantId > 0 && in_array($variantId, $purchasedVariantIds, true)); ?>
                                        <?php
                                        // Use external link button text as display label if variant is external link and label is set
                                        $variantDisplayLabel = ($variantActionEffective === 'external_link' && $variantLinkLabel !== '') ? $variantLinkLabel : $variantLabel;
                                        ?>
                                        <button type="button" class="product-variant-btn<?php echo $isDefault ? $classActive : ''; ?><?php echo $variantOutOfStock ? ' is-out-of-stock' : ''; ?>" data-variant-id="<?php echo e($variantId); ?>" data-price="<?php echo e($variantDisplayPrice); ?>" data-stock="<?php echo e($variantStockValue); ?>" data-unlimited="<?php echo $variantUnlimited ? '1' : '0'; ?>" data-action="<?php echo e($variantActionEffective); ?>" data-download="<?php echo e($variantDownload); ?>" data-link-label="<?php echo e($variantLinkLabel); ?>" data-digital="<?php echo $variantAction === 'download' ? '1' : '0'; ?>" data-owned="<?php echo $variantOwned ? '1' : '0'; ?>">
                                            <?php echo e($variantDisplayLabel); ?>
                                        </button>
                                    <?php endforeach; ?>
                                </div>
                            <?php else: ?>
                                <select class="product-variant-select">
                                    <?php foreach ($productVariants as $variant):
                                        $variantLabel = trim($variant['label'] ?? '');
                                        if ($variantLabel === '') {
                                            continue;
                                        }
                                        $variantId = (int)($variant['id'] ?? 0);
                                        $variantPrice = trim($variant['price'] ?? '');
                                        $variantDisplayPrice = formatProductPrice($variantPrice);
                                        $variantAction = $variant['action_type'] ?? $actionType;
                                        $variantAction = in_array($variantAction, ['cart', 'download', 'external_link'], true) ? $variantAction : $actionType;
                                        $variantDownload = productDownloadUrlToPublic((string)($variant['download_url'] ?? '')) ?: $downloadUrl;
                                        $variantLinkLabel = trim($variant['external_link_label'] ?? '');
                                        $variantActionEffective = $variantAction;
                                        $variantPriceValue = CurrencyService::parsePrice($variantPrice);
                                        if ($variantActionEffective === 'download' && $variantPriceValue > 0) {
                                            $variantActionEffective = 'cart';
                                        }
                                        if ($variantActionEffective === 'external_link') {
                                            $variantDisplayPrice = '';
                                        }
                                        if ($variantActionEffective === 'download') {
                                            $variantDisplayPrice = 'Free';
                                            if ($variantDownload !== '') {
                                                $variantDownload = productFreeDownloadTrackingUrl((int)$product['id'], $variantId);
                                            }
                                        }
                                        $optionDisplayLabel = ($variantActionEffective === 'external_link' && $variantLinkLabel !== '') ? $variantLinkLabel : $variantLabel;
                                        $optionLabel = $optionDisplayLabel;
                                        if ($variantDisplayPrice !== '') {
                                            $optionLabel .= ' (' . $variantDisplayPrice . ')';
                                        }
                                        $isDefault = !empty($variant['is_default']);
                                        if (!$isDefault && $defaultVariant && $variantLabel === ($defaultVariant['label'] ?? '')) {
                                            $isDefault = true;
                                        }
                                        $variantStock = $variant['stock'] ?? null;
                                        $variantUnlimited = $variantStock === null;
                                        $variantStockValue = $variantUnlimited ? '' : (int)$variantStock;
                                        if (!$variantUnlimited && $variantStockValue === 0) {
                                            $optionLabel .= ' (Out of stock)';
                                        }
                                        ?>
                                        <?php $variantOwned = $hasDigitalPurchase || ($variantId > 0 && in_array($variantId, $purchasedVariantIds, true)); ?>
                                        <option value="<?php echo e($variantLabel); ?>" data-variant-id="<?php echo e($variantId); ?>" data-price="<?php echo e($variantDisplayPrice); ?>" data-stock="<?php echo e($variantStockValue); ?>" data-unlimited="<?php echo $variantUnlimited ? '1' : '0'; ?>" data-action="<?php echo e($variantActionEffective); ?>" data-download="<?php echo e($variantDownload); ?>" data-link-label="<?php echo e($variantLinkLabel); ?>" data-digital="<?php echo $variantAction === 'download' ? '1' : '0'; ?>" data-owned="<?php echo $variantOwned ? '1' : '0'; ?>" <?php echo $isDefault ? 'selected' : ''; ?> <?php echo (!$variantUnlimited && $variantStockValue === 0) ? 'disabled' : ''; ?>>
                                            <?php echo e($optionLabel); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <?php if ($allowComments): ?>
        <div class="product-comments-section" id="product-comments">
            <div class="product-comments-header">
                <h3>Comments (<?php echo (int)count($productComments); ?>)</h3>
            </div>

            <?php if ($productCommentFlashMessage !== ''): ?>
                <div class="product-comment-flash product-comment-flash-<?php echo e($productCommentFlashType); ?>">
                    <?php echo e($productCommentFlashMessage); ?>
                </div>
            <?php endif; ?>

            <?php if (empty($productComments)): ?>
                <p class="product-comments-empty">No comments yet. Be the first to comment.</p>
            <?php else: ?>
                <div class="product-comments-list">
                    <?php foreach ($productComments as $productComment): ?>
                        <?php
                            $commentAuthor = trim((string)($productComment['nickname'] ?? ''));
                        if ($commentAuthor === '') {
                            $commentAuthor = trim((string)($productComment['display_name'] ?? ''));
                        }
                        if ($commentAuthor === '') {
                            $commentAuthor = 'User';
                        }
                        $commentCreatedAt = trim((string)($productComment['created_at'] ?? ''));
                        $commentDate = $commentCreatedAt !== '' ? date('M j, Y g:i A', strtotime($commentCreatedAt)) : '';
                        $commentPhoto = trim((string)($productComment['profile_photo'] ?? ''));
                        $commentAvatar = $commentPhoto !== '' ? toPublicFilePath($commentPhoto) : '/assets/images/default-avatar.svg';
                        $commentNickname = trim((string)($productComment['nickname'] ?? ''));
                        $commentProfileLink = $commentNickname !== '' ? '/@' . rawurlencode($commentNickname) : '';
                        ?>
                        <article class="product-comment-item">
                            <div class="product-comment-avatar">
                                <img src="<?php echo e($commentAvatar); ?>" alt="<?php echo e($commentAuthor); ?>" loading="lazy">
                            </div>
                            <div class="product-comment-content">
                                <div class="product-comment-meta">
                                    <?php if ($commentProfileLink !== ''): ?>
                                        <a class="product-comment-author" href="<?php echo e($commentProfileLink); ?>"><?php echo e($commentAuthor); ?></a>
                                    <?php else: ?>
                                        <strong><?php echo e($commentAuthor); ?></strong>
                                    <?php endif; ?>
                                    <?php if ($commentDate !== ''): ?>
                                        <span><?php echo e($commentDate); ?></span>
                                    <?php endif; ?>
                                </div>
                                <div class="product-comment-body"><?php echo nl2br(e((string)($productComment['body'] ?? ''))); ?></div>
                            </div>
                        </article>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>

            <?php if ($canPostProductComment): ?>
                <form method="POST" class="product-comment-form" action="">
                    <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                    <input type="hidden" name="action" value="product_comment_submit">
                    <input type="hidden" name="product_id" value="<?php echo (int)$product['id']; ?>">
                    <label for="product-comment-body-<?php echo (int)$product['id']; ?>">Add a comment</label>
                    <textarea id="product-comment-body-<?php echo (int)$product['id']; ?>" name="comment_body" rows="4" maxlength="2000" required></textarea>
                    <button type="submit" class="btn btn-primary">Post Comment</button>
                </form>
            <?php else: ?>
                <p class="product-comments-login-note">Please <a href="/login">log in</a> to post a comment.</p>
            <?php endif; ?>
        </div>
        <?php endif; ?>

        <?php
        // ── Similar Products ─────────────────────────────────────────
        // Shows other active products from the same collection.
        // All settings are stored in the section JSON (theme editor).
        $showSimilar = (bool)($sectionSettings['show_similar_products'] ?? false);
        $collectionId = (int)($product['collection_id'] ?? 0);
        // Regex to detect video file extensions (used for filtering gallery media)
        $videoExtPattern = '/\.(mp4|webm|mov|m4v|avi|mkv)(\?.*)?$/i';
        // Regex to detect YouTube URLs (embeds won't render in cards)
        $youtubePattern = '~^https?://(www\.)?(youtube\.com|youtu\.be)/~i';
        if ($showSimilar && $collectionId > 0):
            $simTitle = trim($sectionSettings['similar_title'] ?? 'Similar Products');
            $simTitleAlign = $sectionSettings['similar_title_align'] ?? 'center';
            $simTitleAlign = in_array($simTitleAlign, ['left', 'center', 'right'], true) ? $simTitleAlign : 'center';
            $simColumns = max(2, min(4, (int)($sectionSettings['similar_columns'] ?? 4)));
            $simShowPrice = (bool)($sectionSettings['similar_show_price'] ?? true);
            $simShowDescription = (bool)($sectionSettings['similar_show_description'] ?? false);
            $simHoverSwap = (bool)($sectionSettings['similar_hover_swap_media'] ?? true);
            $simMax = max(1, min(16, (int)($sectionSettings['similar_max_products'] ?? 8)));

            // Fetch similar products from the same collection, excluding the current one
            $simStmt = $db->prepare(
                "SELECT p.*, pg.slug AS page_slug, pm.media_url AS gallery_media
                 FROM products p
                 LEFT JOIN pages pg ON pg.id = p.page_id
                 LEFT JOIN product_media pm ON pm.product_id = p.id AND pm.sort_order = 0
                 WHERE p.collection_id = ? AND p.id != ? AND p.is_active = 1
                 ORDER BY p.created_at DESC
                 LIMIT ?"
            );
            $simStmt->execute([$collectionId, (int)$product['id'], $simMax]);
            $similarProducts = $simStmt->fetchAll();

            if (!empty($similarProducts)):
                // Pre-fetch media for hover swap
                $simProductIds = array_map(function ($p) { return (int)$p['id']; }, $similarProducts);
                $simMediaByProduct = [];
                if (!empty($simProductIds)) {
                    $simPlaceholders = implode(',', array_fill(0, count($simProductIds), '?'));
                    $simMediaStmt = $db->prepare(
                        "SELECT product_id, media_url FROM product_media
                         WHERE product_id IN ($simPlaceholders)
                         ORDER BY sort_order ASC, id ASC"
                    );
                    $simMediaStmt->execute($simProductIds);
                    foreach ($simMediaStmt->fetchAll() as $simMediaRow) {
                        $simPid = (int)$simMediaRow['product_id'];
                        $simMediaByProduct[$simPid][] = $simMediaRow['media_url'];
                    }
                }

                // Responsive sizes per column count
                $simSizesMap = [
                    2 => '(max-width: 640px) 100vw, (max-width: 900px) 50vw, 50vw',
                    3 => '(max-width: 640px) 100vw, (max-width: 900px) 50vw, 33vw',
                    4 => '(max-width: 640px) 100vw, (max-width: 900px) 50vw, 25vw',
                ];
                $simSizes = $simSizesMap[$simColumns] ?? $simSizesMap[4];
        ?>
        <div class="similar-products-section">
            <?php if ($simTitle !== ''): ?>
                <div class="similar-products-header text-align-<?php echo e($simTitleAlign); ?>">
                    <h3 class="text-align-<?php echo e($simTitleAlign); ?>"><?php echo e($simTitle); ?></h3>
                </div>
            <?php endif; ?>
            <div class="product-list-grid columns-<?php echo $simColumns; ?> view-grid">
                <?php foreach ($similarProducts as $simProd):
                    $simPid = (int)$simProd['id'];
                    $simName = trim($simProd['name'] ?? '');
                    $simPrice = trim($simProd['price'] ?? '');
                    $simDisplayPrice = formatProductPrice($simPrice);
                    $simDescription = trim($simProd['description'] ?? '');
                    $simShortDesc = $simDescription !== '' ? trim(preg_replace('/\s+/', ' ', strip_tags($simDescription))) : '';

                    // Resolve primary image
                    $simDefaultMedia = trim($simProd['gallery_media'] ?? $simProd['media_url'] ?? '');
                    $simMediaList = $simMediaByProduct[$simPid] ?? [];
                    $simMediaUrl = '';
                    $simPrimaryRaw = '';

                    foreach ($simMediaList as $simItem) {
                        $simItem = trim((string)$simItem);
                        if ($simItem === '' || preg_match($videoExtPattern, $simItem) || preg_match($youtubePattern, $simItem)) {
                            continue;
                        }
                        $simCandidate = toPublicFilePath($simItem);
                        if ($simCandidate !== '') {
                            $simMediaUrl = $simCandidate;
                            $simPrimaryRaw = $simItem;
                            break;
                        }
                    }

                    if ($simMediaUrl === '' && $simDefaultMedia !== '' && !preg_match($videoExtPattern, $simDefaultMedia) && !preg_match($youtubePattern, $simDefaultMedia)) {
                        $simCandidate = toPublicFilePath($simDefaultMedia);
                        if ($simCandidate !== '') {
                            $simMediaUrl = $simCandidate;
                            $simPrimaryRaw = $simDefaultMedia;
                        }
                    }

                    // Hover media
                    $simHoverMedia = '';
                    if ($simHoverSwap && !empty($simMediaList)) {
                        foreach ($simMediaList as $simItem) {
                            $simItem = trim((string)$simItem);
                            if ($simItem === '' || $simItem === $simPrimaryRaw) {
                                continue;
                            }
                            if (preg_match($videoExtPattern, $simItem) || preg_match($youtubePattern, $simItem)) {
                                continue;
                            }
                            $simHoverCandidate = toPublicFilePath($simItem);
                            if ($simHoverCandidate !== '' && $simHoverCandidate !== $simMediaUrl) {
                                $simHoverMedia = $simHoverCandidate;
                                break;
                            }
                        }
                    }

                    // Build product URL
                    $simPageSlug = trim($simProd['page_slug'] ?? '');
                    $simProductSlug = trim((string)($simProd['product_slug'] ?? ''));
                    if ($simPageSlug !== '' && $simProductSlug !== '') {
                        $simProductUrl = '/' . $simPageSlug . '/' . rawurlencode($simProductSlug);
                    } else {
                        $simProductUrl = $simPageSlug !== '' ? '/' . $simPageSlug . '?product=' . $simPid : '/product?product=' . $simPid;
                    }

                    // WebP srcset variants
                    $simWebpSm = $simMediaUrl . '?w=400&format=webp';
                    $simWebpMd = $simMediaUrl . '?w=700&format=webp';
                    $simWebpLg = $simMediaUrl . '?w=1000&format=webp';
                ?>
                    <a class="product-list-card <?php echo ($simHoverSwap && $simHoverMedia !== '') ? 'has-hover-media' : ''; ?>" href="<?php echo e($simProductUrl); ?>" data-hover-media="<?php echo e($simHoverMedia); ?>">
                        <?php if ($simMediaUrl !== ''): ?>
                            <div class="product-list-media">
                                <img src="<?php echo e($simWebpMd); ?>" srcset="<?php echo e($simWebpSm); ?> 400w, <?php echo e($simWebpMd); ?> 700w, <?php echo e($simWebpLg); ?> 1000w" sizes="<?php echo e($simSizes); ?>" alt="<?php echo e($simName); ?>" loading="lazy" decoding="async">
                            </div>
                        <?php endif; ?>
                        <div class="product-list-body">
                            <?php if ($simName !== ''): ?>
                                <h3 class="product-list-title"><?php echo e($simName); ?></h3>
                            <?php endif; ?>
                            <?php if ($simShowPrice && $simDisplayPrice !== ''): ?>
                                <div class="product-list-price"><?php echo e($simDisplayPrice); ?></div>
                            <?php endif; ?>
                            <?php if ($simShowDescription && $simShortDesc !== ''): ?>
                                <p class="product-list-description"><?php echo e($simShortDesc); ?></p>
                            <?php endif; ?>
                        </div>
                    </a>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
            endif; // !empty($similarProducts)
        endif; // $showSimilar && $collectionId > 0
        ?>
    </div>
</section>
