<?php
use NewSite\Auth\SiteAuth;
use NewSite\Cleanup\MissingUploadsService;
use NewSite\Database\DatabaseManager;
use NewSite\User\FriendService;
use NewSite\User\PrivacyService;

/**
 * Public User Profile Page
 * Shows user profile with privacy-respecting field visibility
 */

$isUserLoggedIn = SiteAuth::isLoggedIn();
$viewerId = $isUserLoggedIn ? (int)$_SESSION['site_user_id'] : null;
$profileUserId = $profileUser['id'] ?? 0;
$isOwnProfile = $viewerId && $viewerId === $profileUserId;

// Get visible profile data respecting privacy settings
$castViewerId = $viewerId !== null ? (int)$viewerId : null;
$profileData = $profileUser ? PrivacyService::getVisibleProfileData((int)$profileUserId, $castViewerId) : null;

if (!$profileData) {
    ?>
    <div class="profile-container">
        <div class="profile-not-found">
            <h1>User Not Found</h1>
            <p>This user doesn't exist or their profile is not available.</p>
            <a href="/" class="btn btn-primary">Go Home</a>
        </div>
    </div>
    <?php
    return;
}

// Get friendship status
$friendshipStatus = $viewerId ? FriendService::getStatus((int)$viewerId, (int)$profileUserId) : 'none';

$messengerPrivacy = 'friends';
if ($viewerId && !$isOwnProfile) {
    $privacySettings = PrivacyService::getUserSettings((int)$profileUserId);
    $messengerPrivacy = $privacySettings['messenger'] ?? 'friends';
}
$canMessage = $viewerId && !$isOwnProfile && $friendshipStatus !== 'blocked' && $friendshipStatus !== 'blocked_by' && ($friendshipStatus === 'friends' || $messengerPrivacy === 'everyone');

// Get mutual friends if not own profile and logged in
$mutualFriends = [];
$mutualFriendsCount = 0;
if ($viewerId && !$isOwnProfile) {
    $mutualFriends = FriendService::getMutual((int)$viewerId, (int)$profileUserId, 5);
    $mutualFriendsCount = FriendService::getMutualCount((int)$viewerId, (int)$profileUserId);
}

// Get friend count for this profile
$friendCount = FriendService::getCount((int)$profileUserId);

// Get pending request ID if we received one from this user
$incomingRequestId = 0;
if ($friendshipStatus === 'request_received') {
    $db = DatabaseManager::getWriteConnection();
    $stmt = $db->prepare("SELECT id FROM friend_requests WHERE from_user_id = ? AND to_user_id = ? AND status = 'pending'");
    $stmt->execute([$profileUserId, $viewerId]);
    $incomingRequestId = (int)$stmt->fetchColumn();
}

// Get our sent request ID if we sent one
$sentRequestId = 0;
if ($friendshipStatus === 'request_sent') {
    $db = DatabaseManager::getWriteConnection();
    $stmt = $db->prepare("SELECT id FROM friend_requests WHERE from_user_id = ? AND to_user_id = ? AND status = 'pending'");
    $stmt->execute([$viewerId, $profileUserId]);
    $sentRequestId = (int)$stmt->fetchColumn();
}

// Flash messages
$flashMessage = $_SESSION['friend_flash_message'] ?? '';
$flashError = $_SESSION['friend_flash_error'] ?? '';
unset($_SESSION['friend_flash_message'], $_SESSION['friend_flash_error']);

$memberSince = '';
if (!empty($profileData['member_since'])) {
    $memberSince = date('F Y', is_numeric($profileData['member_since']) ? $profileData['member_since'] : strtotime($profileData['member_since']));
}
$defaultAvatar = '/assets/images/default-avatar.svg';
$canViewProfilePhoto = !empty($profileData['profile_photo_visible']);
$profilePhotoPath = $canViewProfilePhoto ? (string)($profileData['profile_photo'] ?? '') : '';
if ($profilePhotoPath !== '' && MissingUploadsService::isMissingPath($profilePhotoPath)) {
    $profilePhotoPath = '';
}
$profilePhotoUrl = $profilePhotoPath !== '' ? toPublicFilePath($profilePhotoPath) : $defaultAvatar;
?>

<div class="profile-container" itemscope itemtype="https://schema.org/Person">
    <?php if ($flashMessage): ?>
        <div class="alert alert-success"><?php echo e($flashMessage); ?></div>
    <?php endif; ?>
    <?php if ($flashError): ?>
        <div class="alert alert-error"><?php echo e($flashError); ?></div>
    <?php endif; ?>

    <div class="profile-header">
        <div class="profile-photo-section">
            <img src="<?php echo e($profilePhotoUrl); ?>" alt="Profile" class="profile-photo-large"<?php echo ($canViewProfilePhoto && $profilePhotoPath !== '') ? ' itemprop="image"' : ''; ?>>
        </div>

        <div class="profile-info-section">
            <div class="profile-name-row">
                <?php if ($profileData['display_name_visible'] && $profileData['display_name']): ?>
                    <h1 class="profile-display-name" itemprop="name"><?php echo e($profileData['display_name']); ?></h1>
                <?php endif; ?>
                <?php if ($profileData['nickname']): ?>
                    <span class="profile-nickname"<?php echo (!$profileData['display_name_visible'] || !$profileData['display_name']) ? ' itemprop="name"' : ''; ?>>@<?php echo e($profileData['nickname']); ?></span>
                <?php endif; ?>
            </div>

            <div class="profile-meta">
                <?php if ($memberSince): ?>
                    <span class="profile-meta-item">
                        <svg viewBox="0 0 24 24" width="16" height="16" fill="none" stroke="currentColor" stroke-width="2">
                            <rect x="3" y="4" width="18" height="18" rx="2" ry="2"></rect>
                            <line x1="16" y1="2" x2="16" y2="6"></line>
                            <line x1="8" y1="2" x2="8" y2="6"></line>
                            <line x1="3" y1="10" x2="21" y2="10"></line>
                        </svg>
                        Member since <?php echo e($memberSince); ?>
                    </span>
                <?php endif; ?>
                <span class="profile-meta-item">
                    <svg viewBox="0 0 24 24" width="16" height="16" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"></path>
                        <circle cx="9" cy="7" r="4"></circle>
                        <path d="M23 21v-2a4 4 0 0 0-3-3.87"></path>
                        <path d="M16 3.13a4 4 0 0 1 0 7.75"></path>
                    </svg>
                    <?php echo number_format($friendCount); ?> friend<?php echo $friendCount !== 1 ? 's' : ''; ?>
                </span>
            </div>

            <?php if ($profileData['description_visible'] && $profileData['description']): ?>
                <p class="profile-description" itemprop="description"><?php echo nl2br(e($profileData['description'])); ?></p>
            <?php endif; ?>

            <div class="profile-actions">
                <?php if ($isOwnProfile): ?>
                    <a href="/account?tab=settings" class="btn btn-secondary">Edit Profile</a>
                    <a href="/account?tab=friends&ftab=all" class="btn btn-outline">My Friends</a>
                <?php elseif ($viewerId): ?>
                    <?php if ($friendshipStatus === 'friends'): ?>
                        <span class="profile-status-badge status-friends">✓ Friends</span>
                        <button type="button" class="btn btn-primary btn-sm" data-messenger-open="<?php echo $profileUserId; ?>" data-messenger-name="<?php echo e($profileData['display_name'] ?: $profileData['nickname']); ?>" data-messenger-nickname="<?php echo e($profileData['nickname'] ?? $profileData['display_name'] ?? ''); ?>" data-messenger-avatar="<?php echo e($profilePhotoUrl); ?>" data-messenger-status="<?php echo e($profileData['status'] ?? 'offline'); ?>">Message</button>
                        <form method="post" class="inline-form" data-confirm="Remove this friend?">
                            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                            <input type="hidden" name="friend_action" value="remove_friend">
                            <input type="hidden" name="target_user_id" value="<?php echo $profileUserId; ?>">
                            <input type="hidden" name="return_url" value="/@<?php echo e($profileData['nickname']); ?>">
                            <button type="submit" class="btn btn-outline btn-sm">Unfriend</button>
                        </form>
                    <?php elseif ($friendshipStatus === 'request_sent'): ?>
                        <span class="profile-status-badge status-pending">Request Sent</span>
                        <form method="post" class="inline-form">
                            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                            <input type="hidden" name="friend_action" value="cancel_request">
                            <input type="hidden" name="request_id" value="<?php echo $sentRequestId; ?>">
                            <input type="hidden" name="return_url" value="/@<?php echo e($profileData['nickname']); ?>">
                            <button type="submit" class="btn btn-outline btn-sm">Cancel Request</button>
                        </form>
                    <?php elseif ($friendshipStatus === 'request_received'): ?>
                        <span class="profile-status-badge status-pending">Wants to be friends</span>
                        <form method="post" class="inline-form">
                            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                            <input type="hidden" name="friend_action" value="accept_request">
                            <input type="hidden" name="request_id" value="<?php echo $incomingRequestId; ?>">
                            <input type="hidden" name="return_url" value="/@<?php echo e($profileData['nickname']); ?>">
                            <button type="submit" class="btn btn-primary btn-sm">Accept</button>
                        </form>
                        <form method="post" class="inline-form">
                            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                            <input type="hidden" name="friend_action" value="decline_request">
                            <input type="hidden" name="request_id" value="<?php echo $incomingRequestId; ?>">
                            <input type="hidden" name="return_url" value="/@<?php echo e($profileData['nickname']); ?>">
                            <button type="submit" class="btn btn-outline btn-sm">Decline</button>
                        </form>
                    <?php elseif ($friendshipStatus === 'blocked'): ?>
                        <span class="profile-status-badge status-blocked">Blocked</span>
                        <form method="post" class="inline-form">
                            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                            <input type="hidden" name="friend_action" value="unblock_user">
                            <input type="hidden" name="target_user_id" value="<?php echo $profileUserId; ?>">
                            <input type="hidden" name="return_url" value="/@<?php echo e($profileData['nickname']); ?>">
                            <button type="submit" class="btn btn-outline btn-sm">Unblock</button>
                        </form>
                    <?php elseif ($friendshipStatus !== 'blocked_by'): ?>
                        <button type="button" class="btn btn-icon btn-primary has-tooltip" data-friend-action="send_request" data-target-user="<?php echo $profileUserId; ?>" data-tooltip="Send Friend Request">
                            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round"><line x1="12" y1="5" x2="12" y2="19"/><line x1="5" y1="12" x2="19" y2="12"/></svg>
                        </button>
                    <?php endif; ?>

                    <?php if ($canMessage && $friendshipStatus !== 'friends'): ?>
                        <button type="button" class="btn btn-primary btn-sm" data-messenger-open="<?php echo $profileUserId; ?>" data-messenger-name="<?php echo e($profileData['display_name'] ?: $profileData['nickname']); ?>" data-messenger-nickname="<?php echo e($profileData['nickname'] ?? $profileData['display_name'] ?? ''); ?>" data-messenger-avatar="<?php echo e($profilePhotoUrl); ?>" data-messenger-status="<?php echo e($profileData['status'] ?? 'offline'); ?>">Message</button>
                    <?php endif; ?>

                    <?php if ($friendshipStatus !== 'blocked' && $friendshipStatus !== 'blocked_by'): ?>
                        <form method="post" class="inline-form" data-confirm="Block this user? They won't be able to see your profile or send you friend requests.">
                            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                            <input type="hidden" name="friend_action" value="block_user">
                            <input type="hidden" name="target_user_id" value="<?php echo $profileUserId; ?>">
                            <input type="hidden" name="return_url" value="/@<?php echo e($profileData['nickname']); ?>">
                            <button type="submit" class="btn btn-outline btn-sm btn-danger">Block</button>
                        </form>
                    <?php endif; ?>
                <?php else: ?>
                    <a href="/login" class="btn btn-primary">Login to Connect</a>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <?php if ($mutualFriendsCount > 0 && !$isOwnProfile): ?>
    <div class="profile-section mutual-friends-section">
        <h3 class="profile-section-title"><?php echo $mutualFriendsCount; ?> Mutual Friend<?php echo $mutualFriendsCount !== 1 ? 's' : ''; ?></h3>
        <div class="mutual-friends-list">
            <?php foreach ($mutualFriends as $mutual): ?>
                <a href="/@<?php echo e($mutual['nickname']); ?>" class="mutual-friend-item" title="<?php echo e($mutual['display_name'] ?: $mutual['nickname']); ?>">
                    <?php if (!empty($mutual['profile_photo'])): ?>
                        <img src="<?php echo e(toPublicFilePath($mutual['profile_photo'])); ?>" alt="" class="mutual-friend-photo">
                    <?php else: ?>
                        <div class="mutual-friend-placeholder">
                            <?php echo strtoupper(substr($mutual['display_name'] ?? $mutual['nickname'] ?? '?', 0, 1)); ?>
                        </div>
                    <?php endif; ?>
                </a>
            <?php endforeach; ?>
            <?php if ($mutualFriendsCount > 5): ?>
                <span class="mutual-friends-more">+<?php echo $mutualFriendsCount - 5; ?> more</span>
            <?php endif; ?>
        </div>
    </div>
    <?php endif; ?>

    <div class="profile-details">
        <?php
        $detailFields = [
            'website' => ['icon' => '<svg viewBox="0 0 24 24" width="18" height="18" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"></circle><line x1="2" y1="12" x2="22" y2="12"></line><path d="M12 2a15.3 15.3 0 0 1 4 10 15.3 15.3 0 0 1-4 10 15.3 15.3 0 0 1-4-10 15.3 15.3 0 0 1 4-10z"></path></svg>', 'label' => 'Website', 'link' => true],
            'company' => ['icon' => '<svg viewBox="0 0 24 24" width="18" height="18" fill="none" stroke="currentColor" stroke-width="2"><path d="M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0 1 18 0z"></path><circle cx="12" cy="10" r="3"></circle></svg>', 'label' => 'Company'],
            'country' => ['icon' => '<svg viewBox="0 0 24 24" width="18" height="18" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"></circle><path d="M12 2a10 10 0 0 0-7.07 17.07"></path></svg>', 'label' => 'Location'],
        ];

$hasDetails = false;
foreach ($detailFields as $field => $config) {
    if ($profileData[$field . '_visible'] && !empty($profileData[$field])) {
        $hasDetails = true;
        break;
    }
}

if ($hasDetails) {
    ?>
        <div class="profile-section">
            <h3 class="profile-section-title">About</h3>
            <div class="profile-detail-list">
                <?php foreach ($detailFields as $field => $config): ?>
                    <?php if ($profileData[$field . '_visible'] && !empty($profileData[$field])): ?>
                        <div class="profile-detail-item">
                            <span class="profile-detail-icon"><?php echo $config['icon']; ?></span>
                            <span class="profile-detail-label"><?php echo $config['label']; ?>:</span>
                            <?php if (!empty($config['link']) && filter_var($profileData[$field], FILTER_VALIDATE_URL)): ?>
                                <a href="<?php echo e($profileData[$field]); ?>" target="_blank" rel="noopener" class="profile-detail-value profile-detail-link">
                                    <?php echo e(preg_replace('#^https?://#', '', $profileData[$field])); ?>
                                </a>
                            <?php else: ?>
                                <span class="profile-detail-value"><?php echo e($profileData[$field]); ?></span>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                <?php endforeach; ?>
            </div>
        </div>
        <?php } ?>
    </div>
</div>
