<?php
use NewSite\Auth\GoogleOAuthService;
use NewSite\Database\DatabaseManager;
use NewSite\Email\SmtpMailer;
use NewSite\Security\IpService;
use NewSite\Security\RateLimiter;
use NewSite\Settings\SettingsService;
use NewSite\Template\ContentRenderer;
use NewSite\User\UserService;

$email = trim($_POST['email'] ?? '');
$message = '';
$error = $loginTokenError ?? '';
$googleError = $googleOAuthError ?? '';
$allowRegistration = SettingsService::get('allow_registration', '0') === '1';
$googleEnabled = GoogleOAuthService::isEnabled();
$db = DatabaseManager::getWriteConnection();

if ($_SERVER['REQUEST_METHOD'] === 'POST' && $email !== '') {
    $ip = IpService::getClientIP();
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';

    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Please enter a valid email address.';
    } else {
        $rateStatus = RateLimiter::loginStatus($ip);
        if (!$rateStatus['ok']) {
            $wait = (int)$rateStatus['wait'];
            $error = 'Please wait ' . $wait . ' seconds before requesting another link.';
        }
    }

    if ($error === '') {
        $user = UserService::getByEmail($email);
        if (!$allowRegistration) {
            if (!$user) {
                $error = 'Login is currently disabled.';
            } else {
                $stmt = $db->prepare("SELECT COUNT(*) FROM orders WHERE user_id = ? AND status = 'paid'");
                $stmt->execute([(int)$user['id']]);
                $hasPurchases = (int)$stmt->fetchColumn() > 0;
                if (!$hasPurchases) {
                    $error = 'Login is currently disabled for accounts without purchases.';
                }
            }
        }

        if ($error === '') {
            $needsLegal = !$user || UserService::needsLegalAcceptance($user);
            if (!$user) {
                $userId = UserService::create($email, $ip, true);
            } else {
                $userId = (int)$user['id'];
                if ($needsLegal) {
                    UserService::recordLegalAcceptance((int)$userId, $ip);
                }
            }

            $smtp = [
                'host' => SettingsService::get('smtp_host', ''),
                'port' => SettingsService::get('smtp_port', '587'),
                'user' => SettingsService::get('smtp_user', ''),
                'pass' => SettingsService::get('smtp_pass', ''),
                'secure' => SettingsService::get('smtp_secure', 'tls')
            ];

            if ($smtp['host'] === '' || $smtp['user'] === '' || $smtp['pass'] === '') {
                $error = 'SMTP settings are not configured.';
            } else {
                $token = UserService::createLoginToken((int)$userId, $ip, $userAgent);
                $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
                $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
                $loginUrl = $scheme . '://' . $host . '/login?token=' . urlencode($token);

                $siteName = trim((string)SettingsService::get('site_name', ''));
                $subjectTemplate = SettingsService::get('login_email_subject', 'Your login link');
                $bodyTemplate = SettingsService::get('login_email_body', "Hello,\n\nUse the link below to log in:\n{login_url}\n\nIf you did not request this, you can ignore this email.\n\nThank you,\n{site_name}");
                $vars = [
                    '{login_url}' => $loginUrl,
                    '{email}' => $email,
                    '{site_name}' => $siteName
                ];
                $subject = ContentRenderer::renderEmailTemplate($subjectTemplate, $vars);
                $body = ContentRenderer::renderEmailTemplate($bodyTemplate, $vars);

                if (SmtpMailer::send($email, $subject, $body, '', '', $smtp)) {
                    $message = 'System sent a login link to your email.';
                    RateLimiter::loginUpdate($ip);
                } else {
                    $error = 'Failed to send the login email. Please try again later.';
                }
            }
        }
    }
}
?>

<section class="auth-section">
    <div class="container">
        <div class="auth-card">
            <h1>Login</h1>
            <p class="auth-subtitle">Enter your email to receive a secure login link.</p>

            <?php if ($message): ?>
                <div class="auth-alert success"><?php echo e($message); ?></div>
            <?php endif; ?>

            <?php if ($error): ?>
                <div class="auth-alert error"><?php echo e($error); ?></div>
            <?php endif; ?>

            <?php if ($googleError): ?>
                <div class="auth-alert error"><?php echo e($googleError); ?></div>
            <?php endif; ?>

            <form method="POST" action="/login" class="auth-form">
                <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                <label for="login_email">Email address</label>
                <input type="email" id="login_email" name="email" placeholder="you@example.com" value="<?php echo e($email); ?>" required autocomplete="email">
                <div class="auth-terms">
                    By requesting a login link, you agree to our <a href="/terms-of-service">Terms of Service</a> and <a href="/privacy-policy">Privacy Policy</a>.
                </div>
                <button type="submit" class="btn btn-primary">Send login link</button>
            </form>

            <?php if ($googleEnabled): ?>
            <div class="auth-divider">
                <span>or</span>
            </div>
            <a href="<?php echo e(GoogleOAuthService::getAuthUrl()); ?>" class="btn btn-google-login">
                <svg viewBox="0 0 24 24" aria-hidden="true"><path d="M22.56 12.25c0-.78-.07-1.53-.2-2.25H12v4.26h5.92a5.06 5.06 0 0 1-2.2 3.32v2.77h3.57c2.08-1.92 3.28-4.74 3.28-8.1z" fill="#4285F4"></path><path d="M12 23c2.97 0 5.46-.98 7.28-2.66l-3.57-2.77c-.98.66-2.23 1.06-3.71 1.06-2.86 0-5.29-1.93-6.16-4.53H2.18v2.84C3.99 20.53 7.7 23 12 23z" fill="#34A853"></path><path d="M5.84 14.09c-.22-.66-.35-1.36-.35-2.09s.13-1.43.35-2.09V7.07H2.18C1.43 8.55 1 10.22 1 12s.43 3.45 1.18 4.93l2.85-2.22.81-.62z" fill="#FBBC05"></path><path d="M12 5.38c1.62 0 3.06.56 4.21 1.64l3.15-3.15C17.45 2.09 14.97 1 12 1 7.7 1 3.99 3.47 2.18 7.07l3.66 2.84c.87-2.6 3.3-4.53 6.16-4.53z" fill="#EA4335"></path></svg>
                Sign in with Google
            </a>
            <?php endif; ?>
        </div>
    </div>
</section>
