<?php
use NewSite\Database\DatabaseManager;
use NewSite\Util\AssetVersioning;

/**
 * Games Page Template
 * Games hub + individual game launcher views.
 */

$subRouteRaw = $segments[1] ?? '';
$subRoute = strtolower(trim((string)$subRouteRaw));
$subRoute = preg_replace('/[^a-z0-9\-]/', '', $subRoute);

$subRouteInternalSlug = \NewSite\Minigames\MinigameCatalog::resolveFromRoute((string)$subRoute);
$mazeRunnerInternalSlug = 'maze-runner';

try {
    $gamesDb = DatabaseManager::getReadConnection();
} catch (Throwable $e) {
    $gamesDb = null;
}

$miniGameCatalog = \NewSite\Minigames\MinigameCatalog::getCatalog();

// If a sub-route is present but not a known game slug, treat it as not found.
if ($subRoute !== '' && !isset($miniGameCatalog[$subRouteInternalSlug])):
    http_response_code(404);
    ?>
<div class="games-page">
    <div class="container">
        <h1>Page Not Found</h1>
        <p class="games-note">That mini-game does not exist.</p>
        <p><a class="btn btn-outline btn-sm" href="/games">&larr; Back to Mini-Games</a></p>
    </div>
</div>
<?php
        return;
endif;

if ($subRouteInternalSlug === $mazeRunnerInternalSlug):
    if ($gamesDb instanceof PDO && !\NewSite\Minigames\MinigameRepository::isEnabled($gamesDb, 'maze-runner', true)) {
        http_response_code(404);
        ?>
        <div class="games-page">
            <div class="container">
                <h1>Page Not Found</h1>
                <p class="games-note">This mini-game is currently unavailable.</p>
                <p><a class="btn btn-outline btn-sm" href="/games">&larr; Back to Mini-Games</a></p>
            </div>
        </div>
        <?php
        return;
    }

    $isFullscreen = (string)($_GET['fs'] ?? '0') === '1';
    $fullscreenClass = $isFullscreen ? ' is-fullscreen' : '';
    $viewerUserId = (int)($_SESSION['site_user_id'] ?? 0);
    $leaderboardRows = [];
    if ($gamesDb instanceof PDO) {
        $leaderboardRows = \NewSite\Minigames\MinigameRepository::getLeaderboard($gamesDb, 'maze-runner', 50, $viewerUserId);
    }
    ?>
<div class="games-page games-page--game<?php echo $fullscreenClass; ?>">
    <div class="container">
        <div class="games-header">
            <h1>Hardest Maze game V1</h1>
            <div class="games-controls">
                <a class="btn btn-outline btn-sm" href="/games">&larr; Back to Games</a>
                <button type="button" class="btn btn-outline btn-sm" data-game-fullscreen-toggle>Fullscreen</button>
            </div>
        </div>

        <div class="games-game-layout">
            <div class="games-game-main">
                <div class="games-embed" aria-label="Hardest Maze game V1">
                    <div class="maze-runner-stage<?php echo $fullscreenClass; ?>" id="maze-runner-stage" data-start-time-seconds="90" data-door-shuffle-seconds="10">
                        <div class="maze-runner-hud" aria-live="polite">
                            <div class="maze-pill">
                                <span class="maze-label">Level</span>
                                <span id="maze-level" class="maze-value">1</span>
                            </div>
                            <div class="maze-pill">
                                <span class="maze-label">Maze Size</span>
                                <span id="maze-size" class="maze-value">4 × 4</span>
                            </div>
                            <div class="maze-pill">
                                <span class="maze-label">Door Shuffle</span>
                                <span id="maze-door" class="maze-value">10.0s</span>
                            </div>
                            <div class="maze-pill">
                                <span class="maze-label">Time Left</span>
                                <span id="maze-time" class="maze-value">90s</span>
                            </div>
                            <div class="maze-pill">
                                <span class="maze-label">Max Level</span>
                                <span id="maze-score" class="maze-value">1</span>
                            </div>
                            <div class="maze-pill maze-pill-status">
                                <span class="maze-label">Status</span>
                                <span id="maze-status" class="maze-value">Find the exit!</span>
                            </div>
                        </div>

                        <p class="maze-runner-guide">Move with <strong>WASD</strong> or <strong>Arrow keys</strong>. Hidden doors reshuffle every 10 seconds. Reach the exit to level up and gain time equal to <strong>current level × 30</strong>.</p>

                        <div class="maze-runner-canvas-wrap" id="maze-runner-canvas-wrap">
                            <canvas id="maze-runner-canvas" width="860" height="860" aria-label="Hardest Maze game V1 board"></canvas>
                        </div>

                        <div class="maze-runner-actions">
                            <button id="maze-restart-btn" type="button" class="btn btn-outline btn-sm">Start New Run</button>
                        </div>

                        <noscript>
                            <div class="games-note">This game requires JavaScript enabled in your browser.</div>
                        </noscript>
                    </div>
                </div>
            </div>

            <aside
                id="mazeRunnerLeaderboard"
                class="maze-leaderboard"
                data-game-leaderboard
                data-game-slug="maze-runner"
                data-user-logged-in="<?php echo $viewerUserId > 0 ? '1' : '0'; ?>"
                aria-label="Hardest Maze game V1 leaderboard"
            >
                <h2>Top 50 Levels</h2>
                <div id="mazeLeaderboardStatus" class="maze-leaderboard-status">Hardest Maze game V1 Highest Level Leaderboard</div>
                <ol id="mazeLeaderboardList" class="maze-leaderboard-list">
                    <?php if (empty($leaderboardRows)): ?>
                        <li class="maze-leaderboard-empty">No level records yet. Be the first.</li>
                    <?php else: ?>
                        <?php foreach ($leaderboardRows as $entry): ?>
                            <?php
                                $entryRank = (int)($entry['rank'] ?? 0);
                            $entryName = trim((string)($entry['name'] ?? 'Player'));
                            $entryScore = (int)floor((float)($entry['score'] ?? 0));
                            if ($entryScore < 0) {
                                $entryScore = 0;
                            }
                            $entryScoreText = (string)$entryScore;
                            $entryCurrent = !empty($entry['is_current_user']);
                            $entryProfileUrl = trim((string)($entry['profile_url'] ?? ''));
                            ?>
                            <li class="maze-leaderboard-item<?php echo $entryCurrent ? ' is-current' : ''; ?>">
                                <span class="maze-leaderboard-rank">#<?php echo (int)$entryRank; ?></span>
                                <?php if ($entryProfileUrl !== ''): ?>
                                    <a class="maze-leaderboard-name" href="<?php echo e($entryProfileUrl); ?>"><?php echo e($entryName); ?></a>
                                <?php else: ?>
                                    <span class="maze-leaderboard-name"><?php echo e($entryName); ?></span>
                                <?php endif; ?>
                                <span class="maze-leaderboard-score"><?php echo e($entryScoreText); ?></span>
                            </li>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </ol>
                <?php if ($viewerUserId <= 0): ?>
                    <p class="maze-leaderboard-login">Log in to submit your highest level.</p>
                <?php endif; ?>
            </aside>
        </div>
    </div>
</div>

<script type="module" nonce="<?php echo e(getCspNonce()); ?>" src="<?php echo AssetVersioning::url('/assets/js/games/maze-runner.js'); ?>"></script>
<?php
    return;
endif;

if ($subRoute === 'basketball-launch'):
    if ($gamesDb instanceof PDO && !\NewSite\Minigames\MinigameRepository::isEnabled($gamesDb, 'basketball-launch', true)) {
        http_response_code(404);
        ?>
        <div class="games-page">
            <div class="container">
                <h1>Page Not Found</h1>
                <p class="games-note">This mini-game is currently unavailable.</p>
                <p><a class="btn btn-outline btn-sm" href="/games">&larr; Back to Mini-Games</a></p>
            </div>
        </div>
        <?php
        return;
    }

    $isFullscreen = (string)($_GET['fs'] ?? '0') === '1';
    $fullscreenClass = $isFullscreen ? ' is-fullscreen' : '';
    $viewerUserId = (int)($_SESSION['site_user_id'] ?? 0);
    $leaderboardRows = [];
    $basketballSettings = \NewSite\Minigames\MinigameCatalog::getGameplayDefaults('basketball-launch');
    if ($gamesDb instanceof PDO) {
        $leaderboardRows = \NewSite\Minigames\MinigameRepository::getLeaderboard($gamesDb, 'basketball-launch', 50, $viewerUserId);
        $basketballSettings = \NewSite\Minigames\MinigameRepository::getGameplaySettings($gamesDb, 'basketball-launch');
    }
    if (empty($basketballSettings)) {
        $basketballSettings = \NewSite\Minigames\MinigameCatalog::getGameplayDefaults('basketball-launch');
    }

    $guideTimeBonus = (int)($basketballSettings['time_bonus_on_make'] ?? 5);
    $guideTimePenalty = (int)($basketballSettings['time_penalty_on_miss'] ?? 10);
    // Render the game inside the normal site layout.
    // IMPORTANT: Site-wide CSP blocks inline <style>/<script>, so the game code
    // must be loaded from external files with the CSP nonce.
    ?>
<div class="games-page games-page--game<?php echo $fullscreenClass; ?>">
    <div class="container">
        <div class="games-header">
            <h1>Basketball Launch</h1>
            <div class="games-controls">
                <a class="btn btn-outline btn-sm" href="/games">&larr; Back to Games</a>
                <button type="button" class="btn btn-outline btn-sm" data-game-fullscreen-toggle>Fullscreen</button>
            </div>
        </div>

        <div class="games-game-layout">
            <div class="games-game-main">
                <div class="games-embed" aria-label="Basketball Launch game">
                    <div
                        class="basketball-launch-stage<?php echo $fullscreenClass; ?>"
                        id="basketball-launch-stage"
                        data-multiplier-step="<?php echo e((string)($basketballSettings['multiplier_step'] ?? 1.12)); ?>"
                        data-start-time-seconds="<?php echo e((string)($basketballSettings['start_time_seconds'] ?? 30)); ?>"
                        data-time-bonus-on-make="<?php echo e((string)($basketballSettings['time_bonus_on_make'] ?? 5)); ?>"
                        data-time-penalty-on-miss="<?php echo e((string)($basketballSettings['time_penalty_on_miss'] ?? 10)); ?>"
                        data-max-time-seconds="<?php echo e((string)($basketballSettings['max_time_seconds'] ?? 99)); ?>"
                    >
                <div class="basketball-launch-toolbar" aria-label="Game controls">
                    <a class="btn btn-outline btn-sm" href="/games">&larr; Back</a>
                    <button type="button" class="btn btn-primary btn-sm" data-game-fullscreen-toggle>Exit Fullscreen</button>
                </div>

                <div id="hud">
                    <h1>Basketball Launch</h1>
                    <div id="scoreline"><strong id="score">0</strong> points</div>
                    <div id="statsline">Multiplier: <strong id="multiplier">x1.00</strong> | Time: <strong id="timer">30</strong>s</div>
                    <p id="guide">Left click + hold a ball to grab and aim.<br>Each make increases your point multiplier and adds <strong>+<?php echo e((string)$guideTimeBonus); ?>s</strong>.<br>Each miss removes <strong>-<?php echo e((string)$guideTimePenalty); ?>s</strong>. When time hits 0, it’s game over.</p>
                    <div id="status"></div>
                </div>

                <div id="gameOver">
                    <div id="gameOverCard">
                        <h2>Game Over</h2>
                        <p>Max points: <strong id="maxPoints">0</strong></p>
                        <p>Best: <strong id="bestPoints">0</strong></p>
                        <button id="restartBtn" type="button">Start New Game</button>
                    </div>
                </div>

                <div id="crowdTips"></div>

                <div id="basketball-launch-canvas" aria-hidden="true"></div>

                <noscript>
                    <div class="games-note">This game requires JavaScript enabled in your browser.</div>
                </noscript>
                    </div>
                </div>

                <?php if (!$isFullscreen): ?>
                <p class="games-note">Tip: For true browser fullscreen you can also press <strong>F11</strong>.</p>
                <?php endif; ?>
            </div>

            <aside
                id="basketballLeaderboard"
                class="basketball-leaderboard"
                data-game-leaderboard
                data-game-slug="basketball-launch"
                data-user-logged-in="<?php echo $viewerUserId > 0 ? '1' : '0'; ?>"
                aria-label="Basketball Launch leaderboard"
            >
                <h2>Top 50</h2>
                <div id="basketballLeaderboardStatus" class="basketball-leaderboard-status">Basketball Launch Leaderboard</div>
                <ol id="basketballLeaderboardList" class="basketball-leaderboard-list">
                    <?php if (empty($leaderboardRows)): ?>
                        <li class="basketball-leaderboard-empty">No scores yet. Be the first.</li>
                    <?php else: ?>
                        <?php foreach ($leaderboardRows as $entry): ?>
                            <?php
                                $entryRank = (int)($entry['rank'] ?? 0);
                            $entryName = trim((string)($entry['name'] ?? 'Player'));
                            $entryScore = (float)($entry['score'] ?? 0);
                            $entryScoreText = rtrim(rtrim(number_format($entryScore, 2, '.', ''), '0'), '.');
                            if ($entryScoreText === '') {
                                $entryScoreText = '0';
                            }
                            $entryCurrent = !empty($entry['is_current_user']);
                            $entryProfileUrl = trim((string)($entry['profile_url'] ?? ''));
                            ?>
                            <li class="basketball-leaderboard-item<?php echo $entryCurrent ? ' is-current' : ''; ?>">
                                <span class="basketball-leaderboard-rank">#<?php echo (int)$entryRank; ?></span>
                                <?php if ($entryProfileUrl !== ''): ?>
                                    <a class="basketball-leaderboard-name" href="<?php echo e($entryProfileUrl); ?>"><?php echo e($entryName); ?></a>
                                <?php else: ?>
                                    <span class="basketball-leaderboard-name"><?php echo e($entryName); ?></span>
                                <?php endif; ?>
                                <span class="basketball-leaderboard-score"><?php echo e($entryScoreText); ?></span>
                            </li>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </ol>
                <?php if ($viewerUserId <= 0): ?>
                    <p class="basketball-leaderboard-login">Log in to submit your score.</p>
                <?php endif; ?>
            </aside>
        </div>
    </div>
</div>

<script type="module" nonce="<?php echo e(getCspNonce()); ?>" src="<?php echo AssetVersioning::url('/assets/js/games/basketball-launch.js'); ?>"></script>
<?php
    return;
endif;
?>

<div class="games-page">
    <div class="container">
        <h1>Mini-Games</h1>
        <?php
        $displaySiteName = trim((string)($theme['site_name'] ?? ''));
if ($displaySiteName === '') {
    // Generic-site placeholder used across legal pages/settings.
    $displaySiteName = '{SITE_NAME}';
}
?>
        <p class="games-subtitle"><?php echo e($displaySiteName . ' Mini-Games'); ?></p>

        <?php
$enabledGames = [];
if ($gamesDb instanceof PDO) {
    $enabledGames = \NewSite\Minigames\MinigameRepository::getEnabled($gamesDb);
} else {
    // DB not available (or during setup): show catalog defaults.
    $enabledGames = array_values($miniGameCatalog);
    usort($enabledGames, static function (array $a, array $b): int {
        $sortA = (int)($a['sort'] ?? 0);
        $sortB = (int)($b['sort'] ?? 0);
        if ($sortA === $sortB) {
            return strcmp((string)($a['title'] ?? ''), (string)($b['title'] ?? ''));
        }
        return $sortA <=> $sortB;
    });
}
?>

        <?php if (empty($enabledGames)): ?>
            <p class="games-note">No mini-games are available right now.</p>
        <?php else: ?>
            <div class="games-grid">
                <?php foreach ($enabledGames as $game): ?>
                    <?php
            if (!is_array($game)) {
                continue;
            }
                    $slug = isset($game['slug']) ? (string)$game['slug'] : '';
                    $slug = preg_replace('/[^a-z0-9\-]/', '', strtolower(trim($slug)));
                    if ($slug === '') {
                        continue;
                    }
                    $publicSlug = \NewSite\Minigames\MinigameCatalog::getPublicSlug($slug);
                    $title = (string)($game['title'] ?? $slug);
                    $desc = (string)($game['description'] ?? '');
                    $icon = (string)($game['icon'] ?? '🎮');
                    ?>
                    <a class="game-card" href="/games/<?php echo e($publicSlug); ?>" aria-label="Play <?php echo e($title); ?>">
                        <div class="game-card-icon" aria-hidden="true"><?php echo e($icon); ?></div>
                        <div class="game-card-title"><?php echo e($title); ?></div>
                        <?php if ($desc !== ''): ?>
                            <div class="game-card-desc"><?php echo e($desc); ?></div>
                        <?php endif; ?>
                    </a>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>
