<?php
use NewSite\Auth\SiteAuth;
use NewSite\Database\DatabaseManager;
use NewSite\Forum\ForumHelper;
use NewSite\Settings\SettingsService;

$forumEnabled = SettingsService::get('forum_enabled', '0') === '1';
$forumTitle = trim((string)SettingsService::get('forum_title', 'Forum'));
if ($forumTitle === '') {
    $forumTitle = 'Forum';
}
$forumDescription = trim((string)SettingsService::get('forum_description', 'Join the conversation and explore the latest discussions.'));
if ($forumDescription === '') {
    $forumDescription = 'Join the conversation and explore the latest discussions.';
}
if (!$forumEnabled) {
    http_response_code(404);
    ?>
    <div class="page-content">
        <div class="container">
            <h1>Page Not Found</h1>
            <p>The forum is currently disabled.</p>
        </div>
    </div>
    <?php
    return;
}

$db = $db ?? DatabaseManager::getWriteConnection();
$defaultAvatar = '/assets/images/default-avatar.svg';

$selectedCategoryId = 0;
$selectedSubcategoryId = 0;

// Resolve category/subcategory from slug-based path segments
// (Legacy ?category=X / ?subcategory=X are 301-redirected in index.php before reaching here)
if (isset($segments[1]) && $segments[1] === 'subcategory' && isset($segments[2])) {
    $subSlug = trim((string)$segments[2]);
    if ($subSlug !== '') {
        $subSlugStmt = $db->prepare('SELECT id FROM forum_subcategories WHERE slug = ? AND is_active = 1 LIMIT 1');
        $subSlugStmt->execute([$subSlug]);
        $selectedSubcategoryId = (int)$subSlugStmt->fetchColumn();
    }
}

if ($selectedCategoryId <= 0 && isset($segments[1]) && $segments[1] === 'category' && isset($segments[2])) {
    $catSlug = trim((string)$segments[2]);
    if ($catSlug !== '') {
        $catSlugStmt = $db->prepare('SELECT id FROM forum_categories WHERE slug = ? AND is_active = 1 LIMIT 1');
        $catSlugStmt->execute([$catSlug]);
        $selectedCategoryId = (int)$catSlugStmt->fetchColumn();
    }
}

$categories = $db->query("SELECT id, name, slug FROM forum_categories WHERE is_active = 1 ORDER BY sort_order ASC, name ASC")->fetchAll();
$subcategories = $db->query("SELECT id, category_id, parent_id, name, slug, description, depth FROM forum_subcategories WHERE is_active = 1 ORDER BY sort_order ASC, name ASC")->fetchAll();
$isUserLoggedIn = isset($isUserLoggedIn) ? (bool)$isUserLoggedIn : SiteAuth::isLoggedIn();
$canPost = $isUserLoggedIn && !empty($subcategories);

// Build tree structure for nested subcategories
$subcategoriesByCategory = [];
$subcategoryCategoryMap = [];
$subcategorySlugMap = [];
foreach ($subcategories as $sub) {
    $catId = (int)$sub['category_id'];
    if (!isset($subcategoriesByCategory[$catId])) {
        $subcategoriesByCategory[$catId] = [];
    }
    $subcategoriesByCategory[$catId][] = $sub;
    $subcategoryCategoryMap[(int)$sub['id']] = $catId;
    $subcategorySlugMap[(int)$sub['id']] = trim((string)($sub['slug'] ?? ''));
}

// Convert flat subcategories to tree structure per category
foreach ($subcategoriesByCategory as $catId => $catSubs) {
    $subcategoriesByCategory[$catId] = ForumHelper::buildSubcategoryTree($catSubs, null);
}

if ($selectedSubcategoryId > 0 && isset($subcategoryCategoryMap[$selectedSubcategoryId])) {
    $selectedCategoryId = $subcategoryCategoryMap[$selectedSubcategoryId];
}

if ($selectedCategoryId <= 0 && !empty($categories)) {
    $selectedCategoryId = (int)$categories[0]['id'];
}

if ($selectedSubcategoryId > 0 && (!isset($subcategoryCategoryMap[$selectedSubcategoryId]) || $subcategoryCategoryMap[$selectedSubcategoryId] !== $selectedCategoryId)) {
    $selectedSubcategoryId = 0;
}

$subcategoriesPayload = array_map(static function ($sub) {
    return [
        'id' => (int)$sub['id'],
        'category_id' => (int)$sub['category_id'],
        'parent_id' => isset($sub['parent_id']) && $sub['parent_id'] !== null ? (int)$sub['parent_id'] : null,
        'name' => (string)$sub['name']
    ];
}, $subcategories);
$subcategoriesJson = htmlspecialchars(json_encode($subcategoriesPayload, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_AMP | JSON_HEX_QUOT), ENT_QUOTES, 'UTF-8');

// Get recent discussions from entire forum for sidebar
$recentPosts = [];
try {
    $recentPostsSql = "
        SELECT p.id, p.title, p.slug, p.created_at, COALESCE(p.views, 0) AS view_count,
               s.id AS sub_id, s.name AS sub_name,
               c.id AS cat_id, c.name AS cat_name,
               u.display_name, u.nickname,
               COALESCE(comments.comment_count, 0) AS comment_count
        FROM forum_posts p
        JOIN forum_subcategories s ON s.id = p.subcategory_id AND s.is_active = 1
        JOIN forum_categories c ON c.id = s.category_id AND c.is_active = 1
        LEFT JOIN site_users u ON u.id = p.user_id
        LEFT JOIN (
            SELECT post_id, COUNT(*) AS comment_count
            FROM forum_comments
            WHERE status = 'approved'
            GROUP BY post_id
        ) comments ON comments.post_id = p.id
        WHERE p.status = 'approved'
        ORDER BY p.created_at DESC
        LIMIT 15
    ";
    $recentPosts = $db->query($recentPostsSql)->fetchAll();
} catch (Exception $e) {
    // Continue with empty recent posts if query fails
    $recentPosts = [];
}

$perPage = 20;
$pageNum = max(1, (int)($_GET['page'] ?? 1));

$where = "p.status = 'approved'";
$params = [];
if ($selectedSubcategoryId > 0) {
    $where .= " AND p.subcategory_id = ?";
    $params[] = $selectedSubcategoryId;
} elseif ($selectedCategoryId > 0) {
    $where .= " AND s.category_id = ?";
    $params[] = $selectedCategoryId;
}

$countSql = "
    SELECT COUNT(*)
    FROM forum_posts p
    JOIN forum_subcategories s ON s.id = p.subcategory_id AND s.is_active = 1
    JOIN forum_categories c ON c.id = s.category_id AND c.is_active = 1
    WHERE {$where}
";
$stmt = $db->prepare($countSql);
$stmt->execute($params);
$totalPosts = (int)$stmt->fetchColumn();

$totalPages = max(1, (int)ceil($totalPosts / $perPage));
$pageNum = min($pageNum, $totalPages);
$offset = max(0, ($pageNum - 1) * $perPage);

$posts = [];
if ($totalPosts > 0) {
    $postsSql = "
        SELECT p.id, p.title, p.slug, p.body, p.created_at, COALESCE(p.views, 0) AS view_count,
               s.id AS sub_id, s.name AS sub_name,
               c.id AS cat_id, c.name AS cat_name,
               u.display_name, u.nickname, u.profile_photo,
               COALESCE(comments.comment_count, 0) AS comment_count
        FROM forum_posts p
        JOIN forum_subcategories s ON s.id = p.subcategory_id AND s.is_active = 1
        JOIN forum_categories c ON c.id = s.category_id AND c.is_active = 1
        LEFT JOIN site_users u ON u.id = p.user_id
        LEFT JOIN (
            SELECT post_id, COUNT(*) AS comment_count
            FROM forum_comments
            WHERE status = 'approved'
            GROUP BY post_id
        ) comments ON comments.post_id = p.id
        WHERE {$where}
        ORDER BY p.created_at DESC
        LIMIT ? OFFSET ?
    ";
    $stmt = $db->prepare($postsSql);
    $stmt->execute(array_merge($params, [$perPage, $offset]));
    $posts = $stmt->fetchAll();
}

?>

<div class="forum-page" data-forum-root data-forum-subcategories="<?php echo $subcategoriesJson; ?>">
    <div class="container">
        <div class="forum-header">
            <div>
                <h1><?php echo e($forumTitle); ?></h1>
                <p><?php echo e($forumDescription); ?></p>
            </div>
            <div class="forum-header-actions">
                <?php if ($canPost): ?>
                    <button type="button" class="btn btn-primary btn-sm" data-forum-open>Create Post</button>
                <?php elseif (!$isUserLoggedIn): ?>
                    <a class="btn btn-outline btn-sm" href="/login">Login to post</a>
                <?php else: ?>
                    <button type="button" class="btn btn-outline btn-sm" disabled>No categories yet</button>
                <?php endif; ?>
            </div>
        </div>
        <div class="forum-alert is-hidden" data-forum-alert></div>

        <div class="forum-grid">
            <!-- Left Sidebar: Activity Stats -->
            <aside class="forum-sidebar forum-sidebar-left" aria-label="Forum statistics">
                <div class="forum-card forum-activity-card">
                    <div class="forum-card-header">
                        <h3>📊 Forum Stats</h3>
                    </div>
                    <div class="forum-card-body">
                        <?php
                        $totalThreads = $db->query("SELECT COUNT(*) FROM forum_posts WHERE status = 'approved'")->fetchColumn();
$totalReplies = $db->query("SELECT COUNT(*) FROM forum_comments WHERE status = 'approved'")->fetchColumn();
$onlineUsers = $db->query("SELECT COUNT(*) FROM site_users WHERE last_activity > " . (time() - 900))->fetchColumn();
$totalMembers = $db->query("SELECT COUNT(*) FROM site_users")->fetchColumn();
$latestMember = $db->query("SELECT nickname, display_name FROM site_users ORDER BY id DESC LIMIT 1")->fetch();
$latestMemberName = trim((string)($latestMember['nickname'] ?? $latestMember['display_name'] ?? 'Unknown'));
?>
                        <div class="forum-stats">
                            <div class="forum-stat-item">
                                <div class="forum-stat-value"><?php echo number_format($totalThreads); ?></div>
                                <div class="forum-stat-label">Threads</div>
                            </div>
                            <div class="forum-stat-item">
                                <div class="forum-stat-value"><?php echo number_format($totalReplies); ?></div>
                                <div class="forum-stat-label">Replies</div>
                            </div>
                            <div class="forum-stat-item">
                                <div class="forum-stat-value"><?php echo number_format($totalMembers); ?></div>
                                <div class="forum-stat-label">Members</div>
                            </div>
                            <div class="forum-stat-item">
                                <div class="forum-stat-value"><?php echo $onlineUsers; ?></div>
                                <div class="forum-stat-label">Online</div>
                            </div>
                        </div>
                        <div class="forum-activity-footer">
                            <div class="forum-activity-item">
                                <span class="forum-activity-label">📝 Latest Member:</span>
                                <span class="forum-activity-value"><?php echo e($latestMemberName); ?></span>
                            </div>
                        </div>
                    </div>
                </div>
            </aside>

            <!-- Main Content: Forum Categories OR Thread List -->
            <section class="forum-main">
                <?php if ($selectedSubcategoryId > 0): ?>
                    <!-- Thread List for Selected Subcategory -->
                    <div class="forum-card">
                        <div class="forum-card-header">
                            <h3>💬 Threads</h3>
                            <a href="/forum" class="btn-back-forum">← Back to Forum</a>
                        </div>
                        <div class="forum-card-body">
                            <?php if (empty($posts)): ?>
                                <div class="forum-empty" data-forum-empty>No posts yet.</div>
                            <?php else: ?>
                                <div class="forum-posts" data-forum-posts>
                                    <?php foreach ($posts as $post):
                                        $profilePhoto = trim((string)($post['profile_photo'] ?? ''));
                                        $avatarUrl = $profilePhoto !== '' ? toPublicFilePath($profilePhoto) : $defaultAvatar;
                                        $displayName = trim((string)($post['nickname'] ?? ''));
                                        if ($displayName === '') {
                                            $displayName = trim((string)($post['display_name'] ?? ''));
                                        }
                                        if ($displayName === '') {
                                            $displayName = 'User';
                                        }
                                        $createdAt = !empty($post['created_at']) ? date('M j, Y', strtotime($post['created_at'])) : '';
                                        $createdTime = !empty($post['created_at']) ? date('g:i A', strtotime($post['created_at'])) : '';
                                        $commentCount = (int)($post['comment_count'] ?? 0);
                                        $viewCount = (int)($post['view_count'] ?? 0);
                                        $postUrl = FORUM_POST_ROUTE . rawurlencode(trim((string)($post['slug'] ?? '')));
                                        if (trim((string)($post['slug'] ?? '')) === '') {
                                            $postUrl = FORUM_POST_ROUTE . (int)$post['id'];
                                        }
                                        ?>
                                        <a href="<?php echo e($postUrl); ?>" class="forum-post">
                                            <div class="forum-post-icon"></div>
                                            <div class="forum-post-main">
                                                <div class="forum-post-title"><?php echo e($post['title']); ?></div>
                                                <div class="forum-post-meta">
                                                    <span class="forum-post-author">
                                                        <img src="<?php echo e($avatarUrl); ?>" alt="" class="forum-post-avatar">
                                                        <span><?php echo e($displayName); ?></span>
                                                    </span>
                                                    <?php if ($createdAt !== ''): ?>
                                                        <span>• <?php echo e($createdAt); ?></span>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                            <div class="forum-post-stat">
                                                <div class="forum-post-stat-value"><?php echo $commentCount; ?></div>
                                                <div class="forum-post-stat-label">Replies</div>
                                            </div>
                                            <div class="forum-post-stat">
                                                <div class="forum-post-stat-value"><?php echo $viewCount; ?></div>
                                                <div class="forum-post-stat-label">Views</div>
                                            </div>
                                            <div class="forum-post-last">
                                                <div class="forum-post-last-user"><?php echo e($displayName); ?></div>
                                                <div class="forum-post-last-time"><?php echo e($createdAt); ?><?php if ($createdTime): ?> at <?php echo e($createdTime); ?><?php endif; ?></div>
                                            </div>
                                        </a>
                                    <?php endforeach; ?>
                                </div>

                                <?php if ($totalPages > 1):
                                    // Build pagination base URL using subcategory slug
                                    $paginationSubSlug = $selectedSubcategoryId > 0 ? ($subcategorySlugMap[$selectedSubcategoryId] ?? '') : '';
                                    $paginationBase = $paginationSubSlug !== '' ? FORUM_SUBCATEGORY_ROUTE . rawurlencode($paginationSubSlug) : '/forum';
                                ?>
                                    <div class="forum-pagination">
                                        <?php if ($pageNum > 1): ?>
                                            <a class="btn btn-outline btn-sm" href="<?php echo e($paginationBase); ?>?page=<?php echo $pageNum - 1; ?>">Prev</a>
                                        <?php endif; ?>
                                        <span class="forum-page-label">Page <?php echo $pageNum; ?> of <?php echo $totalPages; ?></span>
                                        <?php if ($pageNum < $totalPages): ?>
                                            <a class="btn btn-outline btn-sm" href="<?php echo e($paginationBase); ?>?page=<?php echo $pageNum + 1; ?>">Next</a>
                                        <?php endif; ?>
                                    </div>
                                <?php endif; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php elseif (empty($categories)): ?>
                    <div class="forum-card">
                        <div class="forum-card-body">
                            <div class="forum-empty">No categories yet.</div>
                        </div>
                    </div>
                <?php else: ?>
                    <!-- Category Sections -->
                    <?php foreach ($categories as $category):
                        $catId = (int)$category['id'];
                        $catSubs = $subcategoriesByCategory[$catId] ?? [];
                        ?>
                        <div class="forum-category-section">
                            <div class="forum-category-header">
                                <h2>📂 <?php echo e($category['name']); ?></h2>
                            </div>
                            <?php if (empty($catSubs)): ?>
                                <div class="forum-empty">No subcategories in this section.</div>
                            <?php else: ?>
                                <div class="forum-subcategories">
                                    <?php
                                        /**
                                         * Fetch thread count, reply count, and latest post for a subcategory.
                                         *
                                         * @return array{threads: int, replies: int, latest_post: array|false}
                                         */
                                        function fetchForumSubcategoryStats(PDO $db, int $subId): array
                                        {
                                            $stmt = $db->prepare(
                                                "SELECT COUNT(*) FROM forum_posts WHERE subcategory_id = ? AND status = 'approved'"
                                            );
                                            $stmt->execute([$subId]);
                                            $threads = (int)$stmt->fetchColumn();

                                            $stmt = $db->prepare(
                                                "SELECT COUNT(*) FROM forum_comments c
                                                 JOIN forum_posts p ON p.id = c.post_id
                                                 WHERE p.subcategory_id = ? AND c.status = 'approved'"
                                            );
                                            $stmt->execute([$subId]);
                                            $replies = (int)$stmt->fetchColumn();

                                            $stmt = $db->prepare(
                                                "SELECT p.id, p.title, p.created_at, u.display_name, u.nickname
                                                 FROM forum_posts p
                                                 LEFT JOIN site_users u ON u.id = p.user_id
                                                 WHERE p.subcategory_id = ? AND p.status = 'approved'
                                                 ORDER BY p.created_at DESC
                                                 LIMIT 1"
                                            );
                                            $stmt->execute([$subId]);

                                            return [
                                                'threads' => $threads,
                                                'replies' => $replies,
                                                'latest_post' => $stmt->fetch() ?: null,
                                            ];
                                        }

                                /**
                                 * Recursively render forum subcategory rows with indentation.
                                 */
                                /**
                                                                         * Render one subcategory row (link, stats, latest post).
                                                                         */
                                function renderForumSubcategoryRow(array $sub, array $stats, int $depth): void
                                {
                                    $subId = (int)$sub['id'];
                                    $depthClass = 'forum-subcategory-depth-' . min(10, $depth);
                                    $icon = $depth > 0 ? '└─ 💬' : '💬';
                                    $latestPost = $stats['latest_post'];
                                    $latestAuthor = '';
                                    $latestDate = '';
                                    if ($latestPost) {
                                        $latestAuthor = trim((string)($latestPost['nickname'] ?? ''));
                                        if ($latestAuthor === '') {
                                            $latestAuthor = trim((string)($latestPost['display_name'] ?? ''));
                                        }
                                        if ($latestAuthor === '') {
                                            $latestAuthor = 'User';
                                        }
                                        $latestDate = date('M j, Y', strtotime($latestPost['created_at']));
                                    }
                                    $subSlugValue = trim((string)($sub['slug'] ?? ''));
                                    $subUrl = $subSlugValue !== '' ? FORUM_SUBCATEGORY_ROUTE . rawurlencode($subSlugValue) : '/forum' . ForumHelper::buildQuery(['subcategory' => $subId]);
                                    ?>
                                        <a href="<?php echo e($subUrl); ?>" class="forum-subcategory-row <?php echo $depthClass; ?>">
                                            <div class="forum-subcategory-icon"><?php echo $icon; ?></div>
                                            <div class="forum-subcategory-info">
                                                <div class="forum-subcategory-name">
                                                    <?php echo e($sub['name']); ?>
                                                </div>
                                                <?php if (!empty($sub['description'])): ?>
                                                    <div class="forum-subcategory-desc"><?php echo e($sub['description']); ?></div>
                                                <?php endif; ?>
                                            </div>
                                            <div class="forum-subcategory-stats">
                                                <div class="forum-subcategory-stat">
                                                    <span class="stat-value"><?php echo number_format($stats['threads']); ?></span>
                                                    <span class="stat-label">Threads</span>
                                                </div>
                                                <div class="forum-subcategory-stat">
                                                    <span class="stat-value"><?php echo number_format($stats['replies']); ?></span>
                                                    <span class="stat-label">Replies</span>
                                                </div>
                                            </div>
                                            <div class="forum-subcategory-latest">
                                                <?php if ($latestPost): ?>
                                                    <span class="forum-latest-post">
                                                        <div class="forum-latest-title"><?php echo e($latestPost['title']); ?></div>
                                                        <div class="forum-latest-meta">
                                                            <span>By <?php echo e($latestAuthor); ?></span>
                                                            <span>• <?php echo e($latestDate); ?></span>
                                                        </div>
                                                    </span>
                                                <?php else: ?>
                                                    <div class="forum-latest-empty">No posts yet</div>
                                                <?php endif; ?>
                                            </div>
                                        </a>
                                    <?php
                                }

                        /**
                         * Recursively render forum subcategory rows with indentation.
                         */
                        function renderForumSubcategories(array $subcategories, PDO $db, int $depth = 0): void
                        {
                            foreach ($subcategories as $sub) {
                                $stats = fetchForumSubcategoryStats($db, (int)$sub['id']);
                                renderForumSubcategoryRow($sub, $stats, $depth);

                                if (!empty($sub['children'])) {
                                    renderForumSubcategories($sub['children'], $db, $depth + 1);
                                }
                            }
                        }

                        // Render the tree
                        renderForumSubcategories($catSubs, $db, 0);
                        ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </section>

            <!-- Right Sidebar: Recent Discussions -->
            <aside class="forum-sidebar forum-sidebar-right" aria-label="Recent discussions">
                <div class="forum-card">
                    <div class="forum-card-header">
                        <h3>💬 Recent Discussions</h3>
                        <?php if ($canPost): ?>
                            <button type="button" class="btn-new-thread" data-forum-open title="New Thread">+</button>
                        <?php endif; ?>
                    </div>
                    <div class="forum-card-body">
                        <?php if (empty($recentPosts)): ?>
                            <div class="forum-empty" data-forum-empty>No posts yet.</div>
                        <?php else: ?>
                            <div class="forum-recent-posts">
                                <?php
                                foreach ($recentPosts as $post):
                                    $displayName = trim((string)($post['nickname'] ?? ''));
                                    if ($displayName === '') {
                                        $displayName = trim((string)($post['display_name'] ?? ''));
                                    }
                                    if ($displayName === '') {
                                        $displayName = 'User';
                                    }
                                    $createdAt = !empty($post['created_at']) ? date('M j', strtotime($post['created_at'])) : '';
                                    $commentCount = (int)($post['comment_count'] ?? 0);
                                    $recentPostSlug = trim((string)($post['slug'] ?? ''));
                                    $recentPostUrl = $recentPostSlug !== '' ? FORUM_POST_ROUTE . rawurlencode($recentPostSlug) : FORUM_POST_ROUTE . (int)$post['id'];
                                    ?>
                                    <a href="<?php echo e($recentPostUrl); ?>" class="forum-recent-item">
                                        <div class="forum-recent-title"><?php echo e($post['title']); ?></div>
                                        <div class="forum-recent-meta">
                                            <span class="forum-recent-author"><?php echo e($displayName); ?></span>
                                            <span class="forum-recent-cat"><?php echo e($post['cat_name']); ?></span>
                                        </div>
                                        <div class="forum-recent-stats">
                                            <span>💬 <?php echo $commentCount; ?></span>
                                            <span>📅 <?php echo e($createdAt); ?></span>
                                        </div>
                                    </a>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </aside>
        </div>
    </div>
</div>

<?php if ($canPost): ?>
<div class="forum-modal" data-forum-modal aria-hidden="true">
    <div class="forum-modal-backdrop" data-forum-close></div>
    <dialog class="forum-modal-content" open aria-labelledby="forum-modal-title">
        <div class="forum-modal-header">
            <div>
                <h3 id="forum-modal-title">Create Post</h3>
                <p>Share a new topic with the community.</p>
            </div>
            <button type="button" class="forum-modal-close" data-forum-close aria-label="Close">
                <svg viewBox="0 0 24 24" aria-hidden="true">
                    <path d="M6 6l12 12"></path>
                    <path d="M18 6l-12 12"></path>
                </svg>
            </button>
        </div>
        <div class="forum-modal-body">
            <form class="forum-form" data-forum-form enctype="multipart/form-data">
                <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                <input type="hidden" name="action" value="create_post">
                <div class="forum-form-grid">
                    <div class="forum-field">
                        <label for="forum-category">Category</label>
                        <select id="forum-category" name="category_id" data-forum-category required>
                            <?php foreach ($categories as $category): ?>
                                <option value="<?php echo (int)$category['id']; ?>" <?php echo (int)$category['id'] === $selectedCategoryId ? 'selected' : ''; ?>>
                                    <?php echo e($category['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="forum-field">
                        <label for="forum-subcategory">Subcategory</label>
                        <select id="forum-subcategory" name="subcategory_id" data-forum-subcategory required>
                            <?php
                            // Recursive function to render subcategory options with indentation
                            function renderForumSubcategoryOptions(array $subcategories, int $selectedId = 0, int $depth = 0): string
                            {
                                $html = '';
                                foreach ($subcategories as $sub) {
                                    $subId = (int)$sub['id'];
                                    $indent = str_repeat('&nbsp;&nbsp;&nbsp;&nbsp;', $depth);
                                    $prefix = $depth > 0 ? '└─ ' : '';
                                    $selected = $subId === $selectedId ? ' selected' : '';
                                    $html .= '<option value="' . $subId . '"' . $selected . '>' . $indent . $prefix . e($sub['name']) . '</option>';
                                    if (!empty($sub['children'])) {
                                        $html .= renderForumSubcategoryOptions($sub['children'], $selectedId, $depth + 1);
                                    }
                                }
                                return $html;
                            }

    $initialSubs = $selectedCategoryId > 0 ? ($subcategoriesByCategory[$selectedCategoryId] ?? []) : [];
    echo renderForumSubcategoryOptions($initialSubs, $selectedSubcategoryId, 0);
    ?>
                        </select>
                    </div>
                </div>
                <div class="forum-field">
                    <label for="forum-title">Title</label>
                    <input id="forum-title" name="title" type="text" maxlength="120" required placeholder="Write a clear, short title">
                </div>
                <div class="forum-field">
                    <label for="forum-body">Post</label>
                    <textarea id="forum-body" name="body" rows="5" maxlength="5000" required placeholder="Share the details of your post"></textarea>
                </div>
                <div class="forum-field">
                    <label for="forum-images">Images (optional)</label>
                    <input id="forum-images" name="images[]" type="file" accept="image/*" multiple class="form-control-file" data-forum-images>
                    <div class="forum-upload-help">Up to 5 images, max 5MB each.</div>
                    <div class="forum-upload-list" data-forum-upload-list></div>
                </div>
                <div class="forum-upload-help">
                    By creating a post, you agree to our <a href="/terms-of-service">Terms of Service</a> and <a href="/privacy-policy">Privacy Policy</a>.
                </div>
                <div class="forum-form-message" data-forum-message></div>
                <div class="forum-form-actions">
                    <button type="button" class="btn btn-outline btn-sm" data-forum-close>Cancel</button>
                    <button type="submit" class="btn btn-primary btn-sm" data-forum-submit>Publish</button>
                </div>
            </form>
        </div>
    </dialog>
</div>
<?php endif; ?>
