<?php

use NewSite\Auth\SiteAuth;
use NewSite\Cache\CacheService as Cache;
use NewSite\Database\DatabaseManager;
use NewSite\Security\IpService;
use NewSite\Settings\SettingsService;
use NewSite\Upload\ImageUrlHelper;

$forumEnabled = SettingsService::get('forum_enabled', '0') === '1';
if (!$forumEnabled || empty($forumPostData)) {
    http_response_code(404);
    ?>
    <div class="page-content">
        <div class="container">
            <h1>Page Not Found</h1>
            <p>The page you are looking for does not exist.</p>
        </div>
    </div>
    <?php
    return;
}

$db = $db ?? DatabaseManager::getWriteConnection();
$defaultAvatar = '/assets/images/default-avatar.svg';
$post = $forumPostData;
$postId = (int)($post['id'] ?? 0);
$currentUserId = (int)($_SESSION['site_user_id'] ?? 0);
$canEditPost = $currentUserId > 0 && (int)($post['user_id'] ?? 0) === $currentUserId;

$viewCount = (int)($post['views'] ?? 0);
$sessionKey = session_id() ?: IpService::getClientIP();
if ($postId > 0 && $sessionKey !== '') {
    $viewKey = 'forum_view:' . $postId . ':' . $sessionKey;
    if (!Cache::get($viewKey, null)) {
        try {
            $stmt = $db->prepare("UPDATE forum_posts SET views = COALESCE(views, 0) + 1 WHERE id = ?");
            $stmt->execute([$postId]);
            $viewCount++;
            Cache::set($viewKey, 1, 600);
        } catch (PDOException $e) {
            // ignore view update errors
        }
    }
}

$displayName = trim((string)($post['nickname'] ?? ''));
if ($displayName === '') {
    $displayName = trim((string)($post['display_name'] ?? ''));
}
if ($displayName === '') {
    $displayName = 'User';
}

$profilePhoto = trim((string)($post['profile_photo'] ?? ''));
$avatarUrl = $profilePhoto !== '' ? toPublicFilePath($profilePhoto) : $defaultAvatar;
$createdAt = !empty($post['created_at']) ? date('M j, Y', strtotime($post['created_at'])) : '';
$profileNickname = trim((string)($post['nickname'] ?? ''));
$profileLink = $profileNickname !== '' ? '/@' . rawurlencode($profileNickname) : '';

$stmt = $db->prepare("SELECT c.id, c.user_id, c.body, c.created_at, u.display_name, u.nickname, u.profile_photo FROM forum_comments c LEFT JOIN site_users u ON u.id = c.user_id WHERE c.post_id = ? AND c.status = 'approved' ORDER BY c.created_at ASC");
$stmt->execute([$postId]);
$comments = $stmt->fetchAll();
$repliesCount = count($comments);

$images = [];
if ($postId > 0) {
    $stmt = $db->prepare("SELECT id, image_path, sort_order FROM forum_post_images WHERE post_id = ? ORDER BY sort_order ASC, id ASC");
    $stmt->execute([$postId]);
    $images = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

$commentImagesById = [];
if (!empty($comments)) {
    $commentIds = array_map(static fn ($comment) => (int)($comment['id'] ?? 0), $comments);
    $commentIds = array_values(array_filter($commentIds));
    if (!empty($commentIds)) {
        $placeholders = implode(',', array_fill(0, count($commentIds), '?'));
        $stmt = $db->prepare("SELECT comment_id, id, image_path FROM forum_comment_images WHERE comment_id IN ($placeholders) ORDER BY sort_order ASC, id ASC");
        $stmt->execute($commentIds);
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
            $cid = (int)($row['comment_id'] ?? 0);
            if (!isset($commentImagesById[$cid])) {
                $commentImagesById[$cid] = [];
            }
            if (!empty($row['image_path'])) {
                $commentImagesById[$cid][] = $row;
            }
        }
    }
}

$isUserLoggedIn = SiteAuth::isLoggedIn();
?>

<div class="forum-page forum-post-page" data-forum-post-page>
    <div class="container">
        <div class="forum-post-header">
            <a href="/forum" class="forum-back-link">← Back to forum</a>
            <div class="forum-post-title-wrap">
                <h1 data-forum-post-title><?php echo e($post['title'] ?? 'Forum Post'); ?></h1>
                <div class="forum-post-meta" data-forum-post-meta>
                    <span class="forum-post-author">
                        <span>By</span>
                        <?php if ($profileLink !== ''): ?>
                            <a href="<?php echo e($profileLink); ?>" class="forum-author-link">
                                <img src="<?php echo e($avatarUrl); ?>" alt="<?php echo e($displayName); ?>" class="forum-author-avatar">
                                <span class="forum-author-name"><?php echo e($displayName); ?></span>
                            </a>
                        <?php else: ?>
                            <span class="forum-author-name"><?php echo e($displayName); ?></span>
                        <?php endif; ?>
                    </span>
                    <?php if ($createdAt !== ''): ?>
                        <span>· <?php echo e($createdAt); ?></span>
                    <?php endif; ?>
                    <span>· <span data-forum-replies><?php echo $repliesCount; ?></span> replies</span>
                    <span>· <?php echo $viewCount; ?> views</span>
                </div>
                <?php if ($canEditPost): ?>
                    <div class="forum-post-actions">
                        <button type="button" class="btn btn-outline btn-sm" data-forum-post-edit>Edit Post</button>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <div class="forum-card forum-post-content">
            <div class="forum-card-body">
                <div class="forum-post-body" data-forum-post-body>
                    <?php echo nl2br(e((string)($post['body'] ?? ''))); ?>
                </div>
                <?php if ($canEditPost): ?>
                    <form class="forum-post-edit-form is-hidden" data-forum-post-edit-form data-post-id="<?php echo $postId; ?>" enctype="multipart/form-data">
                        <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                        <input type="hidden" name="action" value="update_post">
                        <input type="hidden" name="post_id" value="<?php echo $postId; ?>">
                        <div class="forum-field">
                            <label for="forum-edit-title">Title</label>
                            <input id="forum-edit-title" name="title" type="text" maxlength="120" required value="<?php echo e((string)($post['title'] ?? '')); ?>">
                        </div>
                        <div class="forum-field">
                            <label for="forum-edit-body">Post</label>
                            <textarea id="forum-edit-body" name="body" rows="5" maxlength="5000" required><?php echo e((string)($post['body'] ?? '')); ?></textarea>
                        </div>
                        <div class="forum-image-edit-list" data-forum-post-images-edit>
                            <?php if (!empty($images)): ?>
                                <div class="forum-image-edit-grid">
                                    <?php foreach ($images as $imageRow):
                                        $imageUrl = ImageUrlHelper::getForumImageUrl($imageRow['image_path'] ?? '');
                                        if (!$imageUrl) {
                                            continue;
                                        }
                                        ?>
                                        <label class="forum-image-edit-item">
                                            <input type="checkbox" name="remove_image_ids[]" value="<?php echo (int)$imageRow['id']; ?>">
                                            <span class="forum-image-edit-thumb"><img src="<?php echo e($imageUrl); ?>" alt="Forum attachment"></span>
                                            <span>Remove</span>
                                        </label>
                                    <?php endforeach; ?>
                                </div>
                            <?php else: ?>
                                <div class="forum-image-edit-empty">No images yet.</div>
                            <?php endif; ?>
                        </div>
                        <div class="forum-field">
                            <label for="forum-edit-images">Add images</label>
                            <input id="forum-edit-images" name="images[]" type="file" accept="image/*" multiple class="form-control-file" data-forum-post-edit-images>
                            <div class="forum-upload-help">Up to 5 images total, max 5MB each.</div>
                        </div>
                        <div class="forum-form-message" data-forum-post-edit-message></div>
                        <div class="forum-form-actions">
                            <button type="button" class="btn btn-outline btn-sm" data-forum-post-edit-cancel>Cancel</button>
                            <button type="submit" class="btn btn-primary btn-sm" data-forum-post-edit-save>Save</button>
                        </div>
                    </form>
                <?php endif; ?>
                <?php if (!empty($images)): ?>
                    <div class="forum-post-images">
                        <?php foreach ($images as $imageRow):
                            $imageUrl = ImageUrlHelper::getForumImageUrl($imageRow['image_path'] ?? '');
                            if (!$imageUrl) {
                                continue;
                            }
                            ?>
                            <button type="button" class="forum-post-image" data-forum-lightbox="<?php echo e($imageUrl); ?>">
                                <img src="<?php echo e($imageUrl); ?>" alt="Forum attachment">
                            </button>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <div class="forum-comments-section">
            <div class="forum-card">
                <div class="forum-card-header">
                    <h3>Replies (<span data-forum-replies-count><?php echo $repliesCount; ?></span>)</h3>
                </div>
                <div class="forum-card-body">
                    <div class="forum-comments" data-forum-comments>
                        <?php if (empty($comments)): ?>
                            <div class="forum-empty" data-forum-empty>No replies yet.</div>
                        <?php else: ?>
                            <?php foreach ($comments as $comment):
                                $commentName = trim((string)($comment['nickname'] ?? ''));
                                if ($commentName === '') {
                                    $commentName = trim((string)($comment['display_name'] ?? ''));
                                }
                                if ($commentName === '') {
                                    $commentName = 'User';
                                }
                                $commentPhoto = trim((string)($comment['profile_photo'] ?? ''));
                                $commentAvatar = $commentPhoto !== '' ? toPublicFilePath($commentPhoto) : $defaultAvatar;
                                $commentDate = !empty($comment['created_at']) ? date('M j, Y', strtotime($comment['created_at'])) : '';
                                $commentNickname = trim((string)($comment['nickname'] ?? ''));
                                $commentProfileLink = $commentNickname !== '' ? '/@' . rawurlencode($commentNickname) : '';
                                $canEditComment = $currentUserId > 0 && (int)($comment['user_id'] ?? 0) === $currentUserId;
                                $commentImageEntries = $commentImagesById[(int)$comment['id']] ?? [];
                                ?>
                                <div class="forum-comment" data-forum-comment data-forum-comment-id="<?php echo (int)($comment['id'] ?? 0); ?>">
                                    <div class="forum-comment-avatar">
                                        <img src="<?php echo e($commentAvatar); ?>" alt="<?php echo e($commentName); ?>">
                                    </div>
                                    <div class="forum-comment-main">
                                        <div class="forum-comment-meta">
                                            <?php if ($commentProfileLink !== ''): ?>
                                                <a class="forum-comment-author" href="<?php echo e($commentProfileLink); ?>"><?php echo e($commentName); ?></a>
                                            <?php else: ?>
                                                <span><?php echo e($commentName); ?></span>
                                            <?php endif; ?>
                                            <?php if ($commentDate !== ''): ?>
                                                <span>· <?php echo e($commentDate); ?></span>
                                            <?php endif; ?>
                                            <?php if ($canEditComment): ?>
                                                <button type="button" class="btn btn-outline btn-sm" data-forum-comment-edit>Edit</button>
                                            <?php endif; ?>
                                        </div>
                                        <?php if (!empty($comment['body'])): ?>
                                            <div class="forum-comment-body" data-forum-comment-body>
                                                <?php echo nl2br(e((string)($comment['body'] ?? ''))); ?>
                                            </div>
                                        <?php endif; ?>
                                        <?php if ($canEditComment): ?>
                                            <form class="forum-comment-edit-form is-hidden" data-forum-comment-edit-form enctype="multipart/form-data">
                                                <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                                <input type="hidden" name="action" value="update_comment">
                                                <input type="hidden" name="comment_id" value="<?php echo (int)($comment['id'] ?? 0); ?>">
                                                <textarea name="body" rows="3" maxlength="2000" required><?php echo e((string)($comment['body'] ?? '')); ?></textarea>
                                                <div class="forum-image-edit-list" data-forum-comment-images-edit>
                                                    <?php if (!empty($commentImageEntries)): ?>
                                                        <div class="forum-image-edit-grid">
                                                            <?php foreach ($commentImageEntries as $commentImageRow):
                                                                $commentEditUrl = ImageUrlHelper::getForumImageUrl($commentImageRow['image_path'] ?? '');
                                                                if (!$commentEditUrl) {
                                                                    continue;
                                                                }
                                                                ?>
                                                                <label class="forum-image-edit-item">
                                                                    <input type="checkbox" name="remove_comment_image_ids[]" value="<?php echo (int)$commentImageRow['id']; ?>">
                                                                    <span class="forum-image-edit-thumb"><img src="<?php echo e($commentEditUrl); ?>" alt="Comment attachment"></span>
                                                                    <span>Remove</span>
                                                                </label>
                                                            <?php endforeach; ?>
                                                        </div>
                                                    <?php else: ?>
                                                        <div class="forum-image-edit-empty">No images yet.</div>
                                                    <?php endif; ?>
                                                </div>
                                                <div class="forum-field">
                                                    <?php $commentEditInputId = 'forum-comment-edit-images-' . (int)($comment['id'] ?? 0); ?>
                                                    <label for="<?php echo e($commentEditInputId); ?>">Add images</label>
                                                    <input id="<?php echo e($commentEditInputId); ?>" name="comment_images[]" type="file" accept="image/*" multiple class="form-control-file">
                                                    <div class="forum-upload-help">Up to 5 images total, max 5MB each.</div>
                                                </div>
                                                <div class="forum-form-message" data-forum-comment-edit-message></div>
                                                <div class="forum-form-actions">
                                                    <button type="button" class="btn btn-outline btn-sm" data-forum-comment-edit-cancel>Cancel</button>
                                                    <button type="submit" class="btn btn-primary btn-sm" data-forum-comment-edit-save>Save</button>
                                                </div>
                                            </form>
                                        <?php endif; ?>
                                        <?php if (!empty($commentImageEntries)): ?>
                                            <div class="forum-comment-images">
                                                <?php foreach ($commentImageEntries as $commentImageRow):
                                                    $commentImageUrl = ImageUrlHelper::getForumImageUrl($commentImageRow['image_path'] ?? '');
                                                    if (!$commentImageUrl) {
                                                        continue;
                                                    }
                                                    ?>
                                                    <button type="button" class="forum-comment-image" data-forum-lightbox="<?php echo e($commentImageUrl); ?>">
                                                        <img src="<?php echo e($commentImageUrl); ?>" alt="Comment attachment">
                                                    </button>
                                                <?php endforeach; ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <div class="forum-card">
                <div class="forum-card-header">
                    <h3>Leave a reply</h3>
                </div>
                <div class="forum-card-body">
                    <?php if ($isUserLoggedIn): ?>
                        <form class="forum-comment-form" data-forum-comment-form data-forum-post-id="<?php echo $postId; ?>" enctype="multipart/form-data">
                            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                            <input type="hidden" name="action" value="create_comment">
                            <input type="hidden" name="post_id" value="<?php echo $postId; ?>">
                            <textarea name="body" rows="4" maxlength="2000" required placeholder="Share your reply"></textarea>
                            <div class="forum-field">
                                <label for="forum-comment-images">Images (optional)</label>
                                <input id="forum-comment-images" name="comment_images[]" type="file" accept="image/*" multiple class="form-control-file" data-forum-comment-images>
                                <div class="forum-upload-help">Up to 5 images, max 5MB each.</div>
                                <div class="forum-upload-list" data-forum-comment-upload-list></div>
                            </div>
                            <div class="forum-form-message" data-forum-comment-message></div>
                            <div class="forum-form-actions">
                                <button type="submit" class="btn btn-primary btn-sm" data-forum-comment-submit>Post Reply</button>
                            </div>
                        </form>
                    <?php else: ?>
                        <div class="forum-empty">Please <a href="/login">log in</a> to reply.</div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>
