<?php

use NewSite\Auth\SiteAuth;
use NewSite\Database\DatabaseManager;

SiteAuth::requireLogin();
$userId = (int)($_SESSION['site_user_id'] ?? 0);
$token = trim($_GET['token'] ?? '');

if ($token === '') {
    http_response_code(400);
    echo 'Invalid download link.';
    exit;
}

$db = DatabaseManager::getWriteConnection();
$stmt = $db->prepare("SELECT dd.*, oi.order_id, oi.product_id AS order_product_id, oi.variant_id, o.user_id AS order_user_id, p.current_version AS product_current_version, p.download_url AS product_download_url, p.action_type AS product_action_type, pv.current_version AS variant_current_version, pv.download_url AS variant_download_url, pv.action_type AS variant_action_type FROM digital_downloads dd JOIN order_items oi ON oi.id = dd.order_item_id JOIN orders o ON o.id = oi.order_id LEFT JOIN products p ON p.id = COALESCE(dd.product_id, oi.product_id) LEFT JOIN product_variants pv ON pv.id = oi.variant_id WHERE dd.token = ? LIMIT 1");
$stmt->execute([$token]);
$download = $stmt->fetch();

function mapDownloadPathToBaseDir(string $path): ?array
{
    $managedUploadsBaseDir = DATA_PATH . '/admin_uploads';
    $prefixMap = [
        '/admin-file/' => $managedUploadsBaseDir,
        '/site-file/' => $managedUploadsBaseDir,
        '/data/admin_uploads/' => $managedUploadsBaseDir,
        '/assets/uploads/' => PUBLIC_PATH . '/assets/uploads',
    ];

    foreach ($prefixMap as $prefix => $baseDir) {
        if (strpos($path, $prefix) === 0) {
            return [
                'base_dir' => $baseDir,
                'relative_path' => substr($path, strlen($prefix)),
            ];
        }
    }

    return null;
}

function resolvePathInsideBaseDir(string $baseDir, string $relativePath): ?string
{
    $resolvedPath = null;
    $relativePath = ltrim(rawurldecode($relativePath), '/');

    if ($relativePath !== '') {
        $baseReal = realpath($baseDir);
        if ($baseReal !== false && is_dir($baseReal)) {
            $candidatePath = realpath($baseReal . '/' . $relativePath);
            if ($candidatePath !== false && is_file($candidatePath) && strpos($candidatePath, $baseReal . DIRECTORY_SEPARATOR) === 0) {
                $resolvedPath = $candidatePath;
            }
        }
    }

    return $resolvedPath;
}

function resolveManagedDownloadFilePath(string $downloadUrl): ?string
{
    $path = parse_url($downloadUrl, PHP_URL_PATH);
    if (!is_string($path) || $path === '') {
        return null;
    }

    $normalizedPath = '/' . ltrim($path, '/');
    $mapping = mapDownloadPathToBaseDir($normalizedPath);
    if ($mapping === null) {
        return null;
    }

    return resolvePathInsideBaseDir($mapping['base_dir'], $mapping['relative_path']);
}

if (!$download || (int)$download['order_user_id'] !== $userId) {
    http_response_code(404);
    echo 'Download not found.';
    exit;
}

$expiresAt = $download['expires_at'] ? (int)$download['expires_at'] : 0;
if ($expiresAt > 0 && time() > $expiresAt) {
    http_response_code(410);
    echo 'This download link has expired.';
    exit;
}

$downloadUrl = trim((string)($download['download_url'] ?? ''));
$variantActionType = trim((string)($download['variant_action_type'] ?? ''));
$variantDownloadUrl = trim((string)($download['variant_download_url'] ?? ''));
$productActionType = trim((string)($download['product_action_type'] ?? ''));
$productDownloadUrl = trim((string)($download['product_download_url'] ?? ''));

$effectiveDownloadUrl = $downloadUrl;
if ($variantActionType === 'download' && $variantDownloadUrl !== '') {
    $effectiveDownloadUrl = $variantDownloadUrl;
} elseif ($productActionType === 'download' && $productDownloadUrl !== '') {
    $effectiveDownloadUrl = $productDownloadUrl;
}

$realPath = resolveManagedDownloadFilePath($effectiveDownloadUrl);

if ($realPath === null) {
    http_response_code(404);
    echo 'File not found.';
    exit;
}

$effectiveVersion = trim((string)($download['product_current_version'] ?? ''));
if ($variantActionType === 'download') {
    $variantVersion = trim((string)($download['variant_current_version'] ?? ''));
    if ($variantVersion !== '') {
        $effectiveVersion = $variantVersion;
    }
}
if ($effectiveVersion === '') {
    $effectiveVersion = trim((string)($download['last_downloaded_version'] ?? ''));
}
if ($effectiveVersion === '') {
    $effectiveVersion = '1.0';
}

$stmt = $db->prepare("UPDATE digital_downloads SET download_count = download_count + 1, last_download_at = ?, last_downloaded_version = ? WHERE id = ?");
$stmt->execute([time(), $effectiveVersion, $download['id']]);

$filename = basename($realPath);
$mimeType = mime_content_type($realPath) ?: 'application/octet-stream';

header('Content-Description: File Transfer');
header('Content-Type: ' . $mimeType);
header('Content-Disposition: attachment; filename="' . $filename . '"');
header('Content-Length: ' . filesize($realPath));
header('Cache-Control: no-store, no-cache, must-revalidate');
header('Pragma: no-cache');
readfile($realPath);
exit;
