<?php
use NewSite\Auth\SiteAuth;
use NewSite\Settings\SettingsService;
use NewSite\Shop\CartService;
use NewSite\Shop\CurrencyService;

$cartId = CartService::getCartId();
$items = CartService::fetchItems($cartId);
$currencySymbol = CurrencyService::getSymbol();
$baseCurrency = SettingsService::get('store_currency', 'USD');
$currentCurrency = CurrencyService::getCurrent();
$isLoggedIn = SiteAuth::isLoggedIn();
$total = 0;
$hasDigital = false;
$hasPhysical = false;

foreach ($items as &$item) {
    $priceRaw = $item['variant_price'] !== '' && $item['variant_price'] !== null ? $item['variant_price'] : $item['product_price'];
    $priceFormatted = CurrencyService::formatValue((string)$priceRaw, $currentCurrency);
    $item['price_formatted'] = $priceFormatted;
    $quantityMax = $item['quantity_max'];
    $item['quantity_max'] = $quantityMax === null ? null : (int)$quantityMax;
    $item['variant_label'] = $item['variant_label'] ?? '';
    $item['line_total'] = '';
    $numeric = 0;
    if ($priceRaw !== '') {
        $numeric = (float)preg_replace('/[^0-9.]/', '', (string)$priceRaw);
        $line = $numeric * (int)$item['quantity'];
        $convertedLine = CurrencyService::convert($line, $baseCurrency, $currentCurrency);
        $item['line_total'] = $currencySymbol . number_format($convertedLine, 2);
        $total += $convertedLine;
    }
    // Check if this is a paid digital product
    $actionType = $item['variant_action_type'] ?? $item['product_action_type'] ?? 'cart';
    $item['is_digital'] = $actionType === 'download' && $numeric > 0;
    if ($actionType === 'download') {
        $hasDigital = true;
    } else {
        $hasPhysical = true;
    }
}
unset($item);

$isFreeOrder = $total <= 0 && !empty($items);
$canCheckout = ($stripeCheckoutEnabled || $isFreeOrder) && $isLoggedIn;
?>
<section class="checkout-page">
    <div class="container">
        <div class="checkout-header">
            <h1>Checkout</h1>
        </div>

        <?php if (empty($items)): ?>
            <div class="checkout-empty">Your cart is empty.</div>
        <?php else: ?>
            <div class="checkout-grid">
                <div class="checkout-summary">
                    <div class="checkout-card">
                        <h3>Order Summary</h3>
                        <div class="checkout-items">
                            <?php foreach ($items as $item): ?>
                                <div class="checkout-item">
                                    <div>
                                        <div class="checkout-item-name"><?php echo e($item['product_name']); ?></div>
                                        <?php if (!empty($item['variant_label'])): ?>
                                            <div class="checkout-item-variant">Version: <?php echo e($item['variant_label']); ?></div>
                                        <?php endif; ?>
                                        <?php if (!empty($item['is_digital'])): ?>
                                            <div class="cart-digital-badge">
                                                <svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/><polyline points="7 10 12 15 17 10"/><line x1="12" y1="15" x2="12" y2="3"/></svg>
                                                <span>Digital Download</span>
                                            </div>
                                        <?php endif; ?>
                                        <div class="checkout-item-qty">Qty: <?php echo (int)$item['quantity']; ?></div>
                                    </div>
                                    <div class="checkout-item-total"><?php echo e($item['line_total']); ?></div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        <div class="checkout-total">
                            <span>Total</span>
                            <strong><?php echo e($currencySymbol . number_format($total, 2)); ?></strong>
                        </div>
                    </div>
                </div>
                <div class="checkout-payment">
                    <div class="checkout-card">
                        <h3>Payment</h3>
                        <?php if (!$isLoggedIn): ?>
                            <div class="checkout-note">Login is required to complete checkout.</div>
                            <a href="/login" class="btn btn-outline">Login to continue</a>
                        <?php elseif ($isFreeOrder): ?>
                            <div class="checkout-note">Your order total is <?php echo e($currencySymbol); ?>0.00. No payment needed.</div>
                            <div class="checkout-note checkout-note--error is-hidden" data-checkout-error></div>
                            <button type="button" class="btn btn-primary checkout-button" data-checkout-start>Complete Order</button>
                            <div class="checkout-note">
                                By continuing, you agree to our <a href="/terms-of-service">Terms of Service</a> and <a href="/privacy-policy">Privacy Policy</a>. See our <a href="/refund-policy">Refund Policy</a> and <a href="/shipping-policy">Shipping Policy</a>.
                                <?php if ($hasDigital): ?>
                                    Digital downloads are delivered immediately.
                                <?php endif; ?>
                                <?php if ($hasPhysical): ?>
                                    Physical goods: standard return window is 30 days from delivery for eligible unused items in original condition (subject to local mandatory law and policy exclusions).
                                <?php endif; ?>
                            </div>
                        <?php elseif (!$stripeCheckoutEnabled): ?>
                            <div class="checkout-note">Stripe is not configured. Add your publishable key in Admin Settings.</div>
                        <?php else: ?>
                            <div class="checkout-note">Stripe checkout is ready. Click below to continue.</div>
                            <div class="checkout-note checkout-note--error is-hidden" data-checkout-error></div>
                            <button type="button" class="btn btn-primary checkout-button" data-checkout-start>Pay with Stripe</button>
                            <div class="checkout-note">
                                By continuing, you agree to our <a href="/terms-of-service">Terms of Service</a> and <a href="/privacy-policy">Privacy Policy</a>. See our <a href="/refund-policy">Refund Policy</a> and <a href="/shipping-policy">Shipping Policy</a>.
                                <?php if ($hasDigital): ?>
                                    Digital downloads are delivered immediately after payment.
                                <?php endif; ?>
                                <?php if ($hasPhysical): ?>
                                    Physical goods: standard return window is 30 days from delivery for eligible unused items in original condition (subject to local mandatory law and policy exclusions).
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</section>
