<?php
use NewSite\Auth\SiteAuth;
use NewSite\Database\DatabaseManager;
use NewSite\Email\TransactionalEmail;
use NewSite\Http\StripeClient;
use NewSite\Settings\SettingsService;
use NewSite\Shop\CartService;
use NewSite\Shop\OrderService;

SiteAuth::requireLogin();
$userId = (int)($_SESSION['site_user_id'] ?? 0);
$db = DatabaseManager::getWriteConnection();
$sessionId = trim($_GET['session_id'] ?? '');
$freeOrderId = (int)($_GET['order_id'] ?? 0);
$stripeSecretKey = SettingsService::get('stripe_secret_key', '');

$success = false;
$error = '';
$order = null;

/* ── Free-order path: order was already created by checkout-api ── */
if ($freeOrderId > 0) {
    $stmt = $db->prepare(
        "SELECT * FROM orders WHERE id = ? AND user_id = ? AND payment_provider = 'free' LIMIT 1"
    );
    $stmt->execute([$freeOrderId, $userId]);
    $order = $stmt->fetch();
    if ($order) {
        $success = true;
    } else {
        $error = 'Order not found or access denied.';
    }
} elseif ($sessionId === '') {
    $error = 'Missing checkout session.';
} elseif ($stripeSecretKey === '') {
    $error = 'Stripe is not configured.';
} else {
    $stmt = $db->prepare("SELECT * FROM orders WHERE stripe_session_id = ? LIMIT 1");
    $stmt->execute([$sessionId]);
    $order = $stmt->fetch();

    if ($order) {
        $success = true;
    } else {
        $stripeResponse = StripeClient::request('GET', '/v1/checkout/sessions/' . urlencode($sessionId), [
            'expand[]' => 'payment_intent'
        ], $stripeSecretKey);

        if (!empty($stripeResponse['error']) || $stripeResponse['status'] >= 400) {
            $error = $stripeResponse['data']['error']['message'] ?? 'Unable to verify payment.';
        } else {
            $session = $stripeResponse['data'];
            $paymentStatus = $session['payment_status'] ?? '';
            $sessionStatus = $session['status'] ?? '';
            $sessionUserId = (int)($session['metadata']['user_id'] ?? 0);

            if ($sessionUserId && $sessionUserId !== $userId) {
                $error = 'This checkout session does not belong to your account.';
            } elseif ($paymentStatus !== 'paid' || $sessionStatus !== 'complete') {
                $error = 'Payment not completed yet.';
            } else {
                $paymentIntent = $session['payment_intent']['id'] ?? $session['payment_intent'] ?? '';
                $orderId = OrderService::createFromCart($userId, CartService::getCartId(), [
                    'status' => 'paid',
                    'payment_provider' => 'stripe',
                    'stripe_session_id' => $sessionId,
                    'stripe_payment_intent' => $paymentIntent
                ]);
                if ($orderId) {
                    $stmt = $db->prepare("SELECT * FROM orders WHERE id = ? LIMIT 1");
                    $stmt->execute([$orderId]);
                    $order = $stmt->fetch();
                    TransactionalEmail::sendPurchaseReceipt($orderId);
                    $success = true;
                } else {
                    $error = 'Order could not be created. Please contact support.';
                }
            }
        }
    }
}
?>

<section class="checkout-success">
    <div class="container">
        <div class="checkout-card">
            <?php if ($success && $order): ?>
                <h1>Purchase complete</h1>
                <p>Your payment was confirmed. You can download your files from Purchases.</p>
                <div class="checkout-success-meta">
                    <div>
                        <span class="checkout-success-label">Order</span>
                        <span class="checkout-success-value">#<?php echo e($order['order_number']); ?></span>
                    </div>
                    <div>
                        <span class="checkout-success-label">Status</span>
                        <span class="checkout-success-value"><?php echo e(ucfirst($order['status'] ?? 'paid')); ?></span>
                    </div>
                </div>
                <div class="checkout-success-actions">
                    <a href="/account?tab=purchases" class="btn btn-primary">View Purchases</a>
                    <a href="/collections" class="btn btn-outline">Continue Shopping</a>
                </div>
            <?php else: ?>
                <h1>Payment not completed</h1>
                <p><?php echo e($error ?: 'We could not confirm your payment yet.'); ?></p>
                <div class="checkout-success-actions">
                    <a href="/checkout" class="btn btn-primary">Back to Checkout</a>
                    <a href="/account?tab=purchases" class="btn btn-outline">View Purchases</a>
                </div>
            <?php endif; ?>
        </div>
    </div>
</section>
