<?php
use NewSite\Settings\SettingsService;
use NewSite\Shop\CartService;
use NewSite\Shop\CurrencyService;

$cartId = CartService::getCartId();
$items = CartService::fetchItems($cartId);
$currencySymbol = CurrencyService::getSymbol();
$baseCurrency = SettingsService::get('store_currency', 'USD');
$currentCurrency = CurrencyService::getCurrent();
$total = 0;

foreach ($items as &$item) {
    $priceRaw = $item['variant_price'] !== '' && $item['variant_price'] !== null ? $item['variant_price'] : $item['product_price'];
    $priceFormatted = CurrencyService::formatValue((string)$priceRaw, $currentCurrency);
    $item['price_formatted'] = $priceFormatted;
    $quantityMax = $item['quantity_max'];
    $item['quantity_max'] = $quantityMax === null ? null : (int)$quantityMax;
    $item['variant_label'] = $item['variant_label'] ?? '';
    $item['line_total'] = '';
    $numeric = 0;
    if ($priceRaw !== '') {
        $numeric = (float)preg_replace('/[^0-9.]/', '', (string)$priceRaw);
        $line = $numeric * (int)$item['quantity'];
        $convertedLine = CurrencyService::convert($line, $baseCurrency, $currentCurrency);
        $item['line_total'] = $currencySymbol . number_format($convertedLine, 2);
        $total += $convertedLine;
    }
    // Check if this is a paid digital product
    $actionType = $item['variant_action_type'] ?? $item['product_action_type'] ?? 'cart';
    $item['is_digital'] = $actionType === 'download' && $numeric > 0;

    // Build product URL: /{page_slug}/{product_slug}
    $slug = trim((string)($item['product_slug'] ?? ''));
    $pageSlugVal = trim((string)($item['page_slug'] ?? ''));
    if ($slug !== '') {
        $item['product_url'] = '/' . ($pageSlugVal !== '' ? $pageSlugVal : 'product') . '/' . rawurlencode($slug);
    } else {
        $item['product_url'] = '';
    }
}
unset($item);
?>
<section class="cart-page">
    <div class="container">
        <div class="cart-header">
            <h1>Your Cart</h1>
        </div>

        <?php if (empty($items)): ?>
            <div class="cart-empty">Your cart is empty.</div>
        <?php else: ?>
            <div class="cart-grid">
                <div class="cart-items">
                    <?php foreach ($items as $item): ?>
                        <div class="cart-item" data-item-id="<?php echo $item['id']; ?>" data-quantity-max="<?php echo $item['quantity_max'] === null ? '' : $item['quantity_max']; ?>">
                            <div class="cart-item-media">
                                <?php if (!empty($item['media_url'])): ?>
                                    <?php if ($item['product_url'] !== ''): ?>
                                        <a href="<?php echo e($item['product_url']); ?>">
                                            <img src="<?php echo e(toPublicFilePath($item['media_url'])); ?>" alt="<?php echo e($item['product_name']); ?>">
                                        </a>
                                    <?php else: ?>
                                        <img src="<?php echo e(toPublicFilePath($item['media_url'])); ?>" alt="<?php echo e($item['product_name']); ?>">
                                    <?php endif; ?>
                                <?php endif; ?>
                            </div>
                            <div class="cart-item-info">
                                <?php if ($item['product_url'] !== ''): ?>
                                    <h3><a href="<?php echo e($item['product_url']); ?>" class="cart-item-link"><?php echo e($item['product_name']); ?></a></h3>
                                <?php else: ?>
                                    <h3><?php echo e($item['product_name']); ?></h3>
                                <?php endif; ?>
                                <?php if (!empty($item['variant_label'])): ?>
                                    <div class="cart-item-variant">Version: <?php echo e($item['variant_label']); ?></div>
                                <?php endif; ?>
                                <?php if (!empty($item['is_digital'])): ?>
                                    <div class="cart-digital-badge">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/><polyline points="7 10 12 15 17 10"/><line x1="12" y1="15" x2="12" y2="3"/></svg>
                                        <span>Digital Download</span>
                                    </div>
                                <?php endif; ?>
                                <div class="cart-item-price"><?php echo e($item['price_formatted']); ?></div>
                                <?php if (!empty($item['is_digital'])): ?>
                                    <!-- Digital products don't need quantity controls -->
                                <?php elseif ($item['quantity_max'] === 1): ?>
                                    <div class="cart-item-qty">Qty: 1</div>
                                <?php else: ?>
                                    <div class="cart-qty-controls">
                                        <button type="button" class="cart-qty-btn" data-qty="-1">-</button>
                                        <input type="number" class="cart-qty-input" min="1" value="<?php echo (int)$item['quantity']; ?>">
                                        <button type="button" class="cart-qty-btn" data-qty="1">+</button>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="cart-item-total">
                                <?php echo e($item['line_total']); ?>
                            </div>
                            <button type="button" class="cart-item-remove" data-remove aria-label="Remove item">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <path d="M3 6h18"></path>
                                    <path d="M8 6V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path>
                                    <path d="M19 6l-1 14a2 2 0 0 1-2 2H8a2 2 0 0 1-2-2L5 6"></path>
                                    <path d="M10 11v6"></path>
                                    <path d="M14 11v6"></path>
                                </svg>
                            </button>
                        </div>
                    <?php endforeach; ?>
                </div>
                <div class="cart-summary">
                    <div class="cart-summary-card">
                        <h3>Summary</h3>
                        <div class="cart-summary-row">
                            <span>Total</span>
                            <strong><?php echo e($currencySymbol . number_format($total, 2)); ?></strong>
                        </div>
                        <a class="btn btn-primary" href="/checkout">Checkout</a>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</section>
