<?php

use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Logging\LogService;
use NewSite\Settings\SettingsService;

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

const STRIPE_INTENT_WHERE = 'stripe_payment_intent = ?';

function webhookResponse(array $payload, int $status = 200): void
{
    http_response_code($status);
    header('Content-Type: application/json');
    echo json_encode($payload);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    webhookResponse(['success' => false, 'error' => 'Method not allowed.'], 405);
}

$secret = trim((string)SettingsService::get('stripe_webhook_secret', ''));
if ($secret === '') {
    webhookResponse(['success' => false, 'error' => 'Webhook secret not configured.'], 400);
}

$signatureHeader = $_SERVER['HTTP_STRIPE_SIGNATURE'] ?? '';
$payload = file_get_contents('php://input');
if ($signatureHeader === '' || $payload === false) {
    webhookResponse(['success' => false, 'error' => 'Invalid signature.'], 400);
}

function verifyStripeSignature(string $payload, string $sigHeader, string $secret, int $tolerance = 300): bool
{
    $parts = explode(',', $sigHeader);
    $timestamp = null;
    $signatures = [];

    foreach ($parts as $part) {
        $kv = explode('=', trim($part), 2);
        if (count($kv) !== 2) {
            continue;
        }
        if ($kv[0] === 't') {
            $timestamp = (int)$kv[1];
        } elseif ($kv[0] === 'v1') {
            $signatures[] = $kv[1];
        }
    }

    $isValid = false;
    if ($timestamp && !empty($signatures) && abs(time() - $timestamp) <= $tolerance) {
        $signedPayload = $timestamp . '.' . $payload;
        $expected = hash_hmac('sha256', $signedPayload, $secret);
        foreach ($signatures as $signature) {
            if (hash_equals($expected, $signature)) {
                $isValid = true;
                break;
            }
        }
    }

    return $isValid;
}

if (!verifyStripeSignature($payload, $signatureHeader, $secret)) {
    webhookResponse(['success' => false, 'error' => 'Signature verification failed.'], 400);
}

$event = json_decode($payload, true);
if (!is_array($event) || empty($event['type']) || empty($event['data']['object'])) {
    webhookResponse(['success' => false, 'error' => 'Invalid payload.'], 400);
}

$db = DatabaseManager::getWriteConnection();
$type = (string)$event['type'];
$object = $event['data']['object'];

function updateOrderStatus(PDO $db, string $whereClause, array $params, string $status): int
{
    $now = DbHelper::nowString();
    $stmt = $db->prepare("UPDATE orders SET status = ?, updated_at = ? WHERE $whereClause");
    $stmt->execute(array_merge([$status, $now], $params));
    return $stmt->rowCount();
}

$handled = false;

switch ($type) {
    case 'checkout.session.completed':
    case 'checkout.session.async_payment_succeeded':
        $sessionId = $object['id'] ?? '';
        if ($sessionId !== '') {
            $rows = updateOrderStatus($db, 'stripe_session_id = ?', [$sessionId], 'paid');
            if ($rows === 0) {
                LogService::add('info', 'Stripe webhook: order not found for session', json_encode(['session_id' => $sessionId]));
            }
        }
        $handled = true;
        break;

    case 'checkout.session.async_payment_failed':
    case 'checkout.session.expired':
        $sessionId = $object['id'] ?? '';
        if ($sessionId !== '') {
            updateOrderStatus($db, 'stripe_session_id = ?', [$sessionId], 'failed');
        }
        $handled = true;
        break;

    case 'payment_intent.succeeded':
        $intentId = $object['id'] ?? '';
        if ($intentId !== '') {
            updateOrderStatus($db, STRIPE_INTENT_WHERE, [$intentId], 'paid');
        }
        $handled = true;
        break;

    case 'payment_intent.processing':
        $intentId = $object['id'] ?? '';
        if ($intentId !== '') {
            updateOrderStatus($db, STRIPE_INTENT_WHERE, [$intentId], 'processing');
        }
        $handled = true;
        break;

    case 'payment_intent.payment_failed':
    case 'payment_intent.canceled':
        $intentId = $object['id'] ?? '';
        if ($intentId !== '') {
            updateOrderStatus($db, STRIPE_INTENT_WHERE, [$intentId], 'failed');
        }
        $handled = true;
        break;

    case 'charge.refunded':
    case 'charge.refund.updated':
        $intentId = $object['payment_intent'] ?? '';
        if ($intentId !== '') {
            updateOrderStatus($db, STRIPE_INTENT_WHERE, [$intentId], 'refunded');
        }
        $handled = true;
        break;

    case 'charge.dispute.created':
    case 'charge.dispute.updated':
        $intentId = $object['payment_intent'] ?? '';
        if ($intentId !== '') {
            updateOrderStatus($db, STRIPE_INTENT_WHERE, [$intentId], 'disputed');
        }
        $handled = true;
        break;

    case 'charge.dispute.closed':
        $intentId = $object['payment_intent'] ?? '';
        $status = $object['status'] ?? '';
        if ($intentId !== '') {
            $finalStatus = $status === 'won' ? 'paid' : 'refunded';
            updateOrderStatus($db, STRIPE_INTENT_WHERE, [$intentId], $finalStatus);
        }
        $handled = true;
        break;

    default:
        break;
}

if (!$handled) {
    LogService::add('info', 'Stripe webhook received', json_encode(['type' => $type]));
}

webhookResponse(['success' => true]);
