<?php

use NewSite\Cache\CacheService as Cache;
use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Security\IpService;

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

header('Content-Type: application/json');

$db = DatabaseManager::getWriteConnection();
$method = strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');
$productId = (int)($_POST['product_id'] ?? $_GET['product_id'] ?? 0);
$userId = !empty($_SESSION['site_user_id']) ? (int)$_SESSION['site_user_id'] : 0;
$sessionId = session_id();
$ipAddress = IpService::getClientIP();

function productLikesResponse(array $data, int $status = 200): void
{
    http_response_code($status);
    echo json_encode($data);
    exit;
}

if ($productId <= 0) {
    productLikesResponse(['success' => false, 'error' => 'Invalid product.'], 400);
}

$stmt = $db->prepare("SELECT id, likes_count, dislikes_count FROM products WHERE id = ? AND is_active = 1");
$stmt->execute([$productId]);
$product = $stmt->fetch();
if (!$product) {
    productLikesResponse(['success' => false, 'error' => 'Product not found.'], 404);
}

function fetchProductLikeState(PDO $db, int $productId, int $userId, string $sessionId, string $ipAddress): string
{
    $lookups = [];
    if ($userId > 0) {
        $lookups[] = [
            'sql' => "SELECT is_liked FROM product_likes WHERE product_id = ? AND user_id = ? LIMIT 1",
            'params' => [$productId, $userId]
        ];
    }

    if ($sessionId !== '') {
        $lookups[] = [
            'sql' => "SELECT is_liked FROM product_likes WHERE product_id = ? AND session_id = ? LIMIT 1",
            'params' => [$productId, $sessionId]
        ];
    }

    if ($ipAddress !== '') {
        $lookups[] = [
            'sql' => "SELECT is_liked FROM product_likes WHERE product_id = ? AND ip_address = ? AND user_id IS NULL AND session_id IS NULL LIMIT 1",
            'params' => [$productId, $ipAddress]
        ];
    }

    foreach ($lookups as $lookup) {
        $stmt = $db->prepare($lookup['sql']);
        $stmt->execute($lookup['params']);
        $row = $stmt->fetch();
        if ($row) {
            return !empty($row['is_liked']) ? 'like' : 'dislike';
        }
    }

    return 'none';
}

if ($method === 'GET') {
    $state = fetchProductLikeState($db, $productId, $userId, $sessionId, $ipAddress);
    productLikesResponse([
        'success' => true,
        'likes' => (int)($product['likes_count'] ?? 0),
        'dislikes' => (int)($product['dislikes_count'] ?? 0),
        'state' => $state
    ]);
}

if ($method !== 'POST') {
    productLikesResponse(['success' => false, 'error' => 'Method not allowed.'], 405);
}

$action = trim((string)($_POST['action'] ?? ''));
if (!in_array($action, ['like', 'dislike'], true)) {
    productLikesResponse(['success' => false, 'error' => 'Invalid action.'], 400);
}

$rateKey = 'product_like_rl:ip:' . $ipAddress;
if ($userId > 0) {
    $rateKey = 'product_like_rl:u:' . $userId;
} elseif ($sessionId !== '') {
    $rateKey = 'product_like_rl:s:' . $sessionId;
}

$now = time();
$rateData = Cache::get($rateKey, null);
if (!is_array($rateData)) {
    $rateData = ['count' => 0, 'window_start' => $now];
}

$window = 60;
$limit = 5;
if ($now - (int)$rateData['window_start'] >= $window) {
    $rateData = ['count' => 0, 'window_start' => $now];
}

if ((int)$rateData['count'] >= $limit) {
    $retryAfter = max(1, $window - ($now - (int)$rateData['window_start']));
    productLikesResponse([
        'success' => false,
        'rate_limited' => true,
        'retry_after' => $retryAfter,
        'likes' => (int)($product['likes_count'] ?? 0),
        'dislikes' => (int)($product['dislikes_count'] ?? 0),
        'state' => fetchProductLikeState($db, $productId, $userId, $sessionId, $ipAddress)
    ], 429);
}

$rateData['count'] = (int)$rateData['count'] + 1;
Cache::set($rateKey, $rateData, $window + 10);

$nowDb = DbHelper::nowString();

try {
    $db->beginTransaction();

    $existing = null;
    if ($userId > 0) {
        $stmt = $db->prepare("SELECT id, is_liked FROM product_likes WHERE product_id = ? AND user_id = ? LIMIT 1");
        $stmt->execute([$productId, $userId]);
        $existing = $stmt->fetch();

        if (!$existing && $sessionId !== '') {
            $stmt = $db->prepare("SELECT id, is_liked FROM product_likes WHERE product_id = ? AND session_id = ? AND user_id IS NULL LIMIT 1");
            $stmt->execute([$productId, $sessionId]);
            $existing = $stmt->fetch();
            if ($existing) {
                $claim = $db->prepare("UPDATE product_likes SET user_id = ?, session_id = NULL, updated_at = ? WHERE id = ?");
                $claim->execute([$userId, $nowDb, $existing['id']]);
            }
        }
    } else {
        if ($sessionId !== '') {
            $stmt = $db->prepare("SELECT id, is_liked FROM product_likes WHERE product_id = ? AND session_id = ? LIMIT 1");
            $stmt->execute([$productId, $sessionId]);
            $existing = $stmt->fetch();
        } elseif ($ipAddress !== '') {
            $stmt = $db->prepare("SELECT id, is_liked FROM product_likes WHERE product_id = ? AND ip_address = ? AND user_id IS NULL AND session_id IS NULL LIMIT 1");
            $stmt->execute([$productId, $ipAddress]);
            $existing = $stmt->fetch();
        }
    }

    $prevState = 'none';
    if ($existing) {
        $prevState = !empty($existing['is_liked']) ? 'like' : 'dislike';
    }
    if ($action === 'like') {
        $nextState = $prevState === 'like' ? 'none' : 'like';
    } else {
        $nextState = $prevState === 'dislike' ? 'none' : 'dislike';
    }

    $likeDelta = 0;
    $dislikeDelta = 0;

    if ($prevState === 'none' && $nextState === 'like') {
        $likeDelta = 1;
    } elseif ($prevState === 'none' && $nextState === 'dislike') {
        $dislikeDelta = 1;
    } elseif ($prevState === 'like' && $nextState === 'none') {
        $likeDelta = -1;
    } elseif ($prevState === 'dislike' && $nextState === 'none') {
        $dislikeDelta = -1;
    } elseif ($prevState === 'like' && $nextState === 'dislike') {
        $likeDelta = -1;
        $dislikeDelta = 1;
    } elseif ($prevState === 'dislike' && $nextState === 'like') {
        $likeDelta = 1;
        $dislikeDelta = -1;
    }

    if ($nextState === 'none') {
        if ($existing) {
            $stmt = $db->prepare("DELETE FROM product_likes WHERE id = ?");
            $stmt->execute([$existing['id']]);
        }
    } else {
        $isLiked = $nextState === 'like' ? 1 : 0;
        if ($existing) {
            $stmt = $db->prepare("UPDATE product_likes SET is_liked = ?, updated_at = ? WHERE id = ?");
            $stmt->execute([$isLiked, $nowDb, $existing['id']]);
        } else {
            $sessionValue = null;
            if ($userId <= 0 && $sessionId !== '') {
                $sessionValue = $sessionId;
            }
            $userValue = null;
            if ($userId > 0) {
                $userValue = $userId;
            }
            $stmt = $db->prepare("INSERT INTO product_likes (product_id, user_id, session_id, ip_address, is_liked, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([
                $productId,
                $userValue,
                $sessionValue,
                $ipAddress !== '' ? $ipAddress : null,
                $isLiked,
                $nowDb,
                $nowDb
            ]);
        }
    }

    if ($likeDelta !== 0 || $dislikeDelta !== 0) {
        $stmt = $db->prepare("UPDATE products SET likes_count = GREATEST(COALESCE(likes_count, 0) + ?, 0), dislikes_count = GREATEST(COALESCE(dislikes_count, 0) + ?, 0) WHERE id = ? RETURNING likes_count, dislikes_count");
        $stmt->execute([$likeDelta, $dislikeDelta, $productId]);
        $counts = $stmt->fetch();
        $likesCount = (int)($counts['likes_count'] ?? 0);
        $dislikesCount = (int)($counts['dislikes_count'] ?? 0);
    } else {
        $stmt = $db->prepare("SELECT likes_count, dislikes_count FROM products WHERE id = ?");
        $stmt->execute([$productId]);
        $counts = $stmt->fetch();
        $likesCount = (int)($counts['likes_count'] ?? 0);
        $dislikesCount = (int)($counts['dislikes_count'] ?? 0);
    }

    $db->commit();

    productLikesResponse([
        'success' => true,
        'likes' => $likesCount,
        'dislikes' => $dislikesCount,
        'state' => $nextState
    ]);
} catch (Throwable $e) {
    if ($db->inTransaction()) {
        $db->rollBack();
    }
    productLikesResponse(['success' => false, 'error' => 'Unable to update like.'], 500);
}
