<?php

use NewSite\Auth\SiteAuth;
use NewSite\Database\DatabaseManager;
use NewSite\Upload\ImageUrlHelper;
use NewSite\Upload\UploadService;
use NewSite\User\PrivacyService;

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

header('Content-Type: application/json');

if (!SiteAuth::isLoggedIn()) {
    $isJson = (isset($_SERVER['HTTP_ACCEPT']) && strpos($_SERVER['HTTP_ACCEPT'], 'application/json') !== false)
        || (isset($_SERVER['CONTENT_TYPE']) && strpos($_SERVER['CONTENT_TYPE'], 'application/json') !== false);

    if ($isJson) {
        http_response_code(401);
        echo json_encode(['success' => false, 'error' => 'Not logged in']);
    } else {
        header('Location: /login');
    }
    exit;
}

$db = DatabaseManager::getWriteConnection();
$userId = (int)$_SESSION['site_user_id'];
$action = $_POST['action'] ?? $_GET['action'] ?? 'get_friends';
$method = strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');

switch ($action) {
    case 'get_friends':
        // Get friends with their status and last message
        $stmt = $db->prepare("
                WITH contacts AS (
                    SELECT CASE WHEN f.user_id = ? THEN f.friend_id ELSE f.user_id END AS contact_id
                    FROM friendships f
                    WHERE f.user_id = ? OR f.friend_id = ?
                    UNION
                    SELECT CASE WHEN um.user_id = ? THEN um.sender_id ELSE um.user_id END AS contact_id
                    FROM user_messages um
                    WHERE um.message_type = 'chat' AND (um.user_id = ? OR um.sender_id = ?)
                )
                SELECT DISTINCT su.id, su.display_name, su.nickname, su.profile_photo, su.status, su.last_activity,
                (SELECT body FROM user_messages
                 WHERE ((user_id = su.id AND sender_id = ?) OR (user_id = ? AND sender_id = su.id))
                 AND message_type = 'chat'
                 ORDER BY created_at DESC LIMIT 1) as last_message,
                (SELECT COUNT(*) FROM user_messages
                 WHERE user_id = ? AND sender_id = su.id AND is_read = 0 AND message_type = 'chat') as unread_count
                FROM contacts c
                JOIN site_users su ON su.id = c.contact_id
                WHERE su.id != ?
                ORDER BY last_activity DESC NULLS LAST
            ");
        $stmt->execute([
            $userId, $userId, $userId,
            $userId, $userId, $userId,
            $userId, $userId,
            $userId,
            $userId
        ]);
        $friends = $stmt->fetchAll(PDO::FETCH_ASSOC);

        foreach ($friends as &$f) {
            $f['profile_photo'] = toPublicFilePath($f['profile_photo'] ?? '');
            $f['is_online'] = ($f['status'] === 'online' && (time() - (int)($f['last_activity'] ?? 0)) < 300);
            if (!$f['is_online'] && $f['status'] === 'online') {
                $f['status'] = 'offline'; // Auto-offline if no activity for 5 mins
            }
        }

        echo json_encode(['success' => true, 'friends' => $friends]);
        break;

    case 'get_messages':
        $friendId = (int)($_POST['friend_id'] ?? $_GET['friend_id'] ?? 0);
        if ($friendId <= 0) {
            echo json_encode(['success' => false, 'error' => 'Invalid friend ID']);
            exit;
        }

        // Only allow read-state mutation on POST (GET stays read-only).
        if ($method === 'POST') {
            $stmt = $db->prepare("UPDATE user_messages SET is_read = 1 WHERE user_id = ? AND sender_id = ? AND message_type = 'chat' AND is_read = 0");
            $stmt->execute([$userId, $friendId]);
        }

        // Fetch messages
        $stmt = $db->prepare("
            SELECT id, user_id, sender_id, body, created_at, image_path, is_read
            FROM user_messages
            WHERE ((user_id = ? AND sender_id = ?) OR (user_id = ? AND sender_id = ?))
            AND message_type = 'chat'
            ORDER BY created_at ASC
            LIMIT 100
        ");
        $stmt->execute([$userId, $friendId, $friendId, $userId]);
        $messages = $stmt->fetchAll(PDO::FETCH_ASSOC);

        foreach ($messages as &$m) {
            $m['image_url'] = ImageUrlHelper::getChatImageUrl($m['image_path'] ?? '');
            $m['body'] = e($m['body'] ?? '');
        }

        echo json_encode(['success' => true, 'messages' => $messages]);
        break;

    case 'send_message':
        $friendId = (int)($_POST['friend_id'] ?? 0);
        $body = trim($_POST['body'] ?? '');
        $imagePath = null;

        if (!empty($_FILES['image']['tmp_name']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
            $upload = UploadService::uploadChatImage($_FILES['image']);
            if ($upload['success']) {
                $imagePath = $upload['path'];
            } else {
                echo json_encode(['success' => false, 'error' => $upload['error']]);
                exit;
            }
        }

        if ($friendId <= 0 || ($body === '' && !$imagePath)) {
            echo json_encode(['success' => false, 'error' => 'Missing data']);
            exit;
        }

        if ($friendId === $userId) {
            echo json_encode(['success' => false, 'error' => 'Invalid recipient']);
            exit;
        }

        // Verify friendship or allow if recipient permits messages from everyone
        $stmt = $db->prepare("SELECT 1 FROM friendships WHERE (user_id = ? AND friend_id = ?) OR (user_id = ? AND friend_id = ?)");
        $stmt->execute([$userId, $friendId, $friendId, $userId]);
        $isFriend = (bool)$stmt->fetch();
        if (!$isFriend) {
            $privacy = PrivacyService::getUserSettings((int)$friendId);
            $messengerAccess = $privacy['messenger'] ?? 'friends';
            if ($messengerAccess !== 'everyone') {
                echo json_encode(['success' => false, 'error' => 'This user only accepts messages from friends']);
                exit;
            }
        }

        $stmt = $db->prepare("INSERT INTO user_messages (user_id, sender_id, body, message_type, created_at, sender_type, image_path) VALUES (?, ?, ?, 'chat', ?, 'user', ?)");
        $stmt->execute([$friendId, $userId, $body, time(), $imagePath]);

        // Push notification (polling fallback will pick it up)
        \NewSite\Push\PushNotification::newMessage($friendId, 'New Chat Message', $body ?: 'Sent an image');

        echo json_encode(['success' => true]);
        break;

    case 'clear_chat':
        $friendId = (int)($_POST['friend_id'] ?? 0);
        if ($friendId <= 0 || $friendId === $userId) {
            echo json_encode(['success' => false, 'error' => 'Invalid recipient']);
            exit;
        }

        $stmt = $db->prepare("DELETE FROM user_messages WHERE message_type = 'chat' AND ((user_id = ? AND sender_id = ?) OR (user_id = ? AND sender_id = ?))");
        $stmt->execute([$userId, $friendId, $friendId, $userId]);
        echo json_encode(['success' => true]);
        break;

    case 'get_unread_total':
        $excludeFriendId = (int)($_GET['exclude_friend_id'] ?? 0);
        if ($excludeFriendId > 0) {
            $stmt = $db->prepare("SELECT COUNT(*) FROM user_messages WHERE user_id = ? AND is_read = 0 AND message_type = 'chat' AND sender_id != ?");
            $stmt->execute([$userId, $excludeFriendId]);
        } else {
            $stmt = $db->prepare("SELECT COUNT(*) FROM user_messages WHERE user_id = ? AND is_read = 0 AND message_type = 'chat'");
            $stmt->execute([$userId]);
        }
        $total = (int)$stmt->fetchColumn();
        echo json_encode(['success' => true, 'total' => $total]);
        break;

    default:
        echo json_encode(['success' => false, 'error' => 'Unknown action']);
}
