<?php

use NewSite\Auth\SiteAuth;
use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Logging\LogService;
use NewSite\Upload\ImageUrlHelper;
use NewSite\Upload\UploadService;
use NewSite\User\UserMessageService;
use NewSite\User\UserService;

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

const JSON_CONTENT_TYPE = 'Content-Type: application/json';
const TRANSCRIPT_DIVIDER = "==========================================================\n";

// Allow non-JSON response for downloads
$isDownload = isset($_GET['download']) && $_GET['download'] === '1';

if (!$isDownload) {
    header(JSON_CONTENT_TYPE);
}

if (!SiteAuth::isLoggedIn()) {
    if ($isDownload) {
        http_response_code(403);
        exit('Access denied');
    }
    echo json_encode(['success' => false]);
    exit;
}

$userId = (int)($_SESSION['site_user_id'] ?? 0);
$method = strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');

// Handle chat transcript download (GDPR - user's own data)
if ($isDownload && $_SERVER['REQUEST_METHOD'] === 'GET') {
    $db = DatabaseManager::getWriteConnection();

    // Rate limit: 3 downloads per 10 minutes, 5 per hour (burst protection)
    $downloadTimesKey = 'chat_download_times_' . $userId;
    $downloadTimes = $_SESSION[$downloadTimesKey] ?? [];
    $now = time();

    // Clean old timestamps (older than 1 hour)
    $downloadTimes = array_filter($downloadTimes, function ($t) use ($now) {
        return $t > ($now - 3600);
    });

    // Count downloads in last 10 minutes and last hour
    $last10Min = array_filter($downloadTimes, function ($t) use ($now) {
        return $t > ($now - 600);
    });
    $lastHour = count($downloadTimes);

    $rateLimitError = null;
    if (count($last10Min) >= 3) {
        $rateLimitError = 'You\'ve downloaded the transcript multiple times recently. Please try again in a few minutes.';
    } elseif ($lastHour >= 5) {
        $rateLimitError = 'Download limit reached. You can download your chat history again later.';
    }

    if ($rateLimitError) {
        header(JSON_CONTENT_TYPE);
        echo json_encode([
            'success' => false,
            'error' => $rateLimitError,
            'rate_limited' => true
        ]);
        exit;
    }

    $user = UserService::getById((int)$userId);
    $userEmail = $user['email'] ?? 'unknown';

    // Get all ticket messages for this user
    $stmt = $db->prepare("SELECT * FROM user_messages WHERE user_id = ? AND message_type = 'ticket' ORDER BY created_at ASC");
    $stmt->execute([$userId]);
    $messages = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (empty($messages)) {
        header(JSON_CONTENT_TYPE);
        echo json_encode(['success' => false, 'error' => 'No messages to download.']);
        exit;
    }

    // Track this download for rate limiting
    $downloadTimes[] = $now;
    $_SESSION[$downloadTimesKey] = $downloadTimes;

    // Build transcript
    $transcript = TRANSCRIPT_DIVIDER;
    $transcript .= "SUPPORT CHAT TRANSCRIPT\n";
    $transcript .= TRANSCRIPT_DIVIDER . "\n";
    $transcript .= "Your Email: " . $userEmail . "\n";
    $transcript .= "Export Date: " . date('Y-m-d H:i:s') . "\n";
    $transcript .= "Total Messages: " . count($messages) . "\n";
    $transcript .= "\n" . TRANSCRIPT_DIVIDER . "\n";

    foreach ($messages as $msg) {
        $sender = ($msg['sender_type'] === 'admin') ? 'SUPPORT' : 'YOU';
        $time = !empty($msg['created_at']) ? date('Y-m-d H:i:s', (int)$msg['created_at']) : 'Unknown';
        $body = trim($msg['body'] ?? '');
        $hasImage = !empty($msg['image_path']) ? ' [Image Attached]' : '';

        $transcript .= "[{$time}] {$sender}{$hasImage}\n";
        $transcript .= "------------------------------------------------------------\n";
        if ($body !== '') {
            $transcript .= $body . "\n";
        }
        $transcript .= "\n";
    }

    $transcript .= TRANSCRIPT_DIVIDER;
    $transcript .= "END OF TRANSCRIPT\n";
    $transcript .= TRANSCRIPT_DIVIDER;

    // Log the download action
    LogService::add('info', 'User downloaded own chat transcript (GDPR)', json_encode([
        'user_id' => $userId,
        'user_email' => $userEmail,
        'message_count' => count($messages)
    ]));

    // Send as download - flush all buffers for immediate delivery
    $filename = 'my_chat_transcript_' . date('Y-m-d') . '.txt';

    // Clear any existing output buffers
    while (ob_get_level()) {
        ob_end_clean();
    }

    // Disable compression for this response
    if (function_exists('apache_setenv')) {
        apache_setenv('no-gzip', '1');
    }
    ini_set('zlib.output_compression', 'Off');

    // Use octet-stream to prevent server-level compression, but keep .txt extension
    header('Content-Type: application/octet-stream');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Content-Encoding: identity');
    header('X-Content-Type-Options: nosniff');
    header('Cache-Control: no-cache, no-store, must-revalidate');
    header('Pragma: no-cache');
    header('Expires: 0');
    header('Connection: close');

    echo $transcript;

    // Force flush
    if (function_exists('fastcgi_finish_request')) {
        fastcgi_finish_request();
    } else {
        flush();
    }
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    // Handle clearing entire chat (user request - GDPR right to erasure)
    if ($action === 'clear_chat') {
        $db = DatabaseManager::getWriteConnection();

        try {
            // Get all image paths before deleting
            $stmt = $db->prepare("SELECT image_path FROM user_messages WHERE user_id = ? AND message_type = 'ticket' AND image_path IS NOT NULL AND image_path != ''");
            $stmt->execute([$userId]);
            $imagePaths = $stmt->fetchAll(PDO::FETCH_COLUMN);

            // Delete all ticket messages for this user
            $stmt = $db->prepare("DELETE FROM user_messages WHERE user_id = ? AND message_type = 'ticket'");
            $stmt->execute([$userId]);
            $deletedCount = $stmt->rowCount();

            // Delete the image files
            $deletedFiles = 0;
            foreach ($imagePaths as $imagePath) {
                $fullPath = DATA_PATH . '/' . $imagePath;
                if (is_file($fullPath)) {
                    @unlink($fullPath);
                    $deletedFiles++;
                }
            }

            // Log the action
            LogService::add('info', 'User cleared own chat history (GDPR erasure)', json_encode([
                'user_id' => $userId,
                'messages_deleted' => $deletedCount,
                'files_deleted' => $deletedFiles
            ]));

            echo json_encode(['success' => true, 'message' => 'Chat history cleared.']);
        } catch (Exception $e) {
            LogService::add('error', 'Failed to clear chat', json_encode(['user_id' => $userId, 'error' => $e->getMessage()]));
            echo json_encode(['success' => false, 'error' => 'Failed to clear chat. Please try again.']);
        }
        exit;
    }

    // Rate limiting for message sending: allow burst but prevent spam
    // 10 messages per minute, 30 messages per hour
    $messageRateLimitKey = 'chat_msg_times_' . $userId;
    $messageTimes = $_SESSION[$messageRateLimitKey] ?? [];
    $now = time();

    // Clean old timestamps (older than 1 hour)
    $messageTimes = array_filter($messageTimes, function ($t) use ($now) {
        return $t > ($now - 3600);
    });

    // Count messages in last minute and last hour
    $lastMinute = array_filter($messageTimes, function ($t) use ($now) {
        return $t > ($now - 60);
    });
    $lastHour = count($messageTimes);

    $rateLimitError = null;
    if (count($lastMinute) >= 10) {
        $rateLimitError = 'You\'re sending messages too fast. Please wait a moment.';
    } elseif ($lastHour >= 30) {
        $rateLimitError = 'You\'ve reached the message limit. Please wait before sending more messages.';
    }

    if ($rateLimitError && ($action === 'start_ticket' || $action === 'reply_message')) {
        echo json_encode(['success' => false, 'error' => $rateLimitError, 'rate_limited' => true]);
        exit;
    }

    // Handle starting a new ticket (first message)
    if ($action === 'start_ticket') {
        $body = trim($_POST['reply_body'] ?? '');

        // Handle image upload
        $imagePath = null;
        if (!empty($_FILES['chat_image']) && $_FILES['chat_image']['error'] !== UPLOAD_ERR_NO_FILE) {
            $uploadResult = UploadService::uploadChatImage($_FILES['chat_image']);
            if (!$uploadResult['success']) {
                echo json_encode(['success' => false, 'error' => $uploadResult['error']]);
                exit;
            }
            $imagePath = $uploadResult['path'];
        }

        if ($body === '' && $imagePath === null) {
            echo json_encode(['success' => false, 'error' => 'Message or image is required.']);
            exit;
        }

        $db = DatabaseManager::getWriteConnection();
        $user = UserService::getById((int)$userId);
        $senderLabel = trim((string)($user['email'] ?? 'User'));
        $createdAt = time();

        try {
            $stmt = $db->prepare("INSERT INTO user_messages (user_id, subject, body, sender_type, sender_label, reply_to_id, message_type, image_path, is_read, admin_read, created_at) VALUES (?, 'Support Chat', ?, 'user', ?, NULL, 'ticket', ?, 1, 0, ?)");
            $stmt->execute([$userId, $body, $senderLabel, $imagePath, $createdAt]);

            // Track this message for rate limiting
            $messageTimes[] = $now;
            $_SESSION[$messageRateLimitKey] = $messageTimes;

            echo json_encode([
                'success' => true,
                'item' => [
                    'id' => (int)DbHelper::lastInsertId($db, 'user_messages'),
                    'body' => $body,
                    'image_path' => $imagePath,
                    'image_url' => $imagePath ? ImageUrlHelper::getChatImageUrl($imagePath) : null,
                    'sender_type' => 'user',
                    'created_at' => $createdAt
                ]
            ]);
        } catch (Exception $e) {
            LogService::add('error', 'Message insert failed', json_encode(['error' => $e->getMessage(), 'user_id' => $userId]));
            echo json_encode(['success' => false, 'error' => 'Unable to save message. Please try again.']);
        }
        exit;
    }

    if ($action !== 'reply_message') {
        echo json_encode(['success' => false, 'error' => 'Invalid action']);
        exit;
    }

    $replyToId = (int)($_POST['reply_to_id'] ?? 0);
    $body = trim($_POST['reply_body'] ?? '');

    // Handle image upload
    $imagePath = null;
    if (!empty($_FILES['chat_image']) && $_FILES['chat_image']['error'] !== UPLOAD_ERR_NO_FILE) {
        $uploadResult = UploadService::uploadChatImage($_FILES['chat_image']);
        if (!$uploadResult['success']) {
            echo json_encode(['success' => false, 'error' => $uploadResult['error']]);
            exit;
        }
        $imagePath = $uploadResult['path'];
    }

    // Require either body text or image
    if ($replyToId <= 0 || ($body === '' && $imagePath === null)) {
        echo json_encode(['success' => false, 'error' => 'Reply message or image is required.']);
        exit;
    }

    $db = DatabaseManager::getWriteConnection();
    $stmt = $db->prepare("SELECT id, subject, sender_type, message_type, reply_to_id FROM user_messages WHERE id = ? AND user_id = ? LIMIT 1");
    $stmt->execute([$replyToId, $userId]);
    $original = $stmt->fetch();
    if (!$original) {
        echo json_encode(['success' => false, 'error' => 'Message not found.']);
        exit;
    }
    // Allow replies to ticket messages (from admin or user's own messages in the thread)
    if (($original['message_type'] ?? 'notice') !== 'ticket') {
        echo json_encode(['success' => false, 'error' => 'Replies are only available for ticket messages.']);
        exit;
    }

    $subject = trim((string)($original['subject'] ?? ''));
    if ($subject === '') {
        $subject = 'Ticket';
    }
    $threadId = (int)($original['reply_to_id'] ?? 0);
    if ($threadId <= 0) {
        $threadId = (int)$original['id'];
    }
    $user = UserService::getById((int)$userId);
    $senderLabel = trim((string)($user['email'] ?? 'User'));

    try {
        $stmt = $db->prepare("INSERT INTO user_messages (user_id, subject, body, sender_type, sender_label, reply_to_id, message_type, image_path, is_read, admin_read, created_at) VALUES (?, ?, ?, 'user', ?, ?, 'ticket', ?, 1, 0, ?)");
        $createdAt = time();
        $stmt->execute([$userId, $subject, $body, $senderLabel, $threadId, $imagePath, $createdAt]);

        // Track this message for rate limiting
        $messageTimes[] = $now;
        $_SESSION[$messageRateLimitKey] = $messageTimes;

        echo json_encode([
            'success' => true,
            'item' => [
                'id' => (int)DbHelper::lastInsertId($db, 'user_messages'),
                'body' => $body,
                'image_path' => $imagePath,
                'image_url' => $imagePath ? ImageUrlHelper::getChatImageUrl($imagePath) : null,
                'sender_type' => 'user',
                'created_at' => $createdAt
            ]
        ]);
    } catch (Exception $e) {
        LogService::add('error', 'Reply insert failed', json_encode(['error' => $e->getMessage(), 'user_id' => $userId]));
        echo json_encode(['success' => false, 'error' => 'Unable to save reply. Please try again.']);
    }
    exit;
}

// Mark ticket messages as read (POST-only to avoid CSRF through GET).
$markRead = $method === 'POST' ? ($_POST['mark_read'] ?? '') : '';
if ($markRead === 'ticket') {
    $db = DatabaseManager::getWriteConnection();
    $stmt = $db->prepare("UPDATE user_messages SET is_read = 1 WHERE user_id = ? AND (message_type = 'ticket' OR subject = 'Ticket' OR LOWER(subject) LIKE '%ticket%') AND is_read = 0");
    $stmt->execute([$userId]);
    \NewSite\Cache\CachedQueries::invalidateNotificationCache($userId);
    echo json_encode(['success' => true]);
    exit;
}

// Mark a specific message as read (POST-only to avoid CSRF through GET).
$markMessageId = $method === 'POST' ? (int)($_POST['mark_message_read'] ?? 0) : 0;
if ($markMessageId > 0) {
    $db = DatabaseManager::getWriteConnection();
    $stmt = $db->prepare("UPDATE user_messages SET is_read = 1 WHERE id = ? AND user_id = ?");
    $stmt->execute([$markMessageId, $userId]);
    \NewSite\Cache\CachedQueries::invalidateNotificationCache($userId);
    echo json_encode(['success' => true]);
    exit;
}

$limit = (int)($_GET['limit'] ?? 10);
if ($limit < 1) {
    $limit = 10;
}
if ($limit > 50) {
    $limit = 50;
}

$type = $_GET['type'] ?? '';
if ($type === 'ticket') {
    $stmt = DatabaseManager::getWriteConnection()->prepare("SELECT * FROM user_messages WHERE user_id = ? AND (message_type = 'ticket' OR subject = 'Ticket') ORDER BY created_at DESC LIMIT ?");
    $stmt->bindValue(1, $userId, PDO::PARAM_INT);
    $stmt->bindValue(2, $limit, PDO::PARAM_INT);
    $stmt->execute();
    $items = $stmt->fetchAll();
    $unread = UserMessageService::getUnreadCount((int)$userId);
} else {
    // Exclude ticket messages from general message list
    $stmt = DatabaseManager::getWriteConnection()->prepare("SELECT * FROM user_messages WHERE user_id = ? AND (message_type IS NULL OR (message_type != 'ticket' AND message_type != 'chat')) AND (subject IS NULL OR LOWER(subject) NOT LIKE '%ticket%') AND (reply_to_id IS NULL OR reply_to_id = 0) ORDER BY created_at DESC LIMIT ?");
    $stmt->bindValue(1, $userId, PDO::PARAM_INT);
    $stmt->bindValue(2, $limit, PDO::PARAM_INT);
    $stmt->execute();
    $items = $stmt->fetchAll();
    $unread = UserMessageService::getUnreadCount((int)$userId);
}

$payload = [];
foreach ($items as $item) {
    $imagePath = $item['image_path'] ?? null;
    $payload[] = [
        'id' => (int)$item['id'],
        'subject' => $item['subject'] ?? 'Message',
        'body' => $item['body'] ?? '',
        'is_read' => !empty($item['is_read']),
        'admin_read' => !empty($item['admin_read']),
        'created_at' => !empty($item['created_at']) ? (int)$item['created_at'] : 0,
        'sender_type' => $item['sender_type'] ?? 'admin',
        'message_type' => $item['message_type'] ?? 'notice',
        'image_path' => $imagePath,
        'image_url' => $imagePath ? ImageUrlHelper::getChatImageUrl($imagePath) : null
    ];
}

echo json_encode([
    'success' => true,
    'unread' => $unread,
    'items' => $payload
]);
