<?php

use NewSite\Config\SetupService;
use NewSite\Database\DatabaseManager;
use NewSite\Settings\SettingsService;
use NewSite\Shop\CurrencyService;

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

if (empty($_SESSION['site_user_id'])) {
    http_response_code(403);
    exit;
}

$orderId = isset($_GET['order']) ? (int)$_GET['order'] : 0;
if ($orderId <= 0) {
    http_response_code(400);
    exit;
}

$db = DatabaseManager::getWriteConnection();
$userId = (int)$_SESSION['site_user_id'];

$stmt = $db->prepare("SELECT * FROM orders WHERE id = ? AND user_id = ? LIMIT 1");
$stmt->execute([$orderId, $userId]);
$order = $stmt->fetch();
if (!$order) {
    http_response_code(404);
    exit;
}

$stmt = $db->prepare("SELECT * FROM order_items WHERE order_id = ? ORDER BY id ASC");
$stmt->execute([$orderId]);
$items = $stmt->fetchAll();

$stmt = $db->prepare("SELECT display_name, email FROM site_users WHERE id = ? LIMIT 1");
$stmt->execute([$userId]);
$user = $stmt->fetch();

$currencyCode = strtoupper((string)($order['currency'] ?? CurrencyService::getCurrent()));
$symbol = CurrencyService::getSymbolFor($currencyCode);
$currencyPrefix = '';
if ($symbol !== '' && preg_match('/^[\x20-\x7E]+$/', $symbol)) {
    $currencyPrefix = $symbol;
} elseif ($currencyCode !== '') {
    $currencyPrefix = $currencyCode . ' ';
}
$orderNumber = $order['order_number'] ?? $orderId;
$orderDate = !empty($order['created_at']) ? date('M j, Y', strtotime($order['created_at'])) : '';
$status = ucfirst($order['status'] ?? 'paid');
$totalAmount = number_format((float)($order['total_amount'] ?? 0), 2);

$siteName = SettingsService::getTheme('site_name', SettingsService::get('site_name', 'Store'));
$displayName = trim($user['display_name'] ?? '') !== '' ? $user['display_name'] : 'Customer';
$email = $user['email'] ?? '';

$contactName = trim((string)SettingsService::get('contact_name', ''));
$contactEmail = trim((string)SettingsService::get('contact_email', ''));
$contactPhone = trim((string)SettingsService::get('contact_phone', ''));
$contactAddress = trim((string)SettingsService::get('contact_address', ''));
$siteUrl = SetupService::getBaseUrl();

$termsEnabled = SettingsService::get('page_terms_enabled', '1') === '1';
$privacyEnabled = SettingsService::get('page_privacy_enabled', '1') === '1';
$refundEnabled = SettingsService::get('page_refund_enabled', '1') === '1';
$shippingEnabled = SettingsService::get('page_shipping_enabled', '1') === '1';
$legalEnabled = SettingsService::get('page_legal_notice_enabled', '1') === '1';

$shippingName = trim($order['shipping_name'] ?? '');
$shippingAddress = trim($order['shipping_address'] ?? '');
$shippingCountry = trim($order['shipping_country'] ?? '');
$shippingPostal = trim($order['shipping_postal_code'] ?? '');

$lines = [];
$lines[] = $siteName !== '' ? $siteName : 'Invoice';
$lines[] = '';
$lines[] = 'Invoice';
$lines[] = 'Order #: ' . $orderNumber;
if ($orderDate !== '') {
    $lines[] = 'Date: ' . $orderDate;
}
$lines[] = 'Status: ' . $status;
$lines[] = '';
$lines[] = 'Supplier';
$lines[] = $siteName !== '' ? $siteName : 'Store';
if ($contactName !== '') {
    $lines[] = $contactName;
}
if ($contactEmail !== '') {
    $lines[] = 'Email: ' . $contactEmail;
}
if ($contactPhone !== '') {
    $lines[] = 'Phone: ' . $contactPhone;
}
if ($contactAddress !== '') {
    $lines[] = $contactAddress;
}
if ($siteUrl !== '') {
    $lines[] = 'Website: ' . $siteUrl;
}
$lines[] = '';
$lines[] = 'Customer: ' . $displayName;
if ($email !== '') {
    $lines[] = 'Email: ' . $email;
}

if ($shippingName !== '' || $shippingAddress !== '' || $shippingCountry !== '' || $shippingPostal !== '') {
    $lines[] = '';
    $lines[] = 'Shipping';
    if ($shippingName !== '') {
        $lines[] = $shippingName;
    }
    if ($shippingAddress !== '') {
        $lines[] = $shippingAddress;
    }
    $shippingLine = trim($shippingCountry . ' ' . $shippingPostal);
    if ($shippingLine !== '') {
        $lines[] = $shippingLine;
    }
}

$lines[] = '';
$lines[] = 'Items';
$subTotal = 0.0;
foreach ($items as $item) {
    $qty = (int)($item['quantity'] ?? 1);
    $unit = (float)($item['unit_price'] ?? 0);
    $lineTotal = $qty * $unit;
    $subTotal += $lineTotal;
    $name = trim($item['product_name'] ?? 'Product');
    $variant = trim($item['variant_label'] ?? '');
    if ($variant !== '') {
        $name .= ' (' . $variant . ')';
    }
    $lines[] = '- ' . $name;
    $lines[] = '  Qty ' . $qty . ' @ ' . $currencyPrefix . number_format($unit, 2) . ' = ' . $currencyPrefix . number_format($lineTotal, 2);
}

$lines[] = '';
$lines[] = 'Total: ' . $currencyPrefix . $totalAmount;

$legalLinks = [];
if ($termsEnabled) {
    $legalLinks[] = 'Terms: ' . $siteUrl . '/terms-of-service';
}
if ($privacyEnabled) {
    $legalLinks[] = 'Privacy: ' . $siteUrl . '/privacy-policy';
}
if ($refundEnabled) {
    $legalLinks[] = 'Refunds: ' . $siteUrl . '/refund-policy';
}
if ($shippingEnabled) {
    $legalLinks[] = 'Shipping: ' . $siteUrl . '/shipping-policy';
}
if ($legalEnabled) {
    $legalLinks[] = 'Legal: ' . $siteUrl . '/legal-notice';
}

if (!empty($legalLinks)) {
    $lines[] = '';
    $lines[] = 'Legal';
    foreach ($legalLinks as $linkLine) {
        $lines[] = $linkLine;
    }
}

function invoicePdfText(string $text): string
{
    $text = str_replace(["\\", "(", ")"], ["\\\\", "\\(", "\\)"], $text);
    if (function_exists('iconv')) {
        $converted = @iconv('UTF-8', 'ISO-8859-1//TRANSLIT', $text);
        if ($converted !== false) {
            $text = $converted;
        }
    }
    $text = preg_replace('/[^\x09\x0A\x0D\x20-\x7E]/', '?', $text);
    return $text;
}

/**
 * Multibyte-safe string length with fallback.
 */
function invoiceSafeStrlen(string $str): int
{
    return function_exists('mb_strlen') ? mb_strlen($str) : strlen($str);
}

/**
 * Wrap a single long line at word boundaries.
 *
 * @return string[] Wrapped lines
 */
function invoiceWrapSingleLine(string $line, int $maxLen): array
{
    $words = preg_split('/\s+/', $line);
    $segments = [];
    $current = '';
    foreach ($words as $word) {
        $next = $current === '' ? $word : $current . ' ' . $word;
        if (invoiceSafeStrlen($next) > $maxLen && $current !== '') {
            $segments[] = $current;
            $current = $word;
        } else {
            $current = $next;
        }
    }
    if ($current !== '') {
        $segments[] = $current;
    }
    return $segments;
}

/**
 * Wrap an array of text lines to fit within a maximum character width.
 */
function invoiceWrapLines(array $lines, int $maxLen = 92): array
{
    $wrapped = [];
    foreach ($lines as $line) {
        $line = (string)$line;
        if (invoiceSafeStrlen($line) <= $maxLen) {
            $wrapped[] = $line;
            continue;
        }
        $wrapped = array_merge($wrapped, invoiceWrapSingleLine($line, $maxLen));
    }
    return $wrapped;
}

$wrappedLines = invoiceWrapLines($lines);

$nl = "\r\n";

$content = "BT" . $nl;
$content .= "/F1 16 Tf" . $nl;
$content .= "72 780 Td" . $nl;
$content .= "18 TL" . $nl;
if (!empty($wrappedLines)) {
    $title = array_shift($wrappedLines);
    $content .= "(" . invoicePdfText($title) . ") Tj" . $nl;
    $content .= "T*" . $nl;
}
$content .= "/F1 12 Tf" . $nl;
$content .= "14 TL" . $nl;
foreach ($wrappedLines as $line) {
    $content .= "(" . invoicePdfText($line) . ") Tj" . $nl;
    $content .= "T*" . $nl;
}
$content .= "ET" . $nl;

$objects = [];
$objects[] = "1 0 obj<< /Type /Catalog /Pages 2 0 R >>endobj";
$objects[] = "2 0 obj<< /Type /Pages /Kids [3 0 R] /Count 1 >>endobj";
$objects[] = "3 0 obj<< /Type /Page /Parent 2 0 R /MediaBox [0 0 612 792] /Resources << /Font << /F1 4 0 R >> >> /Contents 5 0 R >>endobj";
$objects[] = "4 0 obj<< /Type /Font /Subtype /Type1 /BaseFont /Helvetica >>endobj";
$objects[] = "5 0 obj<< /Length " . strlen($content) . " >>stream" . $nl . $content . "endstream" . $nl . "endobj";

$pdf = "%PDF-1.4\n";
$offsets = [];
foreach ($objects as $obj) {
    $offsets[] = strlen($pdf);
    $pdf .= $obj . $nl;
}

$xrefOffset = strlen($pdf);
$pdf .= "xref" . $nl . "0 " . (count($offsets) + 1) . $nl;
$pdf .= "0000000000 65535 f " . $nl;
foreach ($offsets as $offset) {
    $pdf .= sprintf("%010d 00000 n " . $nl, $offset);
}
$pdf .= "trailer<< /Size " . (count($offsets) + 1) . " /Root 1 0 R >>" . $nl;
$pdf .= "startxref" . $nl . $xrefOffset . $nl . "%%EOF" . $nl;

$filename = 'invoice-' . preg_replace('/[^a-zA-Z0-9_-]/', '', (string)$orderNumber) . '.pdf';
if (function_exists('ob_get_level')) {
    while (ob_get_level() > 0) {
        ob_end_clean();
    }
}

header('Content-Type: application/pdf');
header('Content-Disposition: inline; filename="' . $filename . '"');
header('Content-Transfer-Encoding: binary');
header('Accept-Ranges: none');
header('Content-Length: ' . strlen($pdf));
header('Cache-Control: private, no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

echo $pdf;
exit;
