<?php

/**
 * Mini-game leaderboard API
 *
 * Security:
 * - POST requires CSRF token (globally enforced in _init.php)
 * - Score submissions require authenticated users
 * - Lightweight per-user rate limit for score submissions
 */

declare(strict_types=1);

use NewSite\Cache\CacheService as Cache;
use NewSite\Database\DatabaseManager;

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

header('Content-Type: application/json; charset=UTF-8');
header('Cache-Control: no-cache, no-store, must-revalidate');

/**
 * @param array<string,mixed> $payload
 */
function gameLeaderboardResponse(array $payload, int $status = 200): void
{
    http_response_code($status);
    echo json_encode($payload);
    exit;
}

$method = strtoupper((string)($_SERVER['REQUEST_METHOD'] ?? 'GET'));
$input = json_decode((string)file_get_contents('php://input'), true);
if (!is_array($input)) {
    $input = [];
}

$gameSlugRaw = (string)($input['game'] ?? $_POST['game'] ?? $_GET['game'] ?? '');
$gameSlug = \NewSite\Minigames\MinigameCatalog::normalizeSlug($gameSlugRaw);
$catalog = \NewSite\Minigames\MinigameCatalog::getCatalog();
if ($gameSlug === '' || !isset($catalog[$gameSlug])) {
    gameLeaderboardResponse(['success' => false, 'error' => 'Unknown game.'], 404);
}

$db = DatabaseManager::getWriteConnection();
if (!\NewSite\Minigames\MinigameRepository::isEnabled($db, $gameSlug, true)) {
    gameLeaderboardResponse(['success' => false, 'error' => 'Game unavailable.'], 404);
}

$currentUserId = (int)($_SESSION['site_user_id'] ?? 0);

if ($method === 'GET') {
    $action = strtolower(trim((string)($_GET['action'] ?? '')));
    if ($action === 'start-round') {
        if ($currentUserId <= 0) {
            gameLeaderboardResponse(['success' => false, 'error' => 'Login required.'], 401);
        }

        $roundStartRateKey = 'mini_game_round_start_rl:u:' . $currentUserId . ':' . $gameSlug;
        $roundStartWindowSeconds = 60;
        $roundStartLimit = 30;
        $roundStartData = Cache::get($roundStartRateKey, null);
        $roundStartNow = time();

        if (!is_array($roundStartData)) {
            $roundStartData = ['count' => 0, 'window_start' => $roundStartNow];
        }

        $roundStartWindowStart = (int)($roundStartData['window_start'] ?? $roundStartNow);
        if (($roundStartNow - $roundStartWindowStart) >= $roundStartWindowSeconds) {
            $roundStartData = ['count' => 0, 'window_start' => $roundStartNow];
            $roundStartWindowStart = $roundStartNow;
        }

        $roundStartCount = (int)($roundStartData['count'] ?? 0);
        if ($roundStartCount >= $roundStartLimit) {
            $retryAfter = max(1, $roundStartWindowSeconds - ($roundStartNow - $roundStartWindowStart));
            gameLeaderboardResponse([
                'success' => false,
                'error' => 'Too many round starts. Please wait a moment.',
                'retry_after' => $retryAfter,
            ], 429);
        }

        $roundStartData['count'] = $roundStartCount + 1;
        Cache::set($roundStartRateKey, $roundStartData, $roundStartWindowSeconds + 10);

        $round = \NewSite\Minigames\MinigameRoundVerifier::issueRound($gameSlug, $currentUserId);
        if (empty($round)) {
            gameLeaderboardResponse(['success' => false, 'error' => 'Could not start round.'], 500);
        }

        gameLeaderboardResponse([
            'success' => true,
            'game' => $gameSlug,
            'logged_in' => true,
            'round' => $round,
        ]);
    }

    $limit = (int)($_GET['limit'] ?? 50);
    $limit = max(1, min(50, $limit));

    $items = \NewSite\Minigames\MinigameRepository::getLeaderboard($db, $gameSlug, $limit, $currentUserId);
    gameLeaderboardResponse([
        'success' => true,
        'game' => $gameSlug,
        'logged_in' => $currentUserId > 0,
        'items' => $items,
    ]);
}

if ($method !== 'POST') {
    gameLeaderboardResponse(['success' => false, 'error' => 'Method not allowed.'], 405);
}

if ($currentUserId <= 0) {
    gameLeaderboardResponse(['success' => false, 'error' => 'Login required.'], 401);
}

$rateKey = 'mini_game_score_rl:u:' . $currentUserId . ':' . $gameSlug;
$now = time();
$rateWindowSeconds = 60;
$rateLimit = 30;
$rateData = Cache::get($rateKey, null);
if (!is_array($rateData)) {
    $rateData = ['count' => 0, 'window_start' => $now];
}

$windowStart = (int)($rateData['window_start'] ?? $now);
if (($now - $windowStart) >= $rateWindowSeconds) {
    $rateData = ['count' => 0, 'window_start' => $now];
    $windowStart = $now;
}

$currentCount = (int)($rateData['count'] ?? 0);
if ($currentCount >= $rateLimit) {
    $retryAfter = max(1, $rateWindowSeconds - ($now - $windowStart));
    gameLeaderboardResponse([
        'success' => false,
        'error' => 'Too many score submissions. Please wait a moment.',
        'retry_after' => $retryAfter,
    ], 429);
}

$rateData['count'] = $currentCount + 1;
Cache::set($rateKey, $rateData, $rateWindowSeconds + 10);

$scoreRaw = $input['score'] ?? $_POST['score'] ?? null;
if (!is_numeric($scoreRaw)) {
    gameLeaderboardResponse(['success' => false, 'error' => 'Invalid score.'], 400);
}

$score = (float)$scoreRaw;
if (!is_finite($score) || $score < 0 || $score > 1000000) {
    gameLeaderboardResponse(['success' => false, 'error' => 'Score out of range.'], 400);
}

$score = round($score, 2);

$currentBestScore = null;
if (\NewSite\Minigames\MinigameRepository::leaderboardTableExists($db)) {
    try {
        $bestStmt = $db->prepare(
            'SELECT best_score FROM mini_game_leaderboard WHERE game_slug = ? AND user_id = ? LIMIT 1'
        );
        $bestStmt->execute([$gameSlug, $currentUserId]);
        $bestValue = $bestStmt->fetchColumn();
        if ($bestValue !== false && is_numeric($bestValue)) {
            $currentBestScore = (float)$bestValue;
        }
    } catch (Throwable $e) {
        $currentBestScore = null;
    }
}

if ($currentBestScore !== null && $score <= $currentBestScore) {
    $items = \NewSite\Minigames\MinigameRepository::getLeaderboard($db, $gameSlug, 50, $currentUserId);
    gameLeaderboardResponse([
        'success' => true,
        'game' => $gameSlug,
        'items' => $items,
        'submission_skipped' => true,
        'message' => 'Score did not beat your best.',
    ]);
}

$roundId = (string)($input['round_id'] ?? $_POST['round_id'] ?? '');
$roundToken = (string)($input['round_token'] ?? $_POST['round_token'] ?? '');
$roundValidation = \NewSite\Minigames\MinigameRoundVerifier::validateSubmission($gameSlug, $currentUserId, $roundId, $roundToken, $score);
if (empty($roundValidation['ok'])) {
    $status = (int)($roundValidation['status'] ?? 400);
    if ($status < 400 || $status > 599) {
        $status = 400;
    }

    $response = [
        'success' => false,
        'error' => (string)($roundValidation['error'] ?? 'Score verification failed.'),
    ];

    if (isset($roundValidation['max_allowed_score'])) {
        $response['max_allowed_score'] = (float)$roundValidation['max_allowed_score'];
    }
    if (isset($roundValidation['elapsed_seconds'])) {
        $response['elapsed_seconds'] = (int)$roundValidation['elapsed_seconds'];
    }

    gameLeaderboardResponse($response, $status);
}

$saved = \NewSite\Minigames\MinigameRepository::submitScore($db, $gameSlug, $currentUserId, $score);
if (!$saved) {
    gameLeaderboardResponse(['success' => false, 'error' => 'Unable to save score.'], 500);
}

$items = \NewSite\Minigames\MinigameRepository::getLeaderboard($db, $gameSlug, 50, $currentUserId);
gameLeaderboardResponse([
    'success' => true,
    'game' => $gameSlug,
    'items' => $items,
]);
