<?php

use NewSite\User\FriendService;

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

/**
 * Friends Search API
 * Handles AJAX search requests for finding users to add as friends
 */

class FriendsSearchException extends RuntimeException
{
}

// Catch all errors
set_error_handler(function ($severity, $message, $file, $line) {
    throw new FriendsSearchException($message, 0, $severity, $file, $line);
});

// Start output buffering to catch any stray output
ob_start();

try {
    // Clear any output that might have occurred during init
    ob_end_clean();

    header('Content-Type: application/json; charset=utf-8');
    header('Cache-Control: no-cache, no-store, must-revalidate');

    // Must be logged in
    $userId = (int)($_SESSION['site_user_id'] ?? 0);
    if (!$userId) {
        http_response_code(401);
        echo json_encode(['success' => false, 'error' => 'Not authenticated']);
        exit;
    }

    $query = trim($_GET['q'] ?? '');
    if ($query === '') {
        echo json_encode(['success' => true, 'results' => []]);
        exit;
    }

    // Search for users
    $searchResults = FriendService::searchUsers((string)$query, $userId, (int)20);

    // Build response with friendship status
    $results = [];
    foreach ($searchResults as $user) {
        $status = FriendService::getStatus((int)$userId, (int)$user['id']);
        $results[] = [
            'id' => (int)$user['id'],
            'nickname' => $user['nickname'],
            'display_name' => $user['display_name'] ?: $user['nickname'],
            'profile_photo' => $user['profile_photo'] ?: null,
            'status' => $status
        ];
    }

    echo json_encode(['success' => true, 'results' => $results]);

} catch (Throwable $e) {
    // Clear any partial output
    if (ob_get_level()) {
        ob_end_clean();
    }

    error_log('Friends search error: ' . $e->getMessage() . ' in ' . $e->getFile() . ':' . $e->getLine());
    header('Content-Type: application/json; charset=utf-8');
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Search failed'
    ]);
}
