<?php

use NewSite\User\FriendService;

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

/**
 * Friends List API
 * Returns friend data for tabs (all, pending, sent, blocked)
 */

class FriendsListException extends RuntimeException
{
}

set_error_handler(function ($severity, $message, $file, $line) {
    throw new FriendsListException($message, 0, $severity, $file, $line);
});

ob_start();

try {
    ob_end_clean();

    header('Content-Type: application/json; charset=utf-8');
    header('Cache-Control: no-cache, no-store, must-revalidate');

    // Must be logged in
    $userId = (int)($_SESSION['site_user_id'] ?? 0);
    if (!$userId) {
        http_response_code(401);
        echo json_encode(['success' => false, 'error' => 'Not authenticated']);
        exit;
    }

    $tab = $_GET['tab'] ?? 'all';
    $data = [];

    switch ($tab) {
        case 'all':
            $friends = FriendService::getList((int)$userId);
            $data = array_map(function ($f) {
                return [
                    'id' => (int)$f['id'],
                    'nickname' => $f['nickname'],
                    'display_name' => $f['display_name'] ?: $f['nickname'],
                    'profile_photo' => $f['profile_photo'] ?: null,
                    'friends_since' => $f['friends_since'] ?? null
                ];
            }, $friends);
            break;

        case 'pending':
            $requests = FriendService::getPendingRequests((int)$userId);
            $data = array_map(function ($r) {
                return [
                    'request_id' => (int)$r['id'],
                    'user_id' => (int)$r['from_user_id'],
                    'nickname' => $r['nickname'],
                    'display_name' => $r['display_name'] ?: $r['nickname'],
                    'profile_photo' => $r['profile_photo'] ?: null,
                    'message' => $r['message'] ?? null,
                    'created_at' => $r['created_at'] ?? null
                ];
            }, $requests);
            break;

        case 'sent':
            $requests = FriendService::getSentRequests((int)$userId);
            $data = array_map(function ($r) {
                return [
                    'request_id' => (int)$r['id'],
                    'user_id' => (int)$r['to_user_id'],
                    'nickname' => $r['nickname'],
                    'display_name' => $r['display_name'] ?: $r['nickname'],
                    'profile_photo' => $r['profile_photo'] ?: null,
                    'created_at' => $r['created_at'] ?? null
                ];
            }, $requests);
            break;

        case 'blocked':
            $blocked = FriendService::getBlockedUsers((int)$userId);
            $data = array_map(function ($b) {
                return [
                    'user_id' => (int)$b['id'],
                    'nickname' => $b['nickname'],
                    'display_name' => $b['display_name'] ?: $b['nickname'],
                    'profile_photo' => $b['profile_photo'] ?: null,
                    'blocked_at' => $b['blocked_at'] ?? null
                ];
            }, $blocked);
            break;

        default:
            $data = [];
            break;
    }

    echo json_encode(['success' => true, 'tab' => $tab, 'data' => $data]);

} catch (Throwable $e) {
    if (ob_get_level()) {
        ob_end_clean();
    }

    error_log('Friends list error: ' . $e->getMessage());
    header('Content-Type: application/json; charset=utf-8');
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Failed to load data']);
}
