<?php

/**
 * Friends Action API
 * Handles AJAX friend requests (send, accept, decline, cancel, remove, block, unblock)
 */

use NewSite\Database\DbHelper;
use NewSite\Exception\PhpErrorException;
use NewSite\User\FriendService;

set_error_handler(function ($severity, $message, $file, $line) {
    throw new PhpErrorException($message, 0, $severity, $file, $line);
});

ob_start();

try {
    ob_end_clean();

    header('Content-Type: application/json; charset=utf-8');
    header('Cache-Control: no-cache, no-store, must-revalidate');

    // Must be POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['success' => false, 'error' => 'Method not allowed']);
        exit;
    }

    // Must be logged in
    $userId = (int)($_SESSION['site_user_id'] ?? 0);
    if (!$userId) {
        http_response_code(401);
        echo json_encode(['success' => false, 'error' => 'Not authenticated']);
        exit;
    }

    // Get JSON body or form data
    $input = json_decode(file_get_contents('php://input'), true);
    if (!$input) {
        $input = $_POST;
    }

    $action = $input['action'] ?? '';
    $targetUserId = (int)($input['target_user_id'] ?? 0);
    $requestId = (int)($input['request_id'] ?? 0);
    $message = trim($input['message'] ?? '');

    $result = ['success' => false, 'error' => 'Invalid action'];
    $resolvedTargetUserId = $targetUserId; // Will be set for request-based actions

    switch ($action) {
        case 'send_request':
            if ($targetUserId > 0) {
                $result = FriendService::sendRequest((int)$userId, (int)$targetUserId, $message ?: null);
                if ($result['success']) {
                    $result['message'] = 'Friend request sent!';
                    $result['new_status'] = 'request_sent';
                }
            }
            break;

        case 'accept_request':
            if ($requestId > 0) {
                // Get target user from request before processing
                $reqInfo = DbHelper::querySingle("SELECT from_user_id FROM friend_requests WHERE id = ? AND to_user_id = ?", [$requestId, $userId]);
                if ($reqInfo) {
                    $resolvedTargetUserId = (int)$reqInfo['from_user_id'];
                }
                $result = FriendService::respondToRequest((int)$requestId, (int)$userId, true);
                if ($result['success']) {
                    $result['message'] = 'Friend request accepted!';
                    $result['new_status'] = 'friends';
                }
            }
            break;

        case 'decline_request':
            if ($requestId > 0) {
                // Get target user from request before processing
                $reqInfo = DbHelper::querySingle("SELECT from_user_id FROM friend_requests WHERE id = ? AND to_user_id = ?", [$requestId, $userId]);
                if ($reqInfo) {
                    $resolvedTargetUserId = (int)$reqInfo['from_user_id'];
                }
                $result = FriendService::respondToRequest((int)$requestId, (int)$userId, false);
                if ($result['success']) {
                    $result['message'] = 'Friend request declined.';
                    $result['new_status'] = 'none';
                }
            }
            break;

        case 'cancel_request':
            if ($requestId > 0) {
                // Get target user from request before processing
                $reqInfo = DbHelper::querySingle("SELECT to_user_id FROM friend_requests WHERE id = ? AND from_user_id = ?", [$requestId, $userId]);
                if ($reqInfo) {
                    $resolvedTargetUserId = (int)$reqInfo['to_user_id'];
                }
                if (FriendService::cancelRequest((int)$requestId, (int)$userId)) {
                    $result = ['success' => true, 'message' => 'Friend request cancelled.', 'new_status' => 'none'];
                } else {
                    $result = ['success' => false, 'error' => 'Could not cancel request.'];
                }
            }
            break;

        case 'remove_friend':
            if ($targetUserId > 0) {
                if (FriendService::removeFriend((int)$userId, (int)$targetUserId)) {
                    $result = ['success' => true, 'message' => 'Friend removed.', 'new_status' => 'none'];
                } else {
                    $result = ['success' => false, 'error' => 'Could not remove friend.'];
                }
            }
            break;

        case 'block_user':
            if ($targetUserId > 0) {
                if (FriendService::blockUser((int)$userId, (int)$targetUserId)) {
                    $result = ['success' => true, 'message' => 'User blocked.', 'new_status' => 'blocked'];
                } else {
                    $result = ['success' => false, 'error' => 'Could not block user.'];
                }
            }
            break;

        case 'unblock_user':
            if ($targetUserId > 0) {
                if (FriendService::unblockUser((int)$userId, (int)$targetUserId)) {
                    $result = ['success' => true, 'message' => 'User unblocked.', 'new_status' => 'none'];
                } else {
                    $result = ['success' => false, 'error' => 'Could not unblock user.'];
                }
            }
            break;

        default:
            $result = ['success' => false, 'error' => 'Unknown action.'];
            break;
    }

    // Include target_user_id in response for UI updates
    if ($result['success'] && $resolvedTargetUserId > 0) {
        $result['target_user_id'] = $resolvedTargetUserId;
    }

    echo json_encode($result);

} catch (Throwable $e) {
    if (ob_get_level()) {
        ob_end_clean();
    }

    error_log('Friends action error: ' . $e->getMessage() . ' in ' . $e->getFile() . ':' . $e->getLine());
    header('Content-Type: application/json; charset=utf-8');
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Action failed']);
}
