<?php

use NewSite\Auth\SiteAuth;
use NewSite\Cache\CacheService as Cache;
use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Exception\ForumUploadException;
use NewSite\Forum\ForumHelper;
use NewSite\Settings\SettingsService;
use NewSite\Upload\ImageUrlHelper;
use NewSite\Upload\UploadService;
use NewSite\User\UserService;
use NewSite\Util\SlugGenerator;

header('Content-Type: application/json');

function forumPostResponse(array $data, int $status = 200): void
{
    http_response_code($status);
    echo json_encode($data);
    exit;
}

$method = strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');
if ($method !== 'POST') {
    forumPostResponse(['success' => false, 'error' => 'Method not allowed.'], 405);
}

verifyCSRF();

$forumEnabled = SettingsService::get('forum_enabled', '0') === '1';
if (!$forumEnabled) {
    forumPostResponse(['success' => false, 'error' => 'Forum is disabled.'], 404);
}

if (!SiteAuth::isLoggedIn()) {
    forumPostResponse(['success' => false, 'error' => 'Login required.'], 401);
}

$action = trim((string)($_POST['action'] ?? ''));
if ($action === '' && isset($_POST['title']) && isset($_POST['subcategory_id'])) {
    $action = 'create_post';
}
if (!in_array($action, ['create_post', 'update_post'], true)) {
    forumPostResponse(['success' => false, 'error' => 'Invalid action.'], 400);
}

$title = trim((string)($_POST['title'] ?? ''));
$body = trim((string)($_POST['body'] ?? ''));
$subcategoryId = (int)($_POST['subcategory_id'] ?? 0);

if ($action === 'create_post' && $subcategoryId <= 0) {
    forumPostResponse(['success' => false, 'error' => 'Select a subcategory.'], 422);
}
if ($title === '' || $body === '') {
    forumPostResponse(['success' => false, 'error' => 'Title and post content are required.'], 422);
}

$titleLength = function_exists('mb_strlen') ? mb_strlen($title) : strlen($title);
$bodyLength = function_exists('mb_strlen') ? mb_strlen($body) : strlen($body);
if ($titleLength > 120) {
    forumPostResponse(['success' => false, 'error' => 'Title is too long.'], 422);
}
if ($bodyLength > 5000) {
    forumPostResponse(['success' => false, 'error' => 'Post is too long.'], 422);
}

$userId = (int)($_SESSION['site_user_id'] ?? 0);

$maxImages = 5;
$maxImageSize = 5 * 1024 * 1024;
$uploads = [];
if (isset($_FILES['images'])) {
    $fileField = $_FILES['images'];
    if (is_array($fileField['name'])) {
        foreach ($fileField['name'] as $index => $name) {
            $uploads[] = [
                'name' => $fileField['name'][$index] ?? '',
                'type' => $fileField['type'][$index] ?? '',
                'tmp_name' => $fileField['tmp_name'][$index] ?? '',
                'error' => $fileField['error'][$index] ?? UPLOAD_ERR_NO_FILE,
                'size' => $fileField['size'][$index] ?? 0
            ];
        }
    } else {
        $uploads[] = $fileField;
    }
}

$cleanUploads = [];
foreach ($uploads as $file) {
    $error = $file['error'] ?? UPLOAD_ERR_NO_FILE;
    if ($error === UPLOAD_ERR_NO_FILE) {
        continue;
    }
    if ($error !== UPLOAD_ERR_OK) {
        forumPostResponse(['success' => false, 'error' => 'Image upload failed. Please try again.'], 422);
    }
    if (!empty($file['size']) && (int)$file['size'] > $maxImageSize) {
        forumPostResponse(['success' => false, 'error' => 'Each image must be 5MB or less.'], 422);
    }
    $cleanUploads[] = $file;
    if (count($cleanUploads) > $maxImages) {
        forumPostResponse(['success' => false, 'error' => 'You can upload up to 5 images per post.'], 422);
    }
}

if ($action === 'update_post') {
    $postId = (int)($_POST['post_id'] ?? 0);
    if ($postId <= 0) {
        forumPostResponse(['success' => false, 'error' => 'Invalid post.'], 422);
    }
    $db = DatabaseManager::getWriteConnection();
    $stmt = $db->prepare("SELECT id FROM forum_posts WHERE id = ? AND user_id = ? LIMIT 1");
    $stmt->execute([$postId, $userId]);
    if (!$stmt->fetchColumn()) {
        forumPostResponse(['success' => false, 'error' => 'Not authorized to edit this post.'], 403);
    }

    $stmt = $db->prepare("SELECT id, image_path, sort_order FROM forum_post_images WHERE post_id = ? ORDER BY sort_order ASC, id ASC");
    $stmt->execute([$postId]);
    $existingImages = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $existingIds = array_column($existingImages, 'id');

    $removeIds = $_POST['remove_image_ids'] ?? [];
    if (!is_array($removeIds)) {
        $removeIds = [$removeIds];
    }
    $removeIds = array_values(array_filter(array_map('intval', $removeIds)));
    if (!empty($existingIds)) {
        $removeIds = array_values(array_intersect($removeIds, $existingIds));
    } else {
        $removeIds = [];
    }

    $remainingCount = count($existingImages) - count($removeIds);
    if ($remainingCount < 0) {
        $remainingCount = 0;
    }
    if ($remainingCount + count($cleanUploads) > $maxImages) {
        forumPostResponse(['success' => false, 'error' => 'You can upload up to 5 images per post.'], 422);
    }

    $maxSort = -1;
    foreach ($existingImages as $img) {
        if (!in_array((int)$img['id'], $removeIds, true)) {
            $maxSort = max($maxSort, (int)($img['sort_order'] ?? 0));
        }
    }
    $nextSort = $maxSort < 0 ? 0 : $maxSort + 1;

    $uploadedPaths = [];
    $removePaths = [];
    try {
        $db->beginTransaction();
        $stmt = $db->prepare("UPDATE forum_posts SET title = ?, body = ?, updated_at = ? WHERE id = ?");
        $stmt->execute([$title, $body, DbHelper::nowString(), $postId]);

        if (!empty($removeIds)) {
            foreach ($existingImages as $img) {
                if (in_array((int)$img['id'], $removeIds, true)) {
                    $removePaths[] = $img['image_path'];
                }
            }
            $placeholders = implode(',', array_fill(0, count($removeIds), '?'));
            $stmt = $db->prepare("DELETE FROM forum_post_images WHERE post_id = ? AND id IN ($placeholders)");
            $stmt->execute(array_merge([$postId], $removeIds));
        }

        if (!empty($cleanUploads)) {
            $imageStmt = $db->prepare("INSERT INTO forum_post_images (post_id, image_path, sort_order, created_at) VALUES (?, ?, ?, ?)");
            foreach ($cleanUploads as $file) {
                $upload = UploadService::uploadForumImage($file, $maxImageSize);
                if (!$upload['success']) {
                    throw new ForumUploadException($upload['error'] ?? 'Image upload failed.');
                }
                $imageStmt->execute([$postId, $upload['path'], $nextSort, DbHelper::nowString()]);
                $uploadedPaths[] = $upload['path'];
                $nextSort++;
            }
        }

        $db->commit();
    } catch (Throwable $e) {
        if ($db->inTransaction()) {
            $db->rollBack();
        }
        foreach ($uploadedPaths as $path) {
            $fullPath = DATA_PATH . '/' . ltrim($path, '/');
            if (is_file($fullPath)) {
                @unlink($fullPath);
            }
        }
        forumPostResponse(['success' => false, 'error' => 'Unable to update post. Please try again.'], 500);
    }

    foreach ($removePaths as $path) {
        $fullPath = DATA_PATH . '/' . ltrim((string)$path, '/');
        if (is_file($fullPath)) {
            @unlink($fullPath);
        }
    }

    $stmt = $db->prepare("SELECT id, image_path FROM forum_post_images WHERE post_id = ? ORDER BY sort_order ASC, id ASC");
    $stmt->execute([$postId]);
    $updatedImages = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $imagesHtml = '';
    foreach ($updatedImages as $img) {
        $imageUrl = ImageUrlHelper::getForumImageUrl($img['image_path'] ?? '');
        if (!$imageUrl) {
            continue;
        }
        $imagesHtml .= '<button type="button" class="forum-post-image" data-forum-lightbox="' . htmlspecialchars($imageUrl, ENT_QUOTES, 'UTF-8') . '"><img src="' . htmlspecialchars($imageUrl, ENT_QUOTES, 'UTF-8') . '" alt="Post attachment"></button>';
    }

    if (!empty($updatedImages)) {
        $imagesEditHtml = '<div class="forum-image-edit-grid">';
        foreach ($updatedImages as $img) {
            $imageUrl = ImageUrlHelper::getForumImageUrl($img['image_path'] ?? '');
            if (!$imageUrl) {
                continue;
            }
            $imagesEditHtml .= '<label class="forum-image-edit-item">';
            $imagesEditHtml .= '<input type="checkbox" name="remove_image_ids[]" value="' . (int)$img['id'] . '">';
            $imagesEditHtml .= '<span class="forum-image-edit-thumb"><img src="' . htmlspecialchars($imageUrl, ENT_QUOTES, 'UTF-8') . '" alt="Post attachment"></span>';
            $imagesEditHtml .= '<span>Remove</span>';
            $imagesEditHtml .= '</label>';
        }
        $imagesEditHtml .= '</div>';
    } else {
        $imagesEditHtml = '<div class="forum-image-edit-empty">No images yet.</div>';
    }

    forumPostResponse([
        'success' => true,
        'post_id' => $postId,
        'title' => $title,
        'body' => $body,
        'images_html' => $imagesHtml,
        'images_edit_html' => $imagesEditHtml,
        'message' => 'Post updated.'
    ]);
}

$rateKey = 'forum_post_rl:u:' . $userId;
$now = time();
$rateData = Cache::get($rateKey, null);
if (!is_array($rateData)) {
    $rateData = ['count' => 0, 'window_start' => $now];
}

$window = 120;
$limit = 3;
if ($now - (int)$rateData['window_start'] >= $window) {
    $rateData = ['count' => 0, 'window_start' => $now];
}

if ((int)$rateData['count'] >= $limit) {
    $retryAfter = max(1, $window - ($now - (int)$rateData['window_start']));
    forumPostResponse([
        'success' => false,
        'error' => 'Please wait before posting again.',
        'rate_limited' => true,
        'retry_after' => $retryAfter
    ], 429);
}

$rateData['count'] = (int)$rateData['count'] + 1;
Cache::set($rateKey, $rateData, $window + 10);

$db = DatabaseManager::getWriteConnection();
$stmt = $db->prepare("SELECT s.id, s.name, s.slug AS sub_slug, c.id AS category_id, c.name AS category_name, c.slug AS cat_slug FROM forum_subcategories s JOIN forum_categories c ON c.id = s.category_id WHERE s.id = ? AND s.is_active = 1 AND c.is_active = 1 LIMIT 1");
$stmt->execute([$subcategoryId]);
$subcategory = $stmt->fetch();
if (!$subcategory) {
    forumPostResponse(['success' => false, 'error' => 'Invalid subcategory.'], 404);
}

$status = SettingsService::get('forum_auto_approve', '0') === '1' ? 'approved' : 'pending';
$nowDb = DbHelper::nowString();
$uploadedPaths = [];
$postId = 0;
try {
    $db->beginTransaction();

    // Generate unique slug from title for SEO-friendly URLs
    $postSlug = SlugGenerator::generate($title);
    if ($postSlug === '') {
        $postSlug = 'post';
    }
    $candidateSlug = $postSlug;
    $slugSuffix = 2;
    $slugCheckStmt = $db->prepare("SELECT id FROM forum_posts WHERE slug = ? LIMIT 1");
    while (true) {
        $slugCheckStmt->execute([$candidateSlug]);
        if (!$slugCheckStmt->fetch()) {
            break;
        }
        $candidateSlug = $postSlug . '-' . $slugSuffix;
        $slugSuffix++;
    }
    $postSlug = $candidateSlug;

    $stmt = $db->prepare("INSERT INTO forum_posts (subcategory_id, user_id, title, body, slug, status, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
    $stmt->execute([$subcategoryId, $userId, $title, $body, $postSlug, $status, $nowDb, $nowDb]);
    $postId = (int)$db->lastInsertId();

    if (!empty($cleanUploads)) {
        $imageStmt = $db->prepare("INSERT INTO forum_post_images (post_id, image_path, sort_order, created_at) VALUES (?, ?, ?, ?)");
        $sortOrder = 0;
        foreach ($cleanUploads as $file) {
            $upload = UploadService::uploadForumImage($file, $maxImageSize);
            if (!$upload['success']) {
                throw new ForumUploadException($upload['error'] ?? 'Image upload failed.');
            }
            $imageStmt->execute([$postId, $upload['path'], $sortOrder, $nowDb]);
            $uploadedPaths[] = $upload['path'];
            $sortOrder++;
        }
    }

    $db->commit();
} catch (Throwable $e) {
    if ($db->inTransaction()) {
        $db->rollBack();
    }
    foreach ($uploadedPaths as $path) {
        $fullPath = DATA_PATH . '/' . ltrim($path, '/');
        if (is_file($fullPath)) {
            @unlink($fullPath);
        }
    }
    forumPostResponse(['success' => false, 'error' => 'Unable to create post. Please try again.'], 500);
}

$postHtml = '';
if ($status === 'approved') {
    $user = UserService::getById((int)$userId);
    $displayName = trim((string)($user['nickname'] ?? ''));
    if ($displayName === '') {
        $displayName = trim((string)($user['display_name'] ?? ''));
    }
    if ($displayName === '') {
        $displayName = 'User';
    }
    $profilePhoto = trim((string)($user['profile_photo'] ?? ''));
    $avatarUrl = $profilePhoto !== '' ? ForumHelper::toPublicFilePath($profilePhoto) : '/assets/images/default-avatar.svg';
    $createdAt = date('M j, Y', strtotime($nowDb));
    $excerpt = trim(strip_tags($body));
    if (function_exists('mb_strlen')) {
        if (mb_strlen($excerpt) > 160) {
            $excerpt = mb_substr($excerpt, 0, 157) . '...';
        }
    } elseif (strlen($excerpt) > 160) {
        $excerpt = substr($excerpt, 0, 157) . '...';
    }

    $categoryId = (int)$subcategory['category_id'];
    $categoryName = (string)($subcategory['category_name'] ?? 'Category');
    $subcategoryName = (string)($subcategory['name'] ?? 'Subcategory');
    $viewCount = 0;

    ob_start();
    ?>
    <div class="forum-post">
        <a class="forum-post-link" href="/forum/post/<?php echo htmlspecialchars($postSlug, ENT_QUOTES, 'UTF-8'); ?>" aria-label="Open post"></a>
        <div class="forum-post-avatar">
            <img src="<?php echo htmlspecialchars($avatarUrl, ENT_QUOTES, 'UTF-8'); ?>" alt="<?php echo htmlspecialchars($displayName, ENT_QUOTES, 'UTF-8'); ?>">
        </div>
        <div class="forum-post-main">
            <div class="forum-post-title">
                <?php echo htmlspecialchars($title, ENT_QUOTES, 'UTF-8'); ?>
            </div>
            <div class="forum-post-meta">
                <span>By <?php echo htmlspecialchars($displayName, ENT_QUOTES, 'UTF-8'); ?></span>
                <span>· <?php echo htmlspecialchars($createdAt, ENT_QUOTES, 'UTF-8'); ?></span>
                <span>· 0 replies</span>
                <span>· <?php echo (int)$viewCount; ?> views</span>
            </div>
            <?php if ($excerpt !== ''): ?>
                <div class="forum-post-excerpt">
                    <?php echo htmlspecialchars($excerpt, ENT_QUOTES, 'UTF-8'); ?>
                </div>
            <?php endif; ?>
            <div class="forum-post-tags">
                <a href="/forum/category/<?php echo htmlspecialchars(rawurlencode(trim((string)($subcategory['cat_slug'] ?? ''))), ENT_QUOTES, 'UTF-8'); ?>" class="forum-tag">
                    <?php echo htmlspecialchars($categoryName, ENT_QUOTES, 'UTF-8'); ?>
                </a>
                <a href="/forum/subcategory/<?php echo htmlspecialchars(rawurlencode(trim((string)($subcategory['sub_slug'] ?? ''))), ENT_QUOTES, 'UTF-8'); ?>" class="forum-tag">
                    <?php echo htmlspecialchars($subcategoryName, ENT_QUOTES, 'UTF-8'); ?>
                </a>
            </div>
        </div>
    </div>
    <?php
    $postHtml = trim(ob_get_clean());
}

$message = $status === 'approved'
    ? 'Post published successfully.'
    : 'Post submitted and awaiting approval.';

forumPostResponse([
    'success' => true,
    'post_id' => $postId,
    'status' => $status,
    'message' => $message,
    'post_html' => $postHtml
]);
