<?php

use NewSite\Auth\SiteAuth;
use NewSite\Cache\CacheService as Cache;
use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Exception\ForumUploadException;
use NewSite\Settings\SettingsService;
use NewSite\Upload\ImageUrlHelper;
use NewSite\Upload\UploadService;
use NewSite\User\UserService;

header('Content-Type: application/json');

function forumCommentResponse(array $data, int $status = 200): void
{
    http_response_code($status);
    echo json_encode($data);
    exit;
}

$method = strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');
if ($method !== 'POST') {
    forumCommentResponse(['success' => false, 'error' => 'Method not allowed.'], 405);
}

verifyCSRF();

$forumEnabled = SettingsService::get('forum_enabled', '0') === '1';
if (!$forumEnabled) {
    forumCommentResponse(['success' => false, 'error' => 'Forum is disabled.'], 404);
}

if (!SiteAuth::isLoggedIn()) {
    forumCommentResponse(['success' => false, 'error' => 'Login required.'], 401);
}

$action = trim((string)($_POST['action'] ?? ''));
if ($action === '' && isset($_POST['post_id'])) {
    $action = 'create_comment';
}
if (!in_array($action, ['create_comment', 'update_comment'], true)) {
    forumCommentResponse(['success' => false, 'error' => 'Invalid action.'], 400);
}

$postId = (int)($_POST['post_id'] ?? 0);
$body = trim((string)($_POST['body'] ?? ''));
$userId = (int)($_SESSION['site_user_id'] ?? 0);

if ($action === 'create_comment' && $postId <= 0) {
    forumCommentResponse(['success' => false, 'error' => 'Invalid post.'], 400);
}
$maxImages = 5;
$maxImageSize = 5 * 1024 * 1024;
$uploads = [];
if (isset($_FILES['comment_images'])) {
    $fileField = $_FILES['comment_images'];
    if (is_array($fileField['name'])) {
        foreach ($fileField['name'] as $index => $name) {
            $uploads[] = [
                'name' => $fileField['name'][$index] ?? '',
                'type' => $fileField['type'][$index] ?? '',
                'tmp_name' => $fileField['tmp_name'][$index] ?? '',
                'error' => $fileField['error'][$index] ?? UPLOAD_ERR_NO_FILE,
                'size' => $fileField['size'][$index] ?? 0
            ];
        }
    } else {
        $uploads[] = $fileField;
    }
}

$cleanUploads = [];
foreach ($uploads as $file) {
    $error = $file['error'] ?? UPLOAD_ERR_NO_FILE;
    if ($error === UPLOAD_ERR_NO_FILE) {
        continue;
    }
    if ($error !== UPLOAD_ERR_OK) {
        forumCommentResponse(['success' => false, 'error' => 'Image upload failed. Please try again.'], 422);
    }
    if (!empty($file['size']) && (int)$file['size'] > $maxImageSize) {
        forumCommentResponse(['success' => false, 'error' => 'Each image must be 5MB or less.'], 422);
    }
    $cleanUploads[] = $file;
    if (count($cleanUploads) > $maxImages) {
        forumCommentResponse(['success' => false, 'error' => 'You can upload up to 5 images per reply.'], 422);
    }
}

$bodyLength = function_exists('mb_strlen') ? mb_strlen($body) : strlen($body);
if ($bodyLength > 2000) {
    forumCommentResponse(['success' => false, 'error' => 'Reply is too long.'], 422);
}

if ($action === 'update_comment') {
    $commentId = (int)($_POST['comment_id'] ?? 0);
    if ($commentId <= 0) {
        forumCommentResponse(['success' => false, 'error' => 'Invalid comment.'], 422);
    }

    $db = DatabaseManager::getWriteConnection();
    $stmt = $db->prepare("SELECT id FROM forum_comments WHERE id = ? AND user_id = ? LIMIT 1");
    $stmt->execute([$commentId, $userId]);
    if (!$stmt->fetchColumn()) {
        forumCommentResponse(['success' => false, 'error' => 'Not authorized to edit this reply.'], 403);
    }

    $stmt = $db->prepare("SELECT id, image_path, sort_order FROM forum_comment_images WHERE comment_id = ? ORDER BY sort_order ASC, id ASC");
    $stmt->execute([$commentId]);
    $existingImages = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $existingIds = array_column($existingImages, 'id');

    $removeIds = $_POST['remove_comment_image_ids'] ?? [];
    if (!is_array($removeIds)) {
        $removeIds = [$removeIds];
    }
    $removeIds = array_values(array_filter(array_map('intval', $removeIds)));
    if (!empty($existingIds)) {
        $removeIds = array_values(array_intersect($removeIds, $existingIds));
    } else {
        $removeIds = [];
    }

    $remainingCount = count($existingImages) - count($removeIds);
    if ($remainingCount < 0) {
        $remainingCount = 0;
    }
    if ($remainingCount + count($cleanUploads) > $maxImages) {
        forumCommentResponse(['success' => false, 'error' => 'You can upload up to 5 images per reply.'], 422);
    }
    if ($body === '' && ($remainingCount + count($cleanUploads)) === 0) {
        forumCommentResponse(['success' => false, 'error' => 'Reply cannot be empty.'], 422);
    }

    $maxSort = -1;
    foreach ($existingImages as $img) {
        if (!in_array((int)$img['id'], $removeIds, true)) {
            $maxSort = max($maxSort, (int)($img['sort_order'] ?? 0));
        }
    }
    $nextSort = $maxSort < 0 ? 0 : $maxSort + 1;

    $uploadedPaths = [];
    $removePaths = [];
    try {
        $db->beginTransaction();
        $stmt = $db->prepare("UPDATE forum_comments SET body = ? WHERE id = ?");
        $stmt->execute([$body, $commentId]);

        if (!empty($removeIds)) {
            foreach ($existingImages as $img) {
                if (in_array((int)$img['id'], $removeIds, true)) {
                    $removePaths[] = $img['image_path'];
                }
            }
            $placeholders = implode(',', array_fill(0, count($removeIds), '?'));
            $stmt = $db->prepare("DELETE FROM forum_comment_images WHERE comment_id = ? AND id IN ($placeholders)");
            $stmt->execute(array_merge([$commentId], $removeIds));
        }

        if (!empty($cleanUploads)) {
            $imageStmt = $db->prepare("INSERT INTO forum_comment_images (comment_id, image_path, sort_order, created_at) VALUES (?, ?, ?, ?)");
            foreach ($cleanUploads as $file) {
                $upload = UploadService::uploadForumImage($file, $maxImageSize);
                if (!$upload['success']) {
                    throw new ForumUploadException($upload['error'] ?? 'Image upload failed.');
                }
                $imageStmt->execute([$commentId, $upload['path'], $nextSort, DbHelper::nowString()]);
                $uploadedPaths[] = $upload['path'];
                $nextSort++;
            }
        }

        $db->commit();
    } catch (Throwable $e) {
        if ($db->inTransaction()) {
            $db->rollBack();
        }
        foreach ($uploadedPaths as $path) {
            $fullPath = DATA_PATH . '/' . ltrim($path, '/');
            if (is_file($fullPath)) {
                @unlink($fullPath);
            }
        }
        forumCommentResponse(['success' => false, 'error' => 'Unable to update reply. Please try again.'], 500);
    }

    foreach ($removePaths as $path) {
        $fullPath = DATA_PATH . '/' . ltrim((string)$path, '/');
        if (is_file($fullPath)) {
            @unlink($fullPath);
        }
    }

    $stmt = $db->prepare("SELECT id, image_path FROM forum_comment_images WHERE comment_id = ? ORDER BY sort_order ASC, id ASC");
    $stmt->execute([$commentId]);
    $updatedImages = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $imagesHtml = '';
    foreach ($updatedImages as $img) {
        $imageUrl = ImageUrlHelper::getForumImageUrl($img['image_path'] ?? '');
        if (!$imageUrl) {
            continue;
        }
        $imagesHtml .= '<button type="button" class="forum-comment-image" data-forum-lightbox="' . htmlspecialchars($imageUrl, ENT_QUOTES, 'UTF-8') . '"><img src="' . htmlspecialchars($imageUrl, ENT_QUOTES, 'UTF-8') . '" alt="Reply attachment"></button>';
    }

    if (!empty($updatedImages)) {
        $imagesEditHtml = '<div class="forum-image-edit-grid">';
        foreach ($updatedImages as $img) {
            $imageUrl = ImageUrlHelper::getForumImageUrl($img['image_path'] ?? '');
            if (!$imageUrl) {
                continue;
            }
            $imagesEditHtml .= '<label class="forum-image-edit-item">';
            $imagesEditHtml .= '<input type="checkbox" name="remove_comment_image_ids[]" value="' . (int)$img['id'] . '">';
            $imagesEditHtml .= '<span class="forum-image-edit-thumb"><img src="' . htmlspecialchars($imageUrl, ENT_QUOTES, 'UTF-8') . '" alt="Reply attachment"></span>';
            $imagesEditHtml .= '<span>Remove</span>';
            $imagesEditHtml .= '</label>';
        }
        $imagesEditHtml .= '</div>';
    } else {
        $imagesEditHtml = '<div class="forum-image-edit-empty">No images yet.</div>';
    }

    forumCommentResponse([
        'success' => true,
        'comment_id' => $commentId,
        'body' => $body,
        'images_html' => $imagesHtml,
        'images_edit_html' => $imagesEditHtml,
        'message' => 'Reply updated.'
    ]);
}

if ($body === '' && empty($cleanUploads)) {
    forumCommentResponse(['success' => false, 'error' => 'Reply cannot be empty.'], 422);
}

$rateKey = 'forum_comment_rl:u:' . $userId;
$now = time();
$rateData = Cache::get($rateKey, null);
if (!is_array($rateData)) {
    $rateData = ['count' => 0, 'window_start' => $now];
}

$window = 60;
$limit = 6;
if ($now - (int)$rateData['window_start'] >= $window) {
    $rateData = ['count' => 0, 'window_start' => $now];
}

if ((int)$rateData['count'] >= $limit) {
    $retryAfter = max(1, $window - ($now - (int)$rateData['window_start']));
    forumCommentResponse([
        'success' => false,
        'error' => 'Please slow down and try again.',
        'rate_limited' => true,
        'retry_after' => $retryAfter
    ], 429);
}

$rateData['count'] = (int)$rateData['count'] + 1;
Cache::set($rateKey, $rateData, $window + 10);

$db = DatabaseManager::getWriteConnection();
$stmt = $db->prepare("SELECT p.id FROM forum_posts p JOIN forum_subcategories s ON s.id = p.subcategory_id AND s.is_active = 1 JOIN forum_categories c ON c.id = s.category_id AND c.is_active = 1 WHERE p.id = ? AND p.status = 'approved' LIMIT 1");
$stmt->execute([$postId]);
if (!$stmt->fetchColumn()) {
    forumCommentResponse(['success' => false, 'error' => 'Post not found.'], 404);
}

$nowDb = DbHelper::nowString();
$commentId = 0;
$uploadedPaths = [];
$uploadedImages = [];
try {
    $db->beginTransaction();
    $stmt = $db->prepare("INSERT INTO forum_comments (post_id, user_id, body, status, created_at) VALUES (?, ?, ?, 'approved', ?)");
    $stmt->execute([$postId, $userId, $body, $nowDb]);
    $commentId = (int)$db->lastInsertId();

    if (!empty($cleanUploads)) {
        $imageStmt = $db->prepare("INSERT INTO forum_comment_images (comment_id, image_path, sort_order, created_at) VALUES (?, ?, ?, ?)");
        $sortOrder = 0;
        foreach ($cleanUploads as $file) {
            $upload = UploadService::uploadForumImage($file, $maxImageSize);
            if (!$upload['success']) {
                throw new ForumUploadException($upload['error'] ?? 'Image upload failed.');
            }
            $imageStmt->execute([$commentId, $upload['path'], $sortOrder, $nowDb]);
            $imageId = (int)$db->lastInsertId();
            $uploadedPaths[] = $upload['path'];
            $uploadedImages[] = ['id' => $imageId, 'path' => $upload['path']];
            $sortOrder++;
        }
    }

    $db->commit();
} catch (Throwable $e) {
    if ($db->inTransaction()) {
        $db->rollBack();
    }
    foreach ($uploadedPaths as $path) {
        $fullPath = DATA_PATH . '/' . ltrim($path, '/');
        if (is_file($fullPath)) {
            @unlink($fullPath);
        }
    }
    forumCommentResponse(['success' => false, 'error' => 'Unable to post reply. Please try again.'], 500);
}

$user = UserService::getById((int)$userId);
$displayName = trim((string)($user['nickname'] ?? ''));
if ($displayName === '') {
    $displayName = trim((string)($user['display_name'] ?? ''));
}
if ($displayName === '') {
    $displayName = 'User';
}
$nickname = trim((string)($user['nickname'] ?? ''));
$profileLink = $nickname !== '' ? '/@' . rawurlencode($nickname) : '';
$profilePhoto = trim((string)($user['profile_photo'] ?? ''));
$avatarUrl = $profilePhoto !== '' ? str_replace('/admin-file/', '/site-file/', $profilePhoto) : '/assets/images/default-avatar.svg';
$createdAt = date('M j, Y', strtotime($nowDb));

$stmt = $db->prepare("SELECT COUNT(*) FROM forum_comments WHERE post_id = ? AND status = 'approved'");
$stmt->execute([$postId]);
$repliesCount = (int)$stmt->fetchColumn();

ob_start();
?>
<div class="forum-comment" data-forum-comment data-forum-comment-id="<?php echo (int)$commentId; ?>">
    <div class="forum-comment-avatar">
        <img src="<?php echo htmlspecialchars($avatarUrl, ENT_QUOTES, 'UTF-8'); ?>" alt="<?php echo htmlspecialchars($displayName, ENT_QUOTES, 'UTF-8'); ?>">
    </div>
    <div class="forum-comment-main">
        <div class="forum-comment-meta">
            <?php if ($profileLink !== ''): ?>
                <a class="forum-comment-author" href="<?php echo htmlspecialchars($profileLink, ENT_QUOTES, 'UTF-8'); ?>"><?php echo htmlspecialchars($displayName, ENT_QUOTES, 'UTF-8'); ?></a>
            <?php else: ?>
                <span><?php echo htmlspecialchars($displayName, ENT_QUOTES, 'UTF-8'); ?></span>
            <?php endif; ?>
            <span>· <?php echo htmlspecialchars($createdAt, ENT_QUOTES, 'UTF-8'); ?></span>
            <button type="button" class="btn btn-outline btn-sm" data-forum-comment-edit>Edit</button>
        </div>
        <?php if ($body !== ''): ?>
            <div class="forum-comment-body" data-forum-comment-body>
                <?php echo nl2br(htmlspecialchars($body, ENT_QUOTES, 'UTF-8')); ?>
            </div>
        <?php endif; ?>
        <form class="forum-comment-edit-form is-hidden" data-forum-comment-edit-form enctype="multipart/form-data">
            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
            <input type="hidden" name="action" value="update_comment">
            <input type="hidden" name="comment_id" value="<?php echo (int)$commentId; ?>">
            <textarea name="body" rows="3" maxlength="2000" required><?php echo htmlspecialchars($body, ENT_QUOTES, 'UTF-8'); ?></textarea>
            <div class="forum-image-edit-list" data-forum-comment-images-edit>
                <?php if (!empty($uploadedImages)): ?>
                    <div class="forum-image-edit-grid">
                        <?php foreach ($uploadedImages as $imageRow):
                            $imageUrl = ImageUrlHelper::getForumImageUrl($imageRow['path'] ?? '');
                            if (!$imageUrl) {
                                continue;
                            }
                            ?>
                            <label class="forum-image-edit-item">
                                <input type="checkbox" name="remove_comment_image_ids[]" value="<?php echo (int)$imageRow['id']; ?>">
                                <span class="forum-image-edit-thumb"><img src="<?php echo htmlspecialchars($imageUrl, ENT_QUOTES, 'UTF-8'); ?>" alt="Reply attachment"></span>
                                <span>Remove</span>
                            </label>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div class="forum-image-edit-empty">No images yet.</div>
                <?php endif; ?>
            </div>
            <div class="forum-field">
                <label for="comment-images-upload">Add images</label>
                <input id="comment-images-upload" name="comment_images[]" type="file" accept="image/*" multiple class="form-control-file">
                <div class="forum-upload-help">Up to 5 images total, max 5MB each.</div>
            </div>
            <div class="forum-form-message" data-forum-comment-edit-message></div>
            <div class="forum-form-actions">
                <button type="button" class="btn btn-outline btn-sm" data-forum-comment-edit-cancel>Cancel</button>
                <button type="submit" class="btn btn-primary btn-sm" data-forum-comment-edit-save>Save</button>
            </div>
        </form>
        <?php if (!empty($uploadedPaths)): ?>
            <div class="forum-comment-images">
                <?php foreach ($uploadedPaths as $imagePath):
                    $imageUrl = ImageUrlHelper::getForumImageUrl($imagePath);
                    if (!$imageUrl) {
                        continue;
                    }
                    ?>
                    <button type="button" class="forum-comment-image" data-forum-lightbox="<?php echo htmlspecialchars($imageUrl, ENT_QUOTES, 'UTF-8'); ?>">
                        <img src="<?php echo htmlspecialchars($imageUrl, ENT_QUOTES, 'UTF-8'); ?>" alt="Reply attachment">
                    </button>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>
<?php
$commentHtml = trim(ob_get_clean());

forumCommentResponse([
    'success' => true,
    'comment_id' => $commentId,
    'replies_count' => $repliesCount,
    'comment_html' => $commentHtml,
    'message' => 'Reply posted.'
]);
