<?php

use NewSite\Auth\SiteAuth;
use NewSite\Config\SetupService;
use NewSite\Email\TransactionalEmail;
use NewSite\Http\StripeClient;
use NewSite\Settings\SettingsService;
use NewSite\Shop\CartService;
use NewSite\Shop\CurrencyService;
use NewSite\Shop\OrderService;

header('Content-Type: application/json');

function checkoutJsonResponse(array $data, int $status = 200): void
{
    http_response_code($status);
    echo json_encode($data);
    exit;
}

if (!SiteAuth::isLoggedIn()) {
    checkoutJsonResponse(['success' => false, 'error' => 'Login required.'], 401);
}

$cartId = CartService::getCartId();
$items = CartService::fetchItems($cartId);
if (empty($items)) {
    checkoutJsonResponse(['success' => false, 'error' => 'Cart is empty.'], 400);
}

$currency = strtolower(CurrencyService::getCurrent());
$baseCurrency = SettingsService::get('store_currency', 'USD');
$currentCurrency = CurrencyService::getCurrent();
$userId = (int)($_SESSION['site_user_id'] ?? 0);
$baseUrl = SetupService::getBaseUrl();

/* ── Calculate cart total and build Stripe line items ── */
$cartTotal = 0;
$lineItems = [];
foreach ($items as $item) {
    $priceRaw = $item['variant_price'] !== '' && $item['variant_price'] !== null
        ? $item['variant_price']
        : $item['product_price'];
    $unitPrice = CurrencyService::parsePrice((string)$priceRaw);
    $convertedUnit = CurrencyService::convert($unitPrice, $baseCurrency, $currentCurrency);
    $unitAmount = (int)round($convertedUnit * 100);
    $cartTotal += $unitAmount * (int)$item['quantity'];

    /* Only add paid items to the Stripe payload; free items are
       still part of the order via OrderService::createFromCart(). */
    if ($unitAmount > 0) {
        $lineItems[] = [
            'price_data' => [
                'currency' => $currency,
                'product_data' => [
                    'name' => (string)($item['product_name'] ?? 'Product')
                ],
                'unit_amount' => $unitAmount
            ],
            'quantity' => (int)$item['quantity']
        ];
    }
}

/* ── Free order: skip Stripe and fulfil immediately ── */
if ($cartTotal <= 0) {
    $orderId = OrderService::createFromCart($userId, $cartId, [
        'status'                => 'paid',
        'payment_provider'      => 'free',
        'stripe_session_id'     => null,
        'stripe_payment_intent' => null,
    ]);
    if (!$orderId) {
        checkoutJsonResponse(['success' => false, 'error' => 'Order could not be created.'], 500);
    }
    TransactionalEmail::sendPurchaseReceipt($orderId);
    checkoutJsonResponse(['success' => true, 'url' => $baseUrl . '/checkout-success?order_id=' . $orderId]);
}

/* ── Paid order: Stripe checkout session ── */
$stripeSecretKey = SettingsService::get('stripe_secret_key', '');
if ($stripeSecretKey === '') {
    checkoutJsonResponse(['success' => false, 'error' => 'Stripe is not configured.'], 400);
}

$sessionResponse = StripeClient::request('POST', '/v1/checkout/sessions', [
    'mode' => 'payment',
    'success_url' => $baseUrl . '/checkout-success?session_id={CHECKOUT_SESSION_ID}',
    'cancel_url' => $baseUrl . '/checkout',
    'line_items' => $lineItems,
    'metadata' => [
        'cart_id' => (string)$cartId,
        'user_id' => (string)$userId,
    ]
], $stripeSecretKey);

if (!empty($sessionResponse['error']) || $sessionResponse['status'] >= 400) {
    $message = $sessionResponse['data']['error']['message'] ?? 'Unable to start checkout.';
    checkoutJsonResponse(['success' => false, 'error' => $message], 400);
}

$sessionUrl = $sessionResponse['data']['url'] ?? '';
if ($sessionUrl === '') {
    checkoutJsonResponse(['success' => false, 'error' => 'Stripe response missing URL.'], 400);
}

checkoutJsonResponse(['success' => true, 'url' => $sessionUrl]);
