<?php

use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Logging\LogService;
use NewSite\Settings\SettingsService;
use NewSite\Shop\CartService;
use NewSite\Shop\CurrencyService;

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

header('Content-Type: application/json');

$db = DatabaseManager::getWriteConnection();
$action = $_POST['action'] ?? $_GET['action'] ?? 'get';
$cartId = CartService::getCartId();
$currencySymbol = CurrencyService::getSymbol();
$baseCurrency = SettingsService::get('store_currency', 'USD');
$currentCurrency = CurrencyService::getCurrent();

function cartJsonResponse(array $data, int $status = 200): void
{
    http_response_code($status);
    echo json_encode($data);
    exit;
}

function resolveItemPrice(array $item): string
{
    $price = $item['variant_price'] !== '' && $item['variant_price'] !== null
        ? $item['variant_price']
        : $item['product_price'];
    return (string)$price;
}

function buildCartPayload(int $cartId, string $symbol): array
{
    $items = CartService::fetchItems($cartId);
    $total = 0;
    $payloadItems = [];
    $baseCurrency = SettingsService::get('store_currency', 'USD');
    $currentCurrency = CurrencyService::getCurrent();

    foreach ($items as $item) {
        $rawPrice = resolveItemPrice($item);
        $formatted = CurrencyService::formatValue($rawPrice, $currentCurrency);
        $numeric = $rawPrice !== '' ? (float)preg_replace('/[^0-9.]/', '', $rawPrice) : 0;
        $line = $numeric * (int)$item['quantity'];
        $convertedLine = CurrencyService::convert($line, $baseCurrency, $currentCurrency);
        $total += $convertedLine;
        $quantityMax = $item['quantity_max'];

        // Check if this is a paid digital product
        $actionType = $item['variant_action_type'] ?? $item['product_action_type'] ?? 'cart';
        $isDigital = $actionType === 'download' && $numeric > 0;

        // Build product URL: /{page_slug}/{product_slug}
        $slug = trim((string)($item['product_slug'] ?? ''));
        $pageSlugVal = trim((string)($item['page_slug'] ?? ''));
        $productUrl = '';
        if ($slug !== '') {
            $pagePrefix = $pageSlugVal !== '' ? $pageSlugVal : 'product';
            $productUrl = '/' . $pagePrefix . '/' . rawurlencode($slug);
        }

        $payloadItems[] = [
            'id' => (int)$item['id'],
            'product_id' => (int)$item['product_id'],
            'variant_id' => $item['variant_id'] !== null ? (int)$item['variant_id'] : null,
            'name' => $item['product_name'],
            'variant_label' => $item['variant_label'] ?? '',
            'media_url' => toPublicFilePath($item['media_url'] ?? ''),
            'product_url' => $productUrl,
            'quantity' => (int)$item['quantity'],
            'quantity_max' => $quantityMax === null ? null : (int)$quantityMax,
            'price' => $formatted,
            'line_total' => $symbol . number_format($convertedLine, 2),
            'is_digital' => $isDigital
        ];
    }

    return [
        'items' => $payloadItems,
        'total' => $symbol . number_format($total, 2),
        'count' => count($payloadItems)
    ];
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    cartJsonResponse(['success' => false, 'message' => 'Method not allowed.'], 405);
}

try {
    switch ($action) {
        case 'add':
            $productId = (int)($_POST['product_id'] ?? 0);
            $variantId = null;
            if (array_key_exists('variant_id', $_POST)) {
                $variantRaw = trim((string)($_POST['variant_id'] ?? ''));
                if ($variantRaw !== '') {
                    $variantId = (int)$variantRaw;
                    if ($variantId <= 0) {
                        cartJsonResponse(['success' => false, 'message' => 'Invalid variant.'], 400);
                    }
                }
            }
            $quantity = max(1, (int)($_POST['quantity'] ?? 1));

            if (!$productId) {
                cartJsonResponse(['success' => false, 'message' => 'Invalid product.'], 400);
            }

            $stmt = $db->prepare("SELECT id, stock, quantity_max FROM products WHERE id = ? AND is_active = 1");
            $stmt->execute([$productId]);
            $product = $stmt->fetch();
            if (!$product) {
                cartJsonResponse(['success' => false, 'message' => 'Product not found.'], 404);
            }

            $stock = $product['stock'];
            $quantityMax = $product['quantity_max'];
            $variant = null;
            if ($variantId !== null) {
                $stmt = $db->prepare("SELECT id, stock FROM product_variants WHERE id = ? AND product_id = ?");
                $stmt->execute([$variantId, $productId]);
                $variant = $stmt->fetch();
                if (!$variant) {
                    cartJsonResponse(['success' => false, 'message' => 'Variant not found.'], 404);
                }
                $stock = $variant['stock'];
            }

            if ($stock !== null && (int)$stock <= 0) {
                cartJsonResponse(['success' => false, 'message' => 'Out of stock.'], 409);
            }

            if ($quantityMax !== null && $quantityMax > 0) {
                $quantity = min($quantity, (int)$quantityMax);
            }

            if ($variantId === null) {
                $stmt = $db->prepare("SELECT id, quantity FROM cart_items WHERE cart_id = ? AND product_id = ? AND variant_id IS NULL");
                $stmt->execute([$cartId, $productId]);
            } else {
                $stmt = $db->prepare("SELECT id, quantity FROM cart_items WHERE cart_id = ? AND product_id = ? AND variant_id = ?");
                $stmt->execute([$cartId, $productId, $variantId]);
            }
            $existing = $stmt->fetch();

            if ($existing) {
                $newQty = (int)$existing['quantity'] + $quantity;
                if ($quantityMax !== null && $quantityMax > 0) {
                    $newQty = min($newQty, (int)$quantityMax);
                }
                if ($stock !== null) {
                    $newQty = min($newQty, (int)$stock);
                }
                $now = DbHelper::nowString();
                $stmt = $db->prepare("UPDATE cart_items SET quantity = ?, updated_at = ? WHERE id = ?");
                $stmt->execute([$newQty, $now, $existing['id']]);
            } else {
                $stmt = $db->prepare("INSERT INTO cart_items (cart_id, product_id, variant_id, quantity) VALUES (?, ?, ?, ?)");
                $stmt->execute([$cartId, $productId, $variantId, $quantity]);
            }

            cartJsonResponse(['success' => true] + buildCartPayload($cartId, $currencySymbol));
            break;

        case 'update':
            $itemId = (int)($_POST['item_id'] ?? 0);
            $quantity = (int)($_POST['quantity'] ?? 1);
            if (!$itemId) {
                cartJsonResponse(['success' => false, 'message' => 'Invalid cart item.'], 400);
            }

            $stmt = $db->prepare("SELECT ci.*, p.stock AS product_stock, p.quantity_max, v.stock AS variant_stock FROM cart_items ci JOIN products p ON p.id = ci.product_id LEFT JOIN product_variants v ON v.id = ci.variant_id WHERE ci.id = ? AND ci.cart_id = ?");
            $stmt->execute([$itemId, $cartId]);
            $item = $stmt->fetch();
            if (!$item) {
                cartJsonResponse(['success' => false, 'message' => 'Cart item not found.'], 404);
            }

            $quantityMax = $item['quantity_max'];
            $stock = $item['variant_id'] ? $item['variant_stock'] : $item['product_stock'];

            if ($quantity <= 0) {
                $stmt = $db->prepare("DELETE FROM cart_items WHERE id = ?");
                $stmt->execute([$itemId]);
                CartService::cleanupEmptySession($cartId);
            } else {
                if ($quantityMax !== null && $quantityMax > 0) {
                    $quantity = min($quantity, (int)$quantityMax);
                }
                if ($stock !== null) {
                    $quantity = min($quantity, (int)$stock);
                }
                $now = DbHelper::nowString();
                $stmt = $db->prepare("UPDATE cart_items SET quantity = ?, updated_at = ? WHERE id = ?");
                $stmt->execute([$quantity, $now, $itemId]);
            }

            cartJsonResponse(['success' => true] + buildCartPayload($cartId, $currencySymbol));
            break;

        case 'remove':
            $itemId = (int)($_POST['item_id'] ?? 0);
            if ($itemId) {
                $stmt = $db->prepare("DELETE FROM cart_items WHERE id = ? AND cart_id = ?");
                $stmt->execute([$itemId, $cartId]);
                CartService::cleanupEmptySession($cartId);
            }
            cartJsonResponse(['success' => true] + buildCartPayload($cartId, $currencySymbol));
            break;

        case 'get':
        default:
            cartJsonResponse(['success' => true] + buildCartPayload($cartId, $currencySymbol));
    }
} catch (Throwable $e) {
    LogService::add('error', 'Cart API failure: ' . $e->getMessage(), json_encode([
        'action' => $action,
        'cart_id' => $cartId,
    ]));
    cartJsonResponse(['success' => false, 'message' => 'Cart request failed. Please try again.'], 500);
}
