<?php
use NewSite\Auth\SiteAuth;
use NewSite\Database\DatabaseManager;
use NewSite\Logging\LogService;
use NewSite\Settings\SettingsService;
use NewSite\Shop\CurrencyService;
use NewSite\Upload\ImageUrlHelper;
use NewSite\User\FriendService;
use NewSite\User\PrivacyService;
use NewSite\User\UserMessageService;
use NewSite\User\UserService;
use NewSite\Util\CountryData;

SiteAuth::requireLogin();
$userId = (int)($_SESSION['site_user_id'] ?? 0);
$db = DatabaseManager::getWriteConnection();

const ACCOUNT_DATETIME_FORMAT = 'M j, Y g:i A';

$accountFlash = $_SESSION['account_flash_message'] ?? '';
if ($accountFlash) {
    unset($_SESSION['account_flash_message']);
}
$accountFlashError = $_SESSION['account_flash_error'] ?? '';
if ($accountFlashError) {
    unset($_SESSION['account_flash_error']);
}

$user = UserService::getById((int)($_SESSION['site_user_id'] ?? 0));
$nickname = $user['nickname'] ?? '';
if ($nickname === '') {
    $nickname = UserService::generateUniqueNickname($db);
    $stmt = $db->prepare("UPDATE site_users SET nickname = ? WHERE id = ?");
    $stmt->execute([$nickname, $userId]);
    $user['nickname'] = $nickname;
}
$email = $user['email'] ?? '';
$displayName = $user['display_name'] ?? '';
$description = $user['description'] ?? '';
$website = $user['website'] ?? '';
$company = $user['company'] ?? '';
$phone = $user['phone'] ?? '';
$gender = $user['gender'] ?? '';
$genderCustom = $user['gender_custom'] ?? '';
$genderDisplay = '-';
if ($gender === 'custom') {
    $genderDisplay = $genderCustom !== '' ? $genderCustom : 'Custom';
} elseif ($gender !== '') {
    $genderDisplay = $gender;
}
$age = $user['age'] ?? '';
$address = $user['address'] ?? '';
$country = $user['country'] ?? '';
$postalCode = $user['postal_code'] ?? '';
$profilePhoto = $user['profile_photo'] ?? '';
$isSearchable = isset($user['is_searchable']) ? (int)$user['is_searchable'] : 1; // Default to searchable
$countries = CountryData::getOptions();
$createdAt = !empty($user['created_at']) ? date('F j, Y', (int)$user['created_at']) : '';
$lastLogin = !empty($user['last_login_at']) ? date('F j, Y g:i A', (int)$user['last_login_at']) : '';
$messages = UserMessageService::getMessages((int)($_SESSION['site_user_id'] ?? 0), 5);
$unreadCount = UserMessageService::getUnreadCount((int)$userId);
$selectedId = (int)($_GET['message'] ?? 0);
$selectedMessage = null;
$selectedThreadId = (int)($_GET['ticket'] ?? 0);
$activeTab = $_GET['tab'] ?? 'general';
$activeTab = in_array($activeTab, ['general', 'messages', 'settings', 'info', 'purchases', 'friends'], true) ? $activeTab : 'general';
if ($accountFlash && $activeTab === 'general') {
    $activeTab = 'messages';
}

// Friends data
$pendingFriendCount = FriendService::getPendingCount((int)$userId);
$friendsCount = FriendService::getCount((int)$userId);
$privacySettings = PrivacyService::getUserSettings((int)$userId);

// Friends tab data
$friendsTab = $_GET['ftab'] ?? 'all';
$validFriendsTabs = ['all', 'pending', 'sent', 'blocked'];
if (!in_array($friendsTab, $validFriendsTabs)) {
    $friendsTab = 'all';
}
$friends = [];
$pendingRequests = [];
$sentRequests = [];
$blockedUsers = [];
if ($activeTab === 'friends') {
    // Load all data for client-side tab switching
    $friends = FriendService::getList((int)$userId);
    $pendingRequests = FriendService::getPendingRequests((int)$userId);
    $sentRequests = FriendService::getSentRequests((int)$userId);
    $blockedUsers = FriendService::getBlockedUsers((int)$userId);
}
$friendSearchQuery = trim($_GET['fq'] ?? '');
$friendSearchResults = [];
if ($friendSearchQuery !== '' && $activeTab === 'friends') {
    $friendSearchResults = FriendService::searchUsers($friendSearchQuery, $userId, 20);
}
$friendFlashMessage = $_SESSION['friend_flash_message'] ?? '';
$friendFlashError = $_SESSION['friend_flash_error'] ?? '';
unset($_SESSION['friend_flash_message'], $_SESSION['friend_flash_error']);

if ($accountFlash) {
    LogService::add('info', 'Account messages view after action', json_encode([
        'user_id' => $userId,
        'active_tab' => $activeTab,
        'messages_count' => is_array($messages) ? count($messages) : 0,
        'unread_count' => $unreadCount,
        'selected_id' => $selectedId
    ]));
}

if ($selectedId > 0) {
    $stmt = $db->prepare("SELECT * FROM user_messages WHERE id = ? AND user_id = ?");
    $stmt->execute([$selectedId, $userId]);
    $selectedMessage = $stmt->fetch();
    if (!$selectedMessage) {
        $selectedId = 0;
    }
    $stmt = $db->prepare("UPDATE user_messages SET is_read = 1 WHERE id = ? AND user_id = ?");
    $stmt->execute([$selectedId, $userId]);
}

$ticketMessages = [];
try {
    $stmt = $db->prepare("SELECT * FROM user_messages WHERE user_id = ? AND (message_type = 'ticket' OR subject = 'Ticket') ORDER BY created_at DESC LIMIT 200");
    $stmt->execute([$userId]);
    $ticketMessages = $stmt->fetchAll();
} catch (PDOException $e) {
    $stmt = $db->prepare("SELECT * FROM user_messages WHERE user_id = ? AND subject = 'Ticket' ORDER BY created_at DESC LIMIT 200");
    $stmt->execute([$userId]);
    $ticketMessages = $stmt->fetchAll();
}

$ticketChat = null;
$conversation = [];
$ticketReplyToId = 0;

// Always show Support Chat, even if no messages yet
$ticketChat = [
    'root_id' => 1,
    'subject' => 'Support Chat',
    'preview' => '',
    'last_time' => 0
];

if (!empty($ticketMessages)) {
    $latest = $ticketMessages[0];
    $ticketChat['preview'] = $latest['body'] ?? '';
    $ticketChat['last_time'] = (int)($latest['created_at'] ?? 0);
    foreach ($ticketMessages as $row) {
        if (($row['sender_type'] ?? 'admin') === 'admin') {
            $ticketReplyToId = (int)($row['id'] ?? 0);
        }
    }
    if ($ticketReplyToId === 0) {
        $ticketReplyToId = (int)($latest['id'] ?? 0);
    }
}

if ($ticketChat) {
    // Load conversation but don't expand chat unless explicitly requested via ?ticket=1
    $conversation = array_reverse($ticketMessages);
}

$orders = [];
$orderItems = [];
$orderDownloads = [];
$orderRefunds = [];
$productVersions = [];
$hasProductUpdates = false;
$refundWindowDigitalDays = (int)SettingsService::get('refund_window_digital_days', '30');
$refundWindowPhysicalDays = (int)SettingsService::get('refund_window_physical_days', '30');
$stmt = $db->prepare("SELECT * FROM orders WHERE user_id = ? ORDER BY created_at DESC");
$stmt->execute([$userId]);
$orders = $stmt->fetchAll();

if (!empty($orders)) {
    $orderIds = array_column($orders, 'id');
    $placeholders = implode(',', array_fill(0, count($orderIds), '?'));
    $stmt = $db->prepare("SELECT * FROM order_items WHERE order_id IN ($placeholders) ORDER BY id ASC");
    $stmt->execute($orderIds);
    $items = $stmt->fetchAll();
    foreach ($items as $item) {
        $orderItems[$item['order_id']][] = $item;
    }

    $stmt = $db->prepare("SELECT dd.*, oi.order_id, COALESCE(NULLIF(CASE WHEN oi.variant_id IS NOT NULL AND COALESCE(pv.action_type, '') = 'download' THEN TRIM(COALESCE(pv.current_version, '')) ELSE '' END, ''), p.current_version) AS effective_current_version, COALESCE(CASE WHEN oi.variant_id IS NOT NULL AND COALESCE(pv.action_type, '') = 'download' THEN pv.version_changelog END, p.version_changelog) AS effective_version_changelog FROM digital_downloads dd JOIN order_items oi ON oi.id = dd.order_item_id LEFT JOIN products p ON p.id = COALESCE(dd.product_id, oi.product_id) LEFT JOIN product_variants pv ON pv.id = oi.variant_id WHERE oi.order_id IN ($placeholders)");
    $stmt->execute($orderIds);
    $downloads = $stmt->fetchAll();
    foreach ($downloads as $download) {
        $orderDownloads[$download['order_item_id']] = $download;
        // Check if this product has an update available
        $lastDownloaded = trim((string)($download['last_downloaded_version'] ?? ''));
        $currentVersion = trim((string)($download['effective_current_version'] ?? ''));
        if ($lastDownloaded !== '' && $currentVersion !== '' && strcasecmp($lastDownloaded, $currentVersion) !== 0) {
            $hasProductUpdates = true;
        }
    }

    // Fetch refund request status for all order items
    $stmt = $db->prepare(
        "SELECT rr.order_item_id, rr.status, rr.admin_reason, rr.return_instructions
         FROM refund_requests rr
         WHERE rr.order_id IN ($placeholders)"
    );
    $stmt->execute($orderIds);
    $refundRows = $stmt->fetchAll();
    foreach ($refundRows as $rr) {
        $orderRefunds[(int)$rr['order_item_id']] = $rr;
    }
}
?>

<section class="auth-section">
    <div class="container">
        <div class="auth-card auth-card--wide">
            <a href="/@<?php echo e($nickname); ?>" class="btn btn-outline btn-sm account-view-profile-btn">View Profile</a>
            <div class="account-title">
                <?php if ($profilePhoto): ?>
                    <div class="account-title-photo">
                            <img src="<?php echo e($profilePhoto); ?>" alt="Profile">
                    </div>
                <?php else: ?>
                    <div class="account-title-photo account-title-photo--icon" aria-hidden="true">
                        <svg viewBox="0 0 24 24" aria-hidden="true">
                            <circle cx="12" cy="8" r="4"></circle>
                            <path d="M4 20c0-4.4 3.6-8 8-8s8 3.6 8 8"></path>
                        </svg>
                    </div>
                <?php endif; ?>
                <?php if ($nickname): ?>
                    <div class="account-title-name">@<?php echo e($nickname); ?></div>
                <?php endif; ?>
            </div>
            <div class="account-tabs" data-account-tabs data-account-tab-default="<?php echo e($activeTab); ?>">
                <button type="button" class="account-tab <?php echo $activeTab === 'general' ? 'active' : ''; ?>" data-account-tab="general">General</button>
                <button type="button" class="account-tab <?php echo $activeTab === 'purchases' ? 'active' : ''; ?>" data-account-tab="purchases">Purchases<?php if ($hasProductUpdates): ?><span class="purchases-update-indicator" title="Updates available"></span><?php endif; ?></button>
                <button type="button" class="account-tab <?php echo $activeTab === 'info' ? 'active' : ''; ?>" data-account-tab="info">Account Info</button>
                <button type="button" class="account-tab <?php echo $activeTab === 'messages' ? 'active' : ''; ?>" data-account-tab="messages" data-message-center-tab>Message Center<?php if ($unreadCount > 0): ?><span class="message-center-indicator"></span><?php endif; ?></button>
                <button type="button" class="account-tab <?php echo $activeTab === 'settings' ? 'active' : ''; ?>" data-account-tab="settings">Account Settings</button>
                <button type="button" class="account-tab <?php echo $activeTab === 'friends' ? 'active' : ''; ?>" data-account-tab="friends">Friends<?php if ($pendingFriendCount > 0): ?><span class="friend-notification-badge"><?php echo $pendingFriendCount; ?></span><?php endif; ?></button>
            </div>

            <div class="account-panel <?php echo $activeTab === 'general' ? '' : 'is-hidden'; ?>" data-account-panel="general">
            </div>

            <div class="account-panel <?php echo $activeTab === 'purchases' ? '' : 'is-hidden'; ?>" data-account-panel="purchases">
                <div class="account-purchases">
                    <?php if ($accountFlash && $activeTab === 'purchases'): ?>
                        <div class="auth-alert success"><?php echo e($accountFlash); ?></div>
                    <?php endif; ?>
                    <?php if ($accountFlashError && $activeTab === 'purchases'): ?>
                        <div class="auth-alert error"><?php echo e($accountFlashError); ?></div>
                    <?php endif; ?>
                    <?php if (empty($orders)): ?>
                        <div class="auth-alert message-empty">No purchases yet. When you complete checkout, downloads will appear here.</div>
                    <?php else: ?>
                        <?php foreach ($orders as $order): ?>
                            <?php
                            $items = $orderItems[$order['id']] ?? [];
                            $symbol = CurrencyService::getSymbolFor($order['currency'] ?? CurrencyService::getCurrent());
                            $totalValue = number_format((float)($order['total_amount'] ?? 0), 2);
                            $orderDate = !empty($order['created_at']) ? date('M j, Y', strtotime($order['created_at'])) : '';
                            $statusKey = strtolower(preg_replace('/[^a-z]/', '', (string)($order['status'] ?? 'paid')));
                            $shippingStatus = $order['shipping_status'] ?? 'pending';
                            $hasPhysical = false;
                            foreach ($items as $itemCheck) {
                                if (empty($itemCheck['is_digital'])) {
                                    $hasPhysical = true;
                                    break;
                                }
                            }
                            ?>
                            <div class="order-card">
                                <div class="order-card-header">
                                    <div class="order-meta">
                                        <div class="order-number">Order #<?php echo e($order['order_number']); ?></div>
                                        <?php if ($orderDate): ?>
                                            <div class="order-date"><?php echo e($orderDate); ?></div>
                                        <?php endif; ?>
                                    </div>
                                    <div class="order-card-actions">
                                        <span class="status-pill status-pill--<?php echo e($statusKey ?: 'paid'); ?>"><?php echo e(ucfirst($order['status'] ?? 'paid')); ?></span>
                                        <a class="btn btn-outline btn-sm" href="/invoice?order=<?php echo (int)$order['id']; ?>" target="_blank" rel="noopener">Invoice PDF</a>
                                    </div>
                                </div>
                                <div class="order-total">Total: <?php echo e($symbol . $totalValue); ?></div>

                                <div class="order-items">
                                    <?php foreach ($items as $item): ?>
                                        <?php
                                            $unitPrice = number_format((float)($item['unit_price'] ?? 0), 2);
                                        $download = $orderDownloads[$item['id']] ?? null;
                                        $hasUpdate = false;
                                        $changelog = '';
                                        $currentVersion = '';
                                        if ($download) {
                                            $lastDownloaded = trim((string)($download['last_downloaded_version'] ?? ''));
                                            $currentVersion = trim((string)($download['effective_current_version'] ?? ''));
                                            $changelog = (string)($download['effective_version_changelog'] ?? '');
                                            $hasUpdate = $lastDownloaded !== '' && $currentVersion !== '' && strcasecmp($lastDownloaded, $currentVersion) !== 0;
                                        }
                                        ?>
                                        <div class="order-item">
                                            <div class="order-item-info">
                                                <div class="order-item-name"><?php echo e($item['product_name'] ?? 'Product'); ?></div>
                                                <?php if (!empty($item['variant_label'])): ?>
                                                    <div class="order-item-variant">Version: <?php echo e($item['variant_label']); ?></div>
                                                <?php endif; ?>
                                                <div class="order-item-qty">Qty: <?php echo (int)($item['quantity'] ?? 1); ?></div>
                                            </div>
                                            <div class="order-item-actions">
                                                <div class="order-item-price"><?php echo e($symbol . $unitPrice); ?></div>
                                                <?php if (!empty($item['is_digital'])): ?>
                                                    <?php if ($download): ?>
                                                        <?php
                                                            $refundInfo = $orderRefunds[(int)$item['id']] ?? null;
                                                            $refundStatus = $refundInfo ? (string)$refundInfo['status'] : '';
                                                            $dlCount = (int)($download['download_count'] ?? 0);
                                                        ?>
                                                        <div class="order-item-download">
                                                            <a href="/download?token=<?php echo e($download['token']); ?>" class="btn btn-outline btn-sm<?php echo $hasUpdate ? ' has-update' : ''; ?>" data-download-btn>
                                                                <?php if ($hasUpdate): ?><span class="download-update-dot" title="New version available"></span><?php endif; ?>
                                                                Download<?php if ($currentVersion): ?> v<?php echo e($currentVersion); ?><?php endif; ?>
                                                            </a>
                                                            <?php if ($changelog !== ''): ?>
                                                                <button type="button" class="btn btn-text btn-sm" data-toggle-changelog title="View changelog">What's new</button>
                                                            <?php endif; ?>
                                                            <?php if ($refundStatus === ''): ?>
                                                                <?php
                                                                    $digitalWindowOpen = $dlCount === 0
                                                                        && $refundWindowDigitalDays > 0
                                                                        && !empty($order['created_at'])
                                                                        && (strtotime($order['created_at']) + $refundWindowDigitalDays * 86400) > time();
                                                                ?>
                                                                <?php if ($digitalWindowOpen): ?>
                                                                    <form method="POST" action="/account?tab=purchases" class="refund-form-inline">
                                                                        <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                                                        <input type="hidden" name="action" value="request_refund">
                                                                        <input type="hidden" name="order_item_id" value="<?php echo (int)$item['id']; ?>">
                                                                        <button type="submit" class="btn btn-danger btn-sm">Request Refund</button>
                                                                    </form>
                                                                <?php endif; ?>
                                                            <?php elseif ($refundStatus === 'pending'): ?>
                                                                <span class="refund-status refund-status--pending">Refund Pending</span>
                                                                <form method="POST" action="/account?tab=purchases" class="refund-form-inline">
                                                                    <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                                                    <input type="hidden" name="action" value="cancel_refund">
                                                                    <input type="hidden" name="order_item_id" value="<?php echo (int)$item['id']; ?>">
                                                                    <button type="submit" class="btn btn-outline btn-sm">Cancel Refund</button>
                                                                </form>
                                                            <?php elseif ($refundStatus === 'accepted'): ?>
                                                                <div class="refund-status refund-status--accepted">
                                                                    <span>Refund Accepted</span>
                                                                    <p class="refund-accepted-text">Your refund has been approved and will be returned to your original payment method.</p>
                                                                </div>
                                                            <?php elseif ($refundStatus === 'cancelled'): ?>
                                                                <div class="refund-status refund-status--cancelled">
                                                                    <span>Refund Declined</span>
                                                                    <?php if (!empty($refundInfo['admin_reason'])): ?>
                                                                        <p class="refund-reason-text">Reason: <?php echo e($refundInfo['admin_reason']); ?></p>
                                                                    <?php endif; ?>
                                                                </div>
                                                            <?php endif; ?>
                                                        </div>
                                                        <?php if ($changelog !== ''): ?>
                                                            <div class="order-item-changelog is-hidden" data-changelog>
                                                                <div class="changelog-content"><?php echo nl2br(e($changelog)); ?></div>
                                                            </div>
                                                        <?php endif; ?>
                                                    <?php else: ?>
                                                        <span class="order-item-note">Preparing download</span>
                                                    <?php endif; ?>
                                                <?php else: ?>
                                                    <?php
                                                        $refundInfo = $orderRefunds[(int)$item['id']] ?? null;
                                                        $refundStatus = $refundInfo ? (string)$refundInfo['status'] : '';
                                                        $physicalWindowOpen = $refundWindowPhysicalDays > 0
                                                            && !empty($order['created_at'])
                                                            && (strtotime($order['created_at']) + $refundWindowPhysicalDays * 86400) > time();
                                                    ?>
                                                    <div class="order-item-physical">
                                                        <span class="order-item-note">Physical item</span>
                                                        <?php if ($refundStatus === ''): ?>
                                                            <?php if ($physicalWindowOpen): ?>
                                                                <form method="POST" action="/account?tab=purchases" class="refund-form-inline">
                                                                    <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                                                    <input type="hidden" name="action" value="request_refund">
                                                                    <input type="hidden" name="order_item_id" value="<?php echo (int)$item['id']; ?>">
                                                                    <button type="submit" class="btn btn-danger btn-sm">Request return item and refund</button>
                                                                </form>
                                                            <?php endif; ?>
                                                        <?php elseif ($refundStatus === 'pending'): ?>
                                                            <span class="refund-status refund-status--pending">Return &amp; Refund Pending</span>
                                                            <form method="POST" action="/account?tab=purchases" class="refund-form-inline">
                                                                <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                                                <input type="hidden" name="action" value="cancel_refund">
                                                                <input type="hidden" name="order_item_id" value="<?php echo (int)$item['id']; ?>">
                                                                <button type="submit" class="btn btn-outline btn-sm">Cancel Request</button>
                                                            </form>
                                                        <?php elseif ($refundStatus === 'accepted'): ?>
                                                            <div class="refund-status refund-status--accepted">
                                                                <span>Return &amp; Refund Accepted</span>
                                                                <?php
                                                                    $returnFiles = [];
                                                                    $rawInstructions = trim((string)($refundInfo['return_instructions'] ?? ''));
                                                                    if ($rawInstructions !== '') {
                                                                        $decoded = json_decode($rawInstructions, true);
                                                                        if (is_array($decoded)) {
                                                                            $returnFiles = $decoded;
                                                                        }
                                                                    }
                                                                ?>
                                                                <?php if (!empty($returnFiles)): ?>
                                                                    <details class="return-instructions-details">
                                                                        <summary class="btn btn-outline btn-sm">View Instructions</summary>
                                                                        <div class="return-instructions-images">
                                                                            <?php foreach ($returnFiles as $filePath): ?>
                                                                                <?php $fileName = basename((string)$filePath); ?>
                                                                                <a href="/refund-file/<?php echo e($fileName); ?>" target="_blank" rel="noopener">
                                                                                    <img src="/refund-file/<?php echo e($fileName); ?>" alt="Return instruction" loading="lazy">
                                                                                </a>
                                                                            <?php endforeach; ?>
                                                                        </div>
                                                                    </details>
                                                                <?php endif; ?>
                                                            </div>
                                                        <?php elseif ($refundStatus === 'cancelled'): ?>
                                                            <div class="refund-status refund-status--cancelled">
                                                                <span>Return &amp; Refund Declined</span>
                                                                <?php if (!empty($refundInfo['admin_reason'])): ?>
                                                                    <p class="refund-reason-text">Reason: <?php echo e($refundInfo['admin_reason']); ?></p>
                                                                <?php endif; ?>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>

                                <?php if ($hasPhysical): ?>
                                    <div class="order-shipping">
                                        <div class="order-shipping-title">Shipping</div>
                                        <div class="order-shipping-meta">
                                            <span>Status:</span>
                                            <strong><?php echo e(ucfirst($shippingStatus)); ?></strong>
                                        </div>
                                        <?php if (!empty($order['tracking_number'])): ?>
                                            <div class="order-shipping-meta">
                                                <span>Tracking:</span>
                                                <?php if (!empty($order['tracking_url'])): ?>
                                                    <a href="<?php echo e($order['tracking_url']); ?>" target="_blank" rel="noopener"><?php echo e($order['tracking_number']); ?></a>
                                                <?php else: ?>
                                                    <span><?php echo e($order['tracking_number']); ?></span>
                                                <?php endif; ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>

            <div class="account-panel <?php echo $activeTab === 'info' ? '' : 'is-hidden'; ?>" data-account-panel="info">
                <div class="account-info">
                    <div>
                        <span class="account-label">Name</span>
                        <span class="account-value"><?php echo e($displayName ?: '-'); ?></span>
                    </div>
                    <div>
                        <span class="account-label">Description</span>
                        <span class="account-value"><?php echo e($description ?: '-'); ?></span>
                    </div>
                    <div>
                        <span class="account-label">Nickname</span>
                        <span class="account-value"><?php echo e($nickname ?: '-'); ?></span>
                    </div>
                    <div>
                        <span class="account-label">Email</span>
                        <span class="account-value"><?php echo e($email ?: '-'); ?></span>
                    </div>
                    <div>
                        <span class="account-label">Website</span>
                        <span class="account-value"><?php echo e($website ?: '-'); ?></span>
                    </div>
                    <div>
                        <span class="account-label">Company</span>
                        <span class="account-value"><?php echo e($company ?: '-'); ?></span>
                    </div>
                    <div>
                        <span class="account-label">Phone</span>
                        <span class="account-value"><?php echo e($phone ?: '-'); ?></span>
                    </div>
                    <div>
                        <span class="account-label">Gender</span>
                        <span class="account-value"><?php echo e($genderDisplay); ?></span>
                    </div>
                    <div>
                        <span class="account-label">Age</span>
                        <span class="account-value"><?php echo e($age !== '' ? (string)$age : '-'); ?></span>
                    </div>
                    <div>
                        <span class="account-label">Address</span>
                        <span class="account-value"><?php echo e($address ?: '-'); ?></span>
                    </div>
                    <div>
                        <span class="account-label">Country</span>
                        <span class="account-value"><?php echo e($country ?: '-'); ?></span>
                    </div>
                    <div>
                        <span class="account-label">Postal Code</span>
                        <span class="account-value"><?php echo e($postalCode ?: '-'); ?></span>
                    </div>
                    <?php if ($createdAt): ?>
                    <div>
                        <span class="account-label">Member since</span>
                        <span class="account-value"><?php echo e($createdAt); ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if ($lastLogin): ?>
                    <div>
                        <span class="account-label">Last login</span>
                        <span class="account-value"><?php echo e($lastLogin); ?></span>
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <div class="account-panel <?php echo $activeTab === 'messages' ? '' : 'is-hidden'; ?>" data-account-panel="messages" data-message-panel>
                <div class="auth-subtitle">Recent messages</div>
                <?php if ($accountFlash): ?>
                    <div class="auth-alert success"><?php echo e($accountFlash); ?></div>
                <?php endif; ?>
                <?php if ($unreadCount > 0): ?>
                <form method="POST" action="/account?tab=messages" class="messages-toolbar">
                    <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                    <input type="hidden" name="action" value="mark_all_read">
                    <button type="submit" class="btn btn-outline btn-sm">Mark all as read</button>
                </form>
                <?php endif; ?>
                <?php if (empty($messages) && !$ticketChat): ?>
                    <div class="auth-alert message-empty" data-message-empty>No messages yet. New updates will appear here.</div>
                <?php else: ?>
                    <?php if ($ticketChat): ?>
                        <div class="ticket-layout <?php echo $selectedThreadId > 0 ? 'is-collapsed' : ''; ?>" data-ticket-chat data-selected-open="<?php echo $selectedThreadId > 0 ? '1' : '0'; ?>">
                            <div class="ticket-list">
                                <?php
                                $isActive = $selectedThreadId > 0;
                        $preview = trim((string)($ticketChat['preview'] ?? ''));
                        $preview = $preview !== '' ? mb_substr($preview, 0, 80) : '';
                        ?>
                                <a class="ticket-list-item <?php echo $isActive ? 'is-active' : ''; ?>" href="/account?tab=messages&ticket=1" data-ticket-link>
                                    <div class="ticket-list-title">Support Chat</div>
                                    <?php if ($preview !== ''): ?>
                                        <div class="ticket-list-preview"><?php echo e($preview); ?><?php echo strlen($preview) >= 80 ? '…' : ''; ?></div>
                                    <?php endif; ?>
                                    <div class="ticket-list-time"><?php echo !empty($ticketChat['last_time']) ? date(ACCOUNT_DATETIME_FORMAT, (int)$ticketChat['last_time']) : '-'; ?></div>
                                </a>
                            </div>
                            <div class="ticket-chat" data-ticket-chat-box>
                                <?php if (empty($conversation) && $selectedThreadId <= 0): ?>
                                    <div class="text-muted">Click Support Chat to start a conversation.</div>
                                <?php else: ?>
                                    <div class="ticket-chat-header">
                                        <div>
                                            <h4>Support Chat</h4>
                                            <div class="ticket-chat-subject"><?php echo empty($conversation) ? 'Start a new conversation' : 'Ticket conversation'; ?></div>
                                        </div>
                                        <div class="ticket-chat-header-actions" data-chat-actions>
                                            <button type="button" class="btn btn-outline btn-sm<?php echo empty($conversation) ? ' is-hidden-display' : ''; ?>" data-download-chat title="Download your chat history">
                                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="14" height="14" class="ticket-action-icon">
                                                    <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                                                    <polyline points="7 10 12 15 17 10"></polyline>
                                                    <line x1="12" y1="15" x2="12" y2="3"></line>
                                                </svg>
                                                Download
                                            </button>
                                            <button type="button" class="btn btn-outline btn-sm btn-danger-outline<?php echo empty($conversation) ? ' is-hidden-display' : ''; ?>" data-clear-chat title="Clear chat history">
                                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="14" height="14" class="ticket-action-icon">
                                                    <polyline points="3 6 5 6 21 6"></polyline>
                                                    <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path>
                                                </svg>
                                                Clear
                                            </button>
                                            <button type="button" class="btn btn-outline btn-sm ticket-toggle-list" data-ticket-toggle-list>Close Chat</button>
                                        </div>
                                    </div>
                                    <div class="ticket-chat-body" data-ticket-chat-body data-ticket-last-id="<?php echo !empty($conversation) ? (int)($conversation[count($conversation) - 1]['id'] ?? 0) : 0; ?>">
                                        <?php if (empty($conversation)): ?>
                                            <div class="text-muted ticket-empty-state">No messages yet. Send a message to start chatting with support!</div>
                                        <?php else: ?>
                                            <?php foreach ($conversation as $row): ?>
                                                <?php $isAdmin = ($row['sender_type'] ?? 'admin') === 'admin'; ?>
                                                <div class="ticket-bubble <?php echo $isAdmin ? 'ticket-bubble--admin' : 'ticket-bubble--user'; ?>">
                                                    <div class="ticket-bubble-meta">
                                                        <?php echo $isAdmin ? 'Admin' : 'You'; ?>
                                                        · <?php echo !empty($row['created_at']) ? date(ACCOUNT_DATETIME_FORMAT, (int)$row['created_at']) : ''; ?>
                                                        <?php if (!$isAdmin): ?>
                                                            <?php if (!empty($row['admin_read'])): ?>
                                                                <span class="read-receipt read-receipt--read" title="Seen"><svg viewBox="0 0 16 16" fill="currentColor"><path d="M13.854 3.646a.5.5 0 0 1 0 .708l-7 7a.5.5 0 0 1-.708 0l-3.5-3.5a.5.5 0 1 1 .708-.708L6.5 10.293l6.646-6.647a.5.5 0 0 1 .708 0z"/></svg></span>
                                                            <?php else: ?>
                                                                <span class="read-receipt read-receipt--sent" title="Sent"><svg viewBox="0 0 16 16" fill="currentColor"><path d="M13.854 3.646a.5.5 0 0 1 0 .708l-7 7a.5.5 0 0 1-.708 0l-3.5-3.5a.5.5 0 1 1 .708-.708L6.5 10.293l6.646-6.647a.5.5 0 0 1 .708 0z"/></svg></span>
                                                            <?php endif; ?>
                                                        <?php endif; ?>
                                                    </div>
                                                    <?php if (!empty($row['image_path'])): ?>
                                                        <div class="ticket-bubble-image">
                                                            <img src="<?php echo e(ImageUrlHelper::getChatImageUrl($row['image_path'])); ?>" alt="Attachment" class="chat-image" loading="lazy">
                                                        </div>
                                                    <?php endif; ?>
                                                    <div class="ticket-bubble-text"><?php echo e($row['body'] ?? ''); ?></div>
                                                </div>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </div>
                                    <form method="POST" action="/account?tab=messages&ticket=1" class="ticket-reply-form" data-ticket-reply-form enctype="multipart/form-data">
                                        <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                        <input type="hidden" name="action" value="<?php echo empty($conversation) ? 'start_ticket' : 'reply_message'; ?>">
                                        <input type="hidden" name="reply_to_id" value="<?php echo (int)$ticketReplyToId; ?>">
                                        <div class="form-group">
                                            <label for="account-ticket-reply-body">Message</label>
                                            <textarea id="account-ticket-reply-body" name="reply_body" class="form-control" rows="3" placeholder="<?php echo empty($conversation) ? 'Type your message to start a conversation...' : 'Write your reply...'; ?>"></textarea>
                                        </div>
                                        <div class="form-group">
                                            <label for="account-ticket-image" class="upload-label">
                                                <span class="upload-icon">📷</span> Attach Image (max 5MB)
                                            </label>
                                            <input type="file" id="account-ticket-image" name="chat_image" accept="image/jpeg,image/png,image/gif,image/webp" class="form-control-file">
                                            <div class="upload-preview is-hidden-display" data-upload-preview>
                                                <img src="" alt="Preview" class="upload-preview-img">
                                                <button type="button" class="upload-preview-remove" data-remove-preview>&times;</button>
                                            </div>
                                        </div>
                                        <button type="submit" class="btn btn-outline btn-sm"><?php echo empty($conversation) ? 'Send Message' : 'Send Reply'; ?></button>
                                    </form>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endif; ?>

                    <?php
                    $noticeMessages = array_filter($messages, function ($row) {
                        $type = $row['message_type'] ?? 'notice';
                        $subject = $row['subject'] ?? '';
                        $replyToId = $row['reply_to_id'] ?? 0;
                        // Exclude ticket messages, messages with 'Ticket' subject, and replies in ticket threads
                        $isTicketType = $type === 'ticket';
                        $hasTicketSubject = stripos($subject, 'ticket') !== false;
                        $isReplyMessage = !empty($replyToId);
                        return !($isTicketType || $hasTicketSubject || $isReplyMessage);
                    });
?>
                    <?php if (!empty($noticeMessages)): ?>
                        <h3 class="mt-3">Other Messages</h3>
                        <div class="message-list" data-message-list>
                            <?php foreach ($noticeMessages as $message): ?>
                                <?php
            $bodyText = $message['body'] ?? '';
                                $preview = mb_strlen($bodyText) > 100 ? mb_substr($bodyText, 0, 100) . '…' : $bodyText;
                                ?>
                                <div class="message-item message-item--list <?php echo empty($message['is_read']) ? 'is-unread' : ''; ?>" id="message-<?php echo (int)$message['id']; ?>" data-message-id="<?php echo (int)$message['id']; ?>">
                                    <div class="message-head">
                                        <div class="message-title"><?php echo e($message['subject'] ?: 'Message'); ?></div>
                                        <form method="POST" action="/account?tab=messages" class="message-delete-form" data-confirm="Delete this message?">
                                            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                            <input type="hidden" name="action" value="delete_message">
                                            <input type="hidden" name="message_id" value="<?php echo (int)$message['id']; ?>">
                                            <button type="submit" class="message-delete" aria-label="Delete message">🗑️</button>
                                        </form>
                                    </div>
                                    <?php if ($preview !== ''): ?>
                                        <div class="message-preview"><?php echo e($preview); ?></div>
                                    <?php endif; ?>
                                    <div class="message-time">
                                        <?php echo !empty($message['created_at']) ? date(ACCOUNT_DATETIME_FORMAT, (int)$message['created_at']) : ''; ?>
                                    </div>
                                    <?php if (!empty($bodyText)): ?>
                                        <div class="message-body message-body--full is-hidden" data-message-body><?php echo e($bodyText); ?></div>
                                    <?php endif; ?>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                <?php endif; ?>
            </div>

                <div class="account-panel <?php echo $activeTab === 'settings' ? '' : 'is-hidden'; ?>" data-account-panel="settings">
                <?php if (!empty($_SESSION['account_settings_message'])): ?>
                    <div class="auth-alert success"><?php echo e($_SESSION['account_settings_message']);
                    unset($_SESSION['account_settings_message']); ?></div>
                <?php endif; ?>
                <?php if (!empty($_SESSION['account_settings_error'])): ?>
                    <div class="auth-alert error"><?php echo e($_SESSION['account_settings_error']);
                    unset($_SESSION['account_settings_error']); ?></div>
                <?php endif; ?>
                <?php if (!empty($_SESSION['privacy_settings_message'])): ?>
                    <div class="auth-alert success"><?php echo e($_SESSION['privacy_settings_message']);
                    unset($_SESSION['privacy_settings_message']); ?></div>
                <?php endif; ?>
                <?php
                $defaultSettings = PrivacyService::getDefaults();
function getPrivacyValue($field, $privacySettings, $defaultSettings)
{
    return $privacySettings[$field] ?? $defaultSettings[$field] ?? 'hidden';
}
function renderPrivacySelect($field, $privacySettings, $defaultSettings)
{
    $value = getPrivacyValue($field, $privacySettings, $defaultSettings);
    $labels = [
        'everyone' => 'Everyone',
        'friends' => 'Friends',
        'hidden' => 'Hidden'
    ];
    $currentLabel = $labels[$value] ?? 'Hidden';
    ?>
                    <div class="privacy-inline has-tooltip" data-tooltip="Who can see this field">
                        <div class="privacy-dropdown" data-privacy-dropdown>
                            <input type="hidden" name="privacy[<?php echo e($field); ?>]" value="<?php echo e($value); ?>" data-privacy-value>
                            <button type="button" class="privacy-dropdown-trigger" data-privacy-trigger>
                                <span class="privacy-icon" data-privacy-icon="<?php echo e($value); ?>">
                                    <?php if ($value === 'everyone'): ?>
                                    <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><line x1="2" y1="12" x2="22" y2="12"/><path d="M12 2a15.3 15.3 0 0 1 4 10 15.3 15.3 0 0 1-4 10 15.3 15.3 0 0 1-4-10 15.3 15.3 0 0 1 4-10z"/></svg>
                                    <?php elseif ($value === 'friends'): ?>
                                    <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/><circle cx="9" cy="7" r="4"/><path d="M23 21v-2a4 4 0 0 0-3-3.87"/><path d="M16 3.13a4 4 0 0 1 0 7.75"/></svg>
                                    <?php else: ?>
                                    <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="11" width="18" height="11" rx="2" ry="2"/><path d="M7 11V7a5 5 0 0 1 10 0v4"/></svg>
                                    <?php endif; ?>
                                </span>
                                <span class="privacy-label" data-privacy-label><?php echo e($currentLabel); ?></span>
                                <svg class="privacy-caret" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M6 9l6 6 6-6"/></svg>
                            </button>
                            <div class="privacy-dropdown-menu" data-privacy-menu>
                                <button type="button" class="privacy-option <?php echo $value === 'everyone' ? 'active' : ''; ?>" data-privacy-option data-value="everyone">
                                    <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><line x1="2" y1="12" x2="22" y2="12"/><path d="M12 2a15.3 15.3 0 0 1 4 10 15.3 15.3 0 0 1-4 10 15.3 15.3 0 0 1-4-10 15.3 15.3 0 0 1 4-10z"/></svg>
                                    <span>Everyone</span>
                                </button>
                                <button type="button" class="privacy-option <?php echo $value === 'friends' ? 'active' : ''; ?>" data-privacy-option data-value="friends">
                                    <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/><circle cx="9" cy="7" r="4"/><path d="M23 21v-2a4 4 0 0 0-3-3.87"/><path d="M16 3.13a4 4 0 0 1 0 7.75"/></svg>
                                    <span>Friends</span>
                                </button>
                                <button type="button" class="privacy-option <?php echo $value === 'hidden' ? 'active' : ''; ?>" data-privacy-option data-value="hidden">
                                    <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="11" width="18" height="11" rx="2" ry="2"/><path d="M7 11V7a5 5 0 0 1 10 0v4"/></svg>
                                    <span>Hidden</span>
                                </button>
                            </div>
                        </div>
                    </div>
                    <?php
}
?>
                <form method="POST" action="/account?tab=settings" enctype="multipart/form-data" class="auth-form">
                    <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                    <input type="hidden" name="action" value="update_account">

                    <div class="setting-row">
                        <div class="setting-field">
                            <label for="display_name">Name</label>
                            <input type="text" id="display_name" name="display_name" value="<?php echo e($displayName); ?>" maxlength="120" autocomplete="name">
                        </div>
                        <?php renderPrivacySelect('display_name', $privacySettings, $defaultSettings); ?>
                    </div>
                    <div class="settings-help">Used to personalize your profile and receipts.</div>

                    <div class="setting-row">
                        <div class="setting-field">
                            <label for="status">Status</label>
                            <div class="select-field" data-select>
                                <input type="hidden" id="status" name="status" value="<?php echo e($user['status'] ?? 'online'); ?>" data-select-value>
                                <button type="button" class="select-trigger" data-select-trigger>
                                    <span data-select-label>
                                        <?php
                        $statusLabels = [
                            'online' => 'Online',
                            'offline' => 'Offline',
                            'dnd' => 'Do Not Disturb'
                        ];
echo e($statusLabels[$user['status'] ?? 'online'] ?? 'Online');
?>
                                    </span>
                                    <span class="select-caret" aria-hidden="true"></span>
                                </button>
                                <div class="select-menu" data-select-menu>
                                    <button type="button" class="select-option" data-select-option data-value="online">
                                        <span class="status-indicator status-online"></span> Online
                                    </button>
                                    <button type="button" class="select-option" data-select-option data-value="offline">
                                        <span class="status-indicator status-offline"></span> Offline
                                    </button>
                                    <button type="button" class="select-option" data-select-option data-value="dnd">
                                        <span class="status-indicator status-dnd"></span> Do Not Disturb
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="settings-help">Set your availability for friends and the messenger.</div>

                    <div class="setting-row">
                        <div class="setting-field">
                            <div class="toggle-title">Who can message you</div>
                        </div>
                        <?php renderPrivacySelect('messenger', $privacySettings, $defaultSettings); ?>
                    </div>
                    <div class="settings-help">Allow non-friends to start a Messenger chat with you.</div>

                    <div class="setting-row">
                        <div class="setting-field">
                            <label for="description">Description</label>
                            <textarea id="description" name="description" rows="4" maxlength="500" placeholder="Tell us about yourself..."><?php echo e($description); ?></textarea>
                        </div>
                        <?php renderPrivacySelect('description', $privacySettings, $defaultSettings); ?>
                    </div>
                    <div class="settings-help">Shown on your profile to introduce yourself.</div>

                    <div class="setting-row">
                        <div class="setting-field">
                            <label for="website">Website</label>
                            <input type="text" id="website" name="website" value="<?php echo e($website); ?>" maxlength="200" placeholder="https://yourwebsite.com" autocomplete="url">
                        </div>
                        <?php renderPrivacySelect('website', $privacySettings, $defaultSettings); ?>
                    </div>
                    <div class="settings-help">Optional link so others can find your work or portfolio.</div>

                    <div class="setting-row">
                        <div class="setting-field">
                            <label for="company">Company</label>
                            <input type="text" id="company" name="company" value="<?php echo e($company); ?>" maxlength="120" placeholder="Company name" autocomplete="organization">
                        </div>
                        <?php renderPrivacySelect('company', $privacySettings, $defaultSettings); ?>
                    </div>
                    <div class="settings-help">Helps us tailor communications and invoices if needed.</div>

                    <div class="setting-row">
                        <div class="setting-field">
                            <label for="phone">Phone Number</label>
                            <input type="text" id="phone" name="phone" value="<?php echo e($phone); ?>" maxlength="40" placeholder="+1 555 123 4567" autocomplete="tel">
                        </div>
                        <?php renderPrivacySelect('phone', $privacySettings, $defaultSettings); ?>
                    </div>
                    <div class="settings-help">Used only for account support or order issues.</div>

                    <div class="setting-row">
                        <div class="setting-field">
                            <label for="gender">Gender</label>
                            <div class="select-field" data-select>
                                <input type="hidden" id="gender" name="gender" value="<?php echo e($gender); ?>" data-select-value>
                                <button type="button" class="select-trigger" data-select-trigger>
                                    <span data-select-label>
                                        <?php
$genderLabels = [
    '' => 'Select an option',
    'male' => 'Male',
    'female' => 'Female',
    'non-binary' => 'Non-binary',
    'custom' => 'Custom',
    'prefer-not-to-say' => 'Prefer not to say'
];
echo e($genderLabels[$gender] ?? 'Select an option');
?>
                                    </span>
                                    <span class="select-caret" aria-hidden="true"></span>
                                </button>
                                <div class="select-menu" data-select-menu>
                                    <button type="button" class="select-option" data-select-option data-value="">Select an option</button>
                                    <button type="button" class="select-option" data-select-option data-value="male">Male</button>
                                    <button type="button" class="select-option" data-select-option data-value="female">Female</button>
                                    <button type="button" class="select-option" data-select-option data-value="non-binary">Non-binary</button>
                                    <button type="button" class="select-option" data-select-option data-value="custom">Custom</button>
                                    <button type="button" class="select-option" data-select-option data-value="prefer-not-to-say">Prefer not to say</button>
                                </div>
                            </div>
                        </div>
                        <?php renderPrivacySelect('gender', $privacySettings, $defaultSettings); ?>
                    </div>
                    <div class="settings-help">Optional. Shown on your public profile if privacy is set to visible.</div>

                    <div class="setting-row">
                        <div class="setting-field">
                            <label for="gender_custom">Custom Gender</label>
                            <input type="text" id="gender_custom" name="gender_custom" value="<?php echo e($genderCustom); ?>" maxlength="120" placeholder="Enter custom gender">
                        </div>
                    </div>
                    <div class="settings-help">Fill this in if you choose Custom.</div>

                    <div class="setting-row">
                        <div class="setting-field">
                            <label for="age">Age</label>
                            <input type="number" id="age" name="age" value="<?php echo e($age); ?>" min="1" max="120" step="1" placeholder="Age">
                        </div>
                        <?php renderPrivacySelect('age', $privacySettings, $defaultSettings); ?>
                    </div>
                    <div class="settings-help">Helps us ensure age-appropriate experiences.</div>

                    <div class="setting-row">
                        <div class="setting-field">
                            <label for="nickname">Nickname</label>
                            <input type="text" id="nickname" name="nickname" value="<?php echo e($nickname); ?>" maxlength="32" placeholder="Auto-generated if left empty">
                        </div>
                        <?php renderPrivacySelect('nickname', $privacySettings, $defaultSettings); ?>
                    </div>
                    <div class="settings-help">Displayed to other users instead of your email.</div>

                    <div class="setting-row">
                        <div class="setting-field">
                            <label for="address">Address</label>
                            <input type="text" id="address" name="address" value="<?php echo e($address); ?>" maxlength="200">
                        </div>
                        <?php renderPrivacySelect('address', $privacySettings, $defaultSettings); ?>
                    </div>
                    <div class="settings-help">Used for billing or delivery when applicable.</div>

                    <div class="setting-row">
                        <div class="setting-field">
                            <label for="country">Country</label>
                            <div class="select-field" data-select>
                                <input type="hidden" id="country" name="country" value="<?php echo e($country); ?>" data-select-value>
                                <button type="button" class="select-trigger" data-select-trigger>
                                    <span data-select-label><?php echo e($country ?: 'Select country'); ?></span>
                                    <span class="select-caret" aria-hidden="true"></span>
                                </button>
                                <div class="select-menu" data-select-menu>
                                    <button type="button" class="select-option" data-select-option data-value="">Select country</button>
                                    <?php foreach ($countries as $countryOption): ?>
                                        <button type="button" class="select-option" data-select-option data-value="<?php echo e($countryOption); ?>">
                                            <?php echo e($countryOption); ?>
                                        </button>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                        <?php renderPrivacySelect('country', $privacySettings, $defaultSettings); ?>
                    </div>
                    <div class="settings-help">Required for tax calculations and regional settings.</div>

                    <div class="setting-row">
                        <div class="setting-field">
                            <label for="postal_code">Postal Code</label>
                            <input type="text" id="postal_code" name="postal_code" value="<?php echo e($postalCode); ?>" maxlength="20">
                        </div>
                        <?php renderPrivacySelect('postal_code', $privacySettings, $defaultSettings); ?>
                    </div>
                    <div class="settings-help">Used to validate billing or shipping details.</div>

                    <div class="setting-row">
                        <div class="setting-field">
                            <label for="profile_photo">Profile Photo (max 5MB)</label>
                            <input type="file" id="profile_photo" name="profile_photo" accept="image/jpeg,image/png,image/webp">
                        </div>
                        <?php renderPrivacySelect('profile_photo', $privacySettings, $defaultSettings); ?>
                    </div>
                    <div class="settings-help">Shown on your account and messages.</div>

                    <div class="settings-section-divider"></div>

                    <div class="setting-row setting-row-toggle">
                        <div class="setting-field">
                            <label for="is_searchable" class="toggle-label">
                                <span class="toggle-title">Visible in Search</span>
                                <span class="toggle-description">Allow others to find you when searching for friends</span>
                            </label>
                        </div>
                        <div class="toggle-switch">
                            <input type="checkbox" id="is_searchable" name="is_searchable" value="1" <?php echo $isSearchable ? 'checked' : ''; ?>>
                            <span class="toggle-slider" aria-hidden="true"></span>
                        </div>
                    </div>
                    <div class="settings-help">When disabled, your profile won't appear in friend searches. You can still receive friend requests via direct profile link.</div>

                    <button type="submit" class="btn btn-primary">Save Changes</button>
                </form>

                <div class="settings-privacy">
                    <h3>Data & Account</h3>
                    <p class="auth-subtitle">Export or delete your personal data.</p>
                    <form method="POST" action="/account?tab=settings" class="auth-form">
                        <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                        <input type="hidden" name="action" value="export_data">
                        <button type="submit" class="btn btn-outline btn-sm">Send Via Email</button>
                        <div class="settings-help">We will email your export to the address on your account.</div>
                    </form>
                    <form method="POST" action="/account?tab=settings" class="auth-form" data-confirm="This will permanently delete your account data. Continue?">
                        <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                        <input type="hidden" name="action" value="delete_account">
                        <label for="delete_confirm">Type DELETE to confirm</label>
                        <input type="text" id="delete_confirm" name="delete_confirm" placeholder="DELETE">
                        <button type="submit" class="btn btn-danger btn-sm">Delete My Account</button>
                        <div class="settings-help">This permanently removes your account and related data.</div>
                    </form>
                </div>
            </div>

            <div class="account-panel <?php echo $activeTab === 'friends' ? '' : 'is-hidden'; ?>" data-account-panel="friends">
                <div class="friends-container-inline">
                    <?php if ($friendFlashMessage): ?>
                        <div class="alert alert-success"><?php echo e($friendFlashMessage); ?></div>
                    <?php endif; ?>
                    <?php if ($friendFlashError): ?>
                        <div class="alert alert-error"><?php echo e($friendFlashError); ?></div>
                    <?php endif; ?>

                    <!-- Search Section -->
                    <div class="friends-search-section">
                        <form method="get" action="/account" class="friends-search-form">
                            <input type="hidden" name="tab" value="friends">
                            <input type="hidden" name="ftab" value="<?php echo e($friendsTab); ?>">
                            <div class="search-input-wrapper">
                                <svg viewBox="0 0 24 24" width="20" height="20" fill="none" stroke="currentColor" stroke-width="2" class="search-icon">
                                    <circle cx="11" cy="11" r="8"></circle>
                                    <line x1="21" y1="21" x2="16.65" y2="16.65"></line>
                                </svg>
                                <input type="text" name="fq" value="<?php echo e($friendSearchQuery); ?>" placeholder="Search by @nickname or name..." class="friends-search-input">
                            </div>
                            <button type="submit" class="btn btn-primary btn-sm">Search</button>
                            <a href="#" class="btn btn-outline btn-sm <?php echo $friendSearchQuery ? '' : 'is-hidden'; ?>" data-friends-search-clear>Clear</a>
                        </form>
                    </div>

                    <!-- Search Results Container -->
                    <div data-friends-search-results class="<?php echo ($friendSearchQuery && !empty($friendSearchResults)) ? '' : 'is-hidden'; ?>">
                        <?php if ($friendSearchQuery && !empty($friendSearchResults)): ?>
                        <div class="friends-section">
                            <h3 class="friends-section-title">Search Results for "<?php echo e($friendSearchQuery); ?>"</h3>
                            <div class="friends-grid">
                                <?php foreach ($friendSearchResults as $searchUser):
                                    $status = FriendService::getStatus((int)$userId, (int)$searchUser['id']);
                                    ?>
                                <div class="friend-card">
                                    <a href="/@<?php echo e($searchUser['nickname']); ?>" class="friend-card-photo-link">
                                        <?php if (!empty($searchUser['profile_photo'])): ?>
                                            <img src="<?php echo e(toPublicFilePath($searchUser['profile_photo'])); ?>" alt="" class="friend-card-photo">
                                        <?php else: ?>
                                            <div class="friend-card-photo-placeholder">
                                                <?php echo strtoupper(substr($searchUser['display_name'] ?? $searchUser['nickname'] ?? '?', 0, 1)); ?>
                                            </div>
                                        <?php endif; ?>
                                    </a>
                                    <div class="friend-card-info">
                                        <a href="/@<?php echo e($searchUser['nickname']); ?>" class="friend-card-name">
                                            <?php echo e($searchUser['display_name'] ?: $searchUser['nickname']); ?>
                                        </a>
                                        <span class="friend-card-nickname">@<?php echo e($searchUser['nickname']); ?></span>
                                    </div>
                                    <div class="friend-card-actions">
                                        <?php if ($status === 'friends'): ?>
                                            <span class="friend-status-badge">✓ Friends</span>
                                        <?php elseif ($status === 'request_sent'): ?>
                                            <span class="friend-status-badge pending">Pending</span>
                                        <?php elseif ($status === 'request_received'): ?>
                                            <span class="friend-status-badge incoming">Accept?</span>
                                        <?php elseif ($status === 'blocked'): ?>
                                            <span class="friend-status-badge blocked">Blocked</span>
                                        <?php else: ?>
                                            <button type="button" class="btn btn-icon btn-primary has-tooltip" data-friend-action="send_request" data-target-user="<?php echo $searchUser['id']; ?>" data-tooltip="Send Friend Request">
                                                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round"><line x1="12" y1="5" x2="12" y2="19"/><line x1="5" y1="12" x2="19" y2="12"/></svg>
                                            </button>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        <?php elseif ($friendSearchQuery): ?>
                        <div class="friends-empty">
                            <p>No users found matching "<?php echo e($friendSearchQuery); ?>"</p>
                        </div>
                        <?php endif; ?>
                    </div>

                    <!-- Tabs -->
                    <div class="friends-tabs" data-friends-tabs data-friends-tab-default="<?php echo e($friendsTab); ?>">
                        <button type="button" class="friends-tab <?php echo $friendsTab === 'all' ? 'active' : ''; ?>" data-friends-tab="all">
                            All Friends
                            <?php if ($friendsCount > 0): ?>
                                <span class="tab-count"><?php echo $friendsCount; ?></span>
                            <?php endif; ?>
                        </button>
                        <button type="button" class="friends-tab <?php echo $friendsTab === 'pending' ? 'active' : ''; ?>" data-friends-tab="pending">
                            Pending Requests
                            <?php if ($pendingFriendCount > 0): ?>
                                <span class="tab-count tab-count-alert"><?php echo $pendingFriendCount; ?></span>
                            <?php endif; ?>
                        </button>
                        <button type="button" class="friends-tab <?php echo $friendsTab === 'sent' ? 'active' : ''; ?>" data-friends-tab="sent">
                            Sent Requests
                        </button>
                        <button type="button" class="friends-tab <?php echo $friendsTab === 'blocked' ? 'active' : ''; ?>" data-friends-tab="blocked">
                            Blocked
                        </button>
                    </div>

                    <!-- Tab Content -->
                    <div class="friends-panel <?php echo $friendsTab === 'all' ? '' : 'is-hidden'; ?>" data-friends-panel="all">
                        <?php if (count($pendingRequests) > 0): ?>
                        <div class="friends-section pending-highlight">
                            <h3 class="friends-section-title">
                                <span class="pending-dot"></span>
                                <?php echo count($pendingRequests); ?> Pending Request<?php echo count($pendingRequests) !== 1 ? 's' : ''; ?>
                            </h3>
                            <div class="friends-grid">
                                <?php foreach ($pendingRequests as $request): ?>
                                <div class="friend-card friend-card-pending">
                                    <a href="/@<?php echo e($request['nickname']); ?>" class="friend-card-photo-link">
                                        <?php if (!empty($request['profile_photo'])): ?>
                                            <img src="<?php echo e(toPublicFilePath($request['profile_photo'])); ?>" alt="" class="friend-card-photo">
                                        <?php else: ?>
                                            <div class="friend-card-photo-placeholder">
                                                <?php echo strtoupper(substr($request['display_name'] ?? $request['nickname'] ?? '?', 0, 1)); ?>
                                            </div>
                                        <?php endif; ?>
                                    </a>
                                    <div class="friend-card-info">
                                        <a href="/@<?php echo e($request['nickname']); ?>" class="friend-card-name">
                                            <?php echo e($request['display_name'] ?: $request['nickname']); ?>
                                        </a>
                                        <span class="friend-card-nickname">@<?php echo e($request['nickname']); ?></span>
                                    </div>
                                    <div class="friend-card-actions">
                                        <form method="post" action="/account?tab=friends" class="inline-form">
                                            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                            <input type="hidden" name="friend_action" value="accept_request">
                                            <input type="hidden" name="request_id" value="<?php echo $request['id']; ?>">
                                            <input type="hidden" name="return_url" value="/account?tab=friends&ftab=all">
                                            <button type="submit" class="btn btn-primary btn-sm">Accept</button>
                                        </form>
                                        <form method="post" action="/account?tab=friends" class="inline-form">
                                            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                            <input type="hidden" name="friend_action" value="decline_request">
                                            <input type="hidden" name="request_id" value="<?php echo $request['id']; ?>">
                                            <input type="hidden" name="return_url" value="/account?tab=friends&ftab=all">
                                            <button type="submit" class="btn btn-outline btn-sm">Decline</button>
                                        </form>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        <?php endif; ?>

                        <div class="friends-section">
                            <h3 class="friends-section-title">All Friends (<?php echo $friendsCount; ?>)</h3>
                            <div class="friends-grid" data-friends-content="all">
                                <?php if (empty($friends)): ?>
                                <div class="friends-empty">
                                    <p>You don't have any friends yet.</p>
                                    <p class="text-muted">Search for users by their @nickname to connect!</p>
                                </div>
                                <?php else: ?>
                                    <?php foreach ($friends as $friend): ?>
                                    <div class="friend-card">
                                        <a href="/@<?php echo e($friend['nickname']); ?>" class="friend-card-photo-link">
                                            <?php if (!empty($friend['profile_photo'])): ?>
                                                <img src="<?php echo e(toPublicFilePath($friend['profile_photo'])); ?>" alt="" class="friend-card-photo">
                                            <?php else: ?>
                                                <div class="friend-card-photo-placeholder">
                                                    <?php echo strtoupper(substr($friend['display_name'] ?? $friend['nickname'] ?? '?', 0, 1)); ?>
                                                </div>
                                            <?php endif; ?>
                                        </a>
                                        <div class="friend-card-info">
                                            <a href="/@<?php echo e($friend['nickname']); ?>" class="friend-card-name">
                                                <?php echo e($friend['display_name'] ?: $friend['nickname']); ?>
                                            </a>
                                            <span class="friend-card-nickname">@<?php echo e($friend['nickname']); ?></span>
                                        </div>
                                        <div class="friend-card-actions">
                                            <a href="/@<?php echo e($friend['nickname']); ?>" class="btn btn-outline btn-sm">View</a>
                                            <button type="button" class="btn btn-outline btn-sm btn-danger" data-friend-action="remove_friend" data-target-user="<?php echo $friend['id']; ?>">Remove</button>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <div class="friends-panel <?php echo $friendsTab === 'pending' ? '' : 'is-hidden'; ?>" data-friends-panel="pending">
                        <div class="friends-section">
                            <h3 class="friends-section-title">Pending Friend Requests</h3>
                            <div class="friends-grid" data-friends-content="pending">
                                <?php if (empty($pendingRequests)): ?>
                                <div class="friends-empty">
                                    <p>No pending friend requests.</p>
                                </div>
                                <?php else: ?>
                                    <?php foreach ($pendingRequests as $request): ?>
                                    <div class="friend-card friend-card-pending">
                                        <a href="/@<?php echo e($request['nickname']); ?>" class="friend-card-photo-link">
                                            <?php if (!empty($request['profile_photo'])): ?>
                                                <img src="<?php echo e(toPublicFilePath($request['profile_photo'])); ?>" alt="" class="friend-card-photo">
                                            <?php else: ?>
                                                <div class="friend-card-photo-placeholder">
                                                    <?php echo strtoupper(substr($request['display_name'] ?? $request['nickname'] ?? '?', 0, 1)); ?>
                                                </div>
                                            <?php endif; ?>
                                        </a>
                                        <div class="friend-card-info">
                                            <a href="/@<?php echo e($request['nickname']); ?>" class="friend-card-name">
                                                <?php echo e($request['display_name'] ?: $request['nickname']); ?>
                                            </a>
                                            <span class="friend-card-nickname">@<?php echo e($request['nickname']); ?></span>
                                            <span class="friend-card-date">Received <?php echo date('M j', strtotime($request['created_at'])); ?></span>
                                        </div>
                                        <div class="friend-card-actions">
                                            <button type="button" class="btn btn-primary btn-sm" data-friend-action="accept_request" data-request-id="<?php echo $request['id']; ?>">Accept</button>
                                            <button type="button" class="btn btn-outline btn-sm" data-friend-action="decline_request" data-request-id="<?php echo $request['id']; ?>">Decline</button>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <div class="friends-panel <?php echo $friendsTab === 'sent' ? '' : 'is-hidden'; ?>" data-friends-panel="sent">
                        <div class="friends-section">
                            <h3 class="friends-section-title">Sent Friend Requests</h3>
                            <div class="friends-grid" data-friends-content="sent">
                                <?php if (empty($sentRequests)): ?>
                                <div class="friends-empty">
                                    <p>No pending sent requests.</p>
                                </div>
                                <?php else: ?>
                                    <?php foreach ($sentRequests as $request): ?>
                                    <div class="friend-card">
                                        <a href="/@<?php echo e($request['nickname']); ?>" class="friend-card-photo-link">
                                            <?php if (!empty($request['profile_photo'])): ?>
                                                <img src="<?php echo e(toPublicFilePath($request['profile_photo'])); ?>" alt="" class="friend-card-photo">
                                            <?php else: ?>
                                                <div class="friend-card-photo-placeholder">
                                                    <?php echo strtoupper(substr($request['display_name'] ?? $request['nickname'] ?? '?', 0, 1)); ?>
                                                </div>
                                            <?php endif; ?>
                                        </a>
                                        <div class="friend-card-info">
                                            <a href="/@<?php echo e($request['nickname']); ?>" class="friend-card-name">
                                                <?php echo e($request['display_name'] ?: $request['nickname']); ?>
                                            </a>
                                            <span class="friend-card-nickname">@<?php echo e($request['nickname']); ?></span>
                                            <span class="friend-card-date">Sent <?php echo date('M j', strtotime($request['created_at'])); ?></span>
                                        </div>
                                        <div class="friend-card-actions">
                                            <span class="friend-status-badge pending">Awaiting response</span>
                                            <button type="button" class="btn btn-outline btn-sm" data-friend-action="cancel_request" data-request-id="<?php echo $request['id']; ?>">Cancel</button>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <div class="friends-panel <?php echo $friendsTab === 'blocked' ? '' : 'is-hidden'; ?>" data-friends-panel="blocked">
                        <div class="friends-section">
                            <h3 class="friends-section-title">Blocked Users</h3>
                            <div class="friends-grid" data-friends-content="blocked">
                                <?php if (empty($blockedUsers)): ?>
                                <div class="friends-empty">
                                    <p>You haven't blocked anyone.</p>
                                </div>
                                <?php else: ?>
                                    <?php foreach ($blockedUsers as $blocked): ?>
                                    <div class="friend-card friend-card-blocked">
                                        <div class="friend-card-photo-link">
                                            <?php if (!empty($blocked['profile_photo'])): ?>
                                                <img src="<?php echo e(toPublicFilePath($blocked['profile_photo'])); ?>" alt="" class="friend-card-photo">
                                            <?php else: ?>
                                                <div class="friend-card-photo-placeholder">
                                                    <?php echo strtoupper(substr($blocked['display_name'] ?? $blocked['nickname'] ?? '?', 0, 1)); ?>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        <div class="friend-card-info">
                                            <span class="friend-card-name"><?php echo e($blocked['display_name'] ?: $blocked['nickname']); ?></span>
                                            <span class="friend-card-nickname">@<?php echo e($blocked['nickname']); ?></span>
                                            <span class="friend-card-date">Blocked <?php echo date('M j, Y', strtotime($blocked['blocked_at'])); ?></span>
                                        </div>
                                        <div class="friend-card-actions">
                                            <button type="button" class="btn btn-outline btn-sm" data-friend-action="unblock_user" data-target-user="<?php echo $blocked['id']; ?>">Unblock</button>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>
