<?php
/**
 * Front Controller - Handles all public requests
 * Logging is handled globally in _init.php
 */

require_once __DIR__ . '/../vendor/autoload.php'; // Composer PSR-4 + bootstrap

use NewSite\Cache\CacheService as Cache;
use NewSite\Database\DatabaseManager;
use NewSite\Template\TemplateRenderer;
use NewSite\Auth\AdminAuth;
use NewSite\Auth\GoogleOAuthService;
use NewSite\Auth\SiteAuth;
use NewSite\Cleanup\MissingUploadsService;
use NewSite\Config\SetupService;
use NewSite\Contact\ContactService;
use NewSite\Database\DbHelper;
use NewSite\Email\SmtpMailer;
use NewSite\Gdpr\GdprService;
use NewSite\Logging\LogService;
use NewSite\Menu\MenuService;
use NewSite\Security\IpBanService;
use NewSite\Security\IpService;
use NewSite\Security\RateLimiter;
use NewSite\Settings\SettingsService;
use NewSite\Shop\CartService;
use NewSite\Shop\CurrencyService;
use NewSite\Template\ContentRenderer;
use NewSite\Upload\UploadService;
use NewSite\User\FriendService;
use NewSite\User\PrivacyService;
use NewSite\User\UserMessageService;
use NewSite\User\UserService;
use NewSite\Util\AssetVersioning;
use NewSite\Util\CountryData;
use NewSite\Util\CurrencyData;
use NewSite\Util\SlugGenerator;
use NewSite\Visitor\VisitorService;

const HEADER_LOCATION = 'Location: ';
const ROUTE_LOGIN = '/login';
const ACCOUNT_SETTINGS_ROUTE = '/account?tab=settings';
const ACCOUNT_MESSAGES_ROUTE = '/account?tab=messages';
const ACCOUNT_PURCHASES_ROUTE = '/account?tab=purchases';
const SLUG_SANITIZE_PATTERN = '/[^a-z0-9\-]/';
const SQL_SELECT_PUBLISHED_PAGE_BY_SLUG = "SELECT * FROM pages WHERE slug = ? AND is_published = 1";
const SQL_SELECT_PRODUCTS_LIST_PAGE_SLUG = "SELECT p.slug FROM pages p JOIN sections s ON s.page_id = p.id WHERE s.section_type = 'products_list' AND p.is_published = 1 ORDER BY p.id ASC LIMIT 1";
const SQL_HAS_ACTIVE_PRODUCTS_LIST_SECTION = "SELECT 1 FROM sections WHERE page_id = ? AND section_type = 'products_list' AND is_active = 1 LIMIT 1";
const COLLECTION_ROUTE_SEGMENT = '/collection/';
const FORUM_ROUTE = '/forum';
const FORUM_POST_ROUTE = '/forum/post/';
const FORUM_CATEGORY_ROUTE = '/forum/category/';
const FORUM_SUBCATEGORY_ROUTE = '/forum/subcategory/';
const ACCOUNT_ROUTE = '/account';
const CONTACT_SENT_ROUTE = '/contact?sent=1';
const DMCA_SENT_ROUTE = '/dmca?sent=1';
const FILE_NOT_FOUND_MESSAGE = 'File not found';
const IMAGE_NOT_FOUND_MESSAGE = 'Image not found';
const NOT_FOUND_PAGE_TITLE = 'Page Not Found';
const NOT_FOUND_META_TITLE = '404 - Page Not Found';
const NOT_FOUND_META_DESCRIPTION = 'The page you are looking for does not exist.';
const FORUM_POST_DEFAULT_TITLE = 'Forum Post';
const FORUM_META_TITLE_SUFFIX = ' - Forum';
const SQL_UPDATE_SECTION_SETTINGS = "UPDATE sections SET settings = ? WHERE id = ?";
const HEADER_ETAG = 'ETag: ';
const HEADER_CONTENT_TYPE = 'Content-Type: ';
const HEADER_CONTENT_LENGTH = 'Content-Length: ';
const HEADER_LAST_MODIFIED = 'Last-Modified: ';
const HTTP_GMT_DATE_FORMAT = 'D, d M Y H:i:s';
const CACHE_CONTROL_PUBLIC_IMMUTABLE = 'Cache-Control: public, max-age=31536000, immutable';
const CACHE_CONTROL_PRIVATE_IMMUTABLE = 'Cache-Control: private, max-age=31536000, immutable';
const ADMIN_FILE_PATH_SEGMENT = '/admin-file/';
const SITE_FILE_PATH_SEGMENT = '/site-file/';
const PROFILE_PHOTO_PATH_SEGMENT = '/profile-photo/';
const DATA_PROFILE_PHOTOS_SEGMENT = '/../data/profile_photos/';
const DATA_ADMIN_UPLOADS_SEGMENT = '/../data/admin_uploads/';
const PATH_TRAVERSAL_REGEX = '#(^|/)\\.\\.(/|$)#';
const IMAGE_MIME_JPEG = 'image/jpeg';
const IMAGE_MIME_PNG = 'image/png';
const IMAGE_MIME_WEBP = 'image/webp';
const IMAGE_MIME_GIF = 'image/gif';
const ALLOWED_IMAGE_MIMES = [IMAGE_MIME_JPEG, IMAGE_MIME_PNG, IMAGE_MIME_WEBP];
const ALLOWED_IMAGE_MIMES_WITH_GIF = [IMAGE_MIME_JPEG, IMAGE_MIME_PNG, IMAGE_MIME_GIF, IMAGE_MIME_WEBP];
const BLOCKED_EXECUTABLE_EXTENSIONS = ['php', 'php3', 'php4', 'php5', 'phtml', 'phar', 'cgi', 'pl', 'asp', 'aspx', 'js', 'mjs', 'exe', 'dll', 'bat', 'cmd', 'com', 'msi', 'sh'];
const BLOCKED_SCRIPT_MIMES = [
    'application/x-php',
    'application/x-httpd-php',
    'text/x-php',
    'application/x-msdownload',
    'application/x-dosexec',
    'application/x-executable',
    'application/x-sh',
    'text/x-shellscript',
];
const UPLOAD_IMAGE_MIME_MAP = [
    'jpg' => [IMAGE_MIME_JPEG],
    'jpeg' => [IMAGE_MIME_JPEG],
    'png' => [IMAGE_MIME_PNG],
    'webp' => [IMAGE_MIME_WEBP],
];

if (!function_exists('respondNotFoundFile')) {
    function respondNotFoundFile(string $message = FILE_NOT_FOUND_MESSAGE): void
    {
        http_response_code(404);
        echo $message;
        exit;
    }
}

// Check if IP is banned - show connection error
if (IpBanService::isBanned()) {
    // Simulate a connection failure for banned users
    http_response_code(503);
    header('Connection: close');
    header('Retry-After: 3600');
    // Return minimal error that looks like a network failure
    echo '<!DOCTYPE html><html><head><title>Error</title></head><body><h1>This site can\'t be reached</h1><p>Connection was reset. Please check your internet connection and try again.</p></body></html>';
    exit;
}

// Maintenance mode (admins can still access)
$maintenanceMode = SettingsService::get('maintenance_mode', '0') === '1';
if ($maintenanceMode && !AdminAuth::isAllowedIp(IpService::getClientIP())) {
    http_response_code(503);
    TemplateRenderer::renderOnce(__DIR__ . '/templates/maintenance.php');
    exit;
}

// Track this visitor
VisitorService::track();

// Cleanup missing upload references (daily)
if (SetupService::shouldRunDailyTask('missing_uploads_cleanup')) {
    MissingUploadsService::cleanupReferences();
}

// GDPR retention cleanup (daily)
if (SetupService::shouldRunDailyTask('gdpr_retention_cleanup')) {
    GdprService::runRetentionCleanup();
}

// Cache cleanup (daily)
if (SetupService::shouldRunDailyTask('cache_cleanup')) {
    Cache::cleanup();
}

// Get the requested route
$route = isset($_GET['route']) ? trim($_GET['route'], '/') : 'home';

// Dynamic CSS — served through front controller to avoid standalone bootstrap
$dynamicCssAllowed = ['assets/css/theme.php'];
if (in_array($route, $dynamicCssAllowed, true)) {
    TemplateRenderer::renderOnce(__DIR__ . '/' . $route);
    exit;
}

// Parse the route
$segments = explode('/', $route);
$pageSlug = $segments[0] ?: 'home';

if ($pageSlug === 'sitemap.xml') {
    header('Content-Type: application/xml; charset=UTF-8');
    header('Cache-Control: public, max-age=3600');

    $db = DatabaseManager::getWriteConnection();
    $baseUrl = rtrim(SetupService::getBaseUrl(), '/');

    $formatLastmod = function ($value): ?string {
        if ($value === null || $value === '') {
            return null;
        }
        $timestamp = is_numeric($value) ? (int)$value : strtotime((string)$value);
        if ($timestamp <= 0) {
            return null;
        }
        return gmdate('c', $timestamp);
    };

    $urls = [];
    $addUrl = function (string $loc, ?string $lastmod = null) use (&$urls) {
        $urls[$loc] = [
            'loc' => $loc,
            'lastmod' => $lastmod
        ];
    };

    $addUrl($baseUrl . '/', null);

    $pagesStmt = $db->query("SELECT slug, updated_at, created_at FROM pages WHERE is_published = 1");
    foreach ($pagesStmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $slug = trim((string)($row['slug'] ?? ''));
        if ($slug === '') {
            continue;
        }
        $path = $slug === 'home' ? '/' : '/' . $slug;
        $addUrl($baseUrl . $path, $formatLastmod($row['updated_at'] ?? $row['created_at'] ?? null));
    }

    $productPageSlug = 'product';
    $productPageStmt = $db->prepare("SELECT slug FROM pages WHERE slug = ? AND is_published = 1 LIMIT 1");
    $productPageStmt->execute([$productPageSlug]);
    $productPageSlug = trim((string)$productPageStmt->fetchColumn()) ?: 'product';

    $productsStmt = $db->query("SELECT p.id, p.product_slug, p.updated_at, p.created_at, pg.slug AS page_slug FROM products p LEFT JOIN pages pg ON pg.id = p.page_id WHERE p.is_active = 1");
    foreach ($productsStmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $slug = trim((string)($row['page_slug'] ?? ''));
        if ($slug === '') {
            $slug = $productPageSlug;
        }
        $productSlug = trim((string)($row['product_slug'] ?? ''));
        if ($productSlug !== '') {
            $path = '/' . $slug . '/' . rawurlencode($productSlug);
        } else {
            $path = '/' . $slug . '?product=' . (int)$row['id'];
        }
        $addUrl($baseUrl . $path, $formatLastmod($row['updated_at'] ?? $row['created_at'] ?? null));
    }

    $fallbackProductsListSlug = '';
    $fallbackStmt = $db->query(SQL_SELECT_PRODUCTS_LIST_PAGE_SLUG);
    if ($fallbackStmt) {
        $fallbackProductsListSlug = trim((string)$fallbackStmt->fetchColumn());
    }

    $collectionsStmt = $db->query("SELECT c.id, c.collection_slug, c.updated_at, c.created_at, pg.slug AS target_slug FROM collections c LEFT JOIN pages pg ON pg.id = c.target_page_id AND pg.is_published = 1");
    foreach ($collectionsStmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $slug = trim((string)($row['target_slug'] ?? ''));
        if ($slug === '') {
            $slug = $fallbackProductsListSlug;
        }
        if ($slug === '') {
            continue;
        }
        $collectionSlug = trim((string)($row['collection_slug'] ?? ''));
        if ($collectionSlug !== '') {
            $path = '/' . $slug . COLLECTION_ROUTE_SEGMENT . rawurlencode($collectionSlug);
        } else {
            $path = '/' . $slug . '?collection=' . (int)$row['id'];
        }
        $addUrl($baseUrl . $path, $formatLastmod($row['updated_at'] ?? $row['created_at'] ?? null));
    }

    $forumEnabledForSitemap = SettingsService::get('forum_enabled', '0') === '1';
    if ($forumEnabledForSitemap) {
        try {
            $latestForumUpdateStmt = $db->query("SELECT MAX(updated_at) FROM forum_posts WHERE status = 'approved'");
            $latestForumUpdate = $latestForumUpdateStmt ? $latestForumUpdateStmt->fetchColumn() : null;
            $addUrl($baseUrl . FORUM_ROUTE, $formatLastmod($latestForumUpdate));

            $forumPostsStmt = $db->query("SELECT p.id, p.slug, p.updated_at, p.created_at FROM forum_posts p JOIN forum_subcategories s ON s.id = p.subcategory_id JOIN forum_categories c ON c.id = s.category_id AND c.is_active = 1 WHERE p.status = 'approved' ORDER BY p.id ASC");
            foreach ($forumPostsStmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
                $postId = (int)($row['id'] ?? 0);
                if ($postId <= 0) {
                    continue;
                }
                $postSlug = trim((string)($row['slug'] ?? ''));
                $postPath = $postSlug !== '' ? rawurlencode($postSlug) : (string)$postId;
                $addUrl($baseUrl . FORUM_POST_ROUTE . $postPath, $formatLastmod($row['updated_at'] ?? $row['created_at'] ?? null));
            }
        } catch (PDOException $e) {
            // Forum tables may not exist yet on fresh installs.
        }
    }

    $escapeXml = function (string $value): string {
        return htmlspecialchars($value, ENT_XML1 | ENT_COMPAT, 'UTF-8');
    };

    $xml = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
    $xml .= "<urlset xmlns=\"http://www.sitemaps.org/schemas/sitemap/0.9\">\n";
    foreach ($urls as $entry) {
        $xml .= "  <url>\n";
        $xml .= "    <loc>" . $escapeXml($entry['loc']) . "</loc>\n";
        if (!empty($entry['lastmod'])) {
            $xml .= "    <lastmod>" . $escapeXml($entry['lastmod']) . "</lastmod>\n";
        }
        $xml .= "  </url>\n";
    }
    $xml .= "</urlset>";

    echo $xml;
    exit;
}

if ($pageSlug === 'robots.txt') {
    header('Content-Type: text/plain; charset=UTF-8');
    header('Cache-Control: public, max-age=3600');
    $baseUrl = rtrim(SetupService::getBaseUrl(), '/');
    $lines = [
        'User-agent: *',
        'Disallow: /admin',
        'Disallow: /admin/',
        'Disallow: /account',
        'Disallow: /account-settings',
        'Disallow: /login',
        'Disallow: /logout',
        'Disallow: /messages',
        'Disallow: /friends',
        'Disallow: /checkout',
        'Disallow: /checkout-success',
        'Disallow: /cart',
        'Disallow: /download',
        'Disallow: /templates/',
        'Disallow: /api/',
        'Sitemap: ' . $baseUrl . '/sitemap.xml'
    ];
    echo implode("\n", $lines) . "\n";
    exit;
}
$isCartPage = $pageSlug === 'cart';
$isSearchPage = $pageSlug === 'search';
$isLoginPage = $pageSlug === 'login';
$isGoogleCallback = $pageSlug === 'auth'
    && ($segments[1] ?? '') === 'google'
    && ($segments[2] ?? '') === 'callback';
$isAccountPage = $pageSlug === 'account';
$isLogoutPage = $pageSlug === 'logout';
$isMessagesPage = $pageSlug === 'messages';
$isAccountSettingsPage = $pageSlug === 'account-settings';
$isCheckoutPage = $pageSlug === 'checkout';
$isCheckoutSuccessPage = $pageSlug === 'checkout-success';
$isFriendsPage = $pageSlug === 'friends';
$isDmcaPage = $pageSlug === 'dmca';
$isProfilePage = (strlen($pageSlug) > 1 && $pageSlug[0] === '@');
$profileNickname = $isProfilePage ? substr($pageSlug, 1) : null;
$isForumPage = $pageSlug === 'forum';
$isGamesPage = $pageSlug === 'games';
$gamesSubRouteRaw = $isGamesPage && isset($segments[1]) ? (string)$segments[1] : '';
$gamesSubRoute = preg_replace(SLUG_SANITIZE_PATTERN, '', strtolower(trim($gamesSubRouteRaw)));
if (!is_string($gamesSubRoute)) {
    $gamesSubRoute = '';
}
$mazeRunnerInternalSlug = 'maze-runner';
$mazeRunnerPublicSlug = \NewSite\Minigames\MinigameCatalog::getPublicSlug($mazeRunnerInternalSlug);
$isMazeRunnerLegacyRoute = $isGamesPage && $gamesSubRoute === $mazeRunnerInternalSlug;
if ($isMazeRunnerLegacyRoute) {
    $redirectQuery = $_GET;
    unset($redirectQuery['route']);

    $redirectTarget = '/games/' . rawurlencode($mazeRunnerPublicSlug);
    if (!empty($redirectQuery)) {
        $redirectTarget .= '?' . http_build_query($redirectQuery);
    }

    header(HEADER_LOCATION . $redirectTarget, true, 301);
    exit;
}
$isMazeRunnerPage = $isGamesPage && $gamesSubRoute === $mazeRunnerPublicSlug;
$isTextFinderReplacerPage = $pageSlug === 'text-finder-replacer';
$forumEnabled = SettingsService::get('forum_enabled', '0') === '1';

// Forum post routing: support both slug-based (/forum/post/{slug}) and legacy numeric (/forum/post/{id})
$forumPostId = 0;
$forumPostSlug = '';
if ($isForumPage && isset($segments[1]) && $segments[1] === 'post' && isset($segments[2])) {
    $rawPostIdentifier = trim((string)$segments[2]);
    if ($rawPostIdentifier !== '' && ctype_digit($rawPostIdentifier)) {
        // Legacy numeric ID — resolve slug for 301 redirect
        $forumPostId = (int)$rawPostIdentifier;
    } else {
        // Slug-based URL — resolve post ID from slug
        $forumPostSlug = $rawPostIdentifier;
    }
}
$isForumPostPage = $isForumPage && ($forumPostId > 0 || $forumPostSlug !== '');
$searchQuery = trim($_GET['q'] ?? '');
$searchShowMini = SettingsService::get('search_show_mini', '1') === '1';
$searchShowTop = SettingsService::get('search_show_top', '0') === '1';
$searchShowSidebar = SettingsService::get('search_show_sidebar', '0') === '1';
$searchShowFooter = SettingsService::get('search_show_footer', '0') === '1';
$easyMediaAiZipDefault = ContentRenderer::getDefaultEasyMediaAiZipDownloadPath();
$accountShowMini = SettingsService::get('account_show_mini', '0') === '1';
$accountShowTop = SettingsService::get('account_show_top', '1') === '1';
$accountShowSidebar = SettingsService::get('account_show_sidebar', '1') === '1';
$accountShowFooter = SettingsService::get('account_show_footer', '0') === '1';
$cartShowMini = SettingsService::get('cart_show_mini', '0') === '1';
$cartShowTop = SettingsService::get('cart_show_top', '1') === '1';
$cartShowSidebar = SettingsService::get('cart_show_sidebar', '1') === '1';
$cartShowFooter = SettingsService::get('cart_show_footer', '0') === '1';
$isUserLoggedIn = SiteAuth::isLoggedIn();
// account.css only loads on actual account/auth/profile/friends pages
$shouldLoadAccountCss = $isUserLoggedIn
    || $isLoginPage
    || $isAccountPage
    || $isFriendsPage
    || $isProfilePage;
$accountUrl = $isUserLoggedIn ? '/account' : '/login';
$accountLabel = $isUserLoggedIn ? 'Account' : 'Login';
$accountPhoto = '';
if ($isUserLoggedIn) {
    $accountUser = UserService::getById((int)($_SESSION['site_user_id'] ?? 0));
    $accountPhoto = $accountUser['profile_photo'] ?? '';
}
$notificationLocation = SettingsService::get('notification_location', 'mini');

// Server-side cart badge count — eliminates /cart-api fetch on initial page load
$cartItemCount = CartService::getItemCount();
$cartBadgeHidden = $cartItemCount <= 0 ? ' is-hidden' : '';
$allowedNotificationLocations = ['mini', 'top', 'sidebar'];
if (!in_array($notificationLocation, $allowedNotificationLocations, true)) {
    $notificationLocation = 'mini';
}

// Get pending friend request count for logged-in users
$pendingFriendRequests = 0;
if ($isUserLoggedIn) {
    $pendingFriendRequests = FriendService::getPendingCount((int)$_SESSION['site_user_id']);
}

if ($isAccountPage && !$isUserLoggedIn) {
    header(HEADER_LOCATION . ROUTE_LOGIN);
    exit;
}

// Friends page requires login
if ($isFriendsPage && !$isUserLoggedIn) {
    header(HEADER_LOCATION . ROUTE_LOGIN);
    exit;
}

if ($isAccountPage && $_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!SiteAuth::isLoggedIn()) {
        header(HEADER_LOCATION . ROUTE_LOGIN);
        exit;
    }
    $userId = (int)($_SESSION['site_user_id'] ?? 0);
    $db = DatabaseManager::getWriteConnection();
    $action = $_POST['action'] ?? '';
    if ($action === 'export_data') {
        $limitSeconds = 60;
        $stmt = $db->prepare("SELECT last_time FROM gdpr_export_rate_limits WHERE user_id = ?");
        $stmt->execute([$userId]);
        $last = (int)$stmt->fetchColumn();
        if ($last > 0 && (time() - $last) < $limitSeconds) {
            $_SESSION['account_settings_error'] = 'Please wait a minute before requesting another export.';
            header(HEADER_LOCATION . ACCOUNT_SETTINGS_ROUTE);
            exit;
        }

        $user = UserService::getById((int)$userId);
        $to = trim((string)($user['email'] ?? ''));
        if ($to === '' || !filter_var($to, FILTER_VALIDATE_EMAIL)) {
            $_SESSION['account_settings_error'] = 'Your account email is missing or invalid.';
            header(HEADER_LOCATION . ACCOUNT_SETTINGS_ROUTE);
            exit;
        }

        $stmt = $db->prepare("INSERT INTO gdpr_export_rate_limits (user_id, last_time) VALUES (?, ?) ON CONFLICT (user_id) DO UPDATE SET last_time = EXCLUDED.last_time");
        $stmt->execute([$userId, time()]);

        $attachment = null;
        $tmpFile = null;
        $export = null;
        $useMediaExport = function_exists('gdprCollectUserMediaFiles') && function_exists('gdprCreateUserExportArchive');

        if ($useMediaExport) {
            $mediaFiles = GdprService::collectUserMediaFiles($userId);
            $export = GdprService::exportUserData($userId, $mediaFiles);
            $archive = GdprService::createUserExportArchive($userId, $export, $mediaFiles);
            if ($archive['success']) {
                $attachment = [
                    'path' => $archive['path'],
                    'name' => $archive['name'],
                    'mime' => $archive['mime']
                ];
                $tmpFile = $archive['path'];
            }
        }

        if (!$attachment) {
            $export = $export ?? GdprService::exportUserData($userId);
            $payload = json_encode($export, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
            $tmpFile = tempnam(sys_get_temp_dir(), 'gdpr_export_');
            if ($tmpFile === false || $payload === false) {
                $_SESSION['account_settings_error'] = 'Unable to generate your data export.';
                header(HEADER_LOCATION . ACCOUNT_SETTINGS_ROUTE);
                exit;
            }
            file_put_contents($tmpFile, $payload);
            $attachment = [
                'path' => $tmpFile,
                'name' => 'user-data-export-' . $userId . '.json',
                'mime' => 'application/json'
            ];
        }

        $smtp = [
            'host' => SettingsService::get('smtp_host', ''),
            'port' => SettingsService::get('smtp_port', '587'),
            'user' => SettingsService::get('smtp_user', ''),
            'pass' => SettingsService::get('smtp_pass', ''),
            'secure' => SettingsService::get('smtp_secure', 'tls')
        ];

        $subject = 'Your data export request';
        $body = "Hello,\n\nAttached is the data export for your account.";
        if (($attachment['mime'] ?? '') === 'application/zip') {
            $body .= " If available, images (profile photo and support chat images) are included in the ZIP.";
        }
        $body .= "\nIf you did not request this, please contact support.\n\nThank you.";
        $sent = SmtpMailer::send($to, $subject, $body, '', '', $smtp, $attachment);
        if ($tmpFile) {
            @unlink($tmpFile);
        }

        if ($sent) {
            LogService::add('gdpr', 'User data export emailed', json_encode(['user_id' => $userId]));
            $_SESSION['account_settings_message'] = 'We sent your data export to your account email.';
        } else {
            LogService::add('error', 'User data export email failed', json_encode(['user_id' => $userId]));
            $_SESSION['account_settings_error'] = 'Unable to send the export email. Please try again later.';
        }

        header(HEADER_LOCATION . ACCOUNT_SETTINGS_ROUTE);
        exit;
    }
    if ($action === 'delete_account') {
        $confirm = trim($_POST['delete_confirm'] ?? '');
        if ($confirm !== 'DELETE') {
            $_SESSION['account_settings_error'] = 'Type DELETE to confirm account deletion.';
            header(HEADER_LOCATION . ACCOUNT_SETTINGS_ROUTE);
            exit;
        }
        if (GdprService::deleteUserData($userId)) {
            LogService::add('gdpr', 'User account deleted', json_encode(['user_id' => $userId]));
            SiteAuth::logout();
            header(HEADER_LOCATION . '/?account_deleted=1');
            exit;
        }
        $_SESSION['account_settings_error'] = 'Unable to delete account.';
        header(HEADER_LOCATION . ACCOUNT_SETTINGS_ROUTE);
        exit;
    }
    if ($action === 'mark_all_read') {
        $stmt = $db->prepare("UPDATE user_messages SET is_read = 1 WHERE user_id = ?");
        $stmt->execute([$userId]);
        $_SESSION['account_flash_message'] = 'All messages marked as read.';
        LogService::add('info', 'Account messages marked as read', json_encode(['user_id' => $userId]));
    }
    if ($action === 'delete_message') {
        $messageId = (int)($_POST['message_id'] ?? 0);
        if ($messageId > 0) {
            LogService::add('info', 'Account delete message requested', json_encode(['user_id' => $userId, 'message_id' => $messageId]));
            $stmt = $db->prepare("DELETE FROM user_messages WHERE id = ? AND user_id = ?");
            $stmt->execute([$messageId, $userId]);
            LogService::add('info', 'Account message deleted', json_encode(['user_id' => $userId, 'message_id' => $messageId, 'rows' => $stmt->rowCount()]));
            $logPattern = '%"message_id":' . $messageId . '%';
            $logStmt = $db->prepare("DELETE FROM logs WHERE message IN ('Account delete message requested', 'Account message deleted') AND details LIKE ?");
            $logStmt->execute([$logPattern]);
            $_SESSION['account_flash_message'] = 'Message deleted.';
        }
    }
    if ($action === 'reply_message') {
        $replyToId = (int)($_POST['reply_to_id'] ?? 0);
        $body = trim($_POST['reply_body'] ?? '');
        if ($replyToId <= 0 || $body === '') {
            $_SESSION['account_settings_error'] = 'Reply message is required.';
            header(HEADER_LOCATION . ACCOUNT_MESSAGES_ROUTE);
            exit;
        }

        $stmt = $db->prepare("SELECT id, subject, sender_type, message_type FROM user_messages WHERE id = ? AND user_id = ? LIMIT 1");
        $stmt->execute([$replyToId, $userId]);
        $original = $stmt->fetch();
        if (!$original) {
            $_SESSION['account_settings_error'] = 'Message not found.';
            header(HEADER_LOCATION . ACCOUNT_MESSAGES_ROUTE);
            exit;
        }
        if (($original['sender_type'] ?? 'admin') !== 'admin' || ($original['message_type'] ?? 'notice') !== 'ticket') {
            $_SESSION['account_settings_error'] = 'Replies are only available for ticket messages.';
            header(HEADER_LOCATION . ACCOUNT_MESSAGES_ROUTE);
            exit;
        }

        $subject = trim((string)($original['subject'] ?? ''));
        if ($subject === '') {
            $subject = 'Ticket';
        }

        $user = UserService::getById((int)$userId);
        $senderLabel = trim((string)($user['email'] ?? 'User'));
        $stmt = $db->prepare("INSERT INTO user_messages (user_id, subject, body, sender_type, sender_label, reply_to_id, message_type, is_read, created_at) VALUES (?, ?, ?, 'user', ?, ?, 'ticket', 1, ?)");
        $stmt->execute([$userId, $subject, $body, $senderLabel, $replyToId, time()]);
        LogService::add('info', 'User replied to ticket (account)', json_encode(['user_id' => $userId, 'reply_to' => $replyToId]));
        $_SESSION['account_settings_message'] = 'Reply sent successfully.';
        header(HEADER_LOCATION . ACCOUNT_MESSAGES_ROUTE . '&ticket=1');
        exit;
    }
    if ($action === 'request_refund') {
        verifyCSRF();
        $orderItemId = (int)($_POST['order_item_id'] ?? 0);
        if ($orderItemId <= 0) {
            $_SESSION['account_flash_error'] = 'Invalid item.';
            header(HEADER_LOCATION . ACCOUNT_PURCHASES_ROUTE);
            exit;
        }

        // Verify the order item belongs to the user
        $stmt = $db->prepare(
            "SELECT oi.id, oi.is_digital, oi.order_id, o.user_id, o.created_at
             FROM order_items oi
             JOIN orders o ON o.id = oi.order_id
             WHERE oi.id = ? AND o.user_id = ?
             LIMIT 1"
        );
        $stmt->execute([$orderItemId, $userId]);
        $orderItem = $stmt->fetch();
        if (!$orderItem) {
            $_SESSION['account_flash_error'] = 'This item is not eligible for a refund request.';
            header(HEADER_LOCATION . ACCOUNT_PURCHASES_ROUTE);
            exit;
        }

        // Determine refund window based on product type
        $isDigital = !empty($orderItem['is_digital']);
        $windowDays = (int)SettingsService::get(
            $isDigital ? 'refund_window_digital_days' : 'refund_window_physical_days',
            '30'
        );

        // If window is 0, refund requests are disabled for this product type
        if ($windowDays <= 0) {
            $_SESSION['account_flash_error'] = 'Refund requests are not available for this product type.';
            header(HEADER_LOCATION . ACCOUNT_PURCHASES_ROUTE);
            exit;
        }

        // Check if refund window has expired
        $orderCreatedAt = strtotime($orderItem['created_at']);
        $windowEnd = $orderCreatedAt + ($windowDays * 86400);
        if (time() > $windowEnd) {
            $_SESSION['account_flash_error'] = 'The refund request window has expired for this item.';
            header(HEADER_LOCATION . ACCOUNT_PURCHASES_ROUTE);
            exit;
        }

        // For digital items, verify the item was never downloaded
        if ($isDigital) {
            $stmt = $db->prepare(
                "SELECT download_count FROM digital_downloads WHERE order_item_id = ? LIMIT 1"
            );
            $stmt->execute([$orderItemId]);
            $dlRow = $stmt->fetch();
            if ($dlRow && (int)$dlRow['download_count'] > 0) {
                $_SESSION['account_flash_error'] = 'This item has already been downloaded and is not eligible for a refund.';
                header(HEADER_LOCATION . ACCOUNT_PURCHASES_ROUTE);
                exit;
            }
        }

        // Check for existing refund request
        $stmt = $db->prepare("SELECT id, status FROM refund_requests WHERE order_item_id = ? LIMIT 1");
        $stmt->execute([$orderItemId]);
        $existingRefund = $stmt->fetch();
        if ($existingRefund) {
            $_SESSION['account_flash_error'] = 'A refund request already exists for this item.';
            header(HEADER_LOCATION . ACCOUNT_PURCHASES_ROUTE);
            exit;
        }

        // Create refund request
        $now = DbHelper::nowString();
        $stmt = $db->prepare(
            "INSERT INTO refund_requests (order_id, order_item_id, user_id, status, admin_reason, created_at, updated_at)
             VALUES (?, ?, ?, 'pending', '', ?, ?)"
        );
        $stmt->execute([(int)$orderItem['order_id'], $orderItemId, $userId, $now, $now]);
        $refundLabel = $isDigital ? 'Refund' : 'Return & refund';
        LogService::add('refund', 'User requested ' . ($isDigital ? 'refund' : 'return & refund'), json_encode([
            'user_id' => $userId,
            'order_item_id' => $orderItemId,
            'order_id' => (int)$orderItem['order_id'],
            'is_digital' => $isDigital,
        ]));
        $_SESSION['account_flash_message'] = $refundLabel . ' request submitted. We will review it shortly.';
        header(HEADER_LOCATION . ACCOUNT_PURCHASES_ROUTE);
        exit;
    }
    if ($action === 'cancel_refund') {
        verifyCSRF();
        $orderItemId = (int)($_POST['order_item_id'] ?? 0);
        if ($orderItemId <= 0) {
            $_SESSION['account_flash_error'] = 'Invalid item.';
            header(HEADER_LOCATION . ACCOUNT_PURCHASES_ROUTE);
            exit;
        }

        // Only allow cancelling own pending refund requests
        $stmt = $db->prepare(
            "SELECT rr.id FROM refund_requests rr
             JOIN orders o ON o.id = rr.order_id
             WHERE rr.order_item_id = ? AND rr.user_id = ? AND rr.status = 'pending'
             LIMIT 1"
        );
        $stmt->execute([$orderItemId, $userId]);
        $pendingRefund = $stmt->fetch();
        if (!$pendingRefund) {
            $_SESSION['account_flash_error'] = 'No pending refund request found for this item.';
            header(HEADER_LOCATION . ACCOUNT_PURCHASES_ROUTE);
            exit;
        }

        $stmt = $db->prepare("DELETE FROM refund_requests WHERE id = ?");
        $stmt->execute([(int)$pendingRefund['id']]);
        LogService::add('refund', 'User cancelled refund request', json_encode([
            'user_id' => $userId,
            'order_item_id' => $orderItemId,
            'refund_id' => (int)$pendingRefund['id'],
        ]));
        $_SESSION['account_flash_message'] = 'Refund request cancelled.';
        header(HEADER_LOCATION . ACCOUNT_PURCHASES_ROUTE);
        exit;
    }
    if ($action === 'update_account') {
        $displayName = trim($_POST['display_name'] ?? '');
        $nickname = trim($_POST['nickname'] ?? '');
        $address = trim($_POST['address'] ?? '');
        $postalCode = trim($_POST['postal_code'] ?? '');
        $country = trim($_POST['country'] ?? '');
        $allowedCountries = CountryData::getOptions();
        if ($country !== '' && !in_array($country, $allowedCountries, true)) {
            $country = '';
        }

        $description = mb_substr(trim($_POST['description'] ?? ''), 0, 500);
        $website = mb_substr(trim($_POST['website'] ?? ''), 0, 200);
        $company = mb_substr(trim($_POST['company'] ?? ''), 0, 120);
        $phone = mb_substr(trim($_POST['phone'] ?? ''), 0, 40);
        $status = trim($_POST['status'] ?? 'online');
        $allowedStatuses = ['online', 'offline', 'dnd'];
        if (!in_array($status, $allowedStatuses, true)) {
            $status = 'online';
        }

        if ($website !== '' && !filter_var($website, FILTER_VALIDATE_URL)) {
            $normalizedWebsite = 'https://' . ltrim($website, '/');
            if (filter_var($normalizedWebsite, FILTER_VALIDATE_URL)) {
                $website = $normalizedWebsite;
            } else {
                $_SESSION['account_settings_error'] = 'Please enter a valid website URL.';
            }
        }

        $gender = trim($_POST['gender'] ?? '');
        $genderOptions = ['male', 'female', 'non-binary', 'custom', 'prefer-not-to-say'];
        if (!in_array($gender, $genderOptions, true)) {
            $gender = '';
        }
        $genderCustom = '';
        if ($gender === 'custom') {
            $genderCustom = mb_substr(trim($_POST['gender_custom'] ?? ''), 0, 120);
            if ($genderCustom === '') {
                $_SESSION['account_settings_error'] = 'Please enter a custom gender.';
            }
        }

        $ageValue = null;
        $ageRaw = trim($_POST['age'] ?? '');
        if ($ageRaw !== '') {
            if (!ctype_digit($ageRaw)) {
                $_SESSION['account_settings_error'] = 'Age must be a number.';
            } else {
                $ageValue = (int)$ageRaw;
                if ($ageValue < 1 || $ageValue > 120) {
                    $_SESSION['account_settings_error'] = 'Age must be between 1 and 120.';
                }
            }
        }

        $photoPath = null;

        $stmt = $db->prepare("SELECT nickname FROM site_users WHERE id = ?");
        $stmt->execute([$userId]);
        $currentNickname = (string)$stmt->fetchColumn();

        if ($nickname === '') {
            if ($currentNickname === '') {
                $nickname = UserService::generateUniqueNickname($db);
            } else {
                $nickname = $currentNickname;
            }
        } else {
            $stmt = $db->prepare("SELECT id FROM site_users WHERE nickname = ? AND id != ? LIMIT 1");
            $stmt->execute([$nickname, $userId]);
            if ($stmt->fetch()) {
                $_SESSION['account_settings_error'] = 'This username already exists or has been taken.';
            }
        }
        if (isset($_FILES['profile_photo']) && $_FILES['profile_photo']['error'] !== UPLOAD_ERR_NO_FILE) {
            if ($_FILES['profile_photo']['error'] === UPLOAD_ERR_OK) {
                $file = $_FILES['profile_photo'];
                $maxSize = 5 * 1024 * 1024;
                if ($file['size'] > $maxSize) {
                    $_SESSION['account_settings_error'] = 'Profile photo is too large (max 5MB).';
                } else {
                    $allowedMap = UPLOAD_IMAGE_MIME_MAP;
                    $validation = UploadService::validateFile($file, $allowedMap, true);
                    if (!$validation['success']) {
                        $_SESSION['account_settings_error'] = $validation['error'] ?: 'Invalid image file.';
                    } else {
                        $uploadDir = __DIR__ . DATA_PROFILE_PHOTOS_SEGMENT;
                        if (!is_dir($uploadDir)) {
                            @mkdir($uploadDir, 0755, true);
                            file_put_contents($uploadDir . '/.htaccess', "Options -Indexes\n<FilesMatch \"\\.php$\">\n    Deny from all\n</FilesMatch>\nAddType application/octet-stream .php .php5 .phtml");
                        }
                        $fileName = 'profile_' . time() . '_' . bin2hex(random_bytes(6)) . '.' . ($validation['ext'] ?? 'jpg');
                        $dest = $uploadDir . $fileName;
                        if (move_uploaded_file($file['tmp_name'], $dest)) {
                            $photoPath = PROFILE_PHOTO_PATH_SEGMENT . $fileName;
                            $stmt = $db->prepare("SELECT profile_photo FROM site_users WHERE id = ?");
                            $stmt->execute([$userId]);
                            $currentPhoto = $stmt->fetchColumn();
                            if ($currentPhoto && strpos($currentPhoto, PROFILE_PHOTO_PATH_SEGMENT) === 0) {
                                $oldPath = __DIR__ . DATA_PROFILE_PHOTOS_SEGMENT . basename($currentPhoto);
                                if (is_file($oldPath)) {
                                    @unlink($oldPath);
                                }
                            }
                        } else {
                            $_SESSION['account_settings_error'] = 'Failed to upload profile photo.';
                        }
                    }
                }
            } else {
                $_SESSION['account_settings_error'] = 'Profile photo upload failed.';
            }
        }

        if (empty($_SESSION['account_settings_error'])) {
            // Handle is_searchable toggle (checkbox: if not present = invisible)
            $isSearchable = isset($_POST['is_searchable']) ? 1 : 0;

            // Check if is_searchable column exists
            $hasSearchableColumn = false;
            try {
                $checkStmt = $db->query("SELECT column_name FROM information_schema.columns WHERE table_name = 'site_users' AND column_name = 'is_searchable'");
                $hasSearchableColumn = $checkStmt->fetch() !== false;
            } catch (PDOException $e) {
                // Column check failed, assume it doesn't exist
            }

            if ($photoPath !== null) {
                if ($hasSearchableColumn) {
                    $stmt = $db->prepare("UPDATE site_users SET display_name = ?, nickname = ?, description = ?, website = ?, company = ?, phone = ?, gender = ?, gender_custom = ?, age = ?, address = ?, country = ?, postal_code = ?, profile_photo = ?, is_searchable = ?, status = ?, last_activity = ? WHERE id = ?");
                    $stmt->execute([$displayName, $nickname, $description, $website, $company, $phone, $gender, $genderCustom, $ageValue, $address, $country, $postalCode, $photoPath, $isSearchable, $status, time(), $userId]);
                } else {
                    $stmt = $db->prepare("UPDATE site_users SET display_name = ?, nickname = ?, description = ?, website = ?, company = ?, phone = ?, gender = ?, gender_custom = ?, age = ?, address = ?, country = ?, postal_code = ?, profile_photo = ?, status = ?, last_activity = ? WHERE id = ?");
                    $stmt->execute([$displayName, $nickname, $description, $website, $company, $phone, $gender, $genderCustom, $ageValue, $address, $country, $postalCode, $photoPath, $status, time(), $userId]);
                }
            } else {
                if ($hasSearchableColumn) {
                    $stmt = $db->prepare("UPDATE site_users SET display_name = ?, nickname = ?, description = ?, website = ?, company = ?, phone = ?, gender = ?, gender_custom = ?, age = ?, address = ?, country = ?, postal_code = ?, is_searchable = ?, status = ?, last_activity = ? WHERE id = ?");
                    $stmt->execute([$displayName, $nickname, $description, $website, $company, $phone, $gender, $genderCustom, $ageValue, $address, $country, $postalCode, $isSearchable, $status, time(), $userId]);
                } else {
                    $stmt = $db->prepare("UPDATE site_users SET display_name = ?, nickname = ?, description = ?, website = ?, company = ?, phone = ?, gender = ?, gender_custom = ?, age = ?, address = ?, country = ?, postal_code = ?, status = ?, last_activity = ? WHERE id = ?");
                    $stmt->execute([$displayName, $nickname, $description, $website, $company, $phone, $gender, $genderCustom, $ageValue, $address, $country, $postalCode, $status, time(), $userId]);
                }
            }

            // Also save privacy settings if included in form
            $privacyInput = $_POST['privacy'] ?? [];
            if (is_array($privacyInput) && !empty($privacyInput)) {
                $validVisibilities = ['everyone', 'friends', 'hidden'];
                foreach ($privacyInput as $field => $visibility) {
                    $field = preg_replace('/[^a-z_]/', '', $field);
                    if (in_array($visibility, $validVisibilities, true)) {
                        PrivacyService::setUserSetting((int)$userId, $field, $visibility);
                    }
                }
            }

            $_SESSION['account_settings_message'] = 'Account settings updated.';
        }

        header(HEADER_LOCATION . ACCOUNT_SETTINGS_ROUTE);
        exit;
    }

    if ($action === 'save_privacy_settings') {
        $privacyInput = $_POST['privacy'] ?? [];
        if (is_array($privacyInput)) {
            $validVisibilities = ['everyone', 'friends', 'hidden'];
            foreach ($privacyInput as $field => $visibility) {
                $field = preg_replace('/[^a-z_]/', '', $field);
                if (in_array($visibility, $validVisibilities, true)) {
                    PrivacyService::setUserSetting((int)$userId, $field, $visibility);
                }
            }
            $_SESSION['privacy_settings_message'] = 'Privacy settings saved.';
        }
        header(HEADER_LOCATION . ACCOUNT_SETTINGS_ROUTE);
        exit;
    }

    // If it's a friend action, skip this handler and let the friend handler below process it
    if (isset($_POST['friend_action'])) {
        // Don't exit - let the friend action handler below take care of it
    } else {
        header(HEADER_LOCATION . ACCOUNT_MESSAGES_ROUTE);
        exit;
    }
}

// Handle friends page POST actions
if (($isFriendsPage || $isProfilePage || $isAccountPage) && $_SERVER['REQUEST_METHOD'] === 'POST' && $isUserLoggedIn) {
    $userId = (int)$_SESSION['site_user_id'];
    $friendAction = $_POST['friend_action'] ?? '';
    $targetUserId = (int)($_POST['target_user_id'] ?? 0);
    $requestId = (int)($_POST['request_id'] ?? 0);
    $returnUrl = $_POST['return_url'] ?? '/friends';

    if ($friendAction === 'send_request' && $targetUserId > 0) {
        $message = trim($_POST['request_message'] ?? '');
        $result = FriendService::sendRequest((int)$userId, (int)$targetUserId, $message ?: null);
        if ($result['success']) {
            $_SESSION['friend_flash_message'] = 'Friend request sent!';
        } else {
            $_SESSION['friend_flash_error'] = $result['error'] ?? 'Could not send request.';
        }
        header(HEADER_LOCATION . $returnUrl);
        exit;
    }

    if ($friendAction === 'accept_request' && $requestId > 0) {
        $result = FriendService::respondToRequest((int)$requestId, (int)$userId, true);
        if ($result['success']) {
            $_SESSION['friend_flash_message'] = 'Friend request accepted!';
        } else {
            $_SESSION['friend_flash_error'] = $result['error'] ?? 'Could not accept request.';
        }
        header(HEADER_LOCATION . $returnUrl);
        exit;
    }

    if ($friendAction === 'decline_request' && $requestId > 0) {
        $result = FriendService::respondToRequest((int)$requestId, (int)$userId, false);
        if ($result['success']) {
            $_SESSION['friend_flash_message'] = 'Friend request declined.';
        } else {
            $_SESSION['friend_flash_error'] = $result['error'] ?? 'Could not decline request.';
        }
        header(HEADER_LOCATION . $returnUrl);
        exit;
    }

    if ($friendAction === 'cancel_request' && $requestId > 0) {
        if (FriendService::cancelRequest((int)$requestId, (int)$userId)) {
            $_SESSION['friend_flash_message'] = 'Friend request cancelled.';
        } else {
            $_SESSION['friend_flash_error'] = 'Could not cancel request.';
        }
        header(HEADER_LOCATION . $returnUrl);
        exit;
    }

    if ($friendAction === 'remove_friend' && $targetUserId > 0) {
        if (FriendService::removeFriend((int)$userId, (int)$targetUserId)) {
            $_SESSION['friend_flash_message'] = 'Friend removed.';
        } else {
            $_SESSION['friend_flash_error'] = 'Could not remove friend.';
        }
        header(HEADER_LOCATION . $returnUrl);
        exit;
    }

    if ($friendAction === 'block_user' && $targetUserId > 0) {
        if (FriendService::blockUser((int)$userId, (int)$targetUserId)) {
            $_SESSION['friend_flash_message'] = 'User blocked.';
        } else {
            $_SESSION['friend_flash_error'] = 'Could not block user.';
        }
        header(HEADER_LOCATION . $returnUrl);
        exit;
    }

    if ($friendAction === 'unblock_user' && $targetUserId > 0) {
        if (FriendService::unblockUser((int)$userId, (int)$targetUserId)) {
            $_SESSION['friend_flash_message'] = 'User unblocked.';
        } else {
            $_SESSION['friend_flash_error'] = 'Could not unblock user.';
        }
        header(HEADER_LOCATION . $returnUrl);
        exit;
    }

    // Save privacy settings
    if ($friendAction === 'save_privacy') {
        $privacyFields = ['display_name', 'email', 'phone', 'gender', 'age', 'address', 'country', 'postal_code', 'description', 'website', 'company'];
        foreach ($privacyFields as $field) {
            $visibility = $_POST['privacy_' . $field] ?? 'hidden';
            PrivacyService::setUserSetting((int)$userId, $field, $visibility);
        }
        $_SESSION['account_settings_message'] = 'Privacy settings updated.';
        header(HEADER_LOCATION . ACCOUNT_SETTINGS_ROUTE);
        exit;
    }
}

$unreadCount = $isUserLoggedIn ? UserMessageService::getUnreadCount((int)($_SESSION['site_user_id'] ?? 0)) : 0;
$menuMessages = $isUserLoggedIn ? UserMessageService::getMessages((int)($_SESSION['site_user_id'] ?? 0), 3) : [];

ob_start();
?>
<div class="notify-menu" data-notify-menu>
    <?php if ($isUserLoggedIn): ?>
        <div class="notify-section">
            <div class="notify-section-title">Messages</div>
            <div data-notify-list>
                <?php if (empty($menuMessages)): ?>
                    <div class="notify-empty" data-notify-empty>No messages yet.</div>
                <?php else: ?>
                    <?php foreach ($menuMessages as $message): ?>
                        <?php $bodyText = trim(strip_tags((string)($message['body'] ?? ''))); ?>
                        <?php $bodyPreview = substr($bodyText, 0, 100); ?>
                        <?php $bodyHasMore = strlen($bodyText) > 100; ?>
                        <a class="notify-item-link" href="/account?tab=messages&message=<?php echo (int)$message['id']; ?>#message-<?php echo (int)$message['id']; ?>">
                            <div class="notify-item <?php echo empty($message['is_read']) ? 'is-unread' : ''; ?>">
                                <div class="notify-item-title"><?php echo e($message['subject'] ?: 'Message'); ?></div>
                                <?php if ($bodyText !== ''): ?>
                                    <div class="notify-item-body"><?php echo e($bodyPreview); ?><?php echo $bodyHasMore ? '…' : ''; ?></div>
                                <?php endif; ?>
                            </div>
                        </a>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
            <a class="notify-link" href="/account?tab=messages">View all</a>
            <div class="notify-mark-form" data-notify-mark>
                <div class="notify-mark-all">
                    <button type="button" class="notify-link notify-mark-btn js-notify-mark-all">Mark all as read</button>
                </div>
            </div>
        </div>
    <?php else: ?>
        <div class="notify-section">
            <div class="notify-section-title">Quick Actions</div>
            <a class="notify-link" href="/login">Login</a>
        </div>
    <?php endif; ?>
</div>
<?php
$notificationMenu = ob_get_clean();

ob_start();
?>
<div class="account-menu" data-account-menu>
    <?php if ($isUserLoggedIn): ?>
        <a class="account-menu-link" href="/account">Go to Account</a>
        <a class="account-menu-link" href="/account?tab=purchases">Purchases</a>
        <a class="account-menu-link" href="/account?tab=friends&ftab=all" data-friends-menu-link>Friends</a>
        <a class="account-menu-link" href="/account?tab=settings">Account Settings</a>
        <a class="account-menu-link" href="/account?tab=messages" data-message-center-tab>View All Messages</a>
        <div class="account-menu-actions">
            <a class="btn btn-outline btn-sm" href="/logout">Log out</a>
        </div>
    <?php else: ?>
        <a class="account-menu-link" href="/login">Login</a>
    <?php endif; ?>
</div>
<?php
$accountMenu = ob_get_clean();

// ── Google OAuth callback ──────────────────────────────────────────────
$googleOAuthError = '';
if ($isGoogleCallback && GoogleOAuthService::isEnabled()) {
    $code  = trim($_GET['code'] ?? '');
    $state = trim($_GET['state'] ?? '');
    $oauthError = trim($_GET['error'] ?? '');

    if ($oauthError !== '') {
        // User denied consent or another Google-side error
        $googleOAuthError = 'Google login was cancelled or denied.';
    } elseif ($code !== '') {
        $googleUser = GoogleOAuthService::handleCallback($code, $state);
        if ($googleUser !== null) {
            $ip = IpService::getClientIP();
            $allowReg = SettingsService::get('google_oauth_allow_registration', '1') === '1';

            // Try to find by google_id first, then by email
            $user = UserService::getByGoogleId($googleUser['google_id']);
            if (!$user) {
                $user = UserService::getByEmail($googleUser['email']);
            }

            if ($user) {
                // Link google_id if not yet linked
                if (empty($user['google_id'])) {
                    UserService::linkGoogleId((int)$user['id'], $googleUser['google_id']);
                }
                if (UserService::needsLegalAcceptance($user)) {
                    UserService::recordLegalAcceptance((int)$user['id'], $ip);
                }
                SiteAuth::login((int)$user['id'], $ip);
                header(HEADER_LOCATION . ACCOUNT_ROUTE);
                exit;
            }

            if ($allowReg) {
                // Create new user with Google-verified email
                $userId = UserService::create($googleUser['email'], $ip, true);
                UserService::linkGoogleId($userId, $googleUser['google_id']);
                SiteAuth::login($userId, $ip);
                header(HEADER_LOCATION . ACCOUNT_ROUTE);
                exit;
            }

            $googleOAuthError = 'No account found for this email. Registration via Google is currently disabled.';
        } else {
            $googleOAuthError = 'Google login failed. Please try again.';
        }
    }

    // If we reach here, redirect to login with error
    if ($googleOAuthError !== '') {
        $_SESSION['google_oauth_error'] = $googleOAuthError;
    }
    header(HEADER_LOCATION . ROUTE_LOGIN);
    exit;
}

// Pass error from session (set by callback redirect)
$googleOAuthError = '';
if (isset($_SESSION['google_oauth_error'])) {
    $googleOAuthError = $_SESSION['google_oauth_error'];
    unset($_SESSION['google_oauth_error']);
}

$loginTokenError = '';
if ($isLoginPage) {
    if ($isUserLoggedIn) {
        header(HEADER_LOCATION . ACCOUNT_ROUTE);
        exit;
    }

    $token = trim($_GET['token'] ?? '');
    if ($token !== '') {
        $userId = UserService::verifyLoginToken($token);
        if ($userId) {
            $user = UserService::getById((int)$userId);
            if ($user && !UserService::needsLegalAcceptance($user)) {
                SiteAuth::login((int)$userId, IpService::getClientIP());
                header(HEADER_LOCATION . ACCOUNT_ROUTE);
                exit;
            }
            $loginTokenError = $user
                ? 'Please accept our Terms of Service and Privacy Policy on the login page to continue.'
                : 'This login link is invalid or expired.';
        } else {
            $loginTokenError = 'This login link is invalid or expired.';
        }
    }
}

$apiRoutes = ['cart-api', 'search-api', 'set-currency', 'set-language', 'notifications-api', 'messages-api', 'messenger-api', 'checkout-api', 'stripe-webhook', 'cookie-consent', 'friends-search', 'friends-action', 'friends-list', 'product-likes-api', 'forum-posts-api', 'forum-comments-api', 'game-leaderboard-api', 'invoice', 'log-error'];
if (in_array($pageSlug, $apiRoutes, true)) {
    TemplateRenderer::renderOnce(__DIR__ . '/templates/api/' . $pageSlug . '.php');
    exit;
}

// Serve chat images securely
if ($pageSlug === 'chat-image' && isset($segments[1])) {
    $imageName = basename($segments[1]); // Sanitize to prevent directory traversal
    $viewerId = $isUserLoggedIn ? (int)($_SESSION['site_user_id'] ?? 0) : 0;
    $isAdmin = AdminAuth::isLoggedIn();
    if (!$isAdmin && $viewerId <= 0) {
        http_response_code(404);
        exit;
    }

    if (!$isAdmin) {
        $db = DatabaseManager::getReadConnection();
        $imagePathKey = 'chat_images/' . $imageName;
        $stmt = $db->prepare("SELECT 1 FROM user_messages WHERE image_path = ? AND (user_id = ? OR sender_id = ?) LIMIT 1");
        $stmt->execute([$imagePathKey, $viewerId, $viewerId]);
        if (!$stmt->fetchColumn()) {
            http_response_code(404);
            exit;
        }
    }
    $imagePath = __DIR__ . '/../data/chat_images/' . $imageName;

    // Verify the file exists and is an image
    if (file_exists($imagePath) && is_file($imagePath)) {
        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->file($imagePath);
        $allowedMimes = ALLOWED_IMAGE_MIMES_WITH_GIF;

        if (in_array($mimeType, $allowedMimes, true)) {
            // Get file modification time and size for caching
            $lastModified = filemtime($imagePath);
            $fileSize = filesize($imagePath);
            $etag = '"' . md5($imageName . $lastModified . $fileSize) . '"';

            // Check if browser has valid cached version
            $ifNoneMatch = $_SERVER['HTTP_IF_NONE_MATCH'] ?? '';
            $ifModifiedSince = $_SERVER['HTTP_IF_MODIFIED_SINCE'] ?? '';

            if (($ifNoneMatch && $ifNoneMatch === $etag) || ($ifModifiedSince && strtotime($ifModifiedSince) >= $lastModified)) {
                http_response_code(304);
                header(HEADER_ETAG . $etag);
                header(CACHE_CONTROL_PUBLIC_IMMUTABLE);
                exit;
            }

            header(HEADER_CONTENT_TYPE . $mimeType);
            header(HEADER_CONTENT_LENGTH . $fileSize);
            header(HEADER_LAST_MODIFIED . gmdate(HTTP_GMT_DATE_FORMAT, $lastModified) . ' GMT');
            header(HEADER_ETAG . $etag);
            header(CACHE_CONTROL_PUBLIC_IMMUTABLE);
            readfile($imagePath);
            exit;
        }
    }

    http_response_code(404);
    echo IMAGE_NOT_FOUND_MESSAGE;
    exit;
}

// Serve forum post images (public for approved posts)
if ($pageSlug === 'forum-image' && isset($segments[1])) {
    $imageName = basename($segments[1]);
    $imagePath = __DIR__ . '/../data/forum_images/' . $imageName;
    $isAdmin = AdminAuth::isLoggedIn();

    if (!$isAdmin) {
        $db = DatabaseManager::getReadConnection();
        $imagePathKey = 'forum_images/' . $imageName;
        $stmt = $db->prepare("SELECT 1 FROM forum_post_images i JOIN forum_posts p ON p.id = i.post_id WHERE i.image_path = ? AND p.status = 'approved' UNION SELECT 1 FROM forum_comment_images ci JOIN forum_comments c ON c.id = ci.comment_id JOIN forum_posts p2 ON p2.id = c.post_id WHERE ci.image_path = ? AND c.status = 'approved' AND p2.status = 'approved' LIMIT 1");
        $stmt->execute([$imagePathKey, $imagePathKey]);
        if (!$stmt->fetchColumn()) {
            http_response_code(404);
            exit;
        }
    }

    if (file_exists($imagePath) && is_file($imagePath)) {
        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->file($imagePath);
        $allowedMimes = ALLOWED_IMAGE_MIMES_WITH_GIF;

        if (in_array($mimeType, $allowedMimes, true)) {
            $lastModified = filemtime($imagePath);
            $fileSize = filesize($imagePath);
            $etag = '"' . md5($imageName . $lastModified . $fileSize) . '"';

            $ifNoneMatch = $_SERVER['HTTP_IF_NONE_MATCH'] ?? '';
            $ifModifiedSince = $_SERVER['HTTP_IF_MODIFIED_SINCE'] ?? '';

            if (($ifNoneMatch && $ifNoneMatch === $etag) || ($ifModifiedSince && strtotime($ifModifiedSince) >= $lastModified)) {
                http_response_code(304);
                header(HEADER_ETAG . $etag);
                header(CACHE_CONTROL_PUBLIC_IMMUTABLE);
                exit;
            }

            header(HEADER_CONTENT_TYPE . $mimeType);
            header(HEADER_CONTENT_LENGTH . $fileSize);
            header(HEADER_LAST_MODIFIED . gmdate(HTTP_GMT_DATE_FORMAT, $lastModified) . ' GMT');
            header(HEADER_ETAG . $etag);
            header(CACHE_CONTROL_PUBLIC_IMMUTABLE);
            readfile($imagePath);
            exit;
        }
    }

    http_response_code(404);
    echo IMAGE_NOT_FOUND_MESSAGE;
    exit;
}

// Serve profile photos securely
if ($pageSlug === 'profile-photo' && isset($segments[1])) {
    $imageName = basename($segments[1]);
    $viewerId = $isUserLoggedIn ? (int)($_SESSION['site_user_id'] ?? 0) : 0;
    $isAdmin = AdminAuth::isLoggedIn();
    if (!$isAdmin) {
        $db = DatabaseManager::getReadConnection();
        $profilePathKey = PROFILE_PHOTO_PATH_SEGMENT . $imageName;
        $stmt = $db->prepare("SELECT id FROM site_users WHERE profile_photo = ? LIMIT 1");
        $stmt->execute([$profilePathKey]);
        $profileOwnerId = (int)$stmt->fetchColumn();
        if ($profileOwnerId <= 0 || !PrivacyService::canViewField((int)($viewerId ?: 0), (int)$profileOwnerId, 'profile_photo')) {
            http_response_code(404);
            exit;
        }
    }
    $imagePath = __DIR__ . DATA_PROFILE_PHOTOS_SEGMENT . $imageName;

    if (file_exists($imagePath) && is_file($imagePath)) {
        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->file($imagePath);
        $allowedMimes = ALLOWED_IMAGE_MIMES;

        if (in_array($mimeType, $allowedMimes, true)) {
            // Get file modification time and size for caching
            $lastModified = filemtime($imagePath);
            $fileSize = filesize($imagePath);
            $etag = '"' . md5($imageName . $lastModified . $fileSize) . '"';

            // Check if browser has valid cached version
            $ifNoneMatch = $_SERVER['HTTP_IF_NONE_MATCH'] ?? '';
            $ifModifiedSince = $_SERVER['HTTP_IF_MODIFIED_SINCE'] ?? '';

            if (($ifNoneMatch && $ifNoneMatch === $etag) || ($ifModifiedSince && strtotime($ifModifiedSince) >= $lastModified)) {
                http_response_code(304);
                header(HEADER_ETAG . $etag);
                header(CACHE_CONTROL_PUBLIC_IMMUTABLE);
                exit;
            }

            header(HEADER_CONTENT_TYPE . $mimeType);
            header(HEADER_CONTENT_LENGTH . $fileSize);
            header(HEADER_LAST_MODIFIED . gmdate(HTTP_GMT_DATE_FORMAT, $lastModified) . ' GMT');
            header(HEADER_ETAG . $etag);
            header(CACHE_CONTROL_PUBLIC_IMMUTABLE);
            readfile($imagePath);
            exit;
        }
    }

    http_response_code(404);
    echo IMAGE_NOT_FOUND_MESSAGE;
    exit;
}

// Serve contact uploads securely (admin only)
if ($pageSlug === 'contact-upload' && isset($segments[1])) {
    AdminAuth::requireLogin();
    $imageName = basename($segments[1]);
    $imagePath = __DIR__ . '/../data/contact_uploads/' . $imageName;

    if (file_exists($imagePath) && is_file($imagePath)) {
        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->file($imagePath);
        $allowedMimes = ALLOWED_IMAGE_MIMES;

        if (in_array($mimeType, $allowedMimes, true)) {
            // Get file modification time and size for caching
            $lastModified = filemtime($imagePath);
            $fileSize = filesize($imagePath);
            $etag = '"' . md5($imageName . $lastModified . $fileSize) . '"';

            // Check if browser has valid cached version
            $ifNoneMatch = $_SERVER['HTTP_IF_NONE_MATCH'] ?? '';
            $ifModifiedSince = $_SERVER['HTTP_IF_MODIFIED_SINCE'] ?? '';

            if (($ifNoneMatch && $ifNoneMatch === $etag) || ($ifModifiedSince && strtotime($ifModifiedSince) >= $lastModified)) {
                http_response_code(304);
                header(HEADER_ETAG . $etag);
                header(CACHE_CONTROL_PRIVATE_IMMUTABLE);
                exit;
            }

            header(HEADER_CONTENT_TYPE . $mimeType);
            header(HEADER_CONTENT_LENGTH . $fileSize);
            header(HEADER_LAST_MODIFIED . gmdate(HTTP_GMT_DATE_FORMAT, $lastModified) . ' GMT');
            header(HEADER_ETAG . $etag);
            header(CACHE_CONTROL_PRIVATE_IMMUTABLE);
            readfile($imagePath);
            exit;
        }
    }

    http_response_code(404);
    echo FILE_NOT_FOUND_MESSAGE;
    exit;
}

// Serve refund return-instruction images (owning user or admin)
if ($pageSlug === 'refund-file' && isset($segments[1])) {
    $imageName = basename($segments[1]);
    $isAdmin = AdminAuth::isLoggedIn();
    $viewerId = $isUserLoggedIn ? (int)($_SESSION['site_user_id'] ?? 0) : 0;

    if (!$isAdmin && $viewerId <= 0) {
        http_response_code(404);
        exit;
    }

    // Verify the requesting user owns a refund with this file, or is admin
    if (!$isAdmin) {
        $db = DatabaseManager::getReadConnection();
        $stmt = $db->prepare(
            "SELECT 1 FROM refund_requests WHERE user_id = ? AND return_instructions LIKE ? LIMIT 1"
        );
        $stmt->execute([$viewerId, '%' . $imageName . '%']);
        if (!$stmt->fetchColumn()) {
            http_response_code(404);
            exit;
        }
    }

    $imagePath = __DIR__ . '/../data/refund_uploads/' . $imageName;

    if (file_exists($imagePath) && is_file($imagePath)) {
        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->file($imagePath);
        $allowedMimes = ALLOWED_IMAGE_MIMES_WITH_GIF;

        if (in_array($mimeType, $allowedMimes, true)) {
            $lastModified = filemtime($imagePath);
            $fileSize = filesize($imagePath);
            $etag = '"' . md5($imageName . $lastModified . $fileSize) . '"';

            $ifNoneMatch = $_SERVER['HTTP_IF_NONE_MATCH'] ?? '';
            $ifModifiedSince = $_SERVER['HTTP_IF_MODIFIED_SINCE'] ?? '';

            if (($ifNoneMatch && $ifNoneMatch === $etag) || ($ifModifiedSince && strtotime($ifModifiedSince) >= $lastModified)) {
                http_response_code(304);
                header(HEADER_ETAG . $etag);
                header(CACHE_CONTROL_PRIVATE_IMMUTABLE);
                exit;
            }

            header(HEADER_CONTENT_TYPE . $mimeType);
            header(HEADER_CONTENT_LENGTH . $fileSize);
            header(HEADER_LAST_MODIFIED . gmdate(HTTP_GMT_DATE_FORMAT, $lastModified) . ' GMT');
            header(HEADER_ETAG . $etag);
            header(CACHE_CONTROL_PRIVATE_IMMUTABLE);
            readfile($imagePath);
            exit;
        }
    }

    http_response_code(404);
    echo IMAGE_NOT_FOUND_MESSAGE;
    exit;
}

if (!function_exists('normalizeFreeDigitalDownloadPath')) {
    function normalizeFreeDigitalDownloadPath(string $downloadUrl): string
    {
        $normalizedPath = '';
        $downloadUrl = trim($downloadUrl);
        if ($downloadUrl !== '') {
            if (strpos($downloadUrl, ADMIN_FILE_PATH_SEGMENT) !== false) {
                $downloadUrl = str_replace(ADMIN_FILE_PATH_SEGMENT, SITE_FILE_PATH_SEGMENT, $downloadUrl);
            }

            $path = '';
            if (preg_match('~^https?://~i', $downloadUrl)) {
                $parts = parse_url($downloadUrl);
                if (is_array($parts)) {
                    $path = (string)($parts['path'] ?? '');
                }
            } else {
                $path = (string)(parse_url($downloadUrl, PHP_URL_PATH) ?? '');
            }

            if ($path !== '' && strpos($path, SITE_FILE_PATH_SEGMENT) === 0) {
                $normalizedPath = $path;
            }
        }

        return $normalizedPath;
    }
}

// Track and serve free digital product downloads (public, aggregate-only analytics).
if ($pageSlug === 'free-download') {
    $productId = (int)($_GET['product'] ?? 0);
    $variantId = (int)($_GET['variant'] ?? 0);

    if ($productId <= 0 || $variantId < 0) {
        http_response_code(404);
        echo FILE_NOT_FOUND_MESSAGE;
        exit;
    }

    $readDb = DatabaseManager::getReadConnection();
    $lookupVariantId = $variantId > 0 ? $variantId : -1;
    $stmt = $readDb->prepare("SELECT p.id,
                                     p.price AS product_price,
                                     p.action_type AS product_action,
                                     p.download_url AS product_download_url,
                                     pv.id AS variant_id,
                                     pv.price AS variant_price,
                                     pv.action_type AS variant_action,
                                     pv.download_url AS variant_download_url
                              FROM products p
                              LEFT JOIN product_variants pv ON pv.id = ? AND pv.product_id = p.id
                              WHERE p.id = ? AND p.is_active = 1
                              LIMIT 1");
    $stmt->execute([$lookupVariantId, $productId]);
    $downloadRow = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$downloadRow) {
        http_response_code(404);
        echo FILE_NOT_FOUND_MESSAGE;
        exit;
    }

    if ($variantId > 0 && (int)($downloadRow['variant_id'] ?? 0) !== $variantId) {
        http_response_code(404);
        echo FILE_NOT_FOUND_MESSAGE;
        exit;
    }

    $resolvedAction = trim((string)($downloadRow['variant_action'] ?? ''));
    if ($resolvedAction === '') {
        $resolvedAction = trim((string)($downloadRow['product_action'] ?? 'cart'));
    }
    if (!in_array($resolvedAction, ['cart', 'download', 'external_link'], true)) {
        $resolvedAction = 'cart';
    }

    $resolvedPrice = $variantId > 0 ? trim((string)($downloadRow['variant_price'] ?? '')) : '';
    if ($resolvedPrice === '') {
        $resolvedPrice = trim((string)($downloadRow['product_price'] ?? ''));
    }

    $resolvedDownloadUrl = $variantId > 0 ? trim((string)($downloadRow['variant_download_url'] ?? '')) : '';
    if ($resolvedDownloadUrl === '') {
        $resolvedDownloadUrl = trim((string)($downloadRow['product_download_url'] ?? ''));
    }

    if ($resolvedAction !== 'download' || $resolvedDownloadUrl === '' || CurrencyService::parsePrice($resolvedPrice) > 0) {
        http_response_code(404);
        echo FILE_NOT_FOUND_MESSAGE;
        exit;
    }

    $publicPath = normalizeFreeDigitalDownloadPath($resolvedDownloadUrl);
    if ($publicPath === '') {
        http_response_code(404);
        echo FILE_NOT_FOUND_MESSAGE;
        exit;
    }

    $relativePath = ltrim(substr($publicPath, strlen(SITE_FILE_PATH_SEGMENT)), '/');
    if ($relativePath === '' || strpos($relativePath, "\0") !== false || preg_match(PATH_TRAVERSAL_REGEX, $relativePath)) {
        respondNotFoundFile();
    }

    $fileName = basename($relativePath);
    $filePath = __DIR__ . DATA_ADMIN_UPLOADS_SEGMENT . $relativePath;
    $blockedExtensions = BLOCKED_EXECUTABLE_EXTENSIONS;
    $ext = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
    if (!is_file($filePath) || in_array($ext, $blockedExtensions, true)) {
        respondNotFoundFile();
    }

    $finfo = new finfo(FILEINFO_MIME_TYPE);
    $mimeType = (string)$finfo->file($filePath);
    $blockedMimes = BLOCKED_SCRIPT_MIMES;
    if (in_array($mimeType, $blockedMimes, true)) {
        respondNotFoundFile();
    }

    $statsDb = DatabaseManager::getWriteConnection();
    $now = time();
    $variantKey = $variantId > 0 ? $variantId : 0;
    try {
        $statsStmt = $statsDb->prepare("INSERT INTO free_digital_download_stats (product_id, variant_id, download_count, first_download_at, last_download_at)
                                        VALUES (?, ?, 1, ?, ?)
                                        ON CONFLICT (product_id, variant_id)
                                        DO UPDATE SET download_count = free_digital_download_stats.download_count + 1,
                                                      last_download_at = EXCLUDED.last_download_at");
        $statsStmt->execute([$productId, $variantKey, $now, $now]);
    } catch (Throwable $e) {
        LogService::add('warning', 'Free digital download stats update failed', json_encode([
            'product_id' => $productId,
            'variant_id' => $variantKey,
            'error' => $e->getMessage()
        ]));
    }

    header(HEADER_LOCATION . $publicPath, true, 302);
    exit;
}

// Track and serve Easy Media AI ZIP downloads (public, aggregate-only analytics).
if ($pageSlug === 'easy-media-ai-download') {
    $configuredPath = ContentRenderer::normalizeEasyMediaZipDownloadPath((string)SettingsService::get('easy_media_ai_zip_url', $easyMediaAiZipDefault));
    if ($configuredPath === '') {
        respondNotFoundFile();
    }

    $relativePath = ltrim(substr($configuredPath, strlen(SITE_FILE_PATH_SEGMENT)), '/');
    if ($relativePath === '' || strpos($relativePath, "\0") !== false || preg_match(PATH_TRAVERSAL_REGEX, $relativePath)) {
        respondNotFoundFile();
    }

    $fileName = basename($relativePath);
    $filePath = __DIR__ . DATA_ADMIN_UPLOADS_SEGMENT . $relativePath;
    $blockedExtensions = BLOCKED_EXECUTABLE_EXTENSIONS;
    $ext = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
    if (!is_file($filePath) || in_array($ext, $blockedExtensions, true)) {
        respondNotFoundFile();
    }

    $finfo = new finfo(FILEINFO_MIME_TYPE);
    $mimeType = (string)$finfo->file($filePath);
    $blockedMimes = BLOCKED_SCRIPT_MIMES;
    if (in_array($mimeType, $blockedMimes, true)) {
        respondNotFoundFile();
    }

    $statsDb = DatabaseManager::getWriteConnection();
    $now = time();
    $nowDb = DbHelper::nowString();

    try {
        $countStmt = $statsDb->prepare("INSERT INTO settings (setting_key, setting_value, updated_at)
                                        VALUES ('easy_media_ai_zip_download_count', '1', ?)
                                        ON CONFLICT (setting_key)
                                        DO UPDATE SET setting_value = CASE
                                                                        WHEN settings.setting_value ~ '^[0-9]+$' THEN (settings.setting_value::bigint + 1)::text
                                                                        ELSE '1'
                                                                      END,
                                                      updated_at = EXCLUDED.updated_at");
        $countStmt->execute([$nowDb]);

        $firstStmt = $statsDb->prepare("INSERT INTO settings (setting_key, setting_value, updated_at)
                                        VALUES ('easy_media_ai_zip_first_download_at', ?, ?)
                                        ON CONFLICT (setting_key) DO NOTHING");
        $firstStmt->execute([(string)$now, $nowDb]);

        $lastStmt = $statsDb->prepare("INSERT INTO settings (setting_key, setting_value, updated_at)
                                       VALUES ('easy_media_ai_zip_last_download_at', ?, ?)
                                       ON CONFLICT (setting_key)
                                       DO UPDATE SET setting_value = EXCLUDED.setting_value,
                                                     updated_at = EXCLUDED.updated_at");
        $lastStmt->execute([(string)$now, $nowDb]);

        Cache::delete('setting:easy_media_ai_zip_download_count');
        Cache::delete('setting:easy_media_ai_zip_first_download_at');
        Cache::delete('setting:easy_media_ai_zip_last_download_at');
    } catch (Throwable $e) {
        LogService::add('warning', 'Easy Media AI ZIP stats update failed', json_encode([
            'error' => $e->getMessage()
        ]));
    }

    header(HEADER_LOCATION . $configuredPath, true, 302);
    exit;
}

// Serve public site files (logo, favicon, videos, etc.) - no auth required
if ($pageSlug === 'site-file' && isset($segments[1])) {
    // Handle subdirectories (e.g., thumbnails/)
    $relativePath = str_replace('\\', '/', implode('/', array_slice($segments, 1)));
    $relativePath = ltrim($relativePath, '/');
    if ($relativePath === '' || strpos($relativePath, "\0") !== false || preg_match(PATH_TRAVERSAL_REGEX, $relativePath)) {
        respondNotFoundFile();
    }
    $fileName = basename($relativePath);
    $filePath = __DIR__ . DATA_ADMIN_UPLOADS_SEGMENT . $relativePath;

    // Block dangerous executable/script extensions.
    $blockedExtensions = BLOCKED_EXECUTABLE_EXTENSIONS;
    $ext = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
    if (in_array($ext, $blockedExtensions, true)) {
        respondNotFoundFile();
    }

    if (file_exists($filePath) && is_file($filePath)) {
        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->file($filePath);

        // Block dangerous mime types.
        $blockedMimes = BLOCKED_SCRIPT_MIMES;

        if (!in_array($mimeType, $blockedMimes, true)) {
            // On-the-fly image optimization: resize and/or format conversion.
            // Query params: ?w=600&h=400&format=webp  (all optional, images only)
            $servePath = $filePath;
            $serveMime = $mimeType;
            $optHeight = isset($_GET['h']) ? (int) $_GET['h'] : null;
            $optWidth  = isset($_GET['w']) ? (int) $_GET['w'] : null;
            $optFormat = isset($_GET['format']) ? (string) $_GET['format'] : null;

            if (($optHeight !== null || $optWidth !== null || $optFormat !== null)
                && strpos($mimeType, 'image/') === 0
            ) {
                $targetFormat = $optFormat ?? $ext;
                $optimized = \NewSite\Image\ImageOptimizer::getOptimized(
                    $filePath,
                    $optHeight,
                    $targetFormat,
                    $optWidth
                );
                if ($optimized !== null) {
                    $servePath = $optimized['path'];
                    $serveMime = $optimized['mime'];
                }
            }

            $lastModified = filemtime($servePath);
            $fileSize = filesize($servePath);

            // Include optimization params in ETag so each variant is cached independently
            $etagBase = $fileName . $lastModified . $fileSize;
            if ($optHeight !== null || $optWidth !== null || $optFormat !== null) {
                $etagBase .= '|h' . ($optHeight ?? '') . '|w' . ($optWidth ?? '') . '|f' . ($optFormat ?? '');
            }
            $etag = '"' . md5($etagBase) . '"';

            $ifNoneMatch = $_SERVER['HTTP_IF_NONE_MATCH'] ?? '';
            $ifModifiedSince = $_SERVER['HTTP_IF_MODIFIED_SINCE'] ?? '';

            $cacheValid = ($ifNoneMatch && $ifNoneMatch === $etag)
                || ($ifModifiedSince && strtotime($ifModifiedSince) >= $lastModified);

            if ($cacheValid) {
                http_response_code(304);
                header(HEADER_ETAG . $etag);
                header(CACHE_CONTROL_PUBLIC_IMMUTABLE);
                exit;
            }

            $isInlineMime = (strpos($serveMime, 'image/') === 0)
                || (strpos($serveMime, 'video/') === 0)
                || (strpos($serveMime, 'audio/') === 0);

            header(HEADER_CONTENT_TYPE . $serveMime);
            header('X-Content-Type-Options: nosniff');
            header(HEADER_CONTENT_LENGTH . $fileSize);
            header('Content-Disposition: ' . ($isInlineMime ? 'inline' : 'attachment') . '; filename="' . $fileName . '"');
            header(HEADER_LAST_MODIFIED . gmdate(HTTP_GMT_DATE_FORMAT, $lastModified) . ' GMT');
            header(HEADER_ETAG . $etag);
            header(CACHE_CONTROL_PUBLIC_IMMUTABLE);
            readfile($servePath);
            exit;
        }
    }

    http_response_code(404);
    echo FILE_NOT_FOUND_MESSAGE;
    exit;
}

// Serve admin files securely (admin only)
if ($pageSlug === 'admin-file' && isset($segments[1])) {
    AdminAuth::requireLogin();
    // Handle subdirectories (e.g., thumbnails/)
    $relativePath = str_replace('\\', '/', implode('/', array_slice($segments, 1)));
    $relativePath = ltrim($relativePath, '/');
    if ($relativePath === '' || strpos($relativePath, "\0") !== false || preg_match(PATH_TRAVERSAL_REGEX, $relativePath)) {
        respondNotFoundFile();
    }
    $fileName = basename($relativePath);
    $filePath = __DIR__ . DATA_ADMIN_UPLOADS_SEGMENT . $relativePath;

    if (file_exists($filePath) && is_file($filePath)) {
        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->file($filePath);

        // Block dangerous mime types
        $blockedMimes = ['application/x-php', 'application/x-httpd-php', 'text/x-php'];
        if (!in_array($mimeType, $blockedMimes, true)) {
            // Get file modification time and size for caching
            $lastModified = filemtime($filePath);
            $fileSize = filesize($filePath);
            $etag = '"' . md5($fileName . $lastModified . $fileSize) . '"';

            // Check if browser has valid cached version
            $ifNoneMatch = $_SERVER['HTTP_IF_NONE_MATCH'] ?? '';
            $ifModifiedSince = $_SERVER['HTTP_IF_MODIFIED_SINCE'] ?? '';

            $cacheValid = ($ifNoneMatch && $ifNoneMatch === $etag)
                || ($ifModifiedSince && strtotime($ifModifiedSince) >= $lastModified);

            if ($cacheValid) {
                http_response_code(304);
                header(HEADER_ETAG . $etag);
                header(CACHE_CONTROL_PRIVATE_IMMUTABLE);
                exit;
            }

            // Serve the file with caching headers
            header(HEADER_CONTENT_TYPE . $mimeType);
            header('X-Content-Type-Options: nosniff');
            header(HEADER_CONTENT_LENGTH . $fileSize);
            header('Content-Disposition: inline; filename="' . $fileName . '"');
            header(HEADER_LAST_MODIFIED . gmdate(HTTP_GMT_DATE_FORMAT, $lastModified) . ' GMT');
            header(HEADER_ETAG . $etag);
            header(CACHE_CONTROL_PRIVATE_IMMUTABLE);
            readfile($filePath);
            exit;
        }
    }

    http_response_code(404);
    echo FILE_NOT_FOUND_MESSAGE;
    exit;
}

if ($pageSlug === 'download') {
    TemplateRenderer::renderOnce(__DIR__ . '/templates/download.php');
    exit;
}

// Contact settings
$contactEnabled = ContactService::isEnabled();
$contactEmail = ContactService::getEmail();

// Legal page & cookie notice toggles
$pageTermsEnabled = SettingsService::get('page_terms_enabled', '1') === '1';
$pagePrivacyEnabled = SettingsService::get('page_privacy_enabled', '1') === '1';
$pageRefundEnabled = SettingsService::get('page_refund_enabled', '1') === '1';
$pageShippingEnabled = SettingsService::get('page_shipping_enabled', '1') === '1';
$pageLegalNoticeEnabled = SettingsService::get('page_legal_notice_enabled', '1') === '1';
$pageDmcaEnabled = SettingsService::get('page_dmca_enabled', '1') === '1';
$cookieNoticeEnabled = SettingsService::get('cookie_notice_enabled', '1') === '1';

// Map legal page slugs to their toggle — used to 404 disabled pages
$legalPageToggles = [
    'terms-of-service' => $pageTermsEnabled,
    'privacy-policy'   => $pagePrivacyEnabled,
    'refund-policy'    => $pageRefundEnabled,
    'shipping-policy'  => $pageShippingEnabled,
    'legal-notice'     => $pageLegalNoticeEnabled,
    'dmca'             => $pageDmcaEnabled,
];
$contactSubjectDefault = SettingsService::get('contact_subject', 'Contact Form Message');
$stripePublishableKey = SettingsService::get('stripe_publishable_key', '');
$stripeSecretKey = SettingsService::get('stripe_secret_key', '');
$stripeCheckoutEnabled = $stripePublishableKey !== '' && $stripeSecretKey !== '';
$contactSuccess = false;
$contactError = false;
$contactForm = [
    'name' => '',
    'email' => '',
    'subject' => '',
    'message' => ''
];

$dmcaSuccess = false;
$dmcaError = false;
$dmcaVerifyMessage = '';
$dmcaVerifyError = '';
$dmcaForm = [
    'name' => '',
    'email' => '',
    'address' => '',
    'phone' => '',
    'work_description' => '',
    'infringing_urls' => '',
    'signature' => ''
];

$contactFlash = $_SESSION['contact_flash'] ?? null;
if (is_array($contactFlash)) {
    $contactSuccess = $contactFlash['success'] ?? false;
    $contactError = $contactFlash['error'] ?? false;
    unset($_SESSION['contact_flash']);
}

$dmcaFlash = $_SESSION['dmca_flash'] ?? null;
if (is_array($dmcaFlash)) {
    $dmcaSuccess = $dmcaFlash['success'] ?? false;
    $dmcaError = $dmcaFlash['error'] ?? false;
    $dmcaVerifyMessage = $dmcaFlash['verify_message'] ?? '';
    $dmcaVerifyError = $dmcaFlash['verify_error'] ?? '';
    unset($_SESSION['dmca_flash']);
}

$productCommentFlash = $_SESSION['product_comment_flash'] ?? null;
if (is_array($productCommentFlash)) {
    unset($_SESSION['product_comment_flash']);
} else {
    $productCommentFlash = null;
}

// Social links
$socialLinks = [];
$socialMap = [
    'facebook' => ['label' => 'Facebook', 'icon' => '<svg viewBox="0 0 24 24" aria-hidden="true"><path d="M14 9h3V5h-3c-2.2 0-4 1.8-4 4v3H7v4h3v7h4v-7h3l1-4h-4V9z"></path></svg>', 'key' => 'social_facebook'],
    'instagram' => ['label' => 'Instagram', 'icon' => '<svg viewBox="0 0 24 24" aria-hidden="true"><rect x="3" y="3" width="18" height="18" rx="5"></rect><circle cx="12" cy="12" r="4"></circle><circle cx="17" cy="7" r="1"></circle></svg>', 'key' => 'social_instagram'],
    'x' => ['label' => 'X', 'icon' => '<svg viewBox="0 0 24 24" aria-hidden="true"><path d="M4 4l16 16"></path><path d="M20 4L4 20"></path></svg>', 'key' => 'social_x'],
    'youtube' => ['label' => 'YouTube', 'icon' => '<svg viewBox="0 0 24 24" aria-hidden="true"><rect x="3" y="6" width="18" height="12" rx="3"></rect><path d="M10 9l5 3-5 3z"></path></svg>', 'key' => 'social_youtube'],
    'discord' => ['label' => 'Discord', 'icon' => '<svg viewBox="0 0 24 24" aria-hidden="true"><path d="M7 7c3-2 7-2 10 0l2 9c-3 2-11 2-14 0l2-9z"></path><path d="M9 13h.01"></path><path d="M15 13h.01"></path></svg>', 'key' => 'social_discord'],
    'tiktok' => ['label' => 'TikTok', 'icon' => '<svg viewBox="0 0 24 24" aria-hidden="true"><path d="M14 4v10.5a3.5 3.5 0 1 1-3.5-3.5"></path><path d="M14 4c1.5 2 3.5 3 5 3"></path></svg>', 'key' => 'social_tiktok'],
    'linkedin' => ['label' => 'LinkedIn', 'icon' => '<svg viewBox="0 0 24 24" aria-hidden="true"><rect x="3" y="3" width="18" height="18" rx="2"></rect><path d="M8 10v7"></path><path d="M8 7h.01"></path><path d="M12 10v7"></path><path d="M12 13c0-2 3-2 3 0v4"></path></svg>', 'key' => 'social_linkedin']
];
foreach ($socialMap as $item) {
    $url = trim((string)SettingsService::get($item['key'], ''));
    if ($url !== '' && filter_var($url, FILTER_VALIDATE_URL)) {
        $socialLinks[] = ['label' => $item['label'], 'icon' => $item['icon'], 'url' => $url];
    }
}
$hasSocialLinks = !empty($socialLinks);

// Resolve page
$isContactPage = false;
$db = DatabaseManager::getWriteConnection();
$page = null;

if ($pageSlug === 'contact' && $contactEnabled) {
    $isContactPage = true;

    $page = [
        'title' => 'Contact',
        'slug' => 'contact',
        'content' => '',
        'meta_title' => 'Contact',
        'meta_description' => 'Contact us.'
    ];
    $stmt = $db->prepare(SQL_SELECT_PUBLISHED_PAGE_BY_SLUG);
    $stmt->execute(['contact']);
    $pageRow = $stmt->fetch();
    if ($pageRow) {
        $page['id'] = $pageRow['id'];
        if (!empty($pageRow['title'])) {
            $page['title'] = $pageRow['title'];
        }
        if (!empty($pageRow['meta_title'])) {
            $page['meta_title'] = $pageRow['meta_title'];
        }
        if (!empty($pageRow['meta_description'])) {
            $page['meta_description'] = $pageRow['meta_description'];
        }
    }

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $ip = IpService::getClientIP();
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $referrer = $_SERVER['HTTP_REFERER'] ?? '';
        $honey = $_POST['website'] ?? '';

        $contactForm['name'] = trim($_POST['name'] ?? '');
        $contactForm['email'] = trim($_POST['email'] ?? '');
        $contactForm['subject'] = trim($_POST['subject'] ?? '');
        $contactForm['message'] = trim($_POST['message'] ?? '');

        $isAdminSender = AdminAuth::isAllowedIp($ip);

        if (!empty($honey)) {
            $_SESSION['contact_flash'] = ['success' => 'Message sent successfully!'];
            header(HEADER_LOCATION . CONTACT_SENT_ROUTE);
            exit;
        } elseif (!$isAdminSender && !RateLimiter::contactOk($ip, 120)) {
            $contactError = 'You are sending messages too quickly. Please wait a few minutes.';
        } elseif (empty($contactForm['name']) || empty($contactForm['email']) || empty($contactForm['message'])) {
            $contactError = 'Please fill in all required fields.';
        } elseif (!filter_var($contactForm['email'], FILTER_VALIDATE_EMAIL)) {
            $contactError = 'Invalid email address.';
        } else {
            $attachmentLink = '';
            $attachmentForEmail = null;

            if (isset($_FILES['attachment']) && $_FILES['attachment']['error'] !== UPLOAD_ERR_NO_FILE) {
                if ($_FILES['attachment']['error'] === UPLOAD_ERR_OK) {
                    $file = $_FILES['attachment'];

                    $allowedMap = UPLOAD_IMAGE_MIME_MAP;

                    $validation = UploadService::validateFile($file, $allowedMap, true);
                    $maxSize = 5 * 1024 * 1024;

                    if (!$validation['success']) {
                        $contactError = $validation['error'] ?: 'Invalid file format.';
                        LogService::add('warning', 'Contact attachment blocked', json_encode(['mime' => $validation['mime'], 'ext' => $validation['ext'], 'ip' => $ip]));
                    } elseif ($file['size'] > $maxSize) {
                        $contactError = 'File too large. Max 5MB.';
                        LogService::add('warning', 'Contact attachment blocked (size)', json_encode(['size' => $file['size'], 'ip' => $ip]));
                    } else {
                        $uploadDir = __DIR__ . '/../data/contact_uploads/';
                        if (!is_dir($uploadDir)) {
                            @mkdir($uploadDir, 0755, true);
                            file_put_contents($uploadDir . '/.htaccess', "Options -Indexes\n<FilesMatch \"\\.php$\">\n    Deny from all\n</FilesMatch>\nAddType application/octet-stream .php .php5 .phtml");
                        }

                        $newName = 'contact_' . time() . '_' . bin2hex(random_bytes(8)) . '.' . ($validation['ext'] ?? 'jpg');
                        $dest = $uploadDir . $newName;

                        if (move_uploaded_file($file['tmp_name'], $dest)) {
                            $attachmentLink = '/contact-upload/' . $newName;
                            $attachmentForEmail = [
                                'path' => $dest,
                                'name' => $newName,
                                'mime' => $validation['mime']
                            ];
                            LogService::add('info', 'Contact attachment uploaded', json_encode(['path' => $attachmentLink, 'ip' => $ip]));
                        } else {
                            $contactError = 'Failed to upload attachment.';
                            LogService::add('error', 'Contact attachment upload failed', json_encode(['ip' => $ip]));
                        }
                    }
                } else {
                    $contactError = 'Upload error.';
                    LogService::add('error', 'Contact attachment upload error', json_encode(['code' => $_FILES['attachment']['error'], 'ip' => $ip]));
                }
            } else {
                LogService::add('info', 'Contact message without attachment', json_encode(['ip' => $ip]));
            }

            if ($contactError) {
                // stop processing
            } else {
                ContactService::saveMessage(
                    $contactForm['name'],
                    $contactForm['email'],
                    $contactForm['subject'],
                    $contactForm['message'],
                    $ip,
                    $userAgent,
                    $referrer
                );
                RateLimiter::contactUpdate($ip);
                LogService::add('info', 'Contact message received', json_encode([
                    'email' => $contactForm['email'],
                    'subject' => $contactForm['subject'] ?: $contactSubjectDefault,
                    'ip' => $ip,
                    'has_attachment' => (bool)$attachmentForEmail
                ]));
                $smtp = [
                    'host' => SettingsService::get('smtp_host', ''),
                    'port' => SettingsService::get('smtp_port', '587'),
                    'user' => SettingsService::get('smtp_user', ''),
                    'pass' => SettingsService::get('smtp_pass', ''),
                    'secure' => SettingsService::get('smtp_secure', 'tls')
                ];
                $subject = $contactForm['subject'] ? $contactForm['subject'] : $contactSubjectDefault;
                $body = "Name: {$contactForm['name']}\n";
                $body .= "Email: {$contactForm['email']}\n";
                $body .= "Subject: {$subject}\n\n";
                $body .= "Message:\n{$contactForm['message']}\n";
                if (!empty($contactEmail)) {
                    $sent = SmtpMailer::send($contactEmail, $subject, $body, $contactForm['email'], $contactForm['name'], $smtp, $attachmentForEmail);
                    if ($sent) {
                        LogService::add('info', 'Contact email sent', json_encode([
                            'to' => $contactEmail,
                            'subject' => $subject,
                            'has_attachment' => (bool)$attachmentForEmail
                        ]));
                    } else {
                        LogService::add('error', 'Contact email failed', json_encode([
                            'to' => $contactEmail,
                            'subject' => $subject,
                            'has_attachment' => (bool)$attachmentForEmail
                        ]));
                    }
                }
                $_SESSION['contact_flash'] = ['success' => 'Message sent successfully! I will get back to you soon.'];
                header(HEADER_LOCATION . CONTACT_SENT_ROUTE);
                exit;
            }
        }
    }

} elseif ($isDmcaPage && $pageDmcaEnabled) {
    $page = [
        'title' => 'DMCA',
        'slug' => 'dmca',
        'content' => '',
        'meta_title' => 'DMCA',
        'meta_description' => 'DMCA takedown policy and request form.'
    ];
    $stmt = $db->prepare(SQL_SELECT_PUBLISHED_PAGE_BY_SLUG);
    $stmt->execute(['dmca']);
    $pageRow = $stmt->fetch();
    if ($pageRow) {
        $page['id'] = $pageRow['id'];
        if (!empty($pageRow['title'])) {
            $page['title'] = $pageRow['title'];
        }
        if (!empty($pageRow['meta_title'])) {
            $page['meta_title'] = $pageRow['meta_title'];
        }
        if (!empty($pageRow['meta_description'])) {
            $page['meta_description'] = $pageRow['meta_description'];
        }
        $page['content'] = $pageRow['content'] ?? '';
    }

    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        $_SESSION['dmca_form_time'] = time();
        $_SESSION['dmca_form_nonce'] = bin2hex(random_bytes(16));
    }

    $verifyToken = trim($_GET['verify'] ?? '');
    if ($verifyToken !== '') {
        $tokenHash = hash('sha256', $verifyToken);
        $stmt = $db->prepare("SELECT * FROM dmca_requests WHERE verify_token_hash = ? LIMIT 1");
        $stmt->execute([$tokenHash]);
        $dmcaRow = $stmt->fetch();

        if (!$dmcaRow) {
            $dmcaVerifyError = 'Verification link is invalid or expired.';
        } elseif (!empty($dmcaRow['verified_at'])) {
            $dmcaVerifyMessage = 'Your DMCA request is already verified.';
        } elseif (!empty($dmcaRow['verify_expires_at']) && strtotime($dmcaRow['verify_expires_at']) < time()) {
            $dmcaVerifyError = 'Verification link has expired. Please submit again.';
        } else {
            $now = DbHelper::nowString();
            $stmt = $db->prepare("UPDATE dmca_requests SET status = 'verified', verified_at = ?, verify_token_hash = NULL, verify_expires_at = NULL WHERE id = ?");
            $stmt->execute([$now, $dmcaRow['id']]);

            $dmcaVerifyMessage = 'Your DMCA request is verified. Our team will review it soon.';

            $contactEmailForDmca = ContactService::getEmail();
            $smtp = [
                'host' => SettingsService::get('smtp_host', ''),
                'port' => SettingsService::get('smtp_port', '587'),
                'user' => SettingsService::get('smtp_user', ''),
                'pass' => SettingsService::get('smtp_pass', ''),
                'secure' => SettingsService::get('smtp_secure', 'tls')
            ];

            $adminBody = "DMCA request verified.\n\n";
            $adminBody .= "Name: " . ($dmcaRow['name'] ?? '') . "\n";
            $adminBody .= "Email: " . ($dmcaRow['email'] ?? '') . "\n";
            $adminBody .= "Address: " . ($dmcaRow['address'] ?? '') . "\n";
            $adminBody .= "Phone: " . ($dmcaRow['phone'] ?? '') . "\n\n";
            $adminBody .= "Work description:\n" . ($dmcaRow['work_description'] ?? '') . "\n\n";
            $adminBody .= "Infringing URLs:\n" . ($dmcaRow['infringing_urls'] ?? '') . "\n\n";
            $adminBody .= "Signature: " . ($dmcaRow['signature'] ?? '') . "\n";
            $adminBody .= "Good-faith: " . (!empty($dmcaRow['good_faith']) ? 'Yes' : 'No') . "\n";
            $adminBody .= "Perjury: " . (!empty($dmcaRow['perjury']) ? 'Yes' : 'No') . "\n";
            $adminBody .= "Authorized: " . (!empty($dmcaRow['authorized']) ? 'Yes' : 'No') . "\n";

            if ($contactEmailForDmca !== '' && $smtp['host'] !== '' && $smtp['user'] !== '' && $smtp['pass'] !== '') {
                SmtpMailer::send($contactEmailForDmca, 'DMCA request verified', $adminBody, $dmcaRow['email'] ?? '', $dmcaRow['name'] ?? '', $smtp);
            }

            if (!empty($dmcaRow['email']) && $smtp['host'] !== '' && $smtp['user'] !== '' && $smtp['pass'] !== '') {
                $siteName = trim((string)SettingsService::get('site_name', ''));
                $contactEmailForUser = $contactEmailForDmca !== '' ? $contactEmailForDmca : trim((string)ContactService::getEmail());
                $vars = [
                    '{name}' => trim((string)($dmcaRow['name'] ?? '')),
                    '{request_id}' => (string)($dmcaRow['id'] ?? ''),
                    '{contact_email}' => $contactEmailForUser,
                    '{site_name}' => $siteName
                ];
                $subjectTemplate = SettingsService::get('dmca_verified_subject', 'Your DMCA request is verified');
                $bodyTemplate = SettingsService::get('dmca_verified_body', "Hello {name},\n\nYour DMCA request (ID {request_id}) is verified. Our team will review it and contact you if more information is needed.\n\nIf you have questions, contact {contact_email}.\n\nThank you,\n{site_name}");
                $finalSubject = ContentRenderer::renderEmailTemplate($subjectTemplate, $vars);
                $finalBody = ContentRenderer::renderEmailTemplate($bodyTemplate, $vars);
                SmtpMailer::send($dmcaRow['email'], $finalSubject, $finalBody, '', '', $smtp);
            }
        }
    }

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $ip = IpService::getClientIP();
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $honey = $_POST['website'] ?? '';
        $formTime = (int)($_POST['dmca_time'] ?? 0);
        $sessionTime = (int)($_SESSION['dmca_form_time'] ?? 0);
        $formNonce = trim($_POST['dmca_nonce'] ?? '');
        $sessionNonce = trim($_SESSION['dmca_form_nonce'] ?? '');

        $dmcaForm['name'] = trim($_POST['name'] ?? '');
        $dmcaForm['email'] = trim($_POST['email'] ?? '');
        $dmcaForm['address'] = trim($_POST['address'] ?? '');
        $dmcaForm['phone'] = trim($_POST['phone'] ?? '');
        $dmcaForm['work_description'] = trim($_POST['work_description'] ?? '');
        $dmcaForm['infringing_urls'] = trim($_POST['infringing_urls'] ?? '');
        $dmcaForm['signature'] = trim($_POST['signature'] ?? '');

        $goodFaith = !empty($_POST['good_faith']);
        $perjury = !empty($_POST['perjury']);
        $authorized = !empty($_POST['authorized']);

        if (!empty($honey)) {
            $_SESSION['dmca_flash'] = ['success' => 'Thanks. Please check your email to verify your request.'];
            header(HEADER_LOCATION . DMCA_SENT_ROUTE);
            exit;
        } elseif ($sessionTime <= 0 || $formTime !== $sessionTime || (time() - $sessionTime) < 4 || $formNonce === '' || $formNonce !== $sessionNonce) {
            $dmcaError = 'Please wait a moment and try submitting again.';
        } elseif (empty($dmcaForm['name']) || empty($dmcaForm['email']) || empty($dmcaForm['address']) || empty($dmcaForm['work_description']) || empty($dmcaForm['infringing_urls']) || empty($dmcaForm['signature'])) {
            $dmcaError = 'Please fill in all required fields.';
        } elseif (!filter_var($dmcaForm['email'], FILTER_VALIDATE_EMAIL)) {
            $dmcaError = 'Invalid email address.';
        } elseif (!$goodFaith || !$perjury || !$authorized) {
            $dmcaError = 'You must confirm all required statements.';
        } else {
            $rateStatus = RateLimiter::dmcaStatusFor($ip, $dmcaForm['email']);
            if (!$rateStatus['ok']) {
                $waitMinutes = (int)ceil($rateStatus['wait'] / 60);
                $dmcaError = 'Too many requests. Please try again in ' . max(1, $waitMinutes) . ' minute(s).';
            } else {
                $urlLines = preg_split('/\r\n|\r|\n/', $dmcaForm['infringing_urls']);
                $validUrls = [];
                foreach ($urlLines as $line) {
                    $line = trim($line);
                    if ($line === '') {
                        continue;
                    }
                    if (filter_var($line, FILTER_VALIDATE_URL)) {
                        $validUrls[] = $line;
                    }
                }

                if (empty($validUrls)) {
                    $dmcaError = 'Please provide at least one valid URL.';
                } else {
                    $smtp = [
                        'host' => SettingsService::get('smtp_host', ''),
                        'port' => SettingsService::get('smtp_port', '587'),
                        'user' => SettingsService::get('smtp_user', ''),
                        'pass' => SettingsService::get('smtp_pass', ''),
                        'secure' => SettingsService::get('smtp_secure', 'tls')
                    ];
                    if ($smtp['host'] === '' || $smtp['user'] === '' || $smtp['pass'] === '') {
                        $dmcaError = 'Email verification is not configured. Please contact site support.';
                    } else {
                        $token = bin2hex(random_bytes(32));
                        $tokenHash = hash('sha256', $token);
                        $expiresAt = date('Y-m-d H:i:s', time() + 86400);
                        $now = DbHelper::nowString();
                        $urlsText = implode("\n", $validUrls);

                        $stmt = $db->prepare("INSERT INTO dmca_requests (status, name, email, address, phone, work_description, infringing_urls, good_faith, perjury, authorized, signature, ip_address, user_agent, verify_token_hash, verify_expires_at, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                        $stmt->execute([
                            'pending',
                            $dmcaForm['name'],
                            $dmcaForm['email'],
                            $dmcaForm['address'],
                            $dmcaForm['phone'],
                            substr($dmcaForm['work_description'], 0, 4000),
                            substr($urlsText, 0, 4000),
                            $goodFaith ? 1 : 0,
                            $perjury ? 1 : 0,
                            $authorized ? 1 : 0,
                            $dmcaForm['signature'],
                            IpService::gdprStore($ip),
                            substr($userAgent, 0, 500),
                            $tokenHash,
                            $expiresAt,
                            $now
                        ]);

                        $dmcaId = DbHelper::lastInsertId($db, 'dmca_requests');

                        RateLimiter::dmcaUpdate($ip, $dmcaForm['email']);

                        $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
                        $host = $_SERVER['HTTP_HOST'] ?? '';
                        $verifyUrl = $scheme . '://' . $host . '/dmca?verify=' . urlencode($token);

                        $siteName = trim((string)SettingsService::get('site_name', ''));
                        $contactEmailForUser = trim((string)ContactService::getEmail());
                        $vars = [
                            '{name}' => $dmcaForm['name'],
                            '{verify_url}' => $verifyUrl,
                            '{request_id}' => (string)$dmcaId,
                            '{contact_email}' => $contactEmailForUser,
                            '{site_name}' => $siteName
                        ];
                        $subjectTemplate = SettingsService::get('dmca_request_subject', 'Verify your DMCA request');
                        $bodyTemplate = SettingsService::get('dmca_request_body', "Hello {name},\n\nPlease verify your DMCA request by clicking the link below:\n{verify_url}\n\nIf you did not submit this request, you can ignore this email.\n\nThank you,\n{site_name}");
                        $finalSubject = ContentRenderer::renderEmailTemplate($subjectTemplate, $vars);
                        $finalBody = ContentRenderer::renderEmailTemplate($bodyTemplate, $vars);
                        $sent = SmtpMailer::send($dmcaForm['email'], $finalSubject, $finalBody, '', '', $smtp);

                        if ($sent) {
                            $_SESSION['dmca_flash'] = ['success' => 'Thanks. Please check your email to verify your request.'];
                            header(HEADER_LOCATION . DMCA_SENT_ROUTE);
                            exit;
                        }

                        $dmcaError = 'Unable to send verification email. Please try again later.';
                    }
                }
            }
        }
    }

} elseif ($isCartPage) {
    $page = [
        'title' => 'Cart',
        'slug' => 'cart',
        'content' => '',
        'meta_title' => 'Cart',
        'meta_description' => 'Your shopping cart.'
    ];
    $stmt = $db->prepare(SQL_SELECT_PUBLISHED_PAGE_BY_SLUG);
    $stmt->execute(['cart']);
    $pageRow = $stmt->fetch();
    if ($pageRow) {
        $page['id'] = $pageRow['id'];
        if (!empty($pageRow['title'])) {
            $page['title'] = $pageRow['title'];
        }
        if (!empty($pageRow['meta_title'])) {
            $page['meta_title'] = $pageRow['meta_title'];
        }
        if (!empty($pageRow['meta_description'])) {
            $page['meta_description'] = $pageRow['meta_description'];
        }
    }
} elseif ($isSearchPage) {
    $page = [
        'title' => 'Search',
        'slug' => 'search',
        'content' => '',
        'meta_title' => 'Search',
        'meta_description' => 'Search results.'
    ];
} elseif ($isMessagesPage) {
    header(HEADER_LOCATION . ACCOUNT_MESSAGES_ROUTE);
    exit;
} elseif ($isAccountSettingsPage) {
    header(HEADER_LOCATION . ACCOUNT_SETTINGS_ROUTE);
    exit;
} elseif ($isLoginPage) {
    $page = [
        'title' => 'Login',
        'slug' => 'login',
        'content' => '',
        'meta_title' => 'Login',
        'meta_description' => 'Access your account.'
    ];
} elseif ($isAccountPage) {
    $page = [
        'title' => 'Account',
        'slug' => 'account',
        'content' => '',
        'meta_title' => 'Account',
        'meta_description' => 'Your account dashboard.'
    ];
} elseif ($isFriendsPage) {
    $page = [
        'title' => 'Friends',
        'slug' => 'friends',
        'content' => '',
        'meta_title' => 'Friends',
        'meta_description' => 'Manage your friends list.'
    ];
} elseif ($isForumPostPage) {
    if (!$forumEnabled) {
        http_response_code(404);
        $page = [
            'title' => NOT_FOUND_PAGE_TITLE,
            'slug' => '404',
            'content' => '',
            'meta_title' => NOT_FOUND_META_TITLE,
            'meta_description' => NOT_FOUND_META_DESCRIPTION
        ];
    } else {
        // Resolve forum post by slug or legacy numeric ID
        if ($forumPostSlug !== '') {
            $stmt = $db->prepare("
                SELECT p.id, p.user_id, p.title, p.body, p.slug, p.created_at, COALESCE(p.views, 0) AS views,
                       s.id AS sub_id, s.name AS sub_name, s.slug AS sub_slug,
                       c.id AS cat_id, c.name AS cat_name, c.slug AS cat_slug,
                       u.display_name, u.nickname, u.profile_photo,
                       COALESCE(comments.comment_count, 0) AS comment_count
                FROM forum_posts p
                JOIN forum_subcategories s ON s.id = p.subcategory_id AND s.is_active = 1
                JOIN forum_categories c ON c.id = s.category_id AND c.is_active = 1
                LEFT JOIN site_users u ON u.id = p.user_id
                LEFT JOIN (
                    SELECT post_id, COUNT(*) AS comment_count
                    FROM forum_comments
                    WHERE status = 'approved'
                    GROUP BY post_id
                ) comments ON comments.post_id = p.id
                WHERE p.slug = ? AND p.status = 'approved'
                LIMIT 1
            ");
            $stmt->execute([$forumPostSlug]);
        } else {
            $stmt = $db->prepare("
                SELECT p.id, p.user_id, p.title, p.body, p.slug, p.created_at, COALESCE(p.views, 0) AS views,
                       s.id AS sub_id, s.name AS sub_name, s.slug AS sub_slug,
                       c.id AS cat_id, c.name AS cat_name, c.slug AS cat_slug,
                       u.display_name, u.nickname, u.profile_photo,
                       COALESCE(comments.comment_count, 0) AS comment_count
                FROM forum_posts p
                JOIN forum_subcategories s ON s.id = p.subcategory_id AND s.is_active = 1
                JOIN forum_categories c ON c.id = s.category_id AND c.is_active = 1
                LEFT JOIN site_users u ON u.id = p.user_id
                LEFT JOIN (
                    SELECT post_id, COUNT(*) AS comment_count
                    FROM forum_comments
                    WHERE status = 'approved'
                    GROUP BY post_id
                ) comments ON comments.post_id = p.id
                WHERE p.id = ? AND p.status = 'approved'
                LIMIT 1
            ");
            $stmt->execute([$forumPostId]);
        }
        $forumPostData = $stmt->fetch();

        // 301 redirect legacy /forum/post/{id} to /forum/post/{slug}
        if ($forumPostData && $forumPostId > 0) {
            $resolvedSlug = trim((string)($forumPostData['slug'] ?? ''));
            if ($resolvedSlug !== '') {
                header(HEADER_LOCATION . FORUM_POST_ROUTE . rawurlencode($resolvedSlug), true, 301);
                exit;
            }
        }
        // Ensure $forumPostId is set for downstream use
        if ($forumPostData) {
            $forumPostId = (int)($forumPostData['id'] ?? 0);
        }

        if (!$forumPostData) {
            http_response_code(404);
            $page = [
                'title' => NOT_FOUND_PAGE_TITLE,
                'slug' => '404',
                'content' => '',
                'meta_title' => NOT_FOUND_META_TITLE,
                'meta_description' => NOT_FOUND_META_DESCRIPTION
            ];
        } else {
            $metaDescription = trim(strip_tags((string)($forumPostData['body'] ?? '')));
            if (function_exists('mb_strlen') && mb_strlen($metaDescription) > 160) {
                $metaDescription = mb_substr($metaDescription, 0, 157) . '...';
            } elseif (strlen($metaDescription) > 160) {
                $metaDescription = substr($metaDescription, 0, 157) . '...';
            }
            $page = [
                'title' => $forumPostData['title'] ?? FORUM_POST_DEFAULT_TITLE,
                'slug' => 'forum',
                'content' => '',
                'meta_title' => ($forumPostData['title'] ?? FORUM_POST_DEFAULT_TITLE) . FORUM_META_TITLE_SUFFIX,
                'meta_description' => $metaDescription !== '' ? $metaDescription : 'Forum discussion.'
            ];

            $postTitle = trim((string)($forumPostData['title'] ?? FORUM_POST_DEFAULT_TITLE));
            if ($postTitle === '') {
                $postTitle = FORUM_POST_DEFAULT_TITLE;
            }

            $forumCrumbs = [
                ['label' => 'Forum', 'url' => '/forum', 'is_home' => false],
            ];

            $catId = (int)($forumPostData['cat_id'] ?? 0);
            $catName = trim((string)($forumPostData['cat_name'] ?? ''));
            $catSlug = trim((string)($forumPostData['cat_slug'] ?? ''));
            if ($catId > 0 && $catName !== '' && $catSlug !== '') {
                $forumCrumbs[] = [
                    'label' => $catName,
                    'url' => FORUM_CATEGORY_ROUTE . rawurlencode($catSlug),
                    'is_home' => false,
                ];
            }

            // Build nested subcategory chain from leaf -> root, then reverse.
            $subChain = [];
            $seenSubIds = [];
            $currentSubId = (int)($forumPostData['sub_id'] ?? 0);
            $hops = 0;
            while ($currentSubId > 0 && $hops < 12 && !isset($seenSubIds[$currentSubId])) {
                $seenSubIds[$currentSubId] = true;
                $subStmt = $db->prepare('SELECT id, name, slug, parent_id FROM forum_subcategories WHERE id = ? AND is_active = 1 LIMIT 1');
                $subStmt->execute([$currentSubId]);
                $subRow = $subStmt->fetch(PDO::FETCH_ASSOC);
                if (!is_array($subRow)) {
                    break;
                }
                $subChain[] = [
                    'id' => (int)($subRow['id'] ?? 0),
                    'name' => trim((string)($subRow['name'] ?? '')),
                    'slug' => trim((string)($subRow['slug'] ?? '')),
                    'parent_id' => isset($subRow['parent_id']) && $subRow['parent_id'] !== null ? (int)$subRow['parent_id'] : 0,
                ];
                $currentSubId = (int)($subRow['parent_id'] ?? 0);
                $hops += 1;
            }
            $subChain = array_reverse($subChain);
            foreach ($subChain as $subNode) {
                $subNodeId = (int)($subNode['id'] ?? 0);
                $subNodeName = trim((string)($subNode['name'] ?? ''));
                $subNodeSlug = trim((string)($subNode['slug'] ?? ''));
                if ($subNodeId <= 0 || $subNodeName === '' || $subNodeSlug === '') {
                    continue;
                }
                $forumCrumbs[] = [
                    'label' => $subNodeName,
                    'url' => FORUM_SUBCATEGORY_ROUTE . rawurlencode($subNodeSlug),
                    'is_home' => false,
                ];
            }

            $forumCrumbs[] = [
                'label' => $postTitle,
                'url' => null,
                'is_home' => false,
                'is_current' => true,
            ];
            $page['breadcrumbs_override'] = $forumCrumbs;
        }
    }
} elseif ($isForumPage) {
    if (!$forumEnabled) {
        http_response_code(404);
        $page = [
            'title' => NOT_FOUND_PAGE_TITLE,
            'slug' => '404',
            'content' => '',
            'meta_title' => NOT_FOUND_META_TITLE,
            'meta_description' => NOT_FOUND_META_DESCRIPTION
        ];
    } else {
        $page = [
            'title' => 'Forum',
            'slug' => 'forum',
            'content' => '',
            'meta_title' => 'Forum',
            'meta_description' => 'Community forum.'
        ];

        $selectedCategoryId = 0;
        $selectedSubcategoryId = 0;
        $hasCategorySelection = false;
        $hasSubcategorySelection = false;

        // 301 redirect legacy ?category=X and ?subcategory=X query params to slug-based paths
        if (isset($_GET['subcategory']) && (int)($_GET['subcategory'] ?? 0) > 0) {
            $legacySubId = (int)$_GET['subcategory'];
            $legacySubStmt = $db->prepare('SELECT slug FROM forum_subcategories WHERE id = ? AND is_active = 1 LIMIT 1');
            $legacySubStmt->execute([$legacySubId]);
            $legacySubSlug = trim((string)$legacySubStmt->fetchColumn());
            if ($legacySubSlug !== '') {
                header(HEADER_LOCATION . FORUM_SUBCATEGORY_ROUTE . rawurlencode($legacySubSlug), true, 301);
                exit;
            }
        }
        if (isset($_GET['category']) && (int)($_GET['category'] ?? 0) > 0) {
            $legacyCatId = (int)$_GET['category'];
            $legacyCatStmt = $db->prepare('SELECT slug FROM forum_categories WHERE id = ? AND is_active = 1 LIMIT 1');
            $legacyCatStmt->execute([$legacyCatId]);
            $legacyCatSlug = trim((string)$legacyCatStmt->fetchColumn());
            if ($legacyCatSlug !== '') {
                header(HEADER_LOCATION . FORUM_CATEGORY_ROUTE . rawurlencode($legacyCatSlug), true, 301);
                exit;
            }
        }

        // Resolve category/subcategory from slug-based path segments
        if (isset($segments[1], $segments[2]) && (string)$segments[1] === 'subcategory') {
            $subSlugRaw = trim((string)$segments[2]);
            if ($subSlugRaw !== '') {
                // Try slug first, fall back to numeric ID for backward compat
                if (ctype_digit($subSlugRaw)) {
                    $subLookupStmt = $db->prepare('SELECT id, slug FROM forum_subcategories WHERE id = ? AND is_active = 1 LIMIT 1');
                    $subLookupStmt->execute([(int)$subSlugRaw]);
                    $subLookup = $subLookupStmt->fetch(PDO::FETCH_ASSOC);
                    if (is_array($subLookup) && trim((string)($subLookup['slug'] ?? '')) !== '') {
                        // 301 redirect numeric subcategory to slug
                        header(HEADER_LOCATION . FORUM_SUBCATEGORY_ROUTE . rawurlencode(trim((string)$subLookup['slug'])), true, 301);
                        exit;
                    }
                    $selectedSubcategoryId = (int)$subSlugRaw;
                } else {
                    $subSlugStmt = $db->prepare('SELECT id FROM forum_subcategories WHERE slug = ? AND is_active = 1 LIMIT 1');
                    $subSlugStmt->execute([$subSlugRaw]);
                    $selectedSubcategoryId = (int)$subSlugStmt->fetchColumn();
                }
                $hasSubcategorySelection = $selectedSubcategoryId > 0;
            }
        }
        if (!$hasCategorySelection && isset($segments[1], $segments[2]) && (string)$segments[1] === 'category') {
            $catSlugRaw = trim((string)$segments[2]);
            if ($catSlugRaw !== '') {
                if (ctype_digit($catSlugRaw)) {
                    $catLookupStmt = $db->prepare('SELECT id, slug FROM forum_categories WHERE id = ? AND is_active = 1 LIMIT 1');
                    $catLookupStmt->execute([(int)$catSlugRaw]);
                    $catLookup = $catLookupStmt->fetch(PDO::FETCH_ASSOC);
                    if (is_array($catLookup) && trim((string)($catLookup['slug'] ?? '')) !== '') {
                        // 301 redirect numeric category to slug
                        header(HEADER_LOCATION . FORUM_CATEGORY_ROUTE . rawurlencode(trim((string)$catLookup['slug'])), true, 301);
                        exit;
                    }
                    $selectedCategoryId = (int)$catSlugRaw;
                } else {
                    $catSlugStmt = $db->prepare('SELECT id FROM forum_categories WHERE slug = ? AND is_active = 1 LIMIT 1');
                    $catSlugStmt->execute([$catSlugRaw]);
                    $selectedCategoryId = (int)$catSlugStmt->fetchColumn();
                }
                $hasCategorySelection = $selectedCategoryId > 0;
            }
        }

        // Build breadcrumb override only when a specific category/subcategory was requested.
        if ($hasSubcategorySelection) {
            $subStmt = $db->prepare('SELECT id, name, slug, description, category_id, parent_id FROM forum_subcategories WHERE id = ? AND is_active = 1 LIMIT 1');
            $subStmt->execute([$selectedSubcategoryId]);
            $leafSub = $subStmt->fetch(PDO::FETCH_ASSOC);

            if (is_array($leafSub)) {
                $resolvedCategoryId = (int)($leafSub['category_id'] ?? 0);
                $leafSubName = trim((string)($leafSub['name'] ?? ''));
                $leafSubDescription = trim((string)($leafSub['description'] ?? ''));
                $catName = '';
                $catSlug = '';
                if ($resolvedCategoryId > 0) {
                    $catStmt = $db->prepare('SELECT name, slug FROM forum_categories WHERE id = ? AND is_active = 1 LIMIT 1');
                    $catStmt->execute([$resolvedCategoryId]);
                    $catRow = $catStmt->fetch(PDO::FETCH_ASSOC);
                    $catName = trim((string)($catRow['name'] ?? ''));
                    $catSlug = trim((string)($catRow['slug'] ?? ''));
                }

                if ($leafSubName !== '') {
                    $page['title'] = $leafSubName;
                    $page['meta_title'] = $leafSubName . ($catName !== '' ? ' - ' . $catName : '') . FORUM_META_TITLE_SUFFIX;
                    $page['meta_description'] = $leafSubDescription !== ''
                        ? $leafSubDescription
                        : ('Browse forum discussions in ' . $leafSubName . '.');
                }

                $subChain = [];
                $seenSubIds = [];
                $currentSubId = (int)($leafSub['id'] ?? 0);
                $hops = 0;
                while ($currentSubId > 0 && $hops < 12 && !isset($seenSubIds[$currentSubId])) {
                    $seenSubIds[$currentSubId] = true;
                    $chainStmt = $db->prepare('SELECT id, name, slug, parent_id FROM forum_subcategories WHERE id = ? AND is_active = 1 LIMIT 1');
                    $chainStmt->execute([$currentSubId]);
                    $chainRow = $chainStmt->fetch(PDO::FETCH_ASSOC);
                    if (!is_array($chainRow)) {
                        break;
                    }
                    $subChain[] = [
                        'id' => (int)($chainRow['id'] ?? 0),
                        'name' => trim((string)($chainRow['name'] ?? '')),
                        'slug' => trim((string)($chainRow['slug'] ?? '')),
                        'parent_id' => isset($chainRow['parent_id']) && $chainRow['parent_id'] !== null ? (int)$chainRow['parent_id'] : 0,
                    ];
                    $currentSubId = (int)($chainRow['parent_id'] ?? 0);
                    $hops += 1;
                }
                $subChain = array_reverse($subChain);

                $forumCrumbs = [
                    ['label' => 'Forum', 'url' => FORUM_ROUTE, 'is_home' => false],
                ];
                if ($resolvedCategoryId > 0 && $catName !== '' && $catSlug !== '') {
                    $forumCrumbs[] = [
                        'label' => $catName,
                        'url' => FORUM_CATEGORY_ROUTE . rawurlencode($catSlug),
                        'is_home' => false,
                    ];
                }

                $lastIndex = count($subChain) - 1;
                foreach ($subChain as $i => $subNode) {
                    $subNodeId = (int)($subNode['id'] ?? 0);
                    $subNodeName = trim((string)($subNode['name'] ?? ''));
                    $subNodeSlug = trim((string)($subNode['slug'] ?? ''));
                    if ($subNodeId <= 0 || $subNodeName === '') {
                        continue;
                    }
                    $forumCrumbs[] = [
                        'label' => $subNodeName,
                        'url' => $i === $lastIndex ? null : (FORUM_SUBCATEGORY_ROUTE . rawurlencode($subNodeSlug)),
                        'is_home' => false,
                        'is_current' => $i === $lastIndex,
                    ];
                }

                $page['breadcrumbs_override'] = $forumCrumbs;
            }
        } elseif ($hasCategorySelection && $selectedCategoryId > 0) {
            $catStmt = $db->prepare('SELECT name, slug FROM forum_categories WHERE id = ? AND is_active = 1 LIMIT 1');
            $catStmt->execute([$selectedCategoryId]);
            $catRow = $catStmt->fetch(PDO::FETCH_ASSOC);
            $catName = trim((string)($catRow['name'] ?? ''));
            if ($catName !== '') {
                $page['title'] = $catName;
                $page['meta_title'] = $catName . FORUM_META_TITLE_SUFFIX;
                $page['meta_description'] = 'Browse forum discussions in the ' . $catName . ' category.';
                $page['breadcrumbs_override'] = [
                    ['label' => 'Forum', 'url' => FORUM_ROUTE, 'is_home' => false],
                    ['label' => $catName, 'url' => null, 'is_home' => false, 'is_current' => true],
                ];
            }
        }
    }
} elseif ($isProfilePage) {
    $profileUser = UserService::getByNickname($profileNickname);
    if ($profileUser) {
        // Check if blocked
        $viewerId = $isUserLoggedIn ? (int)$_SESSION['site_user_id'] : null;
        if ($viewerId && FriendService::isBlockedEitherWay((int)$viewerId, (int)$profileUser['id'])) {
            $profileUser = null;
        }
    }
    if ($profileUser) {
        $page = [
            'title' => $profileUser['display_name'] ?: ('@' . $profileUser['nickname']),
            'slug' => '@' . $profileUser['nickname'],
            'content' => '',
            'meta_title' => ($profileUser['display_name'] ?: ('@' . $profileUser['nickname'])) . ' - Profile',
            'meta_description' => 'View profile of ' . ($profileUser['display_name'] ?: ('@' . $profileUser['nickname']))
        ];
    } else {
        http_response_code(404);
        $page = [
            'title' => 'User Not Found',
            'slug' => '404',
            'content' => '',
            'meta_title' => 'User Not Found',
            'meta_description' => 'This user does not exist or is unavailable.'
        ];
    }
} elseif ($isLogoutPage) {
    SiteAuth::logout();
    header(HEADER_LOCATION . ROUTE_LOGIN);
    exit;
} elseif ($isGamesPage) {
    $page = [
        'title' => 'Mini-Games',
        'slug' => 'games',
        'content' => '',
        'meta_title' => 'Mini-Games',
        'meta_description' => 'Play free browser mini-games.'
    ];

    // Per-game SEO metadata for /games/<game>
    $gameRouteSlugRaw = $segments[1] ?? '';
    $gameRouteSlug = strtolower(trim((string)$gameRouteSlugRaw));
    $gameRouteSlug = preg_replace(SLUG_SANITIZE_PATTERN, '', $gameRouteSlug);
    if (!is_string($gameRouteSlug)) {
        $gameRouteSlug = '';
    }
    $gameSlug = \NewSite\Minigames\MinigameCatalog::resolveFromRoute($gameRouteSlug);
    $catalog = \NewSite\Minigames\MinigameCatalog::getCatalog();
    $notFoundPage = [
        'title' => NOT_FOUND_PAGE_TITLE,
        'slug' => '404',
        'content' => '',
        'meta_title' => NOT_FOUND_META_TITLE,
        'meta_description' => NOT_FOUND_META_DESCRIPTION
    ];
    if ($gameRouteSlug !== '') {
        // Unknown or disabled slugs should be a true 404 (not the hub).
        if (!isset($catalog[$gameSlug]) || !\NewSite\Minigames\MinigameRepository::isEnabled($db, $gameSlug, true)) {
            http_response_code(404);
            $page = $notFoundPage;
        } else {
            $gameTitle = (string)($catalog[$gameSlug]['title'] ?? $gameSlug);
            $gameDesc = (string)($catalog[$gameSlug]['description'] ?? '');
            $page['meta_title'] = $gameTitle . ' - Mini-Games';
            if ($gameDesc !== '') {
                $page['meta_description'] = 'Play ' . $gameTitle . ' — ' . $gameDesc;
            }
        }
    }
} elseif ($isCheckoutPage) {
    $page = [
        'title' => 'Checkout',
        'slug' => 'checkout',
        'content' => '',
        'meta_title' => 'Checkout',
        'meta_description' => 'Checkout.'
    ];
    $stmt = $db->prepare(SQL_SELECT_PUBLISHED_PAGE_BY_SLUG);
    $stmt->execute(['checkout']);
    $pageRow = $stmt->fetch();
    if ($pageRow) {
        $page['id'] = $pageRow['id'];
        if (!empty($pageRow['title'])) {
            $page['title'] = $pageRow['title'];
        }
        if (!empty($pageRow['meta_title'])) {
            $page['meta_title'] = $pageRow['meta_title'];
        }
        if (!empty($pageRow['meta_description'])) {
            $page['meta_description'] = $pageRow['meta_description'];
        }
    }
} else {
    // If this slug is a disabled legal page, force 404 instead of loading it
    if (isset($legalPageToggles[$pageSlug]) && !$legalPageToggles[$pageSlug]) {
        $page = null; // falls through to the 404 block below
    } else {
        $stmt = $db->prepare(SQL_SELECT_PUBLISHED_PAGE_BY_SLUG);
        $stmt->execute([$pageSlug]);
        $page = $stmt->fetch();
    }
}

if (!$page) {
    // 404 page
    http_response_code(404);
    $page = [
        'title' => NOT_FOUND_PAGE_TITLE,
        'slug' => '404',
        'content' => '',
        'meta_title' => NOT_FOUND_META_TITLE,
        'meta_description' => NOT_FOUND_META_DESCRIPTION
    ];
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && (string)($_POST['action'] ?? '') === 'product_comment_submit') {
    $commentProductId = (int)($_POST['product_id'] ?? 0);

    $buildCommentRedirectTarget = static function (int $productId): string {
        $requestUri = (string)($_SERVER['REQUEST_URI'] ?? '/');
        $parsed = parse_url($requestUri);
        $path = '/';
        $query = '';

        if (is_array($parsed)) {
            $parsedPath = (string)($parsed['path'] ?? '/');
            if ($parsedPath !== '' && str_starts_with($parsedPath, '/')) {
                $path = $parsedPath;
            }
            $parsedQuery = isset($parsed['query']) ? (string)$parsed['query'] : '';
            if ($parsedQuery !== '') {
                $query = '?' . $parsedQuery;
            }
        }

        $target = $path . $query;
        if ($target === '/' && $productId > 0) {
            $target = '/product?product=' . $productId;
        }

        if (strpos($target, '#') === false) {
            $target .= '#product-comments';
        }

        return $target;
    };

    $setCommentFlashAndRedirect = static function (string $type, string $message, int $productId) use ($buildCommentRedirectTarget): void {
        $_SESSION['product_comment_flash'] = [
            'type' => $type,
            'message' => $message,
            'product_id' => $productId
        ];
        header(HEADER_LOCATION . $buildCommentRedirectTarget($productId));
        exit;
    };

    if ($commentProductId <= 0) {
        $setCommentFlashAndRedirect('error', 'Invalid product.', 0);
    }

    $productStmt = $db->prepare("SELECT id, page_id, is_active, allow_comments FROM products WHERE id = ? LIMIT 1");
    $productStmt->execute([$commentProductId]);
    $commentProduct = $productStmt->fetch(PDO::FETCH_ASSOC);

    if (!$commentProduct || (int)($commentProduct['is_active'] ?? 0) !== 1) {
        $setCommentFlashAndRedirect('error', 'Product not found.', $commentProductId);
    }

    $productAllowsComments = (int)($commentProduct['allow_comments'] ?? 1) === 1;
    if (!$productAllowsComments) {
        $setCommentFlashAndRedirect('error', 'Comments are disabled for this product.', $commentProductId);
    }

    $routePageId = isset($page['id']) ? (int)$page['id'] : 0;
    $productPageId = (int)($commentProduct['page_id'] ?? 0);
    if ($routePageId > 0 && $productPageId > 0 && $routePageId !== $productPageId) {
        $setCommentFlashAndRedirect('error', 'Invalid product page.', $commentProductId);
    }

    if (!SiteAuth::isLoggedIn()) {
        $setCommentFlashAndRedirect('error', 'Please log in to post a comment.', $commentProductId);
    }

    $commentBody = trim((string)($_POST['comment_body'] ?? ''));
    if ($commentBody === '') {
        $setCommentFlashAndRedirect('error', 'Comment cannot be empty.', $commentProductId);
    }

    $commentLength = function_exists('mb_strlen') ? mb_strlen($commentBody) : strlen($commentBody);
    if ($commentLength > 2000) {
        $setCommentFlashAndRedirect('error', 'Comment is too long.', $commentProductId);
    }

    $commentUserId = (int)($_SESSION['site_user_id'] ?? 0);
    if ($commentUserId <= 0) {
        $setCommentFlashAndRedirect('error', 'Please log in to post a comment.', $commentProductId);
    }

    $rateStmt = $db->prepare("SELECT created_at FROM product_comments WHERE user_id = ? ORDER BY created_at DESC LIMIT 1");
    $rateStmt->execute([$commentUserId]);
    $lastCommentTimeRaw = $rateStmt->fetchColumn();
    $lastCommentTimestamp = $lastCommentTimeRaw ? strtotime((string)$lastCommentTimeRaw) : 0;
    if ($lastCommentTimestamp > 0 && (time() - $lastCommentTimestamp) < 15) {
        $setCommentFlashAndRedirect('error', 'You are commenting too fast. Please wait a few seconds.', $commentProductId);
    }

    $now = DbHelper::nowString();
    $insertCommentStmt = $db->prepare("INSERT INTO product_comments (product_id, user_id, body, created_at, updated_at) VALUES (?, ?, ?, ?, ?)");
    $insertCommentStmt->execute([$commentProductId, $commentUserId, $commentBody, $now, $now]);

    $setCommentFlashAndRedirect('success', 'Comment posted successfully.', $commentProductId);
}

$requestedProductId = isset($_GET['product']) ? (int)$_GET['product'] : 0;
$isPrettyProductRoute = false;
if ($requestedProductId <= 0 && isset($segments[1])) {
    $productSlugFromPath = SlugGenerator::generate((string)$segments[1]);
    if ($productSlugFromPath !== '') {
        $matchedProductId = 0;
        if (isset($page['id'])) {
            $stmt = $db->prepare("SELECT id FROM products WHERE is_active = 1 AND page_id = ? AND product_slug = ? LIMIT 1");
            $stmt->execute([$page['id'], $productSlugFromPath]);
            $matchedProductId = (int)$stmt->fetchColumn();
        }
        if ($matchedProductId > 0) {
            $requestedProductId = $matchedProductId;
            $_GET['product'] = (string)$matchedProductId;
            $isPrettyProductRoute = true;
        } else {
            $stmt = $db->prepare("SELECT p.id, pg.slug AS page_slug FROM products p LEFT JOIN pages pg ON pg.id = p.page_id WHERE p.is_active = 1 AND p.product_slug = ? LIMIT 1");
            $stmt->execute([$productSlugFromPath]);
            $matchedRow = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($matchedRow) {
                $redirectPageSlug = trim((string)($matchedRow['page_slug'] ?? ''));
                if ($redirectPageSlug === '') {
                    $redirectPageSlug = 'product';
                }
                $redirectTarget = '/' . $redirectPageSlug . '/' . rawurlencode($productSlugFromPath);
                if (($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'GET') {
                    header(HEADER_LOCATION . $redirectTarget, true, 301);
                    exit;
                }
                $requestedProductId = (int)($matchedRow['id'] ?? 0);
                if ($requestedProductId > 0) {
                    $_GET['product'] = (string)$requestedProductId;
                    $isPrettyProductRoute = true;
                }
            }
        }
    }
}

if ($requestedProductId > 0 && isset($page['id'])) {
    $stmt = $db->prepare("SELECT product_slug FROM products WHERE is_active = 1 AND id = ? AND page_id = ? LIMIT 1");
    $stmt->execute([$requestedProductId, $page['id']]);
    $resolvedProductSlug = trim((string)$stmt->fetchColumn());

    if ($resolvedProductSlug !== '') {
        $currentPath = trim((string)(parse_url((string)($_SERVER['REQUEST_URI'] ?? ''), PHP_URL_PATH) ?? ''), '/');
        $expectedPath = trim((string)($page['slug'] ?? $pageSlug), '/') . '/' . $resolvedProductSlug;
        $isPrettyProductRoute = $currentPath === $expectedPath;

        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'GET' && !$isPrettyProductRoute) {
            header(HEADER_LOCATION . '/' . $expectedPath, true, 301);
            exit;
        }
    }
}

if ($requestedProductId > 0 && isset($page['id'])) {
    $stmt = $db->prepare("SELECT name, description, meta_title, meta_description FROM products WHERE id = ? AND is_active = 1 AND page_id = ? LIMIT 1");
    $stmt->execute([$requestedProductId, $page['id']]);
    $productMetaRow = $stmt->fetch();
    if ($productMetaRow) {
        $seoTitle = trim((string)($productMetaRow['meta_title'] ?? ''));
        if ($seoTitle === '') {
            $seoTitle = trim((string)($productMetaRow['name'] ?? ''));
        }
        if ($seoTitle !== '') {
            $page['meta_title'] = $seoTitle;
        }

        $seoDescription = trim((string)($productMetaRow['meta_description'] ?? ''));
        if ($seoDescription === '') {
            $seoDescription = trim((string)($productMetaRow['description'] ?? ''));
        }
        if ($seoDescription !== '') {
            $seoDescription = preg_replace('/\s+/', ' ', strip_tags($seoDescription));
            if (function_exists('mb_substr')) {
                $seoDescription = mb_substr($seoDescription, 0, 180);
            } else {
                $seoDescription = substr($seoDescription, 0, 180);
            }
            $page['meta_description'] = $seoDescription;
        }
    }
}

$requestedCollectionId = isset($_GET['collection']) ? (int)$_GET['collection'] : 0;
$isPrettyCollectionRoute = false;
if ($requestedProductId <= 0 && $requestedCollectionId <= 0 && isset($segments[1], $segments[2]) && (string)$segments[1] === 'collection') {
    $collectionSlugFromPath = SlugGenerator::generate((string)$segments[2]);
    if ($collectionSlugFromPath !== '') {
        $matchedCollectionId = 0;
        if (isset($page['id'])) {
            $stmt = $db->prepare("SELECT id FROM collections WHERE collection_slug = ? AND (target_page_id = ? OR target_page_id IS NULL) LIMIT 1");
            $stmt->execute([$collectionSlugFromPath, $page['id']]);
            $matchedCollectionId = (int)$stmt->fetchColumn();
        }

        if ($matchedCollectionId > 0) {
            $requestedCollectionId = $matchedCollectionId;
            $_GET['collection'] = (string)$matchedCollectionId;
            $isPrettyCollectionRoute = true;
        } else {
            $stmt = $db->prepare("SELECT c.id, pg.slug AS target_slug FROM collections c LEFT JOIN pages pg ON pg.id = c.target_page_id AND pg.is_published = 1 WHERE c.collection_slug = ? LIMIT 1");
            $stmt->execute([$collectionSlugFromPath]);
            $matchedCollectionRow = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($matchedCollectionRow) {
                $redirectPageSlug = trim((string)($matchedCollectionRow['target_slug'] ?? ''));
                if ($redirectPageSlug === '') {
                    $fallbackStmt = $db->query(SQL_SELECT_PRODUCTS_LIST_PAGE_SLUG);
                    $redirectPageSlug = $fallbackStmt ? trim((string)$fallbackStmt->fetchColumn()) : '';
                }
                if ($redirectPageSlug !== '') {
                    $redirectTarget = '/' . $redirectPageSlug . COLLECTION_ROUTE_SEGMENT . rawurlencode($collectionSlugFromPath);
                    if (($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'GET') {
                        header(HEADER_LOCATION . $redirectTarget, true, 301);
                        exit;
                    }
                }

                $requestedCollectionId = (int)($matchedCollectionRow['id'] ?? 0);
                if ($requestedCollectionId > 0) {
                    $_GET['collection'] = (string)$requestedCollectionId;
                    $isPrettyCollectionRoute = true;
                }
            }
        }
    }
}

if ($requestedProductId <= 0 && $requestedCollectionId > 0) {
    $stmt = $db->prepare("SELECT c.collection_slug, c.target_page_id, pg.slug AS target_slug FROM collections c LEFT JOIN pages pg ON pg.id = c.target_page_id AND pg.is_published = 1 WHERE c.id = ? LIMIT 1");
    $stmt->execute([$requestedCollectionId]);
    $resolvedCollectionRow = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($resolvedCollectionRow) {
        $resolvedCollectionSlug = trim((string)($resolvedCollectionRow['collection_slug'] ?? ''));
        $resolvedTargetSlug = trim((string)($resolvedCollectionRow['target_slug'] ?? ''));
        if ($resolvedTargetSlug === '') {
            $fallbackStmt = $db->query(SQL_SELECT_PRODUCTS_LIST_PAGE_SLUG);
            $resolvedTargetSlug = $fallbackStmt ? trim((string)$fallbackStmt->fetchColumn()) : '';
        }

        if ($resolvedCollectionSlug !== '' && $resolvedTargetSlug !== '') {
            $currentPath = trim((string)(parse_url((string)($_SERVER['REQUEST_URI'] ?? ''), PHP_URL_PATH) ?? ''), '/');
            $expectedPath = trim($resolvedTargetSlug, '/') . COLLECTION_ROUTE_SEGMENT . $resolvedCollectionSlug;
            $isPrettyCollectionRoute = $currentPath === $expectedPath;

            if (($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'GET' && !$isPrettyCollectionRoute) {
                header(HEADER_LOCATION . '/' . $expectedPath, true, 301);
                exit;
            }
        }
    }
}

if ($requestedProductId <= 0 && $requestedCollectionId > 0 && isset($page['id'])) {
    $stmt = $db->prepare(SQL_HAS_ACTIVE_PRODUCTS_LIST_SECTION);
    $stmt->execute([$page['id']]);
    $hasProductsList = (bool)$stmt->fetchColumn();
    if ($hasProductsList) {
        $stmt = $db->prepare("SELECT title, description, meta_title, meta_description FROM collections WHERE id = ? LIMIT 1");
        $stmt->execute([$requestedCollectionId]);
        $collectionMetaRow = $stmt->fetch();
        if ($collectionMetaRow) {
            $collectionTitle = trim((string)($collectionMetaRow['title'] ?? ''));

            $collectionSeoTitle = trim((string)($collectionMetaRow['meta_title'] ?? ''));
            if ($collectionSeoTitle === '') {
                $collectionSeoTitle = $collectionTitle;
            }
            if ($collectionSeoTitle !== '') {
                $page['meta_title'] = $collectionSeoTitle;
            }

            $collectionSeoDescription = trim((string)($collectionMetaRow['meta_description'] ?? ''));
            if ($collectionSeoDescription === '') {
                $collectionSeoDescription = trim((string)($collectionMetaRow['description'] ?? ''));
            }
            if ($collectionSeoDescription === '' && $collectionTitle !== '') {
                $collectionSeoDescription = 'Browse products in ' . $collectionTitle . '.';
            }

            if ($collectionSeoDescription !== '') {
                $collectionSeoDescription = preg_replace('/\s+/', ' ', strip_tags($collectionSeoDescription));
                if (function_exists('mb_substr')) {
                    $collectionSeoDescription = mb_substr($collectionSeoDescription, 0, 180);
                } else {
                    $collectionSeoDescription = substr($collectionSeoDescription, 0, 180);
                }
                $page['meta_description'] = $collectionSeoDescription;
            }
        }
    }
}

if (!function_exists('isSeoImagePath')) {
    function isSeoImagePath(string $url): bool
    {
        $url = strtolower(trim($url));
        if ($url === '') {
            return false;
        }
        if (preg_match('~^https?://(www\.)?(youtube\.com|youtu\.be)/~', $url)) {
            return false;
        }
        return (bool)preg_match('/\.(jpg|jpeg|png|gif|webp|avif|svg)(\?.*)?$/', $url);
    }
}

$canonicalQuery = '';
$seoImagePath = '';

if ($requestedProductId > 0 && isset($page['id'])) {
    if (!$isPrettyProductRoute) {
        $canonicalQuery = 'product=' . $requestedProductId;
    }
    $stmt = $db->prepare("SELECT pm.media_url AS gallery_media, p.media_url AS fallback_media FROM products p LEFT JOIN product_media pm ON pm.product_id = p.id AND pm.sort_order = 0 WHERE p.id = ? AND p.page_id = ? LIMIT 1");
    $stmt->execute([$requestedProductId, $page['id']]);
    $mediaRow = $stmt->fetch();
    if ($mediaRow) {
        $candidate = trim((string)($mediaRow['gallery_media'] ?? ''));
        if ($candidate === '') {
            $candidate = trim((string)($mediaRow['fallback_media'] ?? ''));
        }
        if (isSeoImagePath($candidate)) {
            $seoImagePath = $candidate;
        }
    }
} elseif ($requestedCollectionId > 0 && isset($page['id'])) {
    $stmt = $db->prepare(SQL_HAS_ACTIVE_PRODUCTS_LIST_SECTION);
    $stmt->execute([$page['id']]);
    $hasProductsList = (bool)$stmt->fetchColumn();
    if ($hasProductsList) {
        if (!$isPrettyCollectionRoute) {
            $canonicalQuery = 'collection=' . $requestedCollectionId;
        }
        $stmt = $db->prepare("SELECT image_url FROM collections WHERE id = ? LIMIT 1");
        $stmt->execute([$requestedCollectionId]);
        $candidate = trim((string)$stmt->fetchColumn());
        if (isSeoImagePath($candidate)) {
            $seoImagePath = $candidate;
        }
    }
} elseif ($isForumPostPage && $forumPostId > 0) {
    // Resolve first attached image for forum post OG/Twitter cards
    $stmt = $db->prepare("SELECT image_path FROM forum_post_images WHERE post_id = ? ORDER BY sort_order ASC, id ASC LIMIT 1");
    $stmt->execute([$forumPostId]);
    $forumImageCandidate = trim((string)$stmt->fetchColumn());
    if ($forumImageCandidate !== '' && isSeoImagePath($forumImageCandidate)) {
        $seoImagePath = $forumImageCandidate;
    }
} elseif ($isForumPage && !$isForumPostPage) {
    // Forum listing canonical: slug-based paths are already in $route, no query params needed.
    // (Legacy ?category=X / ?subcategory=X URLs are 301-redirected above.)
}

// Get theme settings
$theme = SettingsService::getAllTheme();

/**
 * Render all section templates for the current page.
 *
 * Deduplicates the section-foreach block that was repeated 4× in the page
 * template dispatcher. Relies on SQL_UPDATE_SECTION_SETTINGS from the outer scope.
 *
 * $context carries outer-scope variables that section templates depend on
 * (e.g. $page, $pageSlug, $productCommentFlash) — required because
 * get_defined_vars() inside this function only captures local scope.
 *
 * @param array<int, array<string, mixed>> $sections  Row set from the sections table
 * @param PDO                              $db        Active database connection
 * @param array<string, mixed>             $context   Outer-scope variables for templates
 */
function renderPageSections(array $sections, PDO $db, array $context = []): void
{
    foreach ($sections as $section) {
        $sectionSettings = json_decode($section['settings'], true) ?? [];
        $changed = false;
        $sectionSettings = is_array($sectionSettings) ? MissingUploadsService::stripFromSettings($sectionSettings, $changed) : $sectionSettings;
        if ($changed) {
            $stmt = $db->prepare(SQL_UPDATE_SECTION_SETTINGS);
            $stmt->execute([json_encode($sectionSettings), $section['id']]);
        }
        $sectionType = $section['section_type'];
        if ($sectionType === 'products') {
            $sectionType = 'product';
        }
        // Merge outer-scope context with local vars so templates see both.
        // Local vars take precedence (EXTR_SKIP in TemplateRenderer).
        $templateVars = array_merge($context, get_defined_vars());
        TemplateRenderer::render(
            __DIR__ . '/templates/sections/' . $sectionType . '.php',
            $templateVars
        );
    }
}

// Helper to convert admin file paths to public site file paths
function toPublicFilePath($path)
{
    if (empty($path)) {
        return $path;
    }
    return str_replace('/admin-file/', '/site-file/', $path);
}

// Helper to get the rounded display height for the logo (px)
function getLogoDisplayHeight($height)
{
    $height = (int)$height;
    $validHeights = [30, 40, 50, 60, 70, 80, 90, 100];
    $closest = 40;
    $minDiff = PHP_INT_MAX;
    foreach ($validHeights as $vh) {
        $diff = abs($vh - $height);
        if ($diff < $minDiff) {
            $minDiff = $diff;
            $closest = $vh;
        }
    }
    return $closest;
}

// Helper to get logo height CSS class
function getLogoHeightClass($height)
{
    return 'logo-height-' . getLogoDisplayHeight($height);
}

/**
 * Render an optimized logo using <picture> with WebP source and sized fallback.
 *
 * Outputs a <picture> element with:
 *  - WebP source at 1x and 2x DPR
 *  - Resized original-format fallback at 2x DPR
 *
 * Falls back to a plain <img> if the logo path is empty.
 *
 * @param array  $theme          Theme settings array (needs site_logo, site_name, logo_height)
 * @param string $logoHeightClass CSS class like 'logo-height-40'
 * @param string $extraClasses    Additional CSS classes (e.g. 'footer-logo')
 */
function renderOptimizedLogo(array $theme, string $logoHeightClass, string $extraClasses = ''): void
{
    $logoUrl       = toPublicFilePath($theme['site_logo']);
    $displayHeight = getLogoDisplayHeight((int)($theme['logo_height'] ?? 40));
    $retina        = $displayHeight * 2;
    $altText       = e($theme['site_name'] ?? 'Logo');
    $classes       = trim('logo-image ' . $logoHeightClass . ($extraClasses !== '' ? ' ' . $extraClasses : ''));

    // Compute intrinsic width from source image aspect ratio to prevent CLS
    $displayWidth = $displayHeight; // default: assume square
    $rawLogo = $theme['site_logo'] ?? '';
    if ($rawLogo !== '') {
        $diskPath = str_replace(ADMIN_FILE_PATH_SEGMENT, '', $rawLogo);
        $diskPath = str_replace(SITE_FILE_PATH_SEGMENT, '', $diskPath);
        $absPath  = dirname(__DIR__) . DATA_ADMIN_UPLOADS_SEGMENT . $diskPath;
        if (is_file($absPath)) {
            $dims = @getimagesize($absPath);
            if ($dims !== false && $dims[1] > 0) {
                $displayWidth = (int) round($dims[0] * $displayHeight / $dims[1]);
            }
        }
    }

    // WebP source with 1x and 2x density descriptors
    $webpSuffix = '&format=webp';
    $webp1x = e($logoUrl . '?h=' . $displayHeight . $webpSuffix);
    $webp2x = e($logoUrl . '?h=' . $retina . $webpSuffix);

    // Resized fallback in original format (for browsers without WebP)
    $fallback = e($logoUrl . '?h=' . $retina);

    echo '<picture>'
        . '<source srcset="' . $webp1x . ' 1x, ' . $webp2x . ' 2x" type="image/webp">'
        . '<img src="' . $fallback . '" alt="' . $altText
        . '" class="' . e($classes) . '"'
        . ' width="' . $displayWidth . '" height="' . $displayHeight . '">'
        . '</picture>';
}

/**
 * Render an optimized site image using <picture> with WebP srcset.
 *
 * Uses width descriptors in srcset for responsive image selection.
 * The browser picks the best size based on the `sizes` attribute.
 *
 * @param string $src       Original image URL (e.g. /site-file/xxx.png)
 * @param string $alt       Alt text for the image
 * @param int[]  $widths    Array of widths for srcset (e.g. [400, 800, 1440])
 * @param string $sizes     CSS sizes attribute (e.g. '100vw' or '(max-width:768px) 100vw, 290px')
 * @param string $classes   CSS classes for the img element
 * @param string $extraAttr Additional HTML attributes (e.g. 'loading="lazy" decoding="async"')
 */
function renderSiteImage(
    string $src,
    string $alt,
    array $widths,
    string $sizes,
    string $classes = '',
    string $extraAttr = ''
): void {
    $escapedAlt = e($alt);
    $classAttr  = $classes !== '' ? ' class="' . e($classes) . '"' : '';
    $attrStr    = $extraAttr !== '' ? ' ' . $extraAttr : '';

    // Build WebP srcset with width descriptors
    $webpParts = [];
    foreach ($widths as $w) {
        $webpParts[] = e($src . '?w=' . (int) $w . '&format=webp') . ' ' . (int) $w . 'w';
    }
    $webpSrcset = implode(', ', $webpParts);

    // Fallback: use the largest width, keep original format
    $largestWidth = max($widths);
    $fallbackSrc  = e($src . '?w=' . (int) $largestWidth);

    echo '<picture>'
        . '<source srcset="' . $webpSrcset . '" sizes="' . e($sizes) . '" type="image/webp">'
        . '<img src="' . $fallbackSrc . '" alt="' . $escapedAlt . '"'
        . $classAttr . ' sizes="' . e($sizes) . '"' . $attrStr . '>'
        . '</picture>';
}
$logoHeightClass = getLogoHeightClass($theme['logo_height'] ?? 40);

// Auto-clean missing uploads in theme settings
foreach ($theme as $key => $value) {
    if ((is_string($value) && $value !== '') ? MissingUploadsService::isMissingPath($value) : false) {
        SettingsService::setTheme($key, '');
        $theme[$key] = '';
    }
}

// Get header menu (use theme setting if set, otherwise fall back to location)
$headerMenu = [];
$headerMenuId = !empty($theme['header_menu_id']) ? (int)$theme['header_menu_id'] : null;

if (!$headerMenuId) {
    // Fall back to location-based lookup
    $stmt = $db->prepare("SELECT id FROM menus WHERE location = 'header' LIMIT 1");
    $stmt->execute();
    $menuRow = $stmt->fetch();
    if ($menuRow) {
        $headerMenuId = $menuRow['id'];
    }
}

if ($headerMenuId) {
    $headerMenu = MenuService::getItemsRecursive($db, $headerMenuId);
}

// Get topbar menu (use theme setting if set, otherwise fall back to location)
$topbarMenu = [];
$topbarMenuId = !empty($theme['topbar_menu_id']) ? (int)$theme['topbar_menu_id'] : null;

if (!$topbarMenuId) {
    $stmt = $db->prepare("SELECT id FROM menus WHERE location = 'topbar' LIMIT 1");
    $stmt->execute();
    $menuRow = $stmt->fetch();
    $topbarMenuId = $menuRow ? (int)$menuRow['id'] : null;
}

if ($topbarMenuId) {
    $topbarMenu = MenuService::getItemsRecursive($db, $topbarMenuId);
}

// Get footer menu (use theme setting if set, otherwise fall back to location)
$footerMenu = [];
$footerMenuId = !empty($theme['footer_menu_id']) ? (int)$theme['footer_menu_id'] : null;

if (!$footerMenuId) {
    // Fall back to location-based lookup
    $stmt = $db->prepare("SELECT id FROM menus WHERE location = 'footer' LIMIT 1");
    $stmt->execute();
    $menuRow = $stmt->fetch();
    if ($menuRow) {
        $footerMenuId = $menuRow['id'];
    }
}

if ($footerMenuId) {
    $footerMenu = MenuService::getItemsRecursive($db, $footerMenuId);
}

// Get page sections
$sections = [];
if (isset($page['id'])) {
    $stmt = $db->prepare("SELECT * FROM sections WHERE page_id = ? AND is_active = 1 ORDER BY sort_order");
    $stmt->execute([$page['id']]);
    $sections = $stmt->fetchAll();
}

// Ensure system Collections page always has a collections_list section
if (isset($page['id']) && $pageSlug === 'collections') {
    $hasCollectionsSection = false;
    foreach ($sections as $sectionRow) {
        if (($sectionRow['section_type'] ?? '') === 'collections_list') {
            $hasCollectionsSection = true;
            break;
        }
    }
    if (!$hasCollectionsSection) {
        $defaultSectionSettings = [
            'section_title' => 'Collections',
            'title_align' => 'center',
            'columns' => '3',
            'show_description' => true,
            'show_image' => true,
            'layout_mode' => 'grid'
        ];
        $orderStmt = $db->prepare("SELECT MAX(sort_order) FROM sections WHERE page_id = ?");
        $orderStmt->execute([$page['id']]);
        $maxOrder = (int)$orderStmt->fetchColumn();
        $insertSection = $db->prepare("INSERT INTO sections (page_id, section_type, settings, sort_order) VALUES (?, ?, ?, ?)");
        $insertSection->execute([$page['id'], 'collections_list', json_encode($defaultSectionSettings), $maxOrder + 1]);

        $stmt = $db->prepare("SELECT * FROM sections WHERE page_id = ? AND is_active = 1 ORDER BY sort_order");
        $stmt->execute([$page['id']]);
        $sections = $stmt->fetchAll();
    }
}

// Helper function to get menu item URL
function getMenuItemUrl($item)
{
    if (!empty($item['page_slug'])) {
        return '/' . $item['page_slug'];
    }
    return $item['url'] ?: '#';
}

/**
 * Resolve mini-game breadcrumbs for /games/<game> routes.
 *
 * @return array|null Array of crumbs (Games parent + game title), or null if not a game route
 */
function resolveGameBreadcrumbs(array|null $page, string $pageSlug, array $segments, ?PDO $db): ?array
{
    if ($pageSlug !== 'games' || empty($segments[1])) {
        return null;
    }

    $gameRouteSlug = strtolower(trim((string)$segments[1]));
    $gameRouteSlug = preg_replace('/[^a-z0-9\-]/', '', $gameRouteSlug);
    $gameSlug = is_string($gameRouteSlug)
        ? \NewSite\Minigames\MinigameCatalog::resolveFromRoute($gameRouteSlug)
        : '';
    $catalog = \NewSite\Minigames\MinigameCatalog::getCatalog();
    $isEnabled = $db instanceof PDO
        ? \NewSite\Minigames\MinigameRepository::isEnabled($db, $gameSlug, true)
        : true;

    if ($gameSlug === '' || !isset($catalog[$gameSlug]) || !$isEnabled) {
        return null;
    }

    return [
        ['label' => $page['title'] ?? 'Mini-Games', 'url' => '/games', 'is_home' => false],
        ['label' => (string)($catalog[$gameSlug]['title'] ?? $gameSlug), 'url' => null, 'is_home' => false, 'is_current' => true],
    ];
}

/**
 * Resolve the collection breadcrumb crumb and override the page title
 * when a product or collection filter is active.
 *
 * @return array{crumbs: array, title: string|null} Collection crumbs and optional title override
 */
function resolveCollectionBreadcrumbs(array $page, PDO $db, int $productId, int $collectionId): array
{
    $crumbs = [];
    $titleOverride = null;

    try {
        $titleOverride = resolveCollectionTitleOverride($db, (int)$page['id'], $productId, $collectionId);
        $collection = resolveCollectionRecord($db, (int)$page['id'], $productId);
        if ($collection) {
            $crumbs[] = buildCollectionCrumb($collection, $db);
        }
        $productName = resolveProductName($db, (int)$page['id'], $productId);
        if ($productName !== null) {
            $titleOverride = $productName;
        }
    } catch (\Exception) {
        // Silently fail if collections table structure differs
    }

    return ['crumbs' => $crumbs, 'title' => $titleOverride];
}

/**
 * If viewing a collection (no specific product), try to override the page title with the collection name.
 */
function resolveCollectionTitleOverride(PDO $db, int $pageId, int $productId, int $collectionId): ?string
{
    if ($productId > 0 || $collectionId <= 0) {
        return null;
    }

    $stmt = $db->prepare(SQL_HAS_ACTIVE_PRODUCTS_LIST_SECTION);
    $stmt->execute([$pageId]);
    if (!(bool)$stmt->fetchColumn()) {
        return null;
    }

    $stmt = $db->prepare("SELECT title FROM collections WHERE id = ? LIMIT 1");
    $stmt->execute([$collectionId]);
    $title = trim((string)$stmt->fetchColumn());

    return $title !== '' ? $title : null;
}

/**
 * Find the collection record for this page, preferring a direct product match.
 */
function resolveCollectionRecord(PDO $db, int $pageId, int $productId): array|false
{
    if ($productId > 0) {
        $stmt = $db->prepare(
            "SELECT c.id, c.title, c.collection_slug, c.target_page_id, pg.slug AS target_slug
             FROM products p
             INNER JOIN collections c ON p.collection_id = c.id
             LEFT JOIN pages pg ON pg.id = c.target_page_id AND pg.is_published = 1
             WHERE p.id = ? AND p.page_id = ?
             LIMIT 1"
        );
        $stmt->execute([$productId, $pageId]);
        $collection = $stmt->fetch();
        if ($collection) {
            return $collection;
        }
    }

    $stmt = $db->prepare(
        "SELECT DISTINCT c.id, c.title, c.collection_slug, c.target_page_id, pg.slug AS target_slug
         FROM collections c
         INNER JOIN products p ON p.collection_id = c.id
         LEFT JOIN pages pg ON pg.id = c.target_page_id AND pg.is_published = 1
         WHERE p.page_id = ?
         LIMIT 1"
    );
    $stmt->execute([$pageId]);

    return $stmt->fetch() ?: false;
}

/**
 * Build a breadcrumb crumb array for a collection, resolving URL from target page slug.
 */
function buildCollectionCrumb(array $collection, PDO $db): array
{
    $targetSlug = trim((string)($collection['target_slug'] ?? ''));
    if ($targetSlug === '') {
        $fallbackStmt = $db->query(
            "SELECT p.slug FROM pages p JOIN sections s ON s.page_id = p.id
             WHERE s.section_type = 'products_list' AND p.is_published = 1
             ORDER BY p.id ASC LIMIT 1"
        );
        $targetSlug = $fallbackStmt ? trim((string)$fallbackStmt->fetchColumn()) : '';
    }

    $collectionSlug = trim((string)($collection['collection_slug'] ?? ''));
    $collectionUrl = null;
    if ($targetSlug !== '' && $collectionSlug !== '') {
        $collectionUrl = '/' . $targetSlug . '/collection/' . rawurlencode($collectionSlug);
    } elseif ($targetSlug !== '') {
        $collectionUrl = '/' . $targetSlug . '?collection=' . (int)$collection['id'];
    }

    return ['label' => $collection['title'], 'url' => $collectionUrl, 'is_home' => false];
}

/**
 * Look up the product name to use as the final breadcrumb label.
 */
function resolveProductName(PDO $db, int $pageId, int $productId): ?string
{
    if ($productId <= 0) {
        return null;
    }

    $stmt = $db->prepare("SELECT name FROM products WHERE id = ? AND page_id = ? LIMIT 1");
    $stmt->execute([$productId, $pageId]);
    $name = trim((string)$stmt->fetchColumn());

    return $name !== '' ? $name : null;
}

/**
 * Collect route-level breadcrumb overrides set by the page handler.
 *
 * @return array|null Override crumbs, or null if no override is set
 */
function resolveOverrideBreadcrumbs($page): ?array
{
    if (!is_array($page) || !isset($page['breadcrumbs_override']) || !is_array($page['breadcrumbs_override'])) {
        return null;
    }

    $crumbs = [];
    foreach ($page['breadcrumbs_override'] as $overrideCrumb) {
        if (is_array($overrideCrumb)) {
            $crumbs[] = $overrideCrumb;
        }
    }

    return $crumbs;
}

// Breadcrumbs generation
function generateBreadcrumbs($page, $pageSlug, $themeSettings, $db = null, $segments = [])
{
    $homeLabel = $themeSettings['breadcrumbs_home_label'] ?? 'Home';
    $showHomeSetting = $themeSettings['breadcrumbs_show_home'] ?? '1';
    $showHome = $showHomeSetting === '1' || $showHomeSetting === true;
    $safeSegments = is_array($segments) ? $segments : [];
    $requestedProductId = isset($_GET['product']) ? (int)$_GET['product'] : 0;
    $requestedCollectionId = isset($_GET['collection']) ? (int)$_GET['collection'] : 0;

    // On home page, just show home breadcrumb
    if ($pageSlug === 'home') {
        return $showHome
            ? [['label' => $homeLabel, 'url' => null, 'is_home' => true, 'is_current' => true]]
            : [];
    }

    $breadcrumbs = [];
    if ($showHome) {
        $breadcrumbs[] = ['label' => $homeLabel, 'url' => '/', 'is_home' => true];
    }

    // Try override or game-specific crumbs
    $specialCrumbs = resolveOverrideBreadcrumbs($page)
        ?? resolveGameBreadcrumbs($page, $pageSlug, $safeSegments, $db);
    if ($specialCrumbs !== null) {
        return array_merge($breadcrumbs, $specialCrumbs);
    }

    // Default: resolve collection/product context from DB
    $currentTitle = $page['title'] ?? ucfirst($pageSlug);
    if ($db instanceof PDO && is_array($page) && isset($page['id'])) {
        $result = resolveCollectionBreadcrumbs($page, $db, $requestedProductId, $requestedCollectionId);
        $breadcrumbs = array_merge($breadcrumbs, $result['crumbs']);
        if ($result['title'] !== null) {
            $currentTitle = $result['title'];
        }
    }

    // Add current page (no link)
    $breadcrumbs[] = ['label' => $currentTitle, 'url' => null, 'is_home' => false, 'is_current' => true];

    return $breadcrumbs;
}

// Get breadcrumbs settings
$breadcrumbsEnabled = ($theme['breadcrumbs_enabled'] ?? '0') === '1' || ($theme['breadcrumbs_enabled'] ?? false) === true;
$breadcrumbsSeparator = $theme['breadcrumbs_separator'] ?? 'chevron';
$breadcrumbs = ($breadcrumbsEnabled && $page) ? generateBreadcrumbs($page, $pageSlug, $theme, $db, $segments) : [];

$currencyOptions = CurrencyData::getMap();
$currentCurrency = CurrencyService::getCurrent();
$currentLanguage = SetupService::getCurrentLanguage();
$languageOptions = [
    'en' => 'English'
];

/**
 * Recursively render menu items with nested submenus
 */
function renderNestedMenu($items, $submenuClass = 'submenu', $showIcons = true, $maxDepth = 5, $currentDepth = 0, $itemClass = 'nav-item')
{
    if ($currentDepth >= $maxDepth || empty($items)) {
        return;
    }

    foreach ($items as $item):
        $hasChildren = !empty($item['children']);
        $liClass = $itemClass . ($hasChildren ? ' has-submenu' : '');
        ?>
    <?php echo '<li class="' . e($liClass) . '">'; ?>
        <a href="<?php echo e(getMenuItemUrl($item)); ?>" target="<?php echo e($item['target']); ?>" title="<?php echo e($item['title']); ?>">
            <?php if ($showIcons && !empty($item['icon'])): ?>
            <span class="nav-icon"><?php echo e($item['icon']); ?></span>
            <?php elseif ($showIcons): ?>
            <span class="nav-icon">•</span>
            <?php endif; ?>
            <span class="nav-text"><?php echo e($item['title']); ?></span>
            <?php if ($hasChildren): ?>
            <span class="submenu-arrow">›</span>
            <?php endif; ?>
        </a>
        <?php if ($hasChildren): ?>
        <ul class="<?php echo e($submenuClass); ?> submenu-level-<?php echo $currentDepth + 1; ?>">
            <?php renderNestedMenu($item['children'], $submenuClass, $showIcons, $maxDepth, $currentDepth + 1, $itemClass); ?>
        </ul>
        <?php endif; ?>
    </li>
<?php
    endforeach;
}

/**
 * Render an optional menu icon span for the mobile drawer.
 */
function renderDrawerIcon(array $item, bool $showIcons): void
{
    if ($showIcons && !empty($item['icon'])): ?>
        <span class="mobile-drawer-link-icon" aria-hidden="true"><?php echo e($item['icon']); ?></span>
    <?php endif;
}

/**
 * Recursively render menu items for the mobile drawer.
 * Uses <details>/<summary> for collapsible submenus (pure HTML, no JS needed).
 * @param array  $items       Menu items with optional 'children' arrays
 * @param bool   $showIcons   Whether to render the item icon
 * @param int    $maxDepth    Maximum nesting depth (safety guard)
 * @param int    $currentDepth Current recursion depth
 */
function renderDrawerMenu(array $items, bool $showIcons = false, int $maxDepth = 5, int $currentDepth = 0): void
{
    if ($currentDepth >= $maxDepth || empty($items)) {
        return;
    }

    foreach ($items as $item):
        $hasChildren = !empty($item['children']);
        $depthClass  = $currentDepth > 0 ? ' mobile-drawer-submenu-item' : '';
        if ($hasChildren): ?>
            <details class="mobile-drawer-details<?php echo e($depthClass); ?>">
                <summary class="mobile-drawer-link mobile-drawer-parent">
                    <?php renderDrawerIcon($item, $showIcons); ?>
                    <?php echo e($item['title']); ?>
                    <svg class="mobile-drawer-chevron" viewBox="0 0 24 24" aria-hidden="true"><polyline points="6 9 12 15 18 9"></polyline></svg>
                </summary>
                <div class="mobile-drawer-submenu">
                    <a href="<?php echo e(getMenuItemUrl($item)); ?>" target="<?php echo e($item['target']); ?>" class="mobile-drawer-link mobile-drawer-submenu-item">
                        <?php echo e($item['title']); ?>
                    </a>
                    <?php renderDrawerMenu($item['children'], $showIcons, $maxDepth, $currentDepth + 1); ?>
                </div>
            </details>
        <?php else: ?>
            <a href="<?php echo e(getMenuItemUrl($item)); ?>" target="<?php echo e($item['target']); ?>" class="mobile-drawer-link<?php echo e($depthClass); ?>">
                <?php renderDrawerIcon($item, $showIcons); ?>
                <?php echo e($item['title']); ?>
            </a>
        <?php endif;
    endforeach;
}

// Only load easy-media.js when the page contains the Easy Media widget.
// The widget div (#easyMediaApp) only appears in custom CMS page content;
// built-in templates and section-based pages never include it.
$needsEasyMediaJs = is_string($page['content'] ?? null)
    && strpos($page['content'], 'easyMediaApp') !== false;

// Defer easy-media data attribute computation until we know the widget is needed
if ($needsEasyMediaJs) {
    $easyMediaAiZipUrl = ContentRenderer::normalizeEasyMediaZipDownloadPath(
        (string)SettingsService::get('easy_media_ai_zip_url', $easyMediaAiZipDefault)
    );
    $easyMediaHelpImages = ContentRenderer::getEasyMediaHelpImagePaths(12);
    $easyMediaHelpImagesJson = json_encode($easyMediaHelpImages, JSON_UNESCAPED_SLASHES);
    if (!is_string($easyMediaHelpImagesJson)) {
        $easyMediaHelpImagesJson = '[]';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="csrf-token" content="<?php echo e(getCsrfToken()); ?>">
    <?php
    // Build page title
    $siteName = trim($theme['site_name'] ?? '');
$pageTitle = trim($page['meta_title'] ?? $page['title'] ?? '');

// For home page, show site name (and tagline if set)
if ($pageSlug === 'home' && $siteName) {
    $fullTitle = $siteName;
    if (!empty($theme['site_tagline'])) {
        $fullTitle .= ' - ' . $theme['site_tagline'];
    }
} else {
    // For other pages: Page Title | Site Name
    if ($pageTitle !== '' && $siteName !== '') {
        $fullTitle = $pageTitle . ' | ' . $siteName;
    } elseif ($pageTitle !== '') {
        $fullTitle = $pageTitle;
    } elseif ($siteName !== '') {
        $fullTitle = $siteName;
    } else {
        $fullTitle = 'Untitled Page';
    }
}

// Meta description: use page's, or fall back to site default
$metaDescription = $page['meta_description'] ?? $theme['site_description'] ?? '';

// Dynamic robots meta (page-level):
// - Private/auth pages: noindex,nofollow (should never appear in search results)
// - Search and filtered forum listings: noindex,follow
// - 404 pages: noindex,nofollow
// - Everything else: index,follow
$metaRobots = 'index,follow';
$isNotFoundPage = ((int)http_response_code() === 404) || (($page['slug'] ?? '') === '404');
$isPrivatePage = $isAccountPage || $isLoginPage || $isCartPage || $isCheckoutPage
    || $isCheckoutSuccessPage || $isFriendsPage || $isMessagesPage || $isAccountSettingsPage;
if ($isNotFoundPage) {
    $metaRobots = 'noindex,nofollow';
} elseif ($isPrivatePage) {
    // Private/auth pages must never be indexed even if linked from public pages
    $metaRobots = 'noindex,nofollow';
} elseif ($isSearchPage) {
    $metaRobots = 'noindex,follow';
} elseif ($isForumPage && !$isForumPostPage) {
    // Filtered forum listing views (category/subcategory) should not be indexed
    $hasForumCategoryFilter = isset($segments[1], $segments[2]) && (string)$segments[1] === 'category' && trim((string)$segments[2]) !== '';
    $hasForumSubcategoryFilter = isset($segments[1], $segments[2]) && (string)$segments[1] === 'subcategory' && trim((string)$segments[2]) !== '';
    if ($hasForumCategoryFilter || $hasForumSubcategoryFilter) {
        $metaRobots = 'noindex,follow';
    }
}

// Optional explicit per-page override if present.
$metaRobotsOverride = trim((string)($page['meta_robots'] ?? ''));
if ($metaRobotsOverride !== '') {
    $sanitizedOverride = strtolower((string)preg_replace('/[^a-z,\-\s]/i', '', $metaRobotsOverride));
    if ($sanitizedOverride !== '') {
        $metaRobots = $sanitizedOverride;
    }
}

// Canonical URL (include product/collection query when relevant)
$canonicalPath = ($pageSlug === 'home') ? '/' : '/' . $route;
$canonicalUrl = SetupService::getBaseUrl() . $canonicalPath;
if ($canonicalQuery !== '') {
    $canonicalUrl .= '?' . $canonicalQuery;
}

$seoImageUrl = '';
if ($seoImagePath !== '') {
    if (preg_match('~^https?://~i', $seoImagePath)) {
        $seoImageUrl = $seoImagePath;
    } else {
        $seoImageUrl = SetupService::getBaseUrl() . toPublicFilePath($seoImagePath);
    }
} elseif (!empty($theme['site_logo'])) {
    $seoImageUrl = SetupService::getBaseUrl() . toPublicFilePath($theme['site_logo']);
}
?>
    <title><?php echo e($fullTitle); ?></title>
    <meta name="description" content="<?php echo e($metaDescription); ?>">
    <meta name="robots" content="<?php echo e($metaRobots); ?>">
    <meta name="googlebot" content="<?php echo e($metaRobots); ?>">
    <link rel="canonical" href="<?php echo e($canonicalUrl); ?>">

    <!-- Open Graph -->
    <meta property="og:type" content="<?php echo $isForumPostPage ? 'article' : 'website'; ?>">
    <meta property="og:title" content="<?php echo e($fullTitle); ?>">
    <meta property="og:description" content="<?php echo e($metaDescription); ?>">
    <meta property="og:url" content="<?php echo e($canonicalUrl); ?>">
    <?php if (!empty($siteName)): ?>
    <meta property="og:site_name" content="<?php echo e($siteName); ?>">
    <?php endif; ?>
    <?php if (!empty($seoImageUrl)): ?>
    <meta property="og:image" content="<?php echo e($seoImageUrl); ?>">
    <?php endif; ?>

    <!-- Twitter Card — use large image when a specific SEO image is available -->
    <meta name="twitter:card" content="<?php echo (!empty($seoImageUrl) && $seoImagePath !== '') ? 'summary_large_image' : 'summary'; ?>">
    <meta name="twitter:title" content="<?php echo e($fullTitle); ?>">
    <meta name="twitter:description" content="<?php echo e($metaDescription); ?>">
    <?php if (!empty($seoImageUrl)): ?>
    <meta name="twitter:image" content="<?php echo e($seoImageUrl); ?>">
    <?php endif; ?>

    <!-- JSON-LD: WebSite structured data (enables sitelinks search box in Google) -->
    <script type="application/ld+json">
    <?php
    $jsonLd = [
        '@context' => 'https://schema.org',
        '@type' => 'WebSite',
        'name' => $siteName !== '' ? $siteName : 'Website',
        'url' => SetupService::getBaseUrl() . '/'
    ];
    // Add SearchAction for sitelinks search box
    $jsonLd['potentialAction'] = [
        '@type' => 'SearchAction',
        'target' => [
            '@type' => 'EntryPoint',
            'urlTemplate' => SetupService::getBaseUrl() . '/search?q={search_term_string}'
        ],
        'query-input' => 'required name=search_term_string'
    ];
    echo json_encode($jsonLd, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    ?>
    </script>

    <?php
    // JSON-LD: DiscussionForumPosting for individual forum posts (rich result in Google)
    if ($isForumPostPage && isset($forumPostData) && is_array($forumPostData)):
        $forumPostJsonLd = [
            '@context' => 'https://schema.org',
            '@type' => 'DiscussionForumPosting',
            'headline' => $forumPostData['title'] ?? FORUM_POST_DEFAULT_TITLE,
            'url' => $canonicalUrl,
            'datePublished' => !empty($forumPostData['created_at'])
                ? gmdate('c', strtotime($forumPostData['created_at']))
                : gmdate('c'),
        ];
        // Author
        $forumAuthorName = trim((string)($forumPostData['nickname'] ?? ''));
        if ($forumAuthorName === '') {
            $forumAuthorName = trim((string)($forumPostData['display_name'] ?? ''));
        }
        if ($forumAuthorName !== '') {
            $forumPostJsonLd['author'] = [
                '@type' => 'Person',
                'name' => $forumAuthorName,
                'url' => SetupService::getBaseUrl() . '/@' . rawurlencode($forumAuthorName)
            ];
        }
        // Post body as text
        $forumBodyText = trim(strip_tags((string)($forumPostData['body'] ?? '')));
        if ($forumBodyText !== '') {
            $forumPostJsonLd['text'] = $forumBodyText;
        }
        // Interaction statistics
        $forumPostJsonLd['interactionStatistic'] = [
            [
                '@type' => 'InteractionCounter',
                'interactionType' => 'https://schema.org/CommentAction',
                'userInteractionCount' => (int)($forumPostData['comment_count'] ?? 0)
            ],
            [
                '@type' => 'InteractionCounter',
                'interactionType' => 'https://schema.org/ViewAction',
                'userInteractionCount' => (int)($forumPostData['views'] ?? 0)
            ]
        ];
        // Image if available
        if (!empty($seoImageUrl)) {
            $forumPostJsonLd['image'] = $seoImageUrl;
        }
    ?>
    <script type="application/ld+json">
    <?php echo json_encode($forumPostJsonLd, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT); ?>
    </script>
    <?php endif; ?>

    <?php
// Favicon — auto-detect MIME type from file extension
if (!empty($theme['site_favicon'])):
    $faviconUrl = toPublicFilePath($theme['site_favicon']);
    $faviconExt = strtolower(pathinfo($theme['site_favicon'], PATHINFO_EXTENSION));
    $faviconMime = match($faviconExt) {
        'svg' => 'image/svg+xml',
        'png' => 'image/png',
        'gif' => 'image/gif',
        'jpg', 'jpeg' => 'image/jpeg',
        'webp' => 'image/webp',
        default => 'image/x-icon',
    };
    ?>
    <link rel="icon" href="<?php echo e($faviconUrl); ?>" type="<?php echo $faviconMime; ?>">
    <link rel="apple-touch-icon" href="<?php echo e($faviconUrl); ?>">
    <?php endif; ?>
    <?php if (file_exists(__DIR__ . '/manifest.json')): ?>
    <link rel="manifest" href="/manifest.json">
    <?php endif; ?>
    <meta name="theme-color" content="#6f5bff">
    <?php
    // Preload custom fonts to break the HTML → CSS → font dependency chain.
    // Without this, fonts are only discovered after theme.php CSS is parsed.
    $preloadFonts = json_decode(SettingsService::get('custom_fonts', '[]'), true);
    if (!is_array($preloadFonts)) {
        $preloadFonts = [];
    }
    $fontMimeMap = ['woff2' => 'font/woff2', 'woff' => 'font/woff', 'ttf' => 'font/ttf', 'otf' => 'font/otf'];
    foreach ($preloadFonts as $preloadFont):
        $pfFile = isset($preloadFont['file']) ? basename($preloadFont['file']) : '';
        if ($pfFile === '') {
            continue;
        }
        $pfExt = strtolower(pathinfo($pfFile, PATHINFO_EXTENSION));
        if (!isset($fontMimeMap[$pfExt])) {
            continue;
        }
        $pfPath = __DIR__ . '/assets/uploads/fonts/' . $pfFile;
        if (!is_file($pfPath)) {
            continue;
        }
        $pfUrl  = '/assets/uploads/fonts/' . e($pfFile);
        $pfMime = $fontMimeMap[$pfExt];
    ?>
    <link rel="preload" href="<?php echo $pfUrl; ?>" as="font" type="<?php echo $pfMime; ?>" crossorigin>
    <?php endforeach; ?>
    <link rel="stylesheet" href="<?php echo AssetVersioning::url('/assets/css/style.css'); ?>">
    <?php if ($shouldLoadAccountCss): ?>
    <link rel="stylesheet" href="<?php echo AssetVersioning::url('/assets/css/account.css'); ?>">
    <?php endif; ?>
    <?php if ($isForumPage): ?>
    <link rel="stylesheet" href="<?php echo AssetVersioning::url('/assets/css/forum.css'); ?>">
    <?php endif; ?>
    <?php if ($isUserLoggedIn): ?>
    <link rel="stylesheet" href="<?php echo AssetVersioning::url('/assets/css/messenger.css'); ?>">
    <?php endif; ?>
    <?php if ($isGamesPage): ?>
    <link rel="stylesheet" href="<?php echo AssetVersioning::url('/assets/css/games.css'); ?>">
    <?php endif; ?>
    <?php if ($isMazeRunnerPage): ?>
    <link rel="stylesheet" href="<?php echo AssetVersioning::url('/assets/css/maze-runner.css'); ?>">
    <?php endif; ?>
    <link rel="stylesheet" href="<?php echo AssetVersioning::url('/assets/css/theme.php?page_id=' . (int)($page['id'] ?? 0)); ?>">
</head>
<body class="header-<?php echo e($theme['header_style'] ?? 'top'); ?>" data-page-id="<?php echo (int)($page['id'] ?? 0); ?>" data-cart-sidebar="<?php echo SettingsService::get('cart_sidebar_enabled', '1') === '1' ? '1' : '0'; ?>" data-cart-icon-mode="<?php echo SettingsService::get('cart_icon_items_only', '0') === '1' ? 'items' : 'always'; ?>" data-minibar-sticky="<?php echo SettingsService::get('minibar_sticky', '0') === '1' ? '1' : '0'; ?>" data-cursor-theme="<?php echo SettingsService::get('custom_cursor_enabled', '1') === '1' ? '1' : '0'; ?>" data-search-mini="<?php echo $searchShowMini ? '1' : '0'; ?>" data-search-top="<?php echo $searchShowTop ? '1' : '0'; ?>" data-search-sidebar="<?php echo $searchShowSidebar ? '1' : '0'; ?>" data-search-footer="<?php echo $searchShowFooter ? '1' : '0'; ?>"<?php if ($needsEasyMediaJs): ?> data-easy-media-ai-zip-url="<?php echo e($easyMediaAiZipUrl); ?>" data-easy-media-help-images="<?php echo e($easyMediaHelpImagesJson); ?>"<?php endif; ?>>
    <?php
    // Setup complete notification (one-time, from session)
    if (isset($_SESSION['setup_success']) && $_SESSION['setup_success'] === true):
        $setupMoved = $_SESSION['setup_moved'] ?? false;
        unset($_SESSION['setup_success']); // Clear so it only shows once
        unset($_SESSION['setup_moved']);
        ?>
    <div class="setup-complete-banner" id="setupCompleteBanner">
        <div class="setup-complete-inner">
            <span class="setup-complete-icon">🎉</span>
            <div class="setup-complete-text">
                <strong>Setup Complete!</strong>
                <span>Your site is now configured and ready to use.<?php if (!$setupMoved): ?> <em class="setup-complete-note">(Please delete /public/setup.php manually)</em><?php endif; ?></span>
            </div>
            <a href="/admin" class="setup-complete-btn">Go to Admin →</a>
            <button type="button" class="setup-complete-close" data-action="dismiss-setup-banner" aria-label="Dismiss">&times;</button>
        </div>
    </div>
    <?php endif; ?>

    <!-- Floating Bubbles Background -->
    <div class="bubbles">
        <div class="bubble"></div>
        <div class="bubble"></div>
        <div class="bubble"></div>
        <div class="bubble"></div>
        <div class="bubble"></div>
    </div>

    <div class="mini-bar">
        <div class="mini-bar-inner">
            <div class="mini-bar-left">
                <form class="mini-bar-form mini-bar-icon" method="POST" action="/set-currency" data-mini-dropdown>
                    <input type="hidden" name="currency" value="<?php echo e($currentCurrency); ?>" data-mini-value>
                    <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                    <div class="mini-dropdown" data-mini-dropdown>
                        <button type="button" class="mini-dropdown-trigger" data-mini-trigger>
                            <span class="mini-icon" aria-hidden="true">
                                <svg viewBox="0 0 24 24">
                                    <path d="M12 3v18"></path>
                                    <path d="M8 7h7a3 3 0 0 1 0 6H9a3 3 0 0 0 0 6h7"></path>
                                </svg>
                            </span>
                            <span class="mini-value" data-mini-label><?php echo e($currentCurrency); ?></span>
                            <svg viewBox="0 0 24 24" aria-hidden="true">
                                <path d="M6 9l6 6 6-6"></path>
                            </svg>
                        </button>
                        <div class="mini-dropdown-menu" data-mini-menu>
                            <?php foreach ($currencyOptions as $code => $symbol): ?>
                                <button type="button" class="mini-dropdown-option" data-mini-option="<?php echo e($code); ?>">
                                    <span class="mini-option-icon" aria-hidden="true"><?php echo e($symbol); ?></span>
                                    <span class="mini-option-label"><?php echo e($code); ?></span>
                                </button>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </form>
                <form class="mini-bar-form mini-bar-icon" method="POST" action="/set-language" data-mini-dropdown>
                    <input type="hidden" name="language" value="<?php echo e($currentLanguage); ?>" data-mini-value>
                    <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                    <div class="mini-dropdown" data-mini-dropdown>
                        <button type="button" class="mini-dropdown-trigger" data-mini-trigger>
                            <span class="mini-icon" aria-hidden="true">
                                <svg viewBox="0 0 24 24">
                                    <path d="M4 5h8"></path>
                                    <path d="M8 5v3"></path>
                                    <path d="M6 11h6"></path>
                                    <path d="M14 5h6"></path>
                                    <path d="M14 9h6"></path>
                                    <path d="M14 13h6"></path>
                                </svg>
                            </span>
                            <span class="mini-value" data-mini-label><?php echo e($languageOptions[$currentLanguage] ?? 'English'); ?></span>
                            <svg viewBox="0 0 24 24" aria-hidden="true">
                                <path d="M6 9l6 6 6-6"></path>
                            </svg>
                        </button>
                        <div class="mini-dropdown-menu" data-mini-menu>
                            <?php foreach ($languageOptions as $code => $label): ?>
                                <button type="button" class="mini-dropdown-option" data-mini-option="<?php echo e($code); ?>"><?php echo e($label); ?></button>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </form>
                <?php if ($forumEnabled): ?>
                    <a href="/forum" class="mini-bar-link" aria-label="Forum">
                        <svg viewBox="0 0 24 24" aria-hidden="true">
                            <path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"></path>
                            <path d="M7 9h10"></path>
                            <path d="M7 13h6"></path>
                        </svg>
                        <span>Forum</span>
                    </a>
                <?php endif; ?>
                    <a href="/games" class="mini-bar-link" aria-label="Mini games">
                        <svg viewBox="0 0 24 24" aria-hidden="true">
                            <path d="M6 12h4m-2-2v4"></path>
                            <circle cx="17" cy="10" r="1"></circle>
                            <circle cx="15" cy="14" r="1"></circle>
                            <rect x="2" y="4" width="20" height="16" rx="4"></rect>
                        </svg>
                        <span>MINI-GAMES</span>
                    </a>
            </div>
            <div class="mini-bar-right">
                <?php if (($theme['header_style'] ?? 'top') === 'sidebar'): ?>
                <button type="button" class="sidebar-toggle" data-sidebar-toggle aria-label="Toggle menu">
                    <svg viewBox="0 0 24 24" aria-hidden="true">
                        <line x1="3" y1="6" x2="21" y2="6"></line>
                        <line x1="3" y1="12" x2="21" y2="12"></line>
                        <line x1="3" y1="18" x2="21" y2="18"></line>
                    </svg>
                </button>
                <?php endif; ?>
                <?php if (!empty($topbarMenu)): ?>
                <nav class="mini-bar-nav" aria-label="Top bar menu">
                    <ul>
                        <?php foreach ($topbarMenu as $item): ?>
                            <li class="<?php echo !empty($item['children']) ? 'has-submenu' : ''; ?>">
                                <a href="<?php echo e(getMenuItemUrl($item)); ?>" target="<?php echo e($item['target']); ?>" title="<?php echo e($item['title']); ?>">
                                    <?php if (!empty($item['icon'])): ?>
                                        <span class="mini-bar-menu-icon" aria-hidden="true"><?php echo e($item['icon']); ?></span>
                                    <?php endif; ?>
                                    <span class="mini-bar-menu-text"><?php echo e($item['title']); ?></span>
                                </a>
                                <?php if (!empty($item['children'])): ?>
                                    <ul class="mini-bar-submenu">
                                        <?php foreach ($item['children'] as $child): ?>
                                            <li>
                                                <a href="<?php echo e(getMenuItemUrl($child)); ?>" target="<?php echo e($child['target']); ?>" title="<?php echo e($child['title']); ?>">
                                                    <?php if (!empty($child['icon'])): ?>
                                                        <span class="mini-bar-menu-icon" aria-hidden="true"><?php echo e($child['icon']); ?></span>
                                                    <?php endif; ?>
                                                    <span class="mini-bar-menu-text"><?php echo e($child['title']); ?></span>
                                                </a>
                                            </li>
                                        <?php endforeach; ?>
                                    </ul>
                                <?php endif; ?>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </nav>
                <?php endif; ?>
                <?php if ($searchShowMini): ?>
                <form class="mini-search search-widget" method="GET" action="/search" data-search-widget>
                    <input type="text" name="q" placeholder="Search..." value="<?php echo e($searchQuery); ?>" data-search-input>
                    <button type="submit" aria-label="Search">
                        <svg viewBox="0 0 24 24" aria-hidden="true">
                            <circle cx="11" cy="11" r="7"></circle>
                            <path d="M20 20l-3.5-3.5"></path>
                        </svg>
                    </button>
                    <div class="search-suggest" data-search-suggest></div>
                </form>
                <?php endif; ?>
                <?php if ($cartShowMini): ?>
                    <a href="/cart" class="mini-cart-button" data-cart-toggle aria-label="Cart">
                        <svg viewBox="0 0 24 24" aria-hidden="true">
                            <circle cx="9" cy="20" r="1.5"></circle>
                            <circle cx="17" cy="20" r="1.5"></circle>
                            <path d="M3 4h2l2.4 10.8a2 2 0 0 0 2 1.6h7.4a2 2 0 0 0 2-1.5L21 7H7"></path>
                        </svg>
                        <span class="cart-badge<?php echo $cartBadgeHidden; ?>" data-cart-count><?php echo (int)$cartItemCount; ?></span>
                    </a>
                <?php endif; ?>
                <?php if ($notificationLocation === 'mini' && $isUserLoggedIn): ?>
                    <div class="notify-wrapper">
                        <button type="button" class="notify-button" data-notify-toggle aria-label="Messages">
                            <svg viewBox="0 0 24 24" aria-hidden="true">
                                <path d="M6 8a6 6 0 0 1 12 0v5l2 3H4l2-3z"></path>
                                <path d="M9 19a3 3 0 0 0 6 0"></path>
                            </svg>
                            <?php if ($unreadCount > 0): ?>
                                <span class="notify-indicator"></span>
                            <?php endif; ?>
                        </button>
                        <?php echo $notificationMenu; ?>
                    </div>
                <?php endif; ?>
                <?php if ($accountShowMini): ?>
                    <div class="account-wrapper">
                        <button type="button" class="mini-dropdown-trigger mini-bar-account" data-account-toggle aria-label="<?php echo e($accountLabel); ?>">
                            <?php if ($accountPhoto): ?>
                                <span class="mini-icon mini-account-photo" aria-hidden="true">
                                    <img src="<?php echo e($accountPhoto); ?>" alt="Profile">
                                </span>
                            <?php else: ?>
                                <span class="mini-icon" aria-hidden="true">
                                    <svg viewBox="0 0 24 24" class="mini-account-icon">
                                        <circle cx="12" cy="8" r="4"></circle>
                                        <path d="M4 20c0-4 4-6 8-6s8 2 8 6"></path>
                                    </svg>
                                </span>
                            <?php endif; ?>
                        </button>
                        <?php echo $accountMenu; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <div id="page-loader"></div>
    <button type="button" class="back-to-top" data-back-to-top aria-label="Back to top">
        <svg viewBox="0 0 24 24" aria-hidden="true">
            <path d="M12 5l7 7-1.4 1.4L13 9.8V20h-2V9.8L6.4 13.4 5 12z"></path>
        </svg>
    </button>

    <?php if (($theme['header_style'] ?? 'top') === 'sidebar'): ?>
    <!-- Sidebar Header -->
    <aside class="sidebar-header">
        <div class="sidebar-logo">
            <a href="/">
                <?php if (($theme['logo_type'] ?? 'text') === 'image' && !empty($theme['site_logo'])): ?>
                <?php renderOptimizedLogo($theme, $logoHeightClass); ?>
                <?php else: ?>
                <span class="logo-icon">
                    <svg viewBox="0 0 24 24" aria-hidden="true">
                        <path d="M3 10.5l9-7 9 7"></path>
                        <path d="M5 10v10h14V10"></path>
                    </svg>
                </span>
                <span class="logo-text"><?php echo e($theme['site_name'] ?? 'LOGO'); ?></span>
                <?php endif; ?>
            </a>
        </div>
        <?php if ($searchShowSidebar): ?>
        <div class="sidebar-search">
            <form class="search-widget" method="GET" action="/search" data-search-widget>
                <input type="text" name="q" placeholder="Search..." value="<?php echo e($searchQuery); ?>" data-search-input>
                <button type="submit" aria-label="Search">
                    <svg viewBox="0 0 24 24" aria-hidden="true">
                        <circle cx="11" cy="11" r="7"></circle>
                        <path d="M20 20l-3.5-3.5"></path>
                    </svg>
                </button>
                <div class="search-suggest" data-search-suggest></div>
            </form>
        </div>
        <?php endif; ?>
        <nav class="sidebar-nav" aria-label="Sidebar menu">
            <ul class="nav-menu">
                <?php renderNestedMenu($headerMenu, 'submenu', true); ?>
            </ul>
        </nav>
        <div class="sidebar-footer">
            <div class="sidebar-footer-separator"></div>
            <ul class="nav-menu footer-menu">
                <?php if ($cartShowSidebar): ?>
                <li class="nav-item cart-nav-item" data-cart-icon>
                    <a href="/cart" class="cart-nav-link" data-cart-toggle aria-label="Cart">
                        <span class="nav-icon nav-icon--svg">
                            <svg viewBox="0 0 24 24" aria-hidden="true">
                                <path d="M3 4h2l2 10h11l1.5-8.5H6"></path>
                                <circle cx="9" cy="20" r="1.5"></circle>
                                <circle cx="18" cy="20" r="1.5"></circle>
                            </svg>
                        </span>
                        <span class="nav-text">Cart</span>
                        <span class="cart-badge<?php echo $cartBadgeHidden; ?>" data-cart-count><?php echo (int)$cartItemCount; ?></span>
                    </a>
                </li>
                <?php endif; ?>
                <?php if ($notificationLocation === 'sidebar' && $isUserLoggedIn): ?>
                <li class="nav-item notify-nav-item">
                    <div class="notify-wrapper">
                        <a href="#" class="nav-link notify-button--sidebar" data-notify-toggle aria-label="Notifications">
                            <span class="nav-icon nav-icon--svg">
                                <svg viewBox="0 0 24 24" aria-hidden="true">
                                    <path d="M6 8a6 6 0 0 1 12 0v5l2 3H4l2-3z"></path>
                                    <path d="M9 19a3 3 0 0 0 6 0"></path>
                                </svg>
                            </span>
                            <span class="nav-text">Notifications</span>
                            <?php if ($unreadCount > 0): ?>
                                <span class="notify-indicator"></span>
                            <?php endif; ?>
                        </a>
                        <?php echo $notificationMenu; ?>
                    </div>
                </li>
                <?php endif; ?>
                <?php if ($accountShowSidebar): ?>
                <li class="nav-item <?php echo $pageSlug === 'account' || $pageSlug === 'login' ? 'active' : ''; ?>">
                    <a href="<?php echo e($accountUrl); ?>" aria-label="<?php echo e($accountLabel); ?>">
                        <span class="nav-icon nav-icon--svg">
                            <svg viewBox="0 0 24 24" aria-hidden="true">
                                <circle cx="12" cy="8" r="4"></circle>
                                <path d="M4 20c0-4 4-6 8-6s8 2 8 6"></path>
                            </svg>
                        </span>
                        <span class="nav-text"><?php echo e($accountLabel); ?></span>
                    </a>
                </li>
                <?php endif; ?>
                <?php if ($contactEnabled): ?>
                <li class="nav-item <?php echo $pageSlug === 'contact' ? 'active' : ''; ?>">
                    <a href="/contact">
                        <span class="nav-icon nav-icon--svg">
                            <svg viewBox="0 0 24 24" aria-hidden="true">
                                <path d="M21 15a4 4 0 0 1-4 4H8l-5 4V7a4 4 0 0 1 4-4h10a4 4 0 0 1 4 4z"></path>
                            </svg>
                        </span>
                        <span class="nav-text">Contact</span>
                    </a>
                </li>
                <?php endif; ?>
                <?php if (AdminAuth::isAllowedIp(IpService::getClientIP())): ?>
                <li class="nav-item admin-access-item">
                    <a href="/admin/" class="admin-access-btn" title="Admin Panel" aria-label="Admin Panel">
                        <span class="nav-icon nav-icon--svg">
                            <svg viewBox="0 0 24 24" aria-hidden="true">
                                <circle cx="12" cy="12" r="3"></circle>
                                <path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1 0 2.83 2 2 0 0 1-2.83 0l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-2 2 2 2 0 0 1-2-2v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83 0 2 2 0 0 1 0-2.83l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1-2-2 2 2 0 0 1 2-2h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 0-2.83 2 2 0 0 1 2.83 0l.06.06a1.65 1.65 0 0 0 1.82.33H9a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 2-2 2 2 0 0 1 2 2v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 0 2 2 0 0 1 0 2.83l-.06.06a1.65 1.65 0 0 0-.33 1.82V9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 2 2 2 2 0 0 1-2 2h-.09a1.65 1.65 0 0 0-1.51 1z"></path>
                            </svg>
                        </span>
                        <span class="nav-text">Admin</span>
                    </a>
                </li>
                <?php endif; ?>
                <?php if ($hasSocialLinks): ?>
                <li class="nav-item has-submenu social-links-item">
                    <a href="#" class="social-toggle" aria-label="Social links">
                        <span class="nav-icon nav-icon--svg">
                            <svg viewBox="0 0 24 24" aria-hidden="true">
                                <circle cx="12" cy="12" r="9"></circle>
                                <path d="M3 12h18"></path>
                                <path d="M12 3a15 15 0 0 1 0 18"></path>
                                <path d="M12 3a15 15 0 0 0 0 18"></path>
                            </svg>
                        </span>
                        <span class="nav-text">Social</span>
                    </a>
                    <ul class="submenu submenu-up">
                        <?php foreach ($socialLinks as $link): ?>
                        <li>
                            <a href="<?php echo e($link['url']); ?>" target="_blank" rel="noopener">
                                <span class="nav-icon nav-icon--svg"><?php echo $link['icon']; ?></span>
                                <?php echo e($link['label']); ?>
                            </a>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                </li>
                <?php endif; ?>
            </ul>
        </div>
    </aside>
    <div class="sidebar-overlay" data-sidebar-overlay></div>
    <?php else: ?>
    <!-- Top Header -->
    <header class="top-header">
        <div class="header-container">
            <div class="logo">
                <a href="/">
                    <?php if (($theme['logo_type'] ?? 'text') === 'image' && !empty($theme['site_logo'])): ?>
                    <?php renderOptimizedLogo($theme, $logoHeightClass); ?>
                    <?php else: ?>
                    <span class="logo-text"><?php echo e($theme['site_name'] ?? 'LOGO'); ?></span>
                    <?php endif; ?>
                </a>
            </div>
            <nav class="main-nav" aria-label="Main navigation">
                <button class="mobile-menu-toggle" aria-label="Toggle menu">
                    <span></span>
                    <span></span>
                    <span></span>
                </button>
                <ul class="nav-menu">
                    <?php if ($cartShowTop): ?>
                    <li class="nav-item cart-nav-item" data-cart-icon>
                        <a href="/cart" class="cart-nav-link" data-cart-toggle aria-label="Cart">
                            <span class="nav-icon">🛒</span>
                            <span class="nav-text">Cart</span>
                            <span class="cart-badge<?php echo $cartBadgeHidden; ?>" data-cart-count><?php echo (int)$cartItemCount; ?></span>
                        </a>
                    </li>
                    <?php endif; ?>
                    <?php renderNestedMenu($headerMenu, 'submenu', true); ?>
                    <?php if ($contactEnabled): ?>
                    <li class="nav-item <?php echo $pageSlug === 'contact' ? 'active' : ''; ?>">
                        <a href="/contact">
                            <span class="nav-icon">💬</span>
                            Contact
                        </a>
                    </li>
                    <?php endif; ?>
                </ul>
            </nav>
            <?php if ($searchShowTop): ?>
            <div class="header-search">
                <form class="search-widget" method="GET" action="/search" data-search-widget>
                    <input type="text" name="q" placeholder="Search..." value="<?php echo e($searchQuery); ?>" data-search-input>
                    <button type="submit" aria-label="Search">
                        <svg viewBox="0 0 24 24" aria-hidden="true">
                            <circle cx="11" cy="11" r="7"></circle>
                            <path d="M20 20l-3.5-3.5"></path>
                        </svg>
                    </button>
                    <div class="search-suggest" data-search-suggest></div>
                </form>
            </div>
            <?php endif; ?>
            <?php if ($accountShowTop || ($notificationLocation === 'top' && $isUserLoggedIn) || AdminAuth::isAllowedIp(IpService::getClientIP())): ?>
            <div class="header-actions">
                <?php if ($accountShowTop): ?>
                    <div class="account-wrapper">
                        <button type="button" class="account-access-btn" data-account-toggle title="<?php echo e($accountLabel); ?>" aria-label="<?php echo e($accountLabel); ?>">
                            <?php if ($accountPhoto): ?>
                                <span class="account-access-icon account-access-photo" aria-hidden="true">
                                    <img src="<?php echo e($accountPhoto); ?>" alt="Profile">
                                </span>
                            <?php else: ?>
                                <span class="account-access-icon">👤</span>
                            <?php endif; ?>
                        </button>
                        <?php echo $accountMenu; ?>
                    </div>
                <?php endif; ?>
                <?php if ($notificationLocation === 'top' && $isUserLoggedIn): ?>
                    <div class="notify-wrapper">
                        <button type="button" class="notify-button" data-notify-toggle aria-label="Messages">
                            <svg viewBox="0 0 24 24" aria-hidden="true">
                                <path d="M6 8a6 6 0 0 1 12 0v5l2 3H4l2-3z"></path>
                                <path d="M9 19a3 3 0 0 0 6 0"></path>
                            </svg>
                            <?php if ($unreadCount > 0): ?>
                                <span class="notify-indicator"></span>
                            <?php endif; ?>
                        </button>
                        <?php echo $notificationMenu; ?>
                    </div>
                <?php endif; ?>
                <a href="/admin/" class="admin-access-btn" title="Admin Panel" aria-label="Admin Panel">
                    <span class="admin-access-icon">⚙️</span>
                </a>
            </div>
            <?php endif; ?>
        </div>
    </header>
    <?php endif; ?>

    <?php if (!empty($breadcrumbs)): ?>
    <!-- Breadcrumbs Navigation -->
    <nav class="breadcrumbs-nav" aria-label="Breadcrumb">
        <div class="container">
            <ol class="breadcrumbs" itemscope itemtype="https://schema.org/BreadcrumbList">
                <?php
                    $separators = [
                        'chevron' => '›',
                        'slash' => '/',
                        'arrow' => '→',
                        'dot' => '•'
                    ];
        $sep = $separators[$breadcrumbsSeparator] ?? '›';
        $position = 1;
        foreach ($breadcrumbs as $index => $crumb):
            $isLast = $index === count($breadcrumbs) - 1;
            ?>
                <li class="breadcrumb-item <?php echo $isLast ? 'breadcrumb-item--current' : ''; ?>" itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem">
                    <?php if (!$isLast && $crumb['url']): ?>
                        <a href="<?php echo e($crumb['url']); ?>" itemprop="item">
                            <?php if (!empty($crumb['is_home'])): ?>
                                <svg class="breadcrumb-home-icon" viewBox="0 0 20 20" fill="currentColor" aria-hidden="true">
                                    <path d="M10.707 2.293a1 1 0 00-1.414 0l-7 7a1 1 0 001.414 1.414L4 10.414V17a1 1 0 001 1h2a1 1 0 001-1v-2a1 1 0 011-1h2a1 1 0 011 1v2a1 1 0 001 1h2a1 1 0 001-1v-6.586l.293.293a1 1 0 001.414-1.414l-7-7z"/>
                                </svg>
                                <span itemprop="name"><?php echo e($crumb['label']); ?></span>
                            <?php else: ?>
                                <span itemprop="name"><?php echo e($crumb['label']); ?></span>
                            <?php endif; ?>
                        </a>
                        <span class="breadcrumb-separator" aria-hidden="true"><?php echo $sep; ?></span>
                    <?php else: ?>
                        <?php if (!empty($crumb['is_home'])): ?>
                            <span aria-current="page">
                                <svg class="breadcrumb-home-icon" viewBox="0 0 20 20" fill="currentColor" aria-hidden="true">
                                    <path d="M10.707 2.293a1 1 0 00-1.414 0l-7 7a1 1 0 001.414 1.414L4 10.414V17a1 1 0 001 1h2a1 1 0 001-1v-2a1 1 0 011-1h2a1 1 0 011 1v2a1 1 0 001 1h2a1 1 0 001-1v-6.586l.293.293a1 1 0 001.414-1.414l-7-7z"/>
                                </svg>
                                <span itemprop="name"><?php echo e($crumb['label']); ?></span>
                            </span>
                        <?php else: ?>
                            <span itemprop="name" aria-current="page"><?php echo e($crumb['label']); ?></span>
                        <?php endif; ?>
                    <?php endif; ?>
                    <meta itemprop="position" content="<?php echo $position++; ?>">
                </li>
                <?php endforeach; ?>
            </ol>
        </div>
    </nav>
    <?php endif; ?>

    <?php
    // ── Page template routing (allowlist-driven, single include point) ─────────
    // Same controlled-include pattern as admin front controller (admin/index.php).
    // Template filenames are hardcoded — no user input reaches the include path.
    // Order matters: forum-post must precede forum ($isForumPostPage ⊂ $isForumPage).
    $pageTemplateRoutes = [
        [$isContactPage,            'contact.php',              true],
        [$isDmcaPage,               'dmca.php',                 false],
        [$isCartPage,               'cart.php',                 true],
        [$isCheckoutPage,           'checkout.php',             true],
        [$isCheckoutSuccessPage,    'checkout-success.php',     false],
        [$isSearchPage,             'search.php',               false],
        [$isLoginPage,              'login.php',                false],
        [$isAccountPage,            'account.php',              false],
        [$isFriendsPage,            'friends.php',              false],
        [$isForumPostPage,          'forum-post.php',           false],
        [$isForumPage,              'forum.php',                false],
        [$isGamesPage,              'games.php',                false],
        [$isTextFinderReplacerPage, 'text-finder-replacer.php', false],
        [$isProfilePage,            'profile.php',              false],
        [$isMessagesPage,           'messages.php',             false],
        [$isAccountSettingsPage,    'account-settings.php',     false],
    ];

$activeTemplate      = null;
$templateHasSections = false;
foreach ($pageTemplateRoutes as [$isActive, $tplFile, $hasSections]) {
    if ($isActive) {
        $activeTemplate      = $tplFile;
        $templateHasSections = $hasSections;
        break;
    }
}
?>

    <!-- Main Content -->
    <main class="main-content">
        <?php if ($activeTemplate !== null): ?>
            <?php TemplateRenderer::renderOnce(__DIR__ . '/templates/' . $activeTemplate, get_defined_vars()); ?>
            <?php if ($templateHasSections && !empty($sections)): ?>
                <?php renderPageSections($sections, $db, ['page' => $page, 'pageSlug' => $pageSlug, 'productCommentFlash' => $productCommentFlash ?? null]); ?>
            <?php endif; ?>
        <?php elseif (empty($sections)): ?>
        <div class="page-content">
            <div class="container">
                <h1><?php echo e($page['title']); ?></h1>
                <?php
        $cleanContent = (is_string($page['content']) && $page['content'] !== '') ? MissingUploadsService::stripFromHtml($page['content']) : ($page['content'] ?? '');
            if ($cleanContent !== $page['content'] && isset($page['id'])) {
                $stmt = $db->prepare("UPDATE pages SET content = ? WHERE id = ?");
                $stmt->execute([$cleanContent, $page['id']]);
                $page['content'] = $cleanContent;
            }
            $renderedContent = ContentRenderer::renderPageContentPlaceholders($cleanContent);
            echo $renderedContent;
            ?>
            </div>
        </div>
        <?php else: ?>
            <?php renderPageSections($sections, $db, ['page' => $page, 'pageSlug' => $pageSlug, 'productCommentFlash' => $productCommentFlash ?? null]); ?>
        <?php endif; ?>
    </main>

    <div class="cart-sidebar" id="cart-sidebar" aria-hidden="true">
        <div class="cart-sidebar-overlay" data-cart-close></div>
        <div class="cart-sidebar-panel">
            <div class="cart-sidebar-header">
                <h3>Your Cart</h3>
                <button type="button" class="cart-sidebar-close" data-cart-close>&times;</button>
            </div>
            <div class="cart-sidebar-body" id="cart-sidebar-body">
                <div class="cart-empty">Your cart is empty.</div>
            </div>
            <div class="cart-sidebar-footer">
                <div class="cart-sidebar-total">
                    <span>Total</span>
                    <strong id="cart-sidebar-total">$0</strong>
                </div>
                <a href="/cart" class="btn btn-primary cart-sidebar-checkout">Go to Cart</a>
            </div>
        </div>
    </div>

    <!-- Footer -->
    <footer class="site-footer">
        <div class="footer-container">
            <div class="footer-content">
                <?php if ($searchShowFooter): ?>
                <div class="footer-search">
                    <form class="search-widget" method="GET" action="/search" data-search-widget>
                        <input type="text" name="q" placeholder="Search..." value="<?php echo e($searchQuery); ?>" data-search-input>
                        <button type="submit" aria-label="Search">
                            <svg viewBox="0 0 24 24" aria-hidden="true">
                                <circle cx="11" cy="11" r="7"></circle>
                                <path d="M20 20l-3.5-3.5"></path>
                            </svg>
                        </button>
                        <div class="search-suggest" data-search-suggest></div>
                    </form>
                </div>
                <?php endif; ?>
                <?php if ($cartShowFooter): ?>
                <div class="footer-account">
                    <a href="/cart" class="account-access-btn" aria-label="Cart">
                        <span class="account-access-icon">🛒</span>
                        <span>Cart</span>
                    </a>
                </div>
                <?php endif; ?>
                <?php if ($accountShowFooter): ?>
                <div class="footer-account">
                    <a href="<?php echo e($accountUrl); ?>" class="account-access-btn" aria-label="<?php echo e($accountLabel); ?>">
                        <span class="account-access-icon">👤</span>
                        <span><?php echo e($accountLabel); ?></span>
                    </a>
                </div>
                <?php endif; ?>
                <div class="footer-brand">
                    <?php if (($theme['logo_type'] ?? 'text') === 'image' && !empty($theme['site_logo'])): ?>
                    <?php renderOptimizedLogo($theme, $logoHeightClass, 'footer-logo'); ?>
                    <?php else: ?>
                    <span class="logo-text"><?php echo e($theme['site_name'] ?? 'LOGO'); ?></span>
                    <?php endif; ?>
                    <p><?php echo e($theme['site_tagline'] ?? ''); ?></p>
                </div>
                <nav class="footer-nav" aria-label="Footer navigation">
                    <ul class="footer-menu">
                        <?php
                    function renderFooterMenu($items, $maxDepth = 5, $currentDepth = 0)
                    {
                        if ($currentDepth >= $maxDepth || empty($items)) {
                            return;
                        }
                        foreach ($items as $item):
                            $hasChildren = !empty($item['children']);
                            ?>
                            <li class="footer-menu-item <?php echo $hasChildren ? 'has-submenu' : ''; ?>">
                                <a href="<?php echo e(getMenuItemUrl($item)); ?>" target="<?php echo e($item['target']); ?>">
                                    <?php if (!empty($item['icon'])): ?>
                                    <span class="nav-icon"><?php echo e($item['icon']); ?></span>
                                    <?php endif; ?>
                                    <?php echo e($item['title']); ?>
                                </a>
                                <?php if ($hasChildren): ?>
                                <ul class="footer-submenu submenu-level-<?php echo $currentDepth + 1; ?>">
                                    <?php renderFooterMenu($item['children'], $maxDepth, $currentDepth + 1); ?>
                                </ul>
                                <?php endif; ?>
                            </li>
                        <?php
                        endforeach;
                    }
renderFooterMenu($footerMenu);
?>
                    </ul>
                </nav>
            </div>
            <div class="footer-bottom">
                <p>&copy; <?php echo date('Y'); ?> <?php echo e($theme['site_name'] ?? 'Your Website'); ?>. All rights reserved.</p>
                <div class="footer-links">
                    <?php if ($pageTermsEnabled): ?><a href="/terms-of-service" class="footer-link">Terms of Service</a><?php endif; ?>
                    <?php if ($pagePrivacyEnabled): ?><a href="/privacy-policy" class="footer-link">Privacy Policy</a><?php endif; ?>
                    <?php if ($pageRefundEnabled): ?><a href="/refund-policy" class="footer-link">Refund Policy</a><?php endif; ?>
                    <?php if ($pageShippingEnabled): ?><a href="/shipping-policy" class="footer-link">Shipping Policy</a><?php endif; ?>
                    <?php if ($pageLegalNoticeEnabled): ?><a href="/legal-notice" class="footer-link">Legal Notice</a><?php endif; ?>
                    <?php if ($pageDmcaEnabled): ?><a href="/dmca" class="footer-link">DMCA</a><?php endif; ?>
                    <?php if ($cookieNoticeEnabled): ?><a href="#" class="footer-link js-cookie-settings">Cookie Settings</a><?php endif; ?>
                </div>
            </div>
        </div>
    </footer>

    <?php if ($cookieNoticeEnabled): ?>
    <div id="cookie-consent-banner" class="cookie-consent" aria-live="polite" aria-label="Cookie consent" data-nosnippet>
        <div class="cookie-consent__inner">
            <div class="cookie-consent__text">
                <h3>Cookie Notice</h3>
                <p>We use cookies to improve your experience. You can accept all, reject all, or customize your preferences.</p>
            </div>
            <div class="cookie-consent__actions">
                <button type="button" class="btn btn-outline btn-sm" data-cookie-settings>Settings</button>
                <button type="button" class="btn btn-outline btn-sm" data-cookie-reject>Reject All</button>
                <button type="button" class="btn btn-primary btn-sm" data-cookie-accept>Accept All</button>
            </div>
        </div>
    </div>

    <div id="cookie-settings-modal" class="cookie-modal" aria-label="Cookie preferences" data-nosnippet>
        <div class="cookie-modal__backdrop" data-cookie-close></div>
        <div class="cookie-modal__content">
            <h2>Cookie Preferences</h2>
            <p class="cookie-modal__subtitle">Choose which cookies you want to allow. You can update this at any time.</p>
            <div class="cookie-modal__option">
                <div class="cookie-modal__option-header">
                    <span>Necessary</span>
                    <span class="cookie-toggle is-disabled">Always on</span>
                </div>
                <p>Required for the website to function and cannot be disabled.</p>
            </div>
            <div class="cookie-modal__option">
                <div class="cookie-modal__option-header">
                    <label for="cookie-consent-analytics">Analytics</label>
                    <div class="cookie-switch">
                        <input type="checkbox" id="cookie-consent-analytics" name="analytics" autocomplete="off">
                        <span class="cookie-switch__slider"></span>
                    </div>
                </div>
                <p>Helps us understand how visitors interact with the website.</p>
            </div>
            <div class="cookie-modal__option">
                <div class="cookie-modal__option-header">
                    <label for="cookie-consent-marketing">Marketing</label>
                    <div class="cookie-switch">
                        <input type="checkbox" id="cookie-consent-marketing" name="marketing" autocomplete="off">
                        <span class="cookie-switch__slider"></span>
                    </div>
                </div>
                <p>Personalized offers and relevant promotional content.</p>
            </div>
            <div class="cookie-modal__actions">
                <button type="button" class="btn btn-outline" data-cookie-close>Cancel</button>
                <button type="button" class="btn btn-primary" data-cookie-save>Save Preferences</button>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <script nonce="<?php echo e(getCspNonce()); ?>" src="<?php echo AssetVersioning::url('/assets/js/main.js'); ?>" defer></script>
    <?php if ($needsEasyMediaJs): ?>
    <script nonce="<?php echo e(getCspNonce()); ?>" src="<?php echo AssetVersioning::url('/assets/js/easy-media.js'); ?>" defer></script>
    <?php endif; ?>
    <?php if ($cookieNoticeEnabled): ?>
    <script nonce="<?php echo e(getCspNonce()); ?>" src="<?php echo AssetVersioning::url('/assets/js/cookie-consent.js'); ?>" defer></script>
    <?php endif; ?>

    <?php if ($isUserLoggedIn): ?>
    <!-- Messenger Widget -->
    <div class="messenger-widget" id="messenger-widget">
        <div class="messenger-panel">
            <div class="messenger-header">
                <h3 id="messenger-title">Chat</h3>
                <button type="button" class="messenger-close" id="messenger-close-btn">&times;</button>
            </div>

            <!-- Friend List View -->
            <div class="messenger-body" id="messenger-list-view">
                <div class="messenger-list" id="messenger-friend-list">
                    <!-- Friends will be loaded here -->
                    <div class="text-muted text-center friends-loading">Loading friends...</div>
                </div>
            </div>

            <!-- Chat View -->
            <div class="messenger-chat-view" id="messenger-chat-view">
                <div class="messenger-chat-header">
                    <button type="button" class="messenger-back" id="messenger-back-btn">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M19 12H5M12 19l-7-7 7-7"/></svg>
                    </button>
                    <div class="messenger-avatar-wrap">
                        <img src="" alt="" class="messenger-avatar" id="chat-user-avatar">
                        <span class="messenger-status online" id="chat-user-status"></span>
                    </div>
                    <div class="messenger-friend-info">
                        <a class="messenger-friend-name" id="chat-user-link" href="#" aria-label="View profile">
                            <span id="chat-user-name">Friend Name</span>
                        </a>
                    </div>
                </div>
                <div class="messenger-chat-messages" id="messenger-messages">
                    <!-- Messages will be loaded here -->
                </div>
                <form class="messenger-input-wrap" id="messenger-form">
                    <input type="text" class="messenger-input" id="messenger-input" placeholder="Type a message..." autocomplete="off">
                    <button type="submit" class="messenger-send">
                        <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M22 2L11 13M22 2l-7 20-4-9-9-4 20-7z"/></svg>
                    </button>
                </form>
            </div>
        </div>

        <button type="button" class="messenger-toggle" id="messenger-toggle">
            <span class="messenger-toggle-label">Chat</span>
            <span class="messenger-badge is-hidden-display" id="messenger-unread-badge">0</span>
        </button>
    </div>
    <?php endif; ?>

    <!-- Mobile Bottom Navigation (visible only ≤768px via CSS) -->
    <nav class="mobile-bottom-nav" aria-label="Mobile navigation">
        <a href="/" class="mobile-bottom-nav-item" aria-label="Home">
            <svg viewBox="0 0 24 24" aria-hidden="true"><path d="M3 12l9-9 9 9"></path><path d="M9 21V12h6v9"></path></svg>
            <span>Home</span>
        </a>
        <a href="/search" class="mobile-bottom-nav-item" aria-label="Search">
            <svg viewBox="0 0 24 24" aria-hidden="true"><circle cx="11" cy="11" r="7"></circle><path d="M20 20l-3.5-3.5"></path></svg>
            <span>Search</span>
        </a>
        <a href="/cart" class="mobile-bottom-nav-item" data-cart-toggle aria-label="Cart">
            <svg viewBox="0 0 24 24" aria-hidden="true"><circle cx="9" cy="20" r="1.5"></circle><circle cx="17" cy="20" r="1.5"></circle><path d="M3 4h2l2.4 10.8a2 2 0 0 0 2 1.6h7.4a2 2 0 0 0 2-1.5L21 7H7"></path></svg>
            <span>Cart</span>
            <span class="cart-badge<?php echo $cartBadgeHidden; ?>" data-cart-count><?php echo (int)$cartItemCount; ?></span>
        </a>
        <a href="<?php echo e($accountUrl); ?>" class="mobile-bottom-nav-item" aria-label="<?php echo e($accountLabel); ?>">
            <?php if ($accountPhoto): ?>
                <img src="<?php echo e($accountPhoto); ?>" alt="" class="mobile-bottom-nav-avatar">
            <?php else: ?>
                <svg viewBox="0 0 24 24" aria-hidden="true"><circle cx="12" cy="8" r="4"></circle><path d="M4 20c0-4 4-6 8-6s8 2 8 6"></path></svg>
            <?php endif; ?>
            <span><?php echo e($accountLabel); ?></span>
            <?php if ($pendingFriendRequests > 0): ?>
                <span class="mobile-bottom-nav-badge"></span>
            <?php endif; ?>
        </a>
        <button type="button" class="mobile-bottom-nav-item" data-mobile-drawer-toggle aria-label="More">
            <svg viewBox="0 0 24 24" aria-hidden="true"><line x1="3" y1="6" x2="21" y2="6"></line><line x1="3" y1="12" x2="21" y2="12"></line><line x1="3" y1="18" x2="21" y2="18"></line></svg>
            <span>Menu</span>
        </button>
    </nav>

    <!-- Mobile Drawer (slides up from bottom nav) -->
    <div class="mobile-drawer-overlay" data-mobile-drawer-overlay></div>
    <div class="mobile-drawer" data-mobile-drawer>
        <div class="mobile-drawer-header">
            <span class="mobile-drawer-title">Menu</span>
            <button type="button" class="mobile-drawer-close" data-mobile-drawer-close aria-label="Close menu">
                <svg viewBox="0 0 24 24" aria-hidden="true"><path d="M18 6L6 18"></path><path d="M6 6l12 12"></path></svg>
            </button>
        </div>
        <div class="mobile-drawer-body">
            <?php if ($isUserLoggedIn): ?>
            <div class="mobile-drawer-section">
                <a href="/account" class="mobile-drawer-link">
                    <svg viewBox="0 0 24 24" aria-hidden="true"><circle cx="12" cy="8" r="4"></circle><path d="M4 20c0-4 4-6 8-6s8 2 8 6"></path></svg>
                    My Account
                </a>
                <a href="/account?tab=orders" class="mobile-drawer-link">
                    <svg viewBox="0 0 24 24" aria-hidden="true"><path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path><polyline points="14 2 14 8 20 8"></polyline></svg>
                    Orders
                </a>
                <?php if ($pendingFriendRequests > 0): ?>
                <a href="/friends" class="mobile-drawer-link">
                    <svg viewBox="0 0 24 24" aria-hidden="true"><path d="M16 21v-2a4 4 0 0 0-3-3.87"></path><path d="M22 21v-2a4 4 0 0 0-3-3.87"></path><circle cx="9" cy="7" r="4"></circle><circle cx="16" cy="7" r="4"></circle></svg>
                    Friends
                    <span class="mobile-drawer-badge"><?php echo (int)$pendingFriendRequests; ?></span>
                </a>
                <?php endif; ?>
            </div>
            <?php endif; ?>

            <?php if ($isUserLoggedIn && $unreadCount > 0): ?>
            <div class="mobile-drawer-section">
                <a href="/account?tab=messages" class="mobile-drawer-link">
                    <svg viewBox="0 0 24 24" aria-hidden="true"><path d="M6 8a6 6 0 0 1 12 0v5l2 3H4l2-3z"></path><path d="M9 19a3 3 0 0 0 6 0"></path></svg>
                    Notifications
                    <span class="mobile-drawer-badge"><?php echo (int)$unreadCount; ?></span>
                </a>
            </div>
            <?php endif; ?>

            <?php if ($forumEnabled): ?>
            <div class="mobile-drawer-section">
                <a href="/forum" class="mobile-drawer-link">
                    <svg viewBox="0 0 24 24" aria-hidden="true"><path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"></path></svg>
                    Forum
                </a>
            </div>
            <?php endif; ?>

            <div class="mobile-drawer-section">
                <a href="/games" class="mobile-drawer-link">
                    <svg viewBox="0 0 24 24" aria-hidden="true"><path d="M6 12h4m-2-2v4"></path><circle cx="17" cy="10" r="1"></circle><circle cx="15" cy="14" r="1"></circle><rect x="2" y="4" width="20" height="16" rx="4"></rect></svg>
                    Mini-Games
                </a>
            </div>

            <?php if (!empty($headerMenu)): ?>
            <div class="mobile-drawer-section mobile-drawer-section-menu">
                <?php renderDrawerMenu($headerMenu, false); ?>
            </div>
            <?php endif; ?>

            <?php if (!empty($topbarMenu)): ?>
            <div class="mobile-drawer-section mobile-drawer-section-menu">
                <?php renderDrawerMenu($topbarMenu, true); ?>
            </div>
            <?php endif; ?>

            <div class="mobile-drawer-section mobile-drawer-section-utils">
                <div class="mobile-drawer-row">
                    <form class="mobile-drawer-form" method="POST" action="/set-currency" data-mini-dropdown>
                        <input type="hidden" name="currency" value="<?php echo e($currentCurrency); ?>" data-mini-value>
                        <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                        <div class="mini-dropdown" data-mini-dropdown>
                            <button type="button" class="mobile-drawer-util-btn" data-mini-trigger>
                                <svg viewBox="0 0 24 24" aria-hidden="true"><path d="M12 3v18"></path><path d="M8 7h7a3 3 0 0 1 0 6H9a3 3 0 0 0 0 6h7"></path></svg>
                                <span data-mini-label><?php echo e($currentCurrency); ?></span>
                            </button>
                            <div class="mini-dropdown-menu" data-mini-menu>
                                <?php foreach ($currencyOptions as $code => $symbol): ?>
                                    <button type="button" class="mini-dropdown-option" data-mini-option="<?php echo e($code); ?>">
                                        <span class="mini-option-icon" aria-hidden="true"><?php echo e($symbol); ?></span>
                                        <span class="mini-option-label"><?php echo e($code); ?></span>
                                    </button>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </form>
                    <form class="mobile-drawer-form" method="POST" action="/set-language" data-mini-dropdown>
                        <input type="hidden" name="language" value="<?php echo e($currentLanguage); ?>" data-mini-value>
                        <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                        <div class="mini-dropdown" data-mini-dropdown>
                            <button type="button" class="mobile-drawer-util-btn" data-mini-trigger>
                                <svg viewBox="0 0 24 24" aria-hidden="true"><circle cx="12" cy="12" r="10"></circle><line x1="2" y1="12" x2="22" y2="12"></line><path d="M12 2a15.3 15.3 0 0 1 4 10 15.3 15.3 0 0 1-4 10 15.3 15.3 0 0 1-4-10 15.3 15.3 0 0 1 4-10z"></path></svg>
                                <span data-mini-label><?php echo e($languageOptions[$currentLanguage] ?? 'English'); ?></span>
                            </button>
                            <div class="mini-dropdown-menu" data-mini-menu>
                                <?php foreach ($languageOptions as $code => $label): ?>
                                    <button type="button" class="mini-dropdown-option" data-mini-option="<?php echo e($code); ?>"><?php echo e($label); ?></button>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </form>
                </div>
            </div>

            <?php if ($contactEnabled): ?>
            <div class="mobile-drawer-section">
                <a href="/contact" class="mobile-drawer-link">
                    <svg viewBox="0 0 24 24" aria-hidden="true"><path d="M4 4h16c1.1 0 2 .9 2 2v12c0 1.1-.9 2-2 2H4c-1.1 0-2-.9-2-2V6c0-1.1.9-2 2-2z"></path><polyline points="22,6 12,13 2,6"></polyline></svg>
                    Contact
                </a>
            </div>
            <?php endif; ?>

            <?php if ($isUserLoggedIn): ?>
            <div class="mobile-drawer-section mobile-drawer-section-logout">
                <form method="POST" action="/account">
                    <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                    <input type="hidden" name="action" value="logout">
                    <button type="submit" class="mobile-drawer-link mobile-drawer-logout">
                        <svg viewBox="0 0 24 24" aria-hidden="true"><path d="M9 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h4"></path><polyline points="16 17 21 12 16 7"></polyline><line x1="21" y1="12" x2="9" y2="12"></line></svg>
                        Log Out
                    </button>
                </form>
            </div>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>
