/**
 * Main Frontend JavaScript
 * Neon Theme
 * COMPREHENSIVE ERROR LOGGING ENABLED
 */

// ===========================================
// FRONTEND ERROR LOGGING - Send ALL errors to admin logs
// ===========================================

function debugCatch(context, error) {
  if (globalThis?.console?.debug) {
    globalThis.console.debug(context, error);
  }
}

function resolveRequestUrl(input) {
  if (typeof input === "string") {
    return input;
  }

  return input?.url || "";
}

function setActiveSlide(slides, dots, index) {
  slides.forEach((s) => s.classList.remove("active"));
  dots.forEach((d) => d.classList.remove("active"));

  const currentIndex = (index + slides.length) % slides.length;
  slides[currentIndex].classList.add("active");
  if (dots[currentIndex]) {
    dots[currentIndex].classList.add("active");
  }

  return currentIndex;
}

function easeInOut(t) {
  return t < 0.5 ? 2 * t * t : -1 + (4 - 2 * t) * t;
}

function parsePriceValue(value) {
  if (!value) return 0;
  const numeric = String(value).replaceAll(/[^0-9.]/g, "");
  return numeric ? Number.parseFloat(numeric) : 0;
}

function resolveVariantData(source, fallback) {
  const data = source?.dataset || {};
  return {
    variantPrice: data.price || fallback.variantPrice || "",
    variantStock: data.stock || fallback.variantStock || "",
    variantUnlimited:
      data.unlimited === "1" || Boolean(fallback.variantUnlimited),
    variantAction: data.action || fallback.variantAction || "cart",
    variantDownload: data.download || fallback.variantDownload || "",
    variantLinkLabel: data.linkLabel || fallback.variantLinkLabel || "",
    variantDigital: data.digital === "1" || Boolean(fallback.variantDigital),
    variantOwned: data.owned === "1" || Boolean(fallback.variantOwned),
  };
}

function setActiveVariantButton(buttons, activeButton) {
  buttons.forEach((other) => {
    other.classList.toggle("active", other === activeButton);
  });
}

function restoreButtonText(button, text) {
  button.textContent = text;
}

function formatFileSizeLabel(bytes) {
  if (!bytes) return "0 B";
  const sizes = ["B", "KB", "MB", "GB"];
  const i = Math.min(
    sizes.length - 1,
    Math.floor(Math.log(bytes) / Math.log(1024)),
  );
  return `${(bytes / Math.pow(1024, i)).toFixed(i === 0 ? 0 : 1)} ${sizes[i]}`;
}

function renderSelectedUploadFiles(files, listContainer) {
  if (!listContainer) return;
  listContainer.innerHTML = "";
  if (!files?.length) return;

  const list = document.createElement("ul");
  list.className = "forum-upload-items";
  files.forEach((file) => {
    const item = document.createElement("li");
    item.textContent = `${file.name} (${formatFileSizeLabel(file.size)})`;
    list.appendChild(item);
  });
  listContainer.appendChild(list);
}

function hasMatchingSelectedProductAttribute(groupValues, selectedSet) {
  for (const value of groupValues) {
    if (selectedSet.has(value)) {
      return true;
    }
  }
  return false;
}

function setFormMessage(messageBox, text, mode = "") {
  if (!messageBox) return;
  messageBox.textContent = text;
  messageBox.classList.remove("is-error", "is-success");
  if (mode === "error" || mode === "success") {
    messageBox.classList.add(`is-${mode}`);
  }
}

function setLoadingButtonState(button, isLoading) {
  if (!button) return;
  button.disabled = isLoading;
  button.classList.toggle("is-loading", isLoading);
}

function validateImageUploads(files, maxFiles, maxSizeBytes) {
  if (!Array.isArray(files) || files.length === 0) {
    return { valid: true, error: "" };
  }
  if (files.length > maxFiles) {
    return {
      valid: false,
      error: `You can upload up to ${maxFiles} images.`,
    };
  }
  const tooLarge = files.find((file) => file.size > maxSizeBytes);
  if (tooLarge) {
    return {
      valid: false,
      error: `Each image must be ${Math.round(maxSizeBytes / (1024 * 1024))}MB or less.`,
    };
  }
  return { valid: true, error: "" };
}

// Log function to send errors to server
function logToServer(type, message, details) {
  try {
    const url = "/log-error";
    // Add current timestamp to URL for some lightweight cache busting and to ensure each error is a fresh request
    const cacheBustUrl = url + "?t=" + Date.now();

    // Include CSRF token if available, using FormData for maximum compatibility
    const formData = new FormData();
    formData.append("type", type);
    formData.append("message", message);
    formData.append("details", details);
    if (csrfToken) {
      formData.append("csrf_token", csrfToken);
    }

    fetch(cacheBustUrl, {
      method: "POST",
      body: formData,
    })
      .then(() => {})
      .catch(() => {}); // Silent fail - don't cause more errors
  } catch (error) {
    debugCatch("logToServer failed", error);
  }
}

const csrfTokenMeta = document.querySelector('meta[name="csrf-token"]');
const csrfToken = csrfTokenMeta ? csrfTokenMeta.getAttribute("content") : null;

function isSameOriginRequest(input) {
  try {
    const url = resolveRequestUrl(input);
    if (!url) return true;
    if (url.startsWith("/")) return true;
    const parsed = new URL(url, globalThis.location.origin);
    return parsed.origin === globalThis.location.origin;
  } catch (error) {
    debugCatch("isSameOriginRequest fallback", error);
    return true;
  }
}

// Auto-inject CSRF token but don't log
const originalFetch = globalThis.fetch;
globalThis.fetch = function (...args) {
  const input = args[0];
  const init = args[1] || {};
  const method = (init.method || "GET").toUpperCase();
  const headers = new Headers(init.headers || {});
  if (
    csrfToken &&
    method !== "GET" &&
    method !== "HEAD" &&
    isSameOriginRequest(input) &&
    !headers.has("X-CSRF-Token")
  ) {
    headers.set("X-CSRF-Token", csrfToken);
  }
  init.headers = headers;
  args[1] = init;
  return originalFetch.apply(this, args);
};

document.addEventListener(
  "submit",
  function (event) {
    const form = event.target;
    if (form?.tagName !== "FORM") return;
    const method = (form.getAttribute("method") || "GET").toUpperCase();
    if (method !== "POST") return;
    if (!csrfToken) return;
    if (form.querySelector('input[name="csrf_token"]')) return;

    const input = document.createElement("input");
    input.type = "hidden";
    input.name = "csrf_token";
    input.value = csrfToken;
    form.appendChild(input);
  },
  true,
);

// ===========================================
// END ERROR LOGGING
// ===========================================

document.addEventListener("DOMContentLoaded", function () {
  // Page loader bar
  initPageLoader();

  // Testimonials slider
  initTestimonialsSlider();
  // Product media slider
  initProductMediaSlider();
  initProductMediaLightbox();
  // Product share and variants
  initProductShareButtons();
  initProductLikeButtons();
  initProductVariantSelector();
  initProductTabs();
  initCartSidebar();
  initCartToggle();
  // Cart badge is server-rendered; no initial fetch needed
  initCartActions();
  initCartPage();
  initProductsListControls();
  initProductsListHoverMedia();
  initCollectionsCarouselScroll();
  initMiniBarSelectors();
  initMiniBarScroll();
  initForumCreatePost();
  initForumPostEditing();
  initForumComments();
  initForumImageLightbox();
  initBackToTop();
  initSearchAutocomplete();
  initNotificationMenu();
  initAccountMenu();
  initAccountTabs();
  initNotificationPolling();
  initAccountMessagesPolling();
  initTicketChatPolling();
  initChatImageLightbox();
  initCustomSelects();
  initConfirmDialogs();
  initSetupBannerDismiss();
  initCheckout();
  initSidebarToggle();
  // Mobile menu toggle
  initMobileMenu();
  initMobileDrawer();
  initMessengerWidget();

  // Testimonials
  initTestimonialsSlider();

  // Product media
  initProductMediaSlider();

  // Initialize slideshows
  initSlideshows();
  initImageBanners();

  // Smooth scroll for anchor links
  initSmoothScroll();

  // Add scroll effects
  initScrollEffects();

  // Privacy dropdown
  initPrivacyDropdowns();

  // Custom cursor fire glow
  initCursorFire();

  // Initialize friend action buttons globally (for profile pages, etc.)
  initFriendActionButtons(document);
});

/**
 * Privacy Dropdown for Settings
 */
function initPrivacyDropdowns() {
  const dropdowns = document.querySelectorAll("[data-privacy-dropdown]");

  dropdowns.forEach((dropdown) => {
    const trigger = dropdown.querySelector("[data-privacy-trigger]");
    const menu = dropdown.querySelector("[data-privacy-menu]");
    const hiddenInput = dropdown.querySelector("[data-privacy-value]");
    const iconSpan = dropdown.querySelector("[data-privacy-icon]");
    const labelSpan = dropdown.querySelector("[data-privacy-label]");
    const options = dropdown.querySelectorAll("[data-privacy-option]");

    if (!trigger || !menu) return;

    // Toggle dropdown
    trigger.addEventListener("click", (e) => {
      e.preventDefault();
      e.stopPropagation();

      // Close other dropdowns
      document
        .querySelectorAll("[data-privacy-dropdown].is-open")
        .forEach((d) => {
          if (d !== dropdown) d.classList.remove("is-open");
        });

      dropdown.classList.toggle("is-open");
    });

    // Option selection
    options.forEach((option) => {
      option.addEventListener("click", (e) => {
        e.preventDefault();
        const value = option.dataset.value;
        const label = option.querySelector("span").textContent;
        const iconSvg = option.querySelector("svg").outerHTML;

        // Update hidden input
        if (hiddenInput) hiddenInput.value = value;

        // Update trigger display
        if (labelSpan) labelSpan.textContent = label;
        if (iconSpan) {
          iconSpan.innerHTML = iconSvg;
          iconSpan.dataset.privacyIcon = value;
        }

        // Update active state
        for (const optionItem of options) {
          optionItem.classList.remove("active");
        }
        option.classList.add("active");

        // Close dropdown
        dropdown.classList.remove("is-open");
      });
    });
  });

  // Close on outside click
  document.addEventListener("click", (e) => {
    if (!e.target.closest("[data-privacy-dropdown]")) {
      document
        .querySelectorAll("[data-privacy-dropdown].is-open")
        .forEach((d) => {
          d.classList.remove("is-open");
        });
    }
  });
}

function initConfirmDialogs() {
  document.addEventListener(
    "submit",
    function (event) {
      const form = event.target;
      if (form?.tagName !== "FORM") return;
      const message = form.dataset.confirm;
      if (!message) return;
      if (!confirm(message)) {
        event.preventDefault();
        event.stopPropagation();
      }
    },
    true,
  );

  document.addEventListener(
    "click",
    function (event) {
      const element = event.target.closest("[data-confirm]");
      if (!element || element.tagName === "FORM") return;
      const message = element.dataset.confirm || "Are you sure?";
      if (!confirm(message)) {
        event.preventDefault();
        event.stopPropagation();
      }
    },
    true,
  );
}

function initSetupBannerDismiss() {
  const banner = document.getElementById("setupCompleteBanner");
  const btn = document.querySelector('[data-action="dismiss-setup-banner"]');
  if (!banner || !btn) return;
  btn.addEventListener("click", function () {
    banner.remove();
  });
}

/**
 * Split Slideshows
 */
function initSlideshows() {
  const slideshows = document.querySelectorAll(".section-split-slideshow");
  slideshows.forEach((slideshow) => {
    const slides = slideshow.querySelectorAll(".split-slide");
    const dots = slideshow.querySelectorAll(".slideshow-dot");
    const prevBtn = slideshow.querySelector(".slideshow-arrow.prev");
    const nextBtn = slideshow.querySelector(".slideshow-arrow.next");
    const autoplay = slideshow.dataset.autoplay === "true";
    let currentIndex = 0;
    let timer = null;

    if (slides.length <= 1) return;

    function showSlide(index) {
      currentIndex = setActiveSlide(slides, dots, index);
    }

    function nextSlide() {
      showSlide(currentIndex + 1);
    }

    function prevSlide() {
      showSlide(currentIndex - 1);
    }

    if (nextBtn)
      nextBtn.addEventListener("click", () => {
        nextSlide();
        resetAutoplay();
      });

    if (prevBtn)
      prevBtn.addEventListener("click", () => {
        prevSlide();
        resetAutoplay();
      });

    dots.forEach((dot) => {
      dot.addEventListener("click", () => {
        showSlide(Number.parseInt(dot.dataset.index, 10));
        resetAutoplay();
      });
    });

    function startAutoplay() {
      if (autoplay) {
        timer = setInterval(nextSlide, 5000);
      }
    }

    function resetAutoplay() {
      if (timer) {
        clearInterval(timer);
        startAutoplay();
      }
    }

    startAutoplay();
  });
}

/**
 * Image Banners
 */
function initImageBanners() {
  const banners = document.querySelectorAll(".section-image-banner");
  banners.forEach((banner) => {
    const slides = banner.querySelectorAll(".banner-slide");
    const dots = banner.querySelectorAll(".banner-dot");
    const autoplay = banner.dataset.autoplay === "true";
    const interval = Number.parseInt(banner.dataset.interval, 10) || 5000;
    let currentIndex = 0;
    let timer = null;

    if (slides.length <= 1) return;

    function showSlide(index) {
      currentIndex = setActiveSlide(slides, dots, index);
    }

    dots.forEach((dot) => {
      dot.addEventListener("click", () => {
        showSlide(Number.parseInt(dot.dataset.index, 10));
        if (timer) {
          clearInterval(timer);
          timer = setInterval(nextSlide, interval);
        }
      });
    });

    function nextSlide() {
      showSlide(currentIndex + 1);
    }

    if (autoplay) {
      timer = setInterval(nextSlide, interval);
    }
  });
}

/**
 * Smooth Scroll
 */
function initSmoothScroll() {
  document.querySelectorAll('a[href^="#"]').forEach((anchor) => {
    anchor.addEventListener("click", function (e) {
      const href = this.getAttribute("href");
      if (href === "#") return;

      const target = document.querySelector(href);
      if (target) {
        e.preventDefault();
        target.scrollIntoView({
          behavior: "smooth",
        });
      }
    });
  });
}

/**
 * Scroll Effects
 */
function initScrollEffects() {
  const revealElements = document.querySelectorAll(".reveal-on-scroll");

  const revealObserver = new IntersectionObserver(
    (entries) => {
      entries.forEach((entry) => {
        if (entry.isIntersecting) {
          entry.target.classList.add("revealed");
          revealObserver.unobserve(entry.target);
        }
      });
    },
    {
      threshold: 0.1,
    },
  );

  revealElements.forEach((el) => revealObserver.observe(el));
}

/**
 * Testimonials Slider
 */
function initTestimonialsSlider() {
  const tracks = document.querySelectorAll(".testimonials-track");
  tracks.forEach(function (track) {
    const auto = track.dataset.auto === "1";
    const intervalSec = Number.parseInt(track.dataset.interval || "5", 10);

    const getCardWidth = () => {
      const card = track.querySelector(".testimonial-card");
      if (!card) return 320;
      const rect = card.getBoundingClientRect();
      return Math.round(rect.width || 320);
    };

    const getGap = () => {
      const computed = globalThis.getComputedStyle(track);
      const gapValue = computed.columnGap || computed.gap || "20";
      const gap = Number.parseFloat(gapValue);
      return Number.isFinite(gap) ? gap : 20;
    };

    const getStep = () => getCardWidth() + getGap();
    let animationId = 0;

    function smoothScrollTo(targetLeft, duration) {
      animationId += 1;
      const currentAnimation = animationId;
      const maxLeft = Math.max(0, track.scrollWidth - track.clientWidth);
      const clampedTarget = Math.max(0, Math.min(maxLeft, targetLeft));
      const startLeft = track.scrollLeft;
      const distance = clampedTarget - startLeft;
      const startTime = performance.now();
      const effectiveDuration = Math.max(
        200,
        Math.min(700, duration + Math.abs(distance) * 0.3),
      );

      function stepFrame(now) {
        if (currentAnimation !== animationId) {
          return;
        }
        const elapsed = now - startTime;
        const progress = Math.min(1, elapsed / effectiveDuration);
        const eased = easeInOut(progress);
        track.scrollLeft = startLeft + distance * eased;
        if (progress < 1) {
          requestAnimationFrame(stepFrame);
        }
      }

      requestAnimationFrame(stepFrame);
    }

    track.addEventListener(
      "wheel",
      function (e) {
        if (Math.abs(e.deltaY) > Math.abs(e.deltaX)) {
          e.preventDefault();
          const step = getStep();
          const maxLeft = Math.max(0, track.scrollWidth - track.clientWidth);
          const direction = Math.sign(e.deltaY) || 1;
          const target = Math.max(
            0,
            Math.min(maxLeft, track.scrollLeft + direction * step),
          );
          track.scrollLeft = target;
        }
      },
      { passive: false },
    );

    if (!auto || Number.isNaN(intervalSec)) {
      return;
    }

    let timer = setInterval(
      function () {
        if (track.scrollWidth <= track.clientWidth) return;
        const step = getStep();
        const maxIndex = Math.max(
          0,
          Math.floor((track.scrollWidth - track.clientWidth) / step),
        );
        const currentIndex = Math.round(track.scrollLeft / step);
        let nextIndex = currentIndex;
        if (maxIndex > 0) {
          while (nextIndex === currentIndex) {
            nextIndex = Math.floor(Math.random() * (maxIndex + 1));
          }
        }
        const targetLeft = nextIndex * step;
        smoothScrollTo(targetLeft, 450);
      },
      Math.max(2, intervalSec) * 1000,
    );

    track.addEventListener("mouseenter", function () {
      clearInterval(timer);
    });
    track.addEventListener("mouseleave", function () {
      timer = setInterval(
        function () {
          if (track.scrollWidth <= track.clientWidth) return;
          const step = getStep();
          const maxIndex = Math.max(
            0,
            Math.floor((track.scrollWidth - track.clientWidth) / step),
          );
          const currentIndex = Math.round(track.scrollLeft / step);
          let nextIndex = currentIndex;
          if (maxIndex > 0) {
            while (nextIndex === currentIndex) {
              nextIndex = Math.floor(Math.random() * (maxIndex + 1));
            }
          }
          const targetLeft = nextIndex * step;
          smoothScrollTo(targetLeft, 450);
        },
        Math.max(2, intervalSec) * 1000,
      );
    });
  });
}

/**
 * Product Media Slider
 */
function initProductMediaSlider() {
  const sliders = document.querySelectorAll(".product-media-slider");
  sliders.forEach(function (slider) {
    const wrapper =
      slider.closest(".product-media-wrapper") || slider.parentElement;
    const track = slider.querySelector(".product-media-track");
    const slides = slider.querySelectorAll(".product-media-slide");
    const dots = wrapper.querySelectorAll(".product-media-dot");
    if (!track || slides.length === 0) return;

    let currentIndex = 0;
    const auto = slider.dataset.auto === "1";
    const pauseOnHover = slider.dataset.hoverPause !== "0";
    const intervalSec = Number.parseInt(slider.dataset.interval || "4", 10);
    let timer = null;

    function updateDots() {
      dots.forEach(function (dot, idx) {
        dot.classList.toggle("active", idx === currentIndex);
      });
    }

    function goTo(index, behavior = "smooth") {
      currentIndex = (index + slides.length) % slides.length;
      // Skip clientWidth read when scrolling to index 0 (always left: 0)
      // to avoid a forced synchronous reflow during init.
      const left = currentIndex === 0 ? 0 : slider.clientWidth * currentIndex;
      slider.scrollTo({ left, behavior });
      updateDots();
    }

    if (dots.length) {
      dots.forEach(function (dot) {
        dot.addEventListener("click", function () {
          const target = Number.parseInt(dot.dataset.slide || "0", 10);
          goTo(target);
          if (timer) {
            clearInterval(timer);
            timer = null;
          }
        });
      });
    }

    if (auto && slides.length > 1 && !Number.isNaN(intervalSec)) {
      timer = setInterval(
        function () {
          goTo(currentIndex + 1);
        },
        Math.max(3, intervalSec) * 1000,
      );

      if (pauseOnHover) {
        slider.addEventListener("mouseenter", function () {
          if (timer) {
            clearInterval(timer);
            timer = null;
          }
        });
        slider.addEventListener("mouseleave", function () {
          if (!timer) {
            timer = setInterval(
              function () {
                goTo(currentIndex + 1);
              },
              Math.max(3, intervalSec) * 1000,
            );
          }
        });
      }
    }

    let scrollRaf = 0;
    slider.addEventListener("scroll", () => {
      if (scrollRaf) {
        cancelAnimationFrame(scrollRaf);
      }
      scrollRaf = requestAnimationFrame(() => {
        const width = slider.clientWidth || 1;
        const nextIndex = Math.round(slider.scrollLeft / width);
        if (nextIndex !== currentIndex) {
          currentIndex = nextIndex;
          updateDots();
        }
      });
    });

    goTo(0, "auto");
  });
}

function initProductMediaLightbox() {
  const slides = Array.from(document.querySelectorAll(".product-media-slide"));
  if (!slides.length) return;

  const YT_ID_PATTERNS = [
    /(?:youtu\.be\/|youtube(?:-nocookie)?\.com\/(?:watch\?v=|embed\/|shorts\/))([A-Za-z0-9_-]{11})/,
    /[?&]v=([A-Za-z0-9_-]{11})/,
    /\/embed\/([A-Za-z0-9_-]{11})/,
    /^([A-Za-z0-9_-]{11})$/,
  ];

  const extractYouTubeId = (url) => {
    if (!url) return "";
    const candidate = String(url);
    for (const pattern of YT_ID_PATTERNS) {
      const match = pattern.exec(candidate);
      if (match?.[1]) {
        return match[1];
      }
    }
    return "";
  };

  const buildCleanYouTubeEmbedUrl = (entry) => {
    const id =
      extractYouTubeId(entry.youtubeUrl || "") ||
      extractYouTubeId(entry.src || "");
    if (id === "") {
      return entry.src || "";
    }

    const params = new URLSearchParams({
      autoplay: "1",
      mute: "1",
      loop: "1",
      playlist: id,
      controls: "0",
      disablekb: "1",
      modestbranding: "1",
      rel: "0",
      playsinline: "1",
      showinfo: "0",
      iv_load_policy: "3",
      fs: "0",
      cc_load_policy: "0",
      autohide: "1",
      start: "0",
    });

    return `https://www.youtube-nocookie.com/embed/${id}?${params.toString()}`;
  };

  const getYouTubeWatchUrl = (slide, iframe, embedSrc) => {
    const explicit =
      slide.dataset.youtubeUrl || iframe.dataset.youtubeUrl || "";
    if (explicit !== "") {
      return explicit;
    }

    const embedMatch = /\/embed\/([A-Za-z0-9_-]{11})/.exec(embedSrc);
    if (embedMatch?.[1]) {
      return `https://www.youtube.com/watch?v=${embedMatch[1]}`;
    }

    return "https://www.youtube.com/";
  };

  const mediaEntries = slides.map((slide) => {
    const image = slide.querySelector("img");
    if (image) {
      const src = image.getAttribute("src") || image.currentSrc || "";
      if (src !== "") {
        return {
          type: "image",
          src,
          alt: image.getAttribute("alt") || "Product image",
        };
      }
    }

    const video = slide.querySelector("video");
    if (video) {
      const sourceEl = video.querySelector("source");
      const src =
        sourceEl?.getAttribute("src") || video.getAttribute("src") || "";
      if (src !== "") {
        return {
          type: "video",
          src,
          alt: "Product video",
        };
      }
    }

    const iframe = slide.querySelector("iframe");
    if (iframe) {
      const src = iframe.getAttribute("src") || "";
      if (src !== "") {
        return {
          type: "youtube",
          src,
          alt: "Product video",
          youtubeUrl: getYouTubeWatchUrl(slide, iframe, src),
        };
      }
    }

    return null;
  });

  const validMediaEntries = mediaEntries.filter((item) => item !== null);
  if (!validMediaEntries.length) return;

  let overlay = document.querySelector(".product-lightbox");
  if (!overlay) {
    overlay = document.createElement("div");
    overlay.className = "product-lightbox";
    overlay.innerHTML = `
            <button type="button" class="product-lightbox-close" aria-label="Close">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M6 6l12 12"></path>
                    <path d="M18 6l-12 12"></path>
                </svg>
            </button>
            <button type="button" class="product-lightbox-nav product-lightbox-prev" aria-label="Previous media">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M15 18l-6-6 6-6"/></svg>
            </button>
            <button type="button" class="product-lightbox-nav product-lightbox-next" aria-label="Next media">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M9 6l6 6-6 6"/></svg>
            </button>
            <div class="product-lightbox-stage"></div>
        `;
    document.body.appendChild(overlay);
  }

  const closeBtn = overlay.querySelector(".product-lightbox-close");
  const prevBtn = overlay.querySelector(".product-lightbox-prev");
  const nextBtn = overlay.querySelector(".product-lightbox-next");
  const stage = overlay.querySelector(".product-lightbox-stage");
  const zoomSteps = [100, 120, 140, 160, 180, 200, 250, 300, 350, 400];
  let zoomIndex = 0;
  let zoomClass = "";
  let currentIndex = 0;
  let currentType = "image";

  const clearStage = () => {
    if (!stage) return;
    stage.innerHTML = "";
    overlay.classList.remove("is-zoomed");
  };

  const buildStageMedia = (entry) => {
    if (!stage || !entry) return null;

    let el = null;
    if (entry.type === "image") {
      el = document.createElement("img");
      el.className = "product-lightbox-img product-lightbox-media";
      el.src = entry.src;
      el.alt = entry.alt || "Product image";
    } else if (entry.type === "video") {
      el = document.createElement("video");
      el.className = "product-lightbox-video product-lightbox-media";
      el.src = entry.src;
      el.controls = true;
      el.autoplay = true;
      el.loop = true;
      el.playsInline = true;
      el.preload = "metadata";
    } else if (entry.type === "youtube") {
      const wrapper = document.createElement("div");
      wrapper.className = "product-lightbox-youtube product-lightbox-media";

      const iframeEl = document.createElement("iframe");
      iframeEl.className = "product-lightbox-iframe";
      iframeEl.src = buildCleanYouTubeEmbedUrl(entry);
      iframeEl.setAttribute(
        "allow",
        "autoplay; encrypted-media; picture-in-picture",
      );
      iframeEl.setAttribute("allowfullscreen", "");
      iframeEl.setAttribute("loading", "lazy");
      iframeEl.setAttribute("title", entry.alt || "Product video");
      wrapper.appendChild(iframeEl);

      const topCover = document.createElement("span");
      topCover.className =
        "product-lightbox-youtube-cover product-lightbox-youtube-cover--top";
      wrapper.appendChild(topCover);

      const bottomCover = document.createElement("span");
      bottomCover.className =
        "product-lightbox-youtube-cover product-lightbox-youtube-cover--bottom";
      wrapper.appendChild(bottomCover);

      const youtubeLink = document.createElement("a");
      youtubeLink.className = "product-lightbox-youtube-link";
      youtubeLink.href = entry.youtubeUrl || "https://www.youtube.com/";
      youtubeLink.target = "_blank";
      youtubeLink.rel = "noopener noreferrer";
      youtubeLink.title = "Open on YouTube";
      youtubeLink.setAttribute("aria-label", "Open original video on YouTube");
      youtubeLink.innerHTML = `
                <svg viewBox="0 0 24 24" fill="currentColor" aria-hidden="true" focusable="false">
                    <path d="M23.498 6.186a2.997 2.997 0 0 0-2.11-2.12C19.523 3.5 12 3.5 12 3.5s-7.523 0-9.388.566a2.997 2.997 0 0 0-2.11 2.12C0 8.063 0 12 0 12s0 3.937.502 5.814a2.997 2.997 0 0 0 2.11 2.12C4.477 20.5 12 20.5 12 20.5s7.523 0 9.388-.566a2.997 2.997 0 0 0 2.11-2.12C24 15.937 24 12 24 12s0-3.937-.502-5.814ZM9.75 15.568V8.432L16.05 12l-6.3 3.568Z"/>
                </svg>
            `;
      wrapper.appendChild(youtubeLink);

      el = wrapper;
    }

    if (el) {
      stage.appendChild(el);
    }

    return el;
  };

  const applyZoom = () => {
    const lightboxImg = stage
      ? stage.querySelector(".product-lightbox-img")
      : null;
    if (!lightboxImg || currentType !== "image") {
      overlay.classList.remove("is-zoomed");
      return;
    }

    const value = zoomSteps[zoomIndex] || 100;
    const nextClass = `zoom-${value}`;
    if (zoomClass !== nextClass) {
      if (zoomClass !== "") {
        lightboxImg.classList.remove(zoomClass);
      }
      lightboxImg.classList.add(nextClass);
      zoomClass = nextClass;
    }
    overlay.classList.toggle("is-zoomed", value > 100);
  };

  const setImage = (index) => {
    const entry = validMediaEntries[index];
    if (!entry) return;

    clearStage();
    zoomClass = "";
    buildStageMedia(entry);

    currentIndex = index;
    currentType = entry.type;
    zoomIndex = 0;
    applyZoom();
  };

  const openLightbox = (index) => {
    if (!validMediaEntries[index]) return;
    setImage(index);
    overlay.classList.add("is-open");
    document.body.classList.add("product-lightbox-open");
    const multi = validMediaEntries.length > 1;
    overlay.classList.toggle("is-multi", multi);
  };

  const showNext = (direction) => {
    if (validMediaEntries.length <= 1) return;
    const total = validMediaEntries.length;
    const nextIndex = (currentIndex + direction + total) % total;
    setImage(nextIndex);
  };

  const closeLightbox = () => {
    overlay.classList.remove("is-open", "is-zoomed", "is-multi");
    document.body.classList.remove("product-lightbox-open");
    clearStage();
    zoomClass = "";
    zoomIndex = 0;
    currentType = "image";
  };

  slides.forEach((slide, index) => {
    if (slide.dataset.lightboxBound === "1") {
      return;
    }
    slide.dataset.lightboxBound = "1";

    slide.addEventListener("click", (event) => {
      event.preventDefault();

      const entry = mediaEntries[index];
      if (!entry) {
        return;
      }

      const targetIndex = validMediaEntries.indexOf(entry);
      if (targetIndex >= 0) {
        openLightbox(targetIndex);
      }
    });
  });

  if (closeBtn) {
    closeBtn.addEventListener("click", closeLightbox);
  }
  if (prevBtn) {
    prevBtn.addEventListener("click", (event) => {
      event.stopPropagation();
      showNext(-1);
    });
  }
  if (nextBtn) {
    nextBtn.addEventListener("click", (event) => {
      event.stopPropagation();
      showNext(1);
    });
  }

  overlay.addEventListener("click", (event) => {
    if (event.target === overlay) {
      closeLightbox();
    }
  });

  overlay.addEventListener(
    "wheel",
    (event) => {
      if (!overlay.classList.contains("is-open")) return;
      if (currentType !== "image") return;

      event.preventDefault();
      const direction = event.deltaY > 0 ? -1 : 1;
      const nextIndex = Math.min(
        zoomSteps.length - 1,
        Math.max(0, zoomIndex + direction),
      );
      if (nextIndex !== zoomIndex) {
        zoomIndex = nextIndex;
        applyZoom();
      }
    },
    { passive: false },
  );

  document.addEventListener("keydown", (event) => {
    if (!overlay.classList.contains("is-open")) return;
    if (event.key === "Escape") {
      closeLightbox();
    } else if (event.key === "ArrowRight") {
      showNext(1);
    } else if (event.key === "ArrowLeft") {
      showNext(-1);
    }
  });
}

function initCollectionsCarouselScroll() {
  const carousels = document.querySelectorAll(
    ".collection-list-grid.is-carousel",
  );
  carousels.forEach((carousel) => {
    if (carousel.dataset.wheelBound === "1") return;
    carousel.dataset.wheelBound = "1";

    const getCardWidth = () => {
      const card = carousel.querySelector(".collection-list-card");
      if (!card) return 320;
      const rect = card.getBoundingClientRect();
      return Math.round(rect.width || 320);
    };

    const getGap = () => {
      const computed = globalThis.getComputedStyle(carousel);
      const gapValue = computed.columnGap || computed.gap || "24";
      const gap = Number.parseFloat(gapValue);
      return Number.isFinite(gap) ? gap : 24;
    };

    const getStep = () => getCardWidth() + getGap();

    carousel.addEventListener(
      "wheel",
      (event) => {
        const maxLeft = carousel.scrollWidth - carousel.clientWidth;
        if (maxLeft <= 0) return;

        const isHorizontal = Math.abs(event.deltaX) > Math.abs(event.deltaY);
        const isTrackpad = event.deltaMode === 0 && Math.abs(event.deltaY) < 40;
        let delta = 0;

        if (isHorizontal) {
          delta = event.deltaX;
          if (event.deltaMode === 1) {
            delta *= 16;
          } else if (event.deltaMode === 2) {
            delta *= carousel.clientWidth;
          }
        } else if (isTrackpad) {
          delta = event.deltaY;
        } else {
          delta = (Math.sign(event.deltaY) || 1) * getStep();
        }

        if (!delta) return;

        event.preventDefault();
        const nextLeft = Math.max(
          0,
          Math.min(maxLeft, carousel.scrollLeft + delta),
        );
        carousel.scrollLeft = nextLeft;
      },
      { passive: false },
    );
  });
}

function initCheckout() {
  const checkoutButton = document.querySelector("[data-checkout-start]");
  if (!checkoutButton) return;

  const errorSlot = document.querySelector("[data-checkout-error]");

  checkoutButton.addEventListener("click", async function () {
    if (checkoutButton.disabled) return;
    checkoutButton.disabled = true;
    checkoutButton.classList.add("is-loading");

    if (errorSlot) {
      errorSlot.classList.add("is-hidden");
      errorSlot.textContent = "";
    }

    try {
      const bodyObj = {};
      if (csrfToken) {
        bodyObj.csrf_token = csrfToken;
      }

      const response = await fetch("/checkout-api", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(bodyObj),
      });
      const data = await response.json();
      if (!response.ok || !data.url) {
        throw new Error(data.error || "Unable to start checkout.");
      }
      globalThis.location.href = data.url;
    } catch (error) {
      if (errorSlot) {
        errorSlot.textContent = error.message;
        errorSlot.classList.remove("is-hidden");
      }
    } finally {
      checkoutButton.disabled = false;
      checkoutButton.classList.remove("is-loading");
    }
  });
}

/**
 * Chat Image Lightbox
 * Opens chat images in a fullscreen overlay
 */
function initChatImageLightbox() {
  // Create lightbox element if it doesn't exist
  let lightbox = document.querySelector(".chat-lightbox");
  if (!lightbox) {
    lightbox = document.createElement("div");
    lightbox.className = "chat-lightbox";
    lightbox.innerHTML = `
            <button class="chat-lightbox-close" aria-label="Close">&times;</button>
            <img class="chat-lightbox-img" src="" alt="Full size image" width="800" height="600">
        `;
    document.body.appendChild(lightbox);
  }

  const lightboxImg = lightbox.querySelector(".chat-lightbox-img");
  const closeBtn = lightbox.querySelector(".chat-lightbox-close");

  // Close lightbox on button click
  closeBtn.addEventListener("click", function () {
    lightbox.classList.remove("is-open");
  });

  // Close lightbox on background click
  lightbox.addEventListener("click", function (e) {
    if (e.target === lightbox) {
      lightbox.classList.remove("is-open");
    }
  });

  // Close on Escape key
  document.addEventListener("keydown", function (e) {
    if (e.key === "Escape" && lightbox.classList.contains("is-open")) {
      lightbox.classList.remove("is-open");
    }
  });

  // Delegate click events to chat images
  document.addEventListener("click", function (e) {
    if (e.target.classList.contains("chat-image")) {
      e.preventDefault();
      lightboxImg.src = e.target.src;
      lightbox.classList.add("is-open");
    }
  });
}

function initTicketChatPolling() {
  const container = document.querySelector("[data-ticket-chat]");
  if (!container) {
    return;
  }

  const chatBody = container.querySelector("[data-ticket-chat-body]");
  const replyForm = container.querySelector("[data-ticket-reply-form]");
  const previewEl = container.querySelector(".ticket-list-preview");
  const timeEl = container.querySelector(".ticket-list-time");
  const ticketLink = container.querySelector("[data-ticket-link]");
  const toggleButton = container.querySelector("[data-ticket-toggle-list]");

  if (!chatBody) {
    return;
  }

  const search = new URLSearchParams(globalThis.location.search);
  const isOpen =
    search.get("ticket") === "1" || container.dataset.selectedOpen === "1";
  if (!isOpen) {
    return;
  }

  container.classList.add("is-collapsed");
  const clearMessageCenterIndicators = () => {
    document
      .querySelectorAll("[data-message-center-tab] .message-center-indicator")
      .forEach((el) => el.remove());
    globalThis.__ticketUnreadOverride = 0;
    globalThis.__ticketUnreadOverrideUntil = Date.now() + 30000;
  };
  clearMessageCenterIndicators();

  let lastId = Number.parseInt(chatBody.dataset.ticketLastId || "0", 10);
  let isFetching = false;

  function renderMessages(items) {
    const sorted = items
      .slice()
      .sort((a, b) => (a.created_at || 0) - (b.created_at || 0));
    chatBody.innerHTML = "";
    sorted.forEach((item) => {
      const isAdmin = (item.sender_type || "admin") === "admin";
      const bubble = document.createElement("div");
      bubble.className = `ticket-bubble ${isAdmin ? "ticket-bubble--admin" : "ticket-bubble--user"}`;
      bubble.dataset.ticketMessageId = String(item.id);

      const meta = document.createElement("div");
      meta.className = "ticket-bubble-meta";
      const date = item.created_at ? new Date(item.created_at * 1000) : null;

      // Add read receipt checkmark for user's own messages
      let readReceipt = "";
      if (!isAdmin) {
        if (item.admin_read) {
          readReceipt =
            '<span class="read-receipt read-receipt--read" title="Seen"><svg viewBox="0 0 16 16" fill="currentColor"><path d="M13.854 3.646a.5.5 0 0 1 0 .708l-7 7a.5.5 0 0 1-.708 0l-3.5-3.5a.5.5 0 1 1 .708-.708L6.5 10.293l6.646-6.647a.5.5 0 0 1 .708 0z"/></svg></span>';
        } else {
          readReceipt =
            '<span class="read-receipt read-receipt--sent" title="Sent"><svg viewBox="0 0 16 16" fill="currentColor"><path d="M13.854 3.646a.5.5 0 0 1 0 .708l-7 7a.5.5 0 0 1-.708 0l-3.5-3.5a.5.5 0 1 1 .708-.708L6.5 10.293l6.646-6.647a.5.5 0 0 1 .708 0z"/></svg></span>';
        }
      }
      meta.innerHTML = `${isAdmin ? "Admin" : "You"} · ${date ? date.toLocaleString() : ""} ${readReceipt}`;

      bubble.appendChild(meta);

      // Render image if present
      if (item.image_url) {
        const imageWrap = document.createElement("div");
        imageWrap.className = "ticket-bubble-image";
        const img = document.createElement("img");
        img.src = item.image_url;
        img.alt = "Attached image";
        img.className = "chat-image";
        img.loading = "lazy";
        imageWrap.appendChild(img);
        bubble.appendChild(imageWrap);
      }

      const text = document.createElement("div");
      text.className = "ticket-bubble-text";
      text.textContent = item.body || "";

      bubble.appendChild(text);
      chatBody.appendChild(bubble);
    });

    if (sorted.length) {
      const last = sorted[sorted.length - 1];
      lastId = last.id;
      chatBody.dataset.ticketLastId = String(lastId);
      if (previewEl) {
        const preview = (last.body || "").trim().slice(0, 80);
        previewEl.textContent =
          preview + (last.body && last.body.length > 80 ? "…" : "");
      }
      if (timeEl && last.created_at) {
        const date = new Date(last.created_at * 1000);
        timeEl.textContent = date.toLocaleString();
      }
    }

    // Auto-scroll to latest message
    chatBody.scrollTop = chatBody.scrollHeight;
  }

  async function fetchMessages() {
    if (isFetching) {
      return;
    }
    isFetching = true;
    try {
      // Mark ticket messages as read when chat is expanded (visible)
      const shouldMarkRead = container.classList.contains("is-collapsed");

      // Explicitly use GET for fetching messages
      const response = await fetch("/messages-api?type=ticket&limit=200", {
        method: "GET",
        credentials: "same-origin",
      });
      const data = await response.json();
      if (data?.success && Array.isArray(data?.items)) {
        // Always re-render to update read receipts, not just when new messages
        renderMessages(data.items);
        if (shouldMarkRead) {
          const markData = new URLSearchParams();
          markData.append("mark_read", "ticket");
          if (csrfToken) {
            markData.append("csrf_token", csrfToken);
          }
          await fetch("/messages-api", {
            method: "POST",
            headers: { "Content-Type": "application/x-www-form-urlencoded" },
            body: markData.toString(),
            credentials: "same-origin",
          }).catch(() => {});

          const messageCenterTab = document.querySelector(
            "[data-message-center-tab]",
          );
          if (messageCenterTab) {
            messageCenterTab
              .querySelectorAll(".message-center-indicator")
              .forEach((el) => el.remove());
          }
          fetch("/notifications-api?action=get&force=1")
            .then((r) => r.json())
            .then((payload) => {
              const unreadTotal = Object.hasOwn(payload || {}, "unread_total")
                ? payload.unread_total
                : (payload?.unread || 0) + (payload?.ticket_unread || 0);
              if (messageCenterTab) {
                if (unreadTotal > 0) {
                  if (
                    !messageCenterTab.querySelector(".message-center-indicator")
                  ) {
                    const indicator = document.createElement("span");
                    indicator.className = "message-center-indicator";
                    messageCenterTab.appendChild(indicator);
                  }
                } else {
                  messageCenterTab
                    .querySelectorAll(".message-center-indicator")
                    .forEach((el) => el.remove());
                }
              }
            })
            .catch(() => {});
        }
      }
    } catch (error) {
      debugCatch("ticket polling fetch failed", error);
    } finally {
      isFetching = false;
    }
  }

  if (replyForm) {
    const fileInput = replyForm.querySelector('input[name="chat_image"]');
    const previewWrap = replyForm.querySelector("[data-upload-preview]");
    const previewImg = previewWrap
      ? previewWrap.querySelector(".upload-preview-img")
      : null;
    const removeBtn = replyForm.querySelector("[data-remove-preview]");

    // Handle image preview
    if (fileInput) {
      fileInput.addEventListener("change", function () {
        const file = fileInput.files[0];
        if (file && previewWrap && previewImg) {
          // Validate file size (5MB)
          if (file.size > 5 * 1024 * 1024) {
            alert("Image must be under 5MB");
            fileInput.value = "";
            previewWrap.classList.add("is-hidden-display");
            return;
          }
          // Validate type
          const validTypes = [
            "image/jpeg",
            "image/png",
            "image/gif",
            "image/webp",
          ];
          if (!validTypes.includes(file.type)) {
            alert("Only JPG, PNG, GIF and WEBP images are allowed");
            fileInput.value = "";
            previewWrap.classList.add("is-hidden-display");
            return;
          }
          const reader = new FileReader();
          reader.onload = function (e) {
            previewImg.src = e.target.result;
            previewWrap.classList.remove("is-hidden-display");
          };
          reader.readAsDataURL(file);
        } else if (previewWrap) {
          previewWrap.classList.add("is-hidden-display");
        }
      });
    }

    // Remove preview button
    if (removeBtn) {
      removeBtn.addEventListener("click", function () {
        if (fileInput) {
          fileInput.value = "";
        }
        if (previewWrap) {
          previewWrap.classList.add("is-hidden-display");
        }
      });
    }

    const applyReplyRateLimitState = (data) => {
      const textarea = replyForm.querySelector('textarea[name="reply_body"]');
      if (!textarea) return;
      const originalPlaceholder = textarea.placeholder;
      textarea.placeholder = data.error || "Slow down! Please wait a moment.";
      textarea.classList.add("rate-limited");
      setTimeout(() => {
        textarea.placeholder = originalPlaceholder;
        textarea.classList.remove("rate-limited");
      }, 3000);
    };

    const applyReplySuccessState = async (data, submitBtn) => {
      const textarea = replyForm.querySelector('textarea[name="reply_body"]');
      if (textarea) {
        textarea.value = "";
        textarea.placeholder = "Write your reply...";
      }

      if (fileInput) {
        fileInput.value = "";
      }
      previewWrap?.classList.add("is-hidden-display");

      const actionInput = replyForm.querySelector('input[name="action"]');
      if (actionInput?.value === "start_ticket") {
        actionInput.value = "reply_message";
        const replyToInput = replyForm.querySelector(
          'input[name="reply_to_id"]',
        );
        if (replyToInput && data?.item?.id) {
          replyToInput.value = data.item.id;
        }
      }

      if (submitBtn) {
        submitBtn.textContent = "Send Reply";
      }

      chatBody.querySelector(".ticket-empty-state")?.remove();
      container
        .querySelector("[data-download-chat]")
        ?.classList.remove("is-hidden-display");
      container
        .querySelector("[data-clear-chat]")
        ?.classList.remove("is-hidden-display");

      await fetchMessages();
    };

    replyForm.addEventListener("submit", async function (event) {
      event.preventDefault();
      const submitBtn = replyForm.querySelector('button[type="submit"]');
      const originalBtnText = submitBtn ? submitBtn.textContent : "Send";

      if (submitBtn) {
        submitBtn.disabled = true;
        submitBtn.textContent = "Sending...";
      }

      const formData = new FormData(replyForm);
      try {
        const response = await fetch("/messages-api", {
          method: "POST",
          body: formData,
          credentials: "same-origin",
        });
        if (!response.ok) {
          throw new Error(`HTTP error! status: ${response.status}`);
        }

        const data = await response.json();
        if (data?.success) {
          await applyReplySuccessState(data, submitBtn);
          return;
        }
        if (data?.rate_limited) {
          applyReplyRateLimitState(data);
          return;
        }
        if (data?.error) {
          alert(data.error);
        }
      } catch (error_) {
        debugCatch("Chat submit error", error_);
        console.error("Chat submit error:", error_);
        alert(
          "Failed to send message. Please check your connection and try again.",
        );
      } finally {
        if (submitBtn) {
          submitBtn.disabled = false;
          submitBtn.textContent = originalBtnText;
        }
      }
    });
  }

  if (ticketLink) {
    ticketLink.addEventListener("click", (event) => {
      event.preventDefault();
      // Re-read URL to check current state
      const currentSearch = new URLSearchParams(globalThis.location.search);
      if (currentSearch.get("ticket") === "1") {
        // Already open, toggle collapse
        container.classList.toggle("is-collapsed");
        if (!container.classList.contains("is-collapsed")) {
          // Collapsed back to list view, remove ticket param
          const url = new URL(globalThis.location.href);
          url.searchParams.delete("ticket");
          globalThis.history.pushState({}, "", url.toString());
        }
      } else {
        // Expand the ticket chat without page reload
        container.classList.add("is-collapsed");
        const url = new URL(globalThis.location.href);
        url.searchParams.set("ticket", "1");
        globalThis.history.pushState({}, "", url.toString());
        clearMessageCenterIndicators();
        // Fetch messages and scroll to bottom
        fetchMessages();
        chatBody.scrollTop = chatBody.scrollHeight;
      }
    });
  }

  if (toggleButton) {
    toggleButton.addEventListener("click", () => {
      container.classList.remove("is-collapsed");
      // Update URL to remove ticket parameter for consistent state
      const url = new URL(globalThis.location.href);
      url.searchParams.delete("ticket");
      globalThis.history.replaceState({}, "", url.toString());
    });
  }

  // Download chat transcript button
  const downloadBtn = container.querySelector("[data-download-chat]");
  if (downloadBtn) {
    downloadBtn.addEventListener("click", async function () {
      downloadBtn.disabled = true;
      const originalText = downloadBtn.innerHTML;
      downloadBtn.innerHTML =
        '<span class="inline-spinner">⏳</span> Downloading...';

      try {
        const response = await fetch("/messages-api?download=1", {
          credentials: "same-origin",
        });

        // Check if response is OK
        if (!response.ok) {
          console.error(
            "Download response not OK:",
            response.status,
            response.statusText,
          );
          alert(
            "Download failed (status " +
              response.status +
              "). Please try again.",
          );
          return;
        }

        const contentType = (
          response.headers.get("content-type") || ""
        ).toLowerCase();
        console.log("Download content-type:", contentType);

        if (contentType.includes("application/json")) {
          // Rate limited or error - parse as JSON
          const data = await response.json();
          if (!data.success) {
            alert(data.error || "Download failed. Please try again.");
          }
        } else {
          // Assume it's the file download (text/plain or similar)
          const blob = await response.blob();
          console.log("Download blob size:", blob.size);
          if (blob.size > 0) {
            const url = URL.createObjectURL(blob);
            const a = document.createElement("a");
            a.href = url;
            a.download =
              "my_chat_transcript_" +
              new Date().toISOString().slice(0, 10) +
              ".txt";
            a.classList.add("is-hidden-display");
            document.body.appendChild(a);
            a.click();
            // Small delay before cleanup
            setTimeout(() => {
              a.remove();
              URL.revokeObjectURL(url);
            }, 100);
          } else {
            alert("No messages to download.");
          }
        }
      } catch (error_) {
        debugCatch("Download error", error_);
        console.error("Download error:", error_);
        alert("Download failed: " + (error_.message || "Unknown error"));
      } finally {
        downloadBtn.disabled = false;
        downloadBtn.innerHTML = originalText;
      }
    });
  }

  // Clear chat button
  const clearBtn = container.querySelector("[data-clear-chat]");
  if (clearBtn) {
    const applyClearedChatState = () => {
      chatBody.innerHTML =
        '<div class="text-muted ticket-empty-state">No messages yet. Send a message to start chatting with support!</div>';
      downloadBtn?.classList.add("is-hidden-display");
      clearBtn.classList.add("is-hidden-display");

      const actionInput = replyForm?.querySelector('input[name="action"]');
      if (actionInput) {
        actionInput.value = "start_ticket";
      }

      const replyToInput = replyForm?.querySelector(
        'input[name="reply_to_id"]',
      );
      if (replyToInput) {
        replyToInput.value = "0";
      }

      const submitBtn = replyForm?.querySelector('button[type="submit"]');
      if (submitBtn) {
        submitBtn.textContent = "Send Message";
      }
    };

    const requestClearChat = async () => {
      const formData = new FormData();
      formData.append("action", "clear_chat");
      const response = await fetch("/messages-api", {
        method: "POST",
        body: formData,
        credentials: "same-origin",
      });
      return response.json();
    };

    clearBtn.addEventListener("click", async function () {
      if (
        !confirm(
          "Are you sure you want to clear your entire chat history? This cannot be undone.",
        )
      ) {
        return;
      }

      clearBtn.disabled = true;
      const originalText = clearBtn.innerHTML;
      clearBtn.innerHTML = '<span class="inline-spinner">⏳</span> Clearing...';

      try {
        const data = await requestClearChat();
        if (data.success) {
          applyClearedChatState();
          return;
        }
        alert(data.error || "Failed to clear chat. Please try again.");
      } catch (error_) {
        debugCatch("Clear chat error", error_);
        console.error("Clear chat error:", error_);
        alert("Failed to clear chat. Please try again.");
      } finally {
        clearBtn.disabled = false;
        clearBtn.innerHTML = originalText;
      }
    });
  }

  // Initial scroll to bottom for server-rendered messages
  chatBody.scrollTop = chatBody.scrollHeight;

  fetchMessages();
  setInterval(fetchMessages, 5000);
}

/**
 * Product Share Button
 */
function initProductShareButtons() {
  const buttons = document.querySelectorAll(".product-share-btn");
  if (!buttons.length) return;

  buttons.forEach(function (btn) {
    btn.addEventListener("click", function () {
      const title = btn.dataset.shareTitle || document.title;
      const url = globalThis.location.href;

      if (navigator.share) {
        navigator.share({ title, url }).catch(function () {});
        return;
      }

      if (navigator.clipboard?.writeText) {
        navigator.clipboard.writeText(url).then(function () {
          const original = btn.textContent;
          btn.textContent = "Copied";
          globalThis.setTimeout(restoreButtonText, 1600, btn, original);
        });
      }
    });
  });
}

/**
 * Product Like/Dislike Buttons
 */
function initProductLikeButtons() {
  const wrap = document.querySelector("[data-product-like]");
  if (!wrap) return;

  const likeBtn = wrap.querySelector(".product-like-btn");
  const likeCountEl = wrap.querySelector(
    ".product-like-btn .product-like-count",
  );
  const productId = wrap.dataset.productId || "";
  if (!likeBtn || !productId) return;

  let busy = false;
  let cooldownTimer = null;

  const parseCount = (value) => {
    const num = Number.parseInt(String(value || "0"), 10);
    return Number.isNaN(num) ? 0 : num;
  };

  const applyState = (state, likes, dislikes) => {
    const likeValue =
      typeof likes === "number" ? likes : parseCount(wrap.dataset.likeCount);
    const dislikeValue =
      typeof dislikes === "number"
        ? dislikes
        : parseCount(wrap.dataset.dislikeCount);
    const netValue = likeValue - dislikeValue;

    wrap.dataset.likeState = state;
    wrap.dataset.likeCount = String(likeValue);
    wrap.dataset.dislikeCount = String(dislikeValue);

    if (likeCountEl) likeCountEl.textContent = String(netValue);

    likeBtn.classList.toggle("is-active", state !== "none");
    wrap.classList.toggle("is-liked", state === "like");
    wrap.classList.toggle("is-disliked", state === "dislike");
    const isPressed = state === "none" ? "false" : "true";
    likeBtn.setAttribute("aria-pressed", isPressed);
  };

  const setDisabled = (disabled) => {
    likeBtn.disabled = disabled;
    wrap.classList.toggle("is-disabled", disabled);
  };

  const handleRateLimit = (retryAfter) => {
    const retry = Number.parseInt(String(retryAfter || "60"), 10);
    console.warn(`Like rate limit hit. Try again in ${retry}s.`);
    setDisabled(true);
    if (cooldownTimer) {
      clearTimeout(cooldownTimer);
    }
    cooldownTimer = setTimeout(
      () => {
        setDisabled(false);
      },
      Math.max(1, retry) * 1000,
    );
  };

  const sendAction = (action) => {
    if (busy) return;
    busy = true;
    setDisabled(true);

    const formData = new FormData();
    formData.append("product_id", productId);
    formData.append("action", action);
    if (csrfToken) {
      formData.append("csrf_token", csrfToken);
    }

    fetch("/product-likes-api", {
      method: "POST",
      body: formData,
      credentials: "same-origin",
    })
      .then((r) => r.json())
      .then((data) => {
        if (data?.success) {
          applyState(data.state || "none", data.likes, data.dislikes);
          setDisabled(false);
          return;
        }
        if (data?.rate_limited) {
          handleRateLimit(data.retry_after || 60);
          return;
        }
        console.warn("Like update failed.", data?.error || "Unknown error");
        setDisabled(false);
      })
      .catch((err) => {
        console.warn("Like update failed.", err);
        setDisabled(false);
      })
      .finally(() => {
        busy = false;
      });
  };

  let longPressTimer = null;
  let longPressFired = false;

  likeBtn.addEventListener("click", (event) => {
    if (longPressFired) {
      longPressFired = false;
      return;
    }
    if (event.shiftKey) {
      sendAction("dislike");
      return;
    }
    sendAction("like");
  });

  likeBtn.addEventListener("contextmenu", (event) => {
    event.preventDefault();
    sendAction("dislike");
  });

  likeBtn.addEventListener(
    "touchstart",
    () => {
      longPressFired = false;
      if (longPressTimer) {
        clearTimeout(longPressTimer);
      }
      longPressTimer = setTimeout(() => {
        longPressFired = true;
        sendAction("dislike");
      }, 500);
    },
    { passive: true },
  );

  likeBtn.addEventListener("touchend", () => {
    if (longPressTimer) {
      clearTimeout(longPressTimer);
      longPressTimer = null;
    }
  });

  likeBtn.addEventListener("touchcancel", () => {
    if (longPressTimer) {
      clearTimeout(longPressTimer);
      longPressTimer = null;
    }
  });

  applyState(wrap.dataset.likeState || "none");
}

/**
 * Product Variant Selector
 */
function initProductVariantSelector() {
  document.querySelectorAll(".product-meta-card").forEach(function (card) {
    const priceEl = card.querySelector(".product-price-value");
    const priceWrap = card.querySelector(".product-detail-price");
    const basePrice = priceWrap?.dataset.basePrice || "";
    const cartBtn = card.querySelector(".product-cart-btn");
    const downloadBtn = card.querySelector(".product-download-btn");
    const linkBtn = card.querySelector(".product-link-btn");
    const qtyWrap = card.querySelector("[data-product-qty]");
    const digitalBadge = card.querySelector("[data-digital-badge]");
    const baseAction = card.dataset.baseAction || "cart";
    const baseDownload = card.dataset.baseDownload || "";
    const baseLink = card.dataset.baseLink || "";
    const baseLinkLabel = card.dataset.baseLinkLabel || "";
    const baseDigital = card.dataset.baseDigital === "1";
    const productOwned = card.dataset.productOwned === "1";
    const ownedMessage = card.dataset.ownedMessage || "You already have this.";
    const baseStock = cartBtn?.dataset.stock || "";
    const baseUnlimited = cartBtn?.dataset.unlimited === "1";
    if (!priceEl) return;

    function setCartAvailability(stockValue, unlimited, ownedLock) {
      if (!cartBtn) return;
      if (ownedLock) {
        cartBtn.disabled = true;
        cartBtn.classList.add("is-disabled");
        cartBtn.textContent = "Already Owned";
        cartBtn.setAttribute("title", ownedMessage);
        cartBtn.setAttribute("aria-disabled", "true");
        return;
      }
      cartBtn.classList.remove("is-disabled");
      cartBtn.removeAttribute("title");
      cartBtn.removeAttribute("aria-disabled");
      const outOfStock =
        !unlimited &&
        stockValue !== "" &&
        Number.parseInt(stockValue, 10) === 0;
      cartBtn.disabled = outOfStock;
      cartBtn.classList.toggle("is-out-of-stock", outOfStock);
      cartBtn.textContent = outOfStock ? "Out of Stock" : "Add to Cart";
    }

    function setDownloadState(url) {
      if (!downloadBtn) return;
      if (url) {
        downloadBtn.classList.remove("is-disabled");
        downloadBtn.setAttribute("href", url);
        downloadBtn.setAttribute("download", "");
        downloadBtn.setAttribute("aria-disabled", "false");
        downloadBtn.textContent = "Download";
      } else {
        downloadBtn.classList.add("is-disabled");
        downloadBtn.setAttribute("href", "#");
        downloadBtn.removeAttribute("download");
        downloadBtn.setAttribute("aria-disabled", "true");
        downloadBtn.textContent = "Download Unavailable";
      }
    }

    function setLinkState(url, label) {
      if (!linkBtn) return;
      if (url) {
        linkBtn.classList.remove("is-disabled");
        linkBtn.setAttribute("href", url);
        linkBtn.setAttribute("target", "_blank");
        linkBtn.setAttribute("rel", "noopener");
        linkBtn.setAttribute("aria-disabled", "false");
        linkBtn.textContent = label || baseLinkLabel || "Open link";
      } else {
        linkBtn.classList.add("is-disabled");
        linkBtn.setAttribute("href", "#");
        linkBtn.removeAttribute("target");
        linkBtn.removeAttribute("rel");
        linkBtn.setAttribute("aria-disabled", "true");
        linkBtn.textContent = "Link unavailable";
      }
    }

    function updateDigitalBadge(isPaidDigital, isExternal) {
      digitalBadge?.classList.toggle("is-hidden", !isPaidDigital && !isExternal);
      if (!digitalBadge) return;
      const badgeLabel = digitalBadge.querySelector("[data-badge-label]");
      const iconDownload = digitalBadge.querySelector(".badge-icon-download");
      const iconExternal = digitalBadge.querySelector(".badge-icon-external");
      if (badgeLabel) {
        badgeLabel.textContent = isExternal ? "External Link" : "Digital Download";
      }
      iconDownload?.classList.toggle("is-hidden", isExternal);
      iconExternal?.classList.toggle("is-hidden", !isExternal);
    }

    function setActionState(variant) {
      const {
        variantAction,
        variantDownload,
        variantPrice,
        variantStock,
        variantUnlimited,
        variantDigital,
        variantOwned,
        variantLinkLabel,
      } = variant;

      const resolvedPrice = variantPrice === "" ? basePrice : variantPrice;
      const priceValue = parsePriceValue(resolvedPrice);
      const isDownload = variantAction === "download" && priceValue <= 0;
      const isExternal = variantAction === "external_link";
      const isPaidDigital = variantDigital && priceValue > 0;
      const ownedLock = variantOwned && isPaidDigital;

      cartBtn?.classList.toggle("is-hidden", isDownload || isExternal);
      downloadBtn?.classList.toggle("is-hidden", !isDownload);
      linkBtn?.classList.toggle("is-hidden", !isExternal);
      qtyWrap?.classList.toggle(
        "is-hidden",
        isDownload || variantDigital || isExternal,
      );
      updateDigitalBadge(isPaidDigital, isExternal);
      priceWrap?.classList.toggle("is-hidden", isExternal);

      if (isDownload) {
        setDownloadState(variantDownload);
      }
      if (isExternal) {
        setLinkState(variantDownload || baseLink, variantLinkLabel);
      }

      let nextPriceText = resolvedPrice;
      if (isExternal) {
        nextPriceText = "";
      } else if (isDownload) {
        nextPriceText = "Free";
      }
      priceEl.textContent = nextPriceText;

      if (isDownload || isExternal) {
        return;
      }

      let effectiveStock = variantStock;
      if (variantUnlimited) {
        effectiveStock = "";
      }
      let stockValueForCart = effectiveStock;
      if (stockValueForCart === "") {
        stockValueForCart = baseStock;
      }
      setCartAvailability(
        stockValueForCart,
        variantUnlimited || baseUnlimited,
        ownedLock,
      );
    }

    card.querySelectorAll(".product-variant-select").forEach(function (select) {
      select.addEventListener("change", function () {
        const selected = select.options[select.selectedIndex];
        const variantData = resolveVariantData(selected, {
          variantAction: baseAction,
          variantDownload: baseDownload,
          variantDigital: baseDigital,
          variantOwned: productOwned,
        });
        setActionState(variantData);
      });

      const selected = select.options[select.selectedIndex];
      const variantData = resolveVariantData(selected, {
        variantAction: baseAction,
        variantDownload: baseDownload,
        variantDigital: baseDigital,
        variantOwned: productOwned,
      });
      setActionState(variantData);
    });

    const buttons = card.querySelectorAll(".product-variant-btn");
    if (buttons.length) {
      buttons.forEach(function (btn) {
        const variantStock = btn.dataset.stock || "";
        const variantUnlimited = btn.dataset.unlimited === "1";
        const outOfStock =
          !variantUnlimited &&
          variantStock !== "" &&
          Number.parseInt(variantStock, 10) === 0;
        btn.classList.toggle("is-out-of-stock", outOfStock);
      });
      buttons.forEach(function (btn) {
        btn.addEventListener("click", function () {
          setActiveVariantButton(buttons, btn);
          const variantData = resolveVariantData(btn, {
            variantAction: baseAction,
            variantDownload: baseDownload,
            variantDigital: baseDigital,
            variantOwned: productOwned,
          });
          const outOfStock =
            !variantData.variantUnlimited &&
            variantData.variantStock !== "" &&
            Number.parseInt(variantData.variantStock, 10) === 0;
          btn.classList.toggle("is-out-of-stock", outOfStock);
          setActionState(variantData);
        });
      });
    }

    const activeBtn = card.querySelector(".product-variant-btn.active");
    if (activeBtn) {
      const variantData = resolveVariantData(activeBtn, {
        variantAction: baseAction,
        variantDownload: baseDownload,
        variantDigital: baseDigital,
        variantOwned: productOwned,
      });
      setActionState(variantData);
    } else {
      setActionState({
        variantAction: baseAction,
        variantDownload: baseDownload,
        variantPrice: basePrice,
        variantStock: baseStock,
        variantUnlimited: baseUnlimited,
        variantDigital: baseDigital,
        variantOwned: productOwned,
        variantLinkLabel: "",
      });
    }
  });
}

/**
 * Cart Sidebar
 */
function initCartSidebar() {
  const sidebar = document.getElementById("cart-sidebar");
  if (!sidebar) return;

  const closeButtons = sidebar.querySelectorAll("[data-cart-close]");
  closeButtons.forEach((btn) => {
    btn.addEventListener("click", () => {
      sidebar.classList.remove("open");
      sidebar.setAttribute("aria-hidden", "true");
    });
  });

  sidebar.addEventListener("click", (e) => {
    const qtyBtn = e.target.closest(".cart-mini-qty-btn");
    if (!qtyBtn) return;
    const item = qtyBtn.closest(".cart-mini-item");
    if (!item) return;
    const input = item.querySelector(".cart-mini-qty-input");
    const current = Number.parseInt(input.value || "1", 10);
    const delta = Number.parseInt(qtyBtn.dataset.qty || "0", 10);
    const max = item.dataset.quantityMax;
    let next = current + delta;
    if (max) {
      next = Math.min(next, Number.parseInt(max, 10));
    }
    next = Math.max(1, next);
    input.value = String(next);
    updateCartItem(item.dataset.itemId, next, "sidebar");
  });

  sidebar.addEventListener("click", (e) => {
    const removeBtn = e.target.closest("[data-remove]");
    if (!removeBtn) return;
    const item = removeBtn.closest(".cart-mini-item");
    if (!item) return;
    updateCartItem(item.dataset.itemId, 0, "sidebar");
  });

  sidebar.addEventListener("change", (e) => {
    const input = e.target.closest(".cart-mini-qty-input");
    if (!input) return;
    const item = input.closest(".cart-mini-item");
    const max = item.dataset.quantityMax;
    let next = Number.parseInt(input.value || "1", 10);
    if (max) {
      next = Math.min(next, Number.parseInt(max, 10));
    }
    next = Math.max(1, next);
    input.value = String(next);
    updateCartItem(item.dataset.itemId, next, "sidebar");
  });
}

function openCartSidebar() {
  const sidebar = document.getElementById("cart-sidebar");
  if (!sidebar) return;
  sidebar.classList.add("open");
  sidebar.setAttribute("aria-hidden", "false");
}

function loadCartSidebar() {
  const body = document.body;
  const enabled = body.dataset.cartSidebar === "1";
  if (!enabled) return;
  const container = document.getElementById("cart-sidebar-body");
  const totalEl = document.getElementById("cart-sidebar-total");
  if (!container || !totalEl) return;

  const formData = new FormData();
  formData.append("action", "get");
  if (csrfToken) {
    formData.append("csrf_token", csrfToken);
  }

  fetch("/cart-api", {
    method: "POST",
    body: formData,
  })
    .then(parseCartApiResponse)
    .then((data) => {
      updateCartBadgeFromItems(data.items || []);
      if (!data.items.length) {
        container.innerHTML =
          '<div class="cart-empty">Your cart is empty.</div>';
        totalEl.textContent = data.total || "$0";
        return;
      }

      container.innerHTML = data.items
        .map((item) => {
          const variant = item.variant_label
            ? `<div class="cart-item-variant">${item.variant_label}</div>`
            : "";
          let media = "";
          if (item.media_url && item.product_url) {
            media = `<div class="cart-item-thumb"><a href="${item.product_url}"><img src="${item.media_url}" alt="${item.name}"></a></div>`;
          } else if (item.media_url) {
            media = `<div class="cart-item-thumb"><img src="${item.media_url}" alt="${item.name}"></div>`;
          }
          const quantityMax =
            item.quantity_max === null ? "" : item.quantity_max;
          const showQtyControls = item.quantity_max !== 1 && !item.is_digital;
          const digitalBadge = item.is_digital
            ? `<div class="cart-digital-badge"><svg xmlns="http://www.w3.org/2000/svg" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/><polyline points="7 10 12 15 17 10"/><line x1="12" y1="15" x2="12" y2="3"/></svg><span>Digital</span></div>`
            : "";
          let qtyControls = "";
          if (showQtyControls) {
            qtyControls = `
                    <div class="cart-mini-qty-controls">
                        <button type="button" class="cart-mini-qty-btn" data-qty="-1">-</button>
                        <input type="number" class="cart-mini-qty-input" min="1" value="${item.quantity}">
                        <button type="button" class="cart-mini-qty-btn" data-qty="1">+</button>
                    </div>
                `;
          } else if (!item.is_digital) {
            qtyControls = `<div class="cart-mini-qty">Qty: ${item.quantity}</div>`;
          }
          return `
                    <div class="cart-mini-item" data-item-id="${item.id}" data-quantity-max="${quantityMax}">
                        ${media}
                        <div class="cart-mini-info">
                            <div class="cart-mini-name">${item.product_url ? `<a href="${item.product_url}" class="cart-item-link">${item.name}</a>` : item.name}</div>
                            ${variant}
                            ${digitalBadge}
                            ${qtyControls}
                        </div>
                        <div class="cart-mini-price">${item.line_total}</div>
                        <button type="button" class="cart-mini-remove" data-remove aria-label="Remove item">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                <path d="M3 6h18"></path>
                                <path d="M8 6V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path>
                                <path d="M19 6l-1 14a2 2 0 0 1-2 2H8a2 2 0 0 1-2-2L5 6"></path>
                                <path d="M10 11v6"></path>
                                <path d="M14 11v6"></path>
                            </svg>
                        </button>
                    </div>
                `;
        })
        .join("");

      totalEl.textContent = data.total;
    })
    .catch((error) => {
      if (document.visibilityState === "visible") {
        console.debug("Failed to load cart sidebar:", error);
      }
    });
}

function initCartToggle() {
  document.querySelectorAll("[data-cart-toggle]").forEach((link) => {
    link.addEventListener("click", (e) => {
      const sidebarEnabled = document.body.dataset.cartSidebar === "1";
      if (!sidebarEnabled) return;
      e.preventDefault();
      loadCartSidebar();
      openCartSidebar();
    });
  });
}

function updateCartBadgeFromItems(items) {
  const count = (items || []).reduce(
    (sum, item) => sum + Number.parseInt(item.quantity || 0, 10),
    0,
  );
  const badges = document.querySelectorAll("[data-cart-count]");
  const hideWhenEmpty = document.body.dataset.cartIconMode === "items";
  const cartIcons = document.querySelectorAll("[data-cart-icon]");
  badges.forEach((badge) => {
    badge.textContent = String(count);
    badge.classList.toggle("is-hidden", count <= 0);
  });
  cartIcons.forEach((icon) => {
    icon.classList.toggle("is-hidden", hideWhenEmpty && count <= 0);
  });
}

function parseCartApiResponse(response) {
  return response.text().then((rawText) => {
    const text = (rawText || "").trim();
    let data = null;

    if (text !== "") {
      try {
        data = JSON.parse(text);
      } catch (error) {
        debugCatch("parseCartApiResponse JSON.parse failed", error);
        throw new Error(
          `Cart response was not valid JSON (HTTP ${response.status}).`,
        );
      }
    }

    if (!response.ok) {
      const message =
        data && (data.message || data.error)
          ? data.message || data.error
          : `Cart request failed (HTTP ${response.status}).`;
      throw new Error(message);
    }

    if (!data || typeof data !== "object") {
      throw new Error("Cart response was empty. Please try again.");
    }

    if (data.success === false) {
      throw new Error(data.message || data.error || "Cart request failed.");
    }

    return data;
  });
}

function refreshCartBadge() {
  const formData = new FormData();
  formData.append("action", "get");

  // Add CSRF token for POST request if available
  if (csrfToken) {
    formData.append("csrf_token", csrfToken);
  }

  fetch("/cart-api", {
    method: "POST",
    body: formData,
  })
    .then(parseCartApiResponse)
    .then((data) => {
      updateCartBadgeFromItems(data.items || []);
    })
    .catch((e) => {
      if (document.visibilityState === "visible") {
        console.debug("Failed to refresh cart badge:", e);
      }
    });
}

/**
 * Cart Actions
 */
function initCartActions() {
  document.querySelectorAll(".product-cart-btn").forEach((btn) => {
    btn.addEventListener("click", () => {
      if (btn.disabled) return;
      const productId = btn.dataset.productId;
      const quantityMax = btn.dataset.quantityMax;
      const qtyInput = btn
        .closest(".product-meta-card")
        ?.querySelector(".product-qty-input");
      const qtyValue = qtyInput
        ? Math.max(1, Number.parseInt(qtyInput.value || "1", 10))
        : 1;
      let variantId = null;

      const card = btn.closest(".product-meta-card");
      if (card) {
        const activeBtn = card.querySelector(".product-variant-btn.active");
        if (activeBtn) {
          variantId = activeBtn.dataset.variantId;
        } else {
          const select = card.querySelector(".product-variant-select");
          if (select?.options?.length) {
            const selected = select.options[select.selectedIndex];
            variantId = selected?.dataset?.variantId;
          }
        }
      }

      const formData = new FormData();
      formData.append("action", "add");
      formData.append("product_id", productId || "");
      if (variantId) {
        formData.append("variant_id", variantId);
      }
      formData.append("quantity", quantityMax === "1" ? "1" : String(qtyValue));

      fetch("/cart-api", {
        method: "POST",
        body: formData,
      })
        .then(parseCartApiResponse)
        .then((data) => {
          updateCartBadgeFromItems(data.items || []);
          const sidebarEnabled = document.body.dataset.cartSidebar === "1";
          if (sidebarEnabled) {
            loadCartSidebar();
            openCartSidebar();
          } else {
            globalThis.location.href = "/cart";
          }
        })
        .catch((error) => {
          console.warn("Add to cart failed:", error);
          alert(
            error.message || "Unable to add item to cart. Please try again.",
          );
        });
    });
  });

  document.querySelectorAll(".product-qty-controls").forEach((wrapper) => {
    wrapper.addEventListener("click", (e) => {
      const btn = e.target.closest(".product-qty-btn");
      if (!btn) return;
      const input = wrapper.querySelector(".product-qty-input");
      if (!input) return;
      const current = Number.parseInt(input.value || "1", 10);
      const delta = Number.parseInt(btn.dataset.qty || "0", 10);
      let next = current + delta;
      next = Math.max(1, next);
      input.value = String(next);
    });

    wrapper.addEventListener("change", (e) => {
      const input = e.target.closest(".product-qty-input");
      if (!input) return;
      let next = Number.parseInt(input.value || "1", 10);
      next = Math.max(1, next);
      input.value = String(next);
    });
  });
}

/**
 * Cart Page
 */
function initCartPage() {
  const cartPage = document.querySelector(".cart-page");
  if (!cartPage) return;

  cartPage.addEventListener("click", (e) => {
    const removeBtn = e.target.closest("[data-remove]");
    if (removeBtn) {
      const item = removeBtn.closest(".cart-item");
      const itemId = item?.dataset?.itemId;
      if (!itemId) return;
      updateCartItem(itemId, 0);
      return;
    }

    const qtyBtn = e.target.closest(".cart-qty-btn");
    if (qtyBtn) {
      const item = qtyBtn.closest(".cart-item");
      if (!item) return;
      const input = item.querySelector(".cart-qty-input");
      const current = Number.parseInt(input.value || "1", 10);
      const delta = Number.parseInt(qtyBtn.dataset.qty || "0", 10);
      const max = item.dataset.quantityMax;
      let next = current + delta;
      if (max) {
        next = Math.min(next, Number.parseInt(max, 10));
      }
      next = Math.max(1, next);
      input.value = String(next);
      updateCartItem(item.dataset.itemId, next, "page");
    }
  });

  cartPage.addEventListener("change", (e) => {
    const input = e.target.closest(".cart-qty-input");
    if (!input) return;
    const item = input.closest(".cart-item");
    const max = item.dataset.quantityMax;
    let next = Number.parseInt(input.value || "1", 10);
    if (max) {
      next = Math.min(next, Number.parseInt(max, 10));
    }
    next = Math.max(1, next);
    input.value = String(next);
    updateCartItem(item.dataset.itemId, next, "page");
  });
}

/**
 * Products List Filters / Sort / View
 */
function initProductsListControls() {
  document.querySelectorAll("[data-products-list]").forEach((section) => {
    const grid = section.querySelector("[data-products-grid]");
    if (!grid) return;
    const cards = Array.from(grid.querySelectorAll(".product-list-card"));
    const sortDropdown = section.querySelector("[data-products-sort]");
    const sortTrigger = sortDropdown?.querySelector("[data-sort-trigger]");
    const sortMenu = sortDropdown?.querySelector("[data-sort-menu]");
    const sortLabel = sortDropdown?.querySelector("[data-sort-label]");
    const viewToggle = section.querySelector("[data-view-toggle]");
    const filterInputs = section.querySelectorAll("[data-filter]");
    const filtersToggle = section.querySelector("[data-filters-toggle]");
    const filtersPanel = section.querySelector(".products-filters");
    const countLabel = section.querySelector("[data-products-count]");

    const parseNumber = (value) => {
      const num = Number.parseFloat(value || "0");
      return Number.isNaN(num) ? 0 : num;
    };

    const readAttrs = (card) => {
      const raw = card.dataset.attrs;
      if (!raw) return {};
      try {
        return JSON.parse(raw);
      } catch (error) {
        debugCatch("Invalid product attribute JSON", error);
        return {};
      }
    };

    const matchesAvailability = (inStock, selectedAvailability) => {
      if (!selectedAvailability.length) {
        return true;
      }
      return selectedAvailability.some(
        (value) => (value === "in" && inStock) || (value === "out" && !inStock),
      );
    };

    const matchesAttributes = (card, selectedAttributes) => {
      const groups = Object.keys(selectedAttributes);
      if (!groups.length) {
        return true;
      }
      const attrs = readAttrs(card);
      for (const group of groups) {
        const groupValues = attrs[group] || [];
        const selectedSet = selectedAttributes[group];
        if (!hasMatchingSelectedProductAttribute(groupValues, selectedSet)) {
          return false;
        }
      }
      return true;
    };

    const isCardVisible = (card, selectedAvailability, selectedAttributes) => {
      const inStock = card.dataset.inStock === "1";
      return (
        matchesAvailability(inStock, selectedAvailability) &&
        matchesAttributes(card, selectedAttributes)
      );
    };

    const applyFilters = () => {
      const selectedAvailability = Array.from(
        section.querySelectorAll('[data-filter="availability"]:checked'),
      ).map((i) => i.value);
      const selectedAttributes = {};
      section
        .querySelectorAll('[data-filter="attr"]:checked')
        .forEach((input) => {
          const group = input.dataset.filterGroup || "";
          if (!group) return;
          if (!selectedAttributes[group]) {
            selectedAttributes[group] = new Set();
          }
          selectedAttributes[group].add(input.value);
        });

      cards.forEach((card) => {
        card.classList.toggle(
          "is-hidden-display",
          !isCardVisible(card, selectedAvailability, selectedAttributes),
        );
      });

      if (countLabel) {
        const visibleCount = cards.filter(
          (card) => !card.classList.contains("is-hidden-display"),
        ).length;
        countLabel.textContent = `Showing ${visibleCount} of ${cards.length} Products`;
      }
    };

    const getSortValue = () => {
      if (sortDropdown) {
        return sortDropdown.dataset.sortValue || "featured";
      }
      return "featured";
    };

    const applySort = () => {
      const value = getSortValue();
      const sorted = [...cards].sort((a, b) => {
        if (value === "featured") {
          const fa = parseNumber(a.dataset.featured);
          const fb = parseNumber(b.dataset.featured);
          if (fb !== fa) return fb - fa;
          return parseNumber(b.dataset.date) - parseNumber(a.dataset.date);
        }
        if (value === "bestselling") {
          return (
            parseNumber(b.dataset.bestselling) -
            parseNumber(a.dataset.bestselling)
          );
        }
        if (value === "price_asc") {
          return parseNumber(a.dataset.price) - parseNumber(b.dataset.price);
        }
        if (value === "price_desc") {
          return parseNumber(b.dataset.price) - parseNumber(a.dataset.price);
        }
        if (value === "date_old") {
          return parseNumber(a.dataset.date) - parseNumber(b.dataset.date);
        }
        return parseNumber(b.dataset.date) - parseNumber(a.dataset.date);
      });

      sorted.forEach((card) => grid.appendChild(card));
    };

    if (sortTrigger && sortMenu && sortDropdown) {
      sortTrigger.addEventListener("click", (e) => {
        e.preventDefault();
        sortDropdown.classList.toggle("open");
      });

      sortMenu.querySelectorAll("[data-sort-option]").forEach((option) => {
        option.addEventListener("click", () => {
          const value = option.dataset.sortOption || "featured";
          sortDropdown.dataset.sortValue = value;
          if (sortLabel) {
            sortLabel.textContent = option.textContent || "Featured";
          }
          sortDropdown.classList.remove("open");
          applySort();
          section.classList.add("sort-has-value");
        });
      });

      document.addEventListener("click", (event) => {
        if (!sortDropdown.contains(event.target)) {
          sortDropdown.classList.remove("open");
        }
      });

      applySort();
    }

    if (filterInputs.length) {
      filterInputs.forEach((input) => {
        input.addEventListener("change", applyFilters);
      });
    }

    for (const resetBtn of section.querySelectorAll("[data-filter-reset]")) {
      resetBtn.addEventListener("click", () => {
        const card = resetBtn.closest(".filters-card");
        if (!card) return;
        for (const input of card.querySelectorAll('input[type="checkbox"]')) {
          input.checked = false;
        }
        applyFilters();
      });
    }

    if (filtersToggle && filtersPanel) {
      const label = section.querySelector("[data-filters-label]");
      const updateLabel = () => {
        if (label) {
          label.classList.toggle("is-on", filtersToggle.checked);
        }
      };
      filtersToggle.addEventListener("change", () => {
        section.classList.toggle("filters-hidden", !filtersToggle.checked);
        updateLabel();
      });
      updateLabel();
    }

    if (viewToggle) {
      const buttons = viewToggle.querySelectorAll(".view-btn");
      for (const btn of buttons) {
        btn.addEventListener("click", () => {
          for (const other of Array.from(buttons)) {
            other.classList.toggle("active", other === btn);
          }
          const view = btn.dataset.view || "grid";
          grid.classList.toggle("view-grid", view === "grid");
          grid.classList.toggle("view-list", view === "list");
        });
      }
    }

    applyFilters();
  });
}

function initProductsListHoverMedia() {
  document.querySelectorAll("[data-products-list]").forEach((section) => {
    if (section.dataset.hoverMedia !== "1") {
      return;
    }
    section
      .querySelectorAll(".product-list-card.has-hover-media")
      .forEach((card) => {
        const hoverMedia = card.dataset.hoverMedia;
        const img = card.querySelector(".product-list-media img");
        if (!hoverMedia || !img) return;
        const defaultMedia = img.dataset.defaultMedia || img.src;

        card.addEventListener("mouseenter", () => {
          img.src = hoverMedia;
        });

        card.addEventListener("mouseleave", () => {
          img.src = defaultMedia;
        });
      });
  });
}

function initMiniBarSelectors() {
  document.querySelectorAll("form[data-mini-dropdown]").forEach((form) => {
    const trigger = form.querySelector("[data-mini-trigger]");
    const menu = form.querySelector("[data-mini-menu]");
    const valueInput = form.querySelector("[data-mini-value]");
    const label = form.querySelector("[data-mini-label]");
    const dropdown = form.querySelector(".mini-dropdown");

    if (!trigger || !menu || !valueInput || !label || !dropdown) return;

    trigger.addEventListener("click", (e) => {
      e.preventDefault();
      dropdown.classList.toggle("open");
    });

    menu.querySelectorAll("[data-mini-option]").forEach((option) => {
      option.addEventListener("click", () => {
        const nextValue = option.dataset.miniOption || "";
        valueInput.value = nextValue;
        label.textContent = option.textContent || nextValue;
        dropdown.classList.remove("open");
        form.submit();
      });
    });

    document.addEventListener("click", (event) => {
      if (!dropdown.contains(event.target)) {
        dropdown.classList.remove("open");
      }
    });
  });
}

function initMiniBarScroll() {
  // When minibar-sticky is enabled, keep the bar visible on scroll
  if (document.body.dataset.minibarSticky === "1") {
    document.body.classList.remove("mini-bar-hidden");
    return;
  }

  let ticking = false;
  const update = () => {
    const shouldHide = globalThis.scrollY > 10;
    document.body.classList.toggle("mini-bar-hidden", shouldHide);
    ticking = false;
  };

  globalThis.addEventListener("scroll", () => {
    if (!ticking) {
      globalThis.requestAnimationFrame(update);
      ticking = true;
    }
  });

  // Defer initial check to avoid forced synchronous reflow
  // during DOMContentLoaded init sequence.
  requestAnimationFrame(update);
}

function initForumCreatePost() {
  const root = document.querySelector("[data-forum-root]");
  if (!root) return;

  const modal = document.querySelector("[data-forum-modal]");
  const openBtns = document.querySelectorAll("[data-forum-open]");
  const alertBox = document.querySelector("[data-forum-alert]");
  if (!modal || !openBtns.length) return;

  const form = modal.querySelector("[data-forum-form]");
  const categorySelect = modal.querySelector("[data-forum-category]");
  const subcategorySelect = modal.querySelector("[data-forum-subcategory]");
  const messageBox = modal.querySelector("[data-forum-message]");
  const submitBtn = modal.querySelector("[data-forum-submit]");
  const closeButtons = modal.querySelectorAll("[data-forum-close]");
  const imagesInput = modal.querySelector("[data-forum-images]");
  const imagesList = modal.querySelector("[data-forum-upload-list]");

  const MAX_FORUM_IMAGES = 5;
  const MAX_FORUM_IMAGE_SIZE = 5 * 1024 * 1024;

  let subcategories = [];
  try {
    subcategories = JSON.parse(root.dataset.forumSubcategories || "[]") || [];
  } catch (error_) {
    debugCatch("initForumCreatePost subcategories parse", error_);
    subcategories = [];
  }

  const subcategoriesByCategory = subcategories.reduce((acc, sub) => {
    const key = String(sub.category_id || "");
    if (!acc[key]) acc[key] = [];
    acc[key].push(sub);
    return acc;
  }, {});

  // Build tree structure from flat array
  const buildTree = (subs, parentId = null) => {
    const branch = [];
    subs.forEach((sub) => {
      const subParentId =
        sub.parent_id !== undefined && sub.parent_id !== null
          ? sub.parent_id
          : null;
      if (subParentId === parentId) {
        const item = { ...sub, children: buildTree(subs, sub.id) };
        branch.push(item);
      }
    });
    return branch;
  };

  // Render subcategory options with indentation
  const renderSubcategoryOptions = (subs, depth = 0) => {
    let html = "";
    subs.forEach((sub) => {
      const indent = "&nbsp;&nbsp;&nbsp;&nbsp;".repeat(depth);
      const prefix = depth > 0 ? "└─ " : "";
      html += `<option value="${sub.id}">${indent}${prefix}${sub.name || "Untitled"}</option>`;
      if (sub.children && sub.children.length > 0) {
        html += renderSubcategoryOptions(sub.children, depth + 1);
      }
    });
    return html;
  };

  const renderSubcategories = (categoryId, selectedId) => {
    if (!subcategorySelect) return;
    const list = subcategoriesByCategory[String(categoryId)] || [];
    const tree = buildTree(list, null);
    subcategorySelect.innerHTML = "";
    if (!tree.length) {
      const option = document.createElement("option");
      option.value = "";
      option.textContent = "No subcategories available";
      subcategorySelect.appendChild(option);
      subcategorySelect.disabled = true;
      return;
    }
    subcategorySelect.disabled = false;
    subcategorySelect.innerHTML = renderSubcategoryOptions(tree, 0);
    if (selectedId) {
      subcategorySelect.value = String(selectedId);
    }
  };

  if (categorySelect) {
    renderSubcategories(categorySelect.value, subcategorySelect?.value || "");
    categorySelect.addEventListener("change", () => {
      renderSubcategories(categorySelect.value);
    });
  }

  if (imagesInput) {
    imagesInput.addEventListener("change", () => {
      const files = imagesInput.files ? Array.from(imagesInput.files) : [];
      if (files.length > MAX_FORUM_IMAGES) {
        if (messageBox) {
          messageBox.textContent = "You can upload up to 5 images per post.";
          messageBox.classList.add("is-error");
        }
        imagesInput.value = "";
        renderSelectedUploadFiles([], imagesList);
        return;
      }
      const tooLarge = files.find((file) => file.size > MAX_FORUM_IMAGE_SIZE);
      if (tooLarge) {
        if (messageBox) {
          messageBox.textContent = "Each image must be 5MB or less.";
          messageBox.classList.add("is-error");
        }
        imagesInput.value = "";
        renderSelectedUploadFiles([], imagesList);
        return;
      }
      if (messageBox) {
        messageBox.textContent = "";
        messageBox.classList.remove("is-error");
      }
      renderSelectedUploadFiles(files, imagesList);
    });
  }

  const openModal = () => {
    modal.classList.add("is-open");
    modal.setAttribute("aria-hidden", "false");
    document.body.classList.add("forum-modal-open");
    if (messageBox) {
      messageBox.textContent = "";
      messageBox.classList.remove("is-error", "is-success");
    }
  };

  const closeModal = () => {
    // Remove focus from any focused element inside the modal before hiding
    const focusedElement = document.activeElement;
    if (focusedElement && modal.contains(focusedElement)) {
      focusedElement.blur();
      // Return focus to the first open button if available
      if (openBtns.length > 0) {
        openBtns[0].focus();
      }
    }

    modal.classList.remove("is-open");
    modal.setAttribute("aria-hidden", "true");
    document.body.classList.remove("forum-modal-open");
  };

  openBtns.forEach((btn) => btn.addEventListener("click", openModal));
  closeButtons.forEach((btn) => btn.addEventListener("click", closeModal));
  modal.addEventListener("click", (event) => {
    if (event.target === modal) {
      closeModal();
    }
  });

  if (!form) return;

  const validateCreatePostImages = () => {
    const files = imagesInput?.files ? Array.from(imagesInput.files) : [];
    const validation = validateImageUploads(
      files,
      MAX_FORUM_IMAGES,
      MAX_FORUM_IMAGE_SIZE,
    );
    if (validation.valid) {
      return true;
    }
    setFormMessage(
      messageBox,
      validation.error
        .replace("images.", "images per post.")
        .replace("5MB", "5MB"),
      "error",
    );
    setLoadingButtonState(submitBtn, false);
    return false;
  };

  const showForumAlert = (text, mode) => {
    if (!alertBox) return;
    alertBox.textContent = text;
    alertBox.classList.remove("is-hidden", "is-error", "is-success");
    alertBox.classList.add(mode === "error" ? "is-error" : "is-success");
    if (mode === "success") {
      globalThis.setTimeout(() => {
        alertBox.classList.add("is-hidden");
      }, 4500);
    }
  };

  const prependCreatedPost = (postHtml) => {
    if (!postHtml) return;
    let postsWrap = document.querySelector("[data-forum-posts]");
    const empty = document.querySelector("[data-forum-empty]");
    if (!postsWrap) {
      const body = document.querySelector(".forum-main .forum-card-body");
      postsWrap = document.createElement("div");
      postsWrap.className = "forum-posts";
      postsWrap.dataset.forumPosts = "";
      if (empty?.parentNode) {
        empty.parentNode.replaceChild(postsWrap, empty);
      } else if (body) {
        body.prepend(postsWrap);
      }
    }
    postsWrap.insertAdjacentHTML("afterbegin", postHtml);
  };

  form.addEventListener("submit", async (event) => {
    event.preventDefault();
    setLoadingButtonState(submitBtn, true);
    setFormMessage(messageBox, "");

    if (!validateCreatePostImages()) {
      return;
    }

    const formData = new FormData(form);
    formData.set("action", "create_post");
    try {
      const response = await fetch("/forum-posts-api", {
        method: "POST",
        body: formData,
        credentials: "same-origin",
      });
      const data = await response.json();
      if (!response.ok || !data.success) {
        console.warn("Forum post create failed.", data);
        const error = data?.error || "Unable to create post.";
        setFormMessage(messageBox, error, "error");
        showForumAlert(error, "error");
        return;
      }

      const successMessage = data.message || "Post submitted.";
      setFormMessage(messageBox, successMessage, "success");
      showForumAlert(successMessage, "success");
      prependCreatedPost(data.post_html);

      form.reset();
      categorySelect && renderSubcategories(categorySelect.value);
      if (imagesList) imagesList.innerHTML = "";
      globalThis.setTimeout(closeModal, 400);
    } catch (error_) {
      debugCatch("initForumCreatePost submit failed", error_);
      setFormMessage(
        messageBox,
        "Unable to create post. Please try again.",
        "error",
      );
    } finally {
      setLoadingButtonState(submitBtn, false);
    }
  });
}

function initForumComments() {
  const form = document.querySelector("[data-forum-comment-form]");
  if (!form) return;

  const commentsWrap = document.querySelector("[data-forum-comments]");
  const emptyState = document.querySelector("[data-forum-empty]");
  const messageBox = form.querySelector("[data-forum-comment-message]");
  const submitBtn = form.querySelector("[data-forum-comment-submit]");
  const repliesCountEl = document.querySelector("[data-forum-replies]");
  const repliesCountHeaderEl = document.querySelector(
    "[data-forum-replies-count]",
  );
  const imagesInput = form.querySelector("[data-forum-comment-images]");
  const imagesList = form.querySelector("[data-forum-comment-upload-list]");

  const MAX_FORUM_IMAGES = 5;
  const MAX_FORUM_IMAGE_SIZE = 5 * 1024 * 1024;

  if (imagesInput) {
    imagesInput.addEventListener("change", () => {
      const files = imagesInput.files ? Array.from(imagesInput.files) : [];
      if (files.length > MAX_FORUM_IMAGES) {
        if (messageBox) {
          messageBox.textContent = "You can upload up to 5 images per reply.";
          messageBox.classList.add("is-error");
        }
        imagesInput.value = "";
        renderSelectedUploadFiles([], imagesList);
        return;
      }
      const tooLarge = files.find((file) => file.size > MAX_FORUM_IMAGE_SIZE);
      if (tooLarge) {
        if (messageBox) {
          messageBox.textContent = "Each image must be 5MB or less.";
          messageBox.classList.add("is-error");
        }
        imagesInput.value = "";
        renderSelectedUploadFiles([], imagesList);
        return;
      }
      if (messageBox) {
        messageBox.textContent = "";
        messageBox.classList.remove("is-error");
      }
      renderSelectedUploadFiles(files, imagesList);
    });
  }

  const validateReplyImages = () => {
    const files = imagesInput?.files ? Array.from(imagesInput.files) : [];
    const validation = validateImageUploads(
      files,
      MAX_FORUM_IMAGES,
      MAX_FORUM_IMAGE_SIZE,
    );
    if (validation.valid) {
      return true;
    }
    setFormMessage(
      messageBox,
      validation.error.replace("images.", "images per reply."),
      "error",
    );
    setLoadingButtonState(submitBtn, false);
    return false;
  };

  const updateRepliesCount = (count) => {
    if (count === undefined) return;
    if (repliesCountEl) {
      repliesCountEl.textContent = String(count);
    }
    if (repliesCountHeaderEl) {
      repliesCountHeaderEl.textContent = String(count);
    }
  };

  form.addEventListener("submit", async (event) => {
    event.preventDefault();
    setLoadingButtonState(submitBtn, true);
    setFormMessage(messageBox, "");

    if (!validateReplyImages()) {
      return;
    }

    const formData = new FormData(form);
    formData.set("action", "create_comment");
    if (!formData.get("post_id")) {
      const fallbackPostId = form.dataset.forumPostId;
      if (fallbackPostId) {
        formData.set("post_id", fallbackPostId);
      }
    }
    try {
      const response = await fetch("/forum-comments-api", {
        method: "POST",
        body: formData,
        credentials: "same-origin",
      });
      const data = await response.json();
      if (!response.ok || !data.success) {
        console.warn("Forum reply failed.", data);
        const error = data?.error || "Unable to post reply.";
        setFormMessage(messageBox, error, "error");
        return;
      }

      if (data.comment_html && commentsWrap) {
        if (emptyState) {
          emptyState.remove();
        }
        commentsWrap.insertAdjacentHTML("beforeend", data.comment_html);
      }

      updateRepliesCount(data.replies_count);

      form.reset();
      if (imagesList) imagesList.innerHTML = "";
      setFormMessage(messageBox, data.message || "Reply posted.", "success");
    } catch (error_) {
      debugCatch("initForumComments submit failed", error_);
      setFormMessage(
        messageBox,
        "Unable to post reply. Please try again.",
        "error",
      );
    } finally {
      setLoadingButtonState(submitBtn, false);
    }
  });

  if (commentsWrap) {
    commentsWrap.addEventListener("click", (event) => {
      const editBtn = event.target.closest("[data-forum-comment-edit]");
      if (editBtn) {
        const comment = editBtn.closest("[data-forum-comment]");
        if (!comment) return;
        const body = comment.querySelector("[data-forum-comment-body]");
        const editForm = comment.querySelector(
          "[data-forum-comment-edit-form]",
        );
        if (body) body.classList.add("is-hidden");
        if (editForm) editForm.classList.remove("is-hidden");
        return;
      }

      const cancelBtn = event.target.closest(
        "[data-forum-comment-edit-cancel]",
      );
      if (cancelBtn) {
        const comment = cancelBtn.closest("[data-forum-comment]");
        if (!comment) return;
        const body = comment.querySelector("[data-forum-comment-body]");
        const editForm = comment.querySelector(
          "[data-forum-comment-edit-form]",
        );
        if (editForm) editForm.classList.add("is-hidden");
        if (body) body.classList.remove("is-hidden");
      }
    });

    commentsWrap.addEventListener("submit", async (event) => {
      const editForm = event.target.closest("[data-forum-comment-edit-form]");
      if (!editForm) return;
      event.preventDefault();

      const comment = editForm.closest("[data-forum-comment]");
      const bodyEl = comment?.querySelector("[data-forum-comment-body]");
      const message = editForm.querySelector(
        "[data-forum-comment-edit-message]",
      );
      const saveBtn = editForm.querySelector("[data-forum-comment-edit-save]");

      const applyCommentImagesState = (data) => {
        if (data.images_html !== undefined) {
          let imagesWrap = comment?.querySelector(".forum-comment-images");
          if (data.images_html && data.images_html.trim() !== "") {
            if (!imagesWrap) {
              imagesWrap = document.createElement("div");
              imagesWrap.className = "forum-comment-images";
              comment
                ?.querySelector(".forum-comment-main")
                ?.appendChild(imagesWrap);
            }
            imagesWrap.innerHTML = data.images_html;
          } else {
            imagesWrap?.remove();
          }
        }

        if (data.images_edit_html !== undefined) {
          const editList = editForm.querySelector(
            "[data-forum-comment-images-edit]",
          );
          if (editList) {
            editList.innerHTML = data.images_edit_html;
          }
        }
      };

      setFormMessage(message, "");
      setLoadingButtonState(saveBtn, true);

      const formData = new FormData(editForm);
      try {
        const response = await fetch("/forum-comments-api", {
          method: "POST",
          body: formData,
          credentials: "same-origin",
        });
        const data = await response.json();
        if (!response.ok || !data.success) {
          const error = data?.error || "Unable to update reply.";
          setFormMessage(message, error, "error");
          return;
        }

        if (bodyEl) {
          const safe = document.createElement("div");
          safe.textContent = data.body || "";
          bodyEl.innerHTML = safe.innerHTML.replaceAll("\n", "<br>");
          bodyEl.classList.remove("is-hidden");
        }
        applyCommentImagesState(data);
        const editFileInput = editForm.querySelector('input[type="file"]');
        if (editFileInput) editFileInput.value = "";
        editForm.classList.add("is-hidden");
      } catch (error_) {
        debugCatch("initForumComments edit failed", error_);
        setFormMessage(
          message,
          "Unable to update reply. Please try again.",
          "error",
        );
      } finally {
        setLoadingButtonState(saveBtn, false);
      }
    });
  }
}

function initForumPostEditing() {
  const editBtn = document.querySelector("[data-forum-post-edit]");
  const editForm = document.querySelector("[data-forum-post-edit-form]");
  const titleEl = document.querySelector("[data-forum-post-title]");
  const metaEl = document.querySelector("[data-forum-post-meta]");
  const bodyEl = document.querySelector("[data-forum-post-body]");
  if (!editBtn || !editForm || !titleEl || !bodyEl) return;

  const message = editForm.querySelector("[data-forum-post-edit-message]");
  const cancelBtn = editForm.querySelector("[data-forum-post-edit-cancel]");
  const saveBtn = editForm.querySelector("[data-forum-post-edit-save]");

  const toggleEdit = (isEditing) => {
    editForm.classList.toggle("is-hidden", !isEditing);
    titleEl.classList.toggle("is-hidden", isEditing);
    if (metaEl) metaEl.classList.toggle("is-hidden", isEditing);
    editBtn.classList.toggle("is-hidden", isEditing);
    bodyEl.classList.toggle("is-hidden", isEditing);
    if (message) {
      message.textContent = "";
      message.classList.remove("is-error", "is-success");
    }
  };

  editBtn.addEventListener("click", () => toggleEdit(true));
  if (cancelBtn) {
    cancelBtn.addEventListener("click", () => toggleEdit(false));
  }

  const applyPostImagesState = (data) => {
    if (data.images_html === undefined) {
      return;
    }
    const container = bodyEl?.parentElement;
    let imagesWrap = document.querySelector(".forum-post-images");
    if (data.images_html && data.images_html.trim() !== "") {
      if (!imagesWrap) {
        imagesWrap = document.createElement("div");
        imagesWrap.className = "forum-post-images";
        container?.appendChild(imagesWrap);
      }
      imagesWrap.innerHTML = data.images_html;
      return;
    }
    imagesWrap?.remove();
  };

  editForm.addEventListener("submit", async (event) => {
    event.preventDefault();
    setLoadingButtonState(saveBtn, true);
    setFormMessage(message, "");

    const formData = new FormData(editForm);
    try {
      const response = await fetch("/forum-posts-api", {
        method: "POST",
        body: formData,
        credentials: "same-origin",
      });
      const data = await response.json();
      if (!response.ok || !data.success) {
        const error = data?.error || "Unable to update post.";
        setFormMessage(message, error, "error");
        return;
      }

      if (data.title) {
        titleEl.textContent = data.title;
      }
      if (bodyEl) {
        const safe = document.createElement("div");
        safe.textContent = data.body || "";
        bodyEl.innerHTML = safe.innerHTML.replaceAll("\n", "<br>");
      }
      applyPostImagesState(data);
      if (data.images_edit_html !== undefined) {
        const editList = editForm.querySelector(
          "[data-forum-post-images-edit]",
        );
        if (editList) {
          editList.innerHTML = data.images_edit_html;
        }
      }
      const editFileInput = editForm.querySelector('input[type="file"]');
      if (editFileInput) editFileInput.value = "";
      toggleEdit(false);
    } catch (error_) {
      debugCatch("initForumPostEditing submit failed", error_);
      setFormMessage(
        message,
        "Unable to update post. Please try again.",
        "error",
      );
    } finally {
      setLoadingButtonState(saveBtn, false);
    }
  });
}

function initForumImageLightbox() {
  const buildLightbox = () => {
    const overlay = document.createElement("div");
    overlay.className = "forum-lightbox";
    overlay.innerHTML = `
            <div class="forum-lightbox-backdrop" data-forum-lightbox-close></div>
            <div class="forum-lightbox-content" role="dialog" aria-modal="true">
                <button type="button" class="forum-lightbox-close" data-forum-lightbox-close aria-label="Close">&times;</button>
                <img class="forum-lightbox-image" alt="Forum image">
            </div>
        `;
    document.body.appendChild(overlay);
    return overlay;
  };

  let overlay = null;
  let lightboxImage = null;
  const zoomSteps = [100, 120, 140, 160, 180, 200, 250, 300, 350, 400];
  let zoomIndex = 0;
  let zoomClass = "zoom-100";

  const applyZoom = () => {
    if (!lightboxImage) return;
    const value = zoomSteps[zoomIndex] || 100;
    const nextClass = `zoom-${value}`;
    if (zoomClass !== nextClass) {
      lightboxImage.classList.remove(zoomClass);
      lightboxImage.classList.add(nextClass);
      zoomClass = nextClass;
    }
  };

  const openLightbox = (src) => {
    if (!overlay) {
      overlay = buildLightbox();
      lightboxImage = overlay.querySelector(".forum-lightbox-image");
      overlay.addEventListener("click", (event) => {
        if (
          event.target instanceof HTMLElement &&
          event.target.dataset.forumLightboxClose !== undefined
        ) {
          closeLightbox();
        }
      });
      overlay.addEventListener(
        "wheel",
        (event) => {
          if (!lightboxImage || !overlay.classList.contains("is-open")) return;
          event.preventDefault();
          const direction = event.deltaY > 0 ? -1 : 1;
          const nextIndex = Math.min(
            zoomSteps.length - 1,
            Math.max(0, zoomIndex + direction),
          );
          if (nextIndex !== zoomIndex) {
            zoomIndex = nextIndex;
            applyZoom();
          }
        },
        { passive: false },
      );
    }
    if (lightboxImage) {
      lightboxImage.src = src;
      zoomIndex = 0;
      lightboxImage.classList.add(zoomClass);
      applyZoom();
    }
    overlay.classList.add("is-open");
    document.body.classList.add("forum-lightbox-open");
  };

  const closeLightbox = () => {
    if (!overlay) return;
    overlay.classList.remove("is-open");
    document.body.classList.remove("forum-lightbox-open");
  };

  document.addEventListener("click", (event) => {
    const target = event.target;
    const button = target?.closest?.("[data-forum-lightbox]");
    if (!button) return;
    const src = button.dataset.forumLightbox;
    if (src) {
      openLightbox(src);
    }
  });

  document.addEventListener("keydown", (event) => {
    if (event.key === "Escape") {
      closeLightbox();
    }
  });
}

function initBackToTop() {
  const button = document.querySelector("[data-back-to-top]");
  if (!button) return;

  // Toggle visibility on the button itself (not body) to limit
  // style recalculation scope and avoid forced body-level reflow.
  let ticking = false;
  const update = () => {
    button.classList.toggle("visible", globalThis.scrollY > 300);
    ticking = false;
  };

  globalThis.addEventListener(
    "scroll",
    () => {
      if (!ticking) {
        globalThis.requestAnimationFrame(update);
        ticking = true;
      }
    },
    { passive: true },
  );

  button.addEventListener("click", () => {
    globalThis.scrollTo({ top: 0, behavior: "smooth" });
  });

  // Defer initial check to avoid forced synchronous reflow
  // during DOMContentLoaded init sequence.
  requestAnimationFrame(update);
}

function initNotificationMenu() {
  const toggles = document.querySelectorAll("[data-notify-toggle]");
  if (!toggles.length) return;

  const closeAll = () => {
    document.querySelectorAll(".notify-wrapper.is-open").forEach((wrapper) => {
      wrapper.classList.remove("is-open");
    });
    document.querySelectorAll(".account-wrapper.is-open").forEach((wrapper) => {
      wrapper.classList.remove("is-open");
    });
  };

  toggles.forEach((toggle) => {
    const wrapper = toggle.closest(".notify-wrapper");
    if (!wrapper) return;

    toggle.addEventListener("click", (event) => {
      event.preventDefault();
      event.stopPropagation();
      const isOpen = wrapper.classList.contains("is-open");
      closeAll();
      if (!isOpen) {
        wrapper.classList.add("is-open");
      }
    });

    if (wrapper.closest(".sidebar-footer")) {
      wrapper.addEventListener("mouseleave", () => {
        wrapper.classList.remove("is-open");
      });
    }
  });

  document.addEventListener("click", () => {
    closeAll();
  });

  document.addEventListener("keydown", (event) => {
    if (event.key === "Escape") {
      closeAll();
    }
  });
}

function initAccountMenu() {
  const toggles = document.querySelectorAll("[data-account-toggle]");
  if (!toggles.length) return;

  const closeAll = () => {
    document.querySelectorAll(".account-wrapper.is-open").forEach((wrapper) => {
      wrapper.classList.remove("is-open");
    });
    document.querySelectorAll(".notify-wrapper.is-open").forEach((wrapper) => {
      wrapper.classList.remove("is-open");
    });
  };

  toggles.forEach((toggle) => {
    const wrapper = toggle.closest(".account-wrapper");
    if (!wrapper) return;

    toggle.addEventListener("click", (event) => {
      event.stopPropagation();
      const isOpen = wrapper.classList.contains("is-open");
      closeAll();
      if (!isOpen) {
        wrapper.classList.add("is-open");
      }
    });
  });

  document.addEventListener("click", () => {
    closeAll();
  });

  document.addEventListener("keydown", (event) => {
    if (event.key === "Escape") {
      closeAll();
    }
  });
}

// Friends sub-tabs activation function (global for popstate access)
let activateFriendsTab = null;

function initFriendsTabs() {
  const container = document.querySelector("[data-friends-tabs]");
  if (!container) return;

  const root =
    container.closest(".friends-container-inline") ||
    container.closest(".account-panel") ||
    document;
  const buttons = container.querySelectorAll("[data-friends-tab]");
  const panels = root.querySelectorAll("[data-friends-panel]");
  if (!buttons.length || !panels.length) return;

  const validTabs = new Set(
    Array.from(buttons, (btn) => btn.dataset.friendsTab),
  );

  const activate = async (target, updateUrl = true, forceReload = false) => {
    if (!validTabs.has(target)) return;

    buttons.forEach((btn) => {
      btn.classList.toggle("active", btn.dataset.friendsTab === target);
    });
    panels.forEach((panel) => {
      panel.classList.toggle(
        "is-hidden",
        panel.dataset.friendsPanel !== target,
      );
    });

    if (updateUrl) {
      const url = new URL(globalThis.location.href);
      url.searchParams.set("tab", "friends");
      url.searchParams.set("ftab", target);
      globalThis.history.pushState({ tab: "friends", ftab: target }, "", url);
    }

    // Load content via AJAX for dynamic updates
    if (forceReload || updateUrl) {
      await loadFriendsTabContent(target, root);
    }
  };

  // Expose for popstate handler
  activateFriendsTab = activate;

  buttons.forEach((btn) => {
    btn.addEventListener("click", () => {
      activate(btn.dataset.friendsTab, true, true);
    });
  });

  // Set initial state from URL or default
  const urlFtab = new URL(globalThis.location.href).searchParams.get("ftab");
  const defaultTab = container.dataset.friendsTabDefault || "all";
  const initialTab = urlFtab && validTabs.has(urlFtab) ? urlFtab : defaultTab;

  // Ensure the correct tab is visible and always load fresh content
  activate(initialTab, false, true);

  // Initialize friends search
  initFriendsSearch();

  // Initialize action buttons on existing content
  initFriendActionButtons(root);
}

// Load friends tab content via AJAX
async function loadFriendsTabContent(tab, root) {
  const panel = root.querySelector(`[data-friends-panel="${tab}"]`);
  if (!panel) return;

  // Look for specific content container or fall back to friends-grid
  let grid =
    panel.querySelector(`[data-friends-content="${tab}"]`) ||
    panel.querySelector(".friends-grid");
  if (!grid) return;

  // Show loading
  grid.innerHTML = '<div class="friends-empty"><p>Loading...</p></div>';

  try {
    const encodedTab = encodeURIComponent(tab);
    const primaryUrl = `/friends-list?tab=${encodedTab}`;
    const fallbackUrl = `/index.php?route=friends-list&tab=${encodedTab}`;

    const fetchJson = async (url) => {
      const response = await fetch(url, {
        credentials: "same-origin",
        headers: { "X-Requested-With": "XMLHttpRequest" },
      });
      const contentType = response.headers.get("content-type") || "";
      if (!contentType.includes("application/json")) {
        const text = await response.text();
        throw new Error(
          `Non-JSON response (${response.status}) from ${url}: ${text.slice(0, 120)}`,
        );
      }
      return response.json();
    };

    let data;
    try {
      data = await fetchJson(primaryUrl);
    } catch (error_) {
      debugCatch("loadFriendsTabContent primary endpoint failed", error_);
      data = await fetchJson(fallbackUrl);
    }

    if (data.success && data.data) {
      if (data.data.length === 0) {
        grid.innerHTML = `<div class="friends-empty"><p>${getEmptyMessage(tab)}</p></div>`;
      } else {
        grid.innerHTML = renderFriendsCards(tab, data.data);
        initFriendActionButtons(grid);
      }
    } else {
      grid.innerHTML =
        '<div class="friends-empty"><p>Failed to load.</p></div>';
    }
  } catch (error_) {
    debugCatch("Friends tab load error", error_);
    console.error("Friends tab load error:", error_);
    grid.innerHTML = '<div class="friends-empty"><p>Failed to load.</p></div>';
  }
}

// Reload a friends tab if it's currently visible
function reloadVisibleFriendsTab(tab) {
  const panel = document.querySelector(
    `[data-friends-panel="${tab}"]:not(.is-hidden)`,
  );
  if (panel) {
    const root = panel.closest(".auth-card") || document;
    loadFriendsTabContent(tab, root);
  }
}

// Update search result card status when friend action changes
function updateSearchResultStatus(userId, newStatus) {
  if (!userId) return;

  // Find search result card with this user (by data-user-id on card, or by button data-target-user)
  const searchContainer = document.querySelector(
    "[data-friends-search-results]",
  );
  if (!searchContainer) return;

  // Try to find by card's data-user-id first, then by button
  let card = searchContainer.querySelector(
    `.friend-card[data-user-id="${userId}"]`,
  );
  if (!card) {
    card = searchContainer
      .querySelector(`[data-friend-action][data-target-user="${userId}"]`)
      ?.closest(".friend-card");
  }
  if (!card) return;

  const actionsDiv = card.querySelector(".friend-card-actions");
  if (!actionsDiv) return;

  if (newStatus === "request_sent") {
    actionsDiv.innerHTML =
      '<span class="friend-status-badge pending">Pending</span>';
  } else if (newStatus === "friends") {
    actionsDiv.innerHTML = '<span class="friend-status-badge">✓ Friends</span>';
  } else if (newStatus === "blocked") {
    actionsDiv.innerHTML =
      '<span class="friend-status-badge blocked">Blocked</span>';
  } else if (newStatus === "none") {
    // Show Add Friend button again
    actionsDiv.innerHTML = `<button type="button" class="btn btn-icon btn-primary has-tooltip" data-friend-action="send_request" data-target-user="${userId}" data-tooltip="Send Friend Request"><svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round"><line x1="12" y1="5" x2="12" y2="19"/><line x1="5" y1="12" x2="19" y2="12"/></svg></button>`;
    initFriendActionButtons(actionsDiv);
  }
}

function getEmptyMessage(tab) {
  switch (tab) {
    case "all":
      return "No friends yet. Search for users to add friends!";
    case "pending":
      return "No pending friend requests.";
    case "sent":
      return "No sent requests.";
    case "blocked":
      return "No blocked users.";
    default:
      return "No data.";
  }
}

function renderFriendsCards(tab, data) {
  return data
    .map((item) => {
      const rawNickname = (item.nickname || "").toString();
      const safeNickname = escapeHtml(rawNickname);
      const profileUrl = rawNickname
        ? "/@" + encodeURIComponent(rawNickname)
        : "#";
      const initial = (item.display_name || item.nickname || "?")
        .charAt(0)
        .toUpperCase();
      const photoHtml = item.profile_photo
        ? '<img src="' +
          escapeHtml(item.profile_photo) +
          '" alt="" class="friend-card-photo">'
        : '<div class="friend-card-photo-placeholder">' + initial + "</div>";

      const showDisplayName =
        item.display_name && item.display_name !== item.nickname;
      const safeDisplayName = escapeHtml(item.display_name || "");
      const nameHtml = showDisplayName
        ? '<a href="' +
          profileUrl +
          '" class="friend-card-name">' +
          safeDisplayName +
          "</a>" +
          '<span class="friend-card-nickname">@' +
          safeNickname +
          "</span>"
        : '<a href="' +
          profileUrl +
          '" class="friend-card-name">@' +
          safeNickname +
          "</a>";

      let actionsHtml = "";
      switch (tab) {
        case "all":
          actionsHtml = `<button type="button" class="btn btn-outline btn-sm" data-friend-action="remove_friend" data-target-user="${item.id}">Remove</button>`;
          break;
        case "pending":
          actionsHtml = `
                    <button type="button" class="btn btn-primary btn-sm" data-friend-action="accept_request" data-request-id="${item.request_id}">Accept</button>
                    <button type="button" class="btn btn-outline btn-sm" data-friend-action="decline_request" data-request-id="${item.request_id}">Decline</button>`;
          break;
        case "sent":
          actionsHtml = `<button type="button" class="btn btn-outline btn-sm" data-friend-action="cancel_request" data-request-id="${item.request_id}">Cancel</button>`;
          break;
        case "blocked":
          actionsHtml = `<button type="button" class="btn btn-outline btn-sm" data-friend-action="unblock_user" data-target-user="${item.user_id}">Unblock</button>`;
          break;
      }

      return (
        '<div class="friend-card">' +
        '<a href="' +
        profileUrl +
        '" class="friend-card-photo-link">' +
        photoHtml +
        "</a>" +
        '<div class="friend-card-info">' +
        nameHtml +
        "</div>" +
        '<div class="friend-card-actions">' +
        actionsHtml +
        "</div>" +
        "</div>"
      );
    })
    .join("");
}

function initFriendsSearch() {
  const searchForm = document.querySelector(".friends-search-form");
  if (!searchForm) return;

  const searchInput = searchForm.querySelector(".friends-search-input");
  const searchResultsContainer = document.querySelector(
    "[data-friends-search-results]",
  );
  const clearBtn = document.querySelector("[data-friends-search-clear]");

  if (!searchInput || !searchResultsContainer) return;

  searchForm.addEventListener("submit", async (e) => {
    e.preventDefault();
    const query = searchInput.value.trim();
    if (!query) return;

    // Update URL without reload
    const url = new URL(globalThis.location.href);
    url.searchParams.set("fq", query);
    globalThis.history.pushState({}, "", url);

    // Show loading state
    searchResultsContainer.innerHTML =
      '<div class="friends-empty"><p>Searching...</p></div>';
    searchResultsContainer.classList.remove("is-hidden");

    try {
      const response = await fetch(
        `/friends-search?q=${encodeURIComponent(query)}`,
        {
          method: "GET",
          credentials: "same-origin",
          headers: {
            "X-Requested-With": "XMLHttpRequest",
            Accept: "application/json",
          },
        },
      );

      if (!response.ok) {
        // Try to get error details from response
        const errorData = await response.json().catch(() => ({}));
        console.error("Friends search API error:", response.status, errorData);
        throw new Error(errorData.debug || `HTTP ${response.status}`);
      }

      const data = await response.json();

      if (data.success && data.results && data.results.length > 0) {
        let html =
          '<div class="friends-section"><h3 class="friends-section-title">Search Results for "' +
          escapeHtml(query) +
          '"</h3><div class="friends-grid">';
        data.results.forEach((user) => {
          const rawNickname = (user.nickname || "").toString();
          const safeNickname = escapeHtml(rawNickname);
          const profileUrl = rawNickname
            ? "/@" + encodeURIComponent(rawNickname)
            : "#";
          const initial = (user.display_name || user.nickname || "?")
            .charAt(0)
            .toUpperCase();
          const photoHtml = user.profile_photo
            ? '<img src="' +
              escapeHtml(user.profile_photo) +
              '" alt="" class="friend-card-photo">'
            : '<div class="friend-card-photo-placeholder">' +
              initial +
              "</div>";

          let actionHtml = "";
          if (user.status === "friends") {
            actionHtml = '<span class="friend-status-badge">✓ Friends</span>';
          } else if (user.status === "request_sent") {
            actionHtml =
              '<span class="friend-status-badge pending">Pending</span>';
          } else if (user.status === "request_received") {
            actionHtml =
              '<span class="friend-status-badge incoming">Accept?</span>';
          } else if (user.status === "blocked") {
            actionHtml =
              '<span class="friend-status-badge blocked">Blocked</span>';
          } else {
            actionHtml = `<button type="button" class="btn btn-icon btn-primary has-tooltip" data-friend-action="send_request" data-target-user="${user.id}" data-tooltip="Send Friend Request"><svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round"><line x1="12" y1="5" x2="12" y2="19"/><line x1="5" y1="12" x2="19" y2="12"/></svg></button>`;
          }

          // Only show display_name if different from nickname
          const showDisplayName =
            user.display_name && user.display_name !== user.nickname;
          const safeDisplayName = escapeHtml(user.display_name || "");
          const nameHtml = showDisplayName
            ? '<a href="' +
              profileUrl +
              '" class="friend-card-name">' +
              safeDisplayName +
              "</a>" +
              '<span class="friend-card-nickname">@' +
              safeNickname +
              "</span>"
            : '<a href="' +
              profileUrl +
              '" class="friend-card-name">@' +
              safeNickname +
              "</a>";

          html +=
            '<div class="friend-card" data-user-id="' +
            user.id +
            '">' +
            '<a href="' +
            profileUrl +
            '" class="friend-card-photo-link">' +
            photoHtml +
            "</a>" +
            '<div class="friend-card-info">' +
            nameHtml +
            "</div>" +
            '<div class="friend-card-actions">' +
            actionHtml +
            "</div>" +
            "</div>";
        });
        html += "</div></div>";
        searchResultsContainer.innerHTML = html;
      } else {
        searchResultsContainer.innerHTML =
          '<div class="friends-empty"><p>No users found matching "' +
          escapeHtml(query) +
          '"</p></div>';
      }

      // Show clear button
      if (clearBtn) clearBtn.classList.remove("is-hidden");

      // Attach friend action handlers to new buttons
      initFriendActionButtons(searchResultsContainer);
    } catch (err) {
      console.error("Friends search error:", err);
      searchResultsContainer.innerHTML =
        '<div class="friends-empty"><p>Search failed. Please try again.</p></div>';
    }
  });

  // Clear button handler
  if (clearBtn) {
    clearBtn.addEventListener("click", (e) => {
      e.preventDefault();
      searchInput.value = "";
      searchResultsContainer.innerHTML = "";
      searchResultsContainer.classList.add("is-hidden");
      clearBtn.classList.add("is-hidden");

      // Update URL without reload
      const url = new URL(globalThis.location.href);
      url.searchParams.delete("fq");
      globalThis.history.pushState({}, "", url);
    });
  }
}

// Handle AJAX friend action buttons
function initFriendActionButtons(container) {
  if (!container) container = document;

  const confirmActionMessages = {
    remove_friend: "Remove this friend?",
    decline_request: "Decline this request?",
  };

  const shouldProceedWithFriendAction = (action) => {
    const message = confirmActionMessages[action];
    return !message || confirm(message);
  };

  const markButtonPending = (btn) => {
    btn.disabled = true;
    const originalHTML = btn.innerHTML;
    if (btn.classList.contains("btn-icon")) {
      btn.classList.add("is-action-pending");
    } else {
      btn.textContent = "Sending...";
    }
    return originalHTML;
  };

  const restoreFriendActionButton = (btn, originalHTML) => {
    if (!btn?.isConnected) return;
    btn.disabled = false;
    btn.innerHTML = originalHTML;
    btn.classList.remove("is-action-pending");
  };

  const refreshFriendsTabsForStatus = (status) => {
    if (status === "request_sent") {
      reloadVisibleFriendsTab("sent");
      return;
    }
    if (status === "friends") {
      reloadVisibleFriendsTab("all");
      reloadVisibleFriendsTab("pending");
      return;
    }
    if (status === "blocked") {
      reloadVisibleFriendsTab("blocked");
      return;
    }
    if (status === "none") {
      reloadVisibleFriendsTab("sent");
    }
  };

  const applyProfileBadgeForStatus = (btn, status) => {
    if (status === "request_sent") {
      btn.outerHTML =
        '<span class="profile-status-badge status-pending">Request Sent</span>';
      return;
    }
    if (status === "friends") {
      btn.outerHTML =
        '<span class="profile-status-badge status-friends">✓ Friends</span>';
      return;
    }
    if (status === "blocked") {
      btn.outerHTML =
        '<span class="profile-status-badge status-blocked">Blocked</span>';
    }
  };

  const applyFriendActionSuccess = (btn, data, targetUserId) => {
    const status = data.new_status;
    const actionsDiv =
      btn.closest(".friend-card-actions") || btn.closest(".profile-actions");
    const isProfileAction = Boolean(btn.closest(".profile-actions"));

    if (actionsDiv) {
      if (isProfileAction) {
        applyProfileBadgeForStatus(btn, status);
      } else if (status === "request_sent") {
        actionsDiv.innerHTML =
          '<span class="friend-status-badge pending">Pending</span>';
      } else if (status === "friends") {
        actionsDiv.innerHTML =
          '<span class="friend-status-badge">✓ Friends</span>';
      } else if (status === "blocked") {
        actionsDiv.innerHTML =
          '<span class="friend-status-badge blocked">Blocked</span>';
      } else if (status === "none") {
        btn.closest(".friend-card")?.remove();
      }
    } else if (status === "request_sent") {
      applyProfileBadgeForStatus(btn, status);
    }

    refreshFriendsTabsForStatus(status);
    updateSearchResultStatus(data.target_user_id || targetUserId, status);
  };

  container.querySelectorAll("[data-friend-action]").forEach((btn) => {
    if (btn.dataset.friendActionBound) return;
    btn.dataset.friendActionBound = "true";

    btn.addEventListener("click", async (e) => {
      e.preventDefault();

      const action = btn.dataset.friendAction;
      const targetUserId = btn.dataset.targetUser;
      const requestId = btn.dataset.requestId;
      if (!shouldProceedWithFriendAction(action)) return;

      const originalHTML = markButtonPending(btn);

      try {
        const bodyObj = {
          action: action,
          target_user_id: Number.parseInt(targetUserId, 10) || 0,
          request_id: Number.parseInt(requestId, 10) || 0,
        };

        if (csrfToken) {
          bodyObj.csrf_token = csrfToken;
        }

        const response = await fetch("/friends-action", {
          method: "POST",
          credentials: "same-origin",
          headers: {
            "Content-Type": "application/json",
            "X-Requested-With": "XMLHttpRequest",
          },
          body: JSON.stringify(bodyObj),
        });

        const data = await response.json();

        if (!data.success) {
          alert(data.error || "Action failed");
          restoreFriendActionButton(btn, originalHTML);
          return;
        }

        applyFriendActionSuccess(btn, data, targetUserId);
      } catch (error_) {
        debugCatch("Friend action error", error_);
        console.error("Friend action error:", error_);
        alert("Action failed. Please try again.");
        restoreFriendActionButton(btn, originalHTML);
      }
    });
  });
}

function escapeHtml(text) {
  if (!text) return "";
  const div = document.createElement("div");
  div.textContent = text;
  return div.innerHTML;
}

function initAccountTabs() {
  const container = document.querySelector("[data-account-tabs]");
  if (!container) return;

  const root = container.closest(".auth-card") || document;
  const buttons = container.querySelectorAll("[data-account-tab]");
  const panels = root.querySelectorAll("[data-account-panel]");
  if (!buttons.length || !panels.length) return;
  const validTabs = new Set(
    Array.from(buttons, (btn) => btn.dataset.accountTab),
  );

  const ensureVisible = () => {
    const anyVisible = Array.from(panels).some(
      (panel) => !panel.classList.contains("is-hidden"),
    );
    if (!anyVisible) {
      const fallback =
        Array.from(panels).find(
          (panel) => panel.dataset.accountPanel === "messages",
        ) || panels[0];
      if (fallback) {
        fallback.classList.remove("is-hidden");
      }
    }
  };

  const activate = (target, options = {}) => {
    if (!validTabs.has(target)) {
      panels.forEach((panel) => {
        panel.classList.remove("is-hidden");
      });
      ensureVisible();
      return;
    }
    buttons.forEach((btn) => {
      btn.classList.toggle("active", btn.dataset.accountTab === target);
    });
    panels.forEach((panel) => {
      panel.classList.toggle(
        "is-hidden",
        panel.dataset.accountPanel !== target,
      );
    });
    ensureVisible();

    if (options.updateUrl) {
      const url = new URL(globalThis.location.href);
      url.searchParams.set("tab", target);
      if (target !== "messages") {
        url.searchParams.delete("message");
      }
      // Always include ftab=all when switching to friends tab
      if (target === "friends" && !url.searchParams.get("ftab")) {
        url.searchParams.set("ftab", "all");
      }
      globalThis.history.pushState({ tab: target }, "", url);
    }

    // When switching to friends tab, ensure the friends sub-tabs are properly initialized
    if (target === "friends") {
      // Re-initialize friends tabs and force load content
      setTimeout(() => {
        const friendsContainer = document.querySelector("[data-friends-tabs]");
        if (friendsContainer) {
          const friendsRoot =
            friendsContainer.closest(".friends-container-inline") ||
            friendsContainer.closest(".account-panel") ||
            document;
          const activeTab = friendsContainer.querySelector(
            ".friends-tab.active",
          );
          const ftab = activeTab ? activeTab.dataset.friendsTab : "all";
          // Always load content when switching to friends tab
          loadFriendsTabContent(ftab, friendsRoot);
          initFriendsSearch();
          initFriendActionButtons(friendsRoot);
        }
      }, 0);
    }
  };

  buttons.forEach((btn) => {
    btn.addEventListener("click", () => {
      activate(btn.dataset.accountTab, { updateUrl: true });
    });
  });

  const urlTab = new URL(globalThis.location.href).searchParams.get("tab");
  if (urlTab && validTabs.has(urlTab)) {
    activate(urlTab);
  } else {
    const defaultTab = container.dataset.accountTabDefault;
    if (defaultTab && validTabs.has(defaultTab)) {
      activate(defaultTab);
    } else {
      const active = container.querySelector(".account-tab.active");
      if (active) {
        activate(active.dataset.accountTab);
      } else {
        activate("general");
      }
    }
  }

  globalThis.addEventListener("popstate", () => {
    const url = new URL(globalThis.location.href);
    const tab = url.searchParams.get("tab") || "general";
    if (validTabs.has(tab)) {
      activate(tab);
    }
    // Also handle friends sub-tabs
    if (tab === "friends") {
      const ftab = url.searchParams.get("ftab") || "all";
      if (typeof activateFriendsTab === "function") {
        activateFriendsTab(ftab, false);
      }
    }
  });

  // Initialize friends sub-tabs
  initFriendsTabs();

  // Changelog toggle functionality
  document.querySelectorAll("[data-toggle-changelog]").forEach((btn) => {
    btn.addEventListener("click", function () {
      const actions = this.closest(".order-item-actions");
      if (!actions) return;
      const changelog = actions.querySelector("[data-changelog]");
      if (changelog) {
        changelog.classList.toggle("is-hidden");
        this.textContent = changelog.classList.contains("is-hidden")
          ? "What's new"
          : "Hide";
      }
    });
  });
}

function initNotificationPolling() {
  const wrapper = document.querySelector(".notify-wrapper");
  const list = document.querySelector("[data-notify-list]");
  const markForm = document.querySelector("[data-notify-mark]");
  const markButton = document.querySelector(".js-notify-mark-all");
  if (!wrapper || !list) return;

  let lastUnread = null;
  let lastPendingFriends = null;
  let toastTimeout = null;

  const showToast = (title, body) => {
    if (!title && !body) return;
    const existing = document.querySelector(".notify-toast");
    if (existing) {
      existing.remove();
    }
    const toast = document.createElement("div");
    toast.className = "notify-toast";
    const safeTitle = title || "New notification";
    const safeBody = body || "You have a new message.";
    toast.innerHTML = `
            <div>
                <div class="notify-toast-title">${safeTitle}</div>
                <div class="notify-toast-body">${safeBody}</div>
            </div>
        `;
    document.body.appendChild(toast);
    clearTimeout(toastTimeout);
    toastTimeout = setTimeout(() => {
      toast.remove();
    }, 2100);
  };

  const syncPresenceIndicator = (container, selector, className, show) => {
    if (!container) return;
    const existing = container.querySelector(selector);
    if (show && !existing) {
      const indicator = document.createElement("span");
      indicator.className = className;
      container.appendChild(indicator);
      return;
    }
    if (!show && existing) {
      existing.remove();
    }
  };

  const syncCountBadge = (
    container,
    selector,
    className,
    count,
    alertClass,
  ) => {
    if (!container) return;
    const existing = container.querySelector(selector);
    if (count <= 0) {
      existing?.remove();
      return;
    }

    const badge = existing || document.createElement("span");
    badge.className = className;
    if (alertClass) {
      badge.classList.add(alertClass);
    }
    badge.textContent = String(count);
    if (!existing) {
      container.appendChild(badge);
    }
  };

  const syncMessageCenterIndicator = (unreadTotal) => {
    const messageCenterTab = document.querySelector(
      "[data-message-center-tab]",
    );
    syncPresenceIndicator(
      messageCenterTab,
      ".message-center-indicator",
      "message-center-indicator",
      unreadTotal > 0,
    );
  };

  const syncNotifyBadgeAndMarkForm = (noticeUnread) => {
    const notifyButton = wrapper.querySelector(".notify-button");
    syncPresenceIndicator(
      notifyButton,
      ".notify-indicator",
      "notify-indicator",
      noticeUnread > 0,
    );
    markForm?.classList.toggle("is-hidden-display", noticeUnread <= 0);
  };

  const renderUnreadNotifications = (items) => {
    const unreadItems = (items || []).filter((item) => !item.is_read);
    if (!unreadItems.length) {
      list.innerHTML =
        '<div class="notify-empty" data-notify-empty>No unread messages.</div>';
      return;
    }

    const renderedItems = [];
    for (const item of unreadItems) {
      const safeTitle = item.subject || "Message";
      const bodyText = (item.body || "").toString();
      const bodyPreview =
        bodyText.length > 100 ? `${bodyText.slice(0, 100)}…` : bodyText;
      const body = bodyPreview
        ? `<div class="notify-item-body">${bodyPreview}</div>`
        : "";
      const unreadClass = item.is_read ? "" : " is-unread";
      const linkHref = `/account?tab=messages&message=${item.id}#message-${item.id}`;
      renderedItems.push(
        '<a class="notify-item-link" href="' +
          linkHref +
          '" data-notify-message-id="' +
          item.id +
          '">' +
          '<div class="notify-item' +
          unreadClass +
          '">' +
          '<div class="notify-item-title">' +
          safeTitle +
          "</div>" +
          body +
          "</div>" +
          "</a>",
      );
    }
    list.innerHTML = renderedItems.join("");

    const markNotificationAsRead = async (messageId) => {
      try {
        await fetch("/messages-api", {
          method: "POST",
          headers: { "Content-Type": "application/x-www-form-urlencoded" },
          body: "mark_message_read=" + encodeURIComponent(messageId),
        });
      } catch (error_) {
        debugCatch("markNotificationAsRead failed", error_);
      }
    };

    list.querySelectorAll("[data-notify-message-id]").forEach((link) => {
      link.addEventListener("click", () => {
        const messageId = link.dataset.notifyMessageId;
        if (!messageId) return;
        markNotificationAsRead(messageId);
      });
    });
  };

  const resolveUnreadTotal = (data, noticeUnread) => {
    let unreadTotal =
      data?.unread_total ?? noticeUnread + (data.ticket_unread || 0);
    if (
      globalThis.__ticketUnreadOverrideUntil &&
      Date.now() < globalThis.__ticketUnreadOverrideUntil
    ) {
      const override =
        Number.parseInt(globalThis.__ticketUnreadOverride || 0, 10) || 0;
      unreadTotal = noticeUnread + override;
    }
    return unreadTotal;
  };

  const maybeShowNotificationToast = (data, noticeUnread, pendingFriends) => {
    if (lastPendingFriends !== null && pendingFriends > lastPendingFriends) {
      const requester = data.latest_friend_requester || "Someone";
      showToast(
        "New Friend Request",
        `${requester} sent you a friend request.`,
      );
      return;
    }

    if (lastUnread === null || noticeUnread <= lastUnread) {
      return;
    }

    const newestUnread = (data.items || []).find((item) => !item.is_read);
    showToast(
      newestUnread?.subject || "New notification",
      newestUnread?.body || "You have a new message.",
    );
  };

  const updateFriendsIndicator = (pendingCount) => {
    const friendsMenuLink = document.querySelector("[data-friends-menu-link]");
    syncPresenceIndicator(
      friendsMenuLink,
      ".friends-indicator",
      "friends-indicator",
      pendingCount > 0,
    );

    const pendingTab = document.querySelector('[data-friends-tab="pending"]');
    syncCountBadge(
      pendingTab,
      ".tab-count",
      "tab-count",
      pendingCount,
      "tab-count-alert",
    );

    const friendsNavTab = document.querySelector(
      '[data-account-tab="friends"]',
    );
    syncCountBadge(
      friendsNavTab,
      ".friend-notification-badge",
      "friend-notification-badge",
      pendingCount,
    );
  };

  const updateUI = (data) => {
    if (!data?.success) return;

    const noticeUnread = data.unread || 0;
    const unreadTotal = resolveUnreadTotal(data, noticeUnread);
    const pendingFriends = data.pending_friend_requests || 0;

    maybeShowNotificationToast(data, noticeUnread, pendingFriends);
    lastUnread = noticeUnread;
    lastPendingFriends = pendingFriends;

    updateFriendsIndicator(pendingFriends);
    syncMessageCenterIndicator(unreadTotal);
    syncNotifyBadgeAndMarkForm(noticeUnread);
    renderUnreadNotifications(data.items || []);
  };

  const fetchUpdates = () => {
    // Explicitly using GET for data retrieval
    fetch("/notifications-api?action=get")
      .then((r) => r.json())
      .then(updateUI)
      .catch(() => {});
  };

  if (markButton) {
    markButton.addEventListener("click", () => {
      const formData = new FormData();
      formData.append("action", "mark_all");
      if (csrfToken) {
        formData.append("csrf_token", csrfToken);
      }

      fetch("/notifications-api", {
        method: "POST",
        body: formData,
      })
        .then((r) => r.json())
        .then(updateUI)
        .catch(() => {});
    });
  }

  fetchUpdates();

  // Poll every 30 seconds as fallback
  setInterval(fetchUpdates, 30000);

  document.addEventListener("visibilitychange", () => {
    if (document.visibilityState === "visible") {
      fetchUpdates();
    }
  });
}

function initAccountMessagesPolling() {
  const panel = document.querySelector("[data-message-panel]");
  const list = document.querySelector("[data-message-list]");
  const empty = document.querySelector("[data-message-empty]");
  if (!panel) return;

  // Handle clicks on message items anywhere in the panel
  panel.addEventListener("click", (event) => {
    const deleteBtn = event.target.closest(
      ".message-delete-form, .message-delete",
    );
    if (deleteBtn) return;
    const card = event.target.closest(".message-item--list");
    if (!card) return;
    event.preventDefault();
    const body = card.querySelector("[data-message-body]");
    const isExpanded = card.classList.contains("is-expanded");

    // Collapse all other expanded messages
    panel
      .querySelectorAll(".message-item--list.is-expanded")
      .forEach((item) => {
        item.classList.remove("is-expanded");
        const itemBody = item.querySelector("[data-message-body]");
        if (itemBody) {
          itemBody.classList.add("is-hidden");
        }
      });

    if (isExpanded) {
      const url = new URL(globalThis.location.href);
      url.searchParams.delete("message");
      globalThis.history.pushState({ tab: "messages" }, "", url);
    } else {
      card.classList.add("is-expanded");
      if (body) {
        body.classList.remove("is-hidden");
      }
      const messageId = card.dataset.messageId;
      if (messageId) {
        const url = new URL(globalThis.location.href);
        url.searchParams.set("tab", "messages");
        url.searchParams.set("message", messageId);
        globalThis.history.pushState(
          { tab: "messages", message: messageId },
          "",
          url,
        );

        // Mark message as read via AJAX
        if (card.classList.contains("is-unread")) {
          const markData = new URLSearchParams();
          markData.append("mark_message_read", messageId);
          if (csrfToken) {
            markData.append("csrf_token", csrfToken);
          }

          fetch("/messages-api", {
            method: "POST",
            headers: { "Content-Type": "application/x-www-form-urlencoded" },
            body: markData.toString(),
          })
            .then(() => {
              card.classList.remove("is-unread");
            })
            .catch(() => {});
        }
      }
    }
  });

  const escapeHtml = (value) => {
    return String(value)
      .replaceAll("&", "&amp;")
      .replaceAll("<", "&lt;")
      .replaceAll(">", "&gt;")
      .replaceAll('"', "&quot;")
      .replaceAll("'", "&#39;");
  };

  const formatTime = (timestamp) => {
    if (!timestamp) return "";
    const date = new Date(timestamp * 1000);
    return date.toLocaleString(undefined, {
      month: "short",
      day: "numeric",
      year: "numeric",
      hour: "numeric",
      minute: "2-digit",
    });
  };

  const renderList = (items) => {
    if (!list) return;
    if (!items.length) {
      list.innerHTML = "";
      if (empty) {
        empty.classList.remove("is-hidden");
      }
      return;
    }

    if (empty) {
      empty.classList.add("is-hidden");
    }

    list.innerHTML = items
      .map((item) => {
        const subject = escapeHtml(item.subject || "Message");
        const bodyText = (item.body || "").toString();
        const preview =
          bodyText.length > 100 ? `${bodyText.slice(0, 100)}…` : bodyText;
        const previewHtml = preview
          ? `<div class="message-preview">${escapeHtml(preview)}</div>`
          : "";
        const unreadClass = item.is_read ? "" : " is-unread";
        const timeText = formatTime(item.created_at);
        const timeHtml = timeText
          ? `<div class="message-time">${escapeHtml(timeText)}</div>`
          : "";
        const fullBodyHtml = bodyText
          ? `<div class="message-body message-body--full is-hidden" data-message-body>${escapeHtml(bodyText)}</div>`
          : "";
        const csrfField = csrfToken
          ? `<input type="hidden" name="csrf_token" value="${csrfToken}">`
          : "";
        return `
                    <div class="message-item message-item--list${unreadClass}" id="message-${item.id}" data-message-id="${item.id}" data-message-href="/account?tab=messages&message=${item.id}#message-${item.id}">
                    <div class="message-head">
                        <a class="message-title" href="/account?tab=messages&message=${item.id}#message-${item.id}">
                            ${subject}
                        </a>
                        <form method="POST" action="/account?tab=messages" class="message-delete-form" data-confirm="Delete this message?">
                            ${csrfField}
                            <input type="hidden" name="action" value="delete_message">
                            <input type="hidden" name="message_id" value="${item.id}">
                            <button type="submit" class="message-delete" aria-label="Delete message">🗑️</button>
                        </form>
                    </div>
                    ${previewHtml}
                    ${timeHtml}
                        ${fullBodyHtml}
                </div>
            `;
      })
      .join("");
  };

  const fetchUpdates = () => {
    if (panel.classList.contains("is-hidden")) {
      return;
    }
    const selectedId = new URL(globalThis.location.href).searchParams.get(
      "message",
    );
    if (selectedId) {
      return;
    }
    fetch("/messages-api?limit=10")
      .then((r) => r.json())
      .then((data) => {
        if (!data?.success) return;
        renderList(data.items || []);
      })
      .catch(() => {});
  };

  fetchUpdates();
  // Poll every 30 seconds (reduced from 5s to prevent database overload with many users)
  setInterval(fetchUpdates, 30000);
  document.addEventListener("visibilitychange", () => {
    if (document.visibilityState === "visible") {
      fetchUpdates();
    }
  });
}

function initCustomSelects() {
  const selects = document.querySelectorAll("[data-select]");
  if (!selects.length) return;

  const closeAll = () => {
    selects.forEach((select) => select.classList.remove("is-open"));
  };

  selects.forEach((select) => {
    const trigger = select.querySelector("[data-select-trigger]");
    const label = select.querySelector("[data-select-label]");
    const valueInput = select.querySelector("[data-select-value]");
    const options = select.querySelectorAll("[data-select-option]");
    if (!trigger || !label || !valueInput || !options.length) return;

    const updateActive = (value) => {
      options.forEach((option) => {
        option.classList.toggle("is-active", option.dataset.value === value);
      });
    };

    trigger.addEventListener("click", (event) => {
      event.preventDefault();
      const isOpen = select.classList.contains("is-open");
      closeAll();
      select.classList.toggle("is-open", !isOpen);
    });

    options.forEach((option) => {
      option.addEventListener("click", (event) => {
        event.preventDefault();
        const value = option.dataset.value || "";
        valueInput.value = value;
        label.textContent = option.textContent.trim();
        updateActive(value);
        select.classList.remove("is-open");
      });
    });

    updateActive(valueInput.value || "");
  });

  document.addEventListener("click", (event) => {
    if (!event.target.closest("[data-select]")) {
      closeAll();
    }
  });

  document.addEventListener("keydown", (event) => {
    if (event.key === "Escape") {
      closeAll();
    }
  });
}

function initSearchAutocomplete() {
  const widgets = document.querySelectorAll("[data-search-widget]");
  if (!widgets.length) return;

  const debounce = (fn, wait) => {
    let timeout;
    return (...args) => {
      clearTimeout(timeout);
      timeout = setTimeout(() => fn(...args), wait);
    };
  };

  widgets.forEach((widget) => {
    const input = widget.querySelector("[data-search-input]");
    const suggest = widget.querySelector("[data-search-suggest]");
    if (!input || !suggest) return;

    const renderItems = (items) => {
      if (!items.length) {
        suggest.innerHTML =
          '<div class="search-suggest-empty">No results found</div>';
        suggest.classList.add("open");
        return;
      }

      suggest.innerHTML = items
        .map((item) => {
          //NOSONAR item shaping kept local for performance
          const itemName = item?.name ? String(item.name) : "Untitled";
          const itemIcon = item?.icon
            ? `${escapeHtml(String(item.icon))} `
            : "";
          const itemType = item?.type === "mini_game" ? "Mini-Game" : "Product";
          const mediaUrl = item?.media ? String(item.media) : "";
          const itemUrl =
            typeof item?.url === "string" && item.url.startsWith("/")
              ? item.url
              : "";
          const safeName = escapeHtml(itemName);
          const safeMediaUrl = escapeHtml(mediaUrl);
          const media = safeMediaUrl
            ? `<div class="search-suggest-thumb"><img src="${safeMediaUrl}" alt="${safeName}"></div>`
            : "";
          const text = `<span>${itemIcon}${safeName}</span><span class="search-suggest-type">${itemType}</span>`;
          if (itemUrl) {
            return `
                        <a class="search-suggest-item" href="${itemUrl}">
                            ${media}
                            ${text}
                        </a>
                    `;
          }
          return `
                    <div class="search-suggest-item is-disabled">
                        ${media}
                        ${text}
                    </div>
                `;
        })
        .join("");
      suggest.classList.add("open");
    };

    const fetchResults = debounce(() => {
      const query = input.value.trim();
      if (query.length < 2) {
        suggest.classList.remove("open");
        suggest.innerHTML = "";
        return;
      }

      // Using GET for search
      fetch(`/search-api?q=${encodeURIComponent(query)}`)
        .then((r) => {
          if (!r.ok) {
            throw new Error("Search API unavailable");
          }
          return r.json();
        })
        .then((data) => {
          if (!data.success) return;
          renderItems(data.items || []);
        })
        .catch(() => {
          suggest.innerHTML =
            '<div class="search-suggest-empty">Search unavailable</div>';
          suggest.classList.add("open");
        });
    }, 200);

    input.addEventListener("input", fetchResults);
    input.addEventListener("focus", fetchResults);
    document.addEventListener("click", (event) => {
      if (!widget.contains(event.target)) {
        suggest.classList.remove("open");
      }
    });
  });
}

function updateCartItem(itemId, quantity, mode) {
  const formData = new FormData();
  formData.append("action", "update");
  formData.append("item_id", itemId);
  formData.append("quantity", quantity);

  if (csrfToken) {
    formData.append("csrf_token", csrfToken);
  }

  fetch("/cart-api", {
    method: "POST",
    body: formData,
  })
    .then(parseCartApiResponse)
    .then((data) => {
      updateCartBadgeFromItems(data.items || []);
      if (mode === "sidebar") {
        loadCartSidebar();
        return;
      }
      globalThis.location.reload();
    })
    .catch((error) => {
      console.warn("Cart update failed:", error);
    });
}

/**
 * Product Tabs (bar style)
 */
function initProductTabs() {
  document
    .querySelectorAll("[data-product-tabs]")
    .forEach(function (container) {
      //NOSONAR tab wiring is intentionally in one block
      const buttons = container.querySelectorAll(".product-tab-btn");
      const panels = container.querySelectorAll(".product-tab-panel");
      if (!buttons.length || !panels.length) return;

      buttons.forEach(function (btn) {
        btn.addEventListener("click", function () {
          const target = btn.dataset.tabTarget;
          for (const other of buttons) {
            other.classList.toggle("active", other === btn);
          }
          for (const panel of panels) {
            panel.classList.toggle("active", panel.dataset.tabPanel === target);
          }
        });
      });
    });
}

/**
 * Product Accordions (default open)
 */

/**
 * Page Loader Bar
 */
function initPageLoader() {
  const loader = document.getElementById("page-loader");
  // #page-loader is server-rendered to avoid a forced reflow
  // from document.body.appendChild during DOMContentLoaded.
  if (!loader) return;

  function showLoader() {
    loader.classList.add("active");
    requestAnimationFrame(function () {
      loader.classList.add("complete");
    });
  }

  function hideLoader() {
    loader.classList.add("complete");
    setTimeout(function () {
      loader.classList.remove("active", "complete");
    }, 200);
  }

  document.querySelectorAll("a[href]").forEach(function (link) {
    link.addEventListener("click", function (e) {
      const href = link.getAttribute("href") || "";
      const target = link.getAttribute("target") || "";

      if (
        target === "_blank" ||
        href.startsWith("mailto:") ||
        href.startsWith("tel:")
      ) {
        return;
      }

      if (href.startsWith("#")) {
        return;
      }

      if (
        href.startsWith("http") &&
        !href.startsWith(globalThis.location.origin)
      ) {
        return;
      }

      showLoader();
    });
  });

  globalThis.addEventListener("pageshow", hideLoader);
}

/**
 * Mobile Menu Toggle
 */
function initMobileMenu() {
  const toggle = document.querySelector(".mobile-menu-toggle");
  const menu = document.querySelector(".nav-menu");

  const closeSiblingSubmenus = (item) => {
    const parent = item.parentElement;
    if (!parent) return;
    Array.from(parent.children).forEach((sibling) => {
      if (sibling === item || !sibling.classList.contains("has-submenu")) {
        return;
      }
      sibling.classList.remove("submenu-open");
      sibling
        .querySelectorAll(".submenu-open")
        .forEach((nested) => nested.classList.remove("submenu-open"));
    });
  };

  const handleMobileSubmenuToggle = (event, item) => {
    if (globalThis.innerWidth > 768) {
      return;
    }
    event.preventDefault();
    event.stopPropagation();
    item.classList.toggle("submenu-open");
    closeSiblingSubmenus(item);
  };

  if (toggle && menu) {
    toggle.addEventListener("click", function () {
      menu.classList.toggle("active");
      toggle.classList.toggle("active");
    });

    // Mobile submenu toggle for header - supports nested submenus
    menu.querySelectorAll(".has-submenu").forEach(function (item) {
      const link = item.querySelector(":scope > a");
      if (link) {
        link.addEventListener("click", function (e) {
          handleMobileSubmenuToggle(e, item);
        });
      }
    });
  }

  // Mobile submenu toggle for footer - supports nested submenus
  const footerMenu = document.querySelector(".footer-menu");
  if (footerMenu) {
    footerMenu.querySelectorAll(".has-submenu").forEach(function (item) {
      const link = item.querySelector(":scope > a");
      if (link) {
        link.addEventListener("click", function (e) {
          handleMobileSubmenuToggle(e, item);
        });
      }
    });
  }
}

/**
 * Sidebar Toggle (Header Sidebar Layout)
 */
function initSidebarToggle() {
  const toggle = document.querySelector("[data-sidebar-toggle]");
  const overlay = document.querySelector("[data-sidebar-overlay]");
  const sidebarHeader = document.querySelector(".sidebar-header");

  if (!toggle) {
    return;
  }

  if (sidebarHeader) {
    // Sidebar width now relies on static CSS variables for CSP safety.
  }

  const closeSidebar = () => {
    document.body.classList.remove("sidebar-open");
  };

  toggle.addEventListener("click", () => {
    document.body.classList.toggle("sidebar-open");
  });

  if (overlay) {
    overlay.addEventListener("click", closeSidebar);
  }

  document.addEventListener("keydown", (event) => {
    if (event.key === "Escape") {
      closeSidebar();
    }
  });

  globalThis.addEventListener("resize", () => {
    if (globalThis.innerWidth > 960) {
      closeSidebar();
    }
  });
}

/**
 * Messenger Widget Logic
 */
function initMessengerWidget() {
  const widget = document.getElementById("messenger-widget");
  if (!widget) return;

  document.body.classList.add("has-messenger-widget");

  const toggle = document.getElementById("messenger-toggle");
  const closeBtn = document.getElementById("messenger-close-btn");
  const backBtn = document.getElementById("messenger-back-btn");
  const listView = document.getElementById("messenger-list-view");
  const chatView = document.getElementById("messenger-chat-view");
  const friendList = document.getElementById("messenger-friend-list");
  const messagesContainer = document.getElementById("messenger-messages");
  const messengerForm = document.getElementById("messenger-form");
  const messengerInput = document.getElementById("messenger-input");
  const unreadBadge = document.getElementById("messenger-unread-badge");
  const title = document.getElementById("messenger-title");

  // Floating widget spacing is handled by CSS variables for CSP safety.

  if (messagesContainer) {
    messagesContainer.addEventListener("click", (event) => {
      const img = event.target.closest("[data-open-image]");
      if (!img) return;
      event.preventDefault();
      const url = img.dataset.openImage;
      if (url) {
        globalThis.open(url, "_blank");
      }
    });
  }

  let activeFriendId = null;
  let pollInterval = null;
  let listPollInterval = null;
  const hiddenKey = "messenger_hidden_friends";
  const getHiddenList = () => {
    try {
      const raw = localStorage.getItem(hiddenKey);
      const parsed = raw ? JSON.parse(raw) : [];
      return Array.isArray(parsed) ? parsed.map(String) : [];
    } catch (error_) {
      debugCatch("getHiddenList parse failed", error_);
      return [];
    }
  };
  const setHiddenList = (list) => {
    localStorage.setItem(hiddenKey, JSON.stringify(list.map(String)));
  };
  const hideFriend = (id) => {
    const list = getHiddenList();
    const key = String(id);
    if (!list.includes(key)) {
      list.push(key);
      setHiddenList(list);
    }
  };
  const unhideFriend = (id) => {
    const key = String(id);
    const list = getHiddenList().filter((item) => item !== key);
    setHiddenList(list);
  };

  let contextMenu = document.querySelector(".messenger-context-menu");
  if (!contextMenu) {
    contextMenu = document.createElement("div");
    contextMenu.className = "messenger-context-menu";
    contextMenu.innerHTML = `
            <button type="button" class="messenger-context-item" data-action="clear_chat">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M21 15a4 4 0 0 1-4 4H8l-5 4V7a4 4 0 0 1 4-4h10a4 4 0 0 1 4 4z"/><path d="M8 9h8"/><path d="M8 13h6"/></svg>
                Clear chat
            </button>
            <button type="button" class="messenger-context-item" data-action="hide_chat">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M4 6h16"/><path d="M7 6v12"/><path d="M17 6v12"/><path d="M9 6V4h6v2"/><path d="M10 10v6"/><path d="M14 10v6"/></svg>
                Hide chat
            </button>
        `;
    document.body.appendChild(contextMenu);
  }
  let contextTargetId = null;
  const closeContextMenu = () => {
    contextMenu.classList.remove("is-open");
    const parent = contextMenu.parentElement;
    if (parent?.classList.contains("messenger-friend-item")) {
      parent.classList.remove("is-context-open");
    }
    contextTargetId = null;
  };
  const openContextMenu = (item, friendId) => {
    closeContextMenu();
    contextTargetId = String(friendId);
    if (item?.contains(contextMenu) === false) {
      item.appendChild(contextMenu);
    }
    if (item) {
      item.classList.add("is-context-open");
    }
    contextMenu.classList.add("is-open");
  };

  document.addEventListener("click", (event) => {
    if (
      contextMenu.classList.contains("is-open") &&
      !contextMenu.contains(event.target)
    ) {
      closeContextMenu();
    }
  });
  globalThis.addEventListener("scroll", closeContextMenu, true);
  globalThis.addEventListener("resize", closeContextMenu);

  // Toggle widget
  toggle.addEventListener("click", () => {
    widget.classList.toggle("is-open");
    if (widget.classList.contains("is-open")) {
      localStorage.setItem("messenger_open", "true");
      loadFriendList();
      startListPolling();
    } else {
      localStorage.removeItem("messenger_open");
      stopListPolling();
      stopChatPolling();
    }
  });

  closeBtn.addEventListener("click", () => {
    widget.classList.remove("is-open");
    localStorage.removeItem("messenger_open");
    stopListPolling();
    stopChatPolling();
  });

  backBtn.addEventListener("click", () => {
    chatView.classList.remove("active");
    listView.classList.remove("is-hidden-display");
    activeFriendId = null;
    localStorage.removeItem("messenger_active_friend");
    title.textContent = "Chat";
    stopChatPolling();
    loadFriendList();
  });

  // Load friend list
  async function loadFriendList() {
    if (!widget.classList.contains("is-open")) return;
    try {
      // Using GET since it's just fetching data
      const r = await fetch("/messenger-api?action=get_friends");
      const data = await r.json();
      if (data.success) {
        renderFriendList(data.friends);
        updateUnreadBadge(data.friends);

        // If we are currently in list view, ensure it's visible
        if (!activeFriendId) {
          chatView.classList.remove("active");
          listView.classList.remove("is-hidden-display");
        }
      }
    } catch (error_) {
      debugCatch("Failed to load friends", error_);
      console.error("Failed to load friends", error_);
    }
  }

  const isActiveChatWithFriend = (friendId = null) => {
    if (!activeFriendId) return false;
    if (!widget.classList.contains("is-open")) return false;
    if (!chatView.classList.contains("active")) return false;
    if (friendId === null) return true;
    return String(friendId) === String(activeFriendId);
  };

  function renderFriendList(friends) {
    const hiddenList = getHiddenList();
    const visibleFriends = friends.filter((f) => {
      const idKey = String(f.id);
      const unread = Number.parseInt(f.unread_count, 10) || 0;
      if (!hiddenList.includes(idKey)) return true;
      return unread > 0 || isActiveChatWithFriend(f.id);
    });
    if (!visibleFriends.length) {
      friendList.innerHTML =
        '<div class="text-muted text-center messenger-empty">No chats yet. Start a conversation to see it here.</div>';
      return;
    }

    const makeFallbackAvatar = (name) => {
      const initial = (name || "?").trim().charAt(0).toUpperCase() || "?";
      const svg = `<svg xmlns="http://www.w3.org/2000/svg" width="80" height="80" viewBox="0 0 80 80"><rect width="80" height="80" rx="40" fill="#2a2440"/><text x="50%" y="52%" text-anchor="middle" dominant-baseline="middle" font-family="Inter, Arial, sans-serif" font-size="34" fill="#c9b8ff">${initial}</text></svg>`;
      return `data:image/svg+xml;utf8,${encodeURIComponent(svg)}`;
    };

    friendList.innerHTML = visibleFriends
      .map((f) => {
        const displayName = f.display_name || f.nickname || "";
        const avatar = f.profile_photo || makeFallbackAvatar(displayName);
        const unreadCount = isActiveChatWithFriend(f.id)
          ? 0
          : Number.parseInt(f.unread_count, 10) || 0;
        const unread =
          unreadCount > 0
            ? `<span class="messenger-badge messenger-badge--inline">${unreadCount}</span>`
            : "";
        const statusClass = f.status || "offline";
        const lastMsg = f.last_message ? f.last_message : "No messages yet";
        const nickname = f.nickname || "";

        return `
                <div class="messenger-friend-item" data-id="${f.id}" data-name="${displayName}" data-nickname="${nickname}" data-avatar="${avatar}" data-status="${statusClass}">
                    <div class="messenger-avatar-wrap">
                        <img src="${avatar}" alt="" class="messenger-avatar">
                        <span class="messenger-status ${statusClass}"></span>
                    </div>
                    <div class="messenger-friend-info">
                        <span class="messenger-friend-name">${displayName} ${unread}</span>
                        <span class="messenger-last-msg">${lastMsg}</span>
                    </div>
                </div>
            `;
      })
      .join("");

    // Friend click
    friendList.querySelectorAll(".messenger-friend-item").forEach((item) => {
      item.addEventListener("click", () => {
        const id = item.dataset.id;
        const name = item.dataset.name;
        const nickname = item.dataset.nickname || "";
        const avatar = item.dataset.avatar;
        const status = item.dataset.status;
        openChat(id, name, nickname, avatar, status);
      });
    });

    friendList.querySelectorAll(".messenger-friend-item").forEach((item) => {
      item.addEventListener("contextmenu", (event) => {
        event.preventDefault();
        openContextMenu(item, item.dataset.id);
      });
    });
  }

  function openChat(id, name, nickname, avatar, status) {
    activeFriendId = id;
    unhideFriend(id);
    listView.classList.add("is-hidden-display");
    chatView.classList.add("active");

    localStorage.setItem(
      "messenger_active_friend",
      JSON.stringify({ id, name, nickname, avatar, status }),
    );

    document.getElementById("chat-user-name").textContent = name;
    const profileLink = document.getElementById("chat-user-link");
    if (profileLink) {
      const handle = (nickname || name || "").trim();
      profileLink.href = handle ? `/@${encodeURIComponent(handle)}` : "#";
    }
    const userAvatar = document.getElementById("chat-user-avatar");
    if (userAvatar) userAvatar.src = avatar;
    const userStatus = document.getElementById("chat-user-status");
    if (userStatus) userStatus.className = `messenger-status ${status}`;

    loadMessages();
    startChatPolling();
  }

  async function loadMessages() {
    if (!activeFriendId || !widget.classList.contains("is-open")) return;
    try {
      const formData = new FormData();
      formData.append("action", "get_messages");
      formData.append("friend_id", activeFriendId);
      if (csrfToken) {
        formData.append("csrf_token", csrfToken);
      }
      const r = await fetch("/messenger-api", {
        method: "POST",
        body: formData,
      });
      const data = await r.json();
      if (data.success) {
        renderMessages(data.messages);
      }
    } catch (error_) {
      debugCatch("Failed to load messages", error_);
      console.error("Failed to load messages", error_);
    }
  }

  function renderMessages(messages) {
    if (!messages || messages.length === 0) {
      messagesContainer.innerHTML =
        '<div class="text-muted text-center messenger-empty messenger-empty--compact">No messages yet.</div>';
      return;
    }
    const isAtBottom =
      messagesContainer.scrollHeight - messagesContainer.scrollTop <=
      messagesContainer.clientHeight + 100;

    messagesContainer.innerHTML = messages
      .map((m) => {
        const type = m.sender_id == activeFriendId ? "received" : "sent";
        const time = new Date(m.created_at * 1000).toLocaleTimeString([], {
          hour: "2-digit",
          minute: "2-digit",
        });
        const imageUrl = m.image_url ? String(m.image_url) : "";
        const safeImageUrl = imageUrl ? encodeURI(imageUrl) : "";
        const openImageUrl = safeImageUrl.replaceAll("'", "%27");
        const imageHtml = safeImageUrl
          ? '<div class="messenger-msg-image"><img src="' +
            safeImageUrl +
            '" alt="" data-open-image="' +
            openImageUrl +
            '"></div>'
          : "";
        const bodyHtml = m.body
          ? `<div class="messenger-msg-body">${m.body}</div>`
          : "";
        const isRead = !!m.is_read;
        const receiptClass = isRead
          ? "read-receipt--read"
          : "read-receipt--sent";
        const receiptTitle = isRead ? "Read" : "Not read";
        const receipt =
          type === "sent"
            ? `
                <span class="messenger-receipt-badge ${receiptClass}" title="${receiptTitle}">
                    <svg viewBox="0 0 16 16" fill="currentColor"><path d="M13.854 3.646a.5.5 0 0 1 0 .708l-7 7a.5.5 0 0 1-.708 0l-3.5-3.5a.5.5 0 1 1 .708-.708L6.5 10.293l6.646-6.647a.5.5 0 0 1 .708 0z"/></svg>
                </span>`
            : "";

        return `
                <div class="messenger-msg messenger-msg--${type}">
                    ${imageHtml}
                    ${bodyHtml}
                    <span class="messenger-msg-time">${time}</span>
                    ${receipt}
                </div>
            `;
      })
      .join("");

    if (isAtBottom) {
      messagesContainer.scrollTop = messagesContainer.scrollHeight;
    }
  }

  // Send message
  messengerForm.addEventListener("submit", async (e) => {
    e.preventDefault();
    const body = messengerInput.value.trim();
    if (!body || !activeFriendId) return;

    messengerInput.value = "";

    try {
      const formData = new FormData();
      formData.append("action", "send_message");
      formData.append("friend_id", activeFriendId);
      formData.append("body", body);

      if (csrfToken) {
        formData.append("csrf_token", csrfToken);
      }

      const r = await fetch("/messenger-api", {
        method: "POST",
        body: formData,
      });
      const data = await r.json();
      if (data.success) {
        loadMessages();
      }
    } catch (error_) {
      debugCatch("Failed to send message", error_);
      console.error("Failed to send message", error_);
    }
  });

  function updateUnreadBadge(friends) {
    const total = friends.reduce((acc, f) => {
      const count = Number.parseInt(f.unread_count, 10) || 0;
      return acc + (isActiveChatWithFriend(f.id) ? 0 : count);
    }, 0);
    if (unreadBadge) {
      if (total > 0) {
        unreadBadge.textContent = total;
        unreadBadge.classList.remove("is-hidden-display");
      } else {
        unreadBadge.classList.add("is-hidden-display");
      }
    }
  }

  function startChatPolling() {
    stopChatPolling();
    pollInterval = setInterval(loadMessages, 5000);
  }

  function stopChatPolling() {
    if (pollInterval) clearInterval(pollInterval);
  }

  function startListPolling() {
    stopListPolling();
    listPollInterval = setInterval(loadFriendList, 15000);
  }

  function stopListPolling() {
    if (listPollInterval) clearInterval(listPollInterval);
  }

  // Initial unread check
  async function checkUnreadTotal() {
    try {
      // Using GET for data retrieval
      const excludeActive = isActiveChatWithFriend()
        ? `&exclude_friend_id=${encodeURIComponent(activeFriendId)}`
        : "";
      const r = await fetch(
        `/messenger-api?action=get_unread_total${excludeActive}`,
      );
      const data = await r.json();
      if (data.success) {
        if (data.total > 0) {
          unreadBadge.textContent = data.total;
          unreadBadge.classList.remove("is-hidden-display");
        } else {
          unreadBadge.classList.add("is-hidden-display");
        }
      }
    } catch (error_) {
      debugCatch("checkUnreadTotal failed", error_);
      // Silently fail for background unread check
    }
  }

  checkUnreadTotal();
  setInterval(checkUnreadTotal, 5000);

  const closeActiveChatAndReturnToList = () => {
    if (String(activeFriendId) === String(contextTargetId)) {
      chatView.classList.remove("active");
      listView.classList.remove("is-hidden-display");
      activeFriendId = null;
      localStorage.removeItem("messenger_active_friend");
    }
  };

  const handleHideChatAction = () => {
    hideFriend(contextTargetId);
    closeActiveChatAndReturnToList();
    closeContextMenu();
    loadFriendList();
  };

  const clearChatForContextTarget = async () => {
    if (!confirm("Clear all messages in this chat?")) {
      return;
    }
    try {
      const formData = new FormData();
      formData.append("action", "clear_chat");
      formData.append("friend_id", contextTargetId);
      if (csrfToken) {
        formData.append("csrf_token", csrfToken);
      }
      const response = await fetch("/messenger-api", {
        method: "POST",
        body: formData,
      });
      const data = await response.json();
      if (data?.success) {
        if (String(activeFriendId) === String(contextTargetId)) {
          renderMessages([]);
        }
        loadFriendList();
      } else if (data?.error) {
        alert(data.error);
      }
    } catch (error_) {
      debugCatch("Messenger clear chat failed", error_);
      console.error("Clear chat failed:", error_);
      alert("Failed to clear chat. Please try again.");
    } finally {
      closeContextMenu();
    }
  };

  contextMenu.addEventListener("click", async (event) => {
    const actionBtn = event.target.closest("[data-action]");
    if (!actionBtn || !contextTargetId) return;
    const action = actionBtn.dataset.action;

    if (action === "hide_chat") {
      handleHideChatAction();
      return;
    }

    if (action === "clear_chat") {
      await clearChatForContextTarget();
    }
  });

  // Restore state
  const isMessengerOpen = localStorage.getItem("messenger_open") === "true";
  const savedActiveFriend = localStorage.getItem("messenger_active_friend");

  if (isMessengerOpen) {
    widget.classList.add("is-open");

    if (savedActiveFriend) {
      try {
        const friend = JSON.parse(savedActiveFriend);
        openChat(
          friend.id,
          friend.name,
          friend.nickname || "",
          friend.avatar,
          friend.status,
        );
        startListPolling(); // Also poll list in background
      } catch (error_) {
        debugCatch("Restore messenger state failed", error_);
        localStorage.removeItem("messenger_active_friend");
        loadFriendList();
        startListPolling();
      }
    } else {
      loadFriendList();
      startListPolling();
    }
  }

  // Global listener for message buttons
  document.addEventListener("click", (e) => {
    const btn = e.target.closest("[data-messenger-open]");
    if (btn) {
      const id = btn.dataset.messengerOpen;
      const name = btn.dataset.messengerName;
      const fallback = `data:image/svg+xml;utf8,${encodeURIComponent('<svg xmlns="http://www.w3.org/2000/svg" width="80" height="80" viewBox="0 0 80 80"><rect width="80" height="80" rx="40" fill="#2a2440"/><text x="50%" y="52%" text-anchor="middle" dominant-baseline="middle" font-family="Inter, Arial, sans-serif" font-size="34" fill="#c9b8ff">?</text></svg>')}`;
      const avatar = btn.dataset.messengerAvatar || fallback;
      const status = btn.dataset.messengerStatus || "offline";
      const nickname = btn.dataset.messengerNickname || "";

      if (!widget.classList.contains("is-open")) {
        widget.classList.add("is-open");
        localStorage.setItem("messenger_open", "true");
      }
      openChat(id, name, nickname, avatar, status);
    }
  });
}

/**
 * Cursor fire glow — animated ember effect on hover over interactive elements.
 * Only active when data-cursor-theme="1" is set on <body>.
 */
function initCursorFire() {
  if (document.body.dataset.cursorTheme !== "1") return;

  // Build the fire glow DOM
  const fire = document.createElement("div");
  fire.className = "cursor-fire";
  fire.setAttribute("aria-hidden", "true");
  fire.innerHTML =
    '<div class="cursor-fire-core"></div>' +
    '<div class="cursor-fire-spark"></div>' +
    '<div class="cursor-fire-spark"></div>' +
    '<div class="cursor-fire-spark"></div>';
  document.body.appendChild(fire);

  // Interactive selectors matching the CSS cursor-hover rule
  const hoverSelectors =
    "a, button, [role='button'], input, select, textarea, .btn, .product-list-card, .collection-list-card";

  let isHovering = false;
  let raf = 0;
  let mx = -100;
  let my = -100;

  const moveFire = () => {
    fire.style.left = mx + "px";
    fire.style.top = my + "px";
    raf = 0;
  };

  document.addEventListener(
    "mousemove",
    (e) => {
      mx = e.clientX;
      my = e.clientY;
      if (!raf) raf = requestAnimationFrame(moveFire);
    },
    { passive: true },
  );

  document.addEventListener(
    "mouseover",
    (e) => {
      if (e.target.closest(hoverSelectors)) {
        if (!isHovering) {
          isHovering = true;
          fire.classList.add("is-hovering");
        }
      }
    },
    { passive: true },
  );

  document.addEventListener(
    "mouseout",
    (e) => {
      if (e.target.closest(hoverSelectors)) {
        isHovering = false;
        fire.classList.remove("is-hovering");
      }
    },
    { passive: true },
  );
}

/* ──────────────────────────────────────────────────────────
   Mobile Bottom‑Nav Drawer
   ────────────────────────────────────────────────────────── */
function initMobileDrawer() {
  const toggle = document.querySelector("[data-mobile-drawer-toggle]");
  const drawer = document.querySelector("[data-mobile-drawer]");
  const overlay = document.querySelector("[data-mobile-drawer-overlay]");
  const closeBtn = document.querySelector("[data-mobile-drawer-close]");

  if (!toggle || !drawer || !overlay) {
    return;
  }

  function openDrawer() {
    drawer.classList.add("is-active");
    overlay.classList.add("is-active");
    document.body.classList.add("drawer-open");
  }

  function closeDrawer() {
    drawer.classList.remove("is-active");
    overlay.classList.remove("is-active");
    document.body.classList.remove("drawer-open");
  }

  toggle.addEventListener("click", function (e) {
    e.preventDefault();
    if (drawer.classList.contains("is-active")) {
      closeDrawer();
    } else {
      openDrawer();
    }
  });

  overlay.addEventListener("click", function () {
    closeDrawer();
  });

  if (closeBtn) {
    closeBtn.addEventListener("click", function () {
      closeDrawer();
    });
  }

  /* Close drawer on any navigation link click */
  const drawerLinks = drawer.querySelectorAll("a[href]");
  for (const link of drawerLinks) {
    link.addEventListener("click", function () {
      closeDrawer();
    });
  }

  /* Close drawer on form submit (currency / language selectors) */
  const drawerForms = drawer.querySelectorAll("form");
  for (const form of drawerForms) {
    form.addEventListener("click", function () {
      closeDrawer();
    });
  }
}
