/*
 * Hardest Maze game V1
 *
 * CSP-safe mini-game implementation (external JS only).
 * Uses the shared mini-game leaderboard API.
 */

(() => {
  const stageEl = document.getElementById("maze-runner-stage");
  const canvasWrapEl = document.getElementById("maze-runner-canvas-wrap");
  const canvas = document.getElementById("maze-runner-canvas");
  const pageEl = stageEl ? stageEl.closest(".games-page--game") : null;
  const fullscreenButtons = document.querySelectorAll(
    "[data-game-fullscreen-toggle]",
  );

  if (!stageEl || !canvasWrapEl || !canvas) {
    return;
  }

  const ctx = canvas.getContext("2d");
  if (!ctx) {
    return;
  }

  const hud = {
    level: document.getElementById("maze-level"),
    size: document.getElementById("maze-size"),
    door: document.getElementById("maze-door"),
    time: document.getElementById("maze-time"),
    maxLevel: document.getElementById("maze-score"),
    status: document.getElementById("maze-status"),
  };

  const restartBtn = document.getElementById("maze-restart-btn");
  const leaderboardEl = document.getElementById("mazeRunnerLeaderboard");
  const leaderboardListEl = document.getElementById("mazeLeaderboardList");
  const leaderboardStatusEl = document.getElementById("mazeLeaderboardStatus");

  if (
    !hud.level ||
    !hud.size ||
    !hud.door ||
    !hud.time ||
    !hud.maxLevel ||
    !hud.status ||
    !restartBtn
  ) {
    return;
  }

  const gameSlug = leaderboardEl
    ? String(leaderboardEl.dataset.gameSlug || "")
    : "";
  const isLeaderboardUserLoggedIn = leaderboardEl
    ? leaderboardEl.dataset.userLoggedIn === "1"
    : false;
  const csrfToken =
    document
      .querySelector('meta[name="csrf-token"]')
      ?.getAttribute("content") || "";

  const readFiniteNumberSetting = (value, fallback) => {
    const parsed = Number(value);
    return Number.isFinite(parsed) ? parsed : fallback;
  };

  const clamp = (value, min, max) => Math.min(max, Math.max(min, value));

  const START_TIME_SECONDS = Math.round(
    clamp(
      readFiniteNumberSetting(stageEl.dataset.startTimeSeconds, 90),
      20,
      300,
    ),
  );
  const DOOR_SHUFFLE_SECONDS = clamp(
    readFiniteNumberSetting(stageEl.dataset.doorShuffleSeconds, 10),
    3,
    30,
  );

  const CHAOS_CHANCE = 0.1;
  const CHAOS_RECOVERY_MS = 6000;
  const DOOR_TRANSITION_SECONDS = 1.8;
  const MOVE_COOLDOWN_MS = 60;
  const PLAYER_LERP_SPEED = 20;
  const HOLD_REPEAT_DELAY_MS = 150;
  const LEVEL_TIME_BONUS_MULTIPLIER = 30;
  const INITIAL_DOOR_OPEN_CHANCE = 0.72;
  const NORMAL_SHUFFLE_OPEN_CHANCE = 0.78;
  const CHAOS_SHUFFLE_OPEN_CHANCE = 0.58;
  const NORMAL_SHUFFLE_CHANGE_RATIO = 0.12;
  const CHAOS_SHUFFLE_CHANGE_RATIO = 0.28;
  const NORMAL_SHUFFLE_MIN_CHANGES = 4;
  const NORMAL_SHUFFLE_MAX_CHANGES = 28;
  const CHAOS_SHUFFLE_MIN_CHANGES = 8;
  const CHAOS_SHUFFLE_MAX_CHANGES = 72;

  const DIRS = [
    { dx: 0, dy: -1, wall: "N", opposite: "S" },
    { dx: 1, dy: 0, wall: "E", opposite: "W" },
    { dx: 0, dy: 1, wall: "S", opposite: "N" },
    { dx: -1, dy: 0, wall: "W", opposite: "E" },
  ];

  const keyMap = new Map([
    ["arrowup", [0, -1]],
    ["w", [0, -1]],
    ["arrowdown", [0, 1]],
    ["s", [0, 1]],
    ["arrowleft", [-1, 0]],
    ["a", [-1, 0]],
    ["arrowright", [1, 0]],
    ["d", [1, 0]],
  ]);

  const state = {
    level: 1,
    rows: 4,
    cols: 4,
    cells: [],
    doors: [],
    doorMap: new Map(),
    player: { x: 0, y: 0 },
    playerRender: { x: 0, y: 0 },
    exit: { x: 0, y: 0 },
    cellSize: 24,
    padding: 20,
    nextDoorSwapAt: performance.now() + DOOR_SHUFFLE_SECONDS * 1000,
    chaosActiveUntil: 0,
    lastShuffleWasChaos: false,
    canMoveAt: 0,
    maxLevelReached: 1,
    timeRemaining: START_TIME_SECONDS,
    gameOver: false,
    lastFrameTime: performance.now(),
    canvasSize: 860,
    statusClearAt: 0,
  };

  const heldMoveKeys = new Set();
  const heldMovePriority = [];
  const heldMoveDownAt = new Map();

  let leaderboardFetchSerial = 0;
  let currentUserBestLeaderboardLevel = 0;
  let activeVerifiedRound = null;
  let pendingVerifiedRoundPromise = null;
  let lastMidGameSubmitTime = 0;
  const MIN_MIDGAME_SUBMIT_MS = 4000;

  const setFullscreenButtonLabels = (isOn) => {
    fullscreenButtons.forEach((btn) => {
      if (!btn) return;
      btn.textContent = isOn ? "Exit Fullscreen" : "Fullscreen";
    });
  };

  const applyFullscreenState = (isOn) => {
    stageEl.classList.toggle("is-fullscreen", isOn);
    if (pageEl) {
      pageEl.classList.toggle("is-fullscreen", isOn);
    }
    setFullscreenButtonLabels(isOn);
    resizeCanvas();
  };

  fullscreenButtons.forEach((btn) => {
    btn.addEventListener("click", () => {
      const next = !stageEl.classList.contains("is-fullscreen");
      applyFullscreenState(next);
    });
  });

  applyFullscreenState(stageEl.classList.contains("is-fullscreen"));

  function resizeCanvas() {
    const rect = canvasWrapEl.getBoundingClientRect();
    const maxByViewport = Math.floor(
      globalThis.innerHeight *
        (stageEl.classList.contains("is-fullscreen") ? 0.72 : 0.62),
    );
    const cssSize = clamp(
      Math.floor(Math.min(rect.width || 860, maxByViewport || 860)),
      320,
      980,
    );
    const dpr = Math.min(globalThis.devicePixelRatio || 1, 2);

    canvas.style.width = `${cssSize}px`;
    canvas.style.height = `${cssSize}px`;

    const pixelSize = Math.max(1, Math.floor(cssSize * dpr));
    if (canvas.width !== pixelSize || canvas.height !== pixelSize) {
      canvas.width = pixelSize;
      canvas.height = pixelSize;
    }

    ctx.setTransform(dpr, 0, 0, dpr, 0, 0);
    state.canvasSize = cssSize;
    recomputeBoardSizing();
  }

  globalThis.addEventListener("resize", resizeCanvas);

  const randInt = (max) => Math.floor(Math.random() * max);

  function shuffle(array) {
    for (let i = array.length - 1; i > 0; i -= 1) {
      const j = randInt(i + 1);
      [array[i], array[j]] = [array[j], array[i]];
    }
    return array;
  }

  function cellIndex(x, y) {
    return y * state.cols + x;
  }

  function inBounds(x, y) {
    return x >= 0 && y >= 0 && x < state.cols && y < state.rows;
  }

  function edgeKey(a, b) {
    const first = a.y < b.y || (a.y === b.y && a.x <= b.x) ? a : b;
    const second = first === a ? b : a;
    return `${first.x},${first.y}|${second.x},${second.y}`;
  }

  function makeGrid() {
    state.cells = Array.from({ length: state.rows * state.cols }, () => ({
      walls: { N: true, E: true, S: true, W: true },
      visited: false,
    }));
  }

  function generateMazeDFS() {
    makeGrid();

    const stack = [{ x: 0, y: 0 }];
    state.cells[cellIndex(0, 0)].visited = true;

    while (stack.length) {
      const current = stack.at(-1);
      if (!current) {
        break;
      }

      const neighbors = [];

      for (const dir of DIRS) {
        const nx = current.x + dir.dx;
        const ny = current.y + dir.dy;
        if (!inBounds(nx, ny)) continue;
        if (state.cells[cellIndex(nx, ny)].visited) continue;
        neighbors.push({ dir, nx, ny });
      }

      if (!neighbors.length) {
        stack.pop();
        continue;
      }

      const pick = neighbors[randInt(neighbors.length)];
      const currentCell = state.cells[cellIndex(current.x, current.y)];
      const nextCell = state.cells[cellIndex(pick.nx, pick.ny)];

      currentCell.walls[pick.dir.wall] = false;
      nextCell.walls[pick.dir.opposite] = false;
      nextCell.visited = true;

      stack.push({ x: pick.nx, y: pick.ny });
    }

    for (const cell of state.cells) {
      delete cell.visited;
    }
  }

  function canTravel(x, y, nx, ny, ignoreDoorState = false) {
    if (!inBounds(nx, ny)) return false;

    const dx = nx - x;
    const dy = ny - y;
    let wallKey = null;

    if (dx === 1 && dy === 0) wallKey = "E";
    else if (dx === -1 && dy === 0) wallKey = "W";
    else if (dx === 0 && dy === 1) wallKey = "S";
    else if (dx === 0 && dy === -1) wallKey = "N";
    else return false;

    const current = state.cells[cellIndex(x, y)];
    if (current.walls[wallKey]) return false;

    if (!ignoreDoorState) {
      const key = edgeKey({ x, y }, { x: nx, y: ny });
      const door = state.doorMap.get(key);
      if (door && !door.open) return false;
    }

    return true;
  }

  function rebuildPathFromParent(endNode, parent) {
    const path = [endNode];
    let pKey = `${endNode.x},${endNode.y}`;

    while (parent.has(pKey)) {
      const previousNode = parent.get(pKey);
      if (!previousNode) {
        break;
      }

      path.push(previousNode);
      pKey = `${previousNode.x},${previousNode.y}`;
    }

    path.reverse();
    return path;
  }

  function findPath(start, end, ignoreDoorState = false) {
    const queue = [start];
    let head = 0;
    const visited = new Set([`${start.x},${start.y}`]);
    const parent = new Map();

    while (head < queue.length) {
      const cur = queue[head];
      head += 1;

      if (cur.x === end.x && cur.y === end.y) {
        return rebuildPathFromParent(cur, parent);
      }

      for (const dir of DIRS) {
        const nx = cur.x + dir.dx;
        const ny = cur.y + dir.dy;
        if (!canTravel(cur.x, cur.y, nx, ny, ignoreDoorState)) continue;

        const key = `${nx},${ny}`;
        if (visited.has(key)) continue;

        visited.add(key);
        parent.set(key, cur);
        queue.push({ x: nx, y: ny });
      }
    }

    return null;
  }

  function pathToEdgeSet(path) {
    const edges = new Set();
    if (!path || path.length < 2) return edges;

    for (let i = 0; i < path.length - 1; i += 1) {
      const a = path[i];
      const b = path[i + 1];
      edges.add(edgeKey(a, b));
    }

    return edges;
  }

  function collectOpenEdges() {
    const openEdges = [];

    for (let y = 0; y < state.rows; y += 1) {
      for (let x = 0; x < state.cols; x += 1) {
        const cell = state.cells[cellIndex(x, y)];

        if (x + 1 < state.cols && !cell.walls.E) {
          openEdges.push([
            { x, y },
            { x: x + 1, y },
          ]);
        }
        if (y + 1 < state.rows && !cell.walls.S) {
          openEdges.push([
            { x, y },
            { x, y: y + 1 },
          ]);
        }
      }
    }

    return openEdges;
  }

  function setDoorOpenState(door, isOpen) {
    door.open = isOpen;
    door.targetClosed = isOpen ? 0 : 1;
  }

  function addDoor(a, b) {
    const key = edgeKey(a, b);
    const open = Math.random() < INITIAL_DOOR_OPEN_CHANCE;
    const closedAmount = open ? 0 : 1;

    const door = {
      key,
      a,
      b,
      open,
      closedAmount,
      targetClosed: closedAmount,
      slideFromStart: Math.random() > 0.5,
    };

    state.doors.push(door);
    state.doorMap.set(key, door);
  }

  function tryDisruptPreviousPath(previousPath) {
    if (!Array.isArray(previousPath) || previousPath.length < 3) {
      return false;
    }

    const previousEdges = pathToEdgeSet(previousPath);
    const candidates = state.doors.filter(
      (door) => door.open && previousEdges.has(door.key),
    );

    if (!candidates.length) {
      return false;
    }

    shuffle(candidates);
    for (const door of candidates) {
      setDoorOpenState(door, false);
      if (findPath(state.player, state.exit, false)) {
        // When we close part of the old route, also open another corridor
        // outside that old route so the maze visibly "swaps" routes.
        const alternateOpenCandidates = state.doors.filter(
          (candidateDoor) =>
            !candidateDoor.open &&
            candidateDoor.key !== door.key &&
            !previousEdges.has(candidateDoor.key),
        );

        if (alternateOpenCandidates.length > 0) {
          const openDoor =
            alternateOpenCandidates[randInt(alternateOpenCandidates.length)];
          if (openDoor) {
            setDoorOpenState(openDoor, true);
          }
        }

        return true;
      }
      setDoorOpenState(door, true);
    }

    return false;
  }

  function applyDoorShuffleMutations(allowChaosBlock = false) {
    const shuffledDoors = shuffle([...state.doors]);

    const ratio = allowChaosBlock
      ? CHAOS_SHUFFLE_CHANGE_RATIO
      : NORMAL_SHUFFLE_CHANGE_RATIO;
    const minChanges = allowChaosBlock
      ? CHAOS_SHUFFLE_MIN_CHANGES
      : NORMAL_SHUFFLE_MIN_CHANGES;
    const maxChanges = allowChaosBlock
      ? CHAOS_SHUFFLE_MAX_CHANGES
      : NORMAL_SHUFFLE_MAX_CHANGES;
    const targetChanges = clamp(
      Math.round(state.doors.length * ratio),
      minChanges,
      maxChanges,
    );
    const openChance = allowChaosBlock
      ? CHAOS_SHUFFLE_OPEN_CHANCE
      : NORMAL_SHUFFLE_OPEN_CHANCE;

    let changed = 0;
    for (const door of shuffledDoors) {
      if (changed >= targetChanges) {
        break;
      }

      const nextOpen = Math.random() < openChance;
      if (nextOpen === door.open) {
        continue;
      }

      setDoorOpenState(door, nextOpen);
      changed += 1;
    }

    // Ensure some visible variation when random choices matched previous state too often.
    if (changed < minChanges) {
      for (const door of shuffledDoors) {
        if (changed >= minChanges) {
          break;
        }

        setDoorOpenState(door, !door.open);
        changed += 1;
      }
    }
  }

  function ensurePathFromPlayer() {
    let path = findPath(state.player, state.exit, false);
    if (path) return;

    path = findPath(state.player, state.exit, true);
    if (!path) return;

    for (let i = 0; i < path.length - 1; i += 1) {
      const key = edgeKey(path[i], path[i + 1]);
      const door = state.doorMap.get(key);
      if (door) {
        setDoorOpenState(door, true);
      }
    }
  }

  function forceNoPathFromPlayer() {
    const path = findPath(state.player, state.exit, false);
    if (!path) return true;

    const routeDoors = [];
    for (let i = 0; i < path.length - 1; i += 1) {
      const key = edgeKey(path[i], path[i + 1]);
      const door = state.doorMap.get(key);
      if (door?.open) routeDoors.push(door);
    }

    shuffle(routeDoors);
    for (const door of routeDoors) {
      setDoorOpenState(door, false);
      if (!findPath(state.player, state.exit, false)) return true;
      setDoorOpenState(door, true);
    }

    const openDoors = state.doors.filter((door) => door.open);
    shuffle(openDoors);
    for (const door of openDoors) {
      setDoorOpenState(door, false);
      if (!findPath(state.player, state.exit, false)) return true;
      setDoorOpenState(door, true);
    }

    return false;
  }

  function createDoors() {
    state.doors = [];
    state.doorMap.clear();

    const openEdges = collectOpenEdges();
    for (const pair of openEdges) {
      if (!pair) continue;
      addDoor(pair[0], pair[1]);
    }

    ensurePathFromPlayer();
  }

  function toggleDoors(allowChaosBlock = false, previousPath = null) {
    applyDoorShuffleMutations(allowChaosBlock);

    if (allowChaosBlock) {
      forceNoPathFromPlayer();
      return;
    }

    ensurePathFromPlayer();

    // Prefer a route change after shuffle when an alternate path exists,
    // instead of always preserving the exact same run-to-exit corridor.
    if (previousPath) {
      const rerouted = tryDisruptPreviousPath(previousPath);
      if (rerouted) {
        ensurePathFromPlayer();
      }
    }
  }

  function rollChaosThisShuffle() {
    if (state.lastShuffleWasChaos) {
      state.lastShuffleWasChaos = false;
      return false;
    }

    const willBeChaos = Math.random() < CHAOS_CHANCE;
    if (willBeChaos) {
      state.lastShuffleWasChaos = true;
    }

    return willBeChaos;
  }

  function levelDimensions(level) {
    const safeLevel = Math.max(1, Math.floor(level));
    const baseSize = Math.min(4 + Math.floor((safeLevel - 1) * 0.6), 34);

    if (safeLevel === 1) {
      return { rows: 4, cols: 4 };
    }

    const maxDifference = Math.min(5, Math.max(1, safeLevel - 1));
    const difference = randInt(maxDifference + 1);
    const majorDelta = Math.ceil(difference / 2);
    const minorDelta = Math.floor(difference / 2);
    const rowMajor = Math.random() < 0.5;

    const rows = clamp(baseSize + (rowMajor ? majorDelta : -minorDelta), 4, 34);
    const cols = clamp(baseSize + (rowMajor ? -minorDelta : majorDelta), 4, 34);

    return { rows, cols };
  }

  function recomputeBoardSizing() {
    state.padding = Math.max(12, Math.floor(state.canvasSize * 0.035));
    const maxBoard = state.canvasSize - state.padding * 2;
    const largest = Math.max(state.rows, state.cols);
    const raw = Math.floor(maxBoard / largest);
    state.cellSize = clamp(raw, 12, 48);
  }

  function setStatus(text, tone = "good") {
    hud.status.textContent = text;
    hud.status.classList.remove("is-good", "is-warn", "is-bad");
    if (tone === "warn") {
      hud.status.classList.add("is-warn");
    } else if (tone === "bad") {
      hud.status.classList.add("is-bad");
    } else {
      hud.status.classList.add("is-good");
    }
  }

  function updateHud() {
    hud.level.textContent = String(state.level);
    hud.size.textContent = `${state.cols} × ${state.rows}`;
    hud.maxLevel.textContent = String(
      Math.max(1, Math.floor(state.maxLevelReached)),
    );

    const secondsLeft = Math.max(0, Math.ceil(state.timeRemaining));
    const minutesLeft = Math.floor(secondsLeft / 60);
    const timeLabel = `${secondsLeft}s (${minutesLeft} Min)`;
    hud.time.textContent = timeLabel;
    hud.time.classList.toggle(
      "is-warn",
      state.timeRemaining <= 20 && !state.gameOver,
    );

    const msLeft = Math.max(0, state.nextDoorSwapAt - performance.now());
    hud.door.textContent = `${(msLeft / 1000).toFixed(1)}s`;
  }

  function startLevel(level, keepRunState = true) {
    state.level = level;
    state.maxLevelReached = Math.max(state.maxLevelReached, state.level);
    const dims = levelDimensions(level);
    state.rows = dims.rows;
    state.cols = dims.cols;

    state.player = { x: 0, y: 0 };
    state.playerRender = { x: 0, y: 0 };
    state.exit = { x: state.cols - 1, y: state.rows - 1 };

    generateMazeDFS();
    createDoors();

    recomputeBoardSizing();
    state.nextDoorSwapAt = performance.now() + DOOR_SHUFFLE_SECONDS * 1000;
    state.chaosActiveUntil = 0;
    state.lastShuffleWasChaos = false;
    state.canMoveAt = 0;

    if (!keepRunState) {
      state.maxLevelReached = 1;
      state.timeRemaining = START_TIME_SECONDS;
    }

    state.lastFrameTime = performance.now();
    if (!state.gameOver) {
      setStatus("Find the exit!", "good");
    }

    updateHud();
  }

  function endRun(reason = "Time up") {
    if (state.gameOver) {
      return;
    }

    state.gameOver = true;
    setStatus(`${reason} — run ended`, "bad");
    updateHud();

    const achievedLevel = Math.max(1, Math.floor(state.maxLevelReached));
    if (achievedLevel > 0) {
      submitLeaderboardLevel(achievedLevel);
    } else {
      fetchLeaderboard();
    }
  }

  function handleLevelComplete() {
    if (state.gameOver) {
      return;
    }

    const completedLevel = Math.max(1, Math.floor(state.level));
    const nextLevel = state.level + 1;
    const timeBonus = completedLevel * LEVEL_TIME_BONUS_MULTIPLIER;
    state.timeRemaining += timeBonus;

    setStatus(`Level ${state.level} cleared! +${timeBonus}s`, "good");
    state.statusClearAt = performance.now() + 1400;

    startLevel(nextLevel, true);

    /* Submit score mid-game if this level beats the user's known best.
       Client-side throttle prevents rapid-fire requests on fast early levels;
       the server also enforces per-user rate limits and round timing. */
    const now = performance.now();
    if (
      isLeaderboardUserLoggedIn &&
      completedLevel > currentUserBestLeaderboardLevel &&
      now - lastMidGameSubmitTime >= MIN_MIDGAME_SUBMIT_MS
    ) {
      lastMidGameSubmitTime = now;
      submitLeaderboardLevel(completedLevel, false, true);
    }
  }

  function tryMove(dx, dy) {
    if (state.gameOver) return;

    const now = performance.now();
    if (now < state.canMoveAt) return;
    state.canMoveAt = now + MOVE_COOLDOWN_MS;

    const nx = state.player.x + dx;
    const ny = state.player.y + dy;

    if (!canTravel(state.player.x, state.player.y, nx, ny, false)) return;

    state.player.x = nx;
    state.player.y = ny;
    if (state.player.x === state.exit.x && state.player.y === state.exit.y) {
      handleLevelComplete();
      return;
    }

    updateHud();
  }

  function animateDoors(deltaSeconds) {
    const speed = 1 / DOOR_TRANSITION_SECONDS;

    for (const door of state.doors) {
      const diff = door.targetClosed - door.closedAmount;
      if (Math.abs(diff) < 0.0001) {
        door.closedAmount = door.targetClosed;
        continue;
      }

      const step = Math.sign(diff) * speed * deltaSeconds;
      if (Math.abs(step) >= Math.abs(diff)) {
        door.closedAmount = door.targetClosed;
      } else {
        door.closedAmount += step;
      }
    }
  }

  function animatePlayer(deltaSeconds) {
    const t = clamp(deltaSeconds * PLAYER_LERP_SPEED, 0, 1);

    state.playerRender.x += (state.player.x - state.playerRender.x) * t;
    state.playerRender.y += (state.player.y - state.playerRender.y) * t;

    if (Math.abs(state.player.x - state.playerRender.x) < 0.001) {
      state.playerRender.x = state.player.x;
    }
    if (Math.abs(state.player.y - state.playerRender.y) < 0.001) {
      state.playerRender.y = state.player.y;
    }
  }

  function drawCellBackgrounds(offsetX, offsetY) {
    const cs = state.cellSize;

    ctx.fillStyle = "rgba(82, 196, 26, 0.18)";
    ctx.fillRect(offsetX + 2, offsetY + 2, cs - 4, cs - 4);

    ctx.fillStyle = "rgba(255, 214, 10, 0.22)";
    ctx.fillRect(
      offsetX + state.exit.x * cs + 2,
      offsetY + state.exit.y * cs + 2,
      cs - 4,
      cs - 4,
    );
  }

  function drawWalls(offsetX, offsetY) {
    const cs = state.cellSize;
    ctx.strokeStyle = "#d7deff";
    ctx.lineWidth = 2;

    for (let y = 0; y < state.rows; y += 1) {
      for (let x = 0; x < state.cols; x += 1) {
        const cell = state.cells[cellIndex(x, y)];
        const sx = offsetX + x * cs;
        const sy = offsetY + y * cs;

        if (cell.walls.N) {
          ctx.beginPath();
          ctx.moveTo(sx, sy);
          ctx.lineTo(sx + cs, sy);
          ctx.stroke();
        }
        if (cell.walls.E) {
          ctx.beginPath();
          ctx.moveTo(sx + cs, sy);
          ctx.lineTo(sx + cs, sy + cs);
          ctx.stroke();
        }
        if (cell.walls.S) {
          ctx.beginPath();
          ctx.moveTo(sx, sy + cs);
          ctx.lineTo(sx + cs, sy + cs);
          ctx.stroke();
        }
        if (cell.walls.W) {
          ctx.beginPath();
          ctx.moveTo(sx, sy);
          ctx.lineTo(sx, sy + cs);
          ctx.stroke();
        }
      }
    }
  }

  function drawDoors(offsetX, offsetY) {
    const cs = state.cellSize;
    ctx.strokeStyle = "#d7deff";
    ctx.lineWidth = 2;
    ctx.lineCap = "butt";

    for (const door of state.doors) {
      if (door.closedAmount <= 0.015) continue;

      const a = door.a;
      const b = door.b;

      const verticalDivider = a.y === b.y;
      const x = offsetX + Math.max(a.x, b.x) * cs;
      const y = offsetY + Math.max(a.y, b.y) * cs;

      if (verticalDivider) {
        const minY = offsetY + Math.min(a.y, b.y) * cs;
        const maxY = minY + cs;
        const total = maxY - minY;
        const amount = total * door.closedAmount;
        const startY = door.slideFromStart ? minY : maxY - amount;

        ctx.beginPath();
        ctx.moveTo(x, startY);
        ctx.lineTo(x, startY + amount);
        ctx.stroke();
      } else {
        const minX = offsetX + Math.min(a.x, b.x) * cs;
        const maxX = minX + cs;
        const total = maxX - minX;
        const amount = total * door.closedAmount;
        const startX = door.slideFromStart ? minX : maxX - amount;

        ctx.beginPath();
        ctx.moveTo(startX, y);
        ctx.lineTo(startX + amount, y);
        ctx.stroke();
      }
    }
  }

  function drawPlayer(offsetX, offsetY) {
    const cs = state.cellSize;
    const cx = offsetX + state.playerRender.x * cs + cs / 2;
    const cy = offsetY + state.playerRender.y * cs + cs / 2;

    ctx.beginPath();
    ctx.fillStyle = "#69b1ff";
    ctx.arc(cx, cy, cs * 0.28, 0, Math.PI * 2);
    ctx.fill();

    ctx.beginPath();
    ctx.strokeStyle = "#ffffff";
    ctx.lineWidth = 2;
    ctx.arc(cx, cy, cs * 0.28, 0, Math.PI * 2);
    ctx.stroke();
  }

  function renderBoard() {
    const boardW = state.cols * state.cellSize;
    const boardH = state.rows * state.cellSize;
    const offsetX = (state.canvasSize - boardW) / 2;
    const offsetY = (state.canvasSize - boardH) / 2;

    ctx.clearRect(0, 0, state.canvasSize, state.canvasSize);

    drawCellBackgrounds(offsetX, offsetY);
    drawWalls(offsetX, offsetY);
    drawDoors(offsetX, offsetY);
    drawPlayer(offsetX, offsetY);
  }

  function escapeHtml(value) {
    return String(value)
      .replaceAll("&", "&amp;")
      .replaceAll("<", "&lt;")
      .replaceAll(">", "&gt;")
      .replaceAll('"', "&quot;")
      .replaceAll("'", "&#39;");
  }

  function setLeaderboardStatus(text, isError = false) {
    if (!leaderboardStatusEl) {
      return;
    }
    leaderboardStatusEl.textContent = text;
    leaderboardStatusEl.classList.toggle("is-error", !!isError);
  }

  function renderLeaderboard(items) {
    if (!leaderboardListEl) {
      return;
    }

    currentUserBestLeaderboardLevel = 0;

    if (!Array.isArray(items) || items.length === 0) {
      leaderboardListEl.innerHTML =
        '<li class="maze-leaderboard-empty">No level records yet. Be the first.</li>';
      return;
    }

    leaderboardListEl.innerHTML = items
      .map((item) => {
        const rank = Number.isFinite(Number(item?.rank))
          ? Number(item.rank)
          : 0;
        const safeRank = Math.max(0, Math.floor(rank));
        const name = escapeHtml(item?.name || "Player");
        const rawProfileUrl =
          typeof item?.profile_url === "string" ? item.profile_url : "";
        const hasProfileLink = rawProfileUrl.startsWith("/@");
        const profileUrl = hasProfileLink ? escapeHtml(rawProfileUrl) : "";
        const leaderboardValue = Number(item?.score);
        const safeValue = Number.isFinite(leaderboardValue)
          ? leaderboardValue
          : 0;
        const levelReached = Math.max(0, Math.floor(safeValue));
        const levelText = escapeHtml(String(levelReached));
        const currentClass = item?.is_current_user ? " is-current" : "";

        if (
          item?.is_current_user &&
          levelReached > currentUserBestLeaderboardLevel
        ) {
          currentUserBestLeaderboardLevel = levelReached;
        }

        const nameMarkup = hasProfileLink
          ? `<a class="maze-leaderboard-name" href="${profileUrl}">${name}</a>`
          : `<span class="maze-leaderboard-name">${name}</span>`;

        return `
        <li class="maze-leaderboard-item${currentClass}">
          <span class="maze-leaderboard-rank">#${safeRank}</span>
          ${nameMarkup}
          <span class="maze-leaderboard-score">${levelText}</span>
        </li>
      `;
      })
      .join("");
  }

  function clearVerifiedRound() {
    activeVerifiedRound = null;
  }

  function startVerifiedRound() {
    if (!gameSlug || !isLeaderboardUserLoggedIn) {
      clearVerifiedRound();
      return Promise.resolve(null);
    }

    if (pendingVerifiedRoundPromise) {
      return pendingVerifiedRoundPromise;
    }

    pendingVerifiedRoundPromise = fetch(
      `/game-leaderboard-api?game=${encodeURIComponent(gameSlug)}&action=start-round`,
      {
        credentials: "same-origin",
      },
    )
      .then((r) => {
        if (!r.ok) {
          throw new Error("Could not start verified round");
        }
        return r.json();
      })
      .then((data) => {
        if (
          !data?.success ||
          !data?.round?.round_id ||
          !data?.round?.round_token
        ) {
          throw new Error(data?.error || "Could not start verified round");
        }

        activeVerifiedRound = {
          id: String(data.round.round_id),
          token: String(data.round.round_token),
          issuedAt: Number(data.round.issued_at || 0),
          expiresAt: Number(data.round.expires_at || 0),
          minSubmitSeconds: Number(data.round.min_submit_seconds || 0),
        };

        return activeVerifiedRound;
      })
      .catch((error) => {
        clearVerifiedRound();
        throw error;
      })
      .finally(() => {
        pendingVerifiedRoundPromise = null;
      });

    return pendingVerifiedRoundPromise;
  }

  function fetchLeaderboard() {
    if (!gameSlug || !leaderboardListEl) {
      return;
    }

    const serial = ++leaderboardFetchSerial;
    fetch(`/game-leaderboard-api?game=${encodeURIComponent(gameSlug)}&limit=50`)
      .then((r) => {
        if (!r.ok) {
          throw new Error("Leaderboard unavailable");
        }
        return r.json();
      })
      .then((data) => {
        if (serial !== leaderboardFetchSerial) {
          return;
        }

        if (!data?.success) {
          throw new Error("Leaderboard unavailable");
        }

        renderLeaderboard(data.items || []);
        setLeaderboardStatus("Hardest Maze game V1 Highest Level Leaderboard");
      })
      .catch(() => {
        if (serial !== leaderboardFetchSerial) {
          return;
        }
        setLeaderboardStatus("Leaderboard unavailable", true);
      });
  }

  /**
   * Check whether a round is too fresh for submission (min elapsed time).
   * @returns {boolean} true if the round is still too young to submit.
   */
  function isRoundTooFresh() {
    const minSec = Number(activeVerifiedRound?.minSubmitSeconds || 0);
    const issued = Number(activeVerifiedRound?.issuedAt || 0);
    if (minSec <= 0 || issued <= 0) {
      return false;
    }
    return Date.now() / 1000 - issued < minSec;
  }

  /** Handle a successful leaderboard POST response. */
  function handleSubmitSuccess(data, isMidGame) {
    clearVerifiedRound();
    renderLeaderboard(data.items || []);

    if (isMidGame) {
      /* Round token consumed server-side — obtain a fresh round for the
         next potential mid-game submission without disrupting gameplay. */
      startVerifiedRound().catch(() => {
        /* Next submit will surface the missing-round error if needed. */
      });
      return;
    }

    if (data?.submission_skipped) {
      setLeaderboardStatus(data?.message || "Level did not beat your best");
    } else {
      setLeaderboardStatus("Highest level submitted");
    }
  }

  /** Handle a failed leaderboard POST response (end-game only). */
  function handleSubmitError(error) {
    const message = String(error?.message || "Could not submit level");
    if (message.toLowerCase().includes("too quickly")) {
      setLeaderboardStatus("Level not submitted (round too short yet)");
    } else {
      setLeaderboardStatus(message, true);
    }
  }

  /**
   * Attempt to recover a verified round when none is active.
   * @returns {boolean} true if recovery was initiated (caller should return).
   */
  function awaitPendingRound(levelReached, isMidGame, waitedForPendingRound) {
    if (!pendingVerifiedRoundPromise || waitedForPendingRound) {
      return false;
    }
    pendingVerifiedRoundPromise
      .then(() => submitLeaderboardLevel(levelReached, true, isMidGame))
      .catch(() => {
        if (!isMidGame) {
          setLeaderboardStatus("Level verification unavailable", true);
          fetchLeaderboard();
        }
      });
    return true;
  }

  /**
   * Show leaderboard feedback only during end-of-game submissions.
   * Mid-game submissions stay silent to avoid distracting the player.
   */
  function endGameFeedback(isMidGame, message, isError = false) {
    if (isMidGame) {
      return;
    }
    if (message) {
      setLeaderboardStatus(message, isError);
    }
    fetchLeaderboard();
  }

  function submitLeaderboardLevel(
    levelReached,
    waitedForPendingRound = false,
    isMidGame = false,
  ) {
    if (!gameSlug || levelReached <= 0) {
      return;
    }

    if (!isLeaderboardUserLoggedIn || !csrfToken) {
      endGameFeedback(isMidGame, "");
      return;
    }

    if (levelReached <= currentUserBestLeaderboardLevel) {
      endGameFeedback(isMidGame, "Level did not beat your best");
      return;
    }

    if (!activeVerifiedRound?.id || !activeVerifiedRound?.token) {
      if (awaitPendingRound(levelReached, isMidGame, waitedForPendingRound)) {
        return;
      }
      endGameFeedback(isMidGame, "Round expired. Start a new run.", true);
      return;
    }

    if (isRoundTooFresh()) {
      endGameFeedback(isMidGame, "Level not submitted (round too short yet)");
      return;
    }

    if (!isMidGame) {
      setLeaderboardStatus("Saving highest level...");
    }

    fetch("/game-leaderboard-api", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "X-CSRF-Token": csrfToken,
      },
      credentials: "same-origin",
      body: JSON.stringify({
        game: gameSlug,
        score: levelReached,
        round_id: activeVerifiedRound.id,
        round_token: activeVerifiedRound.token,
      }),
    })
      .then((r) => r.json().then((data) => ({ ok: r.ok, data })))
      .then((response) => {
        if (!response?.ok || !response?.data?.success) {
          throw new Error(response?.data?.error || "Level save failed");
        }
        handleSubmitSuccess(response.data, isMidGame);
      })
      .catch((error) => {
        if (!isMidGame) {
          handleSubmitError(error);
        }
      });
  }

  function startNewRun() {
    state.gameOver = false;
    state.maxLevelReached = 1;
    state.timeRemaining = START_TIME_SECONDS;
    state.statusClearAt = 0;
    lastMidGameSubmitTime = 0;
    clearVerifiedRound();

    startLevel(1, false);
    setStatus("Find the exit!", "good");
    updateHud();

    if (isLeaderboardUserLoggedIn) {
      startVerifiedRound().catch(() => {
        // Ignore transient verification start issues during gameplay.
      });
    }
  }

  function clearHeldMoveState() {
    heldMoveKeys.clear();
    heldMovePriority.length = 0;
    heldMoveDownAt.clear();
  }

  function setHeldMoveKey(key, isPressed, now = performance.now()) {
    if (!keyMap.has(key)) {
      return;
    }

    if (isPressed) {
      if (!heldMoveKeys.has(key)) {
        heldMoveKeys.add(key);
        heldMoveDownAt.set(key, now);
        const existingIndex = heldMovePriority.indexOf(key);
        if (existingIndex >= 0) {
          heldMovePriority.splice(existingIndex, 1);
        }
        heldMovePriority.push(key);
      }
      return;
    }

    heldMoveKeys.delete(key);
    heldMoveDownAt.delete(key);
    const index = heldMovePriority.indexOf(key);
    if (index >= 0) {
      heldMovePriority.splice(index, 1);
    }
  }

  function getCurrentHeldMove(now) {
    for (let i = heldMovePriority.length - 1; i >= 0; i -= 1) {
      const key = heldMovePriority[i];
      if (heldMoveKeys.has(key)) {
        const downAt = Number(heldMoveDownAt.get(key) || 0);
        if (now - downAt < HOLD_REPEAT_DELAY_MS) {
          continue;
        }
        return keyMap.get(key) || null;
      }
    }

    return null;
  }

  function clearExpiredChaosBlock(now) {
    if (state.chaosActiveUntil > 0 && now >= state.chaosActiveUntil) {
      ensurePathFromPlayer();
      state.chaosActiveUntil = 0;
    }
  }

  function handleDoorShuffle(now) {
    if (now < state.nextDoorSwapAt) {
      return;
    }

    const pathBeforeShuffle = findPath(state.player, state.exit, false);
    const chaosThisShuffle = rollChaosThisShuffle();
    toggleDoors(chaosThisShuffle, pathBeforeShuffle);
    state.nextDoorSwapAt = now + DOOR_SHUFFLE_SECONDS * 1000;

    if (chaosThisShuffle) {
      const hasRoute = Boolean(findPath(state.player, state.exit, false));
      state.chaosActiveUntil = hasRoute
        ? 0
        : performance.now() + CHAOS_RECOVERY_MS;
      if (!hasRoute) {
        setStatus("Doors shifted! Path blocked briefly...", "warn");
      }
      return;
    }

    state.chaosActiveUntil = 0;
    if (state.statusClearAt <= now) {
      setStatus("Doors reshuffled. Keep moving!", "good");
      state.statusClearAt = now + 900;
    }
  }

  function clearStatusWhenDue(now) {
    if (state.statusClearAt > 0 && now >= state.statusClearAt) {
      setStatus("Find the exit!", "good");
      state.statusClearAt = 0;
    }
  }

  function processActiveRun(now, deltaSeconds) {
    state.timeRemaining = Math.max(0, state.timeRemaining - deltaSeconds);

    if (state.timeRemaining <= 0) {
      endRun("Time up");
      return;
    }

    clearExpiredChaosBlock(now);
    handleDoorShuffle(now);
    clearStatusWhenDue(now);

    const heldMove = getCurrentHeldMove(now);
    if (heldMove) {
      tryMove(heldMove[0], heldMove[1]);
    }
  }

  function update(now = performance.now()) {
    const deltaSeconds = Math.min(
      0.05,
      Math.max(0, (now - state.lastFrameTime) / 1000),
    );
    state.lastFrameTime = now;

    if (!state.gameOver) {
      processActiveRun(now, deltaSeconds);
    }

    animateDoors(deltaSeconds);
    animatePlayer(deltaSeconds);
    updateHud();
    renderBoard();

    requestAnimationFrame(update);
  }

  globalThis.addEventListener(
    "keydown",
    (e) => {
      const target = e.target;
      if (target instanceof HTMLElement) {
        const tag = target.tagName.toLowerCase();
        if (tag === "input" || tag === "textarea" || target.isContentEditable) {
          return;
        }
      }

      const key = String(e.key || "").toLowerCase();
      if (key === "r") {
        e.preventDefault();
        startNewRun();
        return;
      }

      const move = keyMap.get(key);
      if (!move) {
        return;
      }

      e.preventDefault();
      if (e.repeat) {
        setHeldMoveKey(key, true, performance.now());
        return;
      }

      setHeldMoveKey(key, true, performance.now());
      tryMove(move[0], move[1]);
    },
    { passive: false },
  );

  globalThis.addEventListener("keyup", (e) => {
    const key = String(e.key || "").toLowerCase();
    setHeldMoveKey(key, false);
  });

  globalThis.addEventListener("blur", () => {
    clearHeldMoveState();
  });

  restartBtn.addEventListener("click", () => {
    startNewRun();
  });

  resizeCanvas();
  fetchLeaderboard();
  startNewRun();
  requestAnimationFrame(update);
})();
