/*
 * Basketball Launch (Three.js)
 *
 * NOTE:
 * - External file required for strict CSP (no inline <script>).
 * - Scoped to #basketball-launch-stage so it can live inside the site layout.
 */

import * as THREE from "../vendor/three.module.min.js";

(() => {
  const stageEl = document.getElementById("basketball-launch-stage");
  const canvasHost = document.getElementById("basketball-launch-canvas");
  const pageEl = stageEl ? stageEl.closest(".games-page--game") : null;
  const fullscreenButtons = document.querySelectorAll(
    "[data-game-fullscreen-toggle]",
  );

  if (!stageEl || !canvasHost) {
    return;
  }

  const setFullscreenButtonLabels = (isOn) => {
    fullscreenButtons.forEach((btn) => {
      if (!btn) return;
      btn.textContent = isOn ? "Exit Fullscreen" : "Fullscreen";
    });
  };

  const getStageSize = () => {
    const rect = stageEl.getBoundingClientRect();
    const width = Math.max(1, Math.floor(rect.width || 0));
    const height = Math.max(1, Math.floor(rect.height || 0));
    return { width, height };
  };

  const BALL_RADIUS = 0.38;
  const FLOOR_Y = 0.28;
  const BOUNDS = { minX: -5.2, maxX: 5.2, minZ: -22.5, maxZ: 11.2 };
  const HOOP_Z_OFFSET = 1.25;
  const rimCenter = new THREE.Vector3(0, 3.15, -2.6 + HOOP_Z_OFFSET);
  const rimRadius = 0.9;
  const rimTube = 0.08;
  const gravity = 10.8;
  const BOARD_WIDTH = 3.4;
  const BOARD_HEIGHT = 2.2;
  const BOARD_THICKNESS = 0.12;
  const NET_OUTER_RADIUS = 0.72;
  const NET_HEIGHT = 1.05;

  const scoreEl = document.getElementById("score");
  const multiplierEl = document.getElementById("multiplier");
  const timerEl = document.getElementById("timer");
  const statusEl = document.getElementById("status");
  const gameOverEl = document.getElementById("gameOver");
  const maxPointsEl = document.getElementById("maxPoints");
  const bestPointsEl = document.getElementById("bestPoints");
  const restartBtn = document.getElementById("restartBtn");
  const crowdTipsEl = document.getElementById("crowdTips");
  const leaderboardEl = document.getElementById("basketballLeaderboard");
  const leaderboardListEl = document.getElementById(
    "basketballLeaderboardList",
  );
  const leaderboardStatusEl = document.getElementById(
    "basketballLeaderboardStatus",
  );

  const gameSlug = leaderboardEl
    ? String(leaderboardEl.dataset.gameSlug || "")
    : "";
  const isLeaderboardUserLoggedIn = leaderboardEl
    ? leaderboardEl.dataset.userLoggedIn === "1"
    : false;
  const csrfToken =
    document
      .querySelector('meta[name="csrf-token"]')
      ?.getAttribute("content") || "";

  if (
    !scoreEl ||
    !multiplierEl ||
    !timerEl ||
    !statusEl ||
    !gameOverEl ||
    !maxPointsEl ||
    !bestPointsEl ||
    !restartBtn ||
    !crowdTipsEl
  ) {
    return;
  }

  const readFiniteNumberSetting = (value, fallback) => {
    const parsed = Number(value);
    return Number.isFinite(parsed) ? parsed : fallback;
  };
  const clampNumber = (value, min, max) => Math.min(max, Math.max(min, value));

  const MULTIPLIER_STEP = Number(
    clampNumber(
      readFiniteNumberSetting(stageEl.dataset.multiplierStep, 1.12),
      1.01,
      3,
    ).toFixed(4),
  );
  const START_TIME_SECONDS = Math.round(
    clampNumber(
      readFiniteNumberSetting(stageEl.dataset.startTimeSeconds, 30),
      5,
      240,
    ),
  );
  const TIME_BONUS_ON_MAKE = Math.round(
    clampNumber(
      readFiniteNumberSetting(stageEl.dataset.timeBonusOnMake, 5),
      0,
      60,
    ),
  );
  const TIME_PENALTY_ON_MISS = Math.round(
    clampNumber(
      readFiniteNumberSetting(stageEl.dataset.timePenaltyOnMiss, 10),
      0,
      120,
    ),
  );
  const configuredMaxTimeSeconds = Math.round(
    clampNumber(
      readFiniteNumberSetting(stageEl.dataset.maxTimeSeconds, 99),
      10,
      360,
    ),
  );
  const MAX_TIME_SECONDS = Math.max(
    START_TIME_SECONDS,
    configuredMaxTimeSeconds,
  );

  let score = 0;
  let multiplier = 1;
  let timeRemaining = START_TIME_SECONDS;
  let noFailStreak = 0;
  let bestScore = 0;
  let gameOver = false;
  let statusTimeout = null;
  let lastTimerLabel = "";
  let leaderboardFetchSerial = 0;
  let activeVerifiedRound = null;
  let pendingVerifiedRoundPromise = null;
  let currentUserBestLeaderboardScore = 0;

  const scene = new THREE.Scene();
  scene.background = new THREE.Color(0x070913);
  scene.fog = new THREE.Fog(0x070913, 8, 30);

  const initialSize = getStageSize();
  const camera = new THREE.PerspectiveCamera(
    64,
    initialSize.width / initialSize.height,
    0.1,
    120,
  );
  camera.position.set(0, 4, 8.1);
  camera.lookAt(0, 1, -2.1);

  const renderer = new THREE.WebGLRenderer({ antialias: true });
  renderer.setPixelRatio(Math.min(globalThis.devicePixelRatio || 1, 2));
  renderer.setSize(initialSize.width, initialSize.height);
  renderer.shadowMap.enabled = true;
  renderer.shadowMap.type = THREE.PCFSoftShadowMap;

  // IMPORTANT: mount the canvas inside the stage instead of document.body.
  canvasHost.replaceChildren();
  canvasHost.appendChild(renderer.domElement);

  function makeLaneTexture() {
    const canvas = document.createElement("canvas");
    canvas.width = 1024;
    canvas.height = 2048;
    const ctx = canvas.getContext("2d");
    if (!ctx) {
      return null;
    }

    const baseGrad = ctx.createLinearGradient(0, 0, 0, canvas.height);
    baseGrad.addColorStop(0, "#f6dfa6");
    baseGrad.addColorStop(0.52, "#ecc982");
    baseGrad.addColorStop(1, "#d9b065");
    ctx.fillStyle = baseGrad;
    ctx.fillRect(0, 0, canvas.width, canvas.height);

    const plankW = 56;
    for (let x = 0; x <= canvas.width; x += plankW) {
      ctx.fillStyle =
        x % (plankW * 2) === 0
          ? "rgba(115, 70, 18, 0.14)"
          : "rgba(255, 255, 255, 0.06)";
      ctx.fillRect(x, 0, 2, canvas.height);
    }

    for (let y = 80; y < canvas.height; y += 168) {
      ctx.fillStyle = "rgba(90, 58, 20, 0.08)";
      ctx.fillRect(0, y, canvas.width, 2);
    }

    ctx.save();
    ctx.translate(canvas.width * 0.5, canvas.height * 0.64);
    ctx.rotate(-0.03);
    ctx.beginPath();
    ctx.fillStyle = "rgba(26, 57, 126, 0.84)";
    ctx.ellipse(0, 0, 170, 106, 0, 0, Math.PI * 2);
    ctx.fill();
    ctx.strokeStyle = "rgba(255, 133, 28, 0.94)";
    ctx.lineWidth = 16;
    ctx.stroke();
    ctx.fillStyle = "rgba(255, 255, 255, 0.92)";
    ctx.font = "bold 62px Trebuchet MS";
    ctx.textAlign = "center";
    ctx.fillText("ARCADE", 0, -10);
    ctx.font = "bold 86px Trebuchet MS";
    ctx.fillText("HOOPS", 0, 65);
    ctx.restore();

    const tex = new THREE.CanvasTexture(canvas);
    tex.colorSpace = THREE.SRGBColorSpace;
    return tex;
  }

  function makePanelTexture() {
    const canvas = document.createElement("canvas");
    canvas.width = 512;
    canvas.height = 1024;
    const ctx = canvas.getContext("2d");
    if (!ctx) {
      return null;
    }

    const grad = ctx.createLinearGradient(0, 0, 0, canvas.height);
    grad.addColorStop(0, "#3f8dff");
    grad.addColorStop(0.5, "#2f73f1");
    grad.addColorStop(1, "#1d4bbf");
    ctx.fillStyle = grad;
    ctx.fillRect(0, 0, canvas.width, canvas.height);

    for (let y = 0; y < canvas.height; y += 20) {
      ctx.fillStyle =
        y % 40 === 0 ? "rgba(255, 255, 255, 0.08)" : "rgba(8, 20, 46, 0.08)";
      ctx.fillRect(0, y, canvas.width, 2);
    }

    const shine = ctx.createLinearGradient(0, 0, canvas.width, 0);
    shine.addColorStop(0, "rgba(255,255,255,0.04)");
    shine.addColorStop(0.5, "rgba(255,255,255,0.22)");
    shine.addColorStop(1, "rgba(255,255,255,0.04)");
    ctx.fillStyle = shine;
    ctx.fillRect(0, 0, canvas.width, canvas.height);

    const tex = new THREE.CanvasTexture(canvas);
    tex.colorSpace = THREE.SRGBColorSpace;
    tex.wrapS = THREE.RepeatWrapping;
    tex.wrapT = THREE.RepeatWrapping;
    tex.repeat.set(1, 1.4);
    return tex;
  }

  function buildWirePanel(width, height, cols, rows, color, opacity = 0.84) {
    const verts = [];
    const halfW = width * 0.5;
    const halfH = height * 0.5;

    for (let i = 0; i <= cols; i += 1) {
      const x = -halfW + (width * i) / cols;
      verts.push(x, -halfH, 0, x, halfH, 0);
    }

    for (let j = 0; j <= rows; j += 1) {
      const y = -halfH + (height * j) / rows;
      verts.push(-halfW, y, 0, halfW, y, 0);
    }

    const geo = new THREE.BufferGeometry();
    geo.setAttribute("position", new THREE.Float32BufferAttribute(verts, 3));
    return new THREE.LineSegments(
      geo,
      new THREE.LineBasicMaterial({
        color,
        transparent: true,
        opacity,
        depthWrite: false,
        toneMapped: false,
      }),
    );
  }

  const hemi = new THREE.HemisphereLight(0x5b77ba, 0x04060d, 0.38);
  scene.add(hemi);

  const key = new THREE.SpotLight(0xffffff, 1.55, 46, Math.PI / 5, 0.45, 1.2);
  key.position.set(0, 9.4, 8.4);
  key.target.position.set(0, 2, -2.6 + HOOP_Z_OFFSET);
  key.castShadow = true;
  key.shadow.mapSize.set(1024, 1024);
  key.shadow.camera.near = 1;
  key.shadow.camera.far = 54;
  key.shadow.camera.left = -12;
  key.shadow.camera.right = 12;
  key.shadow.camera.top = 12;
  key.shadow.camera.bottom = -12;
  scene.add(key.target);
  scene.add(key);

  const sideGlowL = new THREE.PointLight(0x2f7aff, 0.75, 22);
  sideGlowL.position.set(-3.2, 1.3, 1.8);
  scene.add(sideGlowL);

  const sideGlowR = new THREE.PointLight(0xff5d98, 0.62, 22);
  sideGlowR.position.set(3.2, 1.3, 1.8);
  scene.add(sideGlowR);

  const laneBlueFront = new THREE.PointLight(0x70b6ff, 1.05, 15);
  laneBlueFront.position.set(-2.2, 1.1, 4.4);
  scene.add(laneBlueFront);

  const laneBlueBack = new THREE.PointLight(0x4c84ff, 0.8, 13);
  laneBlueBack.position.set(-2.2, 1.6, -3.5);
  scene.add(laneBlueBack);

  const lanePinkFront = new THREE.PointLight(0xff7cae, 1.05, 15);
  lanePinkFront.position.set(2.2, 1.1, 4.4);
  scene.add(lanePinkFront);

  const lanePinkBack = new THREE.PointLight(0xff4d85, 0.8, 13);
  lanePinkBack.position.set(2.2, 1.6, -3.5);
  scene.add(lanePinkBack);

  const laneTexture = makeLaneTexture();
  if (laneTexture) {
    laneTexture.anisotropy = renderer.capabilities.getMaxAnisotropy();
  }
  const panelTexture = makePanelTexture();

  const cabinetBase = new THREE.Mesh(
    new THREE.BoxGeometry(6.9, 0.62, 17.2),
    new THREE.MeshStandardMaterial({
      color: 0x11192c,
      roughness: 0.52,
      metalness: 0.4,
    }),
  );
  cabinetBase.position.set(0, FLOOR_Y - 0.29, 2.15);
  cabinetBase.receiveShadow = true;
  cabinetBase.castShadow = true;
  scene.add(cabinetBase);

  const lane = new THREE.Mesh(
    new THREE.PlaneGeometry(4.7, 8.6),
    new THREE.MeshStandardMaterial({
      map: laneTexture || null,
      color: laneTexture ? 0xffffff : 0xe2bc75,
      roughness: 0.5,
      metalness: 0.03,
    }),
  );
  lane.rotation.x = -Math.PI / 2;
  lane.position.set(0, FLOOR_Y + 0.024, -0.2);
  lane.receiveShadow = true;
  scene.add(lane);

  const sidePanelMat = new THREE.MeshStandardMaterial({
    map: panelTexture || null,
    color: 0x2f74ef,
    roughness: 0.35,
    metalness: 0.24,
  });

  const sideWallThickness = 0.14;
  const sideWallHeight = 1.86;
  const sideWallDepth = 15.4;
  const sideWallY = 1.21;
  const sideWallZ = 3.28;
  const sideWallX = 2.46;
  const sideWallHalf = new THREE.Vector3(
    sideWallThickness * 0.5 + 0.12,
    sideWallHeight * 0.5 + 0.08,
    sideWallDepth * 0.5,
  );
  const sideWallColliders = [
    {
      center: new THREE.Vector3(-sideWallX, sideWallY, sideWallZ),
      half: sideWallHalf,
      bounce: 0.74,
    },
    {
      center: new THREE.Vector3(sideWallX, sideWallY, sideWallZ),
      half: sideWallHalf,
      bounce: 0.74,
    },
  ];

  const leftSidePanel = new THREE.Mesh(
    new THREE.BoxGeometry(sideWallThickness, sideWallHeight, sideWallDepth),
    sidePanelMat,
  );
  leftSidePanel.position.set(-sideWallX, sideWallY, sideWallZ);
  leftSidePanel.castShadow = true;
  leftSidePanel.receiveShadow = true;
  scene.add(leftSidePanel);

  const rightSidePanel = new THREE.Mesh(
    new THREE.BoxGeometry(sideWallThickness, sideWallHeight, sideWallDepth),
    sidePanelMat,
  );
  rightSidePanel.position.set(sideWallX, sideWallY, sideWallZ);
  rightSidePanel.castShadow = true;
  rightSidePanel.receiveShadow = true;
  scene.add(rightSidePanel);

  const railMat = new THREE.MeshStandardMaterial({
    color: 0x0f1421,
    roughness: 0.34,
    metalness: 0.67,
  });
  const leftRail = new THREE.Mesh(
    new THREE.BoxGeometry(0.12, 0.16, 15.65),
    railMat,
  );
  leftRail.position.set(-2.58, 2.2, 3.28);
  scene.add(leftRail);

  const rightRail = new THREE.Mesh(
    new THREE.BoxGeometry(0.12, 0.16, 15.65),
    railMat,
  );
  rightRail.position.set(2.58, 2.2, 3.28);
  scene.add(rightRail);

  const frontRail = new THREE.Mesh(
    new THREE.BoxGeometry(5.08, 0.14, 0.12),
    railMat,
  );
  frontRail.position.set(0, 1.45, 10.95);
  scene.add(frontRail);

  const backRail = new THREE.Mesh(
    new THREE.BoxGeometry(5.08, 0.14, 0.12),
    railMat,
  );
  backRail.position.set(0, 1.9, -4.27);
  scene.add(backRail);

  const stripPink = new THREE.MeshStandardMaterial({
    color: 0xff75a4,
    emissive: 0xff2c6b,
    emissiveIntensity: 2.2,
    metalness: 0.2,
    roughness: 0.3,
    toneMapped: false,
  });
  const stripBlue = new THREE.MeshStandardMaterial({
    color: 0x8ac5ff,
    emissive: 0x2f78ff,
    emissiveIntensity: 2.15,
    metalness: 0.2,
    roughness: 0.3,
    toneMapped: false,
  });

  const leftStrip = new THREE.Mesh(
    new THREE.BoxGeometry(0.055, 0.06, 14.3),
    stripPink,
  );
  leftStrip.position.set(-2.24, FLOOR_Y + 0.05, 3.1);
  scene.add(leftStrip);

  const rightStrip = new THREE.Mesh(
    new THREE.BoxGeometry(0.055, 0.06, 14.3),
    stripBlue,
  );
  rightStrip.position.set(2.24, FLOOR_Y + 0.05, 3.1);
  scene.add(rightStrip);

  const leftCage = buildWirePanel(15.2, 3.25, 36, 14, 0xa4c9ff, 0.64);
  leftCage.position.set(-2.43, 2.22, 3.4);
  leftCage.rotation.y = Math.PI / 2;
  scene.add(leftCage);

  const rightCage = buildWirePanel(15.2, 3.25, 36, 14, 0xa4c9ff, 0.64);
  rightCage.position.set(2.43, 2.22, 3.4);
  rightCage.rotation.y = Math.PI / 2;
  scene.add(rightCage);

  const backFence = buildWirePanel(4.84, 2.12, 22, 11, 0xa4c9ff, 0.76);
  backFence.position.set(0, 2.47, -4.19);
  scene.add(backFence);

  const cageWallColliders = [
    {
      center: new THREE.Vector3(-2.43, 2.22, 3.4),
      half: new THREE.Vector3(0.08, 1.68, 7.68),
      bounce: 0.68,
    },
    {
      center: new THREE.Vector3(2.43, 2.22, 3.4),
      half: new THREE.Vector3(0.08, 1.68, 7.68),
      bounce: 0.68,
    },
    {
      center: new THREE.Vector3(0, 2.47, -4.19),
      half: new THREE.Vector3(2.45, 1.1, 0.08),
      bounce: 0.64,
    },
  ];

  const crowdDeckMat = new THREE.MeshStandardMaterial({
    color: 0x0d1322,
    roughness: 0.58,
    metalness: 0.28,
  });
  const leftCrowdDeck = new THREE.Mesh(
    new THREE.BoxGeometry(3.2, 0.22, 15.8),
    crowdDeckMat,
  );
  leftCrowdDeck.position.set(-4.35, FLOOR_Y + 0.57, 3.28);
  leftCrowdDeck.receiveShadow = true;
  leftCrowdDeck.castShadow = true;
  scene.add(leftCrowdDeck);

  const rightCrowdDeck = new THREE.Mesh(
    new THREE.BoxGeometry(3.2, 0.22, 15.8),
    crowdDeckMat,
  );
  rightCrowdDeck.position.set(4.35, FLOOR_Y + 0.57, 3.28);
  rightCrowdDeck.receiveShadow = true;
  rightCrowdDeck.castShadow = true;
  scene.add(rightCrowdDeck);

  const backCrowdDeck = new THREE.Mesh(
    new THREE.BoxGeometry(13.2, 0.24, 3.2),
    crowdDeckMat,
  );
  backCrowdDeck.position.set(0, FLOOR_Y + 0.58, -8.3);
  backCrowdDeck.receiveShadow = true;
  backCrowdDeck.castShadow = true;
  scene.add(backCrowdDeck);

  const spectatorShirtPalette = [
    0xff6b7b, 0x4f91ff, 0x7bffbe, 0xffb459, 0xc07bff, 0xfff16b,
  ];
  const spectatorPantsPalette = [0x22324f, 0x2f1f44, 0x1e2e1e, 0x3b2a19];
  const spectatorSkinPalette = [0xf2c39f, 0xd8a17d, 0xb67b57, 0x87553c];
  const crowdLookTarget = new THREE.Vector3(0, 1.8, -0.9);
  const spectators = [];
  const activeSpectators = [];
  const crowdUnlockOrder = [];
  let crowdCelebrate = false;
  let crowdEnergy = 0;
  let crowdCheerPulse = 0;
  let crowdTime = 0;
  let lastPlayerActionAt = 0;
  let lastLaunchedBall = null;
  let crowdActiveCount = 0;
  let crowdPopulationReady = false;
  const activeCrowdTips = [];
  let nextCrowdChatterAt = 0;
  const tmpCrowdTipWorld = new THREE.Vector3();

  const crowdScoreMessages = [
    "Great shot!",
    "Buckets!",
    "Nice arc!",
    "Clean swish!",
    "That was nasty!",
    "Splash city!",
    "Money ball!",
    "Chef mode!",
    "Laser aim!",
    "You cooked that rim!",
    "Silky release!",
    "Textbook jumper!",
    "That shot had manners!",
    "Perfect timing!",
    "Rainmaker!",
    "Pure net poetry!",
    "Straight highlight!",
    "That was disrespectful!",
    "Wrist snap perfection!",
    "Crowd approves that one!",
  ];
  const crowdStreakMessages = [
    "On fire!",
    "No misses!",
    "Keep cooking!",
    "Unstoppable!",
    "Another one!",
    "Streak machine!",
    "Who can stop this?",
    "Infinite buckets!",
    "This is unfair!",
    "Arcade legend pace!",
    "The rim is scared!",
    "Absolute heater!",
    "You broke the difficulty!",
    "Hot hand detected!",
    "The net is tired!",
    "No mercy streak!",
    "Scoreboard is melting!",
    "Main character run!",
    "Historic run alert!",
    "Keep the chaos going!",
  ];
  const crowdMissMessages = [
    "Almost!",
    "Reset and shoot!",
    "Too strong!",
    "Stay focused!",
    "You'll hit next!",
    "That one had no GPS!",
    "Aim software crashed!",
    "Rim dodged that somehow!",
    "Too much power!",
    "Try less panic, more aim!",
    "That ball saw ghosts!",
    "Calm hands, next shot!",
    "Who approved that release?",
    "Ball went sightseeing!",
    "Even the net looked confused!",
    "You can recover this!",
    "One bad shot only!",
    "Reboot your aim!",
    "Rim said not today!",
    "Shake it off, fire again!",
  ];
  const crowdGameOverMessages = [
    "Ahh, tough break!",
    "Good run!",
    "One more game!",
    "You'll beat that score!",
    "Crowd wants a rematch!",
    "That ending was dramatic!",
    "Respect the effort!",
    "Close, but chaos won!",
    "Game over, hype remains!",
    "Run it back immediately!",
    "Score was spicy though!",
    "You almost broke it!",
    "Final miss was cinematic!",
    "That was a wild ride!",
    "We need round two!",
    "Crowd still believes!",
    "You gave us a show!",
    "Next run goes harder!",
    "Not bad, not safe either!",
    "Rematch button is calling!",
  ];
  const crowdAmbientGoodMessages = [
    "Looking good!",
    "Keep the rhythm!",
    "That form is clean!",
    "Crowd is with you!",
    "This pace is dangerous!",
    "Smooth operator!",
    "Rim is in trouble!",
    "Calm and clinical!",
    "You are dialed in!",
    "Keep this tempo!",
    "Shooter energy!",
    "Warming up nicely!",
    "Eyes on target!",
    "This is quality work!",
    "Precision mode on!",
    "Keep stacking swishes!",
    "Confidence looks good!",
    "This run has aura!",
    "Pure focus right now!",
    "The lane is yours!",
  ];
  const crowdAmbientBadMessages = [
    "Need better aim!",
    "Do not rush it!",
    "Set your shot!",
    "Too many misses!",
    "That release was cursed!",
    "Who taught that form?",
    "Less panic, more control!",
    "Shots are getting wild!",
    "Take a breath first!",
    "The rim is not over there!",
    "Focus up, shooter!",
    "Aim check required!",
    "That looked accidental!",
    "Slow down the chaos!",
    "Your arc is on vacation!",
    "Hands are too jumpy!",
    "That miss was loud!",
    "Rushed shot detected!",
    "This is getting messy!",
    "Do not feed the bricks!",
  ];
  const crowdAmbientStateMessages = [
    "Arena is loud!",
    "Pressure is up!",
    "Crowd is watching!",
    "Stay in control!",
    "This machine is hungry!",
    "Vibes: chaotic neutral.",
    "Crowd status: caffeinated.",
    "Drama level increasing.",
    "Arcade gods are judging.",
    "Noise level: ridiculous.",
    "This feels like overtime.",
    "Tension in the air!",
    "Rim chemistry unstable.",
    "The crowd wants fireworks!",
    "Momentum is wobbling.",
    "Lane is glowing right now.",
    "Energy is bouncing!",
    "Pressure cooker mode.",
    "Everybody is locked in.",
    "This run has plot twists!",
  ];
  const crowdIdleWaitMessages = [
    "Yo, what are you waiting for?",
    "Ball is ready, superstar!",
    "Shoot already, coach is sleeping!",
    "Hello? We came for buckets!",
    "This is basketball, not meditation!",
    "Press launch, not pause!",
    "Clock is fake but pressure is real!",
    "Rim is getting cold!",
    "Stop loading and start shooting!",
    "Are we practicing standing still?",
    "Crowd paid for action!",
    "Throw the ball, not our patience!",
    "We need chaos, not silence!",
    "The hoop is wide awake!",
    "Do something legendary!",
    "My popcorn finished, still no shot!",
    "This lane needs violence!",
    "Wake up, sniper!",
    "No shot? No glory!",
    "Take the shot, hero!",
  ];

  function seededUnit(seed) {
    const value = Math.sin(seed * 12.9898 + 78.233) * 43758.5453;
    return value - Math.floor(value);
  }

  function randomRange(min, max) {
    return min + Math.random() * (max - min);
  }

  function pickRandomLine(lines) {
    return lines[Math.floor(Math.random() * lines.length)];
  }

  function createSpectator(xPos, zPos, index) {
    const seed = index * 17.39 + (xPos > 0 ? 97 : 43) + (zPos + 12) * 0.63;
    const heightScale = 0.9 + seededUnit(seed + 1.1) * 0.28;
    const xJitter = (seededUnit(seed + 2.1) - 0.5) * 0.2;
    const zJitter = (seededUnit(seed + 3.1) - 0.5) * 0.14;

    const shirtColor =
      spectatorShirtPalette[
        Math.floor(seededUnit(seed + 4.1) * spectatorShirtPalette.length)
      ];
    const pantsColor =
      spectatorPantsPalette[
        Math.floor(seededUnit(seed + 5.1) * spectatorPantsPalette.length)
      ];
    const skinColor =
      spectatorSkinPalette[
        Math.floor(seededUnit(seed + 6.1) * spectatorSkinPalette.length)
      ];

    const shirtMat = new THREE.MeshStandardMaterial({
      color: shirtColor,
      roughness: 0.58,
      metalness: 0.08,
    });
    const pantsMat = new THREE.MeshStandardMaterial({
      color: pantsColor,
      roughness: 0.68,
      metalness: 0.03,
    });
    const skinMat = new THREE.MeshStandardMaterial({
      color: skinColor,
      roughness: 0.72,
      metalness: 0.02,
    });

    const root = new THREE.Group();
    root.position.set(xPos + xJitter, FLOOR_Y + 0.62, zPos + zJitter);
    root.scale.setScalar(heightScale);
    const lookOffsetYaw = (seededUnit(seed + 7.1) - 0.5) * 0.2;
    const baseYaw =
      Math.atan2(
        crowdLookTarget.x - root.position.x,
        crowdLookTarget.z - root.position.z,
      ) +
      Math.PI +
      lookOffsetYaw;
    root.rotation.y = baseYaw;

    const torso = new THREE.Mesh(
      new THREE.BoxGeometry(0.34, 0.58, 0.24),
      shirtMat,
    );
    torso.position.y = 0.92;
    torso.castShadow = true;
    torso.receiveShadow = true;
    root.add(torso);

    const hips = new THREE.Mesh(
      new THREE.BoxGeometry(0.32, 0.22, 0.22),
      pantsMat,
    );
    hips.position.y = 0.56;
    hips.castShadow = true;
    hips.receiveShadow = true;
    root.add(hips);

    const legGeo = new THREE.BoxGeometry(0.12, 0.44, 0.12);
    const leftLeg = new THREE.Mesh(legGeo, pantsMat);
    leftLeg.position.set(-0.08, 0.29, 0);
    leftLeg.castShadow = true;
    leftLeg.receiveShadow = true;
    root.add(leftLeg);

    const rightLeg = new THREE.Mesh(legGeo, pantsMat);
    rightLeg.position.set(0.08, 0.29, 0);
    rightLeg.castShadow = true;
    rightLeg.receiveShadow = true;
    root.add(rightLeg);

    const head = new THREE.Mesh(
      new THREE.SphereGeometry(0.14, 16, 12),
      skinMat,
    );
    head.position.y = 1.34;
    head.castShadow = true;
    head.receiveShadow = true;
    root.add(head);

    const armGeo = new THREE.BoxGeometry(0.1, 0.42, 0.1);
    const leftArmPivot = new THREE.Group();
    leftArmPivot.position.set(-0.23, 1.1, 0);
    const leftArm = new THREE.Mesh(armGeo, shirtMat);
    leftArm.position.y = -0.21;
    leftArm.castShadow = true;
    leftArm.receiveShadow = true;
    leftArmPivot.add(leftArm);
    root.add(leftArmPivot);

    const rightArmPivot = new THREE.Group();
    rightArmPivot.position.set(0.23, 1.1, 0);
    const rightArm = new THREE.Mesh(armGeo, shirtMat);
    rightArm.position.y = -0.21;
    rightArm.castShadow = true;
    rightArm.receiveShadow = true;
    rightArmPivot.add(rightArm);
    root.add(rightArmPivot);

    scene.add(root);
    spectators.push({
      root,
      leftArmPivot,
      rightArmPivot,
      baseY: root.position.y,
      lookYaw: baseYaw,
      lookOffsetYaw,
      bobPhase: seededUnit(seed + 8.1) * Math.PI * 2,
      clapPhase: seededUnit(seed + 9.1) * Math.PI * 2,
      cheerPhase: seededUnit(seed + 10.1) * Math.PI * 2,
      clapSpeed: 5.7 + seededUnit(seed + 11.1) * 2.6,
      active: true,
      unlockRank: seededUnit(seed + 21.3),
    });
  }

  const crowdSlots = [];
  function addCrowdSideRow(xPos, zSlots) {
    for (const zPos of zSlots) {
      crowdSlots.push({ x: xPos, z: zPos });
    }
  }

  const sideRowNearZ = [8.2, 6.8, 5.4, 4, 2.6, 1.2, -0.2, -1.6, -3, -4.4];
  const sideRowFarZ = [7.9, 6.3, 4.7, 3.1, 1.5, -0.1, -1.7, -3.3];
  const sideCrowdNearX = sideWallX + 1.12;
  const sideCrowdFarX = sideWallX + 2.46;
  addCrowdSideRow(-sideCrowdNearX, sideRowNearZ);
  addCrowdSideRow(sideCrowdNearX, sideRowNearZ);
  addCrowdSideRow(-sideCrowdFarX, sideRowFarZ);
  addCrowdSideRow(sideCrowdFarX, sideRowFarZ);

  const backRowX = [-5.6, -3.8, -2, -0.2, 1.6, 3.4, 5.2];
  for (const xPos of backRowX) {
    crowdSlots.push({ x: xPos, z: -7.7 });
  }
  for (const xPos of backRowX) {
    crowdSlots.push({ x: xPos, z: -9.1 });
  }

  for (let i = 0; i < crowdSlots.length; i += 1) {
    const slot = crowdSlots[i];
    createSpectator(slot.x, slot.z, i + 1);
  }

  function setCrowdPopulation(count) {
    if (crowdUnlockOrder.length === 0) {
      return;
    }

    const nextCount = THREE.MathUtils.clamp(
      Math.floor(count),
      1,
      crowdUnlockOrder.length,
    );
    if (nextCount === crowdActiveCount) {
      return;
    }

    crowdActiveCount = nextCount;
    activeSpectators.length = 0;
    for (const spectator of spectators) {
      spectator.active = false;
      spectator.root.visible = false;
    }
    for (let i = 0; i < crowdActiveCount; i += 1) {
      const spectator = crowdUnlockOrder[i];
      spectator.active = true;
      spectator.root.visible = true;
      activeSpectators.push(spectator);
    }
  }

  function syncCrowdPopulationToScore() {
    if (!crowdPopulationReady) {
      return;
    }
    const scoreBasedCount = 1 + Math.floor(score / 5);
    setCrowdPopulation(scoreBasedCount);
  }

  crowdUnlockOrder.push(...spectators);
  crowdUnlockOrder.sort((a, b) => a.unlockRank - b.unlockRank);
  let starterIndex = 0;
  let starterBest = Infinity;
  for (let i = 0; i < crowdUnlockOrder.length; i += 1) {
    const spectator = crowdUnlockOrder[i];
    const dx = spectator.root.position.x + 3.6;
    const dz = spectator.root.position.z - 2.2;
    const scoreDist = dx * dx + dz * dz;
    if (scoreDist < starterBest) {
      starterBest = scoreDist;
      starterIndex = i;
    }
  }
  if (starterIndex > 0) {
    const starter = crowdUnlockOrder.splice(starterIndex, 1)[0];
    crowdUnlockOrder.unshift(starter);
  }

  crowdPopulationReady = true;
  setCrowdPopulation(1);

  function pickSpectatorForTip(preferBall = true) {
    if (activeSpectators.length === 0) {
      return null;
    }

    if (preferBall && lastLaunchedBall && balls.includes(lastLaunchedBall)) {
      const ballPos = lastLaunchedBall.mesh.position;
      let nearest = activeSpectators[0];
      let nearestDistSq = Infinity;
      for (const spectator of activeSpectators) {
        const dx = spectator.root.position.x - ballPos.x;
        const dz = spectator.root.position.z - ballPos.z;
        const distSq = dx * dx + dz * dz;
        if (distSq < nearestDistSq) {
          nearest = spectator;
          nearestDistSq = distSq;
        }
      }
      return nearest;
    }

    return activeSpectators[
      Math.floor(Math.random() * activeSpectators.length)
    ];
  }

  function pickSpectatorsForBurst(count, preferBall = true) {
    if (activeSpectators.length === 0) {
      return [];
    }

    const targets = [];
    const used = new Set();
    const firstPick = pickSpectatorForTip(preferBall);
    if (firstPick) {
      targets.push(firstPick);
      used.add(firstPick);
    }

    let guard = 0;
    while (
      targets.length < count &&
      used.size < activeSpectators.length &&
      guard < 120
    ) {
      const candidate =
        activeSpectators[Math.floor(Math.random() * activeSpectators.length)];
      if (!used.has(candidate)) {
        targets.push(candidate);
        used.add(candidate);
      }
      guard += 1;
    }

    return targets;
  }

  function clearCrowdTips() {
    for (const tip of activeCrowdTips) {
      if (tip.el) {
        tip.el.remove();
      }
    }
    activeCrowdTips.length = 0;
  }

  function scheduleAmbientCrowdTip(minDelay = 3.8, maxDelay = 7.2) {
    nextCrowdChatterAt = crowdTime + randomRange(minDelay, maxDelay);
  }

  function showCrowdTip(
    text,
    tone = "state",
    spectator = null,
    duration = randomRange(1.5, 2.2),
    preferBall = true,
  ) {
    if (!crowdTipsEl || !text) {
      return;
    }
    const targetSpectator = spectator || pickSpectatorForTip(preferBall);
    if (!targetSpectator) {
      return;
    }

    // Keep one tooltip per spectator to avoid stacked/double text on a single person.
    for (let i = activeCrowdTips.length - 1; i >= 0; i -= 1) {
      const existing = activeCrowdTips[i];
      if (existing && existing.spectator === targetSpectator) {
        existing.el?.remove();
        activeCrowdTips.splice(i, 1);
      }
    }

    if (activeCrowdTips.length >= 16) {
      const oldest = activeCrowdTips.shift();
      oldest?.el?.remove();
    }

    const el = document.createElement("div");
    el.className = `crowd-tip ${tone}`;
    el.textContent = text;
    crowdTipsEl.appendChild(el);

    const bornAt = crowdTime;
    activeCrowdTips.push({
      el,
      spectator: targetSpectator,
      bornAt,
      expiresAt: bornAt + duration,
      headOffset: randomRange(1.92, 2.16),
      driftX: randomRange(-0.12, 0.12),
      phase: randomRange(0, Math.PI * 2),
    });
  }

  function spawnCrowdTipBurst({
    lines,
    tone = "state",
    chance = 1,
    minCount = 1,
    maxCount = 1,
    preferBall = true,
    durationMin = 1.5,
    durationMax = 2.2,
  }) {
    if (!lines || lines.length === 0 || Math.random() > chance) {
      return 0;
    }

    const clampedMax = Math.max(minCount, maxCount);
    const requestedCount = Math.max(
      1,
      Math.floor(randomRange(minCount, clampedMax + 1)),
    );
    const targets = pickSpectatorsForBurst(
      Math.min(requestedCount, activeSpectators.length),
      preferBall,
    );

    for (const target of targets) {
      showCrowdTip(
        pickRandomLine(lines),
        tone,
        target,
        randomRange(durationMin, durationMax),
        preferBall,
      );
    }

    return targets.length;
  }

  function emitScoreCrowdTips(streakMilestone) {
    const spawned = spawnCrowdTipBurst({
      lines: streakMilestone ? crowdStreakMessages : crowdScoreMessages,
      tone: "good",
      chance: streakMilestone ? 0.72 : 0.3,
      minCount: 1,
      maxCount: streakMilestone ? 3 : 2,
      preferBall: true,
      durationMin: streakMilestone ? 1.95 : 1.6,
      durationMax: streakMilestone ? 2.7 : 2.25,
    });

    if (streakMilestone && spawned > 0 && Math.random() < 0.4) {
      spawnCrowdTipBurst({
        lines: crowdAmbientStateMessages,
        tone: "state",
        chance: 0.62,
        minCount: 1,
        maxCount: 2,
        preferBall: false,
        durationMin: 1.8,
        durationMax: 2.4,
      });
    }

    scheduleAmbientCrowdTip(
      streakMilestone ? 3.6 : 4.2,
      streakMilestone ? 6.2 : 7.6,
    );
  }

  function emitMissCrowdTips(finalMiss = false) {
    const tone = finalMiss || Math.random() >= 0.28 ? "bad" : "state";
    const durationMin = finalMiss ? 2 : 1.55;
    const spawned = spawnCrowdTipBurst({
      lines: finalMiss ? crowdGameOverMessages : crowdMissMessages,
      tone,
      chance: finalMiss ? 0.82 : 0.3,
      minCount: 1,
      maxCount: finalMiss ? 3 : 2,
      preferBall: true,
      durationMin,
      durationMax: finalMiss ? 2.9 : 2.35,
    });

    if (finalMiss && spawned > 0 && Math.random() < 0.46) {
      spawnCrowdTipBurst({
        lines: [
          "Run it back!",
          "Rematch now!",
          "We still believe!",
          "One more game!",
        ],
        tone: "state",
        chance: 0.68,
        minCount: 1,
        maxCount: 2,
        preferBall: false,
        durationMin: 2,
        durationMax: 2.8,
      });
    } else if (!finalMiss && spawned === 0 && Math.random() < 0.12) {
      spawnCrowdTipBurst({
        lines: crowdAmbientStateMessages,
        tone: "state",
        chance: 0.58,
        minCount: 1,
        maxCount: 1,
        preferBall: false,
        durationMin: 1.5,
        durationMax: 2.05,
      });
    }

    scheduleAmbientCrowdTip(finalMiss ? 5.5 : 4.6, finalMiss ? 8.8 : 8);
  }

  const balls = [];
  const ballMaterial = new THREE.MeshStandardMaterial({
    color: 0xd9661f,
    roughness: 0.86,
    metalness: 0.02,
  });

  function hasActiveLaunchedBall() {
    for (const ball of balls) {
      if (ball.launched && ball.groundHitAt === null) {
        return true;
      }
    }
    return false;
  }

  function resolveAmbientCrowdMood() {
    if (crowdEnergy > 0.72) {
      return "Crowd mood: hyped!";
    }
    if (crowdEnergy > 0.45) {
      return "Crowd mood: tense.";
    }
    return "Crowd mood: waiting.";
  }

  function resolveAmbientTone(roll, hotChance, pressureChance) {
    if (roll < hotChance) {
      return "good";
    }
    if (roll < hotChance + pressureChance) {
      return "bad";
    }
    return "state";
  }

  function getAmbientLinesForTone(tone, text) {
    if (tone === "good") {
      return crowdAmbientGoodMessages;
    }
    if (tone === "bad") {
      return crowdAmbientBadMessages;
    }
    return [
      ...crowdAmbientStateMessages,
      `Streak: ${noFailStreak}`,
      `Time left: ${Math.max(0, Math.ceil(timeRemaining))}s`,
      `Multiplier x${multiplier.toFixed(2)}`,
      text,
    ];
  }

  function emitAmbientCrowdTip() {
    if (gameOver) {
      return;
    }

    const idleSeconds = crowdTime - lastPlayerActionAt;
    const playerWaiting =
      !grabbedBall && !hasActiveLaunchedBall() && idleSeconds > 7.4;
    if (playerWaiting) {
      const idleHeat = THREE.MathUtils.clamp((idleSeconds - 7.4) / 18, 0, 1);
      spawnCrowdTipBurst({
        lines: crowdIdleWaitMessages,
        tone: idleHeat > 0.45 ? "bad" : "state",
        chance: 0.58,
        minCount: 1,
        maxCount: idleHeat > 0.72 ? 2 : 1,
        preferBall: false,
        durationMin: 1.75,
        durationMax: 2.8,
      });
      scheduleAmbientCrowdTip(4.4, 7.6);
      return;
    }

    const hotChance = Math.min(
      0.68,
      noFailStreak * 0.07 + Math.max(0, multiplier - 1) * 0.09,
    );
    const timePressure = THREE.MathUtils.clamp((12 - timeRemaining) / 12, 0, 1);
    const pressureChance = 0.22 + timePressure * 0.28;
    const roll = Math.random();
    const tone = resolveAmbientTone(roll, hotChance, pressureChance);
    let text;

    if (tone === "good") {
      text = pickRandomLine(crowdAmbientGoodMessages);
    } else if (tone === "bad") {
      text = pickRandomLine(crowdAmbientBadMessages);
    } else {
      const mood = resolveAmbientCrowdMood();
      const stateLinePool = [
        pickRandomLine(crowdAmbientStateMessages),
        `Streak: ${noFailStreak}`,
        `Time left: ${Math.max(0, Math.ceil(timeRemaining))}s`,
        `Multiplier x${multiplier.toFixed(2)}`,
        mood,
      ];
      text = pickRandomLine(stateLinePool);
    }

    const linePool = getAmbientLinesForTone(tone, text);

    spawnCrowdTipBurst({
      lines: linePool,
      tone,
      chance: 0.48,
      minCount: 1,
      maxCount: crowdEnergy > 0.72 ? 2 : 1,
      preferBall: false,
      durationMin: 1.45,
      durationMax: 2.45,
    });

    scheduleAmbientCrowdTip(4.8, 9);
  }

  function updateCrowdTips() {
    const stageRect = stageEl.getBoundingClientRect();
    const stageWidth = Math.max(1, stageRect.width || 1);
    const stageHeight = Math.max(1, stageRect.height || 1);

    for (let i = activeCrowdTips.length - 1; i >= 0; i -= 1) {
      const tip = activeCrowdTips[i];
      if (
        !tip?.el ||
        crowdTime >= tip.expiresAt ||
        !tip.spectator?.active ||
        !tip.spectator?.root?.visible
      ) {
        tip?.el?.remove();
        activeCrowdTips.splice(i, 1);
        continue;
      }

      tmpCrowdTipWorld.set(
        tip.driftX + Math.sin(crowdTime * 1.9 + tip.phase) * 0.05,
        tip.headOffset + Math.sin(crowdTime * 3.15 + tip.phase) * 0.05,
        0,
      );
      tip.spectator.root.localToWorld(tmpCrowdTipWorld);
      tmpCrowdTipWorld.project(camera);

      if (
        tmpCrowdTipWorld.z > 1 ||
        tmpCrowdTipWorld.z < -1 ||
        Math.abs(tmpCrowdTipWorld.x) > 1.3 ||
        Math.abs(tmpCrowdTipWorld.y) > 1.3
      ) {
        tip.el.style.opacity = "0";
        continue;
      }

      // Map clip-space to stage pixel coordinates.
      const x = (tmpCrowdTipWorld.x * 0.5 + 0.5) * stageWidth;
      const y = (-tmpCrowdTipWorld.y * 0.5 + 0.5) * stageHeight;
      tip.el.style.left = `${x}px`;
      tip.el.style.top = `${y}px`;

      const age = crowdTime - tip.bornAt;
      const remaining = tip.expiresAt - crowdTime;
      let alpha = 1;
      if (age < 0.13) {
        alpha = age / 0.13;
      }
      if (remaining < 0.22) {
        alpha = Math.min(alpha, remaining / 0.22);
      }
      tip.el.style.opacity = String(THREE.MathUtils.clamp(alpha, 0, 1));
    }
  }

  // --- remaining game logic (mostly unchanged from the original HTML build) ---

  const boundaryLineMat = new THREE.LineBasicMaterial({
    color: 0x2f65d0,
    transparent: true,
    opacity: 0.65,
  });
  const boundsShape = [
    new THREE.Vector3(BOUNDS.minX, FLOOR_Y + 0.03, BOUNDS.maxZ),
    new THREE.Vector3(BOUNDS.maxX, FLOOR_Y + 0.03, BOUNDS.maxZ),
    new THREE.Vector3(BOUNDS.maxX, FLOOR_Y + 0.03, BOUNDS.minZ),
    new THREE.Vector3(BOUNDS.minX, FLOOR_Y + 0.03, BOUNDS.minZ),
    new THREE.Vector3(BOUNDS.minX, FLOOR_Y + 0.03, BOUNDS.maxZ),
  ];
  const boundsLine = new THREE.Line(
    new THREE.BufferGeometry().setFromPoints(boundsShape),
    boundaryLineMat,
  );
  scene.add(boundsLine);

  const backWall = new THREE.Mesh(
    new THREE.BoxGeometry(5.45, 4.7, 0.2),
    new THREE.MeshStandardMaterial({
      color: 0x0f172d,
      roughness: 0.42,
      metalness: 0.37,
      emissive: 0x142659,
      emissiveIntensity: 0.4,
    }),
  );
  backWall.position.set(0, 2.75, -4.45 + HOOP_Z_OFFSET);
  backWall.castShadow = true;
  backWall.receiveShadow = true;
  scene.add(backWall);

  const boardFrame = new THREE.Mesh(
    new THREE.BoxGeometry(4.15, 2.85, 0.16),
    new THREE.MeshStandardMaterial({
      color: 0x285fd3,
      roughness: 0.33,
      metalness: 0.52,
      emissive: 0x1444c7,
      emissiveIntensity: 0.9,
    }),
  );
  boardFrame.position.set(0, 4.2, -3.72 + HOOP_Z_OFFSET);
  boardFrame.castShadow = true;
  scene.add(boardFrame);

  const board = new THREE.Mesh(
    new THREE.BoxGeometry(BOARD_WIDTH, BOARD_HEIGHT, BOARD_THICKNESS),
    new THREE.MeshStandardMaterial({
      color: 0x1d2436,
      roughness: 0.34,
      metalness: 0.22,
      emissive: 0x203874,
      emissiveIntensity: 0.48,
    }),
  );
  board.position.set(0, 4.2, -3.65 + HOOP_Z_OFFSET);
  board.castShadow = true;
  board.receiveShadow = true;
  scene.add(board);
  const boardHalfExtents = new THREE.Vector3(
    BOARD_WIDTH * 0.5,
    BOARD_HEIGHT * 0.5,
    BOARD_THICKNESS * 0.5,
  );

  const boardSquare = new THREE.LineSegments(
    new THREE.EdgesGeometry(new THREE.PlaneGeometry(1.1, 0.78)),
    new THREE.LineBasicMaterial({ color: 0xff8fbd, toneMapped: false }),
  );
  boardSquare.position.set(0, 3.78, -3.58 + HOOP_Z_OFFSET);
  scene.add(boardSquare);

  const neonPinkMat = new THREE.MeshStandardMaterial({
    color: 0xff86b3,
    emissive: 0xff2e72,
    emissiveIntensity: 2.9,
    metalness: 0.22,
    roughness: 0.28,
    toneMapped: false,
  });
  const neonWhiteMat = new THREE.MeshStandardMaterial({
    color: 0xfff4f7,
    emissive: 0xffd7e2,
    emissiveIntensity: 2.4,
    metalness: 0.18,
    roughness: 0.25,
    toneMapped: false,
  });

  const outerArch = new THREE.Mesh(
    new THREE.TorusGeometry(1.74, 0.05, 12, 72, Math.PI),
    neonPinkMat,
  );
  outerArch.position.set(0, 4.47, -3.54 + HOOP_Z_OFFSET);
  outerArch.rotation.z = Math.PI;
  scene.add(outerArch);

  const outerArchLeft = new THREE.Mesh(
    new THREE.CylinderGeometry(0.05, 0.05, 1.78, 14),
    neonPinkMat,
  );
  outerArchLeft.position.set(-1.74, 3.58, -3.54 + HOOP_Z_OFFSET);
  scene.add(outerArchLeft);

  const outerArchRight = new THREE.Mesh(
    new THREE.CylinderGeometry(0.05, 0.05, 1.78, 14),
    neonPinkMat,
  );
  outerArchRight.position.set(1.74, 3.58, -3.54 + HOOP_Z_OFFSET);
  scene.add(outerArchRight);

  const innerArch = new THREE.Mesh(
    new THREE.TorusGeometry(1.25, 0.035, 12, 72, Math.PI),
    neonWhiteMat,
  );
  innerArch.position.set(0, 4.17, -3.53 + HOOP_Z_OFFSET);
  innerArch.rotation.z = Math.PI;
  scene.add(innerArch);

  const innerArchLeft = new THREE.Mesh(
    new THREE.CylinderGeometry(0.035, 0.035, 1.28, 14),
    neonWhiteMat,
  );
  innerArchLeft.position.set(-1.25, 3.53, -3.53 + HOOP_Z_OFFSET);
  scene.add(innerArchLeft);

  const innerArchRight = new THREE.Mesh(
    new THREE.CylinderGeometry(0.035, 0.035, 1.28, 14),
    neonWhiteMat,
  );
  innerArchRight.position.set(1.25, 3.53, -3.53 + HOOP_Z_OFFSET);
  scene.add(innerArchRight);

  const rim = new THREE.Mesh(
    new THREE.TorusGeometry(rimRadius, rimTube, 14, 44),
    new THREE.MeshStandardMaterial({
      color: 0xe7671e,
      roughness: 0.28,
      metalness: 0.25,
    }),
  );
  rim.rotation.x = Math.PI / 2;
  rim.position.copy(rimCenter);
  rim.castShadow = true;
  scene.add(rim);

  const rimGlow = new THREE.PointLight(0xff8f56, 1.22, 9);
  rimGlow.position.set(rimCenter.x, rimCenter.y + 0.1, rimCenter.z + 0.3);
  scene.add(rimGlow);

  const net = new THREE.Mesh(
    new THREE.CylinderGeometry(NET_OUTER_RADIUS, 0.38, NET_HEIGHT, 20, 1, true),
    new THREE.MeshStandardMaterial({
      color: 0xf6f7fb,
      roughness: 0.9,
      metalness: 0,
      transparent: true,
      opacity: 0.68,
      side: THREE.DoubleSide,
    }),
  );
  const netRestPos = new THREE.Vector3(
    rimCenter.x,
    rimCenter.y - NET_HEIGHT * 0.52,
    rimCenter.z,
  );
  const netOffset = new THREE.Vector3();
  const netVelocity = new THREE.Vector3();
  const netTilt = new THREE.Vector2();
  const netTiltVel = new THREE.Vector2();
  let netDrop = 0;
  let netDropVel = 0;
  let netPulse = 0;
  let netPulseVel = 0;
  net.position.copy(netRestPos);
  scene.add(net);

  function createBall(x, y, z) {
    const mesh = new THREE.Mesh(
      new THREE.SphereGeometry(BALL_RADIUS, 28, 20),
      ballMaterial.clone(),
    );
    mesh.position.set(x, y, z);
    mesh.castShadow = true;
    mesh.receiveShadow = true;
    scene.add(mesh);

    balls.push({
      mesh,
      velocity: new THREE.Vector3(),
      start: new THREE.Vector3(x, y, z),
      lastY: y,
      attempted: false,
      launched: false,
      canGrab: true,
      grabbed: false,
      netCooldown: 0,
      groundHitAt: null,
      spawnedOnLaunch: false,
    });
  }

  const rackY = FLOOR_Y + BALL_RADIUS + 0.16;
  const rackSlots = [new THREE.Vector3(0, rackY, 4.65)];
  let nextRackSlot = 0;
  let spawnTimer = null;

  function spawnNextBall(delayMs = 0) {
    if (gameOver) {
      return;
    }

    const spawn = () => {
      if (gameOver) {
        return;
      }
      const slot = rackSlots[nextRackSlot];
      nextRackSlot = (nextRackSlot + 1) % rackSlots.length;
      createBall(slot.x, slot.y, slot.z);
    };

    if (delayMs <= 0) {
      spawn();
      return;
    }

    if (spawnTimer) {
      clearTimeout(spawnTimer);
    }
    spawnTimer = setTimeout(() => {
      spawnTimer = null;
      spawn();
    }, delayMs);
  }

  const raycaster = new THREE.Raycaster();
  const pointer = new THREE.Vector2(0, -0.12);
  const pointerVelocity = new THREE.Vector2();
  let grabbedBall = null;
  let grabDistance = 5.4;
  const dragVelocity = new THREE.Vector3();
  const prevGrabPos = new THREE.Vector3();
  let prevGrabTime = performance.now();
  let prevPointerTime = performance.now();

  const tmpDir = new THREE.Vector3();
  const tmpHoriz = new THREE.Vector3();
  const tmpTarget = new THREE.Vector3();
  const tmpClosest = new THREE.Vector3();
  const tmpPush = new THREE.Vector3();
  const tmpRimNearest = new THREE.Vector3();
  const tmpRimDiff = new THREE.Vector3();

  function formatPoints(value) {
    return (Math.round(value * 100) / 100).toLocaleString(undefined, {
      minimumFractionDigits: 0,
      maximumFractionDigits: 2,
    });
  }

  function escapeHtml(value) {
    return String(value)
      .replaceAll("&", "&amp;")
      .replaceAll("<", "&lt;")
      .replaceAll(">", "&gt;")
      .replaceAll('"', "&quot;")
      .replaceAll("'", "&#39;");
  }

  function setLeaderboardStatus(text, isError = false) {
    if (!leaderboardStatusEl) {
      return;
    }
    leaderboardStatusEl.textContent = text;
    leaderboardStatusEl.classList.toggle("is-error", !!isError);
  }

  function renderLeaderboard(items) {
    if (!leaderboardListEl) {
      return;
    }

    currentUserBestLeaderboardScore = 0;

    if (!Array.isArray(items) || items.length === 0) {
      leaderboardListEl.innerHTML =
        '<li class="basketball-leaderboard-empty">No scores yet. Be the first.</li>';
      return;
    }

    leaderboardListEl.innerHTML = items
      .map((item) => {
        const rank = Number.isFinite(Number(item?.rank))
          ? Number(item.rank)
          : 0;
        const safeRank = Math.max(0, Math.floor(rank));
        const name = escapeHtml(item?.name || "Player");
        const rawProfileUrl =
          typeof item?.profile_url === "string" ? item.profile_url : "";
        const hasProfileLink = rawProfileUrl.startsWith("/@");
        const profileUrl = hasProfileLink ? escapeHtml(rawProfileUrl) : "";
        const scoreNum = Number(item?.score);
        const safeScore = Number.isFinite(scoreNum) ? scoreNum : 0;
        const score = escapeHtml(formatPoints(safeScore));
        const currentClass = item?.is_current_user ? " is-current" : "";

        if (
          item?.is_current_user &&
          safeScore > currentUserBestLeaderboardScore
        ) {
          currentUserBestLeaderboardScore = safeScore;
        }

        const nameMarkup = hasProfileLink
          ? `<a class="basketball-leaderboard-name" href="${profileUrl}">${name}</a>`
          : `<span class="basketball-leaderboard-name">${name}</span>`;
        return `
        <li class="basketball-leaderboard-item${currentClass}">
          <span class="basketball-leaderboard-rank">#${safeRank}</span>
          ${nameMarkup}
          <span class="basketball-leaderboard-score">${score}</span>
        </li>
      `;
      })
      .join("");
  }

  function clearVerifiedRound() {
    activeVerifiedRound = null;
  }

  function startVerifiedRound() {
    if (!gameSlug || !isLeaderboardUserLoggedIn) {
      clearVerifiedRound();
      return Promise.resolve(null);
    }

    if (pendingVerifiedRoundPromise) {
      return pendingVerifiedRoundPromise;
    }

    pendingVerifiedRoundPromise = fetch(
      `/game-leaderboard-api?game=${encodeURIComponent(gameSlug)}&action=start-round`,
      {
        credentials: "same-origin",
      },
    )
      .then((r) => {
        if (!r.ok) {
          throw new Error("Could not start verified round");
        }
        return r.json();
      })
      .then((data) => {
        if (
          !data?.success ||
          !data?.round?.round_id ||
          !data?.round?.round_token
        ) {
          throw new Error(data?.error || "Could not start verified round");
        }

        activeVerifiedRound = {
          id: String(data.round.round_id),
          token: String(data.round.round_token),
          issuedAt: Number(data.round.issued_at || 0),
          expiresAt: Number(data.round.expires_at || 0),
          minSubmitSeconds: Number(data.round.min_submit_seconds || 0),
        };
        return activeVerifiedRound;
      })
      .catch((error) => {
        clearVerifiedRound();
        throw error;
      })
      .finally(() => {
        pendingVerifiedRoundPromise = null;
      });

    return pendingVerifiedRoundPromise;
  }

  function fetchLeaderboard() {
    if (!gameSlug || !leaderboardListEl) {
      return;
    }

    const serial = ++leaderboardFetchSerial;
    fetch(`/game-leaderboard-api?game=${encodeURIComponent(gameSlug)}&limit=50`)
      .then((r) => {
        if (!r.ok) {
          throw new Error("Leaderboard unavailable");
        }
        return r.json();
      })
      .then((data) => {
        if (serial !== leaderboardFetchSerial) {
          return;
        }
        if (!data?.success) {
          throw new Error("Leaderboard unavailable");
        }
        renderLeaderboard(data.items || []);
        setLeaderboardStatus("Basketball Launch Leaderboard");
      })
      .catch(() => {
        if (serial !== leaderboardFetchSerial) {
          return;
        }
        setLeaderboardStatus("Leaderboard unavailable", true);
      });
  }

  function submitLeaderboardScore(points, waitedForPendingRound = false) {
    if (!gameSlug || points <= 0) {
      return;
    }

    if (!isLeaderboardUserLoggedIn || !csrfToken) {
      fetchLeaderboard();
      return;
    }

    if (points <= currentUserBestLeaderboardScore) {
      setLeaderboardStatus("Score did not beat your best");
      fetchLeaderboard();
      return;
    }

    if (!activeVerifiedRound?.id || !activeVerifiedRound?.token) {
      if (pendingVerifiedRoundPromise && !waitedForPendingRound) {
        pendingVerifiedRoundPromise
          .then(() => {
            submitLeaderboardScore(points, true);
          })
          .catch(() => {
            setLeaderboardStatus("Score verification unavailable", true);
            fetchLeaderboard();
          });
        return;
      }

      setLeaderboardStatus("Round expired. Start a new game.", true);
      fetchLeaderboard();
      return;
    }

    const minSubmitSeconds = Number(activeVerifiedRound.minSubmitSeconds || 0);
    const issuedAt = Number(activeVerifiedRound.issuedAt || 0);
    if (minSubmitSeconds > 0 && issuedAt > 0) {
      const elapsedSeconds = Date.now() / 1000 - issuedAt;
      if (elapsedSeconds < minSubmitSeconds) {
        setLeaderboardStatus("Score not submitted (round too short yet)");
        fetchLeaderboard();
        return;
      }
    }

    setLeaderboardStatus("Saving score...");
    fetch("/game-leaderboard-api", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "X-CSRF-Token": csrfToken,
      },
      credentials: "same-origin",
      body: JSON.stringify({
        game: gameSlug,
        score: points,
        round_id: activeVerifiedRound.id,
        round_token: activeVerifiedRound.token,
      }),
    })
      .then((r) => {
        return r.json().then((data) => ({ ok: r.ok, data }));
      })
      .then((response) => {
        const data = response?.data;
        if (!response?.ok || !data?.success) {
          throw new Error(data?.error || "Score save failed");
        }
        clearVerifiedRound();
        renderLeaderboard(data.items || []);
        if (data?.submission_skipped) {
          setLeaderboardStatus(data?.message || "Score did not beat your best");
        } else {
          setLeaderboardStatus("Score submitted");
        }
      })
      .catch((error) => {
        const message = String(error?.message || "Could not submit score");
        if (message.toLowerCase().includes("too quickly")) {
          setLeaderboardStatus("Score not submitted (round too short yet)");
        } else {
          setLeaderboardStatus(message, true);
        }
      });
  }

  function formatSecondsLeft(seconds) {
    // Display as an integer countdown (ceil keeps 1..N visible until it truly hits 0).
    const safe = Number.isFinite(seconds) ? seconds : 0;
    return String(Math.max(0, Math.ceil(safe)));
  }

  function updateScore() {
    scoreEl.textContent = formatPoints(score);
    multiplierEl.textContent = `x${multiplier.toFixed(2)}`;
    lastTimerLabel = formatSecondsLeft(timeRemaining);
    timerEl.textContent = lastTimerLabel;
    syncCrowdPopulationToScore();
  }

  function flashStatus(text) {
    statusEl.textContent = text;
    if (statusTimeout) {
      clearTimeout(statusTimeout);
    }
    statusTimeout = setTimeout(() => {
      statusEl.textContent = "";
    }, 800);
  }

  function updatePointer(event) {
    const rect = renderer.domElement.getBoundingClientRect();
    const nextX = ((event.clientX - rect.left) / rect.width) * 2 - 1;
    const nextY = -((event.clientY - rect.top) / rect.height) * 2 + 1;
    const now = performance.now();
    const dt = Math.max(0.001, (now - prevPointerTime) / 1000);
    pointerVelocity.set((nextX - pointer.x) / dt, (nextY - pointer.y) / dt);
    pointer.x = nextX;
    pointer.y = nextY;
    prevPointerTime = now;
  }

  function findBallByMesh(mesh) {
    for (const ball of balls) {
      if (ball.mesh === mesh) {
        return ball;
      }
    }
    return null;
  }

  function grabBall(event) {
    if (event.button !== 0) {
      return;
    }
    if (gameOver) {
      return;
    }

    updatePointer(event);
    raycaster.setFromCamera(pointer, camera);
    const grabbableMeshes = balls
      .filter((ball) => ball.canGrab)
      .map((ball) => ball.mesh);
    if (grabbableMeshes.length === 0) {
      return;
    }
    const hits = raycaster.intersectObjects(grabbableMeshes, false);

    if (hits.length === 0) {
      return;
    }

    const selected = findBallByMesh(hits[0].object);
    if (!selected) {
      return;
    }
    if (!selected.canGrab) {
      return;
    }

    grabbedBall = selected;
    grabbedBall.grabbed = true;
    grabbedBall.velocity.set(0, 0, 0);
    grabbedBall.attempted = false;
    grabbedBall.netCooldown = 0;
    dragVelocity.set(0, 0, 0);
    grabDistance = THREE.MathUtils.clamp(hits[0].distance, 2.6, 13.5);
    prevGrabPos.copy(grabbedBall.mesh.position);
    prevGrabTime = performance.now();
    prevPointerTime = prevGrabTime;
    pointerVelocity.set(0, 0);
    lastPlayerActionAt = crowdTime;
    renderer.domElement.style.cursor = "grabbing";
  }

  function releaseBall() {
    if (!grabbedBall) {
      return;
    }
    if (gameOver) {
      grabbedBall = null;
      renderer.domElement.style.cursor = "grab";
      return;
    }

    const releasedBall = grabbedBall;
    releasedBall.grabbed = false;
    releasedBall.attempted = false;
    releasedBall.groundHitAt = null;
    lastPlayerActionAt = crowdTime;
    const dragSpeed = dragVelocity.length();
    const flickSpeed = pointerVelocity.length();

    // Dead-zone: click/release without a throw gesture should not auto-launch.
    if (dragSpeed < 1.15 && flickSpeed < 1.35) {
      releasedBall.canGrab = true;
      releasedBall.launched = false;
      releasedBall.velocity.set(0, 0, 0);
      grabbedBall = null;
      renderer.domElement.style.cursor = "grab";
      return;
    }

    releasedBall.canGrab = false;
    releasedBall.launched = true;
    lastLaunchedBall = releasedBall;

    if (
      isLeaderboardUserLoggedIn &&
      !activeVerifiedRound &&
      !pendingVerifiedRoundPromise
    ) {
      startVerifiedRound().catch(() => {
        // Keep gameplay smooth; status is shown when a submit is attempted.
      });
    }

    if (!releasedBall.spawnedOnLaunch) {
      releasedBall.spawnedOnLaunch = true;
      spawnNextBall(120);
    }

    // Direction-preserving throw: no forced forward boost.
    const launch = releasedBall.velocity
      .copy(dragVelocity)
      .multiplyScalar(1.06);
    launch.y = THREE.MathUtils.clamp(launch.y, -5.2, 9.8);

    grabbedBall = null;
    renderer.domElement.style.cursor = "grab";
  }

  renderer.domElement.addEventListener("pointerdown", grabBall);
  renderer.domElement.addEventListener("pointermove", updatePointer);
  renderer.domElement.addEventListener("pointerup", (event) => {
    if (event.button === 0) {
      releaseBall();
    }
  });
  renderer.domElement.addEventListener("pointercancel", releaseBall);
  renderer.domElement.addEventListener("pointerleave", releaseBall);

  globalThis.addEventListener("keydown", (event) => {
    if (event.key === "r" || event.key === "R") {
      startNewGame();
    }
  });

  function startNewGame() {
    clearVerifiedRound();

    if (spawnTimer) {
      clearTimeout(spawnTimer);
      spawnTimer = null;
    }

    for (const ball of balls) {
      scene.remove(ball.mesh);
    }
    balls.length = 0;
    nextRackSlot = 0;

    grabbedBall = null;
    dragVelocity.set(0, 0, 0);
    netOffset.set(0, 0, 0);
    netVelocity.set(0, 0, 0);
    netTilt.set(0, 0);
    netTiltVel.set(0, 0);
    netDrop = 0;
    netDropVel = 0;
    netPulse = 0;
    netPulseVel = 0;
    net.position.copy(netRestPos);
    net.scale.set(1, 1, 1);
    net.rotation.set(0, 0, 0);
    score = 0;
    multiplier = 1;
    timeRemaining = START_TIME_SECONDS;
    noFailStreak = 0;
    gameOver = false;
    crowdCelebrate = false;
    crowdCheerPulse = 0;
    lastPlayerActionAt = crowdTime;
    lastLaunchedBall = null;
    clearCrowdTips();
    scheduleAmbientCrowdTip(4.2, 7.2);
    gameOverEl.classList.remove("show");
    updateScore();
    spawnNextBall();
    renderer.domElement.style.cursor = "grab";
    flashStatus("New game");
  }

  function endGame(reason = "Time up") {
    if (gameOver) {
      return;
    }

    gameOver = true;
    if (spawnTimer) {
      clearTimeout(spawnTimer);
      spawnTimer = null;
    }
    bestScore = Math.max(bestScore, score);
    maxPointsEl.textContent = formatPoints(score);
    bestPointsEl.textContent = formatPoints(bestScore);
    gameOverEl.classList.add("show");
    crowdCelebrate = true;
    renderer.domElement.style.cursor = "default";
    flashStatus(`${reason} - game over`);

    if (score > 0) {
      submitLeaderboardScore(score);
    } else {
      fetchLeaderboard();
    }
  }

  function clampBallPosition(pos) {
    pos.x = THREE.MathUtils.clamp(
      pos.x,
      BOUNDS.minX + BALL_RADIUS,
      BOUNDS.maxX - BALL_RADIUS,
    );
    pos.y = THREE.MathUtils.clamp(pos.y, FLOOR_Y + BALL_RADIUS, 7.8);
    pos.z = THREE.MathUtils.clamp(
      pos.z,
      BOUNDS.minZ + BALL_RADIUS,
      BOUNDS.maxZ - BALL_RADIUS,
    );
  }

  function handleRimCollision(ball) {
    const p = ball.mesh.position;
    const v = ball.velocity;
    tmpHoriz.set(p.x - rimCenter.x, 0, p.z - rimCenter.z);
    const radialXZ = tmpHoriz.length();
    if (radialXZ < 0.0001) {
      return;
    }

    tmpHoriz.divideScalar(radialXZ);
    tmpRimNearest.set(
      rimCenter.x + tmpHoriz.x * rimRadius,
      rimCenter.y,
      rimCenter.z + tmpHoriz.z * rimRadius,
    );

    tmpRimDiff.copy(p).sub(tmpRimNearest);
    const dist = tmpRimDiff.length();
    const contactDist = BALL_RADIUS + rimTube;
    if (dist >= contactDist) {
      return;
    }

    if (dist > 0.0001) {
      tmpRimDiff.divideScalar(dist);
    } else {
      tmpRimDiff.set(0, 1, 0);
    }

    const overlap = contactDist - dist;
    const correction = Math.min(overlap * 0.8, 0.09);
    p.addScaledVector(tmpRimDiff, correction);

    const normalSpeed = v.dot(tmpRimDiff);
    if (normalSpeed < 0) {
      const restitution = 0.62;
      v.addScaledVector(tmpRimDiff, -(1 + restitution) * normalSpeed);
      const tangentDrag = 0.988;
      v.multiplyScalar(tangentDrag);
    }
  }

  function resolveEmbeddedAabbPushDirection(point, center, halfExtents, out) {
    const relX = point.x - center.x;
    const relY = point.y - center.y;
    const relZ = point.z - center.z;
    const penX = halfExtents.x - Math.abs(relX);
    const penY = halfExtents.y - Math.abs(relY);
    const penZ = halfExtents.z - Math.abs(relZ);

    if (penX <= penY && penX <= penZ) {
      out.set(relX >= 0 ? 1 : -1, 0, 0);
      return;
    }

    if (penY <= penZ) {
      out.set(0, relY >= 0 ? 1 : -1, 0);
      return;
    }

    out.set(0, 0, relZ >= 0 ? 1 : -1);
  }

  function collideSphereWithAabb(
    ball,
    center,
    halfExtents,
    restitution = 0.72,
    damping = 0.985,
  ) {
    const p = ball.mesh.position;
    const v = ball.velocity;

    tmpClosest.set(
      THREE.MathUtils.clamp(
        p.x,
        center.x - halfExtents.x,
        center.x + halfExtents.x,
      ),
      THREE.MathUtils.clamp(
        p.y,
        center.y - halfExtents.y,
        center.y + halfExtents.y,
      ),
      THREE.MathUtils.clamp(
        p.z,
        center.z - halfExtents.z,
        center.z + halfExtents.z,
      ),
    );

    tmpPush.copy(p).sub(tmpClosest);
    const distSq = tmpPush.lengthSq();
    if (distSq >= BALL_RADIUS * BALL_RADIUS) {
      return false;
    }

    let dist = Math.sqrt(distSq);
    if (dist >= 0.0001) {
      tmpPush.divideScalar(dist);
    } else {
      resolveEmbeddedAabbPushDirection(p, center, halfExtents, tmpPush);
      dist = 0;
    }

    p.addScaledVector(tmpPush, BALL_RADIUS - dist + 0.001);
    const into = v.dot(tmpPush);
    if (into < 0) {
      v.addScaledVector(tmpPush, -(1 + restitution) * into);
    }
    v.multiplyScalar(damping);
    return true;
  }

  function handleBackboardCollision(ball) {
    collideSphereWithAabb(ball, board.position, boardHalfExtents, 0.7, 0.985);
  }

  function handleSideWallCollisions(ball) {
    for (const collider of sideWallColliders) {
      collideSphereWithAabb(
        ball,
        collider.center,
        collider.half,
        collider.bounce,
        0.992,
      );
    }
  }

  function handleCageWallCollisions(ball) {
    for (const collider of cageWallColliders) {
      collideSphereWithAabb(
        ball,
        collider.center,
        collider.half,
        collider.bounce,
        0.994,
      );
    }
  }

  function handleNetInteraction(ball) {
    if (ball.netCooldown > 0) {
      return;
    }

    const p = ball.mesh.position;
    const netTop = netRestPos.y + NET_HEIGHT * 0.52;
    const netBottom = netRestPos.y - NET_HEIGHT * 0.52;
    if (p.y > netTop + BALL_RADIUS || p.y < netBottom - BALL_RADIUS) {
      return;
    }

    tmpHoriz.set(p.x - netRestPos.x, 0, p.z - netRestPos.z);
    const radial = tmpHoriz.length();
    if (radial > NET_OUTER_RADIUS + BALL_RADIUS * 0.7) {
      return;
    }

    const impactSpeed = ball.velocity.length();
    if (impactSpeed < 0.35) {
      return;
    }

    if (radial > 0.0001) {
      tmpHoriz.divideScalar(radial);
    } else {
      tmpHoriz.set(0, 0, 1);
    }

    const impulse = THREE.MathUtils.clamp(impactSpeed * 0.038, 0.03, 0.34);
    netVelocity.x += tmpHoriz.x * impulse + ball.velocity.x * 0.012;
    netVelocity.z += tmpHoriz.z * impulse + ball.velocity.z * 0.012;
    netTiltVel.x += ball.velocity.z * 0.0055;
    netTiltVel.y += -ball.velocity.x * 0.0055;

    const verticalPull = THREE.MathUtils.clamp(
      Math.max(0, -ball.velocity.y) * 0.013 + impactSpeed * 0.004,
      0.008,
      0.085,
    );
    netDropVel -= verticalPull;
    netPulseVel += THREE.MathUtils.clamp(impactSpeed * 0.022, 0.008, 0.12);

    // Let the ball keep momentum; net contact should not force heavy slow-motion.
    ball.velocity.x *= 0.996;
    ball.velocity.y *= 0.998;
    ball.velocity.z *= 0.996;
    ball.netCooldown = 0.03;
  }

  function animateNet(dt) {
    const spring = 22;
    const damping = 7.2;
    netVelocity.x += (-netOffset.x * spring - netVelocity.x * damping) * dt;
    netVelocity.z += (-netOffset.z * spring - netVelocity.z * damping) * dt;
    netOffset.x += netVelocity.x * dt;
    netOffset.z += netVelocity.z * dt;

    const tiltSpring = 18;
    const tiltDamping = 6.4;
    netTiltVel.x += (-netTilt.x * tiltSpring - netTiltVel.x * tiltDamping) * dt;
    netTiltVel.y += (-netTilt.y * tiltSpring - netTiltVel.y * tiltDamping) * dt;
    netTilt.x += netTiltVel.x * dt;
    netTilt.y += netTiltVel.y * dt;

    const dropSpring = 42;
    const dropDamping = 11;
    netDropVel += (-netDrop * dropSpring - netDropVel * dropDamping) * dt;
    netDrop += netDropVel * dt;

    const pulseSpring = 34;
    const pulseDamping = 10;
    netPulseVel += (-netPulse * pulseSpring - netPulseVel * pulseDamping) * dt;
    netPulse += netPulseVel * dt;

    const spread = 1 + netPulse * 0.24;
    const squash = Math.max(0.82, 1 - netPulse * 0.14);
    net.position.set(
      netRestPos.x + netOffset.x,
      netRestPos.y + netDrop,
      netRestPos.z + netOffset.z,
    );
    net.scale.set(spread, squash, spread);
    net.rotation.set(
      netTilt.x + netOffset.z * 0.08,
      0,
      netTilt.y - netOffset.x * 0.08,
    );
  }

  function resolveCrowdFocusTarget() {
    if (
      lastLaunchedBall &&
      balls.includes(lastLaunchedBall) &&
      lastLaunchedBall.launched &&
      lastLaunchedBall.groundHitAt === null
    ) {
      return lastLaunchedBall.mesh.position;
    }

    for (let i = balls.length - 1; i >= 0; i -= 1) {
      const ball = balls[i];
      if (ball.launched && ball.groundHitAt === null) {
        lastLaunchedBall = ball;
        return ball.mesh.position;
      }
    }

    lastLaunchedBall = null;
    return crowdLookTarget;
  }

  function animateCrowd(dt) {
    crowdTime += dt;
    crowdCheerPulse = Math.max(0, crowdCheerPulse - dt * 1.15);
    const targetEnergy = THREE.MathUtils.clamp(
      (crowdCelebrate ? 1 : 0.24) + crowdCheerPulse * 0.7,
      0.24,
      1,
    );
    crowdEnergy += (targetEnergy - crowdEnergy) * Math.min(1, dt * 3.2);
    if (!gameOver && crowdTime >= nextCrowdChatterAt) {
      emitAmbientCrowdTip();
    }
    const focusTarget = resolveCrowdFocusTarget();

    for (const spectator of activeSpectators) {
      const idleBob = Math.sin(crowdTime * 1.45 + spectator.bobPhase) * 0.018;
      const cheerBob =
        Math.max(
          0,
          Math.sin(
            crowdTime * (4.2 + spectator.clapSpeed * 0.1) +
              spectator.cheerPhase,
          ),
        ) *
        (0.062 * crowdEnergy + crowdCheerPulse * 0.045);
      spectator.root.position.y = spectator.baseY + idleBob + cheerBob;
      const targetYaw =
        Math.atan2(
          focusTarget.x - spectator.root.position.x,
          focusTarget.z - spectator.root.position.z,
        ) +
        Math.PI +
        spectator.lookOffsetYaw;
      spectator.lookYaw = THREE.MathUtils.lerp(
        spectator.lookYaw,
        targetYaw,
        THREE.MathUtils.clamp(dt * 9.5, 0, 1),
      );
      spectator.root.rotation.y =
        spectator.lookYaw +
        Math.sin(crowdTime * 0.72 + spectator.cheerPhase) *
          0.05 *
          (1 - crowdEnergy * 0.36);

      const clap = Math.sin(
        crowdTime * (spectator.clapSpeed + crowdCheerPulse * 3.8) +
          spectator.clapPhase,
      );
      const lift = 0.24 + crowdEnergy * 1.08;
      const inward = (0.2 + crowdEnergy * 0.58) * clap;
      spectator.leftArmPivot.rotation.x =
        lift + Math.max(0, clap) * 0.24 * crowdEnergy;
      spectator.rightArmPivot.rotation.x =
        lift + Math.max(0, -clap) * 0.24 * crowdEnergy;
      spectator.leftArmPivot.rotation.z = 0.72 - inward;
      spectator.rightArmPivot.rotation.z = -0.72 + inward;
    }
    updateCrowdTips();
  }

  function updateGrabbedBallState(ball) {
    if (!ball.grabbed) {
      return false;
    }

    raycaster.setFromCamera(pointer, camera);
    tmpDir.copy(raycaster.ray.direction).multiplyScalar(grabDistance);
    const target = tmpTarget.copy(raycaster.ray.origin).add(tmpDir);
    clampBallPosition(target);

    const now = performance.now();
    const grabDt = Math.max(0.001, (now - prevGrabTime) / 1000);
    dragVelocity.copy(target).sub(prevGrabPos).divideScalar(grabDt);
    prevGrabPos.copy(target);
    prevGrabTime = now;

    ball.mesh.position.copy(target);
    ball.lastY = ball.mesh.position.y;
    return true;
  }

  function handleMissedAttempt(ball) {
    ball.attempted = true;
    if (gameOver) {
      return;
    }

    noFailStreak = 0;
    timeRemaining = Math.max(0, timeRemaining - TIME_PENALTY_ON_MISS);
    updateScore();
    if (timeRemaining <= 0) {
      emitMissCrowdTips(true);
      endGame("Time up");
      return;
    }

    emitMissCrowdTips(false);
    flashStatus(`Miss -${TIME_PENALTY_ON_MISS}s`);
    if (!ball.spawnedOnLaunch) {
      spawnNextBall(260);
    }
  }

  function resolveFloorCollision(ball) {
    const p = ball.mesh.position;
    const v = ball.velocity;
    const floorTop = FLOOR_Y + BALL_RADIUS;
    if (p.y >= floorTop) {
      return;
    }

    if (ball.launched && ball.groundHitAt === null) {
      ball.groundHitAt = performance.now() / 1000;
    }
    if (ball.launched && !ball.attempted) {
      handleMissedAttempt(ball);
    }

    p.y = floorTop;
    if (Math.abs(v.y) > 0.34) {
      v.y *= -0.64;
    } else {
      v.y = 0;
    }
    v.x *= 0.9;
    v.z *= 0.9;
  }

  function resolveBoundsCollision(ball) {
    const p = ball.mesh.position;
    const v = ball.velocity;

    if (p.x < BOUNDS.minX + BALL_RADIUS) {
      p.x = BOUNDS.minX + BALL_RADIUS;
      v.x = Math.abs(v.x) * 0.72;
    } else if (p.x > BOUNDS.maxX - BALL_RADIUS) {
      p.x = BOUNDS.maxX - BALL_RADIUS;
      v.x = -Math.abs(v.x) * 0.72;
    }

    if (p.z < BOUNDS.minZ + BALL_RADIUS) {
      p.z = BOUNDS.minZ + BALL_RADIUS;
      v.z = Math.abs(v.z) * 0.72;
    } else if (p.z > BOUNDS.maxZ - BALL_RADIUS) {
      p.z = BOUNDS.maxZ - BALL_RADIUS;
      v.z = -Math.abs(v.z) * 0.72;
    }
  }

  function shouldDespawnBall(ball) {
    if (!ball.launched || ball.groundHitAt === null) {
      return false;
    }

    const nowSec = performance.now() / 1000;
    if (nowSec - ball.groundHitAt < 5) {
      return false;
    }

    if (grabbedBall === ball) {
      grabbedBall = null;
    }
    return true;
  }

  function maybeScore(ball) {
    if (gameOver) {
      return;
    }
    if (!ball.launched || ball.attempted) {
      return;
    }

    const p = ball.mesh.position;
    tmpHoriz.set(p.x - rimCenter.x, 0, p.z - rimCenter.z);
    const radial = tmpHoriz.length();

    const crossedDown =
      ball.lastY > rimCenter.y && p.y <= rimCenter.y && ball.velocity.y < 0;
    if (!crossedDown) {
      return;
    }

    if (radial < rimRadius - BALL_RADIUS * 0.18) {
      const gained = multiplier;
      score += gained;
      multiplier = Math.min(8, multiplier * MULTIPLIER_STEP);
      noFailStreak += 1;
      const streakMilestone = noFailStreak % 10 === 0;
      let doubledBonus = false;
      crowdCheerPulse = Math.min(
        1,
        crowdCheerPulse + (streakMilestone ? 1 : 0.62),
      );

      const timeBonus = TIME_BONUS_ON_MAKE;
      timeRemaining = Math.min(MAX_TIME_SECONDS, timeRemaining + timeBonus);

      if (streakMilestone) {
        score *= 2;
        doubledBonus = true;
      }

      const statusParts = [`Swish! +${formatPoints(gained)}`, `+${timeBonus}s`];
      if (doubledBonus) {
        statusParts.push("2x score bonus");
      }

      emitScoreCrowdTips(streakMilestone);
      flashStatus(statusParts.join(" | "));
      updateScore();
      ball.attempted = true;
      if (!ball.spawnedOnLaunch) {
        spawnNextBall(240);
      }
    }
  }

  function simulateBall(ball, dt) {
    if (updateGrabbedBallState(ball)) {
      return false;
    }

    const p = ball.mesh.position;
    const v = ball.velocity;

    ball.netCooldown = Math.max(0, ball.netCooldown - dt);
    v.y -= gravity * dt;
    v.multiplyScalar(0.996);
    p.addScaledVector(v, dt);

    resolveFloorCollision(ball);

    handleSideWallCollisions(ball);
    handleCageWallCollisions(ball);
    resolveBoundsCollision(ball);

    handleBackboardCollision(ball);
    handleRimCollision(ball);
    handleNetInteraction(ball);
    maybeScore(ball);

    if (shouldDespawnBall(ball)) {
      return true;
    }

    ball.lastY = p.y;
    return false;
  }

  function onResize() {
    const { width, height } = getStageSize();
    camera.aspect = width / height;
    camera.updateProjectionMatrix();
    renderer.setSize(width, height);
  }
  globalThis.addEventListener("resize", onResize);

  const applyFullscreenUi = (isOn) => {
    stageEl.classList.toggle("is-fullscreen", isOn);
    if (pageEl) {
      pageEl.classList.toggle("is-fullscreen", isOn);
    }
    setFullscreenButtonLabels(isOn);
    // Resize after layout/class changes.
    setTimeout(() => {
      onResize();
    }, 50);
  };

  const isFullscreenOn = () => {
    // Either CSS fullscreen mode, or native fullscreen on the stage element.
    return (
      stageEl.classList.contains("is-fullscreen") ||
      document.fullscreenElement === stageEl
    );
  };

  const requestNativeFullscreen = async () => {
    if (!stageEl.requestFullscreen) {
      return false;
    }
    try {
      await stageEl.requestFullscreen();
      return true;
    } catch (error) {
      console.warn("Native fullscreen request failed", error);
      return false;
    }
  };

  const exitNativeFullscreen = async () => {
    if (!document.exitFullscreen || !document.fullscreenElement) {
      return;
    }
    try {
      await document.exitFullscreen();
    } catch (error) {
      console.warn("Exiting native fullscreen failed", error);
    }
  };

  const toggleFullscreen = async () => {
    const currentlyOn = isFullscreenOn();
    if (currentlyOn) {
      // Prefer exiting native fullscreen if we are in it.
      if (document.fullscreenElement) {
        await exitNativeFullscreen();
      }
      applyFullscreenUi(false);
      return;
    }

    // Try native fullscreen first (true fullscreen). If it fails, fallback to CSS fullscreen.
    const entered = await requestNativeFullscreen();
    applyFullscreenUi(true);
    if (!entered) {
      // CSS fullscreen only.
      return;
    }
  };

  fullscreenButtons.forEach((btn) => {
    btn.addEventListener("click", (event) => {
      event.preventDefault();
      toggleFullscreen();
    });
  });

  document.addEventListener("fullscreenchange", () => {
    // Sync CSS classes with native fullscreen changes (ESC key, etc.)
    const isOn = document.fullscreenElement === stageEl;
    if (isOn) {
      applyFullscreenUi(true);
    } else if (!stageEl.classList.contains("is-fullscreen")) {
      // If CSS mode isn't on, ensure buttons reflect non-fullscreen.
      setFullscreenButtonLabels(false);
      setTimeout(() => onResize(), 50);
    }
  });

  // Best-effort: if the container resizes without a window resize (sidebar toggles etc.),
  // refresh renderer size occasionally.
  let lastSize = { ...initialSize };
  setInterval(() => {
    const next = getStageSize();
    if (next.width !== lastSize.width || next.height !== lastSize.height) {
      lastSize = next;
      onResize();
    }
  }, 500);

  restartBtn.addEventListener("click", startNewGame);

  // Initial fullscreen UI state (supports server-side fs=1 without reloading later).
  setFullscreenButtonLabels(isFullscreenOn());

  updateScore();
  bestPointsEl.textContent = String(bestScore);
  fetchLeaderboard();
  startNewGame();

  let prev = performance.now() / 1000;
  let accumulator = 0;
  const fixed = 1 / 120;

  function tick(nowMs) {
    requestAnimationFrame(tick);
    const now = nowMs / 1000;
    let dt = now - prev;
    prev = now;
    dt = Math.min(0.05, dt);
    accumulator += dt;

    while (accumulator >= fixed) {
      for (let i = balls.length - 1; i >= 0; i -= 1) {
        const ball = balls[i];
        const shouldRemove = simulateBall(ball, fixed);
        if (shouldRemove) {
          scene.remove(ball.mesh);
          balls.splice(i, 1);
        }
      }

      if (!gameOver) {
        timeRemaining = Math.max(0, timeRemaining - fixed);
        const nextLabel = formatSecondsLeft(timeRemaining);
        if (nextLabel !== lastTimerLabel) {
          lastTimerLabel = nextLabel;
          timerEl.textContent = nextLabel;
        }
        if (timeRemaining <= 0) {
          emitMissCrowdTips(true);
          endGame("Time up");
        }
      }

      animateNet(fixed);
      animateCrowd(fixed);
      accumulator -= fixed;
    }

    renderer.render(scene, camera);
  }

  requestAnimationFrame(tick);
})();
