/**
 * Arcade Basketball Machine — 3D Perspective with Elliptical Rim
 * Classic coin-op arcade basketball feel with real physics.
 * 3D elliptical hoop ring viewed from below, parabolic ball arc,
 * perspective depth scaling, diamond-mesh net.
 */
(function () {
    'use strict';

    var canvas = document.getElementById('basketball-canvas');
    if (!canvas) return;
    var ctx = canvas.getContext('2d');

    /* ═══════════════════════ CONSTANTS ═══════════════════════ */
    var GRAVITY      = 0.38;
    var BOUNCE_DAMP  = 0.50;
    var WALL_DAMP    = 0.40;
    var AIR_FRICTION = 0.998;
    var RIM_R        = 5;          /* rim endpoint collision radius */
    var MIN_SCALE    = 0.35;       /* ball scale at hoop depth (perspective) */
    var ROUND_SEC    = 45;
    var BONUS_TIME   = 3;
    var BONUS_EVERY  = 3;

    /* ═══════════════════════ LAYOUT ═══════════════════════ */
    var W, H;
    /* cabinet frame */
    var cabSide, cabTop, cabBot, marqueeH;
    /* play area */
    var pX, pY, pW, pH;
    /* backboard (shield) */
    var bbCX, bbTop, bbW, bbH, bbBorderW;
    /* hoop / rim — elliptical ring viewed from below */
    var rimLX, rimRX, rimY, hoopW, rimERX, rimERY;
    /* ball */
    var ballR, startX, startY;
    /* floor platform */
    var floorY, floorH;

    function resize() {
        var c = canvas.parentElement;
        W = c.clientWidth;
        H = Math.min(720, Math.round(W * 0.95));
        canvas.width  = W;
        canvas.height = H;

        cabSide   = Math.max(8, W * 0.015);
        marqueeH  = Math.max(40, H * 0.07);
        cabTop    = cabSide + marqueeH;
        cabBot    = Math.max(6, H * 0.01);

        pX = cabSide;
        pY = cabTop;
        pW = W - cabSide * 2;
        pH = H - cabTop - cabBot;

        /* backboard — smaller, it's "far away" in perspective */
        bbW = Math.max(90, pW * 0.28);
        bbH = bbW * 0.78;
        bbCX = pX + pW * 0.5;
        bbTop = pY + pH * 0.03;
        bbBorderW = Math.max(3, bbW * 0.045);

        /* hoop below backboard — elliptical dimensions */
        hoopW = bbW * 0.44;
        rimY  = bbTop + bbH - bbBorderW * 0.5;
        rimLX = bbCX - hoopW / 2;
        rimRX = bbCX + hoopW / 2;
        rimERX = hoopW / 2;         /* horizontal radius of the rim ellipse */
        rimERY = rimERX * 0.35;     /* vertical radius — perspective compression */

        /* floor platform (ball lives here, close to viewer) */
        floorH = Math.max(45, pH * 0.16);
        floorY = pY + pH - floorH;

        /* ball — large at start (close to viewer), shrinks with perspective */
        ballR = Math.max(28, pW * 0.07);
        startX = pX + pW * 0.5;
        startY = floorY - ballR - 4;

        resetBall();
    }

    /* ═══════════════════════ STATE ═══════════════════════ */
    var IDLE = 0, PLAYING = 1, OVER = 2;
    var state = IDLE;
    var timer = ROUND_SEC;
    var timerID = null;
    var score = 0, shots = 0, streak = 0, best = 0;

    var bx, by, vx, vy;
    var curR;              /* current perspective-scaled radius */
    var inFlight = false;
    var canScore = false;
    var madeShot = false;

    var dragging = false;
    var dsx, dsy, dcx, dcy, dt0;

    var scoreFX = 0, missFX = 0, bonusFX = 0;
    var netSway = 0;
    var bulbPhase = 0;

    function resetBall() {
        bx = startX; by = startY;
        vx = 0; vy = 0;
        curR = ballR;
        inFlight = false;
        canScore = false;
        madeShot = false;
    }

    /* Perspective: 1.0 at ball start (close) → MIN_SCALE at rim (far) */
    function depthScale(y) {
        var t = (startY - y) / (startY - rimY);
        t = Math.max(0, Math.min(1, t));
        return 1 - t * (1 - MIN_SCALE);
    }

    /* ═══════════════════════ ROUND CONTROL ═══════════════════════ */
    function startRound() {
        state = PLAYING;
        score = 0; shots = 0; streak = 0;
        timer = ROUND_SEC;
        resetBall();
        sync();
        if (timerID) clearInterval(timerID);
        timerID = setInterval(function () {
            if (state !== PLAYING) return;
            timer--;
            syncTimer();
            if (timer <= 0) endRound();
        }, 1000);
        syncTimer();
    }
    function endRound() {
        state = OVER;
        if (timerID) { clearInterval(timerID); timerID = null; }
    }
    function sync() {
        txt('basketball-score', score);
        txt('basketball-attempts', shots);
        txt('basketball-streak', streak);
        txt('basketball-best', best);
    }
    function syncTimer() { txt('basketball-timer', timer); }
    function txt(id, v) { var e = document.getElementById(id); if (e) e.textContent = v; }

    /* ═══════════════════════ INPUT ═══════════════════════ */
    function pos(e) {
        var r = canvas.getBoundingClientRect();
        var t = e.touches ? (e.touches[0] || e.changedTouches[0]) : e;
        return {
            x: (t.clientX - r.left) * (W / r.width),
            y: (t.clientY - r.top)  * (H / r.height)
        };
    }
    function onDown(e) {
        e.preventDefault();
        var p = pos(e);
        if (state === IDLE) { startRound(); return; }
        if (state === OVER) { if (!inFlight) startRound(); return; }
        if (inFlight) return;
        var dx = p.x - bx, dy = p.y - by;
        if (Math.sqrt(dx * dx + dy * dy) < ballR * 3) {
            dragging = true;
            dsx = p.x; dsy = p.y;
            dcx = p.x; dcy = p.y;
            dt0 = Date.now();
        }
    }
    function onMove(e) {
        if (!dragging) return;
        e.preventDefault();
        var p = pos(e);
        dcx = p.x; dcy = p.y;
        bx = startX + (p.x - dsx) * 0.5;
        by = startY + (p.y - dsy) * 0.5;
        by = Math.max(pY + pH * 0.4, Math.min(floorY - ballR, by));
        bx = Math.max(pX + ballR, Math.min(pX + pW - ballR, bx));
    }
    function onUp(e) {
        if (!dragging) return;
        dragging = false;
        e.preventDefault();
        var dx = dcx - dsx, dy = dcy - dsy;
        var dist = Math.sqrt(dx * dx + dy * dy);
        var elapsed = Math.max(0.016, (Date.now() - dt0) / 1000);
        if (dy > -10 || dist < 12) { resetBall(); return; }
        shots++;
        madeShot = false;

        /* ── Parabolic arc toward the hoop ── */
        var speed = Math.min(dist / elapsed, 2400) / 2400;  /* 0–1 power */
        var launch = 9 + speed * 14;                         /* 9–23 range */

        /* Height from ball to rim (positive because by > rimY) */
        var heightToRim = by - rimY;

        /* Desired peak height: slightly above the rim for a nice arc */
        var peakH = heightToRim + pH * 0.08;

        /* Ideal vy from kinematics: v = sqrt(2·g·h) */
        var idealVY = Math.sqrt(2 * GRAVITY * Math.max(60, peakH));

        /* Scale by flick power: weak = 50%, strong = 105% of ideal */
        vy = -idealVY * (0.50 + speed * 0.55);

        /* Cap so ball never peaks more than 25% above rim */
        var maxPeakH = heightToRim * 1.25;
        var maxVY = -Math.sqrt(2 * GRAVITY * Math.max(60, maxPeakH));
        if (vy < maxVY) vy = maxVY;

        /* Horizontal: auto-guide toward hoop center + user aim drift */
        var flightT = Math.abs(vy) / GRAVITY;      /* frames to apex */
        var hoopDX  = bbCX - bx;                    /* offset to hoop center */
        var userDrift = (dx / Math.max(1, dist)) * launch * 0.4;
        vx = hoopDX / (flightT * 1.15) + userDrift;

        inFlight = true;
        canScore = false;
        sync();
    }

    canvas.addEventListener('mousedown',  onDown);
    canvas.addEventListener('mousemove',  onMove);
    canvas.addEventListener('mouseup',    onUp);
    canvas.addEventListener('touchstart', onDown, { passive: false });
    canvas.addEventListener('touchmove',  onMove, { passive: false });
    canvas.addEventListener('touchend',   onUp,   { passive: false });

    /* ═══════════════════════ PHYSICS ═══════════════════════ */
    function update() {
        bulbPhase += 0.04;
        if (netSway > 0.01) netSway *= 0.91; else netSway = 0;
        if (scoreFX > 0) scoreFX--;
        if (missFX  > 0) missFX--;
        if (bonusFX > 0) bonusFX--;
        if (!inFlight) return;
        if (state !== PLAYING && state !== OVER) return;

        vy += GRAVITY;
        vx *= AIR_FRICTION;
        bx += vx;
        by += vy;

        /* perspective-scaled collision radius */
        curR = ballR * depthScale(by);

        /* walls */
        if (bx - curR < pX)      { bx = pX + curR;      vx =  Math.abs(vx) * WALL_DAMP; }
        if (bx + curR > pX + pW) { bx = pX + pW - curR; vx = -Math.abs(vx) * WALL_DAMP; }
        if (by - curR < pY)      { by = pY + curR;       vy =  Math.abs(vy) * WALL_DAMP; }

        /* backboard collision rect (inner board area) */
        var bbRectX = bbCX - bbW / 2 + bbBorderW;
        var bbRectY = bbTop + bbBorderW;
        var bbRectW = bbW - bbBorderW * 2;
        var bbRectH = bbH - bbBorderW;
        collideRect(bbRectX, bbRectY, bbRectW, bbRectH);

        /* rim endpoint collisions (left & right of ellipse) */
        collideCircle(rimLX, rimY, RIM_R);
        collideCircle(rimRX, rimY, RIM_R);

        /* front rim collision (bottom of ellipse, closest to viewer) */
        collideCircle(bbCX, rimY + rimERY, RIM_R * 0.7);

        /* rim bar itself — thin rectangle between endpoints */
        /* We skip a full bar collider for clean swish feel. */
        /* The two circle endpoints handle deflections. */

        /* score detection */
        if (!madeShot) {
            if (by + curR < rimY) canScore = true;
            if (canScore && by > rimY + curR * 0.4 &&
                bx > rimLX + curR * 0.2 && bx < rimRX - curR * 0.2 && vy > 0) {
                madeShot = true;
                canScore = false;
                score++;
                streak++;
                if (streak > best) best = streak;
                scoreFX = 55;
                netSway = 1;
                if (streak > 0 && streak % BONUS_EVERY === 0 && state === PLAYING) {
                    timer += BONUS_TIME;
                    bonusFX = 50;
                    syncTimer();
                }
                sync();
            }
        }

        /* floor bounce (ball return) */
        if (by + curR > floorY) {
            by = floorY - curR;
            vy = -Math.abs(vy) * WALL_DAMP;
            vx *= 0.85;
            if (Math.abs(vy) < 1.5) {
                if (!madeShot) { streak = 0; missFX = 40; }
                resetBall();
                sync();
            }
        }

        /* fell off bottom */
        if (by > pY + pH + curR + 20) {
            if (!madeShot) { streak = 0; missFX = 40; }
            resetBall();
            sync();
        }
    }

    function collideCircle(cx, cy, cr) {
        var dx = bx - cx, dy = by - cy;
        var dist = Math.sqrt(dx * dx + dy * dy);
        var min  = curR + cr;
        if (dist < min && dist > 0.01) {
            var nx = dx / dist, ny = dy / dist;
            bx = cx + nx * min;
            by = cy + ny * min;
            var dot = vx * nx + vy * ny;
            vx = (vx - 2 * dot * nx) * BOUNCE_DAMP;
            vy = (vy - 2 * dot * ny) * BOUNCE_DAMP;
        }
    }
    function collideRect(rx, ry, rw, rh) {
        if (bx + curR <= rx || bx - curR >= rx + rw ||
            by + curR <= ry || by - curR >= ry + rh) return;
        var oL = (bx + curR) - rx;
        var oR = (rx + rw) - (bx - curR);
        var oT = (by + curR) - ry;
        var oB = (ry + rh) - (by - curR);
        var m  = Math.min(oL, oR, oT, oB);
        if (m === oB)      { by = ry + rh + curR; vy =  Math.abs(vy) * BOUNCE_DAMP; }
        else if (m === oT) { by = ry - curR;      vy = -Math.abs(vy) * BOUNCE_DAMP; }
        else if (m === oL) { bx = rx - curR;      vx = -Math.abs(vx) * BOUNCE_DAMP; }
        else               { bx = rx + rw + curR;  vx =  Math.abs(vx) * BOUNCE_DAMP; }
    }

    /* ═══════════════════════ DRAWING ═══════════════════════ */
    function draw() {
        ctx.clearRect(0, 0, W, H);
        drawCabinet();
        drawPlayArea();
        drawBackboardShadow();
        drawBackboard();
        drawRimBack();
        drawNet();
        drawBallShadow();
        drawBall();
        drawRimFront();
        drawFloor();
        drawFX();
        drawHUD();
        drawOverlay();
    }

    /* ─── Arcade cabinet shell ─── */
    function drawCabinet() {
        /* outer shell */
        ctx.fillStyle = '#0e0618';
        ctx.fillRect(0, 0, W, H);

        /* side strips — gradient orange → dark */
        var lg = ctx.createLinearGradient(0, 0, cabSide, 0);
        lg.addColorStop(0, '#ff5500'); lg.addColorStop(1, '#661a00');
        ctx.fillStyle = lg;
        ctx.fillRect(0, 0, cabSide, H);
        var rg = ctx.createLinearGradient(W - cabSide, 0, W, 0);
        rg.addColorStop(0, '#661a00'); rg.addColorStop(1, '#ff5500');
        ctx.fillStyle = rg;
        ctx.fillRect(W - cabSide, 0, cabSide, H);

        /* bottom strip */
        ctx.fillStyle = '#1a0a2e';
        ctx.fillRect(cabSide, H - cabBot, pW, cabBot);

        /* marquee */
        var my = cabSide * 0.5;
        var mg = ctx.createLinearGradient(0, my, 0, my + marqueeH);
        mg.addColorStop(0, '#2d0a55'); mg.addColorStop(0.5, '#4a1a8a'); mg.addColorStop(1, '#2d0a55');
        ctx.fillStyle = mg;
        ctx.fillRect(cabSide, my, pW, marqueeH);
        ctx.strokeStyle = '#ff6600'; ctx.lineWidth = 2;
        ctx.strokeRect(cabSide, my, pW, marqueeH);

        /* chaser light bulbs */
        var n = Math.max(10, Math.floor(pW / 16));
        for (var i = 0; i < n; i++) {
            var ph = (bulbPhase + i * 0.3) % (Math.PI * 2);
            var br = 0.4 + Math.sin(ph) * 0.6;
            var lx = cabSide + 8 + (pW - 16) * i / (n - 1);
            var topY = my + 5, botY = my + marqueeH - 5;
            /* top row */
            ctx.fillStyle = 'rgba(255,' + Math.round(120 + br * 80) + ',0,' + (0.3 + br * 0.7) + ')';
            ctx.beginPath(); ctx.arc(lx, topY, 2.5, 0, 6.283); ctx.fill();
            /* bottom row */
            ctx.beginPath(); ctx.arc(lx, botY, 2.5, 0, 6.283); ctx.fill();
        }

        /* title text */
        ctx.save();
        ctx.fillStyle = '#ffcc00';
        ctx.font = 'bold ' + Math.max(16, marqueeH * 0.45) + 'px sans-serif';
        ctx.textAlign = 'center'; ctx.textBaseline = 'middle';
        ctx.shadowColor = '#ff8800'; ctx.shadowBlur = 10;
        ctx.fillText('ARCADE  BASKETBALL', W * 0.5, my + marqueeH * 0.5);
        ctx.restore();
    }

    /* ─── Play area — gym-wall background with 3D perspective ─── */
    function drawPlayArea() {
        /* soft gym-wall green/teal background like the reference */
        var bg = ctx.createLinearGradient(pX, pY, pX, pY + pH);
        bg.addColorStop(0, '#b8cfc4');
        bg.addColorStop(0.6, '#a8bfb4');
        bg.addColorStop(1, '#8faa9e');
        ctx.fillStyle = bg;
        ctx.fillRect(pX, pY, pW, pH);

        /* perspective convergence lines — walls narrow toward the hoop */
        ctx.save();
        ctx.globalAlpha = 0.06;
        ctx.strokeStyle = '#000';
        ctx.lineWidth = 1;
        var vanishX = bbCX;
        var vanishY = bbTop + bbH * 0.5;
        /* left wall edge converging inward */
        ctx.beginPath();
        ctx.moveTo(pX, floorY);
        ctx.lineTo(vanishX - bbW * 0.3, vanishY);
        ctx.stroke();
        /* right wall edge converging inward */
        ctx.beginPath();
        ctx.moveTo(pX + pW, floorY);
        ctx.lineTo(vanishX + bbW * 0.3, vanishY);
        ctx.stroke();
        /* center guide line */
        ctx.beginPath();
        ctx.moveTo(bbCX, floorY);
        ctx.lineTo(bbCX, vanishY + bbH * 0.3);
        ctx.stroke();
        /* quarter-lines for extra depth cue */
        ctx.beginPath();
        ctx.moveTo(pX + pW * 0.25, floorY);
        ctx.lineTo(vanishX - bbW * 0.12, vanishY);
        ctx.stroke();
        ctx.beginPath();
        ctx.moveTo(pX + pW * 0.75, floorY);
        ctx.lineTo(vanishX + bbW * 0.12, vanishY);
        ctx.stroke();
        ctx.restore();

        /* subtle vignette */
        var vg = ctx.createRadialGradient(pX + pW / 2, pY + pH * 0.35, pW * 0.2, pX + pW / 2, pY + pH * 0.4, pW * 0.7);
        vg.addColorStop(0, 'rgba(255,255,255,0.06)');
        vg.addColorStop(1, 'rgba(0,0,0,0.10)');
        ctx.fillStyle = vg;
        ctx.fillRect(pX, pY, pW, pH);

        /* inner border */
        ctx.strokeStyle = 'rgba(0,0,0,0.15)'; ctx.lineWidth = 1;
        ctx.strokeRect(pX, pY, pW, pH);
    }

    /* ─── Backboard — shield/dome shape with thick red-orange border ─── */
    function drawBackboardShadow() {
        var sx = 10, sy = 12;
        ctx.save();
        ctx.translate(sx, sy);
        ctx.globalAlpha = 0.12;
        drawShieldPath(bbCX, bbTop, bbW, bbH);
        ctx.fillStyle = '#000';
        ctx.fill();
        ctx.restore();
    }

    function drawBackboard() {
        /* outer border (red-orange) */
        drawShieldPath(bbCX, bbTop, bbW, bbH);
        var bg = ctx.createLinearGradient(bbCX - bbW / 2, bbTop, bbCX - bbW / 2, bbTop + bbH);
        bg.addColorStop(0, '#d43d1a'); bg.addColorStop(0.5, '#e8501a'); bg.addColorStop(1, '#c0331a');
        ctx.fillStyle = bg;
        ctx.fill();
        ctx.strokeStyle = '#8a2210'; ctx.lineWidth = 2; ctx.stroke();

        /* inner white board */
        var iw = bbW - bbBorderW * 2;
        var ih = bbH - bbBorderW * 1.8;
        drawShieldPath(bbCX, bbTop + bbBorderW, iw, ih);
        var wg = ctx.createLinearGradient(bbCX, bbTop + bbBorderW, bbCX, bbTop + bbBorderW + ih);
        wg.addColorStop(0, '#f5f5f0'); wg.addColorStop(0.5, '#eeeeea'); wg.addColorStop(1, '#e0e0da');
        ctx.fillStyle = wg;
        ctx.fill();
        ctx.strokeStyle = 'rgba(0,0,0,0.08)'; ctx.lineWidth = 1; ctx.stroke();

        /* inner target rectangle (orange outline) */
        var tW = iw * 0.34, tH = ih * 0.3;
        var tX = bbCX - tW / 2;
        var tY = bbTop + bbH * 0.42;
        ctx.strokeStyle = '#d84a1a'; ctx.lineWidth = Math.max(2, bbBorderW * 0.45);
        ctx.strokeRect(tX, tY, tW, tH);
    }

    /* shield / dome shaped backboard path */
    function drawShieldPath(cx, top, w, h) {
        var hW = w / 2;
        var r = hW * 0.45; /* top corner radius */
        ctx.beginPath();
        ctx.moveTo(cx - hW, top + h);                               /* bottom-left */
        ctx.lineTo(cx - hW, top + r);                               /* left edge up */
        ctx.quadraticCurveTo(cx - hW, top, cx - hW + r, top);      /* top-left curve */
        ctx.lineTo(cx + hW - r, top);                               /* top edge */
        ctx.quadraticCurveTo(cx + hW, top, cx + hW, top + r);      /* top-right curve */
        ctx.lineTo(cx + hW, top + h);                               /* right edge down */
        ctx.closePath();
    }

    /* ─── Rim — 3D elliptical ring viewed from below ─── */
    function drawRimBack() {
        /* support bracket from backboard bottom to rim */
        ctx.strokeStyle = '#888';
        ctx.lineWidth = Math.max(3, hoopW * 0.04);
        ctx.beginPath();
        ctx.moveTo(bbCX, bbTop + bbH);
        ctx.lineTo(bbCX, rimY - rimERY);
        ctx.stroke();

        /* back half of 3D rim ring (far side = top arc on screen) */
        /* Drawn BEHIND net and ball */
        var rimThick = Math.max(4, RIM_R * 1.6);

        /* outer edge of the rim tube — dark red */
        ctx.strokeStyle = '#aa1a00';
        ctx.lineWidth = rimThick;
        ctx.beginPath();
        ctx.ellipse(bbCX, rimY, rimERX, rimERY, 0, Math.PI, 0, true);
        ctx.stroke();

        /* inner highlight for 3D tube effect — lighter, slightly smaller */
        ctx.strokeStyle = '#cc3a18';
        ctx.lineWidth = Math.max(1, rimThick * 0.45);
        ctx.beginPath();
        ctx.ellipse(bbCX, rimY, rimERX * 0.92, rimERY * 0.7, 0, Math.PI, 0, true);
        ctx.stroke();
    }
    function drawRimFront() {
        /* front half of 3D rim ring (near side = bottom arc on screen) */
        /* Drawn ON TOP OF net and ball for proper layering */
        var rimThick = Math.max(4, RIM_R * 1.6);

        /* main front arc — bright red */
        ctx.strokeStyle = '#cc2200';
        ctx.lineWidth = rimThick;
        ctx.beginPath();
        ctx.ellipse(bbCX, rimY, rimERX, rimERY, 0, 0, Math.PI, false);
        ctx.stroke();

        /* specular highlight on the front rim — shinier, tube reflection */
        ctx.strokeStyle = '#ee5533';
        ctx.lineWidth = Math.max(1, rimThick * 0.35);
        ctx.beginPath();
        ctx.ellipse(bbCX, rimY, rimERX, rimERY * 1.08, 0, 0.2, Math.PI - 0.2, false);
        ctx.stroke();

        /* metal endpoints at the left and right of the rim */
        ctx.fillStyle = '#cc2200';
        ctx.beginPath(); ctx.arc(rimLX, rimY, RIM_R, 0, 6.283); ctx.fill();
        ctx.strokeStyle = '#991800'; ctx.lineWidth = 1; ctx.stroke();
        ctx.beginPath(); ctx.arc(rimRX, rimY, RIM_R, 0, 6.283); ctx.fill();
        ctx.stroke();

        /* front-center endpoint (bottom of ellipse, closest to viewer) */
        ctx.fillStyle = '#dd3311';
        ctx.beginPath(); ctx.arc(bbCX, rimY + rimERY, RIM_R * 0.7, 0, 6.283); ctx.fill();
        ctx.strokeStyle = '#991800'; ctx.lineWidth = 0.8; ctx.stroke();
    }

    /* ─── Net — diamond mesh hanging from elliptical rim ─── */
    function drawNet() {
        var depth = hoopW * 0.85;   /* how far down the net hangs */
        var rows = 6, cols = 8;
        ctx.lineWidth = Math.max(1, pW * 0.002);

        for (var r = 0; r < rows; r++) {
            var t1 = r / rows, t2 = (r + 1) / rows;
            var hang1 = t1 * depth, hang2 = t2 * depth;
            /* net narrows toward the bottom (cone shape) */
            var narrow1 = 1 - t1 * 0.50;
            var narrow2 = 1 - t2 * 0.50;
            /* elliptical curvature fades as the net tapers */
            var curve1 = rimERY * (1 - t1 * 0.7);
            var curve2 = rimERY * (1 - t2 * 0.7);
            var sway1 = netSway * Math.sin(t1 * 5) * 6;
            var sway2 = netSway * Math.sin(t2 * 5) * 6;

            /* vertical strings (zig-zag for diamond pattern) */
            for (var c = 0; c <= cols; c++) {
                var f1 = c / cols;
                var f2;
                /* offset every other row for diamond effect */
                if (r % 2 === 0) {
                    f2 = (c - 0.5) / cols;
                } else {
                    f2 = (c + 0.5) / cols;
                }
                /* X: distribute across narrowing hoop width */
                var x1 = rimLX + f1 * hoopW * narrow1 + sway1;
                var x2 = rimLX + Math.max(0, Math.min(1, f2)) * hoopW * narrow2 + sway2;
                /* Y: rimY baseline + elliptical bulge (front of rim curves down toward viewer) + hang */
                var bulge1 = curve1 * Math.sin(Math.PI * f1);
                var bulge2 = curve2 * Math.sin(Math.PI * Math.max(0, Math.min(1, f2)));
                var y1 = rimY + bulge1 + hang1;
                var y2 = rimY + bulge2 + hang2;
                /* fade net toward bottom */
                var alpha = 0.55 - t1 * 0.22;
                ctx.strokeStyle = 'rgba(255,255,255,' + alpha + ')';
                ctx.beginPath();
                ctx.moveTo(x1, y1);
                ctx.lineTo(x2, y2);
                ctx.stroke();
            }
        }
    }

    /* ─── Ball shadow ─── */
    function drawBallShadow() {
        var x, y;
        if (state === IDLE) { x = startX; y = startY; }
        else if (state === OVER && !inFlight) { return; }
        else { x = bx; y = by; }

        var vr = ballR * depthScale(y);
        /* shadow on floor */
        var sy = floorY - 1;
        var shadowScale = 1 - Math.max(0, Math.min(1, (sy - y) / (pH * 0.6))) * 0.5;
        var sr = vr * shadowScale;
        ctx.fillStyle = 'rgba(0,0,0,0.15)';
        ctx.beginPath();
        ctx.ellipse(x, sy, sr * 1.3, sr * 0.22, 0, 0, 6.283);
        ctx.fill();
    }

    /* ─── Ball — large, detailed, realistic ─── */
    function drawBall() {
        var x, y;
        if (state === IDLE) { x = startX; y = startY; }
        else if (state === OVER && !inFlight) { return; }
        else { x = bx; y = by; }
        var r = ballR * depthScale(y);
        if (r < 3) return;

        /* main sphere gradient (light in upper-left for 3D) */
        var g = ctx.createRadialGradient(x - r * 0.35, y - r * 0.35, r * 0.05, x, y, r);
        g.addColorStop(0,   '#ffbb55');
        g.addColorStop(0.3, '#f09030');
        g.addColorStop(0.65,'#d06010');
        g.addColorStop(1,   '#803000');
        ctx.fillStyle = g;
        ctx.beginPath(); ctx.arc(x, y, r, 0, 6.283); ctx.fill();

        /* outline */
        ctx.strokeStyle = '#5a2000'; ctx.lineWidth = Math.max(1.5, r * 0.06); ctx.stroke();

        /* seam lines (cross + curved) */
        ctx.save();
        ctx.beginPath();
        ctx.arc(x, y, r - 0.5, 0, 6.283);
        ctx.clip();

        ctx.strokeStyle = 'rgba(60,20,0,0.25)';
        ctx.lineWidth = Math.max(1, r * 0.035);
        /* horizontal seam */
        ctx.beginPath(); ctx.moveTo(x - r, y); ctx.lineTo(x + r, y); ctx.stroke();
        /* vertical seam */
        ctx.beginPath(); ctx.moveTo(x, y - r); ctx.lineTo(x, y + r); ctx.stroke();
        /* curved seams (left & right) */
        ctx.beginPath();
        ctx.arc(x - r * 0.02, y, r * 0.62, -1.0, 1.0);
        ctx.stroke();
        ctx.beginPath();
        ctx.arc(x + r * 0.02, y, r * 0.62, Math.PI - 1.0, Math.PI + 1.0);
        ctx.stroke();
        ctx.restore();

        /* specular highlight (top-left bright spot) */
        var hl = ctx.createRadialGradient(x - r * 0.3, y - r * 0.32, 0, x - r * 0.3, y - r * 0.32, r * 0.45);
        hl.addColorStop(0, 'rgba(255,255,255,0.40)');
        hl.addColorStop(1, 'rgba(255,255,255,0)');
        ctx.fillStyle = hl;
        ctx.beginPath(); ctx.arc(x, y, r, 0, 6.283); ctx.fill();
    }

    /* ─── Floor platform ─── */
    function drawFloor() {
        /* dark platform at the bottom (ball return tray) */
        var fg = ctx.createLinearGradient(pX, floorY, pX, floorY + floorH);
        fg.addColorStop(0, '#2e4a5a'); fg.addColorStop(0.3, '#1e3644'); fg.addColorStop(1, '#152a36');
        ctx.fillStyle = fg;
        ctx.fillRect(pX, floorY, pW, floorH);

        /* top edge highlight */
        ctx.strokeStyle = 'rgba(255,255,255,0.1)'; ctx.lineWidth = 1;
        ctx.beginPath(); ctx.moveTo(pX, floorY + 0.5); ctx.lineTo(pX + pW, floorY + 0.5); ctx.stroke();

        /* subtle grid lines on the tray */
        ctx.strokeStyle = 'rgba(255,255,255,0.03)';
        var g = Math.max(20, pW * 0.06);
        for (var i = 1; i < pW / g; i++) {
            var lx = pX + i * g;
            ctx.beginPath(); ctx.moveTo(lx, floorY + 2); ctx.lineTo(lx, floorY + floorH - 2); ctx.stroke();
        }
    }

    /* ─── Score / miss / bonus FX ─── */
    function drawFX() {
        ctx.textAlign = 'center'; ctx.textBaseline = 'middle';
        if (scoreFX > 0) {
            var a = (scoreFX / 55) * 0.9;
            ctx.save();
            ctx.shadowColor = 'rgba(0,200,80,0.6)'; ctx.shadowBlur = 12;
            ctx.fillStyle = 'rgba(30,220,90,' + a + ')';
            ctx.font = 'bold ' + Math.max(28, pW * 0.065) + 'px sans-serif';
            ctx.fillText('SCORE!', pX + pW / 2, pY + pH * 0.47);
            if (streak > 1) {
                ctx.font = 'bold ' + Math.max(16, pW * 0.035) + 'px sans-serif';
                ctx.fillText(streak + 'x STREAK!', pX + pW / 2, pY + pH * 0.47 + Math.max(30, pW * 0.06));
            }
            ctx.restore();
        }
        if (missFX > 0) {
            ctx.fillStyle = 'rgba(255,70,70,' + (missFX / 40) * 0.55 + ')';
            ctx.font = 'bold ' + Math.max(22, pW * 0.045) + 'px sans-serif';
            ctx.fillText('MISS', pX + pW / 2, pY + pH * 0.47);
        }
        if (bonusFX > 0) {
            ctx.save();
            ctx.shadowColor = 'rgba(255,200,0,0.5)'; ctx.shadowBlur = 8;
            ctx.fillStyle = 'rgba(255,215,40,' + (bonusFX / 50) * 0.9 + ')';
            ctx.font = 'bold ' + Math.max(17, pW * 0.038) + 'px sans-serif';
            ctx.fillText('+' + BONUS_TIME + 's BONUS!', pX + pW / 2, pY + pH * 0.56);
            ctx.restore();
        }
    }

    /* ─── In-canvas HUD (score overlay in top-left) ─── */
    function drawHUD() {
        if (state !== PLAYING) return;
        ctx.save();
        /* score badge */
        ctx.fillStyle = 'rgba(0,0,0,0.35)';
        var bw = Math.max(90, pW * 0.22), bh = Math.max(28, pH * 0.055);
        roundRect(pX + 8, pY + 8, bw, bh, 6); ctx.fill();
        ctx.fillStyle = '#2dff70';
        ctx.font = 'bold ' + Math.max(14, bh * 0.6) + 'px sans-serif';
        ctx.textAlign = 'left'; ctx.textBaseline = 'middle';
        ctx.fillText('SCORE: ' + score, pX + 16, pY + 8 + bh / 2);

        /* timer badge (top-right) */
        var tw = Math.max(50, pW * 0.12);
        var tx = pX + pW - tw - 8;
        var low = timer <= 10;
        ctx.fillStyle = low ? 'rgba(180,25,25,0.8)' : 'rgba(0,0,0,0.35)';
        roundRect(tx, pY + 8, tw, bh, 6); ctx.fill();
        ctx.fillStyle = low ? '#ff8888' : '#fff';
        ctx.font = 'bold ' + Math.max(14, bh * 0.6) + 'px sans-serif';
        ctx.textAlign = 'center';
        ctx.fillText(timer + 's', tx + tw / 2, pY + 8 + bh / 2);
        ctx.restore();
    }

    /* ─── Overlays (idle / game-over) ─── */
    function drawOverlay() {
        ctx.textAlign = 'center'; ctx.textBaseline = 'middle';

        if (state === IDLE) {
            ctx.fillStyle = 'rgba(0,0,0,0.55)';
            ctx.fillRect(pX, pY, pW, pH);

            ctx.save();
            ctx.shadowColor = '#ff8800'; ctx.shadowBlur = 15;
            ctx.fillStyle = '#ffcc00';
            ctx.font = 'bold ' + Math.max(24, pW * 0.06) + 'px sans-serif';
            ctx.fillText('TAP TO PLAY', pX + pW / 2, pY + pH * 0.38);
            ctx.restore();

            ctx.fillStyle = 'rgba(255,255,255,0.55)';
            ctx.font = Math.max(13, pW * 0.028) + 'px sans-serif';
            ctx.fillText('Flick the ball into the hoop!', pX + pW / 2, pY + pH * 0.48);
            ctx.fillText(ROUND_SEC + ' seconds per round', pX + pW / 2, pY + pH * 0.54);
        }

        if (state === OVER && !inFlight) {
            ctx.fillStyle = 'rgba(0,0,0,0.6)';
            ctx.fillRect(pX, pY, pW, pH);

            ctx.save();
            ctx.shadowColor = '#ff4400'; ctx.shadowBlur = 10;
            ctx.fillStyle = '#ff6600';
            ctx.font = 'bold ' + Math.max(24, pW * 0.055) + 'px sans-serif';
            ctx.fillText('TIME\'S UP!', pX + pW / 2, pY + pH * 0.25);
            ctx.restore();

            ctx.fillStyle = '#ffffff';
            ctx.font = 'bold ' + Math.max(36, pW * 0.08) + 'px sans-serif';
            ctx.fillText(score + ' POINTS', pX + pW / 2, pY + pH * 0.4);

            ctx.fillStyle = 'rgba(255,255,255,0.5)';
            ctx.font = Math.max(13, pW * 0.028) + 'px sans-serif';
            ctx.fillText(shots + ' shots  \u2022  Best streak: ' + best, pX + pW / 2, pY + pH * 0.51);

            ctx.save();
            ctx.shadowColor = '#ff8800'; ctx.shadowBlur = 8;
            ctx.fillStyle = '#ffcc00';
            ctx.font = 'bold ' + Math.max(16, pW * 0.036) + 'px sans-serif';
            ctx.fillText('TAP TO PLAY AGAIN', pX + pW / 2, pY + pH * 0.65);
            ctx.restore();
        }

        /* first-shot hint */
        if (state === PLAYING && !inFlight && shots === 0 && !dragging) {
            ctx.fillStyle = 'rgba(0,0,0,0.35)';
            ctx.font = Math.max(13, pW * 0.026) + 'px sans-serif';
            ctx.fillText('Flick the ball toward the hoop!', pX + pW / 2, pY + pH * 0.62);
        }
    }

    /* ─── Rounded rect helper ─── */
    function roundRect(x, y, w, h, r) {
        ctx.beginPath();
        ctx.moveTo(x + r, y);
        ctx.lineTo(x + w - r, y);
        ctx.quadraticCurveTo(x + w, y, x + w, y + r);
        ctx.lineTo(x + w, y + h - r);
        ctx.quadraticCurveTo(x + w, y + h, x + w - r, y + h);
        ctx.lineTo(x + r, y + h);
        ctx.quadraticCurveTo(x, y + h, x, y + h - r);
        ctx.lineTo(x, y + r);
        ctx.quadraticCurveTo(x, y, x + r, y);
        ctx.closePath();
    }

    /* ═══════════════════════ LOOP ═══════════════════════ */
    function loop() {
        update();
        draw();
        requestAnimationFrame(loop);
    }

    resize();
    window.addEventListener('resize', resize);
    loop();
})();
