(function () {
  const HTML_ESCAPE_MAP = {
    "&": "&amp;",
    "<": "&lt;",
    ">": "&gt;",
    '"': "&quot;",
    "'": "&#39;",
  };

  function noop() {}

  function escapeHtml(str) {
    return String(str).replaceAll(/[&<>"']/g, (char) => {
      return HTML_ESCAPE_MAP[char] || char;
    });
  }

  function sanitizeTag(tag) {
    return String(tag || "")
      .trim()
      .toLowerCase()
      .replaceAll(/[_\s]+/g, "-");
  }

  function normalizeSearchText(text) {
    return String(text || "")
      .toLowerCase()
      .replaceAll(/[-_\s]+/g, "");
  }

  function onDbUpgradeNeeded(req, storeName, logFn) {
    logFn("DB Upgrade needed - creating store");
    const db = req.result;
    if (!db.objectStoreNames.contains(storeName)) {
      db.createObjectStore(storeName, { keyPath: "name" });
    }
  }

  function onDbOpenSuccess(req, logFn, resolve) {
    const db = req.result;
    logFn(`DB Opened: ${db.name} v${db.version}`);
    db.onversionchange = () => {
      db.close();
    };
    resolve(db);
  }

  function onDbOpenError(req, logFn, reject) {
    logFn("DB Open Error: " + req.error, "error");
    reject(req.error);
  }

  function resolveRequestResult(req, resolve) {
    resolve(req.result || []);
  }

  function resolveEmptyResult(resolve) {
    resolve([]);
  }

  function itemMatchesSearch(item, query) {
    if (!query) {
      return true;
    }

    const lowerQuery = query.toLowerCase();
    if (
      String(item.name || "")
        .toLowerCase()
        .includes(lowerQuery)
    ) {
      return true;
    }

    const tags = Array.isArray(item.tags) ? item.tags : [];
    for (const tag of tags) {
      if (
        String(tag || "")
          .toLowerCase()
          .includes(lowerQuery)
      ) {
        return true;
      }
    }

    return false;
  }

  function itemMatchesType(item, typeFilter) {
    if (typeFilter === "images") {
      return !item.isVideo;
    }
    if (typeFilter === "videos") {
      return item.isVideo;
    }
    if (typeFilter === "untagged") {
      return !(item.tags && item.tags.length > 0);
    }
    return true;
  }

  function itemMatchesTagFilters(item, activeFilters, strictMode) {
    if (activeFilters.length === 0) {
      return true;
    }

    const tagSet = new Set(item.tags || []);
    const hasAll = activeFilters.every((tag) => tagSet.has(tag));
    if (!hasAll) {
      return false;
    }

    if (!strictMode) {
      return true;
    }

    return tagSet.size === activeFilters.length;
  }

  function handleGeneratedImageThumbBlob(item, imgEl, blob) {
    if (!blob) return;
    item.gridThumbUrl = URL.createObjectURL(blob);
    if (imgEl) {
      imgEl.src = item.gridThumbUrl;
    }
  }

  function handleGeneratedVideoThumbBlob(item, posterEl, blob) {
    if (!blob) return;
    item.thumbUrl = URL.createObjectURL(blob);
    if (posterEl) {
      posterEl.src = item.thumbUrl;
    }
  }

  function countSharedTags(itemTags, activeTags) {
    const sourceTags = Array.isArray(itemTags) ? itemTags : [];
    const activeSet = new Set(Array.isArray(activeTags) ? activeTags : []);
    let sharedCount = 0;

    for (const tag of sourceTags) {
      if (activeSet.has(tag)) {
        sharedCount++;
      }
    }

    return sharedCount;
  }

  function parseTagsFromFilename(filename) {
    const ext = "." + filename.split(".").pop();
    const nameNoExt = filename.slice(0, -ext.length);
    const parts = nameNoExt.split("_");
    if (parts.length <= 1) return { baseName: nameNoExt, tags: [] };
    return {
      baseName: parts[0],
      tags: parts.slice(1).filter((t) => t && t.trim().length > 0),
    };
  }

  function buildFilename(baseName, tags, ext) {
    if (tags.length === 0) return baseName + ext;
    return baseName + "_" + tags.join("_") + ext;
  }

  function setReconnectButtonHidden(hidden) {
    const reconnectBtn = document.getElementById("reconnectBtn");
    reconnectBtn?.classList.toggle("hidden", hidden);
  }

  async function classifyStoredHandle(entry) {
    // NOSONAR - helper is intentionally at module helper scope (outside initEasyMedia)
    const handle = entry?.handle;
    if (!handle) {
      return { kind: "invalid", handle: null };
    }

    const perm = await handle.queryPermission({ mode: "read" });
    if (perm === "granted") {
      return { kind: "granted", handle };
    }
    if (perm === "prompt") {
      return { kind: "pending", handle };
    }

    return { kind: "denied", handle };
  }

  function initEasyMedia() {
    // NOSONAR - legacy UI bootstrap orchestrator
    const root = document.getElementById("easyMediaApp");
    if (!root) return;
    if (root.dataset.emInit === "1") return;
    root.dataset.emInit = "1";

    function revealRoot() {
      if (root.hasAttribute("hidden")) {
        root.removeAttribute("hidden");
      }
    }

    function ensureStyles() {
      const head = document.head || document.getElementsByTagName("head")[0];
      if (!head) {
        revealRoot();
        return;
      }

      const existing = document.getElementById("easy-media-css");
      if (existing) {
        if (existing.sheet) {
          revealRoot();
        } else {
          existing.addEventListener("load", revealRoot, { once: true });
          existing.addEventListener("error", revealRoot, { once: true });
        }
        return;
      }

      const link = document.createElement("link");
      link.id = "easy-media-css";
      link.rel = "stylesheet";
      link.href = "/assets/css/easy-media.css";
      link.addEventListener("load", revealRoot, { once: true });
      link.addEventListener("error", revealRoot, { once: true });
      head.appendChild(link);
    }

    ensureStyles();
    setTimeout(revealRoot, 1500);

    document.body.classList.add("easy-media-active");

    function normalizeAiZipDownloadPath(rawUrl) {
      const adminPrefix = "/admin-file/";
      const sitePrefix = "/site-file/";
      const value = String(rawUrl || "").trim();
      if (!value || value === "#" || value === "{EASY_MEDIA_AI_ZIP_URL}") {
        return "";
      }

      let parsedPath = "";
      try {
        if (/^https?:\/\//i.test(value)) {
          parsedPath = new URL(value).pathname || "";
        } else {
          parsedPath =
            new URL(value, globalThis.location.origin).pathname || "";
        }
      } catch {
        parsedPath = value;
      }

      let normalizedPath = parsedPath;
      if (normalizedPath.startsWith(adminPrefix)) {
        normalizedPath = sitePrefix + normalizedPath.slice(adminPrefix.length);
      }

      const hasTraversal = /(^|\/)\.\.(\/|$)/.test(normalizedPath);
      const isSiteFile = normalizedPath.startsWith(sitePrefix);
      const isZip = /\.zip$/i.test(normalizedPath);

      if (!isSiteFile || hasTraversal || !isZip) {
        return "";
      }

      return normalizedPath;
    }

    function normalizeAiHelpImagePath(rawUrl) {
      const adminPrefix = "/admin-file/";
      const sitePrefix = "/site-file/";
      const value = String(rawUrl || "").trim();
      if (!value || value === "#") {
        return "";
      }

      let parsedPath = "";
      try {
        if (/^https?:\/\//i.test(value)) {
          parsedPath = new URL(value).pathname || "";
        } else {
          parsedPath =
            new URL(value, globalThis.location.origin).pathname || "";
        }
      } catch {
        parsedPath = value;
      }

      let normalizedPath = parsedPath;
      if (normalizedPath.startsWith(adminPrefix)) {
        normalizedPath = sitePrefix + normalizedPath.slice(adminPrefix.length);
      }

      const hasTraversal = /(^|\/|\\)\.\.(\/|\\|$)/.test(normalizedPath);
      const isSiteFile = normalizedPath.startsWith(sitePrefix);
      const isImage = /\.(jpg|jpeg|png|gif|webp|avif|svg)$/i.test(
        normalizedPath,
      );

      if (!isSiteFile || hasTraversal || !isImage) {
        return "";
      }

      return normalizedPath;
    }

    function parseAiHelpImagesConfig() {
      const raw = document.body?.dataset?.easyMediaHelpImages || "[]";
      let values = [];
      try {
        const parsed = JSON.parse(raw);
        if (Array.isArray(parsed)) {
          values = parsed;
        }
      } catch {
        values = [];
      }

      const deduped = [];
      const seen = new Set();
      values.forEach((entry) => {
        const normalized = normalizeAiHelpImagePath(entry);
        if (normalized && !seen.has(normalized)) {
          seen.add(normalized);
          deduped.push(normalized);
        }
      });

      return deduped;
    }

    function ensureAboutUi() {
      const toolbar = root.querySelector(".toolbar");
      const settingsBtn = toolbar?.querySelector(
        '[data-em-action="toggle-settings"]',
      );

      if (settingsBtn) {
        settingsBtn.setAttribute("title", "Settings");
      }

      if (
        toolbar &&
        settingsBtn &&
        !toolbar.querySelector('[data-em-action="toggle-about"]')
      ) {
        const infoBtn = document.createElement("button");
        infoBtn.className = "btn em-btn-info";
        infoBtn.dataset.emAction = "toggle-about";
        infoBtn.setAttribute("title", "About Easy Media");
        infoBtn.setAttribute("aria-label", "About Easy Media");
        infoBtn.innerHTML = `
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="12" cy="12" r="10"></circle>
                        <line x1="12" y1="16" x2="12" y2="12"></line>
                        <line x1="12" y1="8" x2="12.01" y2="8"></line>
                    </svg>
                `;
        settingsBtn.after(infoBtn);
      }

      const settingsModal = document.getElementById("settingsModal");
      let aboutModal = document.getElementById("aboutModal");
      if (!settingsModal || aboutModal) {
        return;
      }

      const aboutTitle = Array.from(
        settingsModal.querySelectorAll(".modal-title"),
      ).find((el) => {
        return (el.textContent || "").trim().toLowerCase() === "about";
      });
      const aboutContent = aboutTitle?.nextElementSibling;
      if (
        !aboutTitle ||
        !aboutContent?.classList.contains("em-about-content")
      ) {
        return;
      }

      aboutModal = document.createElement("div");
      aboutModal.className = "modal";
      aboutModal.id = "aboutModal";
      aboutModal.innerHTML = `
                <div class="modal-content em-about-modal-content">
                    <div class="modal-buttons em-modal-footer">
                        <button class="btn" data-em-action="toggle-about">Close</button>
                    </div>
                </div>
            `;

      const aboutContainer = aboutModal.querySelector(".modal-content");
      if (!aboutContainer) {
        return;
      }

      const closeRow = aboutContainer.querySelector(".modal-buttons");
      aboutContainer.insertBefore(aboutTitle.cloneNode(true), closeRow);
      aboutContainer.insertBefore(aboutContent.cloneNode(true), closeRow);

      settingsModal.parentNode?.insertBefore(
        aboutModal,
        settingsModal.nextSibling,
      );
      aboutTitle.remove();
      aboutContent.remove();
    }

    function ensureAiZipDownloadButton() {
      const settingsModal = document.getElementById("settingsModal");
      if (!settingsModal) {
        return null;
      }

      const failedBtn = settingsModal.querySelector("#failedBtnSettings");
      const utilitiesRow =
        failedBtn?.closest(".em-utilities-row") ||
        settingsModal.querySelector(".em-utilities-row");
      let downloadBtn = settingsModal.querySelector(
        "[data-em-ai-zip-download]",
      );

      if (!downloadBtn) {
        const row = document.createElement("div");
        row.className = "em-utilities-row em-utilities-row-download";

        downloadBtn = document.createElement("a");
        downloadBtn.className = "btn em-flex-btn em-ai-zip-btn";
        downloadBtn.dataset.emAiZipDownload = "1";
        downloadBtn.textContent = "⬇️ Download Media AI ZIP";
        row.appendChild(downloadBtn);

        if (utilitiesRow) {
          utilitiesRow.after(row);
        } else {
          const toggleRow = settingsModal.querySelector(".toggle-row");
          if (toggleRow?.parentNode) {
            toggleRow.parentNode.insertBefore(row, toggleRow);
          }
        }
      }

      if (!downloadBtn) {
        return null;
      }

      const bodyConfigured = normalizeAiZipDownloadPath(
        document.body?.dataset?.easyMediaAiZipUrl || "",
      );
      const buttonConfigured = normalizeAiZipDownloadPath(
        downloadBtn.getAttribute("href") || "",
      );
      const effectiveUrl = buttonConfigured || bodyConfigured;
      const trackedDownloadRoute = "/easy-media-ai-download";

      if (!downloadBtn.dataset.emZipClickBound) {
        downloadBtn.addEventListener("click", (event) => {
          if (downloadBtn.classList.contains("em-ai-zip-btn-disabled")) {
            event.preventDefault();
            toast("Media AI ZIP is not configured by admin yet.");
          }
        });
        downloadBtn.dataset.emZipClickBound = "1";
      }

      if (effectiveUrl) {
        downloadBtn.setAttribute("href", trackedDownloadRoute);
        downloadBtn.dataset.emAiZipSource = effectiveUrl;
        downloadBtn.setAttribute("target", "_blank");
        downloadBtn.setAttribute("rel", "noopener");
        downloadBtn.setAttribute("download", "");
        downloadBtn.classList.remove("disabled", "em-ai-zip-btn-disabled");
        downloadBtn.removeAttribute("aria-disabled");
        downloadBtn.setAttribute("title", "Download the Media AI ZIP package");
      } else {
        downloadBtn.setAttribute("href", "#");
        delete downloadBtn.dataset.emAiZipSource;
        downloadBtn.removeAttribute("target");
        downloadBtn.removeAttribute("rel");
        downloadBtn.removeAttribute("download");
        downloadBtn.classList.add("disabled", "em-ai-zip-btn-disabled");
        downloadBtn.setAttribute("aria-disabled", "true");
        downloadBtn.setAttribute(
          "title",
          "Admin has not configured a ZIP file yet",
        );
      }

      return downloadBtn;
    }

    function ensureAiHelpUi(downloadBtn) {
      if (!downloadBtn) {
        return;
      }

      const settingsModal = document.getElementById("settingsModal");
      if (!settingsModal) {
        return;
      }

      let helpBtn = settingsModal.querySelector(
        '[data-em-action="toggle-ai-help"]',
      );
      if (!helpBtn) {
        helpBtn = document.createElement("button");
        helpBtn.className = "btn em-ai-help-btn";
        helpBtn.setAttribute("type", "button");
        helpBtn.dataset.emAction = "toggle-ai-help";
        helpBtn.setAttribute("aria-label", "Open help images");
        helpBtn.innerHTML = `
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="12" cy="12" r="10"></circle>
                        <line x1="12" y1="16" x2="12" y2="12"></line>
                        <line x1="12" y1="8" x2="12.01" y2="8"></line>
                    </svg>
                `;
        downloadBtn.after(helpBtn);
      }

      let helpModal = document.getElementById("emAiHelpModal");
      if (!helpModal) {
        helpModal = document.createElement("div");
        helpModal.className = "modal";
        helpModal.id = "emAiHelpModal";
        helpModal.innerHTML = `
                    <div class="modal-content em-ai-help-modal-content">
                        <div class="modal-title em-modal-title-cyan">Help Gallery</div>
                        <div class="em-ai-help-gallery" id="emAiHelpGallery"></div>
                        <div class="modal-buttons em-modal-footer">
                            <button class="btn" data-em-action="toggle-ai-help">Close</button>
                        </div>
                    </div>
                `;
        settingsModal.parentNode?.insertBefore(
          helpModal,
          settingsModal.nextSibling,
        );
      }

      const images = parseAiHelpImagesConfig();
      const gallery = helpModal.querySelector("#emAiHelpGallery");
      if (gallery) {
        if (images.length > 0) {
          gallery.innerHTML = images
            .map((imagePath, index) => {
              const safeSrc = escapeHtml(imagePath);
              return `<figure class="em-ai-help-gallery-item"><img src="${safeSrc}" alt="Help image ${index + 1}" loading="lazy"></figure>`;
            })
            .join("");
        } else {
          gallery.innerHTML =
            '<div class="em-ai-help-empty">No help images are configured yet.</div>';
        }
      }

      if (!helpBtn.dataset.emHelpClickBound) {
        helpBtn.addEventListener("click", (event) => {
          if (helpBtn.classList.contains("em-ai-help-btn-disabled")) {
            event.preventDefault();
            toast("No help images configured yet.");
          }
        });
        helpBtn.dataset.emHelpClickBound = "1";
      }

      if (images.length > 0) {
        helpBtn.classList.remove("disabled", "em-ai-help-btn-disabled");
        helpBtn.removeAttribute("aria-disabled");
        helpBtn.setAttribute("title", "Open help images");
      } else {
        helpBtn.classList.add("disabled", "em-ai-help-btn-disabled");
        helpBtn.setAttribute("aria-disabled", "true");
        helpBtn.setAttribute(
          "title",
          "Admin has not configured help images yet",
        );
      }
    }

    ensureAboutUi();
    const aiZipDownloadBtn = ensureAiZipDownloadButton();
    ensureAiHelpUi(aiZipDownloadBtn);

    function syncNoScroll() {
      const settingsOpen = document
        .getElementById("settingsModal")
        ?.classList.contains("active");
      const aboutOpen = document
        .getElementById("aboutModal")
        ?.classList.contains("active");
      const helpOpen = document
        .getElementById("emAiHelpModal")
        ?.classList.contains("active");
      const lightboxOpen = document
        .getElementById("lightbox")
        ?.classList.contains("active");
      const editorOpen = document
        .getElementById("editorModal")
        ?.classList.contains("active");
      const liveOpen = !document
        .getElementById("liveOverlay")
        ?.classList.contains("hidden");
      document.body.classList.toggle(
        "em-no-scroll",
        settingsOpen ||
          aboutOpen ||
          helpOpen ||
          lightboxOpen ||
          editorOpen ||
          liveOpen,
      );
    }

    function log(msg, type = "info") {
      const panel = document.getElementById("logContent");
      if (panel) {
        const entry = document.createElement("div");
        entry.className = `log-entry ${type}`;
        entry.textContent = `[${new Date().toLocaleTimeString()}] ${msg}`;
        panel.appendChild(entry);
        panel.scrollTop = panel.scrollHeight;
      }
      if (type === "error") console.error(msg);
      else if (type === "warn") console.warn(msg);
      else console.log(msg);
    }

    function toggleLogs() {
      const p = document.getElementById("logPanel");
      if (!p) return;
      p.classList.toggle("visible");
      if (p.classList.contains("visible")) {
        const c = document.getElementById("logContent");
        if (c) c.scrollTop = c.scrollHeight;
      }
    }

    function clearLogs() {
      const content = document.getElementById("logContent");
      if (content) content.innerHTML = "";
    }

    function updateFailedBtn() {
      const btn = document.getElementById("failedBtnSettings");
      if (!btn) return;

      btn.classList.remove("hidden");

      if (failedFiles.length > 0) {
        btn.textContent = `⚠️ Errors (${failedFiles.length})`;
        btn.classList.add("em-failed-btn-has");
        btn.classList.remove("em-failed-btn-none");
      } else {
        btn.textContent = "✓ No Errors";
        btn.classList.add("em-failed-btn-none");
        btn.classList.remove("em-failed-btn-has");
      }
    }

    function toggleFailedPanel() {
      let panel = document.getElementById("failedPanel");
      if (!panel) {
        panel = document.createElement("div");
        panel.id = "failedPanel";
        panel.className = "failed-panel hidden";
        panel.innerHTML = `
                    <div class="failed-header">
                        <span class="failed-title">File Operations Log</span>
                        <button class="btn em-failed-close" data-em-action="failed-close">✕ Close</button>
                    </div>
                    <div class="failed-list" id="failedList"></div>
                    <div class="em-failed-footer">
                        <button class="btn em-failed-clear" data-em-action="failed-clear">Clear List</button>
                    </div>
                `;
        document.body.appendChild(panel);
      }

      const isHidden = panel.classList.contains("hidden");
      panel.classList.toggle("hidden");

      if (isHidden) renderFailedFiles();
    }

    function clearFailedList() {
      failedFiles = [];
      updateFailedBtn();
      toggleFailedPanel();
    }

    function renderFailedFiles() {
      const list = document.getElementById("failedList");
      if (!list) return;

      if (failedFiles.length === 0) {
        list.innerHTML =
          '<div class="em-failed-empty">No errors recorded.</div>';
        return;
      }

      list.innerHTML = failedFiles
        .map(
          (f) => `
                <div class="failed-item">
                    <div class="failed-name">${escapeHtml(f.name)}</div>
                    <div class="failed-path">📂 .../${escapeHtml(f.path)}</div>
                    <div class="failed-reason">⚠️ ${escapeHtml(f.reason)}</div>
                </div>
            `,
        )
        .join("");
    }

    globalThis.addEventListener("error", (event) => {
      const msg = event?.message || "Unknown error";
      const line = event?.lineno || 0;
      log(`Global Error: ${msg} (${line})`, "error");
    });

    const IMAGE_EXTS = new Set([
      ".jpg",
      ".jpeg",
      ".png",
      ".gif",
      ".webp",
      ".bmp",
      ".svg",
      ".avif",
    ]);
    const VIDEO_EXTS = new Set([".mp4", ".webm", ".m4v"]);

    let mediaItems = [];
    let filteredItems = [];
    let searchQuery = "";
    let typeFilter = "all";
    let tagRefFilters = new Set();
    let tagStrictMode = false;
    let currentItem = null;
    let similarLimit = 6;
    let watchedDirs = [];
    let fileWatchInterval = null;
    let editingTags = [];
    let isSyncing = false;
    let syncCheckIndex = 0;
    let pendingFolders = [];
    let failedFiles = [];
    const aiServerCandidates = ["https://localhost:8443"];
    let aiServerActive = aiServerCandidates[0];
    let aiEnabled = false;
    let aiCheckInProgress = false;
    let aiTrainingMaxSamples = 100;
    let aiTrainingActive = false;
    let aiTrainingPollTimer = null;

    let cardHeight = 220;
    let cardMinWidth = 280;
    let tagFontSize = 13;
    const BUFFER_ROWS = 3;
    let lastScrollTop = 0;
    let scrollTimeout = null;

    try {
      const storedCardSize = Number.parseInt(
        localStorage.getItem("em_card_size") || "",
        10,
      );
      if (!Number.isNaN(storedCardSize)) {
        cardMinWidth = storedCardSize;
        cardHeight = Math.round(cardMinWidth * 0.78);
        document.documentElement.style.setProperty(
          "--card-min-width",
          `${cardMinWidth}px`,
        );
        document.documentElement.style.setProperty(
          "--card-height",
          `${cardHeight}px`,
        );
      }
      const storedTagSize = Number.parseInt(
        localStorage.getItem("em_tag_size") || "",
        10,
      );
      if (!Number.isNaN(storedTagSize)) {
        tagFontSize = storedTagSize;
        const paddingY = Math.max(4, Math.round(tagFontSize * 0.45));
        const paddingX = Math.max(8, Math.round(tagFontSize * 0.95));
        document.documentElement.style.setProperty(
          "--tag-font-size",
          `${tagFontSize}px`,
        );
        document.documentElement.style.setProperty(
          "--tag-padding-y",
          `${paddingY}px`,
        );
        document.documentElement.style.setProperty(
          "--tag-padding-x",
          `${paddingX}px`,
        );
      }
    } catch {}

    const DB_NAME = "media_wall_v4";
    const DIRS_STORE = "dirs";
    let dbPromise = null;

    function toast(msg) {
      const t = document.getElementById("toast");
      if (!t) return;
      t.textContent = msg;
      t.classList.add("show");
      setTimeout(() => t.classList.remove("show"), 2000);
    }

    function openDb() {
      if (dbPromise) return dbPromise;
      dbPromise = new Promise((resolve, reject) => {
        log(`Opening DB: ${DB_NAME}...`);
        const req = indexedDB.open(DB_NAME, 1);
        req.onupgradeneeded = onDbUpgradeNeeded.bind(
          null,
          req,
          DIRS_STORE,
          log,
        );
        req.onsuccess = onDbOpenSuccess.bind(null, req, log, resolve);
        req.onerror = onDbOpenError.bind(null, req, log, reject);
      });
      return dbPromise;
    }

    async function saveDirHandle(handle) {
      try {
        const db = await openDb();
        const tx = db.transaction(DIRS_STORE, "readwrite");
        tx.objectStore(DIRS_STORE).put({ name: handle.name, handle });
      } catch {}
    }

    async function loadDirHandles() {
      try {
        const db = await openDb();
        return new Promise((resolve) => {
          const tx = db.transaction(DIRS_STORE, "readonly");
          const req = tx.objectStore(DIRS_STORE).getAll();
          req.onsuccess = resolveRequestResult.bind(null, req, resolve);
          req.onerror = resolveEmptyResult.bind(null, resolve);
        });
      } catch (e) {
        log("Load DB Exception: " + e, "error");
        return [];
      }
    }

    async function selectFolder() {
      if (!globalThis.showDirectoryPicker) {
        const isSecure = globalThis.isSecureContext;
        let msg = "Your browser does not support the File System Access API.";

        if (isSecure) {
          msg += "\nPlease use Chrome, Edge, or Opera on Desktop.";
          msg += "\n(Firefox and Safari do not support this standard yet)";
        } else {
          msg += "\n\n⚠️ ISSUE: You are accessing this site via HTTP.";
          msg +=
            "\nThis feature requires a Secure Context (HTTPS) or localhost.";
        }

        alert(msg);
        toast("Feature not supported in this environment");
        return;
      }

      try {
        const handle = await globalThis.showDirectoryPicker({
          mode: "readwrite",
        });
        if (!watchedDirs.some((d) => d.name === handle.name)) {
          watchedDirs.push(handle);
          await saveDirHandle(handle);
        }
        await scanAllFolders();
        startFileWatcher();
        renderTagsBar();
        toast(`Watching: ${handle.name}`);
      } catch (err) {
        if (err.name === "AbortError") return;

        console.error("Folder selection failed:", err);

        let errorMsg = `Error: ${err.message}\n(${err.name})`;
        if (err.name === "SecurityError") {
          errorMsg += "\n\nSecurity Restriction Detected!";
          errorMsg += "\n- Ensure you are on HTTPS";
          errorMsg += "\n- If inside an IFrame/Embed, this API is blocked.";
          errorMsg += "\n- Check browser privacy settings.";
        }

        alert(errorMsg);
        toast("Failed to open folder picker");
      }
    }

    async function scanAllFolders() {
      failedFiles = [];
      updateFailedBtn();
      const before = mediaItems.length;
      for (const dir of watchedDirs) {
        try {
          await scanDirectory(dir, "");
        } catch (e) {
          console.error("Scan folder failed:", e);
        }
      }
      applyFiltersAndSort();
      const content = document.getElementById("content");
      if (content?.classList.contains("hidden")) {
        content.classList.remove("hidden");
      }
      renderGrid();
      const added = mediaItems.length - before;
      if (added > 0) toast(`Loaded ${mediaItems.length} files`);
      if (mediaItems.length === 0 && failedFiles.length > 0) {
        toast("No media found (check Errors)");
      }
    }

    async function scanDirectory(dirHandle, path) {
      try {
        if (path === "") {
          try {
            await dirHandle.requestPermission({ mode: "read" });
          } catch {}
        }

        for await (const entry of dirHandle.values()) {
          if (entry.kind === "file") {
            const file = await entry.getFile();
            addMediaFile(file, entry, path, dirHandle);
          } else if (entry.kind === "directory") {
            await scanDirectory(
              entry,
              path ? `${path}/${entry.name}` : entry.name,
            );
          }
        }
      } catch (e) {
        failedFiles.push({
          name: dirHandle.name,
          path: path,
          reason: "Directory Access Denied",
        });
        updateFailedBtn();
        console.warn("Scan dir error:", e);
      }
    }

    function addMediaFile(file, fileHandle, path, parentHandle) {
      const ext = "." + file.name.split(".").pop().toLowerCase();
      const isImage = IMAGE_EXTS.has(ext);
      const isVideo = VIDEO_EXTS.has(ext);
      if (!isImage && !isVideo) {
        if (ext === ".mp3" || ext === ".log") return;

        failedFiles.push({
          name: file.name,
          path: (path ? path + "/" : "") + file.name,
          reason: `Unsupported extension (${ext})`,
        });
        updateFailedBtn();
        return;
      }

      const id = file.name + file.size + file.lastModified;
      const existing = mediaItems.find((m) => m.id === id);
      if (existing) {
        existing.fileHandle = fileHandle;
        existing.file = file;
        existing.parentHandle = parentHandle;
        return;
      }

      const { baseName, tags } = parseTagsFromFilename(file.name);

      mediaItems.push({
        id,
        file,
        fileHandle,
        name: file.name,
        path,
        isVideo,
        size: file.size,
        lastModified: file.lastModified,
        url: URL.createObjectURL(file),
        thumbUrl: null,
        gridThumbUrl: null,
        baseName,
        tags,
        ext,
        parentHandle,
      });
    }

    function startFileWatcher() {
      if (fileWatchInterval) return;
    }

    function stopFileWatcher() {
      if (fileWatchInterval) {
        clearInterval(fileWatchInterval);
        fileWatchInterval = null;
      }
    }

    async function manualRefresh() {
      await syncFolders(true);
    }

    function getSyncRange(fullScan) {
      if (fullScan) {
        return { startIdx: 0, endIdx: mediaItems.length };
      }

      const batchSize = 50;
      const startIdx = syncCheckIndex;
      const endIdx = Math.min(startIdx + batchSize, mediaItems.length);
      return { startIdx, endIdx };
    }

    async function collectRemovedIds(startIdx, endIdx) {
      const toRemove = [];

      for (let i = startIdx; i < endIdx; i++) {
        const item = mediaItems[i];
        if (!item?.fileHandle) {
          continue;
        }

        try {
          await item.fileHandle.getFile();
        } catch {
          toRemove.push(item.id);
        }
      }

      return toRemove;
    }

    function updateSyncCursor(fullScan, endIdx) {
      if (fullScan) {
        return;
      }

      syncCheckIndex = endIdx >= mediaItems.length ? 0 : endIdx;
    }

    function removeMediaByIds(ids) {
      let removed = 0;

      for (const id of ids) {
        const idx = mediaItems.findIndex((m) => m.id === id);
        if (idx === -1) {
          continue;
        }

        URL.revokeObjectURL(mediaItems[idx].url);
        mediaItems.splice(idx, 1);
        removed++;
      }

      return removed;
    }

    async function scanWatchedDirsForNew() {
      let added = 0;

      for (const dir of watchedDirs) {
        try {
          added += await scanForNew(dir, "");
        } catch (error) {
          console.warn("Sync scan failed:", error);
        }
      }

      return added;
    }

    function refreshAfterSync(added, removed, toRemove, fullScan) {
      if (removed > 0 || added > 0) {
        renderTagsBar();
        applyFiltersAndSort();
        renderGrid();

        if (currentItem && !detailView?.classList.contains("hidden")) {
          if (toRemove.includes(currentItem.id)) {
            closeDetailNow();
          } else {
            renderDetailTags(currentItem);
            renderSimilarGrid(currentItem);
          }
        }

        const parts = [];
        if (added > 0) parts.push(`+${added}`);
        if (removed > 0) parts.push(`-${removed}`);
        toast(parts.join(" "));
        return;
      }

      if (fullScan) {
        toast("Already up to date");
      }
    }

    async function syncFolders(fullScan = false) {
      if (watchedDirs.length === 0 || isSyncing) return;
      isSyncing = true;
      if (fullScan) toast("Syncing...");

      const { startIdx, endIdx } = getSyncRange(fullScan);
      const toRemove = await collectRemovedIds(startIdx, endIdx);
      updateSyncCursor(fullScan, endIdx);

      const removed = removeMediaByIds(toRemove);
      const added = await scanWatchedDirsForNew();

      refreshAfterSync(added, removed, toRemove, fullScan);
      isSyncing = false;
    }

    async function scanForNew(dirHandle, path) {
      let added = 0;
      try {
        for await (const entry of dirHandle.values()) {
          if (entry.kind === "file") {
            try {
              const file = await entry.getFile();
              const id = file.name + file.size + file.lastModified;
              if (!mediaItems.some((m) => m.id === id)) {
                addMediaFile(file, entry, path, dirHandle);
                added++;
              }
            } catch (e) {
              console.warn("Sync read error:", e);
            }
          } else if (entry.kind === "directory") {
            added += await scanForNew(
              entry,
              path ? `${path}/${entry.name}` : entry.name,
            );
          }
        }
      } catch (error) {
        console.warn("Directory sync traversal failed:", error);
      }
      return added;
    }

    async function restoreFolders() {
      // NOSONAR - restore flow intentionally grouped for permission UX
      log("Restoring folders...");
      const stored = await loadDirHandles();

      if (stored.length === 0) {
        log("No stored handles found.");
        setReconnectButtonHidden(true);
        return;
      }

      log(`Found ${stored.length} stored handles.`);

      pendingFolders = [];
      watchedDirs = [];

      for (const entry of stored) {
        try {
          const classified = await classifyStoredHandle(entry);
          if (classified.kind === "granted" && classified.handle) {
            watchedDirs.push(classified.handle);
          } else if (classified.kind === "pending" && classified.handle) {
            pendingFolders.push(classified.handle);
          }
        } catch (e) {
          console.warn("Restore folder failed:", e);
        }
      }

      if (pendingFolders.length > 0) {
        setReconnectButtonHidden(false);
        toast(`Click Reconnect to restore ${pendingFolders.length} folder(s)`);
        log(`UI: Reconnect button shown for ${pendingFolders.length} folders`);
      } else {
        setReconnectButtonHidden(true);
        log("UI: No pending folders, Reconnect button hidden");
      }

      if (watchedDirs.length > 0) {
        await scanAllFolders();
        startFileWatcher();
        renderTagsBar();
        toast(`Watching ${watchedDirs.length} folder(s)`);
      }
    }

    async function reconnectFolders() {
      log("User initiated reconnect...");
      if (pendingFolders.length === 0) return;

      let connected = 0;
      const stillPending = [];

      for (const handle of pendingFolders) {
        try {
          const perm = await handle.requestPermission({ mode: "read" });
          if (perm === "granted") {
            watchedDirs.push(handle);
            connected++;
          } else {
            stillPending.push(handle);
          }
        } catch (e) {
          console.warn("Reconnect permission failed:", e);
          stillPending.push(handle);
        }
      }

      pendingFolders = stillPending;

      const reconnectBtn = document.getElementById("reconnectBtn");
      if (pendingFolders.length === 0) {
        if (reconnectBtn) reconnectBtn.classList.add("hidden");
        log("All pending folders reconnected");
      } else {
        log(`${pendingFolders.length} folders still pending`);
      }

      if (connected > 0) {
        await scanAllFolders();
        startFileWatcher();
        renderTagsBar();
        toast(`Reconnected ${connected} folder(s)`);
      } else if (stillPending.length > 0) {
        toast("Permission denied - try again");
      }
    }

    async function clearAll() {
      log("Starting FULL CLEAR...");
      stopFileWatcher();
      watchedDirs = [];
      pendingFolders = [];

      mediaItems.forEach((m) => URL.revokeObjectURL(m.url));
      mediaItems = [];
      filteredItems = [];

      try {
        if (dbPromise) {
          const db = await dbPromise;
          db.close();
        }

        const dbsToDelete = [
          DB_NAME,
          "media_wall_v3",
          "media_wall_v2",
          "media_wall",
          "dirs",
        ];

        if (indexedDB.databases) {
          const dbs = await indexedDB.databases();
          dbs.forEach((db) => {
            if (db?.name) dbsToDelete.push(db.name);
          });
        }

        log(`Attempting to delete: ${dbsToDelete.join(", ")}`);

        for (const name of dbsToDelete) {
          try {
            indexedDB.deleteDatabase(name);
          } catch {}
        }
      } catch (e) {
        log("Error clearing DB: " + e, "error");
      }

      try {
        log("Clearing LocalStorage...");
        localStorage.clear();
      } catch {}
      try {
        sessionStorage.clear();
      } catch {}

      try {
        if ("caches" in globalThis) {
          const names = await caches.keys();
          await Promise.all(names.map((n) => caches.delete(n)));
        }
      } catch {}

      const btn = document.getElementById("reconnectBtn");
      if (btn) btn.classList.add("hidden");

      log("Clear complete. Reloading in 1s...");
      toast("Fully cleared - reloading...");

      setTimeout(() => {
        location.reload();
      }, 1000);
    }

    function filterMedia(query) {
      searchQuery = query.toLowerCase();
      applyFiltersAndSort();
      renderGrid();
    }

    function setTypeFilter(mode) {
      typeFilter = mode;
      renderTagsBar();
      applyFiltersAndSort();
      renderGrid();
    }

    function toggleTagFilter(tag) {
      if (tag) {
        if (tagRefFilters.has(tag)) {
          tagRefFilters.delete(tag);
        } else {
          tagRefFilters.add(tag);
        }
      } else {
        tagRefFilters.clear();
      }
      renderTagsBar();
      applyFiltersAndSort();
      renderGrid();
    }

    function setTagFilter(tag) {
      tagRefFilters.clear();
      if (tag) {
        tagRefFilters.add(tag);
      }
      renderTagsBar();
      applyFiltersAndSort();
      renderGrid();
    }

    function setAiEnabledState(enabled) {
      aiEnabled = enabled;
      try {
        localStorage.setItem("em_ai_enabled", enabled ? "1" : "0");
      } catch {}
      const aiToggle = document.getElementById("settingAiEnabled");
      if (aiToggle) aiToggle.checked = enabled;
      const status = document.getElementById("aiStatusText");
      if (status) {
        if (enabled) {
          const protocol = aiServerActive
            .replace("https://", "")
            .replace("http://", "");
          status.textContent = `AI status: Connected (${protocol})`;
        } else {
          status.textContent = "AI status: Off";
        }
      }
      const trainingSection = document.getElementById("aiTrainingSection");
      if (trainingSection) trainingSection.classList.toggle("hidden", !enabled);
      const btnAutoTag = document.getElementById("btnAutoTag");
      if (btnAutoTag) {
        const shouldDisable =
          !enabled || currentItem?.isVideo || aiTrainingActive;
        const shouldHide = aiTrainingActive;
        btnAutoTag.disabled = shouldDisable;
        btnAutoTag.classList.toggle("disabled", shouldDisable);
        btnAutoTag.classList.toggle("hidden", shouldHide);
      }
      const btnTrain = document.getElementById("btnTrainAi");
      if (btnTrain) {
        btnTrain.disabled = !enabled || aiTrainingActive;
        btnTrain.classList.toggle("disabled", !enabled || aiTrainingActive);
      }
      if (enabled) {
        fetchServerBatchSize();
      }
    }

    function updateAiMaxSamples(value) {
      const parsed = Number.parseInt(value, 10);
      aiTrainingMaxSamples = Number.isNaN(parsed)
        ? 100
        : Math.max(1, Math.min(parsed, 10000));
      const input = document.getElementById("settingAiMaxSamples");
      if (input) input.value = aiTrainingMaxSamples;
      try {
        localStorage.setItem("em_ai_max_samples", String(aiTrainingMaxSamples));
      } catch {}
    }

    function updateTrainingUi(trained, total, losses, statusText) {
      const status = document.getElementById("aiTrainingStatus");
      const progressText = document.getElementById("aiTrainingProgressText");
      const lossEmbed = document.getElementById("aiTrainingLossEmbed");
      const lossClassifier = document.getElementById(
        "aiTrainingLossClassifier",
      );
      const lossClip = document.getElementById("aiTrainingLossClip");
      const lossVlm = document.getElementById("aiTrainingLossVlm");
      const bar = document.getElementById("aiTrainingBar");
      const btnTrain = document.getElementById("btnTrainAi");
      const btnCancel = document.getElementById("btnCancelTraining");
      const safeTotal = total || 0;
      const safeTrained = trained || 0;
      const pct =
        safeTotal > 0
          ? Math.min(100, Math.round((safeTrained / safeTotal) * 100))
          : 0;
      if (status) status.textContent = statusText || "Training status: Idle";
      if (progressText)
        progressText.textContent = `${safeTrained}/${safeTotal}`;

      const formatLoss = (val) =>
        val !== null && val !== undefined ? val.toFixed(4) : "-";
      if (lossEmbed)
        lossEmbed.textContent = `Embed: ${formatLoss(losses?.embed)}`;
      if (lossClassifier)
        lossClassifier.textContent = `Class: ${formatLoss(losses?.classifier)}`;
      if (lossClip) lossClip.textContent = `CLIP: ${formatLoss(losses?.clip)}`;
      if (lossVlm) {
        const hasVlm = losses?.vlm !== null && losses?.vlm !== undefined;
        lossVlm.textContent = `VLM: ${formatLoss(losses?.vlm)}`;
        lossVlm.classList.toggle("hidden", !hasVlm);
      }

      if (bar) bar.style.width = `${pct}%`;
      const isTraining = aiTrainingActive && safeTrained < safeTotal;
      if (btnCancel) btnCancel.classList.toggle("hidden", !isTraining);
      if (btnTrain) {
        btnTrain.disabled = !aiEnabled || aiTrainingActive;
        btnTrain.classList.toggle("disabled", !aiEnabled || aiTrainingActive);
      }
    }

    async function pollTrainingStatus() {
      try {
        const resp = await fetch(`${aiServerActive}/status`);
        if (!resp.ok) throw new Error("Status failed");
        const data = await resp.json();
        const training = data.training || {};
        const trained = training.trained || 0;
        const total = training.total || 0;
        const losses = {
          embed: training.loss_embed,
          classifier: training.loss_classifier,
          clip: training.loss_clip,
          vlm: training.loss_vlm,
        };
        const active = training.active === true;
        const statusText = active
          ? `Training status: Running (${trained}/${total})`
          : "Training status: Idle";
        updateTrainingUi(trained, total, losses, statusText);
        if (!active) {
          aiTrainingActive = false;
          stopTrainingPoll();
          setAiEnabledState(aiEnabled);
        }
      } catch {
        updateTrainingUi(0, 0, null, "Training status: Failed to fetch");
      }
    }

    async function detectAiServerStatus() {
      const connected = await attemptAiConnection();
      if (!connected) {
        setAiEnabledState(false);
        updateTrainingUi(0, 0, null, "Training status: Idle");
        return;
      }

      try {
        const resp = await fetch(`${aiServerActive}/status`);
        if (!resp.ok) throw new Error("Status failed");
        const data = await resp.json();
        const training = data.training || {};
        const trained = training.trained || 0;
        const total = training.total || 0;
        const losses = {
          embed: training.loss_embed,
          classifier: training.loss_classifier,
          clip: training.loss_clip,
          vlm: training.loss_vlm,
        };
        const active = training.active === true;

        if (active) {
          aiTrainingActive = true;
          setAiEnabledState(true);
          updateTrainingUi(
            trained,
            total,
            losses,
            `Training status: Running (${trained}/${total})`,
          );
          startTrainingPoll();
        } else if (aiEnabled) {
          updateTrainingUi(trained, total, losses, "Training status: Idle");
        } else {
          updateTrainingUi(trained, total, losses, "Training status: Idle");
        }
      } catch {
        updateTrainingUi(0, 0, null, "Training status: Failed to fetch");
      }
    }

    function startTrainingPoll() {
      stopTrainingPoll();
      aiTrainingPollTimer = setInterval(pollTrainingStatus, 1000);
    }

    function stopTrainingPoll() {
      if (aiTrainingPollTimer) {
        clearInterval(aiTrainingPollTimer);
        aiTrainingPollTimer = null;
      }
    }

    async function cancelAiTraining() {
      if (!aiEnabled || !aiTrainingActive) return;
      try {
        const resp = await fetch(`${aiServerActive}/ai-train-cancel`, {
          method: "POST",
        });
        if (resp.ok) {
          aiTrainingActive = false;
          updateTrainingUi(0, 0, null, "Training status: Cancelling...");
        } else {
          updateTrainingUi(0, 0, null, "Training status: Failed to cancel");
        }
      } catch (e) {
        console.error("Cancel training failed:", e);
        toast("Failed to cancel training");
      }
    }

    async function fetchServerBatchSize() {
      const batchSizeText = document.getElementById("aiBatchSizeText");
      if (!batchSizeText) return;
      try {
        const resp = await fetch(`${aiServerActive}/status`);
        if (!resp.ok) throw new Error("Status failed");
        const data = await resp.json();
        const batchSize = data.batch_size || data.batchSize || "Unknown";
        batchSizeText.textContent = `Batch size: ${batchSize} (configured on server)`;
      } catch (e) {
        console.warn("Failed to fetch server batch size:", e);
        batchSizeText.textContent = "Batch size: Unable to fetch";
      }
    }

    function getTaggedTrainingItems() {
      const tagged = mediaItems.filter(
        (item) =>
          !item.isVideo && Array.isArray(item.tags) && item.tags.length > 0,
      );
      tagged.sort((a, b) => b.lastModified - a.lastModified);
      return tagged.slice(0, aiTrainingMaxSamples);
    }

    async function trainAiFromTaggedFiles() {
      if (!aiEnabled) {
        toast("AI auto tag is disabled");
        return;
      }
      if (aiTrainingActive) return;
      const items = getTaggedTrainingItems();
      if (items.length === 0) {
        toast("No tagged images to train");
        return;
      }

      aiTrainingActive = true;
      setAiEnabledState(aiEnabled);
      updateTrainingUi(0, items.length, null, "Training status: Uploading...");

      try {
        const formData = new FormData();
        const tagsArray = [];
        for (const item of items) {
          const file = await item.fileHandle.getFile();
          formData.append("files", file, file.name);
          tagsArray.push({ filename: file.name, tags: item.tags || [] });
        }
        formData.append("tags_json", JSON.stringify(tagsArray));

        const resp = await fetch(`${aiServerActive}/ai-train-batch`, {
          method: "POST",
          body: formData,
        });
        if (!resp.ok) throw new Error("Train failed");
        updateTrainingUi(
          0,
          items.length,
          null,
          "Training status: Running (background)",
        );
        startTrainingPoll();
      } catch (e) {
        console.error("AI training failed:", e);
        updateTrainingUi(0, items.length, null, "Training status: Failed");
        toast("AI training failed");
        aiTrainingActive = false;
        setAiEnabledState(aiEnabled);
      }
    }

    async function attemptAiConnection() {
      try {
        for (const baseUrl of aiServerCandidates) {
          try {
            const resp = await fetch(`${baseUrl}/status`);
            if (!resp.ok) continue;
            aiServerActive = baseUrl;
            return true;
          } catch {
            continue;
          }
        }
        return false;
      } catch {
        return false;
      }
    }

    async function toggleAiEnabled(enabled) {
      if (!enabled) {
        setAiEnabledState(false);
        toast("AI auto tag disabled");
        return;
      }

      if (aiCheckInProgress) return;
      aiCheckInProgress = true;
      const status = document.getElementById("aiStatusText");
      if (status) status.textContent = "AI status: Checking...";
      toast("Checking AI server...");

      let success = false;
      for (let i = 0; i < 5; i++) {
        if (status) status.textContent = `AI status: Attempt ${i + 1} of 5...`;
        success = await attemptAiConnection();
        if (success) break;
        await new Promise((r) => setTimeout(r, 1000));
      }

      if (success) {
        setAiEnabledState(true);
        if (status) status.textContent = "AI status: Connected";
        toast("AI auto tag enabled");
      } else {
        setAiEnabledState(false);
        if (status)
          status.textContent = "AI status: Connection failed (auto-disabled)";
        toast("AI server unavailable. Auto tag disabled");
      }
      aiCheckInProgress = false;
    }

    function toggleSettings() {
      const el = document.getElementById("settingsModal");
      if (!el) return;
      const isActive = el.classList.toggle("active");
      if (isActive) {
        document.getElementById("aboutModal")?.classList.remove("active");
        document.getElementById("emAiHelpModal")?.classList.remove("active");
      }
      syncNoScroll();

      if (isActive) {
        const cb = document.getElementById("settingStrictTags");
        if (cb) cb.checked = tagStrictMode;
        const cardSizeInput = document.getElementById("settingCardSize");
        const tagSizeInput = document.getElementById("settingTagSize");
        if (cardSizeInput) cardSizeInput.value = cardMinWidth;
        if (tagSizeInput) tagSizeInput.value = tagFontSize;
        const aiToggle = document.getElementById("settingAiEnabled");
        if (aiToggle) aiToggle.checked = aiEnabled;
        updateCardSize(cardMinWidth, false);
        updateTagSize(tagFontSize, false);
      }
    }

    function toggleAbout() {
      const el = document.getElementById("aboutModal");
      if (!el) return;
      const isActive = el.classList.toggle("active");
      if (isActive) {
        document.getElementById("settingsModal")?.classList.remove("active");
        document.getElementById("emAiHelpModal")?.classList.remove("active");
      }
      syncNoScroll();
    }

    function toggleAiHelp() {
      const el = document.getElementById("emAiHelpModal");
      if (!el) return;
      const trigger = document.querySelector(
        '[data-em-action="toggle-ai-help"]',
      );
      if (trigger?.classList.contains("em-ai-help-btn-disabled")) {
        return;
      }

      const isActive = el.classList.toggle("active");
      if (isActive) {
        document.getElementById("settingsModal")?.classList.remove("active");
        document.getElementById("aboutModal")?.classList.remove("active");
      }
      syncNoScroll();
    }

    function toggleStrictTags() {
      const cb = document.getElementById("settingStrictTags");
      if (!cb) return;
      tagStrictMode = cb.checked;
      applyFiltersAndSort();
      renderGrid();
    }

    function updateCardSize(value, persist = true) {
      const size = Math.max(
        220,
        Math.min(420, Number.parseInt(value, 10) || 280),
      );
      cardMinWidth = size;
      cardHeight = Math.round(size * 0.78);
      document.documentElement.style.setProperty(
        "--card-min-width",
        `${cardMinWidth}px`,
      );
      document.documentElement.style.setProperty(
        "--card-height",
        `${cardHeight}px`,
      );
      const label = document.getElementById("cardSizeValue");
      if (label) label.textContent = `${cardMinWidth}px`;
      if (persist) {
        try {
          localStorage.setItem("em_card_size", String(cardMinWidth));
        } catch {}
      }
      renderGrid();
    }

    function updateTagSize(value, persist = true) {
      const size = Math.max(10, Math.min(20, Number.parseInt(value, 10) || 13));
      tagFontSize = size;
      const paddingY = Math.max(4, Math.round(size * 0.45));
      const paddingX = Math.max(8, Math.round(size * 0.95));
      document.documentElement.style.setProperty(
        "--tag-font-size",
        `${tagFontSize}px`,
      );
      document.documentElement.style.setProperty(
        "--tag-padding-y",
        `${paddingY}px`,
      );
      document.documentElement.style.setProperty(
        "--tag-padding-x",
        `${paddingX}px`,
      );
      const label = document.getElementById("tagSizeValue");
      if (label) label.textContent = `${tagFontSize}px`;
      if (persist) {
        try {
          localStorage.setItem("em_tag_size", String(tagFontSize));
        } catch {}
      }
      renderTagsBar();
    }

    function toggleLogsFromSettings() {
      toggleSettings();
      toggleLogs();
    }

    function toggleFailedPanelFromSettings() {
      toggleSettings();
      toggleFailedPanel();
    }

    function renderTagsBar() {
      const bar = document.getElementById("tagsBar");
      if (!bar) return;
      const allTags = new Set();
      for (const item of mediaItems) {
        if (typeFilter === "images" && item.isVideo) continue;
        if (typeFilter === "videos" && !item.isVideo) continue;
        if (typeFilter === "untagged" && item.tags && item.tags.length > 0)
          continue;

        if (item.tags)
          item.tags.forEach((t) => {
            if (t && t.trim().length > 0) allTags.add(t);
          });
      }
      const sorted = [...allTags].sort((a, b) => a.localeCompare(b));

      bar.innerHTML = `
                <span class="tags-bar-label">Tags:</span>
                <span class="tag-pill ${tagRefFilters.size === 0 ? "active" : ""}" data-tag="">All</span>
                ${sorted.map((t) => `<span class="tag-pill ${tagRefFilters.has(t) ? "active" : ""}" data-tag="${escapeHtml(t)}">${escapeHtml(t)}</span>`).join("")}
            `;
    }

    function applyFiltersAndSort() {
      const activeFilters = Array.from(tagRefFilters);

      filteredItems = mediaItems.filter((item) => {
        if (!itemMatchesSearch(item, searchQuery)) return false;
        if (!itemMatchesType(item, typeFilter)) return false;
        if (!itemMatchesTagFilters(item, activeFilters, tagStrictMode)) {
          return false;
        }
        return true;
      });
      filteredItems.sort((a, b) => b.lastModified - a.lastModified);
      updateCount();
    }

    function updateCount() {
      const count = document.getElementById("itemCount");
      if (count) count.textContent = `${filteredItems.length} items`;
    }

    function renderGrid() {
      const content = document.getElementById("content");
      if (
        !content ||
        content.classList.contains("hidden") ||
        content.offsetParent === null
      )
        return;

      const grid = document.getElementById("mediaGrid");
      if (!grid) return;

      const gridWidth = grid.clientWidth - 48;
      const cols = Math.max(1, Math.floor(gridWidth / cardMinWidth));
      const totalRows = Math.ceil(filteredItems.length / cols);

      const contentTop = content.offsetTop || 0;
      const scrollTop = globalThis.scrollY;
      const viewportHeight = globalThis.innerHeight;

      const gridScrollY = Math.max(0, scrollTop - contentTop);

      const startRow = Math.max(
        0,
        Math.floor(gridScrollY / cardHeight) - BUFFER_ROWS,
      );
      const endRow = Math.min(
        totalRows,
        Math.ceil((gridScrollY + viewportHeight) / cardHeight) + BUFFER_ROWS,
      );
      const startIdx = startRow * cols;
      const endIdx = Math.min(filteredItems.length, endRow * cols);

      grid.innerHTML = "";

      if (startRow > 0) {
        const topSpacer = document.createElement("div");
        topSpacer.className = "virtual-spacer";
        topSpacer.style.height = `${startRow * cardHeight}px`;
        grid.appendChild(topSpacer);
      }

      const fragment = document.createDocumentFragment();
      for (let i = startIdx; i < endIdx; i++) {
        const item = filteredItems[i];
        fragment.appendChild(createCard(item));
      }
      grid.appendChild(fragment);

      const bottomRows = totalRows - endRow;
      if (bottomRows > 0) {
        const bottomSpacer = document.createElement("div");
        bottomSpacer.className = "virtual-spacer";
        bottomSpacer.style.height = `${bottomRows * cardHeight}px`;
        grid.appendChild(bottomSpacer);
      }
    }

    function createCard(item) {
      const card = document.createElement("div");
      card.className = `media-card ${item.isVideo ? "video-card" : "image-card"}`;
      card.onclick = () => openDetail(item);

      if (item.isVideo) {
        card.innerHTML = `
                    <div class="thumbnail">
                        <img class="poster" src="${item.thumbUrl || item.url}" alt="">
                        <video muted preload="metadata"></video>
                        <div class="play-badge">
                            <svg viewBox="0 0 24 24" fill="currentColor"><path d="M8 5v14l11-7z"/></svg>
                            Video
                        </div>
                    </div>
                    <div class="title">${escapeHtml(item.name)}</div>
                `;

        if (!item.thumbUrl) {
          generateVideoThumbnail(item, card.querySelector(".poster"));
        }

        let hoverVideo = null;
        card.onmouseenter = () => {
          if (hoverVideo || !item.url) return;
          const video = card.querySelector("video");
          if (!video) return;
          hoverVideo = video;
          hoverVideo.src = item.url;
          hoverVideo.muted = true;
          hoverVideo.playsInline = true;
          hoverVideo.loop = true;
          hoverVideo.play().catch(noop);
        };
        card.onmouseleave = () => {
          if (!hoverVideo) return;
          hoverVideo.pause();
          hoverVideo.removeAttribute("src");
          hoverVideo.load();
          hoverVideo = null;
        };
      } else {
        const thumbSrc = item.gridThumbUrl || item.url;
        card.innerHTML = `
                    <div class="thumbnail">
                        <img src="${thumbSrc}" alt="${escapeHtml(item.name)}">
                    </div>
                    <div class="title">${escapeHtml(item.name)}</div>
                `;

        const img = card.querySelector("img");

        img.onerror = () => handleMediaError(item, "Image load failed");

        if (!item.gridThumbUrl) {
          generateImageThumbnail(item, img);
        }
      }
      return card;
    }

    function handleMediaError(item, reason) {
      const idx = mediaItems.findIndex((m) => m.id === item.id);
      if (idx !== -1) {
        mediaItems.splice(idx, 1);
        URL.revokeObjectURL(item.url);
        if (item.gridThumbUrl) URL.revokeObjectURL(item.gridThumbUrl);

        failedFiles.push({
          name: item.name,
          path: item.path || "",
          reason: reason,
        });

        updateFailedBtn();
        if (!globalThis._errorUpdateTimeout) {
          globalThis._errorUpdateTimeout = setTimeout(() => {
            globalThis._errorUpdateTimeout = null;
            applyFiltersAndSort();
            renderGrid();
          }, 200);
        }
      }
    }

    function onContentScroll() {
      if (scrollTimeout) return;
      scrollTimeout = setTimeout(() => {
        scrollTimeout = null;
        if (Math.abs(globalThis.scrollY - lastScrollTop) > cardHeight / 2) {
          lastScrollTop = globalThis.scrollY;
          renderGrid();
        }
      }, 50);
    }

    function generateImageThumbnail(item, imgEl) {
      const img = new Image();
      img.onload = () => {
        try {
          const canvas = document.createElement("canvas");
          const maxW = 320;
          const scale = maxW / img.width;
          canvas.width = maxW;
          canvas.height = Math.round(img.height * scale);
          const ctx = canvas.getContext("2d");
          ctx.drawImage(img, 0, 0, canvas.width, canvas.height);
          canvas.toBlob(
            handleGeneratedImageThumbBlob.bind(null, item, imgEl),
            "image/jpeg",
            0.7,
          );
        } catch {}
      };
      img.src = item.url;
    }

    function generateVideoThumbnail(item, posterEl) {
      const video = document.createElement("video");
      video.src = item.url;
      video.muted = true;
      video.preload = "metadata";
      video.addEventListener(
        "loadeddata",
        () => {
          video.currentTime = 0.1;
        },
        { once: true },
      );
      video.addEventListener(
        "seeked",
        () => {
          try {
            const canvas = document.createElement("canvas");
            const maxW = 320;
            const scale = maxW / video.videoWidth;
            canvas.width = maxW;
            canvas.height = Math.round(video.videoHeight * scale);
            const ctx = canvas.getContext("2d");
            ctx.drawImage(video, 0, 0, canvas.width, canvas.height);
            canvas.toBlob(
              handleGeneratedVideoThumbBlob.bind(null, item, posterEl),
              "image/jpeg",
              0.7,
            );
            video.src = "";
          } catch {}
        },
        { once: true },
      );
    }

    const detailView = document.getElementById("detailView");
    const detailMedia = document.getElementById("detailMedia");
    const detailTitle = document.getElementById("detailTitle");

    function openDetail(item) {
      currentItem = item;
      similarLimit = 6;
      if (detailTitle) detailTitle.textContent = item.name;
      if (detailMedia) {
        detailMedia.innerHTML = item.isVideo
          ? `<video src="${item.url}" controls autoplay loop></video>`
          : `<img src="${item.url}" alt="">`;
      }

      const btnEdit = document.getElementById("btnEditMedia");
      if (btnEdit) btnEdit.classList.toggle("hidden", item.isVideo);
      const btnAutoTag = document.getElementById("btnAutoTag");
      if (btnAutoTag) {
        const shouldHide = item.isVideo || aiTrainingActive;
        btnAutoTag.classList.toggle("hidden", shouldHide);
        btnAutoTag.disabled = item.isVideo || !aiEnabled || aiTrainingActive;
        btnAutoTag.classList.toggle(
          "disabled",
          item.isVideo || !aiEnabled || aiTrainingActive,
        );
      }

      const content = document.getElementById("content");
      if (content) content.classList.add("hidden");
      detailView?.classList.remove("hidden");
      globalThis.scrollTo(0, 0);
      setDetailMode(true);
      renderDetailTags(item);
      renderSimilarGrid(item);
      setTimeout(fitDetailMedia, 0);
      history.pushState({ view: "detail", id: item.id }, "", "#detail");
    }

    function renderDetailTags(item) {
      const container = document.getElementById("detailTags");
      if (!container) return;
      if (!item.tags || item.tags.length === 0) {
        container.innerHTML = '<span class="text-muted">No tags</span>';
        return;
      }
      container.innerHTML = item.tags
        .map(
          (t) =>
            `<span class="tag-pill" data-tag="${escapeHtml(t)}">${escapeHtml(t)}</span>`,
        )
        .join("");
    }

    function closeDetail() {
      if (history.state?.view === "detail") {
        history.back();
        return;
      }
      closeDetailNow();
    }

    function closeDetailNow() {
      detailView?.classList.add("hidden");
      const content = document.getElementById("content");
      if (content) content.classList.remove("hidden");
      currentItem = null;
      setDetailMode(false);
      requestAnimationFrame(() => renderGrid());
    }

    globalThis.addEventListener("popstate", () => {
      if (detailView?.classList.contains("hidden")) return;
      closeDetailNow();
    });

    function setDetailMode(isDetail) {
      document.querySelectorAll(".hide-on-detail").forEach((el) => {
        if (
          !isDetail &&
          el.id === "reconnectBtn" &&
          pendingFolders.length === 0
        )
          return;
        el.classList.toggle("hidden", isDetail);
      });
    }

    function fitDetailMedia() {
      if (detailMedia) {
        detailMedia.style.height = "";
        detailMedia.style.maxHeight = "";
      }
    }

    function navDetail(offset) {
      if (!currentItem || filteredItems.length === 0) return;
      const idx = filteredItems.findIndex((i) => i.id === currentItem.id);
      if (idx === -1) return;

      let newIdx = idx + offset;
      if (newIdx < 0) newIdx = filteredItems.length - 1;
      if (newIdx >= filteredItems.length) newIdx = 0;

      openDetail(filteredItems[newIdx]);
    }

    function renderSimilarGrid(activeItem) {
      const grid = document.getElementById("detailSimilarGrid");
      if (!grid) return;
      grid.innerHTML = "";

      let similar = filteredItems.filter((i) => i.id !== activeItem.id);

      if (activeItem.tags && activeItem.tags.length > 0) {
        similar = similar
          .map((item) => {
            const sharedTags = countSharedTags(item.tags, activeItem.tags);
            return { ...item, sharedTags };
          })
          .sort((a, b) => {
            if (b.sharedTags !== a.sharedTags)
              return b.sharedTags - a.sharedTags;
            return b.lastModified - a.lastModified;
          });
      }

      const toShow = similar.slice(0, similarLimit);
      const loadMore = document.getElementById("detailLoadMore");
      if (loadMore)
        loadMore.style.display = similar.length > similarLimit ? "" : "none";

      for (const item of toShow) {
        const card = createCard(item);
        grid.appendChild(card);
      }
    }

    function loadMoreSimilar() {
      similarLimit += 6;
      if (currentItem) renderSimilarGrid(currentItem);
    }

    async function openOriginalFile() {
      if (!currentItem?.fileHandle) return;
      try {
        const file = await currentItem.fileHandle.getFile();

        const url = URL.createObjectURL(file);
        const a = document.createElement("a");
        a.href = url;
        a.download = file.name;
        document.body.appendChild(a);
        a.click();
        a.remove();

        setTimeout(() => URL.revokeObjectURL(url), 2000);
      } catch (e) {
        console.error("Open original failed:", e);
        alert("Could not open file: " + e.message);
      }
    }

    function openViewer() {
      if (!currentItem) return;
      const lb = document.getElementById("lightbox");
      const content = document.getElementById("lightboxMedia");
      if (!lb || !content) return;
      content.innerHTML = currentItem.isVideo
        ? `<video class="lightbox-zoom" src="${currentItem.url}" controls autoplay loop></video>`
        : `<img class="lightbox-zoom" src="${currentItem.url}" alt="">`;
      const zoomEl = content.querySelector(".lightbox-zoom");
      if (zoomEl) {
        zoomEl.dataset.zoom = "1";
        zoomEl.dataset.panX = "0";
        zoomEl.dataset.panY = "0";
        zoomEl.style.transform = "";
      }
      lb.classList.add("active");
      syncNoScroll();
    }

    function closeLightbox(e) {
      if (
        e &&
        e.target !== e.currentTarget &&
        !e.target.classList.contains("lightbox-close")
      )
        return;
      const lb = document.getElementById("lightbox");
      if (!lb) return;
      const video = lb.querySelector("video");
      if (video) video.pause();
      lb.classList.remove("active");
      syncNoScroll();
      isDragging = false;
    }

    const lightboxEl = document.getElementById("lightbox");
    if (lightboxEl) {
      lightboxEl.addEventListener("wheel", (e) => {
        const zoomEl = document.querySelector(".lightbox-zoom");
        if (!zoomEl) return;
        e.preventDefault();
        const current = Number.parseFloat(zoomEl.dataset.zoom || "1");
        const delta = e.deltaY > 0 ? 0.9 : 1.1;
        const newZoom = Math.min(100, Math.max(0.5, current * delta));
        zoomEl.dataset.zoom = newZoom;
        const panX = Number.parseFloat(zoomEl.dataset.panX || "0");
        const panY = Number.parseFloat(zoomEl.dataset.panY || "0");
        zoomEl.style.transform = `translate(${panX}px, ${panY}px) scale(${newZoom})`;
      });

      lightboxEl.addEventListener("mousedown", (e) => {
        const zoomEl = document.querySelector(".lightbox-zoom");
        if (!zoomEl || e.target.classList.contains("lightbox-close")) return;

        if (e.button !== 0) return;

        const currentZoom = Number.parseFloat(zoomEl.dataset.zoom || "1");
        if (currentZoom <= 1.01) return;

        if (e.target.tagName === "VIDEO" && e.target.controls) {
          const rect = e.target.getBoundingClientRect();
          const controlsHeight = 50;
          if (e.clientY > rect.bottom - controlsHeight) return;
        }
        e.preventDefault();
        isDragging = true;
        dragStartX = e.clientX;
        dragStartY = e.clientY;
        panStartX = Number.parseFloat(zoomEl.dataset.panX || "0");
        panStartY = Number.parseFloat(zoomEl.dataset.panY || "0");
        zoomEl.style.cursor = "grabbing";
      });
    }

    let isDragging = false;
    let dragStartX = 0,
      dragStartY = 0;
    let panStartX = 0,
      panStartY = 0;

    document.addEventListener("mousemove", (e) => {
      if (!isDragging) return;
      const zoomEl = document.querySelector(".lightbox-zoom");
      if (!zoomEl) return;
      const dx = e.clientX - dragStartX;
      const dy = e.clientY - dragStartY;
      const newPanX = panStartX + dx;
      const newPanY = panStartY + dy;
      zoomEl.dataset.panX = newPanX;
      zoomEl.dataset.panY = newPanY;
      const zoom = Number.parseFloat(zoomEl.dataset.zoom || "1");
      zoomEl.style.transform = `translate(${newPanX}px, ${newPanY}px) scale(${zoom})`;
    });

    document.addEventListener("mouseup", () => {
      if (!isDragging) return;
      isDragging = false;
      const zoomEl = document.querySelector(".lightbox-zoom");
      if (zoomEl) zoomEl.style.cursor = "grab";
    });

    function openTagEditor() {
      if (!currentItem) return;
      editingTags = currentItem.tags ? [...currentItem.tags] : [];
      renderModalTags();
      renderModalSuggestions();
      document.getElementById("tagModal")?.classList.add("active");
      document.getElementById("tagInput")?.focus();
      syncNoScroll();
    }

    function closeTagEditor() {
      document.getElementById("tagModal")?.classList.remove("active");
      const input = document.getElementById("tagInput");
      if (input) input.value = "";
      editingTags = [];
      syncNoScroll();
    }

    function mergeTags(existing, incoming) {
      const normalized = new Set();
      (existing || []).forEach((t) => {
        const val = sanitizeTag(String(t || ""));
        if (val) normalized.add(val);
      });
      (incoming || []).forEach((t) => {
        const val = sanitizeTag(String(t || ""));
        if (val) normalized.add(val);
      });
      return Array.from(normalized);
    }

    let aiTagPreview = [];
    let aiTagSelected = new Set();
    let aiTagCaption = "";
    let aiTagKnown = new Set();

    function renderAiTagPreview() {
      const container = document.getElementById("aiTagPreview");
      if (!container) return;
      let captionEl = document.getElementById("aiTagCaption");
      if (!captionEl && container.parentElement) {
        captionEl = document.createElement("div");
        captionEl.id = "aiTagCaption";
        captionEl.className = "ai-caption hidden";
        container.parentElement.insertBefore(captionEl, container);
      }
      if (aiTagPreview.length === 0) {
        container.innerHTML = '<span class="no-tags">No tags suggested.</span>';
      } else {
        container.innerHTML = aiTagPreview
          .map((tag) => {
            const selected = aiTagSelected.has(tag) ? "selected" : "";
            const known = aiTagKnown.has(tag) ? "known" : "";
            return `<span class="tag-pill ai-tag-pill ${selected} ${known}" data-tag="${escapeHtml(tag)}">${escapeHtml(tag)}</span>`;
          })
          .join("");
      }
      if (captionEl) {
        if (aiTagCaption) {
          captionEl.textContent = aiTagCaption;
          captionEl.setAttribute("title", aiTagCaption);
          captionEl.classList.remove("hidden");
        } else {
          captionEl.textContent = "";
          captionEl.removeAttribute("title");
          captionEl.classList.add("hidden");
        }
      }
    }

    function openAiTagPreview(tags, caption = "", knownSet = null) {
      aiTagPreview = Array.isArray(tags) ? tags : [];
      aiTagSelected = new Set(aiTagPreview);
      aiTagCaption = typeof caption === "string" ? caption.trim() : "";
      aiTagKnown = knownSet instanceof Set ? new Set(knownSet) : new Set();
      renderAiTagPreview();
      document.getElementById("aiTagModal")?.classList.add("active");
      syncNoScroll();
    }

    function toggleAiTagSelection(tag) {
      if (!tag) return;
      if (aiTagSelected.has(tag)) {
        aiTagSelected.delete(tag);
      } else {
        aiTagSelected.add(tag);
      }
      renderAiTagPreview();
    }

    function closeAiTagPreview() {
      document.getElementById("aiTagModal")?.classList.remove("active");
      aiTagPreview = [];
      aiTagSelected = new Set();
      aiTagCaption = "";
      aiTagKnown = new Set();
      syncNoScroll();
    }

    async function applyAiTagPreview() {
      if (!currentItem?.fileHandle) {
        closeAiTagPreview();
        return;
      }
      const selectedTags = Array.from(aiTagSelected);
      if (selectedTags.length === 0) {
        toast("No tags selected");
        return;
      }
      const merged = mergeTags(currentItem.tags, selectedTags);
      if (merged.length === (currentItem.tags || []).length) {
        closeAiTagPreview();
        toast("No new tags to apply");
        return;
      }

      try {
        const newName = buildFilename(
          currentItem.baseName,
          merged,
          currentItem.ext,
        );
        if (newName !== currentItem.name) {
          await currentItem.fileHandle.move(newName);
          const newFile = await currentItem.fileHandle.getFile();
          URL.revokeObjectURL(currentItem.url);
          currentItem.file = newFile;
          currentItem.name = newFile.name;
          currentItem.tags = merged;
          currentItem.url = URL.createObjectURL(newFile);
          currentItem.id = newFile.name + newFile.size + newFile.lastModified;
        }

        if (detailTitle) detailTitle.textContent = currentItem.name;
        renderDetailTags(currentItem);
        renderTagsBar();
        applyFiltersAndSort();
        renderGrid();
        toast("AI tags applied");
      } catch (e) {
        console.error("AI tag apply failed:", e);
        toast("Failed to apply AI tags");
      } finally {
        closeAiTagPreview();
      }
    }

    async function autoTagCurrentFile() {
      if (!currentItem?.fileHandle) return;
      if (!aiEnabled) {
        toast("AI auto tag is disabled");
        return;
      }
      if (currentItem.isVideo) {
        toast("AI tagging supports images only");
        return;
      }

      const btn = document.getElementById("btnAutoTag");
      if (btn) {
        btn.disabled = true;
        btn.textContent = "🤖 Tagging...";
      }

      try {
        const aiUrl = aiServerActive;
        const file = await currentItem.fileHandle.getFile();
        const formData = new FormData();
        formData.append("file", file, file.name);
        formData.append("filename", file.name);
        const knownTags = getAllKnownTags();
        formData.append("existingTags", JSON.stringify(knownTags));
        formData.append("maxTags", "8");

        const resp = await fetch(`${aiUrl}/ai-tags`, {
          method: "POST",
          body: formData,
        });

        if (!resp.ok) throw new Error("AI tag failed");

        const data = await resp.json();
        const knownSet = new Set(knownTags);
        const rawTags = Array.isArray(data.tags)
          ? data.tags.map((t) => sanitizeTag(String(t || ""))).filter(Boolean)
          : [];
        const incoming = rawTags.filter((t) => !knownSet.has(t));
        const displayTags = incoming.length > 0 ? incoming : rawTags;
        if (displayTags.length === 0) {
          toast("No tags suggested");
          return;
        }
        if (incoming.length === 0) {
          toast("No new tags; showing existing suggestions");
        }
        openAiTagPreview(displayTags, data.caption || "", knownSet);
      } catch (e) {
        console.error("AI tagging failed:", e);
        toast("AI tagging failed");
      } finally {
        if (btn) {
          btn.disabled = false;
          btn.textContent = "🤖 AI Auto Tag";
        }
      }
    }

    function renderModalTags() {
      const container = document.getElementById("modalTags");
      if (!container) return;
      container.innerHTML = editingTags
        .map(
          (t, i) =>
            `<span class="modal-tag">${escapeHtml(t)} <span class="remove" data-remove-index="${i}">×</span></span>`,
        )
        .join("");
    }

    function renderModalSuggestions(filterText = "") {
      const container = document.getElementById("modalSuggestions");
      if (!container) return;
      const allTags = new Set();
      for (const item of mediaItems) {
        if (item.tags) item.tags.forEach((t) => allTags.add(t));
      }
      const searchLower = (filterText || "").toLowerCase().trim();
      const searchNorm = normalizeSearchText(searchLower);
      let available = [...allTags].filter((t) => !editingTags.includes(t));
      if (searchLower) {
        available = available.filter((t) => {
          const tLower = t.toLowerCase();
          if (tLower.includes(searchLower)) return true;
          if (!searchNorm) return false;
          return normalizeSearchText(tLower).includes(searchNorm);
        });
        available.sort((a, b) => {
          const aStarts = a.toLowerCase().startsWith(searchLower);
          const bStarts = b.toLowerCase().startsWith(searchLower);
          if (aStarts && !bStarts) return -1;
          if (!aStarts && bStarts) return 1;
          return a.localeCompare(b);
        });
      } else {
        available.sort((a, b) => a.localeCompare(b));
      }
      container.innerHTML = available
        .map((t) => {
          let display = escapeHtml(t);
          if (searchLower) {
            const idx = t.toLowerCase().indexOf(searchLower);
            if (idx >= 0) {
              const before = escapeHtml(t.slice(0, idx));
              const match = escapeHtml(t.slice(idx, idx + searchLower.length));
              const after = escapeHtml(t.slice(idx + searchLower.length));
              display = `${before}<strong>${match}</strong>${after}`;
            }
          }
          return `<span class="tag-pill" data-add-tag="${escapeHtml(t)}">${display}</span>`;
        })
        .join("");
    }

    function getAllKnownTags() {
      const allTags = new Set();
      for (const item of mediaItems) {
        if (item.tags)
          item.tags.forEach((t) => {
            const val = sanitizeTag(String(t || ""));
            if (val) allTags.add(val);
          });
      }
      return Array.from(allTags);
    }

    async function updateTagDb() {
      if (!aiEnabled) {
        toast("AI auto tag is disabled");
        return;
      }
      const tags = getAllKnownTags();
      if (tags.length === 0) {
        toast("No tags found to sync");
        return;
      }

      const btn = document.getElementById("btnUpdateTagDb");
      const status = document.getElementById("tagDbStatusText");
      if (btn) {
        btn.disabled = true;
        btn.classList.add("disabled");
      }
      if (status) status.textContent = "Tag DB status: Updating...";

      try {
        const canvas = document.createElement("canvas");
        canvas.width = 1;
        canvas.height = 1;
        const ctx = canvas.getContext("2d");
        ctx.fillStyle = "#000";
        ctx.fillRect(0, 0, 1, 1);
        const blob = await new Promise((resolve) =>
          canvas.toBlob(resolve, "image/png"),
        );
        if (!blob) throw new Error("Failed to generate image");

        const formData = new FormData();
        formData.append("file", blob, "tagdb_sync.png");
        formData.append("filename", "tagdb_sync.png");
        formData.append("existingTags", JSON.stringify(tags));
        formData.append("maxTags", "1");

        const resp = await fetch(`${aiServerActive}/ai-tags`, {
          method: "POST",
          body: formData,
        });

        if (!resp.ok) throw new Error("Tag DB update failed");

        if (status)
          status.textContent = `Tag DB status: Updated (${tags.length} tags)`;
        toast("Tag DB updated");
      } catch (e) {
        console.error("Tag DB update failed:", e);
        if (status) status.textContent = "Tag DB status: Failed";
        toast("Tag DB update failed");
      } finally {
        if (btn) {
          btn.disabled = false;
          btn.classList.remove("disabled");
        }
      }
    }

    function addEditTag(tag) {
      tag = sanitizeTag(tag);
      if (tag && !editingTags.includes(tag)) {
        editingTags.push(tag);
        renderModalTags();
        renderModalSuggestions("");
      }
      const input = document.getElementById("tagInput");
      if (input) input.value = "";
    }

    function removeEditTag(index) {
      editingTags.splice(index, 1);
      renderModalTags();
      renderModalSuggestions("");
    }

    async function saveTags() {
      if (!currentItem?.fileHandle) {
        toast("Cannot rename: no file handle");
        closeTagEditor();
        return;
      }

      const oldName = currentItem.name;
      const newName = buildFilename(
        currentItem.baseName,
        editingTags,
        currentItem.ext,
      );

      if (oldName === newName) {
        closeTagEditor();
        return;
      }

      try {
        await currentItem.fileHandle.move(newName);

        const newFile = await currentItem.fileHandle.getFile();
        URL.revokeObjectURL(currentItem.url);
        currentItem.file = newFile;
        currentItem.name = newName;
        currentItem.tags = [...editingTags];
        currentItem.url = URL.createObjectURL(newFile);
        currentItem.id = newFile.name + newFile.size + newFile.lastModified;

        if (detailTitle) detailTitle.textContent = newName;
        renderDetailTags(currentItem);
        renderTagsBar();
        applyFiltersAndSort();
        renderGrid();

        closeTagEditor();
        toast(`Renamed to: ${newName}`);
      } catch (err) {
        console.error("Rename failed:", err);
        toast("Rename failed. Check permissions.");
      }
    }

    async function renameCurrentFile() {
      if (!currentItem?.fileHandle) return;

      const ext = currentItem.ext || "." + currentItem.name.split(".").pop();
      const nameNoExt = currentItem.name.substring(
        0,
        currentItem.name.lastIndexOf(ext),
      );

      const input = prompt(
        `Enter new filename (extension ${ext} added automatically):`,
        nameNoExt,
      );
      if (!input) return;

      let newName = input.trim();
      if (!newName.toLowerCase().endsWith(ext.toLowerCase())) {
        newName += ext;
      }

      if (newName === currentItem.name) return;

      try {
        await currentItem.fileHandle.move(newName);

        const newFile = await currentItem.fileHandle.getFile();
        URL.revokeObjectURL(currentItem.url);

        const { baseName, tags } = parseTagsFromFilename(newFile.name);

        currentItem.file = newFile;
        currentItem.name = newFile.name;
        currentItem.baseName = baseName;
        currentItem.tags = tags;
        currentItem.url = URL.createObjectURL(newFile);
        currentItem.id = newFile.name + newFile.size + newFile.lastModified;

        if (detailTitle) detailTitle.textContent = newFile.name;
        renderDetailTags(currentItem);
        renderTagsBar();
        applyFiltersAndSort();
        renderGrid();

        toast("File renamed");
      } catch (e) {
        console.error("Rename failed:", e);
        alert("Could not rename file: " + e.message);
      }
    }

    async function deleteCurrentFile() {
      if (!currentItem) return;

      if (
        !confirm(
          `Are you sure you want to delete "${currentItem.name}"?\nThis cannot be undone.`,
        )
      ) {
        return;
      }

      try {
        if (currentItem.parentHandle) {
          await currentItem.parentHandle.removeEntry(currentItem.name);
        } else {
          await currentItem.fileHandle.remove();
        }

        toast("File deleted");

        const idx = mediaItems.findIndex((m) => m.id === currentItem.id);
        if (idx !== -1) {
          mediaItems.splice(idx, 1);
        }

        closeDetailNow();
        renderTagsBar();
        applyFiltersAndSort();
        renderGrid();
      } catch (e) {
        console.error("Delete failed:", e);
        alert("Could not delete file: " + e.message);
      }
    }

    async function generateUniqueId() {
      if (!currentItem?.fileHandle) {
        toast("Cannot rename: no file handle");
        return;
      }

      const chars = "abcdefghijklmnopqrstuvwxyz0123456789";
      let id = "";
      for (let i = 0; i < 16; i++) {
        id += chars.charAt(Math.floor(Math.random() * chars.length));
      }

      const oldName = currentItem.name;
      const newName = buildFilename(
        id,
        currentItem.tags || [],
        currentItem.ext,
      );

      if (oldName === newName) return;

      try {
        await currentItem.fileHandle.move(newName);

        const newFile = await currentItem.fileHandle.getFile();
        URL.revokeObjectURL(currentItem.url);
        currentItem.file = newFile;
        currentItem.name = newName;
        currentItem.baseName = id;
        currentItem.url = URL.createObjectURL(newFile);
        currentItem.id = newFile.name + newFile.size + newFile.lastModified;

        if (detailTitle) detailTitle.textContent = newName;
        applyFiltersAndSort();
        renderGrid();

        toast(`New ID: ${id}`);
      } catch (err) {
        console.error("Generate ID failed:", err);
        toast("Rename failed. Check permissions.");
      }
    }

    let liveTimer = null;
    let isLive = false;
    let liveItems = [];
    let liveIndex = 0;
    let nextSlideEl = null;
    let nextSlideReady = false;

    let liveSettings = {
      imgTime: 0.3,
      vidTime: 0.9,
      zoom: true,
      zoomAmt: 1.5,
    };

    function startLiveMode() {
      liveItems = filteredItems.length > 0 ? filteredItems : mediaItems;

      if (liveItems.length === 0) {
        toast("No media to play!");
        return;
      }

      isLive = true;
      liveIndex = 0;
      const overlay = document.getElementById("liveOverlay");
      if (overlay) overlay.classList.remove("hidden");

      liveItems = [...liveItems].sort(() => Math.random() - 0.5);

      updateLiveSettings();

      const container = document.getElementById("liveContainer");
      if (container) container.innerHTML = "";
      nextSlideEl = null;
      nextSlideReady = false;

      prepareNextSlide();
      checkTransition();
      syncNoScroll();
    }

    function stopLiveMode() {
      isLive = false;
      clearTimeout(liveTimer);
      const overlay = document.getElementById("liveOverlay");
      if (overlay) overlay.classList.add("hidden");
      const container = document.getElementById("liveContainer");
      if (container) container.innerHTML = "";
      if (document.fullscreenElement) {
        document.exitFullscreen().catch(noop);
      }
      syncNoScroll();
    }

    function updateLiveSettings() {
      const imgInput = document.getElementById("liveImgTime");
      const vidInput = document.getElementById("liveVidTime");
      const zoomEnabled = document.getElementById("liveZoomEnabled");
      const zoomAmt = document.getElementById("liveZoomAmt");
      liveSettings.imgTime = Number.parseFloat(imgInput?.value) || 0.3;
      liveSettings.vidTime = Number.parseFloat(vidInput?.value) || 0.9;
      liveSettings.zoom = !!zoomEnabled?.checked;
      liveSettings.zoomAmt = Number.parseFloat(zoomAmt?.value) || 1.5;
    }

    function prepareNextSlide() {
      const container = document.getElementById("liveContainer");
      if (!container) return;
      const item = liveItems[liveIndex];

      nextSlideReady = false;

      let el;
      if (item.isVideo) {
        el = document.createElement("video");
        el.src = item.url;
        el.muted = true;
        el.autoplay = false;
        el.loop = true;
        el.className = "live-media-content";

        el.onloadedmetadata = () => {
          el.currentTime = 0.1;
        };
        el.onseeked = () => {
          nextSlideReady = true;
        };
        if (el.readyState >= 3) nextSlideReady = true;
      } else {
        el = document.createElement("img");
        el.src = item.url;
        el.className = "live-media-content";
        el.onload = () => {
          nextSlideReady = true;
        };
      }

      el.style.opacity = "0";
      el.style.zIndex = "1";
      container.appendChild(el);
      nextSlideEl = el;

      liveIndex++;
      if (liveIndex >= liveItems.length) {
        liveItems.sort(() => Math.random() - 0.5);
        liveIndex = 0;
      }
    }

    function checkTransition() {
      if (!isLive) return;

      if (!nextSlideEl) {
        prepareNextSlide();
      }

      if (nextSlideReady) {
        runTransition();
      } else {
        liveTimer = setTimeout(checkTransition, 50);
      }
    }

    function runTransition() {
      if (!isLive) return;

      const container = document.getElementById("liveContainer");
      if (!container) return;
      const el = nextSlideEl;

      el.style.zIndex = "10";
      el.style.opacity = "1";

      if (el.tagName === "VIDEO") {
        el.play().catch(noop);
      }

      Array.from(container.children).forEach((child) => {
        if (child !== el) child.remove();
      });

      let duration =
        el.tagName === "VIDEO" ? liveSettings.vidTime : liveSettings.imgTime;

      if (liveSettings.zoom) {
        const slope = (liveSettings.zoomAmt - 1) / duration;
        const bigDuration = duration * 10;
        const bigTarget = 1 + slope * bigDuration;

        el.style.transition = `transform ${bigDuration}s linear`;
        el.getBoundingClientRect();
        el.style.transformOrigin = "center center";
        el.style.transform = `scale(${bigTarget})`;
      } else {
        el.style.transform = "none";
      }

      nextSlideEl = null;
      prepareNextSlide();

      liveTimer = setTimeout(checkTransition, duration * 1000);
    }

    function toggleLiveFullScreen() {
      const overlay = document.getElementById("liveOverlay");
      if (!overlay) return;
      if (document.fullscreenElement) {
        document.exitFullscreen();
        return;
      }

      overlay.requestFullscreen().catch((err) => {
        toast(
          `Error attempting to enable full-screen mode: ${err.message} (${err.name})`,
        );
      });
    }

    let editorImg = null,
      editorCtx = null;
    let cropStart = null;
    const editorCanvas = document.getElementById("editorCanvas");
    const cropOverlay = document.getElementById("cropOverlay");

    function openImageEditor() {
      if (!currentItem || currentItem.isVideo) return;
      const modal = document.getElementById("editorModal");
      if (!modal) return;

      editorImg = new Image();
      editorImg.onload = () => {
        editorCanvas.width = editorImg.naturalWidth;
        editorCanvas.height = editorImg.naturalHeight;
        editorCtx = editorCanvas.getContext("2d");
        editorCtx.drawImage(editorImg, 0, 0);

        cropOverlay.style.display = "none";
        cropStart = null;

        modal.classList.add("active");
        syncNoScroll();
      };
      editorImg.src = currentItem.url;

      document.addEventListener("mousemove", onCropMove);
      document.addEventListener("mouseup", onCropUp);
    }

    function closeEditor() {
      const modal = document.getElementById("editorModal");
      if (modal) modal.classList.remove("active");
      editorImg = null;
      document.removeEventListener("mousemove", onCropMove);
      document.removeEventListener("mouseup", onCropUp);
      syncNoScroll();
    }

    function startCropDrag(e) {
      e.preventDefault();
      const canvasRect = editorCanvas.getBoundingClientRect();
      const wsRect = document
        .getElementById("editorWorkspace")
        .getBoundingClientRect();

      const mouseX = e.clientX;
      const mouseY = e.clientY;

      if (
        mouseX < canvasRect.left ||
        mouseX > canvasRect.right ||
        mouseY < canvasRect.top ||
        mouseY > canvasRect.bottom
      )
        return;

      const xRelToCanvas = mouseX - canvasRect.left;
      const yRelToCanvas = mouseY - canvasRect.top;

      cropStart = {
        x: xRelToCanvas,
        y: yRelToCanvas,
        canvasLeft: canvasRect.left,
        canvasTop: canvasRect.top,
        wsLeft: wsRect.left,
        wsTop: wsRect.top,
      };

      cropOverlay.style.display = "block";

      const cssLeft = mouseX - wsRect.left;
      const cssTop = mouseY - wsRect.top;

      cropOverlay.style.left = cssLeft + "px";
      cropOverlay.style.top = cssTop + "px";
      cropOverlay.style.width = "0px";
      cropOverlay.style.height = "0px";
    }

    function onCropMove(e) {
      if (!cropStart) return;
      e.preventDefault();

      const canvasRect = editorCanvas.getBoundingClientRect();
      const wsRect = document
        .getElementById("editorWorkspace")
        .getBoundingClientRect();

      const mouseX = e.clientX;
      const mouseY = e.clientY;

      const clampedX = Math.max(
        canvasRect.left,
        Math.min(mouseX, canvasRect.right),
      );
      const clampedY = Math.max(
        canvasRect.top,
        Math.min(mouseY, canvasRect.bottom),
      );

      const originVPX = cropStart.x + cropStart.canvasLeft;
      const originVPY = cropStart.y + cropStart.canvasTop;

      const leftVP = Math.min(originVPX, clampedX);
      const topVP = Math.min(originVPY, clampedY);
      const width = Math.abs(clampedX - originVPX);
      const height = Math.abs(clampedY - originVPY);

      cropOverlay.style.left = leftVP - wsRect.left + "px";
      cropOverlay.style.top = topVP - wsRect.top + "px";
      cropOverlay.style.width = width + "px";
      cropOverlay.style.height = height + "px";
    }

    function onCropUp() {
      cropStart = null;
    }

    function applyCrop() {
      if (
        cropOverlay.style.display === "none" ||
        Number.parseFloat(cropOverlay.style.width) < 5
      ) {
        toast("Select an area to crop first");
        return;
      }

      const canvasRect = editorCanvas.getBoundingClientRect();
      const wsRect = document
        .getElementById("editorWorkspace")
        .getBoundingClientRect();

      const selLeft = Number.parseFloat(cropOverlay.style.left);
      const selTop = Number.parseFloat(cropOverlay.style.top);
      const selW = Number.parseFloat(cropOverlay.style.width);
      const selH = Number.parseFloat(cropOverlay.style.height);

      const relVisX = wsRect.left + selLeft - canvasRect.left;
      const relVisY = wsRect.top + selTop - canvasRect.top;

      const scaleX = editorCanvas.width / canvasRect.width;
      const scaleY = editorCanvas.height / canvasRect.height;

      const realX = Math.round(relVisX * scaleX);
      const realY = Math.round(relVisY * scaleY);
      const realW = Math.round(selW * scaleX);
      const realH = Math.round(selH * scaleY);

      if (realW <= 0 || realH <= 0) return;

      const tempC = document.createElement("canvas");
      tempC.width = realW;
      tempC.height = realH;
      const tempCtx = tempC.getContext("2d");

      tempCtx.drawImage(
        editorCanvas,
        realX,
        realY,
        realW,
        realH,
        0,
        0,
        realW,
        realH,
      );

      editorCanvas.width = realW;
      editorCanvas.height = realH;
      editorCtx = editorCanvas.getContext("2d");
      editorCtx.drawImage(tempC, 0, 0);

      cropOverlay.style.display = "none";
      cropOverlay.style.width = "0";
    }

    async function saveEditorImage() {
      if (
        cropOverlay.style.display !== "none" &&
        Number.parseFloat(cropOverlay.style.width) > 5
      ) {
        applyCrop();
      }
      if (!confirm("Overwrite original file? Cannot be undone.")) return;
      await saveCanvasToFile(currentItem.fileHandle);
    }

    async function saveEditorCopy() {
      if (
        cropOverlay.style.display !== "none" &&
        Number.parseFloat(cropOverlay.style.width) > 5
      ) {
        applyCrop();
      }
      try {
        const parts = currentItem.name.split(".");
        const ext = parts.pop();
        const name = parts.join(".") + "_edit." + ext;

        const opts = {
          suggestedName: name,
          types: [
            {
              description: "Image",
              accept: { "image/*": [".png", ".jpg", ".jpeg", ".webp"] },
            },
          ],
        };

        const newHandle = await globalThis.showSaveFilePicker(opts);
        await saveCanvasToFile(newHandle);
      } catch (e) {
        if (e.name !== "AbortError") toast("Save Failed: " + e.message);
      }
    }

    async function saveCanvasToFile(handle) {
      try {
        const name = handle.name || currentItem.name;
        const ext = name.split(".").pop().toLowerCase();
        let mime = "image/png";
        if (["jpg", "jpeg"].includes(ext)) mime = "image/jpeg";
        else if (ext === "webp") mime = "image/webp";

        const blob = await new Promise((r) =>
          editorCanvas.toBlob(r, mime, 0.95),
        );
        const writable = await handle.createWritable();
        await writable.write(blob);
        await writable.close();

        toast("File saved!");
        closeEditor();

        if (currentItem && handle === currentItem.fileHandle) {
          const newFile = await currentItem.fileHandle.getFile();
          await updateCurrentWithFile(newFile);
        }
      } catch (e) {
        console.error(e);
        toast("Error saving: " + e.message);
      }
    }

    async function updateCurrentWithFile(file) {
      URL.revokeObjectURL(currentItem.url);
      currentItem.url = URL.createObjectURL(file);
      currentItem.size = file.size;
      currentItem.lastModified = file.lastModified;
      const img = detailMedia?.querySelector("img");
      if (img) img.src = currentItem.url;
    }

    function handleGlobalKeydown(e) {
      const detailHidden = detailView?.classList.contains("hidden") === true;
      const lightboxActive =
        document.getElementById("lightbox")?.classList?.contains("active") ===
        true;

      if (detailHidden || lightboxActive) return;
      if (document.activeElement?.tagName === "INPUT") return;

      if (e.key === "ArrowLeft") navDetail(-1);
      if (e.key === "ArrowRight") navDetail(1);
      if (e.key === "Escape") closeDetail();
    }

    const actionHandlers = {
      "select-folder": selectFolder,
      "reconnect-folders": reconnectFolders,
      "clear-all": clearAll,
      "manual-refresh": manualRefresh,
      "start-live": startLiveMode,
      "toggle-settings": toggleSettings,
      "toggle-about": toggleAbout,
      "toggle-ai-help": toggleAiHelp,
      "close-detail": closeDetail,
      "nav-detail-prev": () => navDetail(-1),
      "nav-detail-next": () => navDetail(1),
      "open-viewer": openViewer,
      "open-original": openOriginalFile,
      "open-editor": openImageEditor,
      "auto-tag": autoTagCurrentFile,
      "rename-file": renameCurrentFile,
      "open-tag-editor": openTagEditor,
      "generate-id": generateUniqueId,
      "delete-file": deleteCurrentFile,
      "load-more-similar": loadMoreSimilar,
      "close-lightbox": () => closeLightbox(),
      "apply-crop": applyCrop,
      "save-editor-overwrite": saveEditorImage,
      "save-editor-copy": saveEditorCopy,
      "close-editor": closeEditor,
      "toggle-logs": toggleLogs,
      "clear-logs": clearLogs,
      "toggle-logs-settings": toggleLogsFromSettings,
      "toggle-failed-settings": toggleFailedPanelFromSettings,
      "train-ai": trainAiFromTaggedFiles,
      "cancel-training": cancelAiTraining,
      "update-tag-db": updateTagDb,
      "close-tag-editor": closeTagEditor,
      "save-tags": saveTags,
      "close-ai-preview": closeAiTagPreview,
      "apply-ai-preview": applyAiTagPreview,
      "live-fullscreen": toggleLiveFullScreen,
      "stop-live": stopLiveMode,
      "failed-close": toggleFailedPanel,
      "failed-clear": clearFailedList,
    };

    function bindUi() {
      const tagsBar = document.getElementById("tagsBar");
      if (tagsBar) {
        tagsBar.addEventListener("click", (e) => {
          const pill = e.target.closest(".tag-pill");
          if (!pill || !tagsBar.contains(pill)) return;
          toggleTagFilter(pill.dataset.tag || "");
        });
      }

      const detailTags = document.getElementById("detailTags");
      if (detailTags) {
        detailTags.addEventListener("click", (e) => {
          const pill = e.target.closest(".tag-pill");
          if (!pill || !detailTags.contains(pill)) return;
          setTagFilter(pill.dataset.tag || "");
          closeDetail();
        });
      }

      const modalTags = document.getElementById("modalTags");
      if (modalTags) {
        modalTags.addEventListener("click", (e) => {
          const removeBtn = e.target.closest("[data-remove-index]");
          if (!removeBtn) return;
          const idx = Number.parseInt(removeBtn.dataset.removeIndex, 10);
          if (!Number.isNaN(idx)) removeEditTag(idx);
        });
      }

      const modalSuggestions = document.getElementById("modalSuggestions");
      if (modalSuggestions) {
        modalSuggestions.addEventListener("click", (e) => {
          const pill = e.target.closest("[data-add-tag]");
          if (!pill) return;
          addEditTag(pill.dataset.addTag || "");
        });
      }

      const aiTagPreviewEl = document.getElementById("aiTagPreview");
      if (aiTagPreviewEl) {
        aiTagPreviewEl.addEventListener("click", (e) => {
          const pill = e.target.closest(".ai-tag-pill");
          if (!pill) return;
          toggleAiTagSelection(pill.dataset.tag || "");
        });
      }

      const lightbox = document.getElementById("lightbox");
      if (lightbox) {
        lightbox.addEventListener("click", closeLightbox);
      }

      const editorWorkspace = document.getElementById("editorWorkspace");
      if (editorWorkspace) {
        editorWorkspace.addEventListener("mousedown", startCropDrag);
      }

      const tagInput = document.getElementById("tagInput");
      if (tagInput) {
        tagInput.addEventListener("keydown", (e) => {
          if (e.key === "Enter") {
            e.preventDefault();
            addEditTag(tagInput.value);
          }
        });
        tagInput.addEventListener("input", (e) => {
          renderModalSuggestions(e.target.value);
        });
      }

      root.addEventListener("input", (e) => {
        const target = e.target;
        if (target.matches('[data-em-input="search"]')) {
          filterMedia(target.value);
        } else if (target.matches('[data-em-input="card-size"]')) {
          updateCardSize(target.value);
        } else if (target.matches('[data-em-input="tag-size"]')) {
          updateTagSize(target.value);
        }
      });

      root.addEventListener("change", (e) => {
        const target = e.target;
        if (target.matches('[data-em-change="type-filter"]')) {
          setTypeFilter(target.value);
        } else if (target.matches('[data-em-change="strict-tags"]')) {
          toggleStrictTags();
        } else if (target.matches('[data-em-change="ai-enabled"]')) {
          toggleAiEnabled(target.checked);
        } else if (target.matches('[data-em-change="ai-max-samples"]')) {
          updateAiMaxSamples(target.value);
        } else if (target.matches('[data-em-change="live-settings"]')) {
          updateLiveSettings();
        }
      });

      document.addEventListener("click", (event) => {
        const actionEl = event.target.closest("[data-em-action]");
        if (!actionEl) return;
        if (!root.contains(actionEl) && !actionEl.closest("#failedPanel"))
          return;
        const action = actionEl.dataset.emAction;
        const handler = actionHandlers[action];
        if (handler) {
          event.preventDefault();
          handler(actionEl, event);
        }
      });

      document.addEventListener("keydown", handleGlobalKeydown);
    }

    function initializeAiControls() {
      try {
        aiEnabled = localStorage.getItem("em_ai_enabled") === "1";
      } catch {
        aiEnabled = false;
      }

      const aiToggle = document.getElementById("settingAiEnabled");
      if (aiToggle) aiToggle.checked = aiEnabled;

      const status = document.getElementById("aiStatusText");
      if (status)
        status.textContent = aiEnabled
          ? `AI status: Connected (${aiServerActive})`
          : "AI status: Off";

      const trainingSection = document.getElementById("aiTrainingSection");
      trainingSection?.classList.toggle("hidden", !aiEnabled);

      const btnAutoTag = document.getElementById("btnAutoTag");
      if (btnAutoTag) {
        const shouldDisable = !aiEnabled || Boolean(currentItem?.isVideo);
        btnAutoTag.disabled = shouldDisable;
        btnAutoTag.classList.toggle("disabled", shouldDisable);
      }

      try {
        const storedMax = Number.parseInt(
          localStorage.getItem("em_ai_max_samples"),
          10,
        );
        if (!Number.isNaN(storedMax)) {
          aiTrainingMaxSamples = storedMax;
        }
      } catch {}

      const maxInput = document.getElementById("settingAiMaxSamples");
      if (maxInput) maxInput.value = aiTrainingMaxSamples;

      updateTrainingUi(0, 0, null, "Training status: Idle");
      detectAiServerStatus();
    }

    let resizeTimeout = null;

    function scheduleGridRender() {
      if (resizeTimeout) clearTimeout(resizeTimeout);
      resizeTimeout = setTimeout(renderGridAfterResize, 150);
    }

    function renderGridAfterResize() {
      resizeTimeout = null;
      renderGrid();
    }

    function bindViewportListeners() {
      globalThis.addEventListener("scroll", onContentScroll);
      globalThis.addEventListener("resize", scheduleGridRender);
    }

    initializeAiControls();
    bindViewportListeners();

    bindUi();
    setDetailMode(false);
    restoreFolders();
  }

  if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", initEasyMedia);
  } else {
    initEasyMedia();
  }
})();
