function isHttps() {
    return globalThis.location.protocol === 'https:';
}

document.addEventListener('DOMContentLoaded', function () {
    const banner = document.getElementById('cookie-consent-banner');
    const modal = document.getElementById('cookie-settings-modal');
    if (!banner || !modal) {
        return;
    }

    const analyticsToggle = document.getElementById('cookie-consent-analytics');
    const marketingToggle = document.getElementById('cookie-consent-marketing');

    const STORAGE_KEY = 'cookie_consent_preferences';
    const API_ENDPOINT = '/cookie-consent';

    function getStoredPrefs() {
        const stored = localStorage.getItem(STORAGE_KEY);
        if (!stored) {
            return null;
        }
        try {
            return JSON.parse(stored);
        } catch (err) {
            console.debug('cookie-consent: stored prefs unreadable', err);
            return null;
        }
    }

    function setConsentCookie(prefs) {
        const maxAgeSeconds = 365 * 24 * 60 * 60;
        const encoded = encodeURIComponent(JSON.stringify(prefs));
        let cookie = `cookie_consent=${encoded};path=/;max-age=${maxAgeSeconds};SameSite=Lax`;
        if (isHttps()) {
            cookie += ';Secure';
        }
        document.cookie = cookie;
    }

    async function sendConsentToServer(prefs) {
        try {
            await fetch(API_ENDPOINT, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                credentials: 'same-origin',
                body: JSON.stringify(prefs)
            });
        } catch (err) {
            console.debug('cookie-consent: server sync failed', err);
        }
    }

    function applyPrefs(prefs) {
        document.documentElement.dataset.cookieConsent = '1';
        document.documentElement.dataset.cookieAnalytics = prefs.analytics ? '1' : '0';
        document.documentElement.dataset.cookieMarketing = prefs.marketing ? '1' : '0';
    }

    function showBanner() {
        banner.classList.add('is-visible');
    }

    function hideBanner() {
        banner.classList.remove('is-visible');
    }

    function openModal() {
        modal.classList.add('is-open');
    }

    function closeModal() {
        modal.classList.remove('is-open');
    }

    function loadDefaults() {
        const defaults = {
            necessary: true,
            analytics: false,
            marketing: false
        };
        const stored = getStoredPrefs();
        const prefs = stored || defaults;
        if (analyticsToggle) {
            analyticsToggle.checked = !!prefs.analytics;
        }
        if (marketingToggle) {
            marketingToggle.checked = !!prefs.marketing;
        }
    }

    async function savePreferences(prefs) {
        const payload = {
            necessary: true,
            analytics: !!prefs.analytics,
            marketing: !!prefs.marketing,
            timestamp: new Date().toISOString()
        };

        localStorage.setItem(STORAGE_KEY, JSON.stringify(payload));
        setConsentCookie(payload);
        hideBanner();
        closeModal();
        applyPrefs(payload);
        await sendConsentToServer(payload);
    }

    const existingPrefs = getStoredPrefs();
    if (existingPrefs) {
        applyPrefs(existingPrefs);
    } else {
        showBanner();
        loadDefaults();
    }

    document.querySelectorAll('[data-cookie-settings], .js-cookie-settings').forEach(function (btn) {
        btn.addEventListener('click', function (event) {
            if (event) {
                event.preventDefault();
            }
            loadDefaults();
            openModal();
        });
    });

    document.querySelectorAll('[data-cookie-close]').forEach(function (btn) {
        btn.addEventListener('click', function () {
            closeModal();
        });
    });

    const acceptBtn = banner.querySelector('[data-cookie-accept]');
    if (acceptBtn) {
        acceptBtn.addEventListener('click', function () {
            savePreferences({ analytics: true, marketing: true });
        });
    }

    const rejectBtn = banner.querySelector('[data-cookie-reject]');
    if (rejectBtn) {
        rejectBtn.addEventListener('click', function () {
            savePreferences({ analytics: false, marketing: false });
        });
    }

    const saveBtn = modal.querySelector('[data-cookie-save]');
    if (saveBtn) {
        saveBtn.addEventListener('click', function () {
            savePreferences({
                analytics: analyticsToggle ? analyticsToggle.checked : false,
                marketing: marketingToggle ? marketingToggle.checked : false
            });
        });
    }
});
