<?php

/**
 * Dynamic Theme CSS Generator
 *
 * Routed through the public front controller (index.php).
 * Reads theme settings from the database and outputs CSS custom properties.
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Database\DatabaseManager;
use NewSite\Settings\SettingsService;
use NewSite\Cleanup\MissingUploadsService;

header('Content-Type: text/css; charset=utf-8');

// --- Custom font-face declarations (merged from custom-fonts.php) ---
$customFonts = json_decode(SettingsService::get('custom_fonts', '[]'), true);
if (!is_array($customFonts)) {
    $customFonts = [];
}

$fontFormatMap = [
    'woff2' => 'woff2',
    'woff'  => 'woff',
    'ttf'   => 'truetype',
    'otf'   => 'opentype',
];

foreach ($customFonts as $customFont) {
    $cfName = isset($customFont['name']) ? trim($customFont['name']) : '';
    $cfFile = isset($customFont['file']) ? basename($customFont['file']) : '';
    if ($cfName === '' || $cfFile === '') {
        continue;
    }
    $cfSafeName = preg_replace('/[^a-zA-Z0-9\s_-]/', '', $cfName);
    if ($cfSafeName === '') {
        continue;
    }
    $cfExt = strtolower(pathinfo($cfFile, PATHINFO_EXTENSION));
    if (!isset($fontFormatMap[$cfExt])) {
        continue;
    }
    $cfPath = __DIR__ . '/../uploads/fonts/' . $cfFile;
    if (!is_file($cfPath)) {
        continue;
    }
    $cfUrl    = '/assets/uploads/fonts/' . $cfFile;
    $cfFormat = $fontFormatMap[$cfExt];
    echo "@font-face {\n";
    echo "  font-family: '" . addslashes($cfSafeName) . "';\n";
    echo "  src: url('" . $cfUrl . "') format('" . $cfFormat . "');\n";
    echo "  font-weight: normal;\n";
    echo "  font-style: normal;\n";
    echo "  font-display: swap;\n";
    echo "}\n";
}
// --- End custom font-face declarations ---

function sanitizeColor(string $value, string $fallback): string
{
    $value = trim($value);
    return preg_match('/^#([0-9a-fA-F]{3}){1,2}$/', $value) ? $value : $fallback;
}

function sanitizeFontFamily(string $value, string $fallback): string
{
    $value = trim($value);
    $value = preg_replace('/[^a-zA-Z0-9 ,\'"-]/', '', $value);
    return $value !== '' ? $value : $fallback;
}

function cssEscapeUrl(string $url): string
{
    return str_replace(["\\", "'"], ["\\\\", "\\'"], $url);
}

function themeToPublicFilePath(string $path): string
{
    if ($path === '') {
        return '';
    }
    return str_replace('/admin-file/', '/site-file/', $path);
}

function intOrDefault($value, int $fallback): int
{
    if (is_numeric($value)) {
        return (int)$value;
    }
    return $fallback;
}

$theme = SettingsService::getAllTheme();

$primary = sanitizeColor((string)($theme['primary_color'] ?? ''), '#a78bff');
$secondary = sanitizeColor((string)($theme['secondary_color'] ?? ''), '#6f5bff');
$accent = sanitizeColor((string)($theme['accent_color'] ?? ''), '#c9b8ff');
$background = sanitizeColor((string)($theme['background_color'] ?? ''), '#0f0d16');
$text = sanitizeColor((string)($theme['text_color'] ?? ''), '#f4f2ff');
$fontFamily = sanitizeFontFamily((string)($theme['font_family'] ?? ''), 'system-ui, sans-serif');
$contentMaxWidth = intOrDefault($theme['content_max_width'] ?? 1400, 1400);
$sectionPadding = intOrDefault($theme['section_padding'] ?? 60, 60);
$sectionHeight = intOrDefault($theme['section_height'] ?? 100, 100);

$rootRules = [
    "--primary-color: {$primary};",
    "--secondary-color: {$secondary};",
    "--accent-color: {$accent};",
    "--bg-color: {$background};",
    "--text-color: {$text};",
    "--font-family: {$fontFamily};",
    "--font-secondary: {$fontFamily};",
    "--content-max-width: min({$contentMaxWidth}px, 100vw);",
    "--section-padding: min({$sectionPadding}px, 12vh);",
    "--section-height: min({$sectionHeight}vh, 100vh);"
];

echo ":root {\n    " . implode("\n    ", $rootRules) . "\n}\n";

$pageId = (int)($_GET['page_id'] ?? 0);
if ($pageId > 0) {
    $db = DatabaseManager::getWriteConnection();
    $stmt = $db->prepare("SELECT use_layout_override, content_max_width, section_padding, section_height FROM pages WHERE id = ? LIMIT 1");
    $stmt->execute([$pageId]);
    $page = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!empty($page['use_layout_override'])) {
        $pageRules = [];
        if ($page['content_max_width'] !== null && $page['content_max_width'] !== '') {
            $pageRules[] = "--content-max-width: min(" . (int)$page['content_max_width'] . "px, 100vw);";
        }
        if ($page['section_padding'] !== null && $page['section_padding'] !== '') {
            $pageRules[] = "--section-padding: min(" . (int)$page['section_padding'] . "px, 12vh);";
        }
        if ($page['section_height'] !== null && $page['section_height'] !== '') {
            $pageRules[] = "--section-height: min(" . (int)$page['section_height'] . "vh, 100vh);";
        }
        if (!empty($pageRules)) {
            echo "body[data-page-id=\"{$pageId}\"] {\n    " . implode("\n    ", $pageRules) . "\n}\n";
        }
    }

    $stmt = $db->prepare("SELECT id, section_type, settings FROM sections WHERE page_id = ? AND is_active = 1 ORDER BY sort_order");
    $stmt->execute([$pageId]);
    $sections = $stmt->fetchAll(PDO::FETCH_ASSOC);

    foreach ($sections as $section) {
        $sectionId = (int)($section['id'] ?? 0);
        if ($sectionId <= 0) {
            continue;
        }
        $sectionType = $section['section_type'] ?? '';
        $settings = json_decode($section['settings'] ?? '', true) ?? [];
        $changed = false;
        $settings = MissingUploadsService::stripFromSettings($settings, $changed);
        $selector = ".section-id-{$sectionId}";

        if ($sectionType === 'image_banner') {
            $images = $settings['images'] ?? [];
            $titleSize = intOrDefault($settings['title_size'] ?? 48, 48);
            $subtitleSize = intOrDefault($settings['subtitle_size'] ?? 18, 18);
            $textAlign = $settings['text_align'] ?? 'center';
            $textAlign = in_array($textAlign, ['left', 'center', 'right'], true) ? $textAlign : 'center';
            $height = intOrDefault($settings['height'] ?? 300, 300);
            $spacingTop = intOrDefault($settings['spacing_top'] ?? 0, 0);
            $spacingBottom = intOrDefault($settings['spacing_bottom'] ?? 0, 0);
            $overlayOpacity = intOrDefault($settings['overlay_opacity'] ?? 50, 50);
            $overlayOpacity = max(0, min(100, $overlayOpacity));
            $imagePosition = $settings['image_position'] ?? 'center';
            $textPosition = $settings['text_position'] ?? 'center';

            $positionMap = [
                'top' => 'top',
                '25' => '25%',
                'center' => 'center',
                '75' => '75%',
                'bottom' => 'bottom'
            ];
            $bgPositionY = $positionMap[$imagePosition] ?? 'center';

            $textPositionMap = [
                'top' => 'flex-start',
                'center' => 'center',
                'bottom' => 'flex-end'
            ];
            $textJustify = $textPositionMap[$textPosition] ?? 'center';

            $overlayColor = 'rgba(0, 0, 0, ' . ($overlayOpacity / 100) . ')';
            echo ".section-image-banner{$selector} { height: {$height}px; margin-top: {$spacingTop}px; margin-bottom: {$spacingBottom}px; --banner-overlay: {$overlayColor}; }\n";
            echo ".section-image-banner{$selector} .banner-content { text-align: {$textAlign}; justify-content: {$textJustify}; }\n";
            echo ".section-image-banner{$selector} .banner-title { font-size: {$titleSize}px; }\n";
            echo ".section-image-banner{$selector} .banner-subtitle { font-size: {$subtitleSize}px; }\n";

            $validImages = [];
            foreach ($images as $img) {
                if (empty($img['image'])) {
                    continue;
                }
                $imagePath = (string)$img['image'];
                if (strpos($imagePath, 'http') === 0 || strpos($imagePath, 'data:') === 0) {
                    $validImages[] = $img;
                    continue;
                }
                if (!MissingUploadsService::isMissingPath($imagePath)) {
                    $validImages[] = $img;
                }
            }
            if (empty($validImages)) {
                $validImages = [['image' => '/assets/images/placeholder-1.svg']];
            }
            foreach ($validImages as $index => $img) {
                $imagePath = (string)($img['image'] ?? '');
                if ($imagePath === '') {
                    continue;
                }
                if (preg_match('#^/assets/images/placeholder(?:-(\d+))?\.jpg$#', $imagePath, $match)) {
                    $suffix = isset($match[1]) && $match[1] !== '' ? '-' . $match[1] : '';
                    $imagePath = '/assets/images/placeholder' . $suffix . '.svg';
                }
                if (strpos($imagePath, 'http') !== 0 && strpos($imagePath, 'data:') !== 0) {
                    $imagePath = themeToPublicFilePath($imagePath);
                }
                $imagePath = cssEscapeUrl($imagePath);
                echo ".section-image-banner{$selector} .banner-slide[data-slide-index=\"{$index}\"] { background-image: url('{$imagePath}'); background-position: center {$bgPositionY}; }\n";
            }
        }

        if ($sectionType === 'split_slideshow') {
            $slides = $settings['slides'] ?? [];
            if (empty($slides)) {
                $slides = [
                    [
                        'title' => 'Welcome to Your Website',
                        'description' => 'Create stunning websites with our powerful theme builder. Customize everything to match your brand.',
                        'button_text' => 'Get Started',
                        'button_url' => '#',
                        'image' => '/assets/images/placeholder-1.svg'
                    ],
                    [
                        'title' => 'Beautiful & Modern',
                        'description' => 'Our neon-inspired design brings your content to life with vibrant colors and smooth animations.',
                        'button_text' => 'Learn More',
                        'button_url' => '#',
                        'image' => '/assets/images/placeholder-2.svg'
                    ]
                ];
            }

            $productById = [];
            $productIds = [];
            foreach ($slides as $slide) {
                $slideProductId = (int)($slide['button_product_id'] ?? 0);
                $slideLinkType = $slide['button_link_type'] ?? 'custom';
                $slideMediaSource = $slide['media_source'] ?? 'manual';
                if ($slideProductId > 0 && ($slideLinkType === 'product' || $slideMediaSource === 'product')) {
                    $productIds[$slideProductId] = true;
                }
            }
            if (!empty($productIds) && isset($db)) {
                $ids = array_keys($productIds);
                $placeholders = implode(',', array_fill(0, count($ids), '?'));
                $stmt = $db->prepare("SELECT p.id, p.media_url, pm.media_url AS gallery_media FROM products p LEFT JOIN product_media pm ON pm.product_id = p.id AND pm.sort_order = 0 WHERE p.id IN ($placeholders) AND p.is_active = 1");
                $stmt->execute($ids);
                foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
                    $pid = (int)($row['id'] ?? 0);
                    if ($pid > 0) {
                        $productById[$pid] = $row;
                    }
                }
            }

            $slideCount = count($slides);
            if ($slideCount > 1) {
                $defaultDotSize = 50;
                if ($slideCount > 10) {
                    $dotSize = max(20, 50 - ($slideCount - 10) * 3);
                } else {
                    $dotSize = $defaultDotSize;
                }
                echo ".section-split-slideshow{$selector} .slideshow-nav { --dot-size: {$dotSize}px; }\n";
            }

            $videoThumbnails = [];
            $videoFiles = [];
            foreach ($slides as $slide) {
                $mediaPath = (string)($slide['image'] ?? '');
                $slideMediaSource = $slide['media_source'] ?? 'manual';
                $slideProductId = (int)($slide['button_product_id'] ?? 0);
                if ($slideMediaSource === 'product' && $slideProductId > 0 && isset($productById[$slideProductId])) {
                    $productMedia = trim((string)($productById[$slideProductId]['gallery_media'] ?? $productById[$slideProductId]['media_url'] ?? ''));
                    if ($productMedia !== '') {
                        $mediaPath = $productMedia;
                    }
                }
                if (preg_match('/\.(mp4|webm|ogg)$/i', $mediaPath)) {
                    $videoFiles[] = basename($mediaPath);
                }
            }
            if (!empty($videoFiles)) {
                $videoFiles = array_unique($videoFiles);
                $placeholders = implode(' OR ', array_fill(0, count($videoFiles), 'file_path LIKE ?'));
                $stmt = $db->prepare("SELECT file_path, thumbnail_path FROM files WHERE {$placeholders}");
                $likeParams = array_map(function ($f) {
                    return '%' . $f;
                }, $videoFiles);
                $stmt->execute($likeParams);
                while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                    if (!empty($row['thumbnail_path'])) {
                        $filename = basename($row['file_path']);
                        $videoThumbnails[$filename] = $row['thumbnail_path'];
                    }
                }
            }

            foreach ($slides as $dotIndex => $dotSlide) {
                $dotMedia = (string)($dotSlide['image'] ?? '');
                $slideMediaSource = $dotSlide['media_source'] ?? 'manual';
                $slideProductId = (int)($dotSlide['button_product_id'] ?? 0);
                if ($slideMediaSource === 'product' && $slideProductId > 0 && isset($productById[$slideProductId])) {
                    $productMedia = trim((string)($productById[$slideProductId]['gallery_media'] ?? $productById[$slideProductId]['media_url'] ?? ''));
                    if ($productMedia !== '') {
                        $dotMedia = $productMedia;
                    }
                }
                if ($dotMedia === '') {
                    continue;
                }
                if (preg_match('#^/assets/images/placeholder(?:-(\d+))?\.jpg$#', $dotMedia, $match)) {
                    $suffix = isset($match[1]) && $match[1] !== '' ? '-' . $match[1] : '';
                    $dotMedia = '/assets/images/placeholder' . $suffix . '.svg';
                }
                $isVideoFile = preg_match('/\.(mp4|webm|ogg)$/i', $dotMedia);
                if ($isVideoFile) {
                    $filename = basename($dotMedia);
                    if (isset($videoThumbnails[$filename])) {
                        $dotMedia = (string)$videoThumbnails[$filename];
                    }
                }
                if (strpos($dotMedia, 'http') !== 0 && strpos($dotMedia, 'data:') !== 0) {
                    $dotMedia = themeToPublicFilePath($dotMedia);
                }
                // Optimise dot thumbnails: resize to 2× dot-size and convert to WebP
                // Dots are tiny (20-50px); loading full images wastes bandwidth
                $dotThumbSize = ($dotSize ?? 50) * 2;
                $isSiteFile = strpos($dotMedia, '/site-file/') === 0;
                if ($isSiteFile) {
                    $dotMedia .= '?w=' . $dotThumbSize . '&format=webp';
                }
                $dotMedia = cssEscapeUrl($dotMedia);
                echo ".section-split-slideshow{$selector} .slideshow-dot[data-index=\"{$dotIndex}\"] { background-image: url('{$dotMedia}'); }\n";
            }
        }

        if ($sectionType === 'testimonials') {
            $cardWidth = intOrDefault($settings['card_width'] ?? 320, 320);
            $cardWidth = max(220, min(520, $cardWidth));
            echo ".section-testimonials{$selector} .testimonial-card { flex: 0 0 {$cardWidth}px; }\n";
        }
    }
}
