<?php
/**
 * Admin Panel - Vendor Libraries
 *
 * Security notes:
 * - This page is admin-only (requireAdmin) and POST actions require CSRF.
 * - We DO NOT auto-download and overwrite vendor JS from arbitrary URLs.
 *   That would be a supply-chain / SSRF risk. Instead we only (optionally)
 *   check the official three.js GitHub latest release tag, and allow an
 *   explicit admin upload to replace the local vendor files.
 */

declare(strict_types=1);

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Auth\AdminAuth;
use NewSite\Logging\LogService;
use NewSite\Admin\AdminLayout;

AdminAuth::requireLogin();

$message = '';
$error = '';

$vendorDir = realpath(__DIR__ . '/../assets/js/vendor');
if ($vendorDir === false) {
    $vendorDir = '';
}

$corePath = $vendorDir !== '' ? $vendorDir . DIRECTORY_SEPARATOR . 'three.core.min.js' : '';
$modulePath = $vendorDir !== '' ? $vendorDir . DIRECTORY_SEPARATOR . 'three.module.min.js' : '';

$readHead = static function (string $path, int $bytes = 8192): string {
    if ($path === '' || !is_file($path) || !is_readable($path)) {
        return '';
    }
    $fh = @fopen($path, 'rb');
    if ($fh === false) {
        return '';
    }
    $data = (string)@fread($fh, $bytes);
    @fclose($fh);
    return $data;
};

$extractThreeRevisionFromCore = static function (string $coreHead): int {
    // three.core.min.js begins with: const t="182",...
    if ($coreHead === '') {
        return 0;
    }

    $revision = 0;
    $patterns = [
        '/const\s+\w+\s*=\s*"(\d{2,4})"\s*,/',
        '/\bREVISION\b\s*=\s*"(\d{2,4})"/'
    ];

    foreach ($patterns as $pattern) {
        if (preg_match($pattern, $coreHead, $m)) {
            $revision = (int)($m[1] ?? 0);
            break;
        }
    }

    return $revision;
};

$sha384 = static function (string $path): string {
    if ($path === '' || !is_file($path) || !is_readable($path)) {
        return '';
    }
    $raw = hash_file('sha384', $path, true);
    if ($raw === false || $raw === '') {
        return '';
    }
    return 'sha384-' . base64_encode($raw);
};

$vendorInfo = [
    'vendor_dir' => $vendorDir,
    'core_exists' => is_file($corePath),
    'module_exists' => is_file($modulePath),
    'core_mtime' => is_file($corePath) ? (int)@filemtime($corePath) : 0,
    'module_mtime' => is_file($modulePath) ? (int)@filemtime($modulePath) : 0,
    'core_size' => is_file($corePath) ? (int)@filesize($corePath) : 0,
    'module_size' => is_file($modulePath) ? (int)@filesize($modulePath) : 0,
    'core_sha384' => $sha384($corePath),
    'module_sha384' => $sha384($modulePath),
    'revision' => 0,
];

$coreHead = $readHead($corePath);
$vendorInfo['revision'] = $extractThreeRevisionFromCore($coreHead);

/**
 * Perform a GET request against a strict allowlist of vendor-related hosts.
 * Security: only these hostnames are ever contacted.
 */
function threeVendorHttpGet(string $url, int $timeoutSeconds = 5, array $extraHeaders = []): string
{
    $parts = parse_url($url);
    $host = strtolower((string)($parts['host'] ?? ''));
    $allowed = ['api.github.com', 'registry.npmjs.org', 'unpkg.com', 'github.com'];
    if ($host === '' || !in_array($host, $allowed, true)) {
        return '';
    }

    $headers = array_merge([
        'User-Agent: NewSite-Admin',
        'Accept: application/json, text/plain, */*',
    ], $extraHeaders);

    $ctx = stream_context_create([
        'http' => [
            'method' => 'GET',
            'timeout' => $timeoutSeconds,
            'header' => implode("\r\n", $headers) . "\r\n",
            'ignore_errors' => true,
        ],
        'ssl' => [
            'verify_peer' => true,
            'verify_peer_name' => true,
        ],
    ]);

    $raw = @file_get_contents($url, false, $ctx);

    return $raw === false ? '' : (string)$raw;
}

/**
 * Probe GitHub API for the latest Three.js release tag (r###).
 */
function probeThreeGitHubApi(): ?array
{
    $raw = threeVendorHttpGet(
        'https://api.github.com/repos/mrdoob/three.js/releases/latest',
        5,
        ['Accept: application/vnd.github+json']
    );

    $json = $raw !== '' ? json_decode($raw, true) : null;
    if (!is_array($json)) {
        return null;
    }

    $tag = isset($json['tag_name']) ? (string)$json['tag_name'] : '';
    if ($tag !== '' && preg_match('/^r(\d{2,4})$/', $tag, $m)) {
        return ['ok' => true, 'source' => 'github-api', 'tag' => $tag, 'revision' => (int)$m[1], 'version' => ''];
    }

    return null;
}

/**
 * Fetch the latest Three.js semver from the npm registry.
 *
 * @return string Semver string (e.g. "0.182.0") or empty if unavailable
 */
function probeThreeNpmVersion(): string
{
    $raw = threeVendorHttpGet('https://registry.npmjs.org/three/latest', 5, ['Accept: application/json']);
    if ($raw === '') {
        return '';
    }

    $json = json_decode($raw, true);
    if (!is_array($json) || !isset($json['version']) || !is_string($json['version'])) {
        return '';
    }

    $version = trim($json['version']);

    return preg_match('/^\d+\.\d+\.\d+$/', $version) ? $version : '';
}

/**
 * Probe the unpkg CDN core build to extract the revision number.
 *
 * @param Closure $extractRevision Callback to extract revision from core JS head
 */
function probeThreeUnpkg(Closure $extractRevision, string $npmVersion): ?array
{
    $unpkgUrl = $npmVersion !== ''
        ? 'https://unpkg.com/three@' . rawurlencode($npmVersion) . '/build/three.core.min.js'
        : 'https://unpkg.com/three@latest/build/three.core.min.js';

    $coreHead = threeVendorHttpGet($unpkgUrl, 6, ['Range: bytes=0-1024', 'Accept: text/plain']);
    if ($coreHead === '') {
        return null;
    }

    $rev = $extractRevision($coreHead);
    if ($rev <= 0) {
        return null;
    }

    return ['ok' => true, 'source' => 'npm/unpkg', 'tag' => 'r' . $rev, 'revision' => $rev, 'version' => $npmVersion];
}

/**
 * Last-ditch fallback: scrape the GitHub HTML releases page for r### tag.
 */
function probeThreeGitHubHtml(string $npmVersion): ?array
{
    $html = threeVendorHttpGet('https://github.com/mrdoob/three.js/releases/latest', 6, ['Accept: text/html']);
    if ($html === '' || !preg_match('/\/mrdoob\/three\.js\/releases\/tag\/(r\d{2,4})/i', $html, $m)) {
        return null;
    }

    $tag = strtolower($m[1]);
    if (!preg_match('/^r(\d{2,4})$/', $tag, $mm)) {
        return null;
    }

    return ['ok' => true, 'source' => 'github-html', 'tag' => $tag, 'revision' => (int)$mm[1], 'version' => $npmVersion];
}

$fetchLatestThreeInfo = static function () use ($extractThreeRevisionFromCore): array {
    // Try GitHub API first (most reliable)
    $result = probeThreeGitHubApi();
    if ($result !== null) {
        return $result;
    }

    // GitHub API failed — fetch npm version, then try unpkg and HTML fallbacks
    $npmVersion = probeThreeNpmVersion();

    return probeThreeUnpkg($extractThreeRevisionFromCore, $npmVersion)
        ?? probeThreeGitHubHtml($npmVersion)
        ?? ['ok' => false, 'error' => 'Could not reach GitHub API (and fallbacks failed). Your host may block outbound HTTP requests.'];
};

$latest = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verifyCSRF();
    $action = (string)($_POST['action'] ?? '');

    if ($action === 'check_latest') {
        $latest = $fetchLatestThreeInfo();
        if (!empty($latest['ok'])) {
            $localRev = (int)$vendorInfo['revision'];
            $remoteRev = (int)($latest['revision'] ?? 0);
            $source = isset($latest['source']) ? (string)$latest['source'] : '';
            $versionNote = isset($latest['version']) && is_string($latest['version']) && $latest['version'] !== ''
                ? ' (npm ' . $latest['version'] . ')'
                : '';
            if ($localRev > 0 && $remoteRev > 0) {
                if ($remoteRev > $localRev) {
                    $message = "Update available: local r{$localRev}, latest {$latest['tag']} (r{$remoteRev}){$versionNote}.";
                } else {
                    $message = "Up to date: local r{$localRev}, latest {$latest['tag']} (r{$remoteRev}){$versionNote}.";
                }
            } else {
                $message = "Latest Three.js release: {$latest['tag']}{$versionNote}.";
            }
            if ($source !== '') {
                $message .= ' Source: ' . $source . '.';
            }
        } else {
            $error = (string)($latest['error'] ?? 'Could not check latest version.');
        }
    }

    if ($action === 'upload_three') {
        // HIGH RISK OPERATION (by nature): replacing executable JS.
        // We only allow replacing the exact two expected files.
        $maxBytes = 6 * 1024 * 1024;

        $moduleFile = $_FILES['three_module'] ?? null;
        $coreFile = $_FILES['three_core'] ?? null;

        if (!is_array($moduleFile) || !is_array($coreFile)) {
            $error = 'Both files are required.';
        } elseif (($moduleFile['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK || ($coreFile['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) {
            $error = 'Upload failed. Please try again.';
        } elseif ((int)($moduleFile['size'] ?? 0) <= 0 || (int)($coreFile['size'] ?? 0) <= 0) {
            $error = 'Uploaded files were empty.';
        } elseif ((int)$moduleFile['size'] > $maxBytes || (int)$coreFile['size'] > $maxBytes) {
            $error = 'Uploaded file is too large.';
        } else {
            $moduleTmp = (string)($moduleFile['tmp_name'] ?? '');
            $coreTmp = (string)($coreFile['tmp_name'] ?? '');

            $moduleHead = $readHead($moduleTmp, 16384);
            $coreHeadUploaded = $readHead($coreTmp, 4096);

            $uploadedRev = $extractThreeRevisionFromCore($coreHeadUploaded);
            if ($uploadedRev <= 0) {
                $error = 'Could not detect a Three.js revision from the uploaded core file.';
            } elseif (strpos($moduleHead, './three.core.min.js') === false) {
                $error = 'Uploaded module file does not appear to reference ./three.core.min.js.';
            } elseif ($vendorDir === '' || !is_dir($vendorDir) || !is_writable($vendorDir)) {
                $error = 'Vendor directory is not writable on the server.';
            } else {
                $targetModule = $vendorDir . DIRECTORY_SEPARATOR . 'three.module.min.js';
                $targetCore = $vendorDir . DIRECTORY_SEPARATOR . 'three.core.min.js';

                $tmpDir = dirname(__DIR__, 2) . '/data/cache';
                if (!is_dir($tmpDir)) {
                    @mkdir($tmpDir, 0755, true);
                }

                $tmpModulePath = $tmpDir . '/three.module.min.' . bin2hex(random_bytes(8)) . '.js';
                $tmpCorePath = $tmpDir . '/three.core.min.' . bin2hex(random_bytes(8)) . '.js';

                if (!@move_uploaded_file($moduleTmp, $tmpModulePath)) {
                    $error = 'Could not move uploaded module file.';
                } elseif (!@move_uploaded_file($coreTmp, $tmpCorePath)) {
                    @unlink($tmpModulePath);
                    $error = 'Could not move uploaded core file.';
                } else {
                    // Atomic-ish replace
                    $ok = true;
                    if (@rename($tmpModulePath, $targetModule) === false) {
                        $ok = false;
                        $error = 'Could not replace three.module.min.js.';
                    }
                    if ($ok && @rename($tmpCorePath, $targetCore) === false) {
                        $ok = false;
                        $error = 'Could not replace three.core.min.js.';
                    }

                    if ($ok) {
                        $message = 'Three.js vendor files updated successfully to r' . $uploadedRev . '.';
                        LogService::add('info', 'Admin updated Three.js vendor files', json_encode([
                            'revision' => $uploadedRev,
                            'files' => ['three.module.min.js', 'three.core.min.js'],
                        ]));
                    } else {
                        // Cleanup temp leftovers
                        @unlink($tmpModulePath);
                        @unlink($tmpCorePath);
                    }
                }
            }
        }

        // Refresh displayed info after upload
        clearstatcache(true, $corePath);
        clearstatcache(true, $modulePath);
        $vendorInfo['core_exists'] = is_file($corePath);
        $vendorInfo['module_exists'] = is_file($modulePath);
        $vendorInfo['core_mtime'] = is_file($corePath) ? (int)@filemtime($corePath) : 0;
        $vendorInfo['module_mtime'] = is_file($modulePath) ? (int)@filemtime($modulePath) : 0;
        $vendorInfo['core_size'] = is_file($corePath) ? (int)@filesize($corePath) : 0;
        $vendorInfo['module_size'] = is_file($modulePath) ? (int)@filesize($modulePath) : 0;
        $vendorInfo['core_sha384'] = $sha384($corePath);
        $vendorInfo['module_sha384'] = $sha384($modulePath);
        $coreHead = $readHead($corePath);
        $vendorInfo['revision'] = $extractThreeRevisionFromCore($coreHead);
    }
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header">
        <h1>Vendor Libraries</h1>
        <p>Check and manage locally hosted third-party JavaScript libraries (security maintenance).</p>
    </div>

    <?php if ($message): ?>
        <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <div class="card">
        <h2>Three.js (self-hosted)</h2>
        <p class="text-muted">Used by games and interactive pages that depend on Three.js. Newer Three.js releases are split into <code>three.module.min.js</code> + <code>three.core.min.js</code>.</p>

        <div class="form-row">
            <div class="form-group">
                <div class="text-muted">Local revision</div>
                <div><strong><?php echo $vendorInfo['revision'] > 0 ? 'r' . (int)$vendorInfo['revision'] : 'Unknown'; ?></strong></div>
            </div>
        </div>

        <div class="form-row">
            <div class="form-group">
                <div class="text-muted">Local files</div>
                <div>
                    <ul>
                        <li><code>three.module.min.js</code> — <?php echo $vendorInfo['module_exists'] ? e(number_format((int)$vendorInfo['module_size'])) . ' bytes' : 'missing'; ?></li>
                        <li><code>three.core.min.js</code> — <?php echo $vendorInfo['core_exists'] ? e(number_format((int)$vendorInfo['core_size'])) . ' bytes' : 'missing'; ?></li>
                    </ul>
                </div>
            </div>
        </div>

        <details>
            <summary>Show SHA-384 hashes (for integrity auditing)</summary>
            <div class="form-group">
                <div><code>three.module.min.js</code>: <code><?php echo e($vendorInfo['module_sha384'] ?: ''); ?></code></div>
                <div><code>three.core.min.js</code>: <code><?php echo e($vendorInfo['core_sha384'] ?: ''); ?></code></div>
            </div>
        </details>

        <form method="POST">
            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
            <input type="hidden" name="action" value="check_latest">
            <button type="submit" class="btn btn-outline">Check latest release</button>
        </form>

        <hr>

        <h3>Update (manual upload)</h3>
        <p class="text-muted">This replaces executable JavaScript on your site. Only do this from a trusted network/IP, and only using official sources.</p>

        <form method="POST" enctype="multipart/form-data">
            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
            <input type="hidden" name="action" value="upload_three">

            <div class="form-group">
                <label for="three-module">three.module.min.js</label>
                <input id="three-module" type="file" name="three_module" accept=".js" class="form-control" required>
            </div>

            <div class="form-group">
                <label for="three-core">three.core.min.js</label>
                <input id="three-core" type="file" name="three_core" accept=".js" class="form-control" required>
            </div>

            <button type="submit" class="btn btn-primary">Upload & Replace</button>
        </form>

        <p class="text-muted">Official download source: <a href="https://github.com/mrdoob/three.js/releases" target="_blank" rel="noopener">three.js releases</a></p>
    </div>
</div>

<?php AdminLayout::renderFooter(); ?>
