<?php
/**
 * Admin Panel - Tickets
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Auth\AdminAuth;
use NewSite\User\UserService;
use NewSite\Upload\UploadService;
use NewSite\Upload\ImageUrlHelper;
use NewSite\Logging\LogService;
use NewSite\Admin\AdminLayout;
use NewSite\Database\DatabaseManager;

AdminAuth::requireLogin();

const ADMIN_TICKET_TRANSCRIPT_DIVIDER = "==========================================================\n";

$db = DatabaseManager::getWriteConnection();
$message = '';
$error = '';
$adminUserId = (int)($_SESSION['admin_user_id'] ?? 0);
$adminUsername = 'Admin';
if ($adminUserId > 0) {
    $stmt = $db->prepare("SELECT username FROM users WHERE id = ? LIMIT 1");
    $stmt->execute([$adminUserId]);
    $adminUsername = (string)$stmt->fetchColumn() ?: 'Admin';
}

$flash = $_SESSION['admin_flash_tickets'] ?? null;
if (is_array($flash)) {
    $message = $flash['message'] ?? '';
    $error = $flash['error'] ?? '';
    unset($_SESSION['admin_flash_tickets']);
}

// Handle chat transcript download (GDPR compliance)
if (isset($_GET['download']) && isset($_GET['user'])) {
    verifyCSRF();
    $downloadUserId = (int)$_GET['user'];
    if ($downloadUserId > 0) {
        // Get user email
        $stmt = $db->prepare("SELECT email FROM site_users WHERE id = ? LIMIT 1");
        $stmt->execute([$downloadUserId]);
        $userEmail = $stmt->fetchColumn() ?: 'unknown';

        // Get all messages for this user
        $stmt = $db->prepare("SELECT um.*, su.email FROM user_messages um JOIN site_users su ON su.id = um.user_id WHERE um.user_id = ? AND um.message_type = 'ticket' ORDER BY um.created_at ASC");
        $stmt->execute([$downloadUserId]);
        $messages = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (!empty($messages)) {
            // Build transcript
            $transcript = ADMIN_TICKET_TRANSCRIPT_DIVIDER;
            $transcript .= "SUPPORT CHAT TRANSCRIPT\n";
            $transcript .= ADMIN_TICKET_TRANSCRIPT_DIVIDER . "\n";
            $transcript .= "User Email: " . $userEmail . "\n";
            $transcript .= "User ID: " . $downloadUserId . "\n";
            $transcript .= "Export Date: " . date('Y-m-d H:i:s') . "\n";
            $transcript .= "Total Messages: " . count($messages) . "\n";
            $transcript .= "\n" . ADMIN_TICKET_TRANSCRIPT_DIVIDER . "\n";

            foreach ($messages as $msg) {
                $sender = ($msg['sender_type'] === 'admin') ? 'ADMIN' : 'USER';
                $senderLabel = !empty($msg['sender_label']) ? " ({$msg['sender_label']})" : '';
                $time = !empty($msg['created_at']) ? date('Y-m-d H:i:s', (int)$msg['created_at']) : 'Unknown';
                $body = trim($msg['body'] ?? '');
                $hasImage = !empty($msg['image_path']) ? ' [Image Attached]' : '';

                $transcript .= "[{$time}] {$sender}{$senderLabel}{$hasImage}\n";
                $transcript .= "------------------------------------------------------------\n";
                if ($body !== '') {
                    $transcript .= $body . "\n";
                }
                $transcript .= "\n";
            }

            $transcript .= ADMIN_TICKET_TRANSCRIPT_DIVIDER;
            $transcript .= "END OF TRANSCRIPT\n";
            $transcript .= ADMIN_TICKET_TRANSCRIPT_DIVIDER;

            // Log the download action
            LogService::add('info', 'Chat transcript downloaded (GDPR)', json_encode([
                'user_id' => $downloadUserId,
                'user_email' => $userEmail,
                'admin_id' => $adminUserId,
                'message_count' => count($messages)
            ]));

            // Send as download
            $filename = 'chat_transcript_' . preg_replace('/[^a-zA-Z0-9]/', '_', $userEmail) . '_' . date('Y-m-d') . '.txt';
            header('Content-Type: text/plain; charset=utf-8');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            header('Content-Length: ' . strlen($transcript));
            header('Cache-Control: no-cache, no-store, must-revalidate');
            echo $transcript;
            exit;
        }
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verifyCSRF();
    $action = $_POST['action'] ?? '';
    try {
        if ($action === 'send_ticket') {
            $email = trim($_POST['user_email'] ?? '');
            $body = trim($_POST['body'] ?? '');

            if ($email === '' || $body === '') {
                $error = 'Email and message are required.';
            } else {
                $user = UserService::getByEmail($email);
                if (!$user) {
                    $error = 'No user found with that email.';
                } else {
                    $subject = 'Ticket';
                    $stmt = $db->prepare("INSERT INTO user_messages (user_id, subject, body, sender_type, sender_label, message_type, is_read, created_at) VALUES (?, ?, ?, 'admin', ?, 'ticket', 0, ?)");
                    $stmt->execute([
                        (int)$user['id'],
                        $subject,
                        $body,
                        $adminUsername,
                        time()
                    ]);
                    \NewSite\Cache\CachedQueries::invalidateNotificationCache((int)$user['id']);
                    LogService::add('info', 'Ticket sent to user', json_encode(['user_id' => (int)$user['id'], 'admin_id' => $adminUserId]));
                    $message = 'Ticket sent to user.';
                }
            }
        } elseif ($action === 'reply_ticket') {
            $userId = (int)($_POST['user_id'] ?? 0);
            $replyToId = (int)($_POST['reply_to_id'] ?? 0);
            $body = trim($_POST['reply_body'] ?? '');

            // Handle image upload
            $imagePath = null;
            if (!empty($_FILES['chat_image']['tmp_name']) && $_FILES['chat_image']['error'] === UPLOAD_ERR_OK) {
                $imagePath = UploadService::uploadChatImage($_FILES['chat_image']);
                if (is_array($imagePath) && isset($imagePath['error'])) {
                    $error = $imagePath['error'];
                    $imagePath = null;
                }
            }

            // Allow message with just image (no text)
            if ($userId <= 0 || ($body === '' && empty($imagePath))) {
                $error = $error ?? 'Reply message or image is required.';
            } else {
                $subject = 'Ticket';
                if ($replyToId > 0) {
                    $stmt = $db->prepare("SELECT subject FROM user_messages WHERE id = ? AND user_id = ? AND message_type = 'ticket' LIMIT 1");
                    $stmt->execute([$replyToId, $userId]);
                    $subjectValue = trim((string)$stmt->fetchColumn());
                    if ($subjectValue !== '') {
                        $subject = $subjectValue;
                    }
                }
                $stmt = $db->prepare("INSERT INTO user_messages (user_id, subject, body, sender_type, sender_label, reply_to_id, message_type, is_read, created_at, image_path) VALUES (?, ?, ?, 'admin', ?, ?, 'ticket', 0, ?, ?)");
                $stmt->execute([
                    $userId,
                    $subject,
                    $body,
                    $adminUsername,
                    $replyToId > 0 ? $replyToId : null,
                    time(),
                    $imagePath
                ]);
                \NewSite\Cache\CachedQueries::invalidateNotificationCache($userId);

                // Push notification to user about ticket reply
                \NewSite\Push\PushNotification::newMessage($userId, 'Support Reply: ' . $subject, $body);

                LogService::add('info', 'Admin replied to ticket', json_encode(['user_id' => $userId, 'reply_to' => $replyToId, 'admin_id' => $adminUserId, 'has_image' => !empty($imagePath)]));
                $message = 'Reply sent.';
            }
        } elseif ($action === 'delete_ticket_thread') {
            $userId = (int)($_POST['user_id'] ?? 0);
            if ($userId > 0) {
                // Get all image paths before deleting
                $stmt = $db->prepare("SELECT image_path FROM user_messages WHERE user_id = ? AND (message_type = 'ticket' OR subject = 'Ticket') AND image_path IS NOT NULL AND image_path != ''");
                $stmt->execute([$userId]);
                $imagePaths = $stmt->fetchAll(PDO::FETCH_COLUMN);

                // Delete the messages
                $stmt = $db->prepare("DELETE FROM user_messages WHERE user_id = ? AND (message_type = 'ticket' OR subject = 'Ticket')");
                $stmt->execute([$userId]);

                // Delete the image files
                $deletedFiles = 0;
                foreach ($imagePaths as $imagePath) {
                    $fullPath = DATA_PATH . '/' . $imagePath;
                    if (is_file($fullPath)) {
                        @unlink($fullPath);
                        $deletedFiles++;
                    }
                }

                $message = 'Ticket deleted.';
                LogService::add('info', 'Ticket thread deleted', json_encode(['user_id' => $userId, 'admin_id' => $adminUserId, 'rows' => $stmt->rowCount(), 'files_deleted' => $deletedFiles]));
            }
        } elseif ($action === 'delete_ticket_message') {
            $messageId = (int)($_POST['message_id'] ?? 0);
            if ($messageId > 0) {
                // Get image path before deleting
                $stmt = $db->prepare("SELECT image_path FROM user_messages WHERE id = ? AND sender_type = 'admin' AND message_type = 'ticket'");
                $stmt->execute([$messageId]);
                $imagePath = $stmt->fetchColumn();

                // Delete the message
                $stmt = $db->prepare("DELETE FROM user_messages WHERE id = ? AND sender_type = 'admin' AND message_type = 'ticket'");
                $stmt->execute([$messageId]);

                // Delete the image file if exists
                if ($imagePath) {
                    $fullPath = DATA_PATH . '/' . $imagePath;
                    if (is_file($fullPath)) {
                        @unlink($fullPath);
                    }
                }

                $message = 'Ticket message deleted.';
                LogService::add('info', 'Ticket message deleted', json_encode(['message_id' => $messageId, 'admin_id' => $adminUserId, 'had_image' => !empty($imagePath)]));
            }
        }
    } catch (PDOException $e) {
        $error = 'Failed to send ticket.';
        LogService::add('error', 'Ticket admin error: ' . $e->getMessage());
    }

    if ($error === '') {
        $_SESSION['admin_flash_tickets'] = ['message' => $message, 'error' => $error];
        header('Location: /admin/tickets.php');
        exit;
    }
}

$selectedUserId = (int)($_GET['user'] ?? 0);
$stmt = $db->query("SELECT um.*, su.email FROM user_messages um JOIN site_users su ON su.id = um.user_id WHERE um.message_type = 'ticket' ORDER BY um.created_at DESC LIMIT 200");
$ticketMessages = $stmt->fetchAll();

$threads = [];
foreach ($ticketMessages as $row) {
    $uid = (int)($row['user_id'] ?? 0);
    if ($uid <= 0) {
        continue;
    }
    if (!isset($threads[$uid])) {
        $threads[$uid] = [
            'user_id' => $uid,
            'email' => $row['email'] ?? '',
            'subject' => $row['subject'] ?? '',
            'preview' => $row['body'] ?? '',
            'last_time' => (int)($row['created_at'] ?? 0),
            'unread' => 0
        ];
    }
    // Count unread messages from users (using admin_read column)
    if (($row['sender_type'] ?? '') === 'user' && (empty($row['admin_read']) || $row['admin_read'] === null)) {
        $threads[$uid]['unread']++;
    }
}

$conversation = [];
if ($selectedUserId > 0) {
    foreach ($ticketMessages as $row) {
        if ((int)($row['user_id'] ?? 0) === $selectedUserId) {
            $conversation[] = $row;
        }
    }
    $conversation = array_reverse($conversation);

    // Mark user messages as read by admin when viewing the conversation
    $stmt = $db->prepare("UPDATE user_messages SET admin_read = 1 WHERE user_id = ? AND message_type = 'ticket' AND sender_type = 'user' AND (admin_read = 0 OR admin_read IS NULL)");
    $stmt->execute([$selectedUserId]);
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header">
        <h1>Tickets</h1>
        <p>Send direct messages to users and reply to incoming messages.</p>
    </div>

    <?php if ($message): ?>
        <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <div class="card">
        <div class="card-header">
            <h3>Send Ticket</h3>
        </div>
        <div class="card-body">
            <form method="POST" action="">
                <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                <input type="hidden" name="action" value="send_ticket">
                <div class="form-group">
                    <label for="user_email">User Email</label>
                    <input type="email" id="user_email" name="user_email" class="form-control" placeholder="user@example.com" required>
                </div>
                <div class="form-group">
                    <label for="ticket_body">Message</label>
                    <textarea id="ticket_body" name="body" class="form-control" rows="4" placeholder="Write your message..." required></textarea>
                </div>
                <button type="submit" class="btn btn-primary">Send Ticket</button>
                <div class="text-muted mt-1">Message will appear in the user message center.</div>
            </form>
        </div>
    </div>

    <div class="card">
        <div class="card-header">
            <h3>Open Tickets</h3>
        </div>
        <div class="card-body">
            <?php if (empty($threads)): ?>
                <div class="text-muted">No open tickets yet.</div>
            <?php else: ?>
                <div class="ticket-layout <?php echo $selectedUserId > 0 ? 'is-collapsed' : ''; ?>" data-ticket-chat-admin data-selected-user-id="<?php echo (int)$selectedUserId; ?>">
                    <div class="ticket-list">
                        <?php foreach ($threads as $thread): ?>
                            <?php
                            $isActive = (int)$thread['user_id'] === $selectedUserId;
                            $preview = trim((string)($thread['preview'] ?? ''));
                            $preview = $preview !== '' ? mb_substr($preview, 0, 80) : '';
                            ?>
                            <div class="ticket-list-item <?php echo $isActive ? 'is-active' : ''; ?> <?php echo !empty($thread['unread']) ? 'has-unread' : ''; ?>" data-ticket-item>
                                <a class="ticket-list-link" href="/admin/tickets.php?user=<?php echo (int)$thread['user_id']; ?>" data-ticket-link>
                                    <div class="ticket-list-title">
                                        <?php echo e($thread['email'] ?? ''); ?>
                                        <?php if (!empty($thread['unread'])): ?>
                                            <span class="ticket-unread-badge"><?php echo (int)$thread['unread']; ?></span>
                                        <?php endif; ?>
                                    </div>
                                    <div class="ticket-list-subject"><?php echo e($thread['subject'] ?: 'Ticket'); ?></div>
                                    <?php if ($preview !== ''): ?>
                                        <div class="ticket-list-preview"><?php echo e($preview); ?><?php echo strlen($preview) >= 80 ? '…' : ''; ?></div>
                                    <?php endif; ?>
                                    <div class="ticket-list-time"><?php echo !empty($thread['last_time']) ? date('Y-m-d H:i', (int)$thread['last_time']) : '-'; ?></div>
                                </a>
                                <form method="POST" action="" class="ticket-list-delete" data-confirm="Delete this ticket?">
                                    <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                                    <input type="hidden" name="action" value="delete_ticket_thread">
                                    <input type="hidden" name="user_id" value="<?php echo (int)$thread['user_id']; ?>">
                                    <button type="submit" class="btn btn-outline btn-sm">Delete</button>
                                </form>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    <div class="ticket-chat" data-ticket-chat-box data-ticket-user-id="<?php echo (int)$selectedUserId; ?>">
                        <?php if (empty($conversation)): ?>
                            <div class="text-muted">Select a ticket to view the conversation.</div>
                        <?php else: ?>
                            <?php
                            $threadSubject = '';
                            $replyToId = 0;
                            $threadRootId = 0;
                            foreach ($conversation as $row) {
                                if ($threadSubject === '' && !empty($row['subject'])) {
                                    $threadSubject = (string)$row['subject'];
                                }
                                if ($threadRootId === 0 && empty($row['reply_to_id'])) {
                                    $threadRootId = (int)($row['id'] ?? 0);
                                }
                                $replyToId = (int)($row['id'] ?? 0);
                            }
                            if ($threadRootId === 0 && $replyToId > 0) {
                                $threadRootId = $replyToId;
                            }
                            $activeEmail = $threads[$selectedUserId]['email'] ?? '';
                            ?>
                            <div class="ticket-chat-header">
                                <div>
                                    <h4><?php echo e($activeEmail); ?></h4>
                                    <div class="ticket-chat-subject"><?php echo e($threadSubject ?: 'Ticket'); ?></div>
                                </div>
                                <div class="ticket-chat-header-actions">
                                    <a href="/admin/tickets.php?user=<?php echo (int)$selectedUserId; ?>&amp;download=1&amp;csrf_token=<?php echo getCsrfToken(); ?>" class="btn btn-outline btn-sm" title="Download chat transcript (GDPR)">
                                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="16" height="16" class="ticket-download-icon">
                                            <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                                            <polyline points="7 10 12 15 17 10"></polyline>
                                            <line x1="12" y1="15" x2="12" y2="3"></line>
                                        </svg>
                                        Download
                                    </a>
                                    <button type="button" class="btn btn-outline ticket-toggle-list" data-ticket-toggle-list>All Tickets</button>
                                </div>
                            </div>
                            <div class="ticket-chat-body" data-ticket-chat-body data-ticket-last-id="<?php echo !empty($conversation) ? (int)($conversation[count($conversation) - 1]['id'] ?? 0) : 0; ?>">
                                <?php foreach ($conversation as $row): ?>
                                    <?php $isAdmin = ($row['sender_type'] ?? 'admin') === 'admin'; ?>
                                    <div class="ticket-bubble <?php echo $isAdmin ? 'ticket-bubble--admin' : 'ticket-bubble--user'; ?>" data-ticket-message-id="<?php echo (int)$row['id']; ?>">
                                        <div class="ticket-bubble-meta">
                                            <?php echo $isAdmin ? 'Admin' : 'User'; ?>
                                            · <?php echo !empty($row['created_at']) ? date('Y-m-d H:i', (int)$row['created_at']) : '-'; ?>
                                            <?php if ($isAdmin && !empty($row['is_read'])): ?>
                                                <span class="read-receipt read-receipt--read" title="Read"><svg viewBox="0 0 16 16" fill="currentColor"><path d="M13.854 3.646a.5.5 0 0 1 0 .708l-7 7a.5.5 0 0 1-.708 0l-3.5-3.5a.5.5 0 1 1 .708-.708L6.5 10.293l6.646-6.647a.5.5 0 0 1 .708 0z"/></svg></span>
                                            <?php endif; ?>
                                        </div>
                                        <?php if (!empty($row['image_path'])): ?>
                                            <div class="ticket-bubble-image">
                                                <img src="<?php echo e(ImageUrlHelper::getChatImageUrl($row['image_path'])); ?>" alt="Attachment" class="chat-image" loading="lazy">
                                            </div>
                                        <?php endif; ?>
                                        <div class="ticket-bubble-text"><?php echo e($row['body'] ?? ''); ?></div>
                                        <?php if ($isAdmin): ?>
                                            <form method="POST" action="" class="ticket-delete-form" data-confirm="Delete this ticket message?">
                                                <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                                                <input type="hidden" name="action" value="delete_ticket_message">
                                                <input type="hidden" name="message_id" value="<?php echo (int)$row['id']; ?>">
                                                <button type="submit" class="btn btn-outline btn-sm">Delete</button>
                                            </form>
                                        <?php endif; ?>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                            <form method="POST" action="" class="ticket-reply-form" enctype="multipart/form-data">
                                <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                                <input type="hidden" name="action" value="reply_ticket">
                                <input type="hidden" name="user_id" value="<?php echo (int)$selectedUserId; ?>">
                                <input type="hidden" name="reply_to_id" value="<?php echo (int)$threadRootId; ?>">
                                <div class="form-group">
                                    <label for="ticket-reply-body">Message</label>
                                    <textarea id="ticket-reply-body" name="reply_body" class="form-control" rows="3" placeholder="Write a reply..."></textarea>
                                </div>
                                <div class="form-group">
                                    <label for="admin-ticket-image" class="upload-label">
                                        <span class="upload-icon">📷</span> Attach Image (max 5MB)
                                    </label>
                                    <input type="file" id="admin-ticket-image" name="chat_image" accept="image/jpeg,image/png,image/gif,image/webp" class="form-control-file">
                                    <div class="upload-preview is-hidden" data-upload-preview>
                                        <img src="" alt="Preview" class="upload-preview-img">
                                        <button type="button" class="upload-preview-remove" data-remove-preview>&times;</button>
                                    </div>
                                </div>
                                <button type="submit" class="btn btn-primary">Send Reply</button>
                            </form>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php AdminLayout::renderFooter(); ?>
