<?php
/**
 * Admin Panel - Live Theme Editor (Shopify-style)
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Admin\AdminLayout;
use NewSite\Auth\AdminAuth;
use NewSite\Config\SetupService;
use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Logging\LogService;
use NewSite\Settings\SettingsService;

AdminAuth::requireLogin();

$db = DatabaseManager::getWriteConnection();
$systemPageSlugs = SetupService::getSystemPageSlugs();

/**
 * Ensure a known system page exists and return its page ID.
 */
$ensureSystemPage = static function (
    PDO $db,
    array $systemPageSlugs,
    array $seed
): int {
    $pageId = 0;
    $slug = strtolower(trim((string)($seed['slug'] ?? '')));
    $title = (string)($seed['title'] ?? '');
    $content = (string)($seed['content'] ?? '');
    $metaTitle = (string)($seed['meta_title'] ?? '');
    $metaDescription = (string)($seed['meta_description'] ?? '');
    $isPublished = (int)($seed['is_published'] ?? 1);

    if ($slug !== '' && $title !== '' && in_array($slug, $systemPageSlugs, true)) {
        $stmt = $db->prepare("SELECT id FROM pages WHERE slug = ? LIMIT 1");
        $stmt->execute([$slug]);
        $row = $stmt->fetch();
        if ($row) {
            $pageId = (int)$row['id'];
        } else {
            $insertStmt = $db->prepare("INSERT INTO pages (title, slug, content, meta_title, meta_description, is_published) VALUES (?, ?, ?, ?, ?, ?)");
            $insertStmt->execute([$title, $slug, $content, $metaTitle, $metaDescription, $isPublished]);
            $pageId = (int)DbHelper::lastInsertId($db, 'pages');
        }
    }

    return $pageId;
};

// Ensure key system pages needed by theme editor exist.
$ensureSystemPage($db, $systemPageSlugs, [
    'slug' => 'contact',
    'title' => 'Contact',
    'content' => '',
    'meta_title' => 'Contact',
    'meta_description' => 'Contact page',
    'is_published' => 1
]);
$ensureSystemPage($db, $systemPageSlugs, [
    'slug' => 'cart',
    'title' => 'Cart',
    'content' => '',
    'meta_title' => 'Cart',
    'meta_description' => 'Cart page',
    'is_published' => 1
]);
$ensureSystemPage($db, $systemPageSlugs, [
    'slug' => 'checkout',
    'title' => 'Checkout',
    'content' => '',
    'meta_title' => 'Checkout',
    'meta_description' => 'Checkout page',
    'is_published' => 1
]);
$ensureSystemPage($db, $systemPageSlugs, [
    'slug' => 'games',
    'title' => 'Mini-Games',
    'content' => '',
    'meta_title' => 'Mini-Games',
    'meta_description' => 'Play free browser mini-games.',
    'is_published' => 1
]);

// Ensure Product page exists (system page for product detail routing)
$productPageId = $ensureSystemPage(
    $db,
    $systemPageSlugs,
    [
        'slug' => 'product',
        'title' => 'Product',
        'content' => '',
        'meta_title' => 'Product',
        'meta_description' => 'Product details page',
        'is_published' => 1
    ]
);

if (!empty($productPageId)) {
    $sectionStmt = $db->prepare("SELECT id FROM sections WHERE page_id = ? AND section_type = 'products' LIMIT 1");
    $sectionStmt->execute([$productPageId]);
    if (!$sectionStmt->fetch()) {
        $defaultSectionSettings = [
            'section_title' => 'Products',
            'title_align' => 'center',
            'product_id' => '',
            'accordion_style' => 'separated',
            'variant_style' => 'select',
            'show_media' => true,
            'show_description' => true,
            'show_price' => true
        ];
        $orderStmt = $db->prepare("SELECT MAX(sort_order) FROM sections WHERE page_id = ?");
        $orderStmt->execute([$productPageId]);
        $maxOrder = (int)$orderStmt->fetchColumn();
        $insertSection = $db->prepare("INSERT INTO sections (page_id, section_type, settings, sort_order) VALUES (?, ?, ?, ?)");
        $insertSection->execute([$productPageId, 'products', json_encode($defaultSectionSettings), $maxOrder + 1]);
    }
}

// Ensure DMCA page exists (system page)
$defaultDmcaContent = '<h2>DMCA Takedown Policy</h2><p>This page explains how to submit a DMCA takedown notice or counter-notice.</p><h3>How to Submit a Takedown Notice</h3><p>Please provide the following:</p><ul><li>Your full legal name, address, phone number, and email address.</li><li>Identification of the copyrighted work you claim has been infringed.</li><li>Exact URL(s) or location(s) of the allegedly infringing content.</li><li>A statement that you have a good-faith belief the use is not authorized.</li><li>A statement made under penalty of perjury that your notice is accurate and you are the rights holder or authorized to act on their behalf.</li><li>Your physical or electronic signature.</li></ul><h3>Where to Send Notices</h3><p>Email: {CONTACT_EMAIL}</p><h3>Counter-Notice</h3><p>If you believe content was removed in error, you may submit a counter-notice with your contact details, the removed content location, a statement under penalty of perjury that removal was a mistake, and consent to jurisdiction.</p>';
$ensureSystemPage(
    $db,
    $systemPageSlugs,
    [
        'slug' => 'dmca',
        'title' => 'DMCA',
        'content' => $defaultDmcaContent,
        'meta_title' => 'DMCA',
        'meta_description' => 'Digital Millennium Copyright Act (DMCA) policy',
        'is_published' => 1
    ]
);

// Ensure Easy Media AI page exists (system page)
$ensureSystemPage(
    $db,
    $systemPageSlugs,
    [
        'slug' => 'easy-media-ai',
        'title' => 'Easy Media AI',
        'content' => '',
        'meta_title' => 'Easy Media AI',
        'meta_description' => 'Private local media manager with AI tagging',
        'is_published' => 1
    ]
);

// Ensure Text Finder & Replacer page exists (system page)
$ensureSystemPage(
    $db,
    $systemPageSlugs,
    [
        'slug' => 'text-finder-replacer',
        'title' => 'Text Finder & Replacer',
        'content' => '<h2>Text Finder &amp; Replacer</h2><p>Find and replace text in pasted content, text files, or zip archives containing text files.</p><p>Open the tool at <a href="/text-finder-replacer">/text-finder-replacer</a>.</p>',
        'meta_title' => 'Text Finder & Replacer',
        'meta_description' => 'Find and replace text in pasted content and uploaded text files.',
        'is_published' => 1
    ]
);

// Get page to edit
$pageId = isset($_GET['page']) ? (int)$_GET['page'] : null;

// Get all pages
$pages = $db->query("SELECT id, title, slug FROM pages ORDER BY title")->fetchAll();

// If no page specified, use home
if (!$pageId && !empty($pages)) {
    foreach ($pages as $p) {
        if ($p['slug'] === 'home') {
            $pageId = $p['id'];
            break;
        }
    }
    if (!$pageId) {
        $pageId = $pages[0]['id'];
    }
}

// Get current page
$currentPage = null;
if ($pageId) {
    $stmt = $db->prepare("SELECT * FROM pages WHERE id = ?");
    $stmt->execute([$pageId]);
    $currentPage = $stmt->fetch();
}

// Get page sections
$sections = [];
if ($pageId) {
    $stmt = $db->prepare("SELECT * FROM sections WHERE page_id = ? ORDER BY sort_order");
    $stmt->execute([$pageId]);
    $sections = $stmt->fetchAll();
}

// Get theme settings
$themeSettings = SettingsService::getAllTheme();

// Custom fonts
$customFonts = json_decode(SettingsService::get('custom_fonts', '[]'), true);
if (!is_array($customFonts)) {
    $customFonts = [];
}

// Get all menus for menu selection dropdowns
$allMenus = $db->query("SELECT id, name, location FROM menus ORDER BY sort_order ASC, id ASC")->fetchAll();

// Products available for current page (used in Products section settings)
$productOptions = [];
if ($pageId) {
    $stmt = $db->prepare("SELECT id, name, price FROM products WHERE is_active = 1 AND page_id = ? ORDER BY created_at DESC");
    $stmt->execute([$pageId]);
    $productOptions = $stmt->fetchAll();
}

// Products available for split slideshow (all active products)
$slideProductOptions = $db->query("SELECT id, name, price FROM products WHERE is_active = 1 ORDER BY created_at DESC")->fetchAll();

const SECTION_NAME_IMAGE_BANNER = 'Image Banner';
const SECTION_NAME_PRODUCTS_LIST = 'Products List';
const SECTION_NAME_COLLECTIONS_LIST = 'Collections List';
const CLASS_IS_HIDDEN = ' is-hidden';

// Available section types
$sectionTypes = [
    [
        'type' => 'image_banner',
        'name' => SECTION_NAME_IMAGE_BANNER,
        'description' => 'Compact banner with background images and text overlay',
        'icon' => '🎯'
    ],
    [
        'type' => 'split_slideshow',
        'name' => 'Split Slideshow',
        'description' => 'Full-screen slideshow with text and image',
        'icon' => '🖼️'
    ],
    [
        'type' => 'text_box',
        'name' => 'Text Box',
        'description' => 'Rich text content for policies or static pages',
        'icon' => '📝'
    ],
    [
        'type' => 'testimonials',
        'name' => 'Testimonials',
        'description' => 'Star ratings, titles, and quotes from visitors',
        'icon' => '💬'
    ],
    [
        'type' => 'products',
        'name' => 'Product',
        'description' => 'Single product details page section',
        'icon' => '🛍️'
    ],
    [
        'type' => 'products_list',
        'name' => SECTION_NAME_PRODUCTS_LIST,
        'description' => 'Show all products in a grid',
        'icon' => '🧩'
    ],
    [
        'type' => 'collections_list',
        'name' => SECTION_NAME_COLLECTIONS_LIST,
        'description' => 'Show all collections in a grid',
        'icon' => '🗂️'
    ]
];

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax'])) {
    verifyCSRF();
    header('Content-Type: application/json');

    $action = $_POST['action'] ?? '';

    try {
        switch ($action) {
            case 'add_section':
                $type = $_POST['type'] ?? '';
                $pageIdPost = (int)($_POST['page_id'] ?? 0);

                // Default settings based on type
                $defaultSettings = [];
                if ($type === 'image_banner') {
                    $defaultSettings = [
                        'images' => [
                            ['image' => '/assets/images/placeholder-1.svg']
                        ],
                        'title' => 'Banner Title',
                        'subtitle' => 'Add a subtitle or description here',
                        'title_size' => '48',
                        'subtitle_size' => '18',
                        'text_align' => 'center',
                        'text_position' => 'center',
                        'height' => '300',
                        'spacing_top' => '0',
                        'spacing_bottom' => '0',
                        'image_position' => 'center',
                        'autoplay' => true,
                        'interval' => '5',
                        'overlay_opacity' => '50',
                        'link_enabled' => false,
                        'link_type' => 'custom',
                        'link_url' => '',
                        'link_page_id' => '',
                        'link_new_tab' => false
                    ];
                } elseif ($type === 'split_slideshow') {
                    $defaultSettings = [
                        'slides' => [
                            [
                                'title' => 'Welcome to Your Website',
                                'description' => 'Create stunning websites with our powerful theme builder.',
                                'button_text' => 'Get Started',
                                'button_link_type' => 'custom',
                                'button_url' => '#',
                                'button_product_id' => '',
                                'media_source' => 'manual',
                                'image' => '/assets/images/placeholder-1.svg'
                            ]
                        ],
                        'autoplay' => true,
                        'show_arrows' => true,
                        'show_dots' => true
                    ];
                } elseif ($type === 'text_box') {
                    $defaultSettings = [
                        'title' => 'Text Section',
                        'content' => '<p>Add your content here. This section is perfect for policies, FAQs, or long-form text.</p>',
                        'title_align' => 'left',
                        'text_align' => 'left',
                        'title_size' => '32',
                        'text_size' => '16'
                    ];
                } elseif ($type === 'testimonials') {
                    $defaultSettings = [
                        'section_title' => 'Testimonials',
                        'card_width' => '320',
                        'auto_scroll' => true,
                        'auto_interval' => '5',
                        'testimonials' => [
                            [
                                'title' => 'Amazing experience',
                                'author' => 'Jane Doe',
                                'comment' => 'This site exceeded my expectations. Clean, fast, and beautiful.',
                                'stars' => 5
                            ]
                        ]
                    ];
                } elseif ($type === 'products') {
                    $defaultSettings = [
                        'section_title' => 'Products',
                        'title_align' => 'center',
                        'product_id' => '',
                        'accordion_style' => 'separated',
                        'content_align' => 'left',
                        'variant_style' => 'select',
                        'show_media' => true,
                        'show_description' => true,
                        'show_price' => true,
                        'show_similar_products' => false,
                        'similar_title' => 'Similar Products',
                        'similar_title_align' => 'center',
                        'similar_columns' => '4',
                        'similar_show_price' => true,
                        'similar_show_description' => false,
                        'similar_hover_swap_media' => true,
                        'similar_max_products' => '8'
                    ];
                } elseif ($type === 'products_list') {
                    $defaultSettings = [
                        'section_title' => 'Products',
                        'title_align' => 'center',
                        'columns' => '3',
                        'show_description' => true,
                        'show_title' => true,
                        'show_price' => true
                    ];
                } elseif ($type === 'collections_list') {
                    $defaultSettings = [
                        'section_title' => 'Collections',
                        'title_align' => 'center',
                        'columns' => '3',
                        'layout_mode' => 'carousel',
                        'show_description' => true,
                        'show_image' => true
                    ];
                }

                // Get max sort order
                $stmt = $db->prepare("SELECT MAX(sort_order) FROM sections WHERE page_id = ?");
                $stmt->execute([$pageIdPost]);
                $maxOrder = (int)$stmt->fetchColumn();

                $stmt = $db->prepare("INSERT INTO sections (page_id, section_type, settings, sort_order) VALUES (?, ?, ?, ?)");
                $stmt->execute([$pageIdPost, $type, json_encode($defaultSettings), $maxOrder + 1]);

                $newId = DbHelper::lastInsertId($db, 'sections');

                echo json_encode(['success' => true, 'id' => $newId]);
                exit;

            case 'remove_section':
                $sectionId = (int)($_POST['section_id'] ?? 0);
                $stmt = $db->prepare("DELETE FROM sections WHERE id = ?");
                $stmt->execute([$sectionId]);
                echo json_encode(['success' => true]);
                exit;

            case 'reorder_sections':
                $order = $_POST['order'] ?? [];
                foreach ($order as $index => $sectionId) {
                    $stmt = $db->prepare("UPDATE sections SET sort_order = ? WHERE id = ?");
                    $stmt->execute([$index, $sectionId]);
                }
                echo json_encode(['success' => true]);
                exit;

            case 'update_section':
                $sectionId = (int)($_POST['section_id'] ?? 0);
                $settingsRaw = $_POST['settings'] ?? '{}';

                // Settings come as JSON string from JavaScript
                $settings = is_string($settingsRaw) ? json_decode($settingsRaw, true) : $settingsRaw;

                // Ensure settings is an array
                if (!is_array($settings)) {
                    $settings = [];
                }

                // Clean up slides array - remove empty indices and reindex
                if (isset($settings['slides']) && is_array($settings['slides'])) {
                    $settings['slides'] = array_values(array_filter($settings['slides'], function ($slide) {
                        if (!is_array($slide)) {
                            return false;
                        }
                        return !empty($slide['title']) ||
                            !empty($slide['description']) ||
                            !empty($slide['button_text']) ||
                            !empty($slide['button_url']) ||
                            !empty($slide['image']) ||
                            !empty($slide['button_product_id']);
                    }));
                }

                // Clean up images array for image banner - remove empty entries and reindex
                if (isset($settings['images']) && is_array($settings['images'])) {
                    $settings['images'] = array_values(array_filter($settings['images'], function ($img) {
                        return is_array($img) && !empty($img['image']);
                    }));
                }

                // Clean up testimonials array
                if (isset($settings['testimonials']) && is_array($settings['testimonials'])) {
                    $settings['testimonials'] = array_values(array_filter($settings['testimonials'], function ($t) {
                        return is_array($t) && (!empty($t['title']) || !empty($t['author']) || !empty($t['comment']));
                    }));
                }

                $stmt = $db->prepare("UPDATE sections SET settings = ? WHERE id = ?");
                $stmt->execute([json_encode($settings), $sectionId]);
                echo json_encode(['success' => true]);
                exit;

            case 'update_theme':
                $key = $_POST['key'] ?? '';
                $value = $_POST['value'] ?? '';

                if ($key) {
                    SettingsService::setTheme($key, $value);
                    echo json_encode(['success' => true]);
                } else {
                    echo json_encode(['success' => false, 'error' => 'Invalid key']);
                }
                exit;
            default:
                echo json_encode(['success' => false, 'error' => 'Invalid action']);
                exit;
        }
    } catch (PDOException $e) {
        LogService::add('error', 'Theme editor operation failed: ' . $action, json_encode(['error' => $e->getMessage()]));
        echo json_encode(['success' => false, 'error' => 'Database error occurred']);
        exit;
    }

    exit;
}

// Get selected section for settings panel
$selectedSection = null;
$selectedSectionSettings = [];
if (isset($_GET['section'])) {
    $sectionParam = $_GET['section'];
    // Only query database for numeric section IDs, not special string values like 'general', 'header', 'footer'
    if (is_numeric($sectionParam)) {
        $stmt = $db->prepare("SELECT * FROM sections WHERE id = ?");
        $stmt->execute([(int)$sectionParam]);
        $selectedSection = $stmt->fetch();
        if ($selectedSection) {
            $selectedSectionSettings = json_decode($selectedSection['settings'], true) ?? [];
        }
    }
}

AdminLayout::renderHeader();
?>

<div class="admin-content admin-content--no-padding">
    <div class="theme-editor-wrapper">
        <!-- Left Sidebar - Sections List -->
        <div class="theme-editor-sidebar">
            <div class="theme-editor-sidebar-header">
                <h3>Theme Editor</h3>
            </div>

            <!-- Page Selector -->
            <div class="theme-editor-page-selector">
                <label for="page-selector" class="theme-editor-page-label">Editing Page</label>
                <select id="page-selector" class="form-control" data-page-selector>
                    <?php foreach ($pages as $p): ?>
                    <option value="<?php echo $p['id']; ?>" <?php echo $p['id'] == $pageId ? 'selected' : ''; ?>>
                        <?php echo e($p['title']); ?>
                    </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="section-list">
                <!-- General Settings -->
                <div class="settings-section">
                    <h4>General</h4>
                    <a href="/admin/theme-editor.php?page=<?php echo (int)$pageId; ?>&section=general" class="section-type-item section-type-item--clickable <?php echo isset($_GET['section']) && $_GET['section'] === 'general' ? 'active' : ''; ?>">
                        <div class="icon">⚙️</div>
                        <div class="info">
                            <h4>Site Settings</h4>
                            <p>Name, favicon & SEO</p>
                        </div>
                    </a>
                </div>

                <!-- Header Settings -->
                <div class="settings-section">
                    <h4>Header</h4>
                    <a href="/admin/theme-editor.php?page=<?php echo (int)$pageId; ?>&section=header" class="section-type-item section-type-item--clickable <?php echo isset($_GET['section']) && $_GET['section'] === 'header' ? 'active' : ''; ?>">
                        <div class="icon">📌</div>
                        <div class="info">
                            <h4>Header Settings</h4>
                            <p>Navigation style & logo</p>
                        </div>
                    </a>
                </div>

                <!-- Page Sections -->
                <div class="settings-section">
                    <h4>Page Sections</h4>
                    <ul class="page-sections-list" id="sections-list" data-sortable data-page="<?php echo (int)$pageId; ?>">
                        <?php foreach ($sections as $section): ?>
                        <li class="page-section-item <?php echo (isset($_GET['section']) && $_GET['section'] == $section['id']) ? 'active' : ''; ?>"
                            draggable="true"
                            data-id="<?php echo $section['id']; ?>"
                            data-section-link="/admin/theme-editor.php?page=<?php echo (int)$pageId; ?>&section=<?php echo (int)$section['id']; ?>">
                            <span class="drag-handle">⋮⋮</span>
                            <?php
                            if ($section['section_type'] === 'products') {
                                $sectionLabel = 'Product';
                            } elseif ($section['section_type'] === 'products_list') {
                                $sectionLabel = SECTION_NAME_PRODUCTS_LIST;
                            } elseif ($section['section_type'] === 'collections_list') {
                                $sectionLabel = SECTION_NAME_COLLECTIONS_LIST;
                            } elseif ($section['section_type'] === 'image_banner') {
                                $sectionLabel = SECTION_NAME_IMAGE_BANNER;
                            } else {
                                $sectionLabel = ucwords(str_replace('_', ' ', $section['section_type']));
                            }
                            ?>
                            <span class="name"><?php echo $sectionLabel; ?></span>
                            <button class="remove" data-remove-section="<?php echo (int)$section['id']; ?>" aria-label="Remove section">✕</button>
                        </li>
                        <?php endforeach; ?>
                    </ul>

                    <button class="btn btn-secondary btn-block mt-2" data-modal="add-section-modal">+ Add Section</button>
                </div>

                <!-- Footer Settings -->
                <div class="settings-section">
                    <h4>Footer</h4>
                    <a href="/admin/theme-editor.php?page=<?php echo (int)$pageId; ?>&section=footer" class="section-type-item section-type-item--clickable <?php echo isset($_GET['section']) && $_GET['section'] === 'footer' ? 'active' : ''; ?>">
                        <div class="icon">📎</div>
                        <div class="info">
                            <h4>Footer Settings</h4>
                            <p>Links & copyright</p>
                        </div>
                    </a>
                </div>
            </div>
        </div>

        <!-- Preview Area -->
        <div class="theme-editor-preview">
            <div class="preview-toolbar">
                <div class="device-switcher">
                    <button class="device-btn active" data-device="desktop" title="Desktop">🖥️</button>
                    <button class="device-btn" data-device="tablet" title="Tablet">📱</button>
                    <button class="device-btn" data-device="mobile" title="Mobile">📲</button>
                </div>
                <div>
                    <a href="/<?php echo e($currentPage['slug'] ?? ''); ?>" target="_blank" class="btn btn-secondary btn-sm">View Live</a>
                    <button class="btn btn-primary btn-sm" data-action="refresh-preview">Refresh</button>
                </div>
            </div>
            <div class="preview-frame-container">
                <iframe id="preview-frame" class="preview-frame" title="Theme preview" src="/<?php echo e($currentPage['slug'] ?? ''); ?>?preview=1"></iframe>
            </div>
        </div>

        <!-- Right Sidebar - Settings Panel -->
        <div class="theme-editor-settings">
            <div class="theme-editor-settings-header">
                <h3>
                    <?php
                    if (isset($_GET['section'])) {
                        if ($_GET['section'] === 'general') {
                            echo 'Site Settings';
                        } elseif ($_GET['section'] === 'header') {
                            echo 'Header Settings';
                        } elseif ($_GET['section'] === 'footer') {
                            echo 'Footer Settings';
                        } elseif ($selectedSection) {
                            if ($selectedSection['section_type'] === 'products') {
                                echo 'Product';
                            } elseif ($selectedSection['section_type'] === 'products_list') {
                                echo SECTION_NAME_PRODUCTS_LIST;
                            } elseif ($selectedSection['section_type'] === 'collections_list') {
                                echo SECTION_NAME_COLLECTIONS_LIST;
                            } elseif ($selectedSection['section_type'] === 'image_banner') {
                                echo SECTION_NAME_IMAGE_BANNER;
                            } else {
                                echo ucwords(str_replace('_', ' ', $selectedSection['section_type']));
                            }
                        }
                    } else {
                        echo 'Theme Colors';
                    }
?>
                </h3>
            </div>
            <div class="theme-editor-settings-body">
                <?php if (isset($_GET['section']) && $_GET['section'] === 'general'): ?>
                <!-- General Site Settings -->
                <div class="settings-section">
                    <h4>Site Identity</h4>

                    <!-- Logo Type Selector -->
                    <div class="form-group">
                        <label for="logo-type">Logo Type</label>
                        <select class="form-control" id="logo-type" data-theme-setting="logo_type" data-theme-toggle="logo">
                            <option value="text" <?php echo ($themeSettings['logo_type'] ?? 'text') === 'text' ? 'selected' : ''; ?>>Text (Site Name)</option>
                            <option value="image" <?php echo ($themeSettings['logo_type'] ?? '') === 'image' ? 'selected' : ''; ?>>Image Logo</option>
                        </select>
                    </div>

                    <!-- Text Logo Options -->
                    <div id="logo-text-options" class="<?php echo ($themeSettings['logo_type'] ?? 'text') === 'image' ? 'is-hidden' : ''; ?>">
                        <div class="form-group">
                            <label for="site-name">Site Name</label>
                            <input type="text" class="form-control" id="site-name" value="<?php echo e($themeSettings['site_name'] ?? ''); ?>" data-theme-setting="site_name" placeholder="My Website">
                            <small class="text-muted">Displayed in browser tab and header</small>
                        </div>
                    </div>

                    <!-- Image Logo Options -->
                    <div id="logo-image-options" class="<?php echo ($themeSettings['logo_type'] ?? 'text') !== 'image' ? 'is-hidden' : ''; ?>">
                        <div class="form-group">
                            <label for="site-logo">Logo Image</label>
                            <div class="input-with-picker">
                                <input type="text" class="form-control" id="site-logo" value="<?php echo e($themeSettings['site_logo'] ?? ''); ?>" data-theme-setting="site_logo" placeholder="/admin-file/logo.png">
                                <button type="button" class="btn btn-secondary" data-action="open-media-picker" data-picker-target="site-logo" data-picker-type="image">Browse</button>
                            </div>
                            <small class="text-muted">Recommended: PNG or SVG with transparent background</small>
                            <?php if (!empty($themeSettings['site_logo'])): ?>
                            <div class="logo-preview-box mt-small">
                                <img src="<?php echo e($themeSettings['site_logo']); ?>" alt="Logo preview" class="logo-preview-img">
                            </div>
                            <?php endif; ?>
                        </div>
                        <div class="form-group">
                            <label for="logo-height">Logo Height (px)</label>
                            <input type="number" class="form-control" id="logo-height" value="<?php echo e($themeSettings['logo_height'] ?? '40'); ?>" data-theme-setting="logo_height" min="20" max="120" step="5">
                            <small class="text-muted">Height in pixels (width will auto-adjust to maintain aspect ratio)</small>
                        </div>
                        <div class="form-group">
                            <label for="site-name-hidden">Hidden Site Name (for SEO)</label>
                            <input type="text" class="form-control" id="site-name-hidden" value="<?php echo e($themeSettings['site_name'] ?? ''); ?>" data-theme-setting="site_name" placeholder="My Website">
                            <small class="text-muted">Used for browser tab and SEO (not displayed on page)</small>
                        </div>
                    </div>

                    <div class="form-group mt-medium">
                        <label for="site-tagline">Site Tagline</label>
                        <input type="text" class="form-control" id="site-tagline" value="<?php echo e($themeSettings['site_tagline'] ?? ''); ?>" data-theme-setting="site_tagline" placeholder="Your amazing tagline">
                        <small class="text-muted">Short description of your site</small>
                    </div>
                </div>

                <div class="settings-section">
                    <h4>Layout</h4>
                    <div class="form-group">
                        <label for="content-max-width">Content Max Width (px)</label>
                        <input type="number" class="form-control" id="content-max-width" value="<?php echo e($themeSettings['content_max_width'] ?? '1400'); ?>" data-theme-setting="content_max_width" min="900" max="2200" step="50">
                        <small class="text-muted">Applies to all sections using the container</small>
                    </div>
                    <div class="form-group">
                        <label for="section-padding">Section Vertical Padding (px)</label>
                        <input type="number" class="form-control" id="section-padding" value="<?php echo e($themeSettings['section_padding'] ?? '60'); ?>" data-theme-setting="section_padding" min="20" max="160" step="5">
                        <small class="text-muted">Controls top/bottom spacing for sections</small>
                    </div>
                    <div class="form-group">
                        <label for="section-height">Section Height (vh)</label>
                        <input type="number" class="form-control" id="section-height" value="<?php echo e($themeSettings['section_height'] ?? '100'); ?>" data-theme-setting="section_height" min="40" max="100" step="5">
                        <small class="text-muted">Controls section height (100 = full screen)</small>
                    </div>
                </div>

                <div class="settings-section">
                    <h4>Favicon</h4>
                    <div class="form-group">
                        <label for="site-favicon">Favicon Image</label>
                        <div class="input-with-picker">
                            <input type="text" class="form-control" id="site-favicon" value="<?php echo e($themeSettings['site_favicon'] ?? ''); ?>" data-theme-setting="site_favicon" placeholder="/admin-file/favicon.ico">
                            <button type="button" class="btn btn-secondary" data-action="open-media-picker" data-picker-target="site-favicon" data-picker-type="image">Browse</button>
                        </div>
                        <small class="text-muted">Upload .ico, .png or .svg (recommended: 32x32 or 64x64)</small>
                        <?php if (!empty($themeSettings['site_favicon'])): ?>
                        <div class="mt-small">
                            <img src="<?php echo e($themeSettings['site_favicon']); ?>" alt="Favicon preview" class="favicon-preview-img">
                        </div>
                        <?php endif; ?>
                    </div>
                </div>

                <div class="settings-section">
                    <h4>SEO Defaults</h4>
                    <div class="form-group">
                        <label for="site-description">Default Meta Description</label>
                        <textarea class="form-control" id="site-description" rows="3" data-theme-setting="site_description" placeholder="A brief description of your website for search engines"><?php echo e($themeSettings['site_description'] ?? ''); ?></textarea>
                        <small class="text-muted">Used when pages don't have their own description</small>
                    </div>
                </div>

                <div class="settings-section">
                    <h4>Breadcrumbs</h4>
                    <div class="form-group">
                        <label class="form-check">
                            <input type="checkbox" id="breadcrumbs-enabled" <?php echo ($themeSettings['breadcrumbs_enabled'] ?? false) ? 'checked' : ''; ?> data-theme-setting="breadcrumbs_enabled" data-theme-type="checkbox">
                            <span>Enable breadcrumbs navigation</span>
                        </label>
                        <small class="text-muted">Shows navigation path below header (Home > Page > ...)</small>
                    </div>
                    <div class="form-group">
                        <label for="breadcrumbs-home-label">Home Label</label>
                        <input type="text" class="form-control" id="breadcrumbs-home-label" value="<?php echo e($themeSettings['breadcrumbs_home_label'] ?? 'Home'); ?>" data-theme-setting="breadcrumbs_home_label" placeholder="Home">
                        <small class="text-muted">Text for the home link in breadcrumbs</small>
                    </div>
                    <div class="form-group">
                        <label for="breadcrumbs-separator">Separator</label>
                        <select class="form-control" id="breadcrumbs-separator" data-theme-setting="breadcrumbs_separator">
                            <option value="chevron" <?php echo ($themeSettings['breadcrumbs_separator'] ?? 'chevron') === 'chevron' ? 'selected' : ''; ?>>Chevron (›)</option>
                            <option value="slash" <?php echo ($themeSettings['breadcrumbs_separator'] ?? '') === 'slash' ? 'selected' : ''; ?>>Slash (/)</option>
                            <option value="arrow" <?php echo ($themeSettings['breadcrumbs_separator'] ?? '') === 'arrow' ? 'selected' : ''; ?>>Arrow (→)</option>
                            <option value="dot" <?php echo ($themeSettings['breadcrumbs_separator'] ?? '') === 'dot' ? 'selected' : ''; ?>>Dot (•)</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label class="form-check">
                            <input type="checkbox" id="breadcrumbs-show-home" <?php echo ($themeSettings['breadcrumbs_show_home'] ?? true) ? 'checked' : ''; ?> data-theme-setting="breadcrumbs_show_home" data-theme-type="checkbox">
                            <span>Show home icon</span>
                        </label>
                    </div>
                </div>

                <?php elseif (isset($_GET['section']) && $_GET['section'] === 'header'): ?>
                <!-- Header Settings -->
                <div class="settings-section">
                    <h4>Header Style</h4>
                    <div class="form-group">
                        <label for="header-style">Navigation Type</label>
                        <select class="form-control" id="header-style" data-theme-setting="header_style">
                            <option value="top" <?php echo ($themeSettings['header_style'] ?? 'top') === 'top' ? 'selected' : ''; ?>>Top Bar</option>
                            <option value="sidebar" <?php echo ($themeSettings['header_style'] ?? '') === 'sidebar' ? 'selected' : ''; ?>>Left Sidebar (Hover Expand)</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="header-menu">Header Menu</label>
                        <select class="form-control" id="header-menu" data-theme-setting="header_menu_id">
                            <option value="">-- Auto (location: header) --</option>
                            <?php foreach ($allMenus as $menu): ?>
                            <option value="<?php echo $menu['id']; ?>" <?php echo ($themeSettings['header_menu_id'] ?? '') == $menu['id'] ? 'selected' : ''; ?>>
                                <?php echo e($menu['name']); ?> (<?php echo e($menu['location']); ?>)
                            </option>
                            <?php endforeach; ?>
                        </select>
                        <small class="text-muted">Select which menu to display in the header</small>
                    </div>
                    <div class="form-group">
                        <label for="topbar-menu">Top Mini Bar Menu</label>
                        <select class="form-control" id="topbar-menu" data-theme-setting="topbar_menu_id">
                            <option value="">-- Auto (location: topbar) --</option>
                            <?php foreach ($allMenus as $menu): ?>
                            <option value="<?php echo $menu['id']; ?>" <?php echo ($themeSettings['topbar_menu_id'] ?? '') == $menu['id'] ? 'selected' : ''; ?>>
                                <?php echo e($menu['name']); ?> (<?php echo e($menu['location']); ?>)
                            </option>
                            <?php endforeach; ?>
                        </select>
                        <small class="text-muted">Select which menu to display in the top mini bar</small>
                    </div>
                </div>

                <?php elseif (isset($_GET['section']) && $_GET['section'] === 'footer'): ?>
                <!-- Footer Settings -->
                <div class="settings-section">
                    <h4>Footer Menu</h4>
                    <div class="form-group">
                        <label for="footer-menu">Footer Menu</label>
                        <select class="form-control" id="footer-menu" data-theme-setting="footer_menu_id">
                            <option value="">-- Auto (location: footer) --</option>
                            <?php foreach ($allMenus as $menu): ?>
                            <option value="<?php echo $menu['id']; ?>" <?php echo ($themeSettings['footer_menu_id'] ?? '') == $menu['id'] ? 'selected' : ''; ?>>
                                <?php echo e($menu['name']); ?> (<?php echo e($menu['location']); ?>)
                            </option>
                            <?php endforeach; ?>
                        </select>
                        <small class="text-muted">Select which menu to display in the footer</small>
                    </div>
                    <p class="text-muted mt-medium">Manage menu items in the <a href="/admin/menus.php">Menus</a> section.</p>
                </div>

                <?php elseif ($selectedSection && $selectedSection['section_type'] === 'split_slideshow'): ?>
                <!-- Split Slideshow Settings -->
                <form data-section-settings-form>
                    <input type="hidden" name="section_id" value="<?php echo $selectedSection['id']; ?>">

                    <div class="settings-section">
                        <h4>Slideshow Options</h4>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[autoplay]" <?php echo ($selectedSectionSettings['autoplay'] ?? true) ? 'checked' : ''; ?>>
                                <span>Autoplay</span>
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[show_arrows]" <?php echo ($selectedSectionSettings['show_arrows'] ?? true) ? 'checked' : ''; ?>>
                                <span>Show Arrows</span>
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[show_dots]" <?php echo ($selectedSectionSettings['show_dots'] ?? true) ? 'checked' : ''; ?>>
                                <span>Show Dots</span>
                            </label>
                        </div>
                    </div>

                    <div class="settings-section">
                        <h4>Slides</h4>
                        <div id="slides-container">
                            <?php
        $slides = $selectedSectionSettings['slides'] ?? [];
                    foreach ($slides as $i => $slide):
                        $slideLinkType = $slide['button_link_type'] ?? 'custom';
                        $slideMediaSource = $slide['media_source'] ?? 'manual';
                        $slideProductId = (string)($slide['button_product_id'] ?? '');
                        ?>
                            <div class="card slide-card" data-slide-index="<?php echo $i; ?>">
                                <div class="card-header">
                                    <span>Slide <?php echo $i + 1; ?></span>
                                    <button type="button" class="btn btn-sm btn-danger btn-slide-remove" data-action="remove-slide">Remove</button>
                                </div>
                                <div class="card-body">
                                    <div class="form-group">
                                        <div class="form-label">Title</div>
                                        <input type="text" name="settings[slides][<?php echo $i; ?>][title]" class="form-control" value="<?php echo e($slide['title'] ?? ''); ?>">
                                    </div>
                                    <div class="form-group">
                                        <div class="form-label">Description</div>
                                        <textarea name="settings[slides][<?php echo $i; ?>][description]" class="form-control" rows="2"><?php echo e($slide['description'] ?? ''); ?></textarea>
                                    </div>
                                    <div class="form-group">
                                        <div class="form-label">Button Text</div>
                                        <input type="text" name="settings[slides][<?php echo $i; ?>][button_text]" class="form-control" value="<?php echo e($slide['button_text'] ?? ''); ?>">
                                    </div>
                                    <div class="form-group">
                                        <div class="form-label">Button Link</div>
                                        <select name="settings[slides][<?php echo $i; ?>][button_link_type]" class="form-control js-slide-link-type">
                                            <option value="custom" <?php echo $slideLinkType === 'custom' ? 'selected' : ''; ?>>Custom URL</option>
                                            <option value="product" <?php echo $slideLinkType === 'product' ? 'selected' : ''; ?>>Product</option>
                                        </select>
                                    </div>
                                    <div class="form-group js-slide-custom-url">
                                        <div class="form-label">Button URL</div>
                                        <input type="text" name="settings[slides][<?php echo $i; ?>][button_url]" class="form-control" value="<?php echo e($slide['button_url'] ?? ''); ?>">
                                        <small class="text-muted">Use https:// or a /relative path.</small>
                                    </div>
                                    <div class="form-group js-slide-product-select">
                                        <div class="form-label">Product (link or image)</div>
                                        <select name="settings[slides][<?php echo $i; ?>][button_product_id]" class="form-control js-slide-product">
                                            <option value="">-- Select product --</option>
                                            <?php foreach ($slideProductOptions as $product): ?>
                                            <option value="<?php echo (int)$product['id']; ?>" <?php echo $slideProductId === (string)$product['id'] ? 'selected' : ''; ?>>
                                                <?php echo e($product['name']); ?>
                                            </option>
                                            <?php endforeach; ?>
                                        </select>
                                        <?php if (empty($slideProductOptions)): ?>
                                        <small class="text-muted">No products available yet.</small>
                                        <?php endif; ?>
                                    </div>
                                    <div class="form-group">
                                        <div class="form-label">Image/Video Source</div>
                                        <select name="settings[slides][<?php echo $i; ?>][media_source]" class="form-control js-slide-media-source">
                                            <option value="manual" <?php echo $slideMediaSource === 'manual' ? 'selected' : ''; ?>>Manual</option>
                                            <option value="product" <?php echo $slideMediaSource === 'product' ? 'selected' : ''; ?>>Product image</option>
                                        </select>
                                        <small class="text-muted">Product image requires a selected product.</small>
                                    </div>
                                    <div class="form-group js-slide-media-manual">
                                        <div class="form-label">Image/Video</div>
                                        <input type="text" name="settings[slides][<?php echo $i; ?>][image]" class="form-control" value="<?php echo e($slide['image'] ?? ''); ?>" data-media-picker="image">
                                    </div>
                                    <div class="form-group js-slide-media-product">
                                        <div class="text-muted">Uses the selected product's primary media.</div>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <template id="slide-product-options">
                            <?php if (!empty($slideProductOptions)): ?>
                                <option value="">-- Select product --</option>
                                <?php foreach ($slideProductOptions as $product): ?>
                                <option value="<?php echo (int)$product['id']; ?>"><?php echo e($product['name']); ?></option>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <option value="">-- No products available --</option>
                            <?php endif; ?>
                        </template>
                        <button type="button" class="btn btn-secondary btn-sm" data-action="add-slide">+ Add Slide</button>
                    </div>

                    <button type="submit" class="btn btn-primary btn-block mt-2">Save Changes</button>
                </form>

                <?php elseif ($selectedSection && $selectedSection['section_type'] === 'text_box'): ?>
                <!-- Text Box Settings -->
                <form data-section-settings-form>
                    <input type="hidden" name="section_id" value="<?php echo $selectedSection['id']; ?>">

                    <div class="settings-section">
                        <h4>Content</h4>
                        <div class="form-group">
                            <div class="form-label">Title</div>
                            <input type="text" name="settings[title]" class="form-control" value="<?php echo e($selectedSectionSettings['title'] ?? ''); ?>">
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <div class="form-label">Title Alignment</div>
                                <select name="settings[title_align]" class="form-control">
                                    <?php
                                $titleAlign = $selectedSectionSettings['title_align'] ?? 'left';
                    $alignOptions = ['left' => 'Left', 'center' => 'Center', 'right' => 'Right'];
                    foreach ($alignOptions as $value => $label):
                        ?>
                                        <option value="<?php echo e($value); ?>" <?php echo $titleAlign === $value ? 'selected' : ''; ?>><?php echo e($label); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="form-group">
                                <div class="form-label">Text Alignment</div>
                                <select name="settings[text_align]" class="form-control">
                                    <?php
                        $textAlign = $selectedSectionSettings['text_align'] ?? 'left';
                    foreach ($alignOptions as $value => $label):
                        ?>
                                        <option value="<?php echo e($value); ?>" <?php echo $textAlign === $value ? 'selected' : ''; ?>><?php echo e($label); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <div class="form-label">Title Size (px)</div>
                                <input type="number" name="settings[title_size]" class="form-control" min="18" max="64" step="1" value="<?php echo e($selectedSectionSettings['title_size'] ?? '32'); ?>">
                            </div>
                            <div class="form-group">
                                <div class="form-label">Text Size (px)</div>
                                <input type="number" name="settings[text_size]" class="form-control" min="12" max="28" step="1" value="<?php echo e($selectedSectionSettings['text_size'] ?? '16'); ?>">
                            </div>
                        </div>
                        <div class="form-group">
                            <div class="form-label">Text (HTML allowed)</div>
                            <textarea name="settings[content]" class="form-control" rows="10"><?php echo e($selectedSectionSettings['content'] ?? ''); ?></textarea>
                        </div>
                        <div class="text-muted">Use &lt;p&gt; for paragraphs, &lt;strong&gt; for bold text, &lt;em&gt; for emphasis/italics, &lt;ul&gt; for bullet lists, and &lt;a&gt; for links.</div>
                    </div>

                    <button type="submit" class="btn btn-primary btn-block mt-2">Save Changes</button>
                </form>

                <?php elseif ($selectedSection && $selectedSection['section_type'] === 'testimonials'): ?>
                <!-- Testimonials Settings -->
                <form data-section-settings-form>
                    <input type="hidden" name="section_id" value="<?php echo $selectedSection['id']; ?>">

                    <div class="settings-section">
                        <h4>Section Title</h4>
                        <div class="form-group">
                            <div class="form-label">Title</div>
                            <input type="text" name="settings[section_title]" class="form-control" value="<?php echo e($selectedSectionSettings['section_title'] ?? ''); ?>">
                            <div class="text-muted mt-1">Leave blank to use the selected product name.</div>
                        </div>
                    </div>

                    <div class="settings-section">
                        <h4>Slider Settings</h4>
                        <div class="form-row">
                            <div class="form-group">
                                <div class="form-label">Card Width (px)</div>
                                <input type="number" name="settings[card_width]" class="form-control" min="220" max="520" step="10" value="<?php echo e($selectedSectionSettings['card_width'] ?? '320'); ?>">
                            </div>
                            <div class="form-group">
                                <div class="form-label">Auto Slide Interval (seconds)</div>
                                <input type="number" name="settings[auto_interval]" class="form-control" min="2" max="20" step="1" value="<?php echo e($selectedSectionSettings['auto_interval'] ?? '5'); ?>">
                            </div>
                        </div>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[auto_scroll]" <?php echo ($selectedSectionSettings['auto_scroll'] ?? true) ? 'checked' : ''; ?>>
                                <span>Enable auto sliding</span>
                            </label>
                        </div>
                    </div>

                    <div class="settings-section">
                        <h4>Testimonials</h4>
                        <div id="testimonials-container">
                            <?php
                            $testimonials = $selectedSectionSettings['testimonials'] ?? [];
                    foreach ($testimonials as $i => $testimonial):
                        ?>
                            <div class="card section-item-card mb-2" data-testimonial-index="<?php echo $i; ?>">
                                <div class="card-header section-item-header">
                                    <span>Testimonial <?php echo $i + 1; ?></span>
                                    <button type="button" class="btn btn-sm btn-danger" data-action="remove-testimonial">Remove</button>
                                </div>
                                <div class="card-body section-item-body">
                                    <div class="form-group">
                                        <div class="form-label">Title</div>
                                        <input type="text" name="settings[testimonials][<?php echo $i; ?>][title]" class="form-control" value="<?php echo e($testimonial['title'] ?? ''); ?>">
                                    </div>
                                    <div class="form-group">
                                        <div class="form-label">Author Name</div>
                                        <input type="text" name="settings[testimonials][<?php echo $i; ?>][author]" class="form-control" value="<?php echo e($testimonial['author'] ?? ''); ?>">
                                    </div>
                                    <div class="form-group">
                                        <div class="form-label">Comment</div>
                                        <textarea name="settings[testimonials][<?php echo $i; ?>][comment]" class="form-control" rows="3"><?php echo e($testimonial['comment'] ?? ''); ?></textarea>
                                    </div>
                                    <div class="form-group">
                                        <div class="form-label">Stars</div>
                                        <select name="settings[testimonials][<?php echo $i; ?>][stars]" class="form-control">
                                            <?php
                                        $stars = (int)($testimonial['stars'] ?? 5);
                        for ($s = 1; $s <= 5; $s++):
                            ?>
                                                <option value="<?php echo $s; ?>" <?php echo $stars === $s ? 'selected' : ''; ?>><?php echo $s; ?></option>
                                            <?php endfor; ?>
                                        </select>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <button type="button" class="btn btn-secondary btn-sm" data-action="add-testimonial">+ Add Testimonial</button>
                    </div>

                    <button type="submit" class="btn btn-primary btn-block mt-2">Save Changes</button>
                </form>

                <?php elseif ($selectedSection && $selectedSection['section_type'] === 'products'): ?>
                <!-- Products Settings -->
                <form data-section-settings-form>
                    <input type="hidden" name="section_id" value="<?php echo $selectedSection['id']; ?>">

                    <div class="settings-section">
                        <h4>Section Title</h4>
                        <div class="form-group">
                            <div class="form-label">Title</div>
                            <input type="text" name="settings[section_title]" class="form-control" value="<?php echo e($selectedSectionSettings['section_title'] ?? ''); ?>">
                        </div>
                        <div class="form-group">
                            <div class="form-label">Title Alignment</div>
                            <select name="settings[title_align]" class="form-control">
                                <?php
                                $titleAlign = $selectedSectionSettings['title_align'] ?? 'center';
                    $alignOptions = ['left' => 'Left', 'center' => 'Center', 'right' => 'Right'];
                    foreach ($alignOptions as $value => $label):
                        ?>
                                    <option value="<?php echo e($value); ?>" <?php echo $titleAlign === $value ? 'selected' : ''; ?>><?php echo e($label); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>

                    <div class="settings-section">
                        <h4>Product</h4>
                        <div class="form-group">
                            <div class="form-label">Select Product</div>
                            <select name="settings[product_id]" class="form-control">
                                <option value="">-- Select a product for this page --</option>
                                <?php
                        $selectedProductId = $selectedSectionSettings['product_id'] ?? '';
                    foreach ($productOptions as $productOption):
                        $optionLabel = $productOption['name'];
                        if (!empty($productOption['price'])) {
                            $optionLabel .= ' (' . $productOption['price'] . ')';
                        }
                        ?>
                                    <option value="<?php echo $productOption['id']; ?>" <?php echo (string)$selectedProductId === (string)$productOption['id'] ? 'selected' : ''; ?>>
                                        <?php echo e($optionLabel); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <?php if (empty($productOptions)): ?>
                                <div class="text-muted mt-1">Add products for this page in Products to enable selection.</div>
                            <?php endif; ?>
                        </div>
                    </div>

                    <div class="settings-section">
                        <h4>Additional Content Areas</h4>
                        <div class="form-group">
                            <div class="form-label">Style</div>
                            <?php $accordionStyle = $selectedSectionSettings['accordion_style'] ?? 'separated'; ?>
                            <select name="settings[accordion_style]" class="form-control">
                                <option value="separated" <?php echo $accordionStyle === 'separated' ? 'selected' : ''; ?>>Separated cards</option>
                                <option value="bar" <?php echo $accordionStyle === 'bar' ? 'selected' : ''; ?>>Single bar list</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <div class="form-label">Text Alignment</div>
                            <?php
                            $contentAlign = $selectedSectionSettings['content_align'] ?? 'left';
$contentAlign = in_array($contentAlign, ['left', 'center', 'right'], true) ? $contentAlign : 'left';
?>
                            <select name="settings[content_align]" class="form-control">
                                <option value="left" <?php echo $contentAlign === 'left' ? 'selected' : ''; ?>>Left</option>
                                <option value="center" <?php echo $contentAlign === 'center' ? 'selected' : ''; ?>>Center</option>
                                <option value="right" <?php echo $contentAlign === 'right' ? 'selected' : ''; ?>>Right</option>
                            </select>
                        </div>
                    </div>

                    <div class="settings-section">
                        <h4>Version Selector</h4>
                        <div class="form-group">
                            <div class="form-label">Style</div>
                            <?php $variantStyle = $selectedSectionSettings['variant_style'] ?? 'select'; ?>
                            <select name="settings[variant_style]" class="form-control">
                                <option value="select" <?php echo $variantStyle === 'select' ? 'selected' : ''; ?>>Dropdown</option>
                                <option value="buttons" <?php echo $variantStyle === 'buttons' ? 'selected' : ''; ?>>Buttons</option>
                            </select>
                        </div>
                    </div>

                    <div class="settings-section">
                        <h4>Visibility</h4>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[show_media]" <?php echo ($selectedSectionSettings['show_media'] ?? true) ? 'checked' : ''; ?>>
                                <span>Show media</span>
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[show_description]" <?php echo ($selectedSectionSettings['show_description'] ?? true) ? 'checked' : ''; ?>>
                                <span>Show description</span>
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[show_price]" <?php echo ($selectedSectionSettings['show_price'] ?? true) ? 'checked' : ''; ?>>
                                <span>Show price</span>
                            </label>
                        </div>
                    </div>

                    <div class="settings-section">
                        <h4>Similar Products</h4>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[show_similar_products]" <?php echo ($selectedSectionSettings['show_similar_products'] ?? false) ? 'checked' : ''; ?>>
                                <span>Show similar products below comments</span>
                            </label>
                        </div>
                        <div class="form-group">
                            <div class="form-label">Section Title</div>
                            <input type="text" name="settings[similar_title]" class="form-control" value="<?php echo e($selectedSectionSettings['similar_title'] ?? 'Similar Products'); ?>">
                        </div>
                        <div class="form-group">
                            <div class="form-label">Title Alignment</div>
                            <select name="settings[similar_title_align]" class="form-control">
                                <?php
                                $simTitleAlign = $selectedSectionSettings['similar_title_align'] ?? 'center';
                    foreach (['left' => 'Left', 'center' => 'Center', 'right' => 'Right'] as $value => $label):
                        ?>
                                    <option value="<?php echo e($value); ?>" <?php echo $simTitleAlign === $value ? 'selected' : ''; ?>><?php echo e($label); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <div class="form-label">Columns</div>
                                <select name="settings[similar_columns]" class="form-control">
                                    <?php
                                    $simCols = (int)($selectedSectionSettings['similar_columns'] ?? 4);
                    foreach ([2, 3, 4] as $col):
                        ?>
                                        <option value="<?php echo $col; ?>" <?php echo $simCols === $col ? 'selected' : ''; ?>><?php echo $col; ?> Columns</option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="form-group">
                                <div class="form-label">Max Products</div>
                                <select name="settings[similar_max_products]" class="form-control">
                                    <?php
                                    $simMax = (int)($selectedSectionSettings['similar_max_products'] ?? 8);
                    foreach ([4, 8, 12, 16] as $max):
                        ?>
                                        <option value="<?php echo $max; ?>" <?php echo $simMax === $max ? 'selected' : ''; ?>><?php echo $max; ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[similar_show_price]" <?php echo ($selectedSectionSettings['similar_show_price'] ?? true) ? 'checked' : ''; ?>>
                                <span>Show price</span>
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[similar_show_description]" <?php echo ($selectedSectionSettings['similar_show_description'] ?? false) ? 'checked' : ''; ?>>
                                <span>Show description</span>
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[similar_hover_swap_media]" <?php echo ($selectedSectionSettings['similar_hover_swap_media'] ?? true) ? 'checked' : ''; ?>>
                                <span>Swap image on hover</span>
                            </label>
                        </div>
                    </div>

                    <button type="submit" class="btn btn-primary btn-block mt-2">Save Changes</button>
                </form>

                <?php elseif ($selectedSection && $selectedSection['section_type'] === 'products_list'): ?>
                <!-- Products List Settings -->
                <form data-section-settings-form>
                    <input type="hidden" name="section_id" value="<?php echo $selectedSection['id']; ?>">

                    <div class="settings-section">
                        <h4>Section Title</h4>
                        <div class="form-group">
                            <div class="form-label">Title</div>
                            <input type="text" name="settings[section_title]" class="form-control" value="<?php echo e($selectedSectionSettings['section_title'] ?? ''); ?>">
                        </div>
                        <div class="form-group">
                            <div class="form-label">Title Alignment</div>
                            <select name="settings[title_align]" class="form-control">
                                <?php
    $titleAlign = $selectedSectionSettings['title_align'] ?? 'center';
                    $alignOptions = ['left' => 'Left', 'center' => 'Center', 'right' => 'Right'];
                    foreach ($alignOptions as $value => $label):
                        ?>
                                    <option value="<?php echo e($value); ?>" <?php echo $titleAlign === $value ? 'selected' : ''; ?>><?php echo e($label); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>

                    <div class="settings-section">
                        <h4>Layout</h4>
                        <div class="form-row">
                            <div class="form-group">
                                <div class="form-label">Columns</div>
                                <select name="settings[columns]" class="form-control">
                                    <?php
                            $columns = (int)($selectedSectionSettings['columns'] ?? 3);
                    foreach ([2, 3, 4] as $col):
                        ?>
                                        <option value="<?php echo $col; ?>" <?php echo $columns === $col ? 'selected' : ''; ?>><?php echo $col; ?> Columns</option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>

                    <div class="settings-section">
                        <h4>Visibility</h4>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[show_title]" <?php echo ($selectedSectionSettings['show_title'] ?? true) ? 'checked' : ''; ?>>
                                <span>Show section title</span>
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[show_description]" <?php echo ($selectedSectionSettings['show_description'] ?? true) ? 'checked' : ''; ?>>
                                <span>Show product description</span>
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[show_price]" <?php echo ($selectedSectionSettings['show_price'] ?? true) ? 'checked' : ''; ?>>
                                <span>Show price</span>
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[hover_swap_media]" <?php echo ($selectedSectionSettings['hover_swap_media'] ?? true) ? 'checked' : ''; ?>>
                                <span>Swap image on hover</span>
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[show_filters]" <?php echo ($selectedSectionSettings['show_filters'] ?? true) ? 'checked' : ''; ?>>
                                <span>Show filters sidebar</span>
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[show_sort]" <?php echo ($selectedSectionSettings['show_sort'] ?? true) ? 'checked' : ''; ?>>
                                <span>Show sorting</span>
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[show_view_toggle]" <?php echo ($selectedSectionSettings['show_view_toggle'] ?? true) ? 'checked' : ''; ?>>
                                <span>Show view toggle</span>
                            </label>
                        </div>
                        <div class="form-group">
                            <div class="form-label">Default view</div>
                            <?php $defaultView = $selectedSectionSettings['default_view'] ?? 'grid'; ?>
                            <select name="settings[default_view]" class="form-control">
                                <option value="grid" <?php echo $defaultView === 'grid' ? 'selected' : ''; ?>>Grid</option>
                                <option value="list" <?php echo $defaultView === 'list' ? 'selected' : ''; ?>>List</option>
                            </select>
                        </div>
                    </div>

                    <button type="submit" class="btn btn-primary btn-block mt-2">Save Changes</button>
                </form>

                <?php elseif ($selectedSection && $selectedSection['section_type'] === 'image_banner'): ?>
                <!-- Image Banner Settings -->
                <form data-section-settings-form>
                    <input type="hidden" name="section_id" value="<?php echo $selectedSection['id']; ?>">

                    <div class="settings-section">
                        <h4>Images</h4>
                        <p class="text-muted">Add multiple images for auto-sliding.</p>
                        <div id="banner-images-container">
                            <?php
                            $bannerImages = $selectedSectionSettings['images'] ?? [];
                    foreach ($bannerImages as $imgIndex => $imgItem):
                        ?>
                            <div class="slide-item" data-index="<?php echo $imgIndex; ?>">
                                <div class="slide-item-header">
                                    <span>Image <?php echo $imgIndex + 1; ?></span>
                                    <button type="button" class="btn-icon" data-action="remove-banner-image" data-index="<?php echo $imgIndex; ?>" title="Remove">&times;</button>
                                </div>
                                <div class="form-group">
                                    <div class="form-label">Image</div>
                                    <div class="image-picker">
                                        <input type="hidden" name="settings[images][<?php echo $imgIndex; ?>][image]" value="<?php echo e($imgItem['image'] ?? ''); ?>">
                                        <div class="image-preview">
                                            <?php if (!empty($imgItem['image'])): ?>
                                            <img src="<?php echo e($imgItem['image']); ?>" alt="Banner preview" class="image-preview-img">
                                            <?php else: ?>
                                            <span>No image</span>
                                            <?php endif; ?>
                                        </div>
                                        <button type="button" class="btn btn-secondary btn-sm" data-action="banner-choose-image">Choose Image</button>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <button type="button" class="btn btn-secondary btn-sm" data-action="add-banner-image">+ Add Image</button>
                    </div>

                    <div class="settings-section">
                        <h4>Text Content</h4>
                        <div class="form-group">
                            <div class="form-label">Title</div>
                            <input type="text" name="settings[title]" class="form-control" value="<?php echo e($selectedSectionSettings['title'] ?? ''); ?>">
                        </div>
                        <div class="form-group">
                            <div class="form-label">Subtitle</div>
                            <textarea name="settings[subtitle]" class="form-control" rows="2"><?php echo e($selectedSectionSettings['subtitle'] ?? ''); ?></textarea>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <div class="form-label">Title Size (px)</div>
                                <input type="number" name="settings[title_size]" class="form-control" min="24" max="96" value="<?php echo e($selectedSectionSettings['title_size'] ?? '48'); ?>">
                            </div>
                            <div class="form-group">
                                <div class="form-label">Subtitle Size (px)</div>
                                <input type="number" name="settings[subtitle_size]" class="form-control" min="12" max="36" value="<?php echo e($selectedSectionSettings['subtitle_size'] ?? '18'); ?>">
                            </div>
                        </div>
                        <div class="form-group">
                            <div class="form-label">Text Alignment</div>
                            <select name="settings[text_align]" class="form-control">
                                <?php
                            $textAlign = $selectedSectionSettings['text_align'] ?? 'center';
$alignOptions = ['left' => 'Left', 'center' => 'Center', 'right' => 'Right'];
foreach ($alignOptions as $value => $label):
    ?>
                                    <option value="<?php echo e($value); ?>" <?php echo $textAlign === $value ? 'selected' : ''; ?>><?php echo e($label); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <div class="form-label">Text Vertical Position</div>
                            <select name="settings[text_position]" class="form-control">
                                <?php
    $textPosition = $selectedSectionSettings['text_position'] ?? 'center';
$textPosOptions = [
    'top' => 'Top',
    'center' => 'Center',
    'bottom' => 'Bottom'
];
foreach ($textPosOptions as $value => $label):
    ?>
                                    <option value="<?php echo e($value); ?>" <?php echo $textPosition === $value ? 'selected' : ''; ?>><?php echo e($label); ?></option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">Position the text content within the banner</small>
                        </div>
                    </div>

                    <div class="settings-section">
                        <h4>Appearance</h4>
                        <div class="form-group">
                            <div class="form-label">Banner Height (px)</div>
                            <input type="number" name="settings[height]" class="form-control" min="150" max="600" step="10" value="<?php echo e($selectedSectionSettings['height'] ?? '300'); ?>">
                        </div>
                        <div class="form-group">
                            <div class="form-label">Top Spacing (px)</div>
                            <input type="number" name="settings[spacing_top]" class="form-control" min="0" max="200" step="5" value="<?php echo e($selectedSectionSettings['spacing_top'] ?? '0'); ?>">
                            <small class="text-muted">Space above the banner</small>
                        </div>
                        <div class="form-group">
                            <div class="form-label">Bottom Spacing (px)</div>
                            <input type="number" name="settings[spacing_bottom]" class="form-control" min="0" max="200" step="5" value="<?php echo e($selectedSectionSettings['spacing_bottom'] ?? '0'); ?>">
                            <small class="text-muted">Space below the banner</small>
                        </div>
                        <div class="form-group">
                            <div class="form-label">Image Vertical Position</div>
                            <select name="settings[image_position]" class="form-control">
                                <?php
    $imagePosition = $selectedSectionSettings['image_position'] ?? 'center';
$positionOptions = [
    'top' => 'Top',
    '25' => 'Upper (25%)',
    'center' => 'Center (50%)',
    '75' => 'Lower (75%)',
    'bottom' => 'Bottom'
];
foreach ($positionOptions as $value => $label):
    ?>
                                    <option value="<?php echo e($value); ?>" <?php echo $imagePosition === $value ? 'selected' : ''; ?>><?php echo e($label); ?></option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">Adjust which part of the image is visible</small>
                        </div>
                        <div class="form-group">
                            <div class="form-label">Overlay Opacity (%)</div>
                            <input type="range" name="settings[overlay_opacity]" class="form-control" min="0" max="90" value="<?php echo e($selectedSectionSettings['overlay_opacity'] ?? '50'); ?>" data-overlay-range>
                            <span data-overlay-output><?php echo e($selectedSectionSettings['overlay_opacity'] ?? '50'); ?>%</span>
                        </div>
                    </div>

                    <div class="settings-section">
                        <h4>Slideshow</h4>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[autoplay]" <?php echo ($selectedSectionSettings['autoplay'] ?? true) ? 'checked' : ''; ?>>
                                <span>Auto-play slides</span>
                            </label>
                        </div>
                        <div class="form-group">
                            <div class="form-label">Slide Interval (seconds)</div>
                            <input type="number" name="settings[interval]" class="form-control" min="2" max="15" value="<?php echo e($selectedSectionSettings['interval'] ?? '5'); ?>">
                        </div>
                    </div>

                    <div class="settings-section">
                        <h4>Link (Optional)</h4>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[link_enabled]" id="banner-link-enabled" <?php echo ($selectedSectionSettings['link_enabled'] ?? false) ? 'checked' : ''; ?> data-banner-link-enabled>
                                <span>Make banner clickable</span>
                            </label>
                        </div>
                        <div id="banner-link-options" class="banner-link-options<?php echo ($selectedSectionSettings['link_enabled'] ?? false) ? '' : CLASS_IS_HIDDEN; ?>">
                            <div class="form-group">
                                <div class="form-label">Link Type</div>
                                <select name="settings[link_type]" class="form-control" id="banner-link-type" data-banner-link-type>
                                    <option value="custom" <?php echo ($selectedSectionSettings['link_type'] ?? 'custom') === 'custom' ? 'selected' : ''; ?>>Custom URL</option>
                                    <option value="page" <?php echo ($selectedSectionSettings['link_type'] ?? 'custom') === 'page' ? 'selected' : ''; ?>>Existing Page</option>
                                </select>
                            </div>
                            <div class="form-group<?php echo ($selectedSectionSettings['link_type'] ?? 'custom') === 'page' ? CLASS_IS_HIDDEN : ''; ?>" id="banner-custom-url">
                                <div class="form-label">Custom URL</div>
                                <input type="text" name="settings[link_url]" class="form-control" value="<?php echo e($selectedSectionSettings['link_url'] ?? ''); ?>" placeholder="https://example.com or /page-slug">
                            </div>
                            <div class="form-group<?php echo ($selectedSectionSettings['link_type'] ?? 'custom') === 'page' ? '' : CLASS_IS_HIDDEN; ?>" id="banner-page-select">
                                <div class="form-label">Select Page</div>
                                <select name="settings[link_page_id]" class="form-control">
                                    <option value="">-- Select a page --</option>
                                    <?php foreach ($pages as $p): ?>
                                    <option value="<?php echo $p['id']; ?>" <?php echo ($selectedSectionSettings['link_page_id'] ?? '') == $p['id'] ? 'selected' : ''; ?>><?php echo e($p['title']); ?> (/<?php echo e($p['slug']); ?>)</option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="form-group">
                                <label class="form-check">
                                    <input type="checkbox" name="settings[link_new_tab]" <?php echo ($selectedSectionSettings['link_new_tab'] ?? false) ? 'checked' : ''; ?>>
                                    <span>Open in new tab</span>
                                </label>
                            </div>
                        </div>
                    </div>

                    <button type="submit" class="btn btn-primary btn-block mt-2">Save Changes</button>
                </form>

                <?php elseif ($selectedSection && $selectedSection['section_type'] === 'collections_list'): ?>
                <!-- Collections List Settings -->
                <form data-section-settings-form>
                    <input type="hidden" name="section_id" value="<?php echo $selectedSection['id']; ?>">

                    <div class="settings-section">
                        <h4>Section Title</h4>
                        <div class="form-group">
                            <div class="form-label">Title</div>
                            <input type="text" name="settings[section_title]" class="form-control" value="<?php echo e($selectedSectionSettings['section_title'] ?? ''); ?>">
                        </div>
                        <div class="form-group">
                            <div class="form-label">Title Alignment</div>
                            <select name="settings[title_align]" class="form-control">
                                <?php
    $titleAlign = $selectedSectionSettings['title_align'] ?? 'center';
                    $alignOptions = ['left' => 'Left', 'center' => 'Center', 'right' => 'Right'];
                    foreach ($alignOptions as $value => $label):
                        ?>
                                    <option value="<?php echo e($value); ?>" <?php echo $titleAlign === $value ? 'selected' : ''; ?>><?php echo e($label); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>

                    <div class="settings-section">
                        <h4>Layout</h4>
                        <div class="form-row">
                            <div class="form-group">
                                <div class="form-label">Columns</div>
                                <select name="settings[columns]" class="form-control">
                                    <?php
                            $columns = (int)($selectedSectionSettings['columns'] ?? 3);
                    foreach ([2, 3, 4] as $col):
                        ?>
                                        <option value="<?php echo $col; ?>" <?php echo $columns === $col ? 'selected' : ''; ?>><?php echo $col; ?> Columns</option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="form-group">
                                <div class="form-label">Layout Style</div>
                                <?php $layoutMode = $selectedSectionSettings['layout_mode'] ?? 'carousel'; ?>
                                <select name="settings[layout_mode]" class="form-control">
                                    <option value="carousel" <?php echo $layoutMode === 'carousel' ? 'selected' : ''; ?>>Carousel</option>
                                    <option value="grid" <?php echo $layoutMode === 'grid' ? 'selected' : ''; ?>>Grid</option>
                                </select>
                            </div>
                        </div>
                    </div>

                    <div class="settings-section">
                        <h4>Visibility</h4>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[show_image]" <?php echo ($selectedSectionSettings['show_image'] ?? true) ? 'checked' : ''; ?>>
                                <span>Show image</span>
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="form-check">
                                <input type="checkbox" name="settings[show_description]" <?php echo ($selectedSectionSettings['show_description'] ?? true) ? 'checked' : ''; ?>>
                                <span>Show description</span>
                            </label>
                        </div>
                    </div>

                    <button type="submit" class="btn btn-primary btn-block mt-2">Save Changes</button>
                </form>

                <?php else: ?>
                <!-- Default: Theme Colors -->
                <div class="settings-section">
                    <h4>Colors</h4>

                    <div class="color-picker-group">
                        <input type="color" id="primary-color" value="<?php echo e($themeSettings['primary_color'] ?? '#a78bff'); ?>" data-theme-setting="primary_color">
                        <input type="text" value="<?php echo e($themeSettings['primary_color'] ?? '#a78bff'); ?>" readonly>
                        <span>Primary</span>
                    </div>

                    <div class="color-picker-group">
                        <input type="color" id="secondary-color" value="<?php echo e($themeSettings['secondary_color'] ?? '#6f5bff'); ?>" data-theme-setting="secondary_color">
                        <input type="text" value="<?php echo e($themeSettings['secondary_color'] ?? '#6f5bff'); ?>" readonly>
                        <span>Secondary</span>
                    </div>

                    <div class="color-picker-group">
                        <input type="color" id="accent-color" value="<?php echo e($themeSettings['accent_color'] ?? '#c9b8ff'); ?>" data-theme-setting="accent_color">
                        <input type="text" value="<?php echo e($themeSettings['accent_color'] ?? '#c9b8ff'); ?>" readonly>
                        <span>Accent</span>
                    </div>

                    <div class="color-picker-group">
                        <input type="color" id="bg-color" value="<?php echo e($themeSettings['background_color'] ?? '#0f0d16'); ?>" data-theme-setting="background_color">
                        <input type="text" value="<?php echo e($themeSettings['background_color'] ?? '#0f0d16'); ?>" readonly>
                        <span>Background</span>
                    </div>
                </div>

                <div class="settings-section">
                    <h4>Typography</h4>
                    <div class="form-group">
                        <div class="form-label">Font Family</div>
                        <select class="form-control" data-theme-setting="font_family">
                            <?php
                            $fontOptions = [
                                ['label' => 'System Default', 'value' => 'system-ui, sans-serif']
                            ];

                    foreach ($customFonts as $font) {
                        $name = isset($font['name']) ? trim($font['name']) : '';
                        if ($name === '') {
                            continue;
                        }
                        $fontOptions[] = [
                            'label' => $name . ' (Custom)',
                            'value' => '"' . $name . '", sans-serif'
                        ];
                    }

                    $currentFont = $themeSettings['font_family'] ?? 'system-ui, sans-serif';
                    foreach ($fontOptions as $option):
                        ?>
                                <option value="<?php echo e($option['value']); ?>" <?php echo $currentFont === $option['value'] ? 'selected' : ''; ?>><?php echo e($option['label']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Add Section Modal -->
<div class="modal-overlay" id="add-section-modal">
    <div class="modal">
        <div class="modal-header">
            <h3>Add Section</h3>
            <button class="modal-close">&times;</button>
        </div>
        <div class="modal-body">
            <div class="section-type-list">
                <?php foreach ($sectionTypes as $type): ?>
                <div class="section-type-item" data-section-type="<?php echo e($type['type']); ?>">
                    <div class="icon"><?php echo $type['icon']; ?></div>
                    <div class="info">
                        <h4><?php echo e($type['name']); ?></h4>
                        <p><?php echo e($type['description']); ?></p>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>
</div>

<script nonce="<?php echo e(getCspNonce()); ?>">
const pageId = <?php echo (int)$pageId; ?>;

function addSection(type) {
    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'add_section');
    formData.append('type', type);
    formData.append('page_id', pageId);

    fetch('/admin/theme-editor.php', {
        method: 'POST',
        headers: {
            'X-CSRF-Token': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        },
        body: formData
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) {
            location.href = '/admin/theme-editor.php?page=' + pageId + '&section=' + data.id;
        }
    });
}

document.querySelectorAll('#add-section-modal .section-type-item[data-section-type]').forEach(item => {
    item.addEventListener('click', () => {
        const type = item.getAttribute('data-section-type');
        if (type) {
            addSection(type);
        }
    });
});

document.querySelectorAll('.page-section-item[data-section-link]').forEach(item => {
    item.addEventListener('click', () => {
        const link = item.getAttribute('data-section-link');
        if (link) {
            location.href = link;
        }
    });
});

document.querySelectorAll('.page-section-item [data-remove-section]').forEach(btn => {
    btn.addEventListener('click', (event) => {
        event.preventDefault();
        event.stopPropagation();
        const id = btn.getAttribute('data-remove-section');
        if (id) {
            removeSection(id);
        }
    });
});

const pageSelector = document.querySelector('[data-page-selector]');
if (pageSelector) {
    pageSelector.addEventListener('change', () => {
        const value = pageSelector.value;
        if (value) {
            location.href = '/admin/theme-editor.php?page=' + value;
        }
    });
}

// Theme setting inputs
document.querySelectorAll('[data-theme-setting]').forEach(input => {
    const handler = () => {
        const key = input.getAttribute('data-theme-setting');
        if (!key) return;
        let value = input.value;
        const isCheckbox = input.getAttribute('data-theme-type') === 'checkbox' || input.type === 'checkbox';
        if (isCheckbox) {
            value = input.checked ? '1' : '0';
        }
        updateThemeSetting(key, value);
        if (input.getAttribute('data-theme-toggle') === 'logo') {
            toggleLogoOptions(value);
        }
    };

    input.addEventListener('change', handler);
    if (input.type === 'color') {
        input.addEventListener('input', handler);
    }
});

// Banner link toggles
document.querySelectorAll('[data-banner-link-enabled]').forEach(input => {
    input.addEventListener('change', toggleBannerLinkOptions);
});
document.querySelectorAll('[data-banner-link-type]').forEach(select => {
    select.addEventListener('change', toggleBannerLinkType);
});

// Overlay opacity range
document.querySelectorAll('[data-overlay-range]').forEach(range => {
    range.addEventListener('input', () => {
        const output = range.parentElement?.querySelector('[data-overlay-output]') || range.nextElementSibling;
        if (output) {
            output.textContent = range.value + '%';
        }
    });
});

// Theme editor action buttons
document.addEventListener('click', (event) => {
    const btn = event.target.closest('[data-action]');
    if (!btn) return;
    const action = btn.getAttribute('data-action');
    if (!action) return;

    if (action === 'refresh-preview') {
        const frame = document.getElementById('preview-frame');
        if (frame && frame.contentWindow) {
            frame.contentWindow.location.reload();
        }
        return;
    }

    if (action === 'open-media-picker') {
        const targetId = btn.getAttribute('data-picker-target');
        const type = btn.getAttribute('data-picker-type') || 'image';
        const input = targetId ? document.getElementById(targetId) : null;
        if (input) {
            openMediaPicker(input, type);
        }
        return;
    }

    if (action === 'add-slide') {
        addSlide();
        return;
    }

    if (action === 'remove-slide') {
        removeSlide(btn);
        return;
    }

    if (action === 'add-testimonial') {
        addTestimonial();
        return;
    }

    if (action === 'remove-testimonial') {
        removeTestimonial(btn);
        return;
    }

    if (action === 'add-banner-image') {
        addBannerImage();
        return;
    }

    if (action === 'remove-banner-image') {
        const index = parseInt(btn.getAttribute('data-index') || '', 10);
        if (!Number.isNaN(index)) {
            removeBannerImage(index);
        }
        return;
    }

    if (action === 'banner-choose-image') {
        openImagePickerForBanner(btn);
    }
});

function removeSection(sectionId) {
    if (!confirm('Remove this section?')) return;

    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'remove_section');
    formData.append('section_id', sectionId);

    fetch('/admin/theme-editor.php', {
        method: 'POST',
        headers: {
            'X-CSRF-Token': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        },
        body: formData
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) {
            location.href = '/admin/theme-editor.php?page=' + pageId;
        }
    });
}

function toggleLogoOptions(type) {
    const textOptions = document.getElementById('logo-text-options');
    const imageOptions = document.getElementById('logo-image-options');

    if (type === 'image') {
        textOptions.classList.add('is-hidden');
        imageOptions.classList.remove('is-hidden');
    } else {
        textOptions.classList.remove('is-hidden');
        imageOptions.classList.add('is-hidden');
    }
}

function updateThemeSetting(key, value) {
    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'update_theme');
    formData.append('key', key);
    formData.append('value', value);

    fetch('/admin/theme-editor.php', {
        method: 'POST',
        headers: {
            'X-CSRF-Token': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        },
        body: formData
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) {
            document.getElementById('preview-frame').contentWindow.location.reload();
        }
    });
}

// Section settings form
const settingsForm = document.querySelector('[data-section-settings-form]');
if (settingsForm) {
    settingsForm.addEventListener('submit', function(e) {
        e.preventDefault();

        const formData = new FormData(this);
        formData.append('ajax', '1');
        formData.append('action', 'update_section');

        // Convert form data to proper settings object
        const settings = {};
        for (let [key, value] of formData.entries()) {
            if (key.startsWith('settings[')) {
                // Parse nested keys like settings[slides][0][title]
                const matches = key.match(/settings\[([^\]]+)\](?:\[([^\]]+)\])?(?:\[([^\]]+)\])?/);
                if (matches) {
                    const k1 = matches[1];
                    const k2 = matches[2];
                    const k3 = matches[3];

                    if (k3) {
                        if (!settings[k1]) settings[k1] = [];
                        if (!settings[k1][k2]) settings[k1][k2] = {};
                        settings[k1][k2][k3] = value;
                    } else if (k2) {
                        if (!settings[k1]) settings[k1] = {};
                        settings[k1][k2] = value;
                    } else {
                        settings[k1] = value === 'on' ? true : value;
                    }
                }
            }
        }

        // Handle checkboxes that might be unchecked
        if (settingsForm.querySelector('input[name="settings[autoplay]"]')) {
            settings.autoplay = formData.has('settings[autoplay]');
        }
        if (settingsForm.querySelector('input[name="settings[show_arrows]"]')) {
            settings.show_arrows = formData.has('settings[show_arrows]');
        }
        if (settingsForm.querySelector('input[name="settings[show_dots]"]')) {
            settings.show_dots = formData.has('settings[show_dots]');
        }
        if (settingsForm.querySelector('input[name="settings[auto_scroll]"]')) {
            settings.auto_scroll = formData.has('settings[auto_scroll]');
        }
        if (settingsForm.querySelector('input[name="settings[show_media]"]')) {
            settings.show_media = formData.has('settings[show_media]');
        }
        if (settingsForm.querySelector('input[name="settings[show_description]"]')) {
            settings.show_description = formData.has('settings[show_description]');
        }
        if (settingsForm.querySelector('input[name="settings[show_price]"]')) {
            settings.show_price = formData.has('settings[show_price]');
        }
        if (settingsForm.querySelector('input[name="settings[show_image]"]')) {
            settings.show_image = formData.has('settings[show_image]');
        }
        // Image banner checkboxes
        if (settingsForm.querySelector('input[name="settings[link_enabled]"]')) {
            settings.link_enabled = formData.has('settings[link_enabled]');
        }
        if (settingsForm.querySelector('input[name="settings[link_new_tab]"]')) {
            settings.link_new_tab = formData.has('settings[link_new_tab]');
        }

        // Convert array-like objects to proper arrays for slides, images, testimonials
        ['slides', 'images', 'testimonials'].forEach(key => {
            if (settings[key] && typeof settings[key] === 'object') {
                // Convert object with numeric keys to array
                const arr = [];
                Object.keys(settings[key]).forEach(k => {
                    if (settings[key][k] && typeof settings[key][k] === 'object') {
                        arr.push(settings[key][k]);
                    }
                });
                settings[key] = arr;
            }
        });

        const postData = new FormData();
        postData.append('ajax', '1');
        postData.append('action', 'update_section');
        postData.append('section_id', formData.get('section_id'));
        postData.append('settings', JSON.stringify(settings));

        fetch('/admin/theme-editor.php', {
            method: 'POST',
            headers: {
                'X-CSRF-Token': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: postData
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                document.getElementById('preview-frame').contentWindow.location.reload();
                showNotification('Settings saved!', 'success');
            }
        });
    });
}

let slideCount = <?php echo isset($selectedSectionSettings['slides']) ? count($selectedSectionSettings['slides']) : 0; ?>;
let testimonialCount = <?php echo isset($selectedSectionSettings['testimonials']) ? count($selectedSectionSettings['testimonials']) : 0; ?>;

const slideProductOptionsTemplate = document.getElementById('slide-product-options');
const slideProductOptionsHtml = slideProductOptionsTemplate ? slideProductOptionsTemplate.innerHTML : '<option value="">-- No products available --</option>';

function syncSlideControls(card) {
    if (!card) return;

    const linkType = card.querySelector('.js-slide-link-type');
    const customUrlGroup = card.querySelector('.js-slide-custom-url');
    const productGroup = card.querySelector('.js-slide-product-select');
    const productSelect = card.querySelector('.js-slide-product');
    const mediaSource = card.querySelector('.js-slide-media-source');
    const mediaManual = card.querySelector('.js-slide-media-manual');
    const mediaProduct = card.querySelector('.js-slide-media-product');

    const linkTypeValue = linkType ? linkType.value : 'custom';
    const hasProduct = productSelect && productSelect.value !== '';

    if (customUrlGroup) {
        customUrlGroup.classList.toggle('is-hidden', linkTypeValue === 'product');
    }

    if (productGroup) {
        const showProduct = linkTypeValue === 'product' ||
            (mediaSource && mediaSource.value === 'product') ||
            (productSelect && productSelect.value !== '');
        productGroup.classList.toggle('is-hidden', !showProduct);
    }

    if (mediaSource) {
        const productOption = mediaSource.querySelector('option[value="product"]');
        if (productOption) {
            productOption.disabled = !hasProduct;
        }
        if (!hasProduct && mediaSource.value === 'product') {
            mediaSource.value = 'manual';
        }
    }

    if (mediaManual) {
        const isManual = !mediaSource || mediaSource.value === 'manual';
        mediaManual.classList.toggle('is-hidden', !isManual);
    }

    if (mediaProduct) {
        const isProduct = mediaSource && mediaSource.value === 'product';
        mediaProduct.classList.toggle('is-hidden', !isProduct);
    }
}

function initSlideControls() {
    document.querySelectorAll('.slide-card').forEach(card => {
        syncSlideControls(card);
    });
}

document.addEventListener('change', (event) => {
    const target = event.target;
    if (!target) return;
    if (target.classList.contains('js-slide-link-type') ||
        target.classList.contains('js-slide-media-source') ||
        target.classList.contains('js-slide-product')) {
        const card = target.closest('.slide-card');
        if (card) {
            syncSlideControls(card);
        }
    }
});

function addSlide() {
    const container = document.getElementById('slides-container');
    const i = slideCount++;
    const slideNum = i + 1;

    const slideHtml = `
        <div class="card slide-card" data-slide-index="${i}">
            <div class="card-header">
                <span>Slide ${slideNum}</span>
                <button type="button" class="btn btn-sm btn-danger btn-slide-remove" data-action="remove-slide">Remove</button>
            </div>
            <div class="card-body">
                <div class="form-group">
                    <label>Title</label>
                    <input type="text" name="settings[slides][${i}][title]" class="form-control" value="Slide ${slideNum} Title">
                </div>
                <div class="form-group">
                    <label>Description</label>
                    <textarea name="settings[slides][${i}][description]" class="form-control" rows="2">Enter your description here.</textarea>
                </div>
                <div class="form-group">
                    <label>Button Text</label>
                    <input type="text" name="settings[slides][${i}][button_text]" class="form-control" value="Learn More">
                </div>
                <div class="form-group">
                    <label>Button Link</label>
                    <select name="settings[slides][${i}][button_link_type]" class="form-control js-slide-link-type">
                        <option value="custom" selected>Custom URL</option>
                        <option value="product">Product</option>
                    </select>
                </div>
                <div class="form-group js-slide-custom-url">
                    <label>Button URL</label>
                    <input type="text" name="settings[slides][${i}][button_url]" class="form-control" value="#">
                    <small class="text-muted">Use https:// or a /relative path.</small>
                </div>
                <div class="form-group js-slide-product-select">
                    <label>Product (link or image)</label>
                    <select name="settings[slides][${i}][button_product_id]" class="form-control js-slide-product"></select>
                    <small class="text-muted">Select a product to use its link or image.</small>
                </div>
                <div class="form-group">
                    <label>Image/Video Source</label>
                    <select name="settings[slides][${i}][media_source]" class="form-control js-slide-media-source">
                        <option value="manual" selected>Manual</option>
                        <option value="product">Product image</option>
                    </select>
                    <small class="text-muted">Product image requires a selected product.</small>
                </div>
                <div class="form-group js-slide-media-manual">
                    <label>Image/Video</label>
                    <div class="input-with-picker">
                        <input type="text" name="settings[slides][${i}][image]" class="form-control" value="/assets/images/placeholder-${(i % 3) + 1}.svg" id="slide-image-${i}">
                        <button type="button" class="btn btn-secondary btn-picker" data-action="open-media-picker" data-picker-target="slide-image-${i}" data-picker-type="image">🖼️ Browse</button>
                    </div>
                </div>
                <div class="form-group js-slide-media-product">
                    <div class="text-muted">Uses the selected product's primary media.</div>
                </div>
            </div>
        </div>
    `;

    container.insertAdjacentHTML('beforeend', slideHtml);
    const newCard = container.querySelector(`.slide-card[data-slide-index="${i}"]`);
    if (newCard) {
        const productSelect = newCard.querySelector('.js-slide-product');
        if (productSelect) {
            productSelect.innerHTML = slideProductOptionsHtml;
        }
        syncSlideControls(newCard);
    }
    showNotification('New slide added. Remember to save changes!', 'info');
}

initSlideControls();

function addTestimonial() {
    const container = document.getElementById('testimonials-container');
    if (!container) return;

    const i = testimonialCount++;
    const itemHtml = `
        <div class="card section-item-card mb-2" data-testimonial-index="${i}">
            <div class="card-header section-item-header">
                <span>Testimonial ${i + 1}</span>
                <button type="button" class="btn btn-sm btn-danger" data-action="remove-testimonial">Remove</button>
            </div>
            <div class="card-body section-item-body">
                <div class="form-group">
                    <label>Title</label>
                    <input type="text" name="settings[testimonials][${i}][title]" class="form-control" value="">
                </div>
                <div class="form-group">
                    <label>Author Name</label>
                    <input type="text" name="settings[testimonials][${i}][author]" class="form-control" value="">
                </div>
                <div class="form-group">
                    <label>Comment</label>
                    <textarea name="settings[testimonials][${i}][comment]" class="form-control" rows="3"></textarea>
                </div>
                <div class="form-group">
                    <label>Stars</label>
                    <select name="settings[testimonials][${i}][stars]" class="form-control">
                        <option value="1">1</option>
                        <option value="2">2</option>
                        <option value="3">3</option>
                        <option value="4">4</option>
                        <option value="5" selected>5</option>
                    </select>
                </div>
            </div>
        </div>
    `;

    container.insertAdjacentHTML('beforeend', itemHtml);
}

function removeTestimonial(button) {
    const card = button.closest('.section-item-card');
    if (card) {
        card.remove();
    }
}

// Image Banner functions
let bannerImageCount = <?php echo isset($selectedSectionSettings['images']) ? count($selectedSectionSettings['images']) : 0; ?>;

function addBannerImage() {
    const container = document.getElementById('banner-images-container');
    if (!container) return;

    const i = bannerImageCount++;
    const imageHtml = `
        <div class="slide-item" data-index="${i}">
            <div class="slide-item-header">
                <span>Image ${i + 1}</span>
                <button type="button" class="btn-icon" data-action="remove-banner-image" data-index="${i}" title="Remove">&times;</button>
            </div>
            <div class="form-group">
                <label>Image</label>
                <div class="image-picker">
                    <input type="hidden" name="settings[images][${i}][image]" value="">
                    <div class="image-preview">
                        <span>No image</span>
                    </div>
                    <button type="button" class="btn btn-secondary btn-sm" data-action="banner-choose-image">Choose Image</button>
                </div>
            </div>
        </div>
    `;

    container.insertAdjacentHTML('beforeend', imageHtml);
    showNotification('Image slot added. Choose an image and save!', 'info');
}

function removeBannerImage(index) {
    const container = document.getElementById('banner-images-container');
    if (!container) return;

    const item = container.querySelector(`.slide-item[data-index="${index}"]`);
    if (item && confirm('Remove this image?')) {
        item.remove();
        showNotification('Image removed. Remember to save changes!', 'warning');
    }
}

function toggleBannerLinkOptions() {
    const checkbox = document.getElementById('banner-link-enabled');
    const options = document.getElementById('banner-link-options');
    if (checkbox && options) {
        options.classList.toggle('is-hidden', !checkbox.checked);
    }
}

function toggleBannerLinkType() {
    const select = document.getElementById('banner-link-type');
    const customUrl = document.getElementById('banner-custom-url');
    const pageSelect = document.getElementById('banner-page-select');

    if (select && customUrl && pageSelect) {
        if (select.value === 'page') {
            customUrl.classList.add('is-hidden');
            pageSelect.classList.remove('is-hidden');
        } else {
            customUrl.classList.remove('is-hidden');
            pageSelect.classList.add('is-hidden');
        }
    }
}

// Image picker helper for image-picker divs
function openImagePickerForBanner(button) {
    const picker = button.closest('.image-picker');
    const hiddenInput = picker.querySelector('input[type="hidden"]');
    const preview = picker.querySelector('.image-preview');

    openMediaPicker(function(path) {
        // Update hidden input
        hiddenInput.value = path;

        // Update preview
        let img = preview.querySelector('.image-preview-img');
        if (!img) {
            preview.innerHTML = '';
            img = document.createElement('img');
            img.className = 'image-preview-img';
            img.alt = 'Banner image preview';
            preview.appendChild(img);
        }
        img.src = path;
    }, 'image');
}

function removeSlide(btn) {
    if (confirm('Remove this slide?')) {
        btn.closest('.slide-card').remove();
        showNotification('Slide removed. Remember to save changes!', 'warning');
    }
}

// Device switcher
document.querySelectorAll('.device-btn').forEach(btn => {
    btn.addEventListener('click', function() {
        document.querySelectorAll('.device-btn').forEach(b => b.classList.remove('active'));
        this.classList.add('active');

        const frame = document.getElementById('preview-frame');
        const device = this.dataset.device;
        frame.classList.remove('tablet', 'mobile');
        if (device !== 'desktop') {
            frame.classList.add(device);
        }
    });
});

// Sections drag reorder
const sectionsList = document.getElementById('sections-list');
if (sectionsList) {
    sectionsList.addEventListener('orderChanged', function() {
        const order = Array.from(this.querySelectorAll('.page-section-item')).map(el => el.dataset.id);

        const formData = new FormData();
        formData.append('ajax', '1');
        formData.append('action', 'reorder_sections');
        order.forEach((id, i) => formData.append('order[]', id));

        fetch('/admin/theme-editor.php', {
            method: 'POST',
            body: formData
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                document.getElementById('preview-frame').contentWindow.location.reload();
            }
        });
    });
}
</script>

<?php AdminLayout::renderFooter(); ?>
