<?php
/**
 * Admin Panel - Text Finder & Replacer
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Auth\AdminAuth;
use NewSite\Admin\AdminLayout;

AdminAuth::requireLogin();

$message = '';
$error = '';

$searchText = (string)($_POST['search_text'] ?? '');
$replaceText = (string)($_POST['replace_text'] ?? '');
$inputText = (string)($_POST['input_text'] ?? '');
$caseSensitive = isset($_POST['case_sensitive']);

$pasteResultText = null;
$pasteReplacementCount = 0;
$fileRunRows = [];
$downloadArchiveName = '';

$maxUploadBytesPerFile = 5 * 1024 * 1024; // 5 MB
$maxPastedTextBytes = 5 * 1024 * 1024; // 5 MB
$maxUploadCount = 25;
$allowedTextExtensions = ['txt'];

// No disk persistence — zip is built in a temp file and streamed on download,
// then deleted immediately. Nothing is stored inside the project folder.

function normalizeUploadedFiles(array $files): array
{
    $normalized = [];

    if (!isset($files['name'])) {
        return $normalized;
    }

    if (is_array($files['name'])) {
        foreach ($files['name'] as $index => $name) {
            $normalized[] = [
                'name' => (string)($name ?? ''),
                'type' => (string)($files['type'][$index] ?? ''),
                'tmp_name' => (string)($files['tmp_name'][$index] ?? ''),
                'error' => (int)($files['error'][$index] ?? UPLOAD_ERR_NO_FILE),
                'size' => (int)($files['size'][$index] ?? 0),
            ];
        }

        return $normalized;
    }

    $normalized[] = [
        'name' => (string)($files['name'] ?? ''),
        'type' => (string)($files['type'] ?? ''),
        'tmp_name' => (string)($files['tmp_name'] ?? ''),
        'error' => (int)($files['error'] ?? UPLOAD_ERR_NO_FILE),
        'size' => (int)($files['size'] ?? 0),
    ];

    return $normalized;
}

function isAllowedTextFileByName(string $path, array $allowedExtensions): bool
{
    $extension = strtolower((string)pathinfo($path, PATHINFO_EXTENSION));
    return in_array($extension, $allowedExtensions, true);
}

function sanitizeArchiveEntryPath(string $entryPath): string
{
    $normalized = str_replace('\\', '/', trim($entryPath));
    $normalized = ltrim($normalized, '/');

    $isValid = true;
    if ($normalized === '' || str_ends_with($normalized, '/')) {
        $isValid = false;
    }

    if ($isValid && (strpos($normalized, '../') !== false || strpos($normalized, '/..') !== false)) {
        $isValid = false;
    }

    if ($isValid && str_starts_with($normalized, '..')) {
        $isValid = false;
    }

    return $isValid ? $normalized : '';
}

function makeSafeOutputName(string $name, string $fallbackExtension = 'txt'): string
{
    $clean = preg_replace('/[^a-zA-Z0-9._-]/', '_', $name);
    $clean = trim((string)$clean, '._-');

    if ($clean === '') {
        $clean = 'file.' . $fallbackExtension;
    }

    return $clean;
}

function applyTextReplacement(string $source, string $needle, string $replacement, bool $caseSensitive, int &$replacementCount): string
{
    $replacementCount = 0;
    if ($needle === '') {
        return $source;
    }

    if ($caseSensitive) {
        $replacementCount = substr_count($source, $needle);
        return str_replace($needle, $replacement, $source);
    }

    $pattern = '/' . preg_quote($needle, '/') . '/iu';
    $result = preg_replace_callback(
        $pattern,
        static function () use ($replacement) {
            return $replacement;
        },
        $source,
        -1,
        $replacementCount
    );

    return is_string($result) ? $result : $source;
}

// ── Handle download request (zip is stored in a temp file, path kept in session) ──
$downloadRequested = isset($_GET['download']) && $_GET['download'] === '1';
if ($downloadRequested) {
    $tempPath = (string)($_SESSION['text_finder_zip_path'] ?? '');
    $tempName = (string)($_SESSION['text_finder_zip_name'] ?? 'text-replace.zip');
    unset($_SESSION['text_finder_zip_path'], $_SESSION['text_finder_zip_name']);

    if ($tempPath !== '' && is_file($tempPath)) {
        header('Content-Type: application/zip');
        header('Content-Disposition: attachment; filename="' . basename($tempName) . '"');
        header('Content-Length: ' . (string)filesize($tempPath));
        header('Cache-Control: private, no-store, no-cache, must-revalidate, max-age=0');
        header('Pragma: no-cache');
        readfile($tempPath);
        @unlink($tempPath); // delete temp file immediately after streaming
        exit;
    }

    $error = 'Export file has expired or was already downloaded. Please run the replacement again.';
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verifyCSRF();

    $action = (string)($_POST['action'] ?? '');
    $caseSensitive = isset($_POST['case_sensitive']);

    if ($searchText === '') {
        $error = 'Search text is required.';
    } elseif ($action === 'replace_paste') {
        if (strlen($inputText) > $maxPastedTextBytes) {
            $error = 'Pasted text is too large. Maximum size is 5 MB.';
        } else {
            $pasteResultText = applyTextReplacement($inputText, $searchText, $replaceText, $caseSensitive, $pasteReplacementCount);
            $message = 'Replacement complete. Matches replaced: ' . $pasteReplacementCount . '.';
        }
    } elseif ($action === 'replace_files') {
        $uploadedFiles = normalizeUploadedFiles($_FILES['source_files'] ?? []);
        $uploadedFiles = array_values(array_filter($uploadedFiles, static function (array $file): bool {
            return (int)($file['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_NO_FILE;
        }));

        if (empty($uploadedFiles)) {
            $error = 'Please upload at least one .txt file or a .zip archive containing .txt files.';
        } elseif (count($uploadedFiles) > $maxUploadCount) {
            $error = 'Too many uploads. Maximum allowed files per run: ' . $maxUploadCount . '.';
        } elseif (!class_exists('ZipArchive')) {
            $error = 'ZipArchive is not available on this server. Please upload plain .txt files.';
        } else {
            $totalProcessedFiles = 0;
            $totalSkippedFiles = 0;
            $totalReplacementCount = 0;

            $downloadArchiveName = 'text-replace-' . date('Ymd-His') . '-' . bin2hex(random_bytes(4)) . '.zip';
            $downloadArchiveTmp = tempnam(sys_get_temp_dir(), 'tfr_');
            if ($downloadArchiveTmp === false) {
                $downloadArchiveTmp = '';
            }

            $outputZip = new ZipArchive();
            $openResult = $downloadArchiveTmp !== ''
                ? $outputZip->open($downloadArchiveTmp, ZipArchive::CREATE | ZipArchive::OVERWRITE)
                : false;
            if ($openResult !== true) {
                $error = 'Could not create export archive.';
                $downloadArchiveName = '';
            } else {
                foreach ($uploadedFiles as $uploadedFile) {
                    $uploadName = basename((string)($uploadedFile['name'] ?? ''));
                    $uploadName = $uploadName !== '' ? $uploadName : 'upload.txt';
                    $uploadError = (int)($uploadedFile['error'] ?? UPLOAD_ERR_NO_FILE);
                    $uploadSize = (int)($uploadedFile['size'] ?? 0);
                    $tmpName = (string)($uploadedFile['tmp_name'] ?? '');

                    if ($uploadError !== UPLOAD_ERR_OK || $tmpName === '' || !is_uploaded_file($tmpName)) {
                        $fileRunRows[] = [
                            'source' => $uploadName,
                            'processed' => 0,
                            'replacements' => 0,
                            'note' => 'Upload failed (code ' . $uploadError . ').'
                        ];
                        $totalSkippedFiles++;
                        continue;
                    }

                    $uploadExtension = strtolower((string)pathinfo($uploadName, PATHINFO_EXTENSION));

                    if ($uploadExtension === 'txt') {
                        if ($uploadSize > $maxUploadBytesPerFile) {
                            $fileRunRows[] = [
                                'source' => $uploadName,
                                'processed' => 0,
                                'replacements' => 0,
                                'note' => 'Skipped (file larger than 5 MB).'
                            ];
                            $totalSkippedFiles++;
                            continue;
                        }

                        $fileContent = @file_get_contents($tmpName);
                        if (!is_string($fileContent)) {
                            $fileRunRows[] = [
                                'source' => $uploadName,
                                'processed' => 0,
                                'replacements' => 0,
                                'note' => 'Skipped (unable to read file).'
                            ];
                            $totalSkippedFiles++;
                            continue;
                        }

                        $replacementCount = 0;
                        $resultContent = applyTextReplacement($fileContent, $searchText, $replaceText, $caseSensitive, $replacementCount);
                        $safeOutputName = makeSafeOutputName($uploadName, 'txt');

                        if ($outputZip->addFromString($safeOutputName, $resultContent) === false) {
                            $fileRunRows[] = [
                                'source' => $uploadName,
                                'processed' => 0,
                                'replacements' => 0,
                                'note' => 'Skipped (failed to add to export).'
                            ];
                            $totalSkippedFiles++;
                            continue;
                        }

                        $fileRunRows[] = [
                            'source' => $uploadName,
                            'processed' => 1,
                            'replacements' => $replacementCount,
                            'note' => 'Processed as plain text file.'
                        ];

                        $totalProcessedFiles++;
                        $totalReplacementCount += $replacementCount;
                        continue;
                    }

                    if ($uploadExtension !== 'zip') {
                        $fileRunRows[] = [
                            'source' => $uploadName,
                            'processed' => 0,
                            'replacements' => 0,
                            'note' => 'Skipped (only .txt and .zip are supported).'
                        ];
                        $totalSkippedFiles++;
                        continue;
                    }

                    if ($uploadSize > ($maxUploadBytesPerFile * 4)) {
                        $fileRunRows[] = [
                            'source' => $uploadName,
                            'processed' => 0,
                            'replacements' => 0,
                            'note' => 'Skipped archive (larger than 20 MB).'
                        ];
                        $totalSkippedFiles++;
                        continue;
                    }

                    $inputZip = new ZipArchive();
                    $zipOpenResult = $inputZip->open($tmpName);
                    if ($zipOpenResult !== true) {
                        $fileRunRows[] = [
                            'source' => $uploadName,
                            'processed' => 0,
                            'replacements' => 0,
                            'note' => 'Skipped archive (cannot open).'
                        ];
                        $totalSkippedFiles++;
                        continue;
                    }

                    $archivePrefix = makeSafeOutputName((string)pathinfo($uploadName, PATHINFO_FILENAME), 'txt');
                    $archiveProcessed = 0;
                    $archiveSkipped = 0;
                    $archiveReplacements = 0;

                    for ($i = 0; $i < $inputZip->numFiles; $i++) {
                        $entryNameRaw = (string)$inputZip->getNameIndex($i);
                        $entryName = sanitizeArchiveEntryPath($entryNameRaw);
                        if ($entryName === '' || !isAllowedTextFileByName($entryName, $allowedTextExtensions)) {
                            $archiveSkipped++;
                            continue;
                        }

                        $entryStat = $inputZip->statIndex($i);
                        $entrySize = (int)($entryStat['size'] ?? 0);
                        if ($entrySize > $maxUploadBytesPerFile) {
                            $archiveSkipped++;
                            continue;
                        }

                        $entryContent = $inputZip->getFromIndex($i);
                        if (!is_string($entryContent)) {
                            $archiveSkipped++;
                            continue;
                        }

                        $entryReplacementCount = 0;
                        $entryResult = applyTextReplacement($entryContent, $searchText, $replaceText, $caseSensitive, $entryReplacementCount);

                        $outputEntryName = $archivePrefix . '/' . $entryName;
                        if ($outputZip->addFromString($outputEntryName, $entryResult) === false) {
                            $archiveSkipped++;
                            continue;
                        }

                        $archiveProcessed++;
                        $archiveReplacements += $entryReplacementCount;
                    }

                    $inputZip->close();

                    $fileRunRows[] = [
                        'source' => $uploadName,
                        'processed' => $archiveProcessed,
                        'replacements' => $archiveReplacements,
                        'note' => 'Archive processed. Skipped entries: ' . $archiveSkipped . '.'
                    ];

                    $totalProcessedFiles += $archiveProcessed;
                    $totalSkippedFiles += $archiveSkipped;
                    $totalReplacementCount += $archiveReplacements;
                }

                $outputZip->close();

                if ($totalProcessedFiles <= 0) {
                    @unlink($downloadArchiveTmp);
                    $downloadArchiveName = '';
                    $error = 'No valid .txt content was processed from the provided files.';
                } else {
                    // Store temp path in session so the download link can stream it
                    $_SESSION['text_finder_zip_path'] = $downloadArchiveTmp;
                    $_SESSION['text_finder_zip_name'] = $downloadArchiveName;
                    $message = 'Replacement complete. Processed files: ' . $totalProcessedFiles
                        . '. Total matches replaced: ' . $totalReplacementCount
                        . '. Skipped entries: ' . $totalSkippedFiles . '.';
                }
            }
        }
    } else {
        $error = 'Invalid action requested.';
    }
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header content-header-actions">
        <div>
            <h1>Text Finder &amp; Replacer</h1>
            <p>Find and replace text in pasted content, .txt files, or .zip archives containing .txt files.</p>
        </div>
        <a href="/admin/pages.php" class="btn btn-secondary">System Pages</a>
    </div>

    <?php if ($message !== ''): ?>
    <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>

    <?php if ($error !== ''): ?>
    <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <div class="card">
        <div class="card-header">
            <h3>Replace in Pasted Text</h3>
        </div>
        <div class="card-body">
            <form method="POST" action="/admin/text-finder.php">
                <input type="hidden" name="action" value="replace_paste">
                <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">

                <div class="form-row">
                    <div class="form-group">
                        <label for="paste-search-text">Find</label>
                        <input type="text" id="paste-search-text" name="search_text" class="form-control" value="<?php echo e($searchText); ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="paste-replace-text">Replace with</label>
                        <input type="text" id="paste-replace-text" name="replace_text" class="form-control" value="<?php echo e($replaceText); ?>">
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="case_sensitive" <?php echo $caseSensitive ? 'checked' : ''; ?>>
                        <span>Case sensitive match</span>
                    </label>
                </div>

                <div class="form-group">
                    <label for="paste-input-text">Input text</label>
                    <textarea id="paste-input-text" name="input_text" class="form-control" rows="12" placeholder="Paste source text here..."><?php echo e($inputText); ?></textarea>
                </div>

                <button type="submit" class="btn btn-primary">Find &amp; Replace</button>
            </form>
        </div>
    </div>

    <?php if ($pasteResultText !== null): ?>
    <div class="card">
        <div class="card-header">
            <h3>Paste Result</h3>
        </div>
        <div class="card-body">
            <p class="text-muted">Replacements made: <?php echo (int)$pasteReplacementCount; ?></p>
            <div class="form-group">
                <label for="paste-output-text">Output text</label>
                <textarea id="paste-output-text" class="form-control" rows="12"><?php echo e($pasteResultText); ?></textarea>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <div class="card">
        <div class="card-header">
            <h3>Replace in Files / Archives</h3>
        </div>
        <div class="card-body">
            <form method="POST" action="/admin/text-finder.php" enctype="multipart/form-data">
                <input type="hidden" name="action" value="replace_files">
                <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">

                <div class="form-row">
                    <div class="form-group">
                        <label for="files-search-text">Find</label>
                        <input type="text" id="files-search-text" name="search_text" class="form-control" value="<?php echo e($searchText); ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="files-replace-text">Replace with</label>
                        <input type="text" id="files-replace-text" name="replace_text" class="form-control" value="<?php echo e($replaceText); ?>">
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="case_sensitive" <?php echo $caseSensitive ? 'checked' : ''; ?>>
                        <span>Case sensitive match</span>
                    </label>
                </div>

                <div class="form-group">
                    <label for="source-files">Upload files</label>
                    <input type="file" id="source-files" name="source_files[]" class="form-control" accept=".txt,.zip" multiple required>
                    <small class="text-muted">Supported: .txt and .zip (only .txt files inside archives are processed). Max 25 uploads per run.</small>
                </div>

                <button type="submit" class="btn btn-primary">Process Files</button>
            </form>
        </div>
    </div>

    <?php if (!empty($fileRunRows)): ?>
    <div class="card">
        <div class="card-header file-manager-header">
            <h3>Processing Summary</h3>
            <?php if ($downloadArchiveName !== ''): ?>
            <a class="btn btn-primary btn-sm" href="/admin/text-finder.php?download=1">Download Replaced Files (.zip)</a>
            <?php endif; ?>
        </div>
        <div class="card-body">
            <div class="table-container">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Source</th>
                            <th>Processed Files</th>
                            <th>Replacements</th>
                            <th>Notes</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($fileRunRows as $row): ?>
                        <tr>
                            <td><?php echo e((string)($row['source'] ?? '')); ?></td>
                            <td><?php echo (int)($row['processed'] ?? 0); ?></td>
                            <td><?php echo (int)($row['replacements'] ?? 0); ?></td>
                            <td><?php echo e((string)($row['note'] ?? '')); ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    <?php endif; ?>
</div>

<?php AdminLayout::renderFooter(); ?>
