<?php
/**
 * Admin Panel - Statistics & Visitor Tracking
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Admin\AdminLayout;
use NewSite\Auth\AdminAuth;
use NewSite\Database\DatabaseManager;
use NewSite\Geo\GeoService;
use NewSite\Security\IpBanService;
use NewSite\Settings\SettingsService;
use NewSite\Shop\CurrencyService;
use NewSite\Util\CountryData;
use NewSite\Visitor\VisitorService;

AdminAuth::requireLogin();

$db = DatabaseManager::getWriteConnection();

const STATS_BIND_LIMIT = ':limit';
const STATS_BIND_OFFSET = ':offset';
const STATS_DATE_TIME_FORMAT = 'Y-m-d H:i';

function formatTimeAgo($timestamp)
{
    if (empty($timestamp)) {
        return 'Never';
    }

    $diff = time() - (int)$timestamp;

    if ($diff < 60) {
        return $diff . 's ago';
    }

    $unitSeconds = 60;
    $unitSuffix = 'm ago';

    if ($diff >= 86400) {
        $unitSeconds = 86400;
        $unitSuffix = 'd ago';
    } elseif ($diff >= 3600) {
        $unitSeconds = 3600;
        $unitSuffix = 'h ago';
    }

    return floor($diff / $unitSeconds) . $unitSuffix;
}

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax'])) {
    header('Content-Type: application/json');

    $action = $_POST['action'] ?? '';

    try {
        switch ($action) {
            case 'ban_ip':
                $ip = $_POST['ip'] ?? '';
                $reason = $_POST['reason'] ?? 'Banned from statistics panel';

                if (!filter_var($ip, FILTER_VALIDATE_IP)) {
                    echo json_encode(['success' => false, 'error' => 'Invalid IP address']);
                    exit;
                }

                IpBanService::ban($ip, $reason);
                echo json_encode(['success' => true]);
                exit;

            case 'unban_ip':
                $ip = $_POST['ip'] ?? '';
                IpBanService::unban($ip);
                echo json_encode(['success' => true]);
                exit;

            case 'get_visitor_logs':
                $visitorId = (int)($_POST['visitor_id'] ?? 0);
                $logs = VisitorService::getLogs((int)$visitorId, (int)100);
                echo json_encode(['success' => true, 'logs' => $logs]);
                exit;

            case 'refresh_visitors':
                $minutes = (int)($_POST['minutes'] ?? 15);
                $visitors = VisitorService::getActive((int)$minutes);
                foreach ($visitors as &$visitor) {
                    $visitor['country_name'] = CountryData::getName($visitor['country_code'] ?? '');
                }
                unset($visitor);
                echo json_encode(['success' => true, 'visitors' => $visitors]);
                exit;

            case 'refresh_geo':
                $minutes = (int)($_POST['minutes'] ?? 15);
                $visitors = VisitorService::getActive((int)$minutes);
                $updated = 0;
                foreach ($visitors as $v) {
                    $newCode = GeoService::resolveCountry($v['ip_address'], true);
                    if (!empty($newCode) && $newCode !== $v['country_code']) {
                        $stmt = $db->prepare("UPDATE visitors SET country_code = ? WHERE id = ?");
                        $stmt->execute([$newCode, $v['id']]);
                        $updated++;
                    }
                }
                echo json_encode(['success' => true, 'updated' => $updated]);
                exit;

            case 'clear_logs':
                $db->exec("DELETE FROM visitor_logs");
                $db->exec("DELETE FROM visitors");
                $db->exec("DELETE FROM geo_cache");
                echo json_encode(['success' => true]);
                exit;

            default:
                echo json_encode(['success' => false, 'error' => 'Unknown action']);
                exit;
        }
    } catch (PDOException $e) {
        echo json_encode(['success' => false, 'error' => 'Database error']);
        exit;
    }
}

// Get current view (default: visitors)
$view = $_GET['view'] ?? 'visitors';
$allowedViews = ['visitors', 'all', 'countries', 'banned', 'users', 'downloads'];
if (!in_array($view, $allowedViews, true)) {
    $view = 'visitors';
}

$perPage = 50;
$page = max(1, (int)($_GET['page'] ?? 1));
$offset = ($page - 1) * $perPage;

// Get active visitors
$activeVisitors = VisitorService::getActive((int)15);

// All visitors (paginated) — exclude banned IPs
$allVisitors = [];
$totalAllVisitors = 0;
if ($view === 'all') {
    $totalAllVisitors = (int)$db->query(
        "SELECT COUNT(*) FROM visitors v
         WHERE NOT EXISTS (SELECT 1 FROM banned_ips b WHERE LOWER(b.ip_address) = LOWER(v.ip_address))"
    )->fetchColumn();
    $stmt = $db->prepare("
        SELECT v.*,
               (SELECT COUNT(*) FROM visitor_logs WHERE visitor_id = v.id) as page_count,
               (SELECT page_url FROM visitor_logs WHERE visitor_id = v.id AND page_url NOT LIKE '/assets/%' ORDER BY created_at DESC LIMIT 1) as last_page
        FROM visitors v
        WHERE NOT EXISTS (SELECT 1 FROM banned_ips b WHERE LOWER(b.ip_address) = LOWER(v.ip_address))
        ORDER BY v.last_activity DESC
        LIMIT :limit OFFSET :offset
    ");
    $stmt->bindValue(STATS_BIND_LIMIT, $perPage, PDO::PARAM_INT);
    $stmt->bindValue(STATS_BIND_OFFSET, $offset, PDO::PARAM_INT);
    $stmt->execute();
    $allVisitors = $stmt->fetchAll();
}

// No automatic geo refresh here to minimize background lookups

// Get country stats
$countryStats = VisitorService::getCountryStats();

// Get banned IPs
$bannedIPs = IpBanService::getAll();

// Get site users (only count confirmed users who have logged in at least once)
$siteUsers = [];
$siteUsersCount = (int)$db->query("SELECT COUNT(DISTINCT email) FROM site_users WHERE last_login_at IS NOT NULL")->fetchColumn();
$totalSiteUsers = $siteUsersCount;
if ($totalSiteUsers === 0) {
    $totalSiteUsers = (int)$db->query("SELECT COUNT(DISTINCT email) FROM users WHERE email IS NOT NULL AND email != ''")->fetchColumn();
}
if ($view === 'users') {
    if ($siteUsersCount === 0) {
        $seed = $db->query("SELECT email, created_at FROM users WHERE email IS NOT NULL AND email != ''")->fetchAll();
        foreach ($seed as $row) {
            $email = trim((string)($row['email'] ?? ''));
            if ($email === '') {
                continue;
            }
            $createdAt = $row['created_at'] ?? null;
            $createdTs = $createdAt ? strtotime($createdAt) : time();
            $stmt = $db->prepare("INSERT INTO site_users (email, created_at) VALUES (?, ?) ON CONFLICT (email) DO NOTHING");
            $stmt->execute([$email, $createdTs]);
        }
        $siteUsersCount = (int)$db->query("SELECT COUNT(DISTINCT email) FROM site_users WHERE last_login_at IS NOT NULL")->fetchColumn();
        $totalSiteUsers = $siteUsersCount;
    }

    $stmt = $db->prepare("SELECT email,
                                 MAX(is_active) AS is_active,
                                 MIN(created_at) AS created_at,
                                 MAX(last_login_at) AS last_login_at,
                                 MAX(created_ip) AS created_ip,
                                 MAX(last_login_ip) AS last_login_ip
                          FROM site_users
                          WHERE last_login_at IS NOT NULL
                          GROUP BY email
                          ORDER BY created_at DESC
                          LIMIT :limit OFFSET :offset");
    $stmt->bindValue(STATS_BIND_LIMIT, $perPage, PDO::PARAM_INT);
    $stmt->bindValue(STATS_BIND_OFFSET, $offset, PDO::PARAM_INT);
    $stmt->execute();
    $siteUsers = $stmt->fetchAll();
}

// Free digital download aggregate stats
$freeDownloadStats = [];
$totalFreeDownloadRows = 0;
$totalFreeDownloadCount = 0;
$easyMediaAiZipDownloadCount = max(0, (int)SettingsService::get('easy_media_ai_zip_download_count', '0'));
$easyMediaAiZipFirstDownloadAt = max(0, (int)SettingsService::get('easy_media_ai_zip_first_download_at', '0'));
$easyMediaAiZipLastDownloadAt = max(0, (int)SettingsService::get('easy_media_ai_zip_last_download_at', '0'));
$easyMediaAiZipVersionLabel = 'ZIP package';
$hasEasyMediaAiZipDownloads = $easyMediaAiZipDownloadCount > 0;
try {
    $totalFreeDownloadCount = (int)$db->query("SELECT COALESCE(SUM(download_count), 0) FROM free_digital_download_stats")->fetchColumn();
    $totalFreeDownloadCount += $easyMediaAiZipDownloadCount;

    if ($view === 'downloads') {
        $totalFreeDownloadRows = (int)$db->query("SELECT COUNT(*) FROM free_digital_download_stats")->fetchColumn();
        if ($hasEasyMediaAiZipDownloads) {
            $totalFreeDownloadRows += 1;
        }

        $downloadsLimit = $perPage;
        $downloadsOffset = $offset;

        if ($hasEasyMediaAiZipDownloads) {
            if ($page === 1) {
                $downloadsLimit = max(0, $perPage - 1);
                $downloadsOffset = 0;
                $freeDownloadStats[] = [
                    'product_id' => 0,
                    'variant_id' => 0,
                    'download_count' => $easyMediaAiZipDownloadCount,
                    'first_download_at' => $easyMediaAiZipFirstDownloadAt,
                    'last_download_at' => $easyMediaAiZipLastDownloadAt,
                    'product_name' => 'Easy Media AI ZIP',
                    'variant_label' => $easyMediaAiZipVersionLabel,
                    'is_easy_media_ai_zip' => 1,
                ];
            } else {
                $downloadsOffset = max(0, $offset - 1);
            }
        }

        $stmt = $db->prepare("SELECT fdds.product_id,
                                     fdds.variant_id,
                                     fdds.download_count,
                                     fdds.first_download_at,
                                     fdds.last_download_at,
                                     p.name AS product_name,
                                     pv.label AS variant_label
                              FROM free_digital_download_stats fdds
                              LEFT JOIN products p ON p.id = fdds.product_id
                              LEFT JOIN product_variants pv ON pv.id = fdds.variant_id
                              ORDER BY fdds.download_count DESC, fdds.last_download_at DESC
                              LIMIT :limit OFFSET :offset");
        $stmt->bindValue(STATS_BIND_LIMIT, $downloadsLimit, PDO::PARAM_INT);
        $stmt->bindValue(STATS_BIND_OFFSET, $downloadsOffset, PDO::PARAM_INT);
        $stmt->execute();
        $freeDownloadStats = array_merge($freeDownloadStats, $stmt->fetchAll());
    }
} catch (PDOException $e) {
    $freeDownloadStats = [];
    $totalFreeDownloadRows = $hasEasyMediaAiZipDownloads ? 1 : 0;
    $totalFreeDownloadCount = $easyMediaAiZipDownloadCount;
    if ($view === 'downloads' && $hasEasyMediaAiZipDownloads) {
        $freeDownloadStats[] = [
            'product_id' => 0,
            'variant_id' => 0,
            'download_count' => $easyMediaAiZipDownloadCount,
            'first_download_at' => $easyMediaAiZipFirstDownloadAt,
            'last_download_at' => $easyMediaAiZipLastDownloadAt,
            'product_name' => 'Easy Media AI ZIP',
            'variant_label' => $easyMediaAiZipVersionLabel,
            'is_easy_media_ai_zip' => 1,
        ];
    }
}

// Get total stats (exclude admin IPs from totals so they are not inflated)
$totalVisitors = $db->query("SELECT COUNT(*) FROM visitors WHERE is_admin_ip = false")->fetchColumn();
$totalPageViews = $db->query("SELECT COUNT(*) FROM visitor_logs vl JOIN visitors v ON v.id = vl.visitor_id WHERE v.is_admin_ip = false")->fetchColumn();
$todayVisitorsSql = "SELECT COUNT(*) FROM visitors WHERE first_visit::date = CURRENT_DATE AND is_admin_ip = false";
$todayVisitors = $db->query($todayVisitorsSql)->fetchColumn();
$totalRevenue = (float)$db->query("SELECT COALESCE(SUM(total_amount), 0) FROM orders WHERE status = 'paid'")->fetchColumn();
// Convert revenue from store currency to current display currency
$storeCurrency = SettingsService::get('store_currency', 'USD');
$currentCurrency = CurrencyService::getCurrent();
$totalRevenueConverted = CurrencyService::convert($totalRevenue, $storeCurrency, $currentCurrency);
$currencySymbol = CurrencyService::getSymbol();

// IP search
$searchIp = trim($_GET['ip'] ?? '');
$searchError = '';
$searchResult = null;
$searchLogs = [];
if ($searchIp !== '') {
    if (!filter_var($searchIp, FILTER_VALIDATE_IP)) {
        $searchError = 'Invalid IP address.';
    } else {
        $stmt = $db->prepare("
            SELECT v.*,
                   (SELECT COUNT(*) FROM visitor_logs WHERE visitor_id = v.id) as page_count,
                   (SELECT page_url FROM visitor_logs WHERE visitor_id = v.id AND page_url NOT LIKE '/assets/%' ORDER BY created_at DESC LIMIT 1) as last_page
            FROM visitors v
            WHERE v.ip_address = ?
            ORDER BY v.last_activity DESC
            LIMIT 1
        ");
        $stmt->execute([$searchIp]);
        $searchResult = $stmt->fetch();
        if ($searchResult) {
            $searchLogs = VisitorService::getLogs((int)$searchResult['id'], (int)50);
        } else {
            $searchError = 'No visitor found for that IP.';
        }
    }
}

$hasMoreAllVisitors = false;
$showMoreUrl = '';
if ($view === 'all' && $totalAllVisitors > ($page * $perPage)) {
    $hasMoreAllVisitors = true;
    $params = ['view' => 'all', 'page' => $page + 1];
    if ($searchIp !== '') {
        $params['ip'] = $searchIp;
    }
    $showMoreUrl = '?' . http_build_query($params);
}

$hasMoreFreeDownloads = false;
$showMoreDownloadsUrl = '';
if ($view === 'downloads' && $totalFreeDownloadRows > ($page * $perPage)) {
    $hasMoreFreeDownloads = true;
    $showMoreDownloadsUrl = '?' . http_build_query([
        'view' => 'downloads',
        'page' => $page + 1,
    ]);
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header">
        <h1>📊 Statistics</h1>
        <p>Monitor website traffic and visitor activity</p>
    </div>

<!-- Stats Overview Cards -->
<div class="stats-grid">
    <div class="stat-card">
        <div class="stat-icon">👥</div>
        <div class="stat-info">
            <span class="stat-value" data-stat-total-visitors><?php echo number_format($totalVisitors); ?></span>
            <span class="stat-label">Total Visitors</span>
        </div>
    </div>
    <div class="stat-card">
        <div class="stat-icon">📄</div>
        <div class="stat-info">
            <span class="stat-value" data-stat-total-views><?php echo number_format($totalPageViews); ?></span>
            <span class="stat-label">Total Page Views</span>
        </div>
    </div>
    <div class="stat-card">
        <div class="stat-icon">🆕</div>
        <div class="stat-info">
            <span class="stat-value" data-stat-today-visitors><?php echo number_format($todayVisitors); ?></span>
            <span class="stat-label">Today's Visitors</span>
        </div>
    </div>
    <div class="stat-card">
        <div class="stat-icon">🟢</div>
        <div class="stat-info">
            <span class="stat-value" data-stat-active-visitors><?php echo count($activeVisitors); ?></span>
            <span class="stat-label">Active Now</span>
        </div>
    </div>
    <div class="stat-card">
        <div class="stat-icon">💰</div>
        <div class="stat-info">
            <span class="stat-value"><?php echo e($currencySymbol . number_format($totalRevenueConverted, 2)); ?></span>
            <span class="stat-label">Total Brutto Earned</span>
        </div>
    </div>
    <div class="stat-card">
        <div class="stat-icon">⬇️</div>
        <div class="stat-info">
            <span class="stat-value"><?php echo number_format($totalFreeDownloadCount); ?></span>
            <span class="stat-label">Free Downloads</span>
        </div>
    </div>
</div>

<!-- Search IP -->
<div class="stats-search">
    <form method="GET" action="" class="stats-search-form">
        <input type="hidden" name="view" value="<?php echo e($view); ?>">
        <?php if ($view === 'all' && $page > 1): ?>
            <input type="hidden" name="page" value="<?php echo (int)$page; ?>">
        <?php endif; ?>
        <input type="text" name="ip" class="form-control stats-search-input" placeholder="Search IP address" value="<?php echo e($searchIp); ?>">
        <div class="stats-search-actions">
            <button type="submit" class="btn btn-secondary btn-sm has-tooltip" data-tooltip="Search for a specific IP address.">Search</button>
            <?php if ($searchIp !== ''): ?>
                <a href="?view=<?php echo e($view); ?>" class="btn btn-secondary btn-sm has-tooltip" data-tooltip="Clear the IP search filter.">Clear</a>
            <?php endif; ?>
        </div>
    </form>
</div>

<?php if ($searchIp !== ''): ?>
<div class="card stats-ip-card">
    <div class="card-header stats-card-header">
        <h3>🔎 IP Lookup</h3>
    </div>
    <div class="card-body">
        <?php if ($searchError): ?>
            <p class="text-muted text-center stats-empty"><?php echo e($searchError); ?></p>
        <?php elseif ($searchResult): ?>
            <div class="stats-ip-grid">
                <div class="stats-ip-item">
                    <div class="stats-ip-label">IP Address</div>
                    <div class="stats-ip-value"><code><?php echo e($searchResult['ip_address']); ?></code>
                        <?php if (!empty($searchResult['is_admin_ip'])): ?>
                            <span class="admin-ip-badge" title="Admin IP">👑</span>
                        <?php endif; ?>
                    </div>
                </div>
                <div class="stats-ip-item">
                    <div class="stats-ip-label">Country</div>
                    <div class="stats-ip-value"><?php echo e(CountryData::getName($searchResult['country_code'])); ?></div>
                </div>
                <div class="stats-ip-item">
                    <div class="stats-ip-label">First Visit</div>
                    <div class="stats-ip-value"><?php echo e($searchResult['first_visit']); ?></div>
                </div>
                <div class="stats-ip-item">
                    <div class="stats-ip-label">Last Activity</div>
                    <div class="stats-ip-value"><?php echo e($searchResult['last_activity']); ?></div>
                </div>
                <div class="stats-ip-item">
                    <div class="stats-ip-label">Visits</div>
                    <div class="stats-ip-value"><?php echo e($searchResult['visit_count']); ?></div>
                </div>
                <div class="stats-ip-item">
                    <div class="stats-ip-label">Pages Logged</div>
                    <div class="stats-ip-value"><?php echo e($searchResult['page_count']); ?></div>
                </div>
                <div class="stats-ip-item stats-ip-wide">
                    <div class="stats-ip-label">Last Page</div>
                    <div class="stats-ip-value text-truncate"><?php echo e($searchResult['last_page'] ?? ''); ?></div>
                </div>
                <div class="stats-ip-item stats-ip-wide">
                    <div class="stats-ip-label">User Agent</div>
                    <div class="stats-ip-value text-truncate"><?php echo e($searchResult['user_agent']); ?></div>
                </div>
                <div class="stats-ip-item stats-ip-wide">
                    <div class="stats-ip-label">Referrer</div>
                    <div class="stats-ip-value text-truncate"><?php echo e($searchResult['referrer']); ?></div>
                </div>
                <div class="stats-ip-item">
                    <div class="stats-ip-label">Status</div>
                    <div class="stats-ip-value"><?php echo IpBanService::isBanned($searchResult['ip_address']) ? 'Banned' : 'Active'; ?></div>
                </div>
            </div>

            <div class="stats-card-actions stats-ip-actions">
                <button class="btn btn-sm btn-secondary js-visitor-logs has-tooltip" data-tooltip="View visit logs for this IP." data-visitor-id="<?php echo $searchResult['id']; ?>" data-ip="<?php echo e($searchResult['ip_address']); ?>">📋 Logs</button>
                <?php if (!IpBanService::isBanned($searchResult['ip_address'])): ?>
                    <button class="btn btn-sm btn-danger js-ban-ip has-tooltip" data-tooltip="Ban this IP from the site." data-ip="<?php echo e($searchResult['ip_address']); ?>">🚫 Ban</button>
                <?php endif; ?>
            </div>

            <div class="stats-ip-logs">
                <h4>Recent Logs</h4>
                <?php if (!empty($searchLogs)): ?>
                    <div class="log-list">
                        <?php foreach ($searchLogs as $log): ?>
                            <div class="log-item">
                                <div class="log-url"><?php echo e($log['page_url']); ?></div>
                                <div class="log-time"><?php echo e($log['created_at']); ?></div>
                                <?php if (!empty($log['referrer'])): ?>
                                    <div class="log-referrer">From: <?php echo e($log['referrer']); ?></div>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <p class="text-muted text-center stats-empty">No logs for this IP yet.</p>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>
</div>
<?php endif; ?>

<!-- View Tabs -->
<div class="view-tabs stats-tabs">
    <a href="?view=visitors" class="btn <?php echo $view === 'visitors' ? 'btn-primary' : 'btn-secondary'; ?>">
        🌐 Live Visitors
    </a>
    <a href="?view=all" class="btn <?php echo $view === 'all' ? 'btn-primary' : 'btn-secondary'; ?>">
        🧾 All Visitors
    </a>
    <a href="?view=countries" class="btn <?php echo $view === 'countries' ? 'btn-primary' : 'btn-secondary'; ?>">
        🗺️ Country Stats
    </a>
    <a href="?view=banned" class="btn <?php echo $view === 'banned' ? 'btn-primary' : 'btn-secondary'; ?>">
        🚫 Banned IPs (<?php echo count($bannedIPs); ?>)
    </a>
    <a href="?view=users" class="btn <?php echo $view === 'users' ? 'btn-primary' : 'btn-secondary'; ?>">
        👤 User Accounts (<?php echo (int)$totalSiteUsers; ?>)
    </a>
    <a href="?view=downloads" class="btn <?php echo $view === 'downloads' ? 'btn-primary' : 'btn-secondary'; ?>">
        ⬇️ Free Downloads (<?php echo number_format($totalFreeDownloadCount); ?>)
    </a>
</div>

<?php if ($view === 'visitors'): ?>
<!-- Live Visitors -->
<div class="card">
    <div class="card-header stats-card-header">
        <h3>🟢 Active Visitors (Last 15 minutes)</h3>
        <div class="stats-card-actions">
            <button class="btn btn-secondary btn-sm has-tooltip" data-tooltip="Refresh the active visitors list." data-action="refresh-visitors">
                🔄 Refresh
            </button>
            <button class="btn btn-secondary btn-sm has-tooltip" data-tooltip="Recheck country info for active visitors." data-action="refresh-geo">
                🌍 Refresh Geo
            </button>
            <button class="btn btn-danger btn-sm has-tooltip" data-tooltip="Delete all visitor logs and reset visitor records." data-action="clear-logs">
                🧹 Clear Logs
            </button>
        </div>
    </div>
    <div class="card-body" id="visitors-card-body">
        <p class="text-muted text-center stats-empty<?php echo empty($activeVisitors) ? '' : ' is-hidden'; ?>" data-visitors-empty>No active visitors at the moment.</p>
        <div class="table-responsive<?php echo empty($activeVisitors) ? ' is-hidden' : ''; ?>" data-visitors-table-wrap>
            <table class="admin-table" id="visitors-table">
                <thead>
                    <tr>
                        <th>Country</th>
                        <th>IP Address</th>
                        <th>Last Page</th>
                        <th>Pages Visited</th>
                        <th>Last Activity</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($activeVisitors as $visitor): ?>
                    <tr data-ip="<?php echo e($visitor['ip_address']); ?>">
                        <td>
                            <?php if (!in_array($visitor['country_code'], ['XX', 'LO'])): ?>
                               <img data-src="https://flagcdn.com/24x18/<?php echo strtolower($visitor['country_code']); ?>.png"
                                   src="data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 24 18'%3E%3C/svg%3E"
                                   alt="<?php echo e($visitor['country_code']); ?>"
                                   class="stats-flag lazy-image"
                                   data-hide-on-error="1">
                            <?php else: ?>
                               <span class="stats-flag-unknown">?</span>
                            <?php endif; ?>
                            <?php echo e(CountryData::getName($visitor['country_code'])); ?>
                        </td>
                        <td>
                            <code><?php echo e($visitor['ip_address']); ?></code>
                            <?php if (!empty($visitor['is_admin_ip'])): ?>
                                <span class="admin-ip-badge" title="Admin IP">👑</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <span class="text-truncate stats-last-page" title="<?php echo e($visitor['last_page']); ?>">
                                <?php echo e($visitor['last_page']); ?>
                            </span>
                        </td>
                        <td><?php echo $visitor['page_count']; ?></td>
                        <td>
                            <span class="time-ago" data-time="<?php echo $visitor['last_activity']; ?>">
                                <?php echo $visitor['last_activity']; ?>
                            </span>
                        </td>
                        <td>
                            <button class="btn btn-sm btn-secondary js-visitor-logs has-tooltip" data-tooltip="View visit logs for this IP." data-visitor-id="<?php echo $visitor['id']; ?>" data-ip="<?php echo e($visitor['ip_address']); ?>">
                                📋 Logs
                            </button>
                            <button class="btn btn-sm btn-danger js-ban-ip has-tooltip" data-tooltip="Ban this IP from the site." data-ip="<?php echo e($visitor['ip_address']); ?>">
                                🚫 Ban
                            </button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<?php elseif ($view === 'all'): ?>
<!-- All Visitors -->
<div class="card">
    <div class="card-header stats-card-header">
        <h3>🧾 All Visitors (All Time)</h3>
        <div class="stats-card-actions">
            <span class="text-muted">Showing <?php echo min($page * $perPage, $totalAllVisitors); ?> of <?php echo number_format($totalAllVisitors); ?></span>
        </div>
    </div>
    <div class="card-body">
        <?php if (empty($allVisitors)): ?>
        <p class="text-muted text-center stats-empty">No visitors logged yet.</p>
        <?php else: ?>
        <div class="table-responsive">
            <table class="admin-table" id="all-visitors-table">
                <thead>
                    <tr>
                        <th>Country</th>
                        <th>IP Address</th>
                        <th>Last Page</th>
                        <th>Pages Visited</th>
                        <th>Last Activity</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($allVisitors as $visitor): ?>
                    <tr data-ip="<?php echo e($visitor['ip_address']); ?>">
                        <td>
                            <?php if (!in_array($visitor['country_code'], ['XX', 'LO'])): ?>
                               <img data-src="https://flagcdn.com/24x18/<?php echo strtolower($visitor['country_code']); ?>.png"
                                 src="data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 24 18'%3E%3C/svg%3E"
                                   alt="<?php echo e($visitor['country_code']); ?>"
                                 class="stats-flag lazy-image"
                                 data-hide-on-error="1">
                            <?php else: ?>
                            <span class="stats-flag-unknown">?</span>
                            <?php endif; ?>
                            <?php echo e(CountryData::getName($visitor['country_code'])); ?>
                        </td>
                        <td>
                            <code><?php echo e($visitor['ip_address']); ?></code>
                            <?php if (!empty($visitor['is_admin_ip'])): ?>
                                <span class="admin-ip-badge" title="Admin IP">👑</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <span class="text-truncate stats-last-page" title="<?php echo e($visitor['last_page']); ?>">
                                <?php echo e($visitor['last_page']); ?>
                            </span>
                        </td>
                        <td><?php echo $visitor['page_count']; ?></td>
                        <td>
                            <span class="time-ago" data-time="<?php echo $visitor['last_activity']; ?>">
                                <?php echo $visitor['last_activity']; ?>
                            </span>
                        </td>
                        <td>
                            <button class="btn btn-sm btn-secondary js-visitor-logs has-tooltip" data-tooltip="View visit logs for this IP." data-visitor-id="<?php echo $visitor['id']; ?>" data-ip="<?php echo e($visitor['ip_address']); ?>">
                                📋 Logs
                            </button>
                            <button class="btn btn-sm btn-danger js-ban-ip has-tooltip" data-tooltip="Ban this IP from the site." data-ip="<?php echo e($visitor['ip_address']); ?>">
                                🚫 Ban
                            </button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
    </div>
</div>

<?php if ($hasMoreAllVisitors): ?>
<div class="stats-pagination">
    <a class="btn btn-secondary has-tooltip" data-tooltip="Load more visitors." href="<?php echo e($showMoreUrl); ?>">Show More</a>
</div>
<?php endif; ?>

<?php elseif ($view === 'users'): ?>
<div class="card">
    <div class="card-header stats-card-header">
        <h3>👤 User Accounts</h3>
    </div>
    <div class="card-body">
        <?php if (empty($siteUsers)): ?>
            <p class="text-muted text-center stats-empty">No user accounts yet.</p>
        <?php else: ?>
            <div class="table-responsive">
                <table class="admin-table">
                    <thead>
                        <tr>
                            <th>Email</th>
                            <th>Status</th>
                            <th>Member Since</th>
                            <th>Last Login</th>
                            <th>Created IP</th>
                            <th>Last Login IP</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($siteUsers as $user): ?>
                            <tr>
                                <td><?php echo e($user['email'] ?? ''); ?></td>
                                <td><?php echo !empty($user['is_active']) ? 'Active' : 'Disabled'; ?></td>
                                <td><?php echo !empty($user['created_at']) ? date(STATS_DATE_TIME_FORMAT, (int)$user['created_at']) : '-'; ?></td>
                                <td title="<?php echo !empty($user['last_login_at']) ? date(STATS_DATE_TIME_FORMAT, (int)$user['last_login_at']) : 'Never'; ?>">
                                    <?php echo e(formatTimeAgo($user['last_login_at'] ?? null)); ?>
                                </td>
                                <td><?php echo e($user['created_ip'] ?? '-'); ?></td>
                                <td><?php echo e($user['last_login_ip'] ?? '-'); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php if ($totalSiteUsers > $perPage): ?>
            <div class="stats-pagination">
                <?php if ($page > 1): ?>
                    <a href="?view=users&page=<?php echo $page - 1; ?>" class="btn btn-secondary btn-sm has-tooltip" data-tooltip="Previous page of users.">← Prev</a>
                <?php endif; ?>
                <?php if ($page * $perPage < $totalSiteUsers): ?>
                    <a href="?view=users&page=<?php echo $page + 1; ?>" class="btn btn-secondary btn-sm has-tooltip" data-tooltip="Next page of users.">Next →</a>
                <?php endif; ?>
            </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<?php elseif ($view === 'downloads'): ?>
<div class="card">
    <div class="card-header stats-card-header">
        <h3>⬇️ Free Digital Downloads</h3>
        <div class="stats-card-actions">
            <span class="text-muted">Showing <?php echo min($page * $perPage, $totalFreeDownloadRows); ?> of <?php echo number_format($totalFreeDownloadRows); ?> items</span>
        </div>
    </div>
    <div class="card-body">
        <?php if (empty($freeDownloadStats)): ?>
            <p class="text-muted text-center stats-empty">No free digital downloads tracked yet.</p>
        <?php else: ?>
            <div class="table-responsive">
                <table class="admin-table">
                    <thead>
                        <tr>
                            <th>Product</th>
                            <th>Version</th>
                            <th>Downloads</th>
                            <th>First Download</th>
                            <th>Last Download</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($freeDownloadStats as $row): ?>
                            <?php
                                $productId = (int)($row['product_id'] ?? 0);
                            $variantId = (int)($row['variant_id'] ?? 0);
                            $isEasyMediaAiZipRow = !empty($row['is_easy_media_ai_zip']);
                            $productName = trim((string)($row['product_name'] ?? ''));
                            if ($productName === '') {
                                $productName = 'Product #' . $productId;
                            }
                            $variantLabel = trim((string)($row['variant_label'] ?? ''));
                            $versionText = 'Base product';
                            if ($isEasyMediaAiZipRow) {
                                $versionText = $variantLabel !== '' ? $variantLabel : $easyMediaAiZipVersionLabel;
                            }
                            if (!$isEasyMediaAiZipRow && $variantId > 0) {
                                $versionText = $variantLabel !== '' ? $variantLabel : ('Variant #' . $variantId);
                            }
                            $firstAt = (int)($row['first_download_at'] ?? 0);
                            $lastAt = (int)($row['last_download_at'] ?? 0);
                            ?>
                            <tr>
                                <td><?php echo e($productName); ?></td>
                                <td><?php echo e($versionText); ?></td>
                                <td><?php echo number_format((int)($row['download_count'] ?? 0)); ?></td>
                                <td><?php echo $firstAt > 0 ? e(date(STATS_DATE_TIME_FORMAT, $firstAt)) : '-'; ?></td>
                                <td><?php echo $lastAt > 0 ? e(date(STATS_DATE_TIME_FORMAT, $lastAt)) : '-'; ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

<?php if ($hasMoreFreeDownloads): ?>
<div class="stats-pagination">
    <a class="btn btn-secondary has-tooltip" data-tooltip="Load more free download records." href="<?php echo e($showMoreDownloadsUrl); ?>">Show More</a>
</div>
<?php endif; ?>

<?php elseif ($view === 'countries'): ?>
<!-- Country Statistics -->
<div class="card">
    <div class="card-header">
        <h3>🗺️ Visitors by Country</h3>
    </div>
    <div class="card-body">
        <?php if (empty($countryStats)): ?>
        <p class="text-muted text-center stats-empty">No country data available yet.</p>
        <?php else: ?>
        <div class="country-stats-grid stats-country-grid">
            <?php foreach ($countryStats as $stat): ?>
            <div class="country-stat-item stats-country-item">
                <?php if (!in_array($stat['country_code'], ['XX', 'LO'])): ?>
                 <img data-src="https://flagcdn.com/48x36/<?php echo strtolower($stat['country_code']); ?>.png"
                     src="data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 48 36'%3E%3C/svg%3E"
                     alt="<?php echo e($stat['country_code']); ?>"
                     class="stats-country-flag lazy-image"
                     data-hide-on-error="1">
                <?php else: ?>
                <span class="stats-country-placeholder">?</span>
                <?php endif; ?>
                <div class="stats-country-meta">
                    <div class="stats-country-name"><?php echo e(CountryData::getName($stat['country_code'])); ?></div>
                    <div class="text-muted stats-country-count">
                        <?php echo $stat['visitor_count']; ?> visitors
                    </div>
                </div>
                <div class="stats-country-percent">
                    <div class="percentage-badge stats-percentage-badge">
                        <?php echo $stat['percentage']; ?>%
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
    </div>
</div>

<?php elseif ($view === 'banned'): ?>
<!-- Banned IPs -->
<div class="card">
    <div class="card-header stats-card-header">
        <h3>🚫 Banned IP Addresses</h3>
        <button class="btn btn-primary btn-sm has-tooltip" data-tooltip="Ban a new IP address." data-action="show-ban">
            ➕ Ban IP Manually
        </button>
    </div>
    <div class="card-body">
        <?php if (empty($bannedIPs)): ?>
        <p class="text-muted text-center stats-empty">No banned IPs.</p>
        <?php else: ?>
        <div class="table-responsive">
            <table class="admin-table">
                <thead>
                    <tr>
                        <th>IP Address</th>
                        <th>Reason</th>
                        <th>Banned At</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($bannedIPs as $banned): ?>
                    <tr>
                        <td><code><?php echo e($banned['ip_address']); ?></code></td>
                        <td><?php echo e($banned['reason'] ?: 'No reason specified'); ?></td>
                        <td><?php echo $banned['banned_at']; ?></td>
                        <td>
                            <button class="btn btn-sm btn-success js-unban-ip has-tooltip" data-tooltip="Remove this IP from the ban list." data-ip="<?php echo e($banned['ip_address']); ?>">
                                ✅ Unban
                            </button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>
    </div>
</div>
<?php endif; ?>

</div><!-- /.admin-content -->

<!-- Visitor Logs Modal -->
<div class="modal-overlay" id="logs-modal">
    <div class="modal stats-modal stats-modal-lg">
        <div class="modal-header">
            <h3>📋 Visitor Logs - <span id="logs-ip"></span></h3>
            <button class="modal-close" data-action="close-logs">&times;</button>
        </div>
        <div class="modal-body stats-modal-body" id="logs-content">
            <p class="text-center">Loading...</p>
        </div>
    </div>
</div>

<!-- Ban IP Modal -->
<div class="modal-overlay" id="ban-modal">
    <div class="modal stats-modal stats-modal-sm">
        <div class="modal-header">
            <h3>🚫 Ban IP Address</h3>
            <button class="modal-close" data-action="close-ban">&times;</button>
        </div>
        <div class="modal-body">
            <form id="ban-form">
                <div class="form-group">
                    <label for="ban-ip">IP Address</label>
                    <input type="text" id="ban-ip" class="form-control" placeholder="e.g., 192.168.1.1" required>
                </div>
                <div class="form-group">
                    <label for="ban-reason">Reason (optional)</label>
                    <input type="text" id="ban-reason" class="form-control" placeholder="Reason for ban">
                </div>
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" data-action="close-ban">Cancel</button>
                    <button type="submit" class="btn btn-danger">🚫 Ban IP</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script nonce="<?php echo e(getCspNonce()); ?>">
const statsNumberFormatter = new Intl.NumberFormat();

function getStatsCsrfToken() {
    const meta = document.querySelector('meta[name="csrf-token"]');
    if (!meta) {
        return '';
    }

    return String(meta.getAttribute('content') || '').trim();
}

function appendStatsCsrfToken(formData) {
    if (!(formData instanceof FormData)) {
        return;
    }

    const csrfToken = getStatsCsrfToken();
    if (csrfToken !== '') {
        formData.append('csrf_token', csrfToken);
    }
}

document.addEventListener('DOMContentLoaded', function() {
    bindVisitorRowActions(document);

    const banForm = document.getElementById('ban-form');
    if (banForm) {
        banForm.addEventListener('submit', submitBan);
    }

    document.querySelectorAll('[data-action="show-ban"]').forEach(function(btn) {
        btn.addEventListener('click', function() {
            showBanModal();
        });
    });

    document.querySelectorAll('[data-action="refresh-visitors"]').forEach(function(btn) {
        btn.addEventListener('click', function() {
            refreshVisitors();
        });
    });

    document.querySelectorAll('[data-action="refresh-geo"]').forEach(function(btn) {
        btn.addEventListener('click', function() {
            refreshGeo();
        });
    });

    document.querySelectorAll('[data-action="clear-logs"]').forEach(function(btn) {
        btn.addEventListener('click', function() {
            clearAllLogs();
        });
    });

    document.querySelectorAll('[data-action="close-logs"]').forEach(function(btn) {
        btn.addEventListener('click', function() {
            closeLogsModal();
        });
    });

    document.querySelectorAll('[data-action="close-ban"]').forEach(function(btn) {
        btn.addEventListener('click', function() {
            closeBanModal();
        });
    });
});

function formatNumber(value) {
    const num = parseInt(value, 10);
    if (Number.isNaN(num)) {
        return String(value ?? '0');
    }
    return statsNumberFormatter.format(num);
}

function setStatValue(attr, value) {
    const el = document.querySelector(`[${attr}]`);
    if (el) {
        el.textContent = value;
    }
}

function bindVisitorRowActions(root) {
    if (!root) return;
    root.querySelectorAll('.js-visitor-logs').forEach(function(btn) {
        if (btn.dataset.bound === '1') return;
        btn.dataset.bound = '1';
        btn.addEventListener('click', function() {
            showVisitorLogs(this.getAttribute('data-visitor-id'), this.getAttribute('data-ip'));
        });
    });

    root.querySelectorAll('.js-ban-ip').forEach(function(btn) {
        if (btn.dataset.bound === '1') return;
        btn.dataset.bound = '1';
        btn.addEventListener('click', function() {
            banVisitorIP(this.getAttribute('data-ip'));
        });
    });

    root.querySelectorAll('.js-unban-ip').forEach(function(btn) {
        if (btn.dataset.bound === '1') return;
        btn.dataset.bound = '1';
        btn.addEventListener('click', function() {
            unbanIP(this.getAttribute('data-ip'));
        });
    });
}

function createFlagElement(countryCode) {
    const code = String(countryCode || 'XX').toUpperCase();
    if (code === 'XX' || code === 'LO') {
        const span = document.createElement('span');
        span.className = 'stats-flag-unknown';
        span.textContent = '?';
        return span;
    }

    const img = document.createElement('img');
    img.src = `https://flagcdn.com/24x18/${code.toLowerCase()}.png`;
    img.alt = code;
    img.className = 'stats-flag';
    img.setAttribute('data-hide-on-error', '1');
    return img;
}

function ensureVisitorsTable() {
    const cardBody = document.getElementById('visitors-card-body');
    if (!cardBody) return null;

    let tableWrap = cardBody.querySelector('[data-visitors-table-wrap]');
    if (!tableWrap) {
        tableWrap = document.createElement('div');
        tableWrap.className = 'table-responsive';
        tableWrap.setAttribute('data-visitors-table-wrap', '');
        cardBody.appendChild(tableWrap);
    }

    let table = cardBody.querySelector('#visitors-table');
    if (!table) {
        table = document.createElement('table');
        table.className = 'admin-table';
        table.id = 'visitors-table';
        table.innerHTML = `
            <thead>
                <tr>
                    <th>Country</th>
                    <th>IP Address</th>
                    <th>Last Page</th>
                    <th>Pages Visited</th>
                    <th>Last Activity</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody></tbody>
        `;
        tableWrap.appendChild(table);
    }

    return { cardBody, tableWrap, table };
}

function updateVisitorsTable(visitors) {
    const payload = Array.isArray(visitors) ? visitors : [];
    const tableParts = ensureVisitorsTable();
    if (!tableParts) return;

    const { cardBody, tableWrap, table } = tableParts;
    const emptyEl = cardBody.querySelector('[data-visitors-empty]');
    const tbody = table.querySelector('tbody') || table.appendChild(document.createElement('tbody'));
    tbody.innerHTML = '';

    if (payload.length) {
        payload.forEach(function(visitor) {
            const tr = document.createElement('tr');
            const ip = visitor.ip_address || '';
            tr.setAttribute('data-ip', ip);

            const countryTd = document.createElement('td');
            const flagEl = createFlagElement(visitor.country_code);
            countryTd.appendChild(flagEl);
            const countryName = visitor.country_name || visitor.country_code || 'Unknown';
            const countryText = document.createTextNode(' ' + countryName);
            countryTd.appendChild(countryText);

            const ipTd = document.createElement('td');
            const ipCode = document.createElement('code');
            ipCode.textContent = ip;
            ipTd.appendChild(ipCode);
            if (visitor.is_admin_ip) {
                const badge = document.createElement('span');
                badge.className = 'admin-ip-badge';
                badge.title = 'Admin IP';
                badge.textContent = '👑';
                ipTd.appendChild(badge);
            }

            const lastPageTd = document.createElement('td');
            const lastPage = visitor.last_page || '';
            const lastPageSpan = document.createElement('span');
            lastPageSpan.className = 'text-truncate stats-last-page';
            lastPageSpan.title = lastPage;
            lastPageSpan.textContent = lastPage;
            lastPageTd.appendChild(lastPageSpan);

            const pageCountTd = document.createElement('td');
            pageCountTd.textContent = String(visitor.page_count ?? 0);

            const lastActivityTd = document.createElement('td');
            const lastActivitySpan = document.createElement('span');
            lastActivitySpan.className = 'time-ago';
            const lastActivity = visitor.last_activity || '';
            lastActivitySpan.setAttribute('data-time', lastActivity);
            lastActivitySpan.textContent = lastActivity;
            lastActivityTd.appendChild(lastActivitySpan);

            const actionsTd = document.createElement('td');
            const logsBtn = document.createElement('button');
            logsBtn.className = 'btn btn-sm btn-secondary js-visitor-logs has-tooltip';
            logsBtn.setAttribute('data-tooltip', 'View visit logs for this IP.');
            logsBtn.setAttribute('data-visitor-id', visitor.id || '');
            logsBtn.setAttribute('data-ip', ip);
            logsBtn.textContent = '📋 Logs';

            const banBtn = document.createElement('button');
            banBtn.className = 'btn btn-sm btn-danger js-ban-ip has-tooltip';
            banBtn.setAttribute('data-tooltip', 'Ban this IP from the site.');
            banBtn.setAttribute('data-ip', ip);
            banBtn.textContent = '🚫 Ban';

            actionsTd.appendChild(logsBtn);
            actionsTd.appendChild(banBtn);

            tr.appendChild(countryTd);
            tr.appendChild(ipTd);
            tr.appendChild(lastPageTd);
            tr.appendChild(pageCountTd);
            tr.appendChild(lastActivityTd);
            tr.appendChild(actionsTd);

            tbody.appendChild(tr);
        });
    }

    const showTable = payload.length > 0;
    if (emptyEl) {
        emptyEl.classList.toggle('is-hidden', showTable);
    }
    if (tableWrap) {
        tableWrap.classList.toggle('is-hidden', !showTable);
    }

    setStatValue('data-stat-active-visitors', formatNumber(payload.length));
    bindVisitorRowActions(cardBody);
}

function fetchActiveVisitors(minutes) {
    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'refresh_visitors');
    formData.append('minutes', String(minutes || 15));
    appendStatsCsrfToken(formData);

    return fetch(window.location.pathname, {
        method: 'POST',
        body: formData
    }).then(r => {
        if (!r.ok) throw new Error('Network response was not ok');
        return r.json();
    });
}

function refreshVisitors() {
    fetchActiveVisitors(15)
        .then(data => {
            if (data.success) {
                updateVisitorsTable(data.visitors || []);
                showNotification('Active visitors refreshed.', 'success');
            } else {
                showNotification(data.error || 'Failed to refresh visitors', 'error');
            }
        })
        .catch(err => {
            console.error('Error refreshing visitors:', err);
            showNotification('Failed to refresh visitors', 'error');
        });
}

function clearAllLogs() {
    if (!confirm('Clear all visitor logs? This cannot be undone.')) {
        return;
    }

    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'clear_logs');
    appendStatsCsrfToken(formData);

    fetch(window.location.pathname, {
        method: 'POST',
        body: formData
    })
    .then(r => {
        if (!r.ok) throw new Error('Network response was not ok');
        return r.json();
    })
    .then(data => {
        if (data.success) {
            showNotification('Visitor logs cleared.', 'success');
            updateVisitorsTable([]);
            setStatValue('data-stat-total-visitors', formatNumber(0));
            setStatValue('data-stat-total-views', formatNumber(0));
            setStatValue('data-stat-today-visitors', formatNumber(0));
        } else {
            showNotification(data.error || 'Failed to clear logs', 'error');
        }
    })
    .catch(err => {
        console.error('Error clearing logs:', err);
        showNotification('Failed to clear logs', 'error');
    });
}

function refreshGeo() {
    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'refresh_geo');
    formData.append('minutes', '15');
    appendStatsCsrfToken(formData);

    fetch(window.location.pathname, {
        method: 'POST',
        body: formData
    })
    .then(r => {
        if (!r.ok) throw new Error('Network response was not ok');
        return r.json();
    })
    .then(data => {
        if (data.success) {
            showNotification(`Geo refreshed (${data.updated} updated).`, 'success');
            return fetchActiveVisitors(15)
                .then(payload => {
                    if (payload.success) {
                        updateVisitorsTable(payload.visitors || []);
                    }
                });
        } else {
            showNotification(data.error || 'Failed to refresh geo', 'error');
        }
    })
    .catch(err => {
        console.error('Error refreshing geo:', err);
        showNotification('Failed to refresh geo', 'error');
    });
}

function showVisitorLogs(visitorId, ip) {
    document.getElementById('logs-modal').classList.add('active');
    document.body.classList.add('is-modal-open');
    document.getElementById('logs-ip').textContent = ip;
    document.getElementById('logs-content').innerHTML = '<p class="text-center">Loading...</p>';

    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'get_visitor_logs');
    formData.append('visitor_id', visitorId);
    appendStatsCsrfToken(formData);

    fetch(window.location.pathname, {
        method: 'POST',
        body: formData
    })
    .then(r => {
        if (!r.ok) throw new Error('Network response was not ok');
        return r.json();
    })
    .then(data => {
        if (data.success && data.logs && data.logs.length > 0) {
            let html = '';
            data.logs.forEach(log => {
                const rawUrl = log.page_url || '';
                const safeUrl = escapeHtml(rawUrl);
                const href = rawUrl.startsWith('http') ? rawUrl : `${window.location.origin}${rawUrl}`;
                html += `
                    <div class="log-item">
                        <div class="log-url"><a href="${escapeHtml(href)}" target="_blank" rel="noopener noreferrer">${safeUrl}</a></div>
                        <div class="log-time">${log.created_at}</div>
                        ${log.referrer ? `<div class="log-referrer">From: <a href="${escapeHtml(log.referrer)}" target="_blank" rel="noopener noreferrer">${escapeHtml(log.referrer)}</a></div>` : ''}
                    </div>
                `;
            });
            document.getElementById('logs-content').innerHTML = html;
        } else {
            document.getElementById('logs-content').innerHTML = '<p class="text-muted text-center">No logs found.</p>';
        }
    })
    .catch(err => {
        console.error('Error fetching logs:', err);
        document.getElementById('logs-content').innerHTML = '<p class="text-muted text-center">Error loading logs.</p>';
    });
}

function closeLogsModal() {
    document.getElementById('logs-modal').classList.remove('active');
    document.body.classList.remove('is-modal-open');
}

function banVisitorIP(ip) {
    if (!confirm(`Are you sure you want to ban IP: ${ip}?\n\nBanned users will not be able to access the website.`)) {
        return;
    }

    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'ban_ip');
    formData.append('ip', ip);
    formData.append('reason', 'Banned from statistics panel');
    appendStatsCsrfToken(formData);

    fetch(window.location.pathname, {
        method: 'POST',
        body: formData
    })
    .then(r => {
        if (!r.ok) throw new Error('Network response was not ok');
        return r.json();
    })
    .then(data => {
        if (data.success) {
            showNotification('IP banned successfully!', 'success');
            // Remove row from table
            document.querySelector(`tr[data-ip="${ip}"]`)?.remove();
        } else {
            showNotification(data.error || 'Failed to ban IP', 'error');
        }
    })
    .catch(err => {
        console.error('Error banning IP:', err);
        showNotification('Failed to ban IP', 'error');
    });
}

function unbanIP(ip) {
    if (!confirm(`Unban IP: ${ip}?`)) {
        return;
    }

    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'unban_ip');
    formData.append('ip', ip);
    appendStatsCsrfToken(formData);

    fetch(window.location.pathname, {
        method: 'POST',
        body: formData
    })
    .then(r => {
        if (!r.ok) throw new Error('Network response was not ok');
        return r.json();
    })
    .then(data => {
        if (data.success) {
            showNotification('IP unbanned successfully!', 'success');
            location.reload();
        } else {
            showNotification(data.error || 'Failed to unban IP', 'error');
        }
    })
    .catch(err => {
        console.error('Error unbanning IP:', err);
        showNotification('Failed to unban IP', 'error');
    });
}

function showBanModal() {
    document.getElementById('ban-modal').classList.add('active');
    document.body.classList.add('is-modal-open');
    document.getElementById('ban-ip').value = '';
    document.getElementById('ban-reason').value = '';
}

function closeBanModal() {
    document.getElementById('ban-modal').classList.remove('active');
    document.body.classList.remove('is-modal-open');
}

function submitBan(e) {
    e.preventDefault();

    const ip = document.getElementById('ban-ip').value;
    const reason = document.getElementById('ban-reason').value;

    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'ban_ip');
    formData.append('ip', ip);
    formData.append('reason', reason);
    appendStatsCsrfToken(formData);

    fetch(window.location.pathname, {
        method: 'POST',
        body: formData
    })
    .then(r => {
        if (!r.ok) throw new Error('Network response was not ok');
        return r.json();
    })
    .then(data => {
        if (data.success) {
            showNotification('IP banned successfully!', 'success');
            closeBanModal();
            location.reload();
        } else {
            showNotification(data.error || 'Failed to ban IP', 'error');
        }
    })
    .catch(err => {
        console.error('Error banning IP:', err);
        showNotification('Failed to ban IP', 'error');
    });
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Close modals on escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeLogsModal();
        closeBanModal();
    }
});

// Close modals on backdrop click
document.querySelectorAll('.modal-overlay').forEach(overlay => {
    overlay.addEventListener('click', function(e) {
        if (e.target === this) {
            this.classList.remove('active');
            document.body.classList.remove('is-modal-open');
        }
    });
});
</script>

<?php AdminLayout::renderFooter(); ?>
