<?php
/**
 * Admin Panel - Site Files
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Auth\AdminAuth;
use NewSite\Config\SetupService;
use NewSite\Settings\SettingsService;
use NewSite\Logging\LogService;
use NewSite\Admin\AdminLayout;
use NewSite\Database\DatabaseManager;

AdminAuth::requireLogin();

$baseUrl = SetupService::getBaseUrl();
$publicRoot = realpath(__DIR__ . '/..');
$message = '';
$error = '';

$contactEmail = trim((string)SettingsService::get('contact_email', ''));
$securityContact = $contactEmail !== '' ? 'mailto:' . $contactEmail : $baseUrl . '/contact';
$db = DatabaseManager::getWriteConnection();

// Resolve the favicon path for manifest/browserconfig icons
$faviconPath = trim((string)SettingsService::getTheme('site_favicon', ''));
$faviconPublicUrl = '';
if ($faviconPath !== '') {
    $faviconPublicUrl = str_replace('/admin-file/', '/site-file/', $faviconPath);
}

const SITE_SITEMAP_PATH = '/sitemap.xml';

function sitemapDateFromValue($value): string
{
    if ($value === null || $value === '') {
        return '';
    }
    $timestamp = is_numeric($value) ? (int)$value : strtotime((string)$value);
    if ($timestamp <= 0) {
        return '';
    }
    return date('Y-m-d', $timestamp);
}

function sitemapAddEntry(array &$entries, string $loc, string $changefreq, string $priority, string $lastmod = ''): void
{
    $entries[$loc] = [
        'loc' => $loc,
        'changefreq' => $changefreq,
        'priority' => $priority,
        'lastmod' => $lastmod,
    ];
}

function sitemapAddPublishedPages(array &$entries, string $baseUrl, PDO $db): void
{
    $excludedSlugs = ['login', 'logout', 'account', 'checkout', 'cart', 'messages', 'account-settings', 'checkout-success'];
    $pages = $db->query("SELECT slug, updated_at, created_at FROM pages WHERE is_published = 1 ORDER BY id ASC")->fetchAll(PDO::FETCH_ASSOC);

    foreach ($pages as $pageItem) {
        $slug = trim((string)($pageItem['slug'] ?? ''));
        if ($slug === '' || in_array($slug, $excludedSlugs, true)) {
            continue;
        }

        $path = '/' . ltrim($slug, '/');
        if ($path === '/index' || $path === '/home') {
            continue;
        }

        sitemapAddEntry(
            $entries,
            $baseUrl . $path,
            'weekly',
            '0.7',
            sitemapDateFromValue($pageItem['updated_at'] ?? $pageItem['created_at'] ?? null)
        );
    }
}

function sitemapResolveProductPageSlug(PDO $db): string
{
    $productPageSlug = 'product';
    $stmt = $db->prepare("SELECT slug FROM pages WHERE slug = ? AND is_published = 1 LIMIT 1");
    $stmt->execute([$productPageSlug]);
    return trim((string)$stmt->fetchColumn()) ?: 'product';
}

function sitemapAddProductUrls(array &$entries, string $baseUrl, PDO $db): void
{
    $productPageSlug = sitemapResolveProductPageSlug($db);

    try {
        $products = $db->query("SELECT p.id, p.product_slug, p.updated_at, p.created_at, pg.slug AS page_slug FROM products p LEFT JOIN pages pg ON pg.id = p.page_id WHERE p.is_active = 1")->fetchAll(PDO::FETCH_ASSOC);
        foreach ($products as $product) {
            $pageSlug = trim((string)($product['page_slug'] ?? '')) ?: $productPageSlug;
            $productSlug = trim((string)($product['product_slug'] ?? ''));
            $path = $productSlug !== ''
                ? ('/' . $pageSlug . '/' . rawurlencode($productSlug))
                : ('/' . $pageSlug . '?product=' . (int)($product['id'] ?? 0));

            sitemapAddEntry(
                $entries,
                $baseUrl . $path,
                'weekly',
                '0.7',
                sitemapDateFromValue($product['updated_at'] ?? $product['created_at'] ?? null)
            );
        }
    } catch (PDOException $e) {
        $products = $db->query("SELECT p.id, p.updated_at, p.created_at, pg.slug AS page_slug FROM products p LEFT JOIN pages pg ON pg.id = p.page_id WHERE p.is_active = 1")->fetchAll(PDO::FETCH_ASSOC);
        foreach ($products as $product) {
            $pageSlug = trim((string)($product['page_slug'] ?? '')) ?: $productPageSlug;
            $path = '/' . $pageSlug . '?product=' . (int)($product['id'] ?? 0);
            sitemapAddEntry(
                $entries,
                $baseUrl . $path,
                'weekly',
                '0.7',
                sitemapDateFromValue($product['updated_at'] ?? $product['created_at'] ?? null)
            );
        }
    }
}

function sitemapResolveProductsListPageSlug(PDO $db): string
{
    $stmt = $db->query("SELECT p.slug FROM pages p JOIN sections s ON s.page_id = p.id WHERE s.section_type = 'products_list' AND p.is_published = 1 ORDER BY p.id ASC LIMIT 1");
    return $stmt ? trim((string)$stmt->fetchColumn()) : '';
}

function sitemapCollectionTargetSlug(array $collection, string $fallbackProductsListSlug): string
{
    return trim((string)($collection['target_slug'] ?? '')) ?: $fallbackProductsListSlug;
}

function sitemapCollectionPath(array $collection, string $targetSlug, bool $allowSlug): string
{
    $collectionSlug = trim((string)($collection['collection_slug'] ?? ''));
    if ($allowSlug && $collectionSlug !== '') {
        return '/' . $targetSlug . '/collection/' . rawurlencode($collectionSlug);
    }

    return '/' . $targetSlug . '?collection=' . (int)($collection['id'] ?? 0);
}

function sitemapAppendCollectionEntries(array &$entries, array $collections, string $baseUrl, string $fallbackProductsListSlug, bool $allowSlug): void
{
    foreach ($collections as $collection) {
        $targetSlug = sitemapCollectionTargetSlug($collection, $fallbackProductsListSlug);
        if ($targetSlug === '') {
            continue;
        }

        sitemapAddEntry(
            $entries,
            $baseUrl . sitemapCollectionPath($collection, $targetSlug, $allowSlug),
            'weekly',
            '0.7',
            sitemapDateFromValue($collection['updated_at'] ?? $collection['created_at'] ?? null)
        );
    }
}

function sitemapAddCollectionUrls(array &$entries, string $baseUrl, PDO $db): void
{
    $fallbackProductsListSlug = sitemapResolveProductsListPageSlug($db);

    try {
        $collections = $db->query("SELECT c.id, c.collection_slug, c.updated_at, c.created_at, pg.slug AS target_slug FROM collections c LEFT JOIN pages pg ON pg.id = c.target_page_id AND pg.is_published = 1")->fetchAll(PDO::FETCH_ASSOC);
        sitemapAppendCollectionEntries($entries, $collections, $baseUrl, $fallbackProductsListSlug, true);
    } catch (PDOException $e) {
        $collections = $db->query("SELECT c.id, c.updated_at, c.created_at, pg.slug AS target_slug FROM collections c LEFT JOIN pages pg ON pg.id = c.target_page_id AND pg.is_published = 1")->fetchAll(PDO::FETCH_ASSOC);
        sitemapAppendCollectionEntries($entries, $collections, $baseUrl, $fallbackProductsListSlug, false);
    }
}

function sitemapAddMiniGameUrls(array &$entries, string $baseUrl, PDO $db): void
{
    sitemapAddEntry($entries, $baseUrl . '/games', 'weekly', '0.6', date('Y-m-d'));

    $enabledGames = \NewSite\Minigames\MinigameRepository::getEnabled($db);
    foreach ($enabledGames as $game) {
        $gameSlug = strtolower(trim((string)($game['slug'] ?? '')));
        $gameSlug = preg_replace('/[^a-z0-9\-]/', '', $gameSlug);
        if (!is_string($gameSlug) || $gameSlug === '') {
            continue;
        }

        $publicGameSlug = \NewSite\Minigames\MinigameCatalog::getPublicSlug($gameSlug);
        if ($publicGameSlug === '') {
            continue;
        }

        sitemapAddEntry($entries, $baseUrl . '/games/' . rawurlencode($publicGameSlug), 'weekly', '0.6', date('Y-m-d'));
    }
}

function sitemapAddForumPostUrls(array &$entries, string $baseUrl, PDO $db): void
{
    $forumEnabled = SettingsService::get('forum_enabled', '0') === '1';
    if (!$forumEnabled) {
        return;
    }

    $latestForumUpdate = '';
    $latestStmt = $db->query("SELECT MAX(updated_at) AS latest_updated_at FROM forum_posts WHERE status = 'approved'");
    if ($latestStmt) {
        $latestForumUpdate = sitemapDateFromValue($latestStmt->fetchColumn());
    }
    sitemapAddEntry($entries, $baseUrl . '/forum', 'daily', '0.6', $latestForumUpdate);

    $forumPosts = $db->query("SELECT p.id, p.slug, p.updated_at FROM forum_posts p JOIN forum_subcategories s ON s.id = p.subcategory_id JOIN forum_categories c ON c.id = s.category_id AND c.is_active = 1 WHERE p.status = 'approved' ORDER BY p.id ASC")->fetchAll(PDO::FETCH_ASSOC);
    foreach ($forumPosts as $fp) {
        $fpSlug = trim((string)($fp['slug'] ?? ''));
        $fpPath = $fpSlug !== '' ? rawurlencode($fpSlug) : (string)(int)($fp['id'] ?? 0);
        sitemapAddEntry(
            $entries,
            $baseUrl . '/forum/post/' . $fpPath,
            'weekly',
            '0.5',
            sitemapDateFromValue($fp['updated_at'] ?? null)
        );
    }
}

function buildSitemap(string $baseUrl, PDO $db): string
{
    $entries = [];
    sitemapAddEntry($entries, $baseUrl . '/', 'weekly', '1.0', date('Y-m-d'));

    try {
        sitemapAddPublishedPages($entries, $baseUrl, $db);
        sitemapAddProductUrls($entries, $baseUrl, $db);
        sitemapAddCollectionUrls($entries, $baseUrl, $db);
        sitemapAddMiniGameUrls($entries, $baseUrl, $db);
        sitemapAddForumPostUrls($entries, $baseUrl, $db);
    } catch (PDOException $e) {
        sitemapAddEntry($entries, $baseUrl . '/collections', 'weekly', '0.7', '');
        sitemapAddEntry($entries, $baseUrl . '/contact', 'monthly', '0.5', '');
        sitemapAddEntry($entries, $baseUrl . '/games', 'weekly', '0.6', '');
    }

    $sitemap = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
    $sitemap .= "<urlset xmlns=\"http://www.sitemaps.org/schemas/sitemap/0.9\">\n";
    foreach ($entries as $entry) {
        $sitemap .= "  <url>\n";
        $sitemap .= "    <loc>{$entry['loc']}</loc>\n";
        if (!empty($entry['lastmod'])) {
            $sitemap .= "    <lastmod>{$entry['lastmod']}</lastmod>\n";
        }
        $sitemap .= "    <changefreq>{$entry['changefreq']}</changefreq>\n";
        $sitemap .= "    <priority>{$entry['priority']}</priority>\n";
        $sitemap .= "  </url>\n";
    }
    $sitemap .= "</urlset>\n";

    return $sitemap;
}

$defaultSitemap = buildSitemap($baseUrl, $db);

$siteFiles = [
    'sitemap' => [
        'label' => 'Sitemap.xml',
        'path' => SITE_SITEMAP_PATH,
        'description' => 'Search engine sitemap file for indexing.',
        'editable' => true,
        'default' => $defaultSitemap,
    ],
    'robots' => [
        'label' => 'Robots.txt',
        'path' => '/robots.txt',
        'description' => 'Crawler rules and sitemap reference.',
        'editable' => true,
        'default' => "User-agent: *\n" .
            "Disallow: /admin/\n" .
            "Disallow: /account\n" .
            "Disallow: /account-settings\n" .
            "Disallow: /login\n" .
            "Disallow: /logout\n" .
            "Disallow: /messages\n" .
            "Disallow: /checkout\n" .
            "Disallow: /checkout-success\n" .
            "Disallow: /cart\n" .
            "Disallow: /friends\n" .
            "Disallow: /download\n" .
            "Disallow: /api/\n" .
            "Allow: /\n" .
            "Allow: /assets/\n" .
            "Sitemap: {$baseUrl}" . SITE_SITEMAP_PATH . "\n",
    ],
    'manifest' => [
        'label' => 'Manifest.json',
        'path' => '/manifest.json',
        'description' => 'PWA manifest for installable experiences.',
        'editable' => true,
        'default' => (function () use ($faviconPublicUrl) {
            // Build icons array from the actual favicon, or use placeholder paths
            $iconSrc = $faviconPublicUrl !== '' ? $faviconPublicUrl : '/assets/icons/icon-512x512.png';
            $iconMime = 'image/png';
            if ($faviconPublicUrl !== '') {
                $ext = strtolower(pathinfo($faviconPublicUrl, PATHINFO_EXTENSION));
                $iconMime = match($ext) {
                    'svg' => 'image/svg+xml',
                    'ico' => 'image/x-icon',
                    'gif' => 'image/gif',
                    'jpg', 'jpeg' => 'image/jpeg',
                    'webp' => 'image/webp',
                    default => 'image/png',
                };
            }
            $icons = [
                ['src' => $iconSrc, 'sizes' => '192x192', 'type' => $iconMime],
                ['src' => $iconSrc, 'sizes' => '512x512', 'type' => $iconMime],
                ['src' => $iconSrc, 'sizes' => '512x512', 'type' => $iconMime, 'purpose' => 'maskable']
            ];
            return json_encode([
                'name' => SettingsService::get('site_name', 'Site'),
                'short_name' => SettingsService::get('site_name', 'Site'),
                'description' => SettingsService::get('site_description', ''),
                'start_url' => '/',
                'scope' => '/',
                'display' => 'standalone',
                'background_color' => '#0f0d16',
                'theme_color' => '#6f5bff',
                'icons' => $icons
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES) . "\n";
        })(),
    ],
    'browserconfig' => [
        'label' => 'Browserconfig.xml',
        'path' => '/browserconfig.xml',
        'description' => 'Windows tiles configuration.',
        'editable' => true,
        'default' => "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n" .
            "<browserconfig>\n" .
            "  <msapplication>\n" .
            "    <tile>\n" .
            "      <square150x150logo src=\"" . ($faviconPublicUrl !== '' ? $faviconPublicUrl : '/assets/icons/mstile-150x150.png') . "\"/>\n" .
            "      <TileColor>#0f0d16</TileColor>\n" .
            "    </tile>\n" .
            "  </msapplication>\n" .
            "</browserconfig>\n",
    ],
    'humans' => [
        'label' => 'Humans.txt',
        'path' => '/humans.txt',
        'description' => 'Credits and contact information for humans.',
        'editable' => true,
        'default' => "/* TEAM */\n" .
            "Site: " . SettingsService::get('site_name', 'Site') . "\n" .
            "Contact: {$securityContact}\n" .
            "\n/* THANKS */\n" .
            "Powered by: Celestia\n" .
            "\n/* SITE */\n" .
            "URL: {$baseUrl}\n" .
            "Last update: " . date('Y-m-d') . "\n" .
            "\n",
    ],
    'offline' => [
        'label' => 'Offline.html',
        'path' => '/offline.html',
        'description' => 'Offline fallback page for PWA or service worker.',
        'editable' => true,
        'default' => "<!doctype html>\n" .
            "<html lang=\"en\">\n" .
            "<head>\n" .
            "  <meta charset=\"utf-8\">\n" .
            "  <meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">\n" .
            "  <meta name=\"theme-color\" content=\"#0f0d16\">\n" .
            "  <title>Offline</title>\n" .
            "  <link rel=\"stylesheet\" href=\"/assets/css/style.css\">\n" .
            "</head>\n" .
            "<body class=\"header-top\">\n" .
            "  <div class=\"page-content\">\n" .
            "    <section class=\"section-text-box\">\n" .
            "      <div class=\"container\">\n" .
            "        <div class=\"text-box-card\">\n" .
            "          <h1>You're offline</h1>\n" .
            "          <p>Please check your connection and try again.</p>\n" .
            "          <a class=\"btn btn-primary\" href=\"/\">Go to homepage</a>\n" .
            "        </div>\n" .
            "      </div>\n" .
            "    </section>\n" .
            "  </div>\n" .
            "</body>\n" .
            "</html>\n",
    ],
    'security' => [
        'label' => 'Security.txt',
        'path' => '/.well-known/security.txt',
        'description' => 'Security contact and disclosure policy.',
        'editable' => true,
        'default' => "Contact: {$securityContact}\n" .
            "Expires: " . date('Y-m-d', strtotime('+1 year')) . "\n" .
            "Preferred-Languages: en\n" .
            "Canonical: {$baseUrl}/.well-known/security.txt\n" .
            "\n",
    ],
];

foreach ($siteFiles as $key => $file) {
    $siteFiles[$key]['file'] = $publicRoot . $file['path'];
    $siteFiles[$key]['url'] = $file['path'];
}

$selectedKey = $_GET['file'] ?? '';
$selectedFile = $siteFiles[$selectedKey] ?? null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verifyCSRF();
    $action = $_POST['action'] ?? '';
    $key = $_POST['key'] ?? '';
    $selectedFile = $siteFiles[$key] ?? null;

    if (!$selectedFile) {
        $error = 'Invalid file selection.';
    } elseif (($selectedFile['editable'] ?? false) !== true) {
        $error = 'This file is view-only.';
    } elseif ($action === 'save') {
        $content = (string)($_POST['content'] ?? '');
        $targetPath = $selectedFile['file'];
        $dir = dirname($targetPath);
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
        if (@file_put_contents($targetPath, $content, LOCK_EX) === false) {
            $error = 'Failed to write file.';
        } else {
            $message = 'File saved.';
            LogService::add('info', 'Site file saved', json_encode(['file' => $selectedFile['path']]));
        }
    } elseif ($action === 'delete') {
        $targetPath = $selectedFile['file'];
        if (file_exists($targetPath)) {
            if (@unlink($targetPath) === false) {
                $error = 'Failed to delete file.';
            } else {
                $message = 'File deleted.';
                LogService::add('info', 'Site file deleted', json_encode(['file' => $selectedFile['path']]));
            }
        } else {
            $error = 'File not found.';
        }
    } elseif ($action === 'refresh_sitemap') {
        if (($selectedFile['path'] ?? '') !== '/sitemap.xml') {
            $error = 'Sitemap auto-complete is only available for sitemap.xml.';
        } else {
            $content = buildSitemap($baseUrl, $db);
            $targetPath = $selectedFile['file'];
            $dir = dirname($targetPath);
            if (!is_dir($dir)) {
                mkdir($dir, 0755, true);
            }
            if (@file_put_contents($targetPath, $content, LOCK_EX) === false) {
                $error = 'Failed to refresh sitemap.';
            } else {
                $message = 'Sitemap refreshed from published pages.';
                LogService::add('info', 'Sitemap refreshed', json_encode(['file' => $selectedFile['path']]));
            }
        }
    }
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header">
        <h1>Site Files</h1>
        <p>Manage SEO and platform files for search engines, PWA, and system metadata.</p>
    </div>

    <?php if ($message): ?>
        <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <div class="card">
        <div class="card-header">
            <h3>File Status</h3>
        </div>
        <div class="card-body">
            <div class="table-container">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>File</th>
                            <th>Purpose</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($siteFiles as $key => $file):
                            $exists = file_exists($file['file']);
                            ?>
                            <tr>
                                <td><?php echo e($file['label']); ?><div class="text-muted"><?php echo e($file['path']); ?></div></td>
                                <td><?php echo e($file['description']); ?></td>
                                <td>
                                    <?php if ($exists): ?>
                                        <span class="status-badge status-active">Available</span>
                                    <?php else: ?>
                                        <span class="status-badge status-inactive">Missing</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <div class="table-actions">
                                        <a class="btn btn-sm btn-secondary" href="<?php echo e($file['url']); ?>" target="_blank">View</a>
                                        <?php if (!empty($file['editable'])): ?>
                                            <a class="btn btn-sm btn-primary" href="/admin/site-files.php?file=<?php echo e($key); ?>">Edit</a>
                                        <?php endif; ?>
                                        <?php if ($exists): ?>
                                            <form method="POST" action="/admin/site-files.php?file=<?php echo e($key); ?>">
                                                <input type="hidden" name="action" value="delete">
                                                <input type="hidden" name="key" value="<?php echo e($key); ?>">
                                                <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                                                <button type="submit" class="btn btn-sm btn-outline" data-confirm="Delete this file?">Delete</button>
                                            </form>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <?php if ($selectedFile):
        $fileExists = file_exists($selectedFile['file']);
        $currentContent = $fileExists ? file_get_contents($selectedFile['file']) : ($selectedFile['default'] ?? '');
        $buttonLabel = $fileExists ? 'Save Changes' : 'Create File';
        ?>
        <div class="card">
            <div class="card-header">
                <h3>Edit <?php echo e($selectedFile['label']); ?></h3>
            </div>
            <div class="card-body">
                <form method="POST" action="/admin/site-files.php?file=<?php echo e($selectedKey); ?>">
                    <input type="hidden" name="action" value="save">
                    <input type="hidden" name="key" value="<?php echo e($selectedKey); ?>">
                    <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                    <div class="form-group">
                        <label for="site-file-content"><?php echo e($selectedFile['path']); ?></label>
                        <textarea id="site-file-content" name="content" class="form-control" rows="14"><?php echo e($currentContent); ?></textarea>
                    </div>
                    <div class="table-actions">
                        <button type="submit" class="btn btn-primary"><?php echo e($buttonLabel); ?></button>
                        <?php if ($selectedFile['path'] === SITE_SITEMAP_PATH): ?>
                            <button type="submit" class="btn btn-secondary" name="action" value="refresh_sitemap">Auto-complete Sitemap</button>
                        <?php endif; ?>
                    </div>
                </form>
            </div>
        </div>
    <?php endif; ?>
</div>

<?php AdminLayout::renderFooter(); ?>
