<?php
/**
 * Admin Panel - Settings
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Admin\AdminLayout;
use NewSite\Auth\AdminAuth;
use NewSite\Contact\ContactService;
use NewSite\Database\DatabaseManager;
use NewSite\Email\SmtpMailer;
use NewSite\Gdpr\GdprService;
use NewSite\Logging\LogService;
use NewSite\Settings\SettingsService;
use NewSite\Template\ContentRenderer;
use NewSite\Upload\UploadService;
use NewSite\User\UserService;

AdminAuth::requireLogin();

$db = DatabaseManager::getWriteConnection();

const SETTINGS_REDIRECT_HEADER = 'Location: /admin/settings.php';
const SETTINGS_FONTS_UPLOAD_DIR = '/../assets/uploads/fonts/';
const SETTINGS_MIME_OCTET_STREAM = 'application/octet-stream';

$message = '';
$error = '';
$adminUserId = (int)($_SESSION['admin_user_id'] ?? 0);
$adminUsername = '';
if ($adminUserId > 0) {
    $stmt = $db->prepare("SELECT username FROM users WHERE id = ? LIMIT 1");
    $stmt->execute([$adminUserId]);
    $adminUsername = (string)$stmt->fetchColumn();
}

$flash = $_SESSION['admin_flash_settings'] ?? null;
if (is_array($flash)) {
    $message = $flash['message'] ?? '';
    $error = $flash['error'] ?? '';
    unset($_SESSION['admin_flash_settings']);
}

$action = $_POST['action'] ?? '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'gdpr_user_tools') {
    verifyCSRF();
    $identifier = trim($_POST['gdpr_user_identifier'] ?? '');
    $requestAction = $_POST['gdpr_request_action'] ?? 'export';
    $confirmDelete = trim($_POST['gdpr_delete_confirm'] ?? '');

    if ($identifier === '') {
        $_SESSION['admin_flash_settings'] = ['error' => 'Please enter a user email or ID.'];
        header(SETTINGS_REDIRECT_HEADER);
        exit;
    }

    $userId = 0;
    if (ctype_digit($identifier)) {
        $userId = (int)$identifier;
    } else {
        $user = UserService::getByEmail($identifier);
        $userId = (int)($user['id'] ?? 0);
    }

    if ($userId <= 0) {
        $_SESSION['admin_flash_settings'] = ['error' => 'User not found.'];
        header(SETTINGS_REDIRECT_HEADER);
        exit;
    }

    if ($requestAction === 'delete') {
        if ($confirmDelete !== 'DELETE') {
            $_SESSION['admin_flash_settings'] = ['error' => 'Type DELETE to confirm account deletion.'];
            header(SETTINGS_REDIRECT_HEADER);
            exit;
        }

        if (GdprService::deleteUserData($userId)) {
            LogService::add('gdpr', 'Admin deleted user data', json_encode([
                'admin_id' => $adminUserId,
                'user_id' => $userId
            ]));
            $_SESSION['admin_flash_settings'] = ['message' => 'User data deleted successfully.'];
        } else {
            $_SESSION['admin_flash_settings'] = ['error' => 'Unable to delete user data.'];
        }

        header(SETTINGS_REDIRECT_HEADER);
        exit;
    }

    $export = GdprService::exportUserData($userId);
    LogService::add('gdpr', 'Admin exported user data', json_encode([
        'admin_id' => $adminUserId,
        'user_id' => $userId
    ]));
    header('Content-Type: application/json');
    header('Content-Disposition: attachment; filename="user-data-export-' . $userId . '.json"');
    echo json_encode($export, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    exit;
}

if ($action === 'gdpr_breach_alert') {
    $title = mb_substr(trim($_POST['breach_title'] ?? ''), 0, 140);
    $summary = mb_substr(trim($_POST['breach_summary'] ?? ''), 0, 1000);
    $impact = mb_substr(trim($_POST['breach_impact'] ?? ''), 0, 500);
    $severity = trim($_POST['breach_severity'] ?? 'medium');
    $records = (int)($_POST['breach_records'] ?? 0);
    $detectedAt = trim($_POST['breach_detected_at'] ?? '');
    $resolvedAt = trim($_POST['breach_resolved_at'] ?? '');
    $notify = isset($_POST['breach_notify']);

    if ($title === '' || $summary === '') {
        $_SESSION['admin_flash_settings'] = ['error' => 'Breach title and summary are required.'];
        header(SETTINGS_REDIRECT_HEADER);
        exit;
    }

    $details = [
        'title' => $title,
        'summary' => $summary,
        'impact' => $impact,
        'severity' => $severity,
        'records' => $records,
        'detected_at' => $detectedAt,
        'resolved_at' => $resolvedAt,
        'admin_id' => $adminUserId,
        'admin_username' => $adminUsername
    ];

    LogService::add('security', 'Data breach alert logged', json_encode($details));

    if ($notify) {
        $to = ContactService::getEmail();
        if (filter_var($to, FILTER_VALIDATE_EMAIL)) {
            $smtp = [
                'host' => SettingsService::get('smtp_host', ''),
                'port' => SettingsService::get('smtp_port', '587'),
                'user' => SettingsService::get('smtp_user', ''),
                'pass' => SettingsService::get('smtp_pass', ''),
                'secure' => SettingsService::get('smtp_secure', 'tls')
            ];
            $subject = 'Data breach alert: ' . $title;
            $body = "A data breach alert has been logged.\n\n";
            $body .= "Title: {$title}\n";
            $body .= "Summary: {$summary}\n";
            if ($impact !== '') {
                $body .= "Impact: {$impact}\n";
            }
            if ($records > 0) {
                $body .= "Records: {$records}\n";
            }
            if ($detectedAt !== '') {
                $body .= "Detected: {$detectedAt}\n";
            }
            if ($resolvedAt !== '') {
                $body .= "Resolved: {$resolvedAt}\n";
            }
            $body .= "\nLogged by: {$adminUsername}\n";

            $sent = SmtpMailer::send($to, $subject, $body, '', '', $smtp);
            if (!$sent) {
                LogService::add('error', 'Breach alert email failed', json_encode(['to' => $to]));
            }
        } else {
            LogService::add('error', 'Breach alert email skipped (invalid contact email)', json_encode(['email' => $to]));
        }
    }

    $_SESSION['admin_flash_settings'] = ['message' => 'Breach alert logged successfully.'];
    header(SETTINGS_REDIRECT_HEADER);
    exit;
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($action === 'general' || $action === 'account')) {
    try {
        if ($action === 'general') {
            SettingsService::set('maintenance_mode', isset($_POST['maintenance_mode']) ? '1' : '0');
        }
        SettingsService::set('allow_registration', isset($_POST['allow_registration']) ? '1' : '0');
        SettingsService::set('contact_enabled', isset($_POST['contact_enabled']) ? '1' : '0');
        SettingsService::set('page_terms_enabled', isset($_POST['page_terms_enabled']) ? '1' : '0');
        SettingsService::set('page_privacy_enabled', isset($_POST['page_privacy_enabled']) ? '1' : '0');
        SettingsService::set('page_refund_enabled', isset($_POST['page_refund_enabled']) ? '1' : '0');
        SettingsService::set('page_shipping_enabled', isset($_POST['page_shipping_enabled']) ? '1' : '0');
        SettingsService::set('page_legal_notice_enabled', isset($_POST['page_legal_notice_enabled']) ? '1' : '0');
        SettingsService::set('page_dmca_enabled', isset($_POST['page_dmca_enabled']) ? '1' : '0');
        SettingsService::set('cookie_notice_enabled', isset($_POST['cookie_notice_enabled']) ? '1' : '0');
        SettingsService::set('contact_email', trim($_POST['contact_email'] ?? ''));
        SettingsService::set('contact_subject', trim($_POST['contact_subject'] ?? ''));
        SettingsService::set('site_name', trim($_POST['site_name'] ?? ''));
        SettingsService::set('contact_name', trim($_POST['contact_name'] ?? ''));
        SettingsService::set('contact_phone', trim($_POST['contact_phone'] ?? ''));
        SettingsService::set('contact_address', trim($_POST['contact_address'] ?? ''));
        SettingsService::set('legal_entity_name', trim($_POST['legal_entity_name'] ?? ''));
        SettingsService::set('legal_legal_form', trim($_POST['legal_legal_form'] ?? ''));
        SettingsService::set('legal_representative', trim($_POST['legal_representative'] ?? ''));
        SettingsService::set('legal_street', trim($_POST['legal_street'] ?? ''));
        SettingsService::set('legal_postal_code', trim($_POST['legal_postal_code'] ?? ''));
        SettingsService::set('legal_city', trim($_POST['legal_city'] ?? ''));
        SettingsService::set('legal_country', trim($_POST['legal_country'] ?? ''));
        SettingsService::set('legal_register_court', trim($_POST['legal_register_court'] ?? ''));
        SettingsService::set('legal_register_number', trim($_POST['legal_register_number'] ?? ''));
        SettingsService::set('legal_vat_id', trim($_POST['legal_vat_id'] ?? ''));
        SettingsService::set('legal_business_id', trim($_POST['legal_business_id'] ?? ''));
        SettingsService::set('legal_editor_name', trim($_POST['legal_editor_name'] ?? ''));
        SettingsService::set('legal_editor_address', trim($_POST['legal_editor_address'] ?? ''));
        SettingsService::set('legal_profession_name', trim($_POST['legal_profession_name'] ?? ''));
        SettingsService::set('legal_professional_title', trim($_POST['legal_professional_title'] ?? ''));
        SettingsService::set('legal_professional_title_country', trim($_POST['legal_professional_title_country'] ?? ''));
        SettingsService::set('legal_professional_rules', trim($_POST['legal_professional_rules'] ?? ''));
        SettingsService::set('legal_professional_rules_url', trim($_POST['legal_professional_rules_url'] ?? ''));
        SettingsService::set('legal_chamber', trim($_POST['legal_chamber'] ?? ''));
        SettingsService::set('legal_supervisory_authority', trim($_POST['legal_supervisory_authority'] ?? ''));
        SettingsService::set('legal_odr_url', trim($_POST['legal_odr_url'] ?? ''));
        SettingsService::set('legal_dispute_participation', trim($_POST['legal_dispute_participation'] ?? ''));
        SettingsService::set('legal_hosting_provider_name', trim($_POST['legal_hosting_provider_name'] ?? ''));
        SettingsService::set('legal_hosting_provider_address', trim($_POST['legal_hosting_provider_address'] ?? ''));
        SettingsService::set('smtp_host', trim($_POST['smtp_host'] ?? ''));
        SettingsService::set('smtp_port', trim($_POST['smtp_port'] ?? ''));
        SettingsService::set('smtp_user', trim($_POST['smtp_user'] ?? ''));
        SettingsService::set('smtp_pass', trim($_POST['smtp_pass'] ?? ''));
        SettingsService::set('smtp_secure', trim($_POST['smtp_secure'] ?? 'tls'));
        SettingsService::set('social_facebook', trim($_POST['social_facebook'] ?? ''));
        SettingsService::set('social_instagram', trim($_POST['social_instagram'] ?? ''));
        SettingsService::set('social_x', trim($_POST['social_x'] ?? ''));
        SettingsService::set('social_youtube', trim($_POST['social_youtube'] ?? ''));
        SettingsService::set('social_discord', trim($_POST['social_discord'] ?? ''));
        SettingsService::set('social_tiktok', trim($_POST['social_tiktok'] ?? ''));
        SettingsService::set('social_linkedin', trim($_POST['social_linkedin'] ?? ''));
        SettingsService::set('stripe_publishable_key', trim($_POST['stripe_publishable_key'] ?? ''));
        SettingsService::set('stripe_secret_key', trim($_POST['stripe_secret_key'] ?? ''));
        SettingsService::set('stripe_webhook_secret', trim($_POST['stripe_webhook_secret'] ?? ''));
        SettingsService::set('google_oauth_enabled', isset($_POST['google_oauth_enabled']) ? '1' : '0');
        SettingsService::set('google_oauth_client_id', trim($_POST['google_oauth_client_id'] ?? ''));
        SettingsService::set('google_oauth_client_secret', trim($_POST['google_oauth_client_secret'] ?? ''));
        SettingsService::set('google_oauth_allow_registration', isset($_POST['google_oauth_allow_registration']) ? '1' : '0');
        SettingsService::set('cart_sidebar_enabled', isset($_POST['cart_sidebar_enabled']) ? '1' : '0');
        SettingsService::set('cart_icon_items_only', isset($_POST['cart_icon_items_only']) ? '1' : '0');
        SettingsService::set('minibar_sticky', isset($_POST['minibar_sticky']) ? '1' : '0');
        SettingsService::set('refund_window_digital_days', (string)max(0, (int)($_POST['refund_window_digital_days'] ?? 30)));
        SettingsService::set('refund_window_physical_days', (string)max(0, (int)($_POST['refund_window_physical_days'] ?? 30)));
        SettingsService::set('custom_cursor_enabled', isset($_POST['custom_cursor_enabled']) ? '1' : '0');
        $videoThumbnailMode = $_POST['video_thumbnail_mode'] ?? 'auto';
        $allowedVideoThumbnailModes = ['auto', 'client', 'server', 'off'];
        if (!in_array($videoThumbnailMode, $allowedVideoThumbnailModes, true)) {
            $videoThumbnailMode = 'auto';
        }
        SettingsService::set('video_thumbnail_mode', $videoThumbnailMode);

        $easyMediaAiZipRaw = trim((string)($_POST['easy_media_ai_zip_url'] ?? ''));
        $easyMediaAiZipUrl = ContentRenderer::normalizeEasyMediaZipDownloadPath($easyMediaAiZipRaw);
        if ($easyMediaAiZipRaw !== '' && $easyMediaAiZipUrl === '') {
            LogService::add('warning', 'Rejected invalid Easy Media AI ZIP setting', json_encode([
                'submitted_value' => $easyMediaAiZipRaw,
                'admin_id' => $adminUserId
            ]));
        }
        SettingsService::set('easy_media_ai_zip_url', $easyMediaAiZipUrl);

        for ($helpImageIndex = 1; $helpImageIndex <= 12; $helpImageIndex++) {
            $helpImageKey = 'easy_media_ai_help_image_' . $helpImageIndex;
            $helpImageRaw = trim((string)($_POST[$helpImageKey] ?? ''));
            $helpImagePath = ContentRenderer::normalizeEasyMediaHelpImagePath($helpImageRaw);
            if ($helpImageRaw !== '' && $helpImagePath === '') {
                LogService::add('warning', 'Rejected invalid Easy Media help image setting', json_encode([
                    'slot' => $helpImageIndex,
                    'submitted_value' => $helpImageRaw,
                    'admin_id' => $adminUserId
                ]));
            }
            SettingsService::set($helpImageKey, $helpImagePath);
        }

        SettingsService::set('search_show_mini', isset($_POST['search_show_mini']) ? '1' : '0');
        SettingsService::set('search_show_top', isset($_POST['search_show_top']) ? '1' : '0');
        SettingsService::set('search_show_sidebar', isset($_POST['search_show_sidebar']) ? '1' : '0');
        SettingsService::set('search_show_footer', isset($_POST['search_show_footer']) ? '1' : '0');
        SettingsService::set('account_show_mini', isset($_POST['account_show_mini']) ? '1' : '0');
        SettingsService::set('account_show_top', isset($_POST['account_show_top']) ? '1' : '0');
        SettingsService::set('account_show_sidebar', isset($_POST['account_show_sidebar']) ? '1' : '0');
        SettingsService::set('account_show_footer', isset($_POST['account_show_footer']) ? '1' : '0');
        SettingsService::set('cart_show_mini', isset($_POST['cart_show_mini']) ? '1' : '0');
        SettingsService::set('cart_show_top', isset($_POST['cart_show_top']) ? '1' : '0');
        SettingsService::set('cart_show_sidebar', isset($_POST['cart_show_sidebar']) ? '1' : '0');
        SettingsService::set('cart_show_footer', isset($_POST['cart_show_footer']) ? '1' : '0');
        SettingsService::set('gdpr_ip_anonymization', isset($_POST['gdpr_ip_anonymization']) ? '1' : '0');
        SettingsService::set('gdpr_auto_purge', isset($_POST['gdpr_auto_purge']) ? '1' : '0');
        SettingsService::set('gdpr_retention_logs_days', (string)max(1, (int)($_POST['gdpr_retention_logs_days'] ?? 30)));
        SettingsService::set('gdpr_retention_visitors_days', (string)max(1, (int)($_POST['gdpr_retention_visitors_days'] ?? 30)));
        SettingsService::set('gdpr_retention_messages_days', (string)max(1, (int)($_POST['gdpr_retention_messages_days'] ?? 365)));
        SettingsService::set('gdpr_retention_contact_days', (string)max(1, (int)($_POST['gdpr_retention_contact_days'] ?? 365)));
        SettingsService::set('gdpr_retention_carts_days', (string)max(1, (int)($_POST['gdpr_retention_carts_days'] ?? 7)));
        SettingsService::set('gdpr_retention_rate_limits_days', (string)max(1, (int)($_POST['gdpr_retention_rate_limits_days'] ?? 7)));
        SettingsService::set('gdpr_retention_geo_cache_days', (string)max(1, (int)($_POST['gdpr_retention_geo_cache_days'] ?? 7)));
        $notificationLocation = $_POST['notification_location'] ?? 'mini';
        $allowedLocations = ['mini', 'top', 'sidebar'];
        if (!in_array($notificationLocation, $allowedLocations, true)) {
            $notificationLocation = 'mini';
        }
        SettingsService::set('notification_location', $notificationLocation);

        $currencyMap = [
            'USD' => '$',
            'EUR' => '€',
            'GBP' => '£',
            'JPY' => '¥',
            'AUD' => 'A$',
            'CAD' => 'C$',
            'CHF' => 'CHF',
            'SEK' => 'SEK',
            'NOK' => 'NOK',
            'DKK' => 'DKK',
            'PLN' => 'PLN',
            'BRL' => 'R$',
            'MXN' => 'MX$',
            'INR' => '₹',
            'KRW' => '₩',
            'CNY' => '¥'
        ];
        $storeCurrency = strtoupper(trim($_POST['store_currency'] ?? 'USD'));
        if (!array_key_exists($storeCurrency, $currencyMap)) {
            $storeCurrency = 'USD';
        }
        SettingsService::set('store_currency', $storeCurrency);
        SettingsService::set('store_currency_symbol', $currencyMap[$storeCurrency]);

        $customFonts = json_decode(SettingsService::get('custom_fonts', '[]'), true);
        if (!is_array($customFonts)) {
            $customFonts = [];
        }

        if (isset($_POST['remove_custom_font']) && $_POST['remove_custom_font'] !== '') {
            $removeFile = basename((string)$_POST['remove_custom_font']);
            $customFonts = array_values(array_filter($customFonts, function ($font) use ($removeFile) {
                return ($font['file'] ?? '') !== $removeFile;
            }));

            $fontPath = __DIR__ . SETTINGS_FONTS_UPLOAD_DIR . $removeFile;
            if (is_file($fontPath)) {
                unlink($fontPath);
            }

            SettingsService::set('custom_fonts', json_encode($customFonts));
            $message = 'Custom font removed.';
            LogService::add('info', 'Custom font removed', json_encode(['file' => $removeFile]));
        }

        if (isset($_POST['add_custom_font'])) {
            $fontName = trim($_POST['custom_font_name'] ?? '');
            if ($fontName === '') {
                $error = 'Custom font name is required.';
            } elseif (!isset($_FILES['custom_font_file']) || $_FILES['custom_font_file']['error'] !== UPLOAD_ERR_OK) {
                $error = 'Custom font file upload failed.';
            } else {
                $file = $_FILES['custom_font_file'];
                $allowedMap = [
                    'woff2' => ['font/woff2', 'application/woff2', 'application/font-woff2', 'application/x-font-woff2', SETTINGS_MIME_OCTET_STREAM],
                    'woff' => ['font/woff', 'application/woff', 'application/font-woff', 'application/x-font-woff', SETTINGS_MIME_OCTET_STREAM],
                    'ttf' => ['font/ttf', 'application/font-sfnt', 'application/x-font-ttf', 'application/x-font-truetype', SETTINGS_MIME_OCTET_STREAM],
                    'otf' => ['font/otf', 'application/font-sfnt', 'application/x-font-opentype', SETTINGS_MIME_OCTET_STREAM]
                ];
                $validation = UploadService::validateFile($file, $allowedMap, false);

                if (!$validation['success']) {
                    $error = $validation['error'] ?: 'Unsupported font format. Use WOFF2, WOFF, TTF, or OTF.';
                    if (!empty($validation['mime'])) {
                        $error .= ' (Detected: ' . $validation['mime'] . ')';
                    }
                } else {
                    $uploadDir = __DIR__ . SETTINGS_FONTS_UPLOAD_DIR;
                    if (!is_dir($uploadDir)) {
                        @mkdir($uploadDir, 0755, true);
                    }

                    $safeName = preg_replace('/[^a-zA-Z0-9\s_-]/', '', $fontName);
                    $fileName = 'font_' . time() . '_' . bin2hex(random_bytes(4)) . '.' . ($validation['ext'] ?? 'woff');
                    $dest = $uploadDir . $fileName;

                    if (move_uploaded_file($file['tmp_name'], $dest)) {
                        foreach ($customFonts as $existing) {
                            if (($existing['name'] ?? '') === $safeName && !empty($existing['file'])) {
                                $oldPath = __DIR__ . SETTINGS_FONTS_UPLOAD_DIR . basename($existing['file']);
                                if (is_file($oldPath)) {
                                    unlink($oldPath);
                                }
                            }
                        }

                        $customFonts = array_values(array_filter($customFonts, function ($font) use ($safeName) {
                            return ($font['name'] ?? '') !== $safeName;
                        }));

                        $customFonts[] = [
                            'name' => $safeName,
                            'file' => $fileName
                        ];

                        SettingsService::set('custom_fonts', json_encode($customFonts));
                        $message = 'Custom font added successfully.';
                        LogService::add('info', 'Custom font added', json_encode(['name' => $safeName, 'file' => $fileName]));
                    } else {
                        $error = 'Failed to save the font file.';
                    }
                }
            }
        }

        if ($error === '') {
            $newUsername = trim($_POST['admin_username'] ?? '');
            if ($newUsername !== '' && $newUsername !== $adminUsername) {
                if (strlen($newUsername) < 3) {
                    $error = 'Username must be at least 3 characters.';
                    LogService::add('warning', 'Admin username change rejected: too short');
                } else {
                    $stmt = $db->prepare("SELECT id FROM users WHERE username = ? AND id != ? LIMIT 1");
                    $stmt->execute([$newUsername, $adminUserId]);
                    if ($stmt->fetch()) {
                        $error = 'That username is already in use.';
                        LogService::add('warning', 'Admin username change rejected: duplicate');
                    } else {
                        $stmt = $db->prepare("UPDATE users SET username = ? WHERE id = ?");
                        $stmt->execute([$newUsername, $adminUserId]);
                        $_SESSION['admin_username'] = $newUsername;
                        $adminUsername = $newUsername;
                        $message = $message ?: 'Settings updated successfully.';
                        LogService::add('info', 'Admin username changed');
                    }
                }
            }
        }

        // Update admin password if provided
        if ($error === '' && !empty($_POST['new_password'])) {
            if (strlen($_POST['new_password']) < 6) {
                $error = 'Password must be at least 6 characters.';
                LogService::add('warning', 'Password change rejected: too short');
            } elseif ($_POST['new_password'] !== $_POST['confirm_password']) {
                $error = 'Passwords do not match.';
                LogService::add('warning', 'Password change rejected: mismatch');
            } else {
                $hash = password_hash($_POST['new_password'], PASSWORD_ARGON2ID);
                $stmt = $db->prepare("UPDATE users SET password = ? WHERE id = ?");
                $stmt->execute([$hash, $adminUserId]);
                $message = $message ?: 'Settings and password updated successfully.';
                LogService::add('info', 'Admin password changed');
            }
        }

        if ($message === '' && $error === '') {
            $message = 'Settings updated successfully.';
        }

        if (isset($_POST['send_test_email'])) {
            $smtp = [
                'host' => SettingsService::get('smtp_host', ''),
                'port' => SettingsService::get('smtp_port', '587'),
                'user' => SettingsService::get('smtp_user', ''),
                'pass' => SettingsService::get('smtp_pass', ''),
                'secure' => SettingsService::get('smtp_secure', 'tls')
            ];
            $to = SettingsService::get('contact_email', '');
            if ($to && SmtpMailer::send($to, 'SMTP Test Email', 'This is a test email from the admin panel.', '', '', $smtp)) {
                $message = 'Test email sent successfully.';
            } else {
                $error = 'Failed to send test email. Check SMTP settings.';
            }
        }

        if (isset($_POST['clear_contact_uploads'])) {
            $uploadDir = __DIR__ . '/../../data/contact_uploads/';
            $removed = 0;
            if (is_dir($uploadDir)) {
                $files = glob($uploadDir . '*');
                foreach ($files as $file) {
                    $basename = basename($file);
                    if ($basename === '.gitkeep' || $basename === '.htaccess') {
                        continue;
                    }
                    if (is_file($file)) {
                        unlink($file);
                        $removed++;
                    }
                }
            }
            $message = 'Cleared contact uploads (' . $removed . ' files).';
            LogService::add('info', 'Cleared contact uploads', json_encode(['count' => $removed]));
        }
    } catch (PDOException $e) {
        $error = 'Failed to save settings.';
        LogService::add('error', 'Settings save failed: ' . $e->getMessage());
    }

    if ($error === '') {
        $_SESSION['admin_flash_settings'] = ['message' => $message, 'error' => $error];
        header(SETTINGS_REDIRECT_HEADER);
        exit;
    }
}

// Get current settings
$maintenanceMode = SettingsService::get('maintenance_mode', '0') === '1';
$allowRegistration = SettingsService::get('allow_registration', '0') === '1';
$contactEnabled = SettingsService::get('contact_enabled', '1') === '1';
$pageTermsEnabled = SettingsService::get('page_terms_enabled', '1') === '1';
$pagePrivacyEnabled = SettingsService::get('page_privacy_enabled', '1') === '1';
$pageRefundEnabled = SettingsService::get('page_refund_enabled', '1') === '1';
$pageShippingEnabled = SettingsService::get('page_shipping_enabled', '1') === '1';
$pageLegalNoticeEnabled = SettingsService::get('page_legal_notice_enabled', '1') === '1';
$pageDmcaEnabled = SettingsService::get('page_dmca_enabled', '1') === '1';
$cookieNoticeEnabled = SettingsService::get('cookie_notice_enabled', '1') === '1';
$contactEmail = SettingsService::get('contact_email', '');
$contactSubject = SettingsService::get('contact_subject', 'Contact Form Message');
$themeSiteName = SettingsService::getTheme('site_name', '');
$siteNameSetting = SettingsService::get('site_name', $themeSiteName);
$contactName = SettingsService::get('contact_name', $siteNameSetting);
$contactPhone = SettingsService::get('contact_phone', '');
$contactAddress = SettingsService::get('contact_address', '');
$legalEntityName = SettingsService::get('legal_entity_name', $contactName);
$legalLegalForm = SettingsService::get('legal_legal_form', '');
$legalRepresentative = SettingsService::get('legal_representative', '');
$legalStreet = SettingsService::get('legal_street', '');
$legalPostalCode = SettingsService::get('legal_postal_code', '');
$legalCity = SettingsService::get('legal_city', '');
$legalCountry = SettingsService::get('legal_country', '');
$legalRegisterCourt = SettingsService::get('legal_register_court', '');
$legalRegisterNumber = SettingsService::get('legal_register_number', '');
$legalVatId = SettingsService::get('legal_vat_id', '');
$legalBusinessId = SettingsService::get('legal_business_id', '');
$legalEditorName = SettingsService::get('legal_editor_name', '');
$legalEditorAddress = SettingsService::get('legal_editor_address', '');
$legalProfessionName = SettingsService::get('legal_profession_name', '');
$legalProfessionalTitle = SettingsService::get('legal_professional_title', '');
$legalProfessionalTitleCountry = SettingsService::get('legal_professional_title_country', '');
$legalProfessionalRules = SettingsService::get('legal_professional_rules', '');
$legalProfessionalRulesUrl = SettingsService::get('legal_professional_rules_url', '');
$legalChamber = SettingsService::get('legal_chamber', '');
$legalSupervisoryAuthority = SettingsService::get('legal_supervisory_authority', '');
$legalOdrUrl = SettingsService::get('legal_odr_url', 'https://ec.europa.eu/consumers/odr');
$legalDisputeParticipation = SettingsService::get('legal_dispute_participation', 'We are not willing or obliged to participate in dispute resolution proceedings before a consumer arbitration board.');
$legalHostingProviderName = SettingsService::get('legal_hosting_provider_name', '');
$legalHostingProviderAddress = SettingsService::get('legal_hosting_provider_address', '');
$smtpHost = SettingsService::get('smtp_host', '');
$smtpPort = SettingsService::get('smtp_port', '587');
$smtpUser = SettingsService::get('smtp_user', '');
$smtpPass = SettingsService::get('smtp_pass', '');
$smtpSecure = SettingsService::get('smtp_secure', 'tls');
$socialFacebook = SettingsService::get('social_facebook', '');
$socialInstagram = SettingsService::get('social_instagram', '');
$socialX = SettingsService::get('social_x', '');
$socialYouTube = SettingsService::get('social_youtube', '');
$socialDiscord = SettingsService::get('social_discord', '');
$socialTikTok = SettingsService::get('social_tiktok', '');
$socialLinkedIn = SettingsService::get('social_linkedin', '');
$stripePublishableKey = SettingsService::get('stripe_publishable_key', '');
$stripeSecretKey = SettingsService::get('stripe_secret_key', '');
$stripeWebhookSecret = SettingsService::get('stripe_webhook_secret', '');
$googleOAuthEnabled = SettingsService::get('google_oauth_enabled', '0') === '1';
$googleOAuthClientId = SettingsService::get('google_oauth_client_id', '');
$googleOAuthClientSecret = SettingsService::get('google_oauth_client_secret', '');
$googleOAuthAllowRegistration = SettingsService::get('google_oauth_allow_registration', '1') === '1';
$purchaseEmailSubject = SettingsService::get('purchase_email_subject', 'Thanks for your purchase');
$purchaseEmailBody = SettingsService::get('purchase_email_body', "Hi {name},\n\nThanks for your purchase!\nYou can view your purchase anytime by logging into your account: {account_url}\n\nOrder: {order_number}\nTotal: {total}\n\nThank you!\n");
$productUpdateEmailSubject = SettingsService::get('product_update_email_subject', 'New update available: {product_name}');
$productUpdateEmailBody = SettingsService::get('product_update_email_body', "Hi {name},\n\nGreat news! A new update is available for {product_name}.\n\nVersion: {version}\n\n{changelog}\n\nLog in to your account to download the latest version:\n{account_url}\n\nThank you!");
$storeCurrency = SettingsService::get('store_currency', 'USD');
$storeCurrencySymbol = SettingsService::get('store_currency_symbol', '$');
$cartSidebarEnabled = SettingsService::get('cart_sidebar_enabled', '1') === '1';
$cartIconItemsOnly = SettingsService::get('cart_icon_items_only', '0') === '1';
$minibarSticky = SettingsService::get('minibar_sticky', '0') === '1';
$refundWindowDigitalDays = (int)SettingsService::get('refund_window_digital_days', '30');
$refundWindowPhysicalDays = (int)SettingsService::get('refund_window_physical_days', '30');
$customCursorEnabled = SettingsService::get('custom_cursor_enabled', '1') === '1';
$videoThumbnailMode = SettingsService::get('video_thumbnail_mode', 'auto');
if (!in_array($videoThumbnailMode, ['auto', 'client', 'server', 'off'], true)) {
    $videoThumbnailMode = 'auto';
}
$easyMediaAiZipDefault = ContentRenderer::getDefaultEasyMediaAiZipDownloadPath();
$easyMediaAiZipUrl = ContentRenderer::normalizeEasyMediaZipDownloadPath((string)SettingsService::get('easy_media_ai_zip_url', $easyMediaAiZipDefault));
$easyMediaAiHelpImageSettings = [];
for ($helpImageIndex = 1; $helpImageIndex <= 12; $helpImageIndex++) {
    $helpImageKey = 'easy_media_ai_help_image_' . $helpImageIndex;
    $easyMediaAiHelpImageSettings[$helpImageIndex] = ContentRenderer::normalizeEasyMediaHelpImagePath((string)SettingsService::get($helpImageKey, ''));
}
$searchShowMini = SettingsService::get('search_show_mini', '1') === '1';
$searchShowTop = SettingsService::get('search_show_top', '0') === '1';
$searchShowSidebar = SettingsService::get('search_show_sidebar', '0') === '1';
$searchShowFooter = SettingsService::get('search_show_footer', '0') === '1';
$accountShowMini = SettingsService::get('account_show_mini', '0') === '1';
$accountShowTop = SettingsService::get('account_show_top', '1') === '1';
$accountShowSidebar = SettingsService::get('account_show_sidebar', '1') === '1';
$accountShowFooter = SettingsService::get('account_show_footer', '0') === '1';
$cartShowMini = SettingsService::get('cart_show_mini', '0') === '1';
$cartShowTop = SettingsService::get('cart_show_top', '1') === '1';
$cartShowSidebar = SettingsService::get('cart_show_sidebar', '1') === '1';
$cartShowFooter = SettingsService::get('cart_show_footer', '0') === '1';
$gdprIpAnonymization = SettingsService::get('gdpr_ip_anonymization', '1') === '1';
$gdprAutoPurge = SettingsService::get('gdpr_auto_purge', '1') === '1';
$gdprRetentionLogs = (int)SettingsService::get('gdpr_retention_logs_days', '30');
$gdprRetentionVisitors = (int)SettingsService::get('gdpr_retention_visitors_days', '30');
$gdprRetentionMessages = (int)SettingsService::get('gdpr_retention_messages_days', '365');
$gdprRetentionContact = (int)SettingsService::get('gdpr_retention_contact_days', '365');
$gdprRetentionCarts = (int)SettingsService::get('gdpr_retention_carts_days', '7');
$gdprRetentionRateLimits = (int)SettingsService::get('gdpr_retention_rate_limits_days', '7');
$gdprRetentionGeoCache = (int)SettingsService::get('gdpr_retention_geo_cache_days', '7');
$notificationLocation = SettingsService::get('notification_location', 'mini');
$customFonts = json_decode(SettingsService::get('custom_fonts', '[]'), true);
if (!is_array($customFonts)) {
    $customFonts = [];
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header">
        <h1>Admin Settings</h1>
        <p>Configure admin panel and technical settings</p>
    </div>

    <?php if ($message): ?>
    <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
    <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <div class="alert alert-info">
        <strong>Note:</strong> For site name, favicon, and SEO settings, use the <a href="/admin/theme-editor.php?section=general">Theme Editor → Site Settings</a>.
    </div>

    <form method="POST" action="" enctype="multipart/form-data">
        <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
        <input type="hidden" name="action" value="general">
        <button type="submit" name="save_settings" value="1" class="visually-hidden">Save Settings</button>
        <div class="card">
            <div class="card-header">
                <h3>Site Options</h3>
            </div>
            <div class="card-body">
                <div class="form-group">
                    <label for="store_currency">Store Currency <span class="info-tooltip" data-tooltip="Sets the default currency used for pricing and totals.">ⓘ</span></label>
                    <select id="store_currency" name="store_currency" class="form-control">
                        <?php
                        $currencyOptions = [
                            'USD' => 'USD ($)',
                            'EUR' => 'EUR (€)',
                            'GBP' => 'GBP (£)',
                            'JPY' => 'JPY (¥)',
                            'AUD' => 'AUD (A$)',
                            'CAD' => 'CAD (C$)',
                            'CHF' => 'CHF',
                            'SEK' => 'SEK',
                            'NOK' => 'NOK',
                            'DKK' => 'DKK',
                            'PLN' => 'PLN',
                            'BRL' => 'BRL (R$)',
                            'MXN' => 'MXN (MX$)',
                            'INR' => 'INR (₹)',
                            'KRW' => 'KRW (₩)',
                            'CNY' => 'CNY (¥)'
                        ];
foreach ($currencyOptions as $value => $label) {
    ?>
                            <option value="<?php echo e($value); ?>" <?php echo $storeCurrency === $value ? 'selected' : ''; ?>><?php echo e($label); ?></option>
                        <?php } ?>
                    </select>
                    <small class="text-muted">Used for product pricing display.</small>
                </div>
                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="cart_sidebar_enabled" <?php echo $cartSidebarEnabled ? 'checked' : ''; ?>>
                        <span>Enable Cart Sidebar</span>
                        <span class="info-tooltip" data-tooltip="Shows a sliding cart panel instead of redirecting to the cart page.">ⓘ</span>
                    </label>
                    <small class="text-muted form-help">When disabled, Add to Cart redirects to the cart page.</small>
                </div>
                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="minibar_sticky" <?php echo $minibarSticky ? 'checked' : ''; ?>>
                        <span>Sticky Top Bar</span>
                        <span class="info-tooltip" data-tooltip="When enabled, the top mini bar stays visible at the top of the page while scrolling. When disabled, it hides when the user scrolls down.">ⓘ</span>
                    </label>
                    <small class="text-muted form-help">Keep the top mini bar always visible on scroll.</small>
                </div>
                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="custom_cursor_enabled" <?php echo $customCursorEnabled ? 'checked' : ''; ?>>
                        <span>Enable Custom Cursor Theme</span>
                        <span class="info-tooltip" data-tooltip="Applies the neon custom cursor styling on the public site.">ⓘ</span>
                    </label>
                    <small class="text-muted form-help">Toggle custom cursors for the public site.</small>
                </div>
                <div class="form-group">
                    <label for="video_thumbnail_mode">Video Thumbnail Generation <span class="info-tooltip" data-tooltip="Controls how video thumbnails are generated for the media library and picker. Server mode requires FFmpeg; if it is not available, thumbnails may fall back to browser mode or be skipped.">ⓘ</span></label>
                    <select id="video_thumbnail_mode" name="video_thumbnail_mode" class="form-control">
                        <option value="auto" <?php echo $videoThumbnailMode === 'auto' ? 'selected' : ''; ?>>Auto (browser + server fallback)</option>
                        <option value="client" <?php echo $videoThumbnailMode === 'client' ? 'selected' : ''; ?>>Browser only (no server FFmpeg)</option>
                        <option value="server" <?php echo $videoThumbnailMode === 'server' ? 'selected' : ''; ?>>Server only (FFmpeg)</option>
                        <option value="off" <?php echo $videoThumbnailMode === 'off' ? 'selected' : ''; ?>>Off (no thumbnails)</option>
                    </select>
                    <small class="text-muted form-help">Large videos can slow uploads when server thumbnails are enabled.</small>
                </div>
                <div class="form-group">
                    <label for="easy_media_ai_zip_url">Easy Media AI ZIP Download <span class="info-tooltip" data-tooltip="Select the ZIP file exposed by the Download button inside Easy Media settings.">ⓘ</span></label>
                    <input type="text" id="easy_media_ai_zip_url" name="easy_media_ai_zip_url" class="form-control" value="<?php echo e($easyMediaAiZipUrl); ?>" placeholder="/admin-file/media-ai.zip" data-media-picker="file">
                    <small class="text-muted form-help">Allowed values: local <code>/admin-file/</code> or <code>/site-file/</code> ZIP paths only. Leave empty to disable the button.</small>
                    <small class="text-muted form-help">Default preloaded file: <code>/site-file/AI_TAG_SERVER_V1.zip</code> (used automatically when this setting is not yet saved).</small>
                </div>
                <div class="form-group">
                    <div class="settings-label">Easy Media Help Images <span class="info-tooltip" data-tooltip="Upload or select screenshots shown in the help gallery popup for end users.">ⓘ</span></div>
                    <small class="text-muted form-help">Add up to 12 images. Empty slots are ignored.</small>
                    <div class="form-row">
                        <?php for ($helpImageIndex = 1; $helpImageIndex <= 12; $helpImageIndex++): ?>
                        <div class="form-group">
                            <label for="easy_media_ai_help_image_<?php echo $helpImageIndex; ?>">Help Image <?php echo $helpImageIndex; ?></label>
                            <input type="text" id="easy_media_ai_help_image_<?php echo $helpImageIndex; ?>" name="easy_media_ai_help_image_<?php echo $helpImageIndex; ?>" class="form-control" value="<?php echo e($easyMediaAiHelpImageSettings[$helpImageIndex] ?? ''); ?>" placeholder="/admin-file/help-<?php echo $helpImageIndex; ?>.png" data-media-picker="image">
                        </div>
                        <?php endfor; ?>
                    </div>
                </div>
                <div class="form-group">
                    <div class="form-label">Search Bar Locations <span class="info-tooltip" data-tooltip="Choose where the search bar appears on the public site.">ⓘ</span></div>
                    <div class="form-check">
                        <input type="checkbox" name="search_show_mini" <?php echo $searchShowMini ? 'checked' : ''; ?>>
                        <span>Mini Top Bar</span>
                    </div>
                    <div class="form-check">
                        <input type="checkbox" name="search_show_top" <?php echo $searchShowTop ? 'checked' : ''; ?>>
                        <span>Top Header</span>
                    </div>
                    <div class="form-check">
                        <input type="checkbox" name="search_show_sidebar" <?php echo $searchShowSidebar ? 'checked' : ''; ?>>
                        <span>Sidebar</span>
                    </div>
                    <div class="form-check">
                        <input type="checkbox" name="search_show_footer" <?php echo $searchShowFooter ? 'checked' : ''; ?>>
                        <span>Footer</span>
                    </div>
                </div>
                <div class="form-group">
                    <div class="form-label">Account Icon Locations <span class="info-tooltip" data-tooltip="Choose where the account icon appears across the public layout.">ⓘ</span></div>
                    <div class="form-check">
                        <input type="checkbox" name="account_show_mini" <?php echo $accountShowMini ? 'checked' : ''; ?>>
                        <span>Mini Top Bar</span>
                    </div>
                    <div class="form-check">
                        <input type="checkbox" name="account_show_top" <?php echo $accountShowTop ? 'checked' : ''; ?>>
                        <span>Top Header</span>
                    </div>
                    <div class="form-check">
                        <input type="checkbox" name="account_show_sidebar" <?php echo $accountShowSidebar ? 'checked' : ''; ?>>
                        <span>Sidebar</span>
                    </div>
                    <div class="form-check">
                        <input type="checkbox" name="account_show_footer" <?php echo $accountShowFooter ? 'checked' : ''; ?>>
                        <span>Footer</span>
                    </div>
                </div>
                <div class="form-group">
                    <div class="form-label">Cart Icon Locations <span class="info-tooltip" data-tooltip="Choose where the cart icon appears across the public layout.">ⓘ</span></div>
                    <div class="form-check">
                        <input type="checkbox" name="cart_show_mini" <?php echo $cartShowMini ? 'checked' : ''; ?>>
                        <span>Mini Top Bar</span>
                    </div>
                    <div class="form-check">
                        <input type="checkbox" name="cart_show_top" <?php echo $cartShowTop ? 'checked' : ''; ?>>
                        <span>Top Header</span>
                    </div>
                    <div class="form-check">
                        <input type="checkbox" name="cart_show_sidebar" <?php echo $cartShowSidebar ? 'checked' : ''; ?>>
                        <span>Sidebar</span>
                    </div>
                    <div class="form-check">
                        <input type="checkbox" name="cart_show_footer" <?php echo $cartShowFooter ? 'checked' : ''; ?>>
                        <span>Footer</span>
                    </div>
                </div>
                <div class="form-group">
                    <div class="form-label">Notifications Icon Location <span class="info-tooltip" data-tooltip="Select the location for the notifications icon (only visible to logged-in users).">ⓘ</span></div>
                    <div class="form-check">
                        <input type="radio" name="notification_location" value="mini" <?php echo $notificationLocation === 'mini' ? 'checked' : ''; ?>>
                        <span>Mini Top Bar</span>
                    </div>
                    <div class="form-check">
                        <input type="radio" name="notification_location" value="top" <?php echo $notificationLocation === 'top' ? 'checked' : ''; ?>>
                        <span>Top Header</span>
                    </div>
                    <div class="form-check">
                        <input type="radio" name="notification_location" value="sidebar" <?php echo $notificationLocation === 'sidebar' ? 'checked' : ''; ?>>
                        <span>Sidebar</span>
                    </div>
                </div>
                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="cart_icon_items_only" <?php echo $cartIconItemsOnly ? 'checked' : ''; ?>>
                        <span>Show Cart Icon Only When Items Exist</span>
                        <span class="info-tooltip" data-tooltip="Hide the cart icon until at least one item is added.">ⓘ</span>
                    </label>
                    <small class="text-muted form-help">When enabled, the cart icon hides until there are items in the cart.</small>
                </div>
                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="maintenance_mode" <?php echo $maintenanceMode ? 'checked' : ''; ?>>
                        <span>Maintenance Mode</span>
                        <span class="info-tooltip" data-tooltip="Shows a maintenance page to visitors while admins can still access the site.">ⓘ</span>
                    </label>
                    <small class="text-muted form-help">When enabled, visitors will see a maintenance page. Admins can still access the site.</small>
                </div>

                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="allow_registration" <?php echo $allowRegistration ? 'checked' : ''; ?>>
                        <span>Allow User Login</span>
                        <span class="info-tooltip" data-tooltip="When disabled, only customers with paid purchases can request login links.">ⓘ</span>
                    </label>
                    <small class="text-muted form-help">Allow visitors to request a login link and create accounts if needed.</small>
                </div>
                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="contact_enabled" <?php echo $contactEnabled ? 'checked' : ''; ?>>
                        <span>Enable Contact Page</span>
                        <span class="info-tooltip" data-tooltip="Shows or hides the built-in contact page.">ⓘ</span>
                    </label>
                    <small class="text-muted form-help">Show or hide the built-in contact page.</small>
                </div>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <h3>Legal Pages &amp; Cookie Notice</h3>
            </div>
            <div class="card-body">
                <p class="text-muted form-help">Enable or disable individual legal pages and the cookie consent banner. Disabled pages return a 404 and their footer links are hidden.</p>
                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="page_terms_enabled" <?php echo $pageTermsEnabled ? 'checked' : ''; ?>>
                        <span>Terms of Service</span>
                    </label>
                </div>
                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="page_privacy_enabled" <?php echo $pagePrivacyEnabled ? 'checked' : ''; ?>>
                        <span>Privacy Policy</span>
                    </label>
                </div>
                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="page_refund_enabled" <?php echo $pageRefundEnabled ? 'checked' : ''; ?>>
                        <span>Refund Policy</span>
                    </label>
                </div>
                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="page_shipping_enabled" <?php echo $pageShippingEnabled ? 'checked' : ''; ?>>
                        <span>Shipping Policy</span>
                    </label>
                </div>
                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="page_legal_notice_enabled" <?php echo $pageLegalNoticeEnabled ? 'checked' : ''; ?>>
                        <span>Legal Notice</span>
                    </label>
                </div>
                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="page_dmca_enabled" <?php echo $pageDmcaEnabled ? 'checked' : ''; ?>>
                        <span>DMCA</span>
                    </label>
                </div>
                <hr>
                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="cookie_notice_enabled" <?php echo $cookieNoticeEnabled ? 'checked' : ''; ?>>
                        <span>Cookie Consent Banner</span>
                        <span class="info-tooltip" data-tooltip="Shows the cookie consent popup and settings modal to visitors.">ⓘ</span>
                    </label>
                    <small class="text-muted form-help">When disabled, the cookie banner and cookie settings link are hidden from the site.</small>
                </div>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <h3>Legal / Impressum Identity</h3>
            </div>
            <div class="card-body">
                <p class="text-muted form-help">These values are used by legal placeholders and should be kept accurate for GDPR/Impressum compliance.</p>

                <div class="form-group">
                    <label for="site_name">Site Name (Legal) <span class="info-tooltip" data-tooltip="Used for legal page placeholders and email templates.">ⓘ</span></label>
                    <input type="text" id="site_name" name="site_name" class="form-control" value="<?php echo e($siteNameSetting); ?>" placeholder="My Website">
                    <small class="text-muted">Placeholder: {SITE_NAME}</small>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="legal_entity_name">Legal Entity Name <span class="info-tooltip" data-tooltip="Full name of the operator/company shown in legal notice pages.">ⓘ</span></label>
                        <input type="text" id="legal_entity_name" name="legal_entity_name" class="form-control" value="<?php echo e($legalEntityName); ?>" placeholder="Example GmbH">
                        <small class="text-muted">Placeholder: {LEGAL_ENTITY_NAME}</small>
                    </div>
                    <div class="form-group">
                        <label for="legal_legal_form">Legal Form <span class="info-tooltip" data-tooltip="Optional legal form, for example GmbH, UG, LLC, Ltd.">ⓘ</span></label>
                        <input type="text" id="legal_legal_form" name="legal_legal_form" class="form-control" value="<?php echo e($legalLegalForm); ?>" placeholder="GmbH">
                        <small class="text-muted">Placeholder: {LEGAL_LEGAL_FORM}</small>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="legal_representative">Authorized Representative <span class="info-tooltip" data-tooltip="Managing director / legal representative.">ⓘ</span></label>
                        <input type="text" id="legal_representative" name="legal_representative" class="form-control" value="<?php echo e($legalRepresentative); ?>" placeholder="Max Mustermann">
                        <small class="text-muted">Placeholder: {LEGAL_REPRESENTATIVE}</small>
                    </div>
                    <div class="form-group">
                        <label for="contact_name">Public Contact Name <span class="info-tooltip" data-tooltip="Fallback display name for {CONTACT_NAME} placeholders.">ⓘ</span></label>
                        <input type="text" id="contact_name" name="contact_name" class="form-control" value="<?php echo e($contactName); ?>" placeholder="Support Team">
                        <small class="text-muted">Placeholder: {CONTACT_NAME}</small>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="legal_street">Street + No. <span class="info-tooltip" data-tooltip="Business address for legal notice pages.">ⓘ</span></label>
                        <input type="text" id="legal_street" name="legal_street" class="form-control" value="<?php echo e($legalStreet); ?>" placeholder="Musterstraße 1">
                        <small class="text-muted">Placeholder: {LEGAL_STREET}</small>
                    </div>
                    <div class="form-group">
                        <label for="legal_postal_code">Postal Code <span class="info-tooltip" data-tooltip="ZIP or postal code for legal address.">ⓘ</span></label>
                        <input type="text" id="legal_postal_code" name="legal_postal_code" class="form-control" value="<?php echo e($legalPostalCode); ?>" placeholder="10115">
                        <small class="text-muted">Placeholder: {LEGAL_POSTAL_CODE}</small>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="legal_city">City <span class="info-tooltip" data-tooltip="City used in legal address output.">ⓘ</span></label>
                        <input type="text" id="legal_city" name="legal_city" class="form-control" value="<?php echo e($legalCity); ?>" placeholder="Berlin">
                        <small class="text-muted">Placeholder: {LEGAL_CITY}</small>
                    </div>
                    <div class="form-group">
                        <label for="legal_country">Country <span class="info-tooltip" data-tooltip="Country used in legal address output.">ⓘ</span></label>
                        <input type="text" id="legal_country" name="legal_country" class="form-control" value="<?php echo e($legalCountry); ?>" placeholder="Germany">
                        <small class="text-muted">Placeholder: {LEGAL_COUNTRY}</small>
                    </div>
                </div>

                <div class="form-group">
                    <label for="contact_address">Public Contact Address <span class="info-tooltip" data-tooltip="Used by {CONTACT_ADDRESS}. Line breaks are preserved.">ⓘ</span></label>
                    <textarea id="contact_address" name="contact_address" class="form-control" rows="3" placeholder="123 Main St&#10;City, State ZIP"><?php echo e($contactAddress); ?></textarea>
                    <small class="text-muted">Placeholder: {CONTACT_ADDRESS}</small>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="legal_register_court">Register Court / Registry <span class="info-tooltip" data-tooltip="Commercial register court or registry authority.">ⓘ</span></label>
                        <input type="text" id="legal_register_court" name="legal_register_court" class="form-control" value="<?php echo e($legalRegisterCourt); ?>" placeholder="Amtsgericht Berlin-Charlottenburg">
                        <small class="text-muted">Placeholder: {LEGAL_REGISTER_COURT}</small>
                    </div>
                    <div class="form-group">
                        <label for="legal_register_number">Register Number <span class="info-tooltip" data-tooltip="Commercial register number (e.g., HRB ...).">ⓘ</span></label>
                        <input type="text" id="legal_register_number" name="legal_register_number" class="form-control" value="<?php echo e($legalRegisterNumber); ?>" placeholder="HRB 123456">
                        <small class="text-muted">Placeholder: {LEGAL_REGISTER_NUMBER}</small>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="legal_vat_id">VAT ID <span class="info-tooltip" data-tooltip="USt-IdNr / VAT identification number.">ⓘ</span></label>
                        <input type="text" id="legal_vat_id" name="legal_vat_id" class="form-control" value="<?php echo e($legalVatId); ?>" placeholder="DE123456789">
                        <small class="text-muted">Placeholder: {LEGAL_VAT_ID}</small>
                    </div>
                    <div class="form-group">
                        <label for="legal_business_id">Business ID <span class="info-tooltip" data-tooltip="Optional local business/tax ID.">ⓘ</span></label>
                        <input type="text" id="legal_business_id" name="legal_business_id" class="form-control" value="<?php echo e($legalBusinessId); ?>" placeholder="123/456/7890">
                        <small class="text-muted">Placeholder: {LEGAL_BUSINESS_ID}</small>
                    </div>
                </div>

                <hr>
                <h4>Editorial Responsibility (§18 MStV)</h4>
                <div class="form-row">
                    <div class="form-group">
                        <label for="legal_editor_name">Responsible Editor</label>
                        <input type="text" id="legal_editor_name" name="legal_editor_name" class="form-control" value="<?php echo e($legalEditorName); ?>" placeholder="Max Mustermann">
                        <small class="text-muted">Placeholder: {LEGAL_EDITOR_NAME}</small>
                    </div>
                </div>
                <div class="form-group">
                    <label for="legal_editor_address">Editor Address</label>
                    <textarea id="legal_editor_address" name="legal_editor_address" class="form-control" rows="3" placeholder="Musterstraße 1&#10;10115 Berlin"><?php echo e($legalEditorAddress); ?></textarea>
                    <small class="text-muted">Placeholder: {LEGAL_EDITOR_ADDRESS}</small>
                </div>

                <hr>
                <h4>Regulated Profession (optional)</h4>
                <div class="form-row">
                    <div class="form-group">
                        <label for="legal_profession_name">Profession</label>
                        <input type="text" id="legal_profession_name" name="legal_profession_name" class="form-control" value="<?php echo e($legalProfessionName); ?>" placeholder="Attorney-at-law">
                        <small class="text-muted">Placeholder: {LEGAL_PROFESSION_NAME}</small>
                    </div>
                    <div class="form-group">
                        <label for="legal_professional_title">Professional Title</label>
                        <input type="text" id="legal_professional_title" name="legal_professional_title" class="form-control" value="<?php echo e($legalProfessionalTitle); ?>" placeholder="Rechtsanwalt">
                        <small class="text-muted">Placeholder: {LEGAL_PROFESSIONAL_TITLE}</small>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="legal_professional_title_country">Title Awarded In</label>
                        <input type="text" id="legal_professional_title_country" name="legal_professional_title_country" class="form-control" value="<?php echo e($legalProfessionalTitleCountry); ?>" placeholder="Germany">
                        <small class="text-muted">Placeholder: {LEGAL_PROFESSIONAL_TITLE_COUNTRY}</small>
                    </div>
                    <div class="form-group">
                        <label for="legal_chamber">Chamber / Professional Body</label>
                        <input type="text" id="legal_chamber" name="legal_chamber" class="form-control" value="<?php echo e($legalChamber); ?>" placeholder="Berlin Bar Association">
                        <small class="text-muted">Placeholder: {LEGAL_CHAMBER}</small>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="legal_professional_rules">Professional Rules</label>
                        <input type="text" id="legal_professional_rules" name="legal_professional_rules" class="form-control" value="<?php echo e($legalProfessionalRules); ?>" placeholder="BRAO, BORA, FAO">
                        <small class="text-muted">Placeholder: {LEGAL_PROFESSIONAL_RULES}</small>
                    </div>
                    <div class="form-group">
                        <label for="legal_professional_rules_url">Professional Rules URL</label>
                        <input type="url" id="legal_professional_rules_url" name="legal_professional_rules_url" class="form-control" value="<?php echo e($legalProfessionalRulesUrl); ?>" placeholder="https://www.brak.de/">
                        <small class="text-muted">Placeholder: {LEGAL_PROFESSIONAL_RULES_URL}</small>
                    </div>
                </div>

                <hr>
                <h4>Supervisory Authority &amp; Disputes</h4>
                <div class="form-group">
                    <label for="legal_supervisory_authority">Supervisory Authority</label>
                    <input type="text" id="legal_supervisory_authority" name="legal_supervisory_authority" class="form-control" value="<?php echo e($legalSupervisoryAuthority); ?>" placeholder="State Media Authority ...">
                    <small class="text-muted">Placeholder: {LEGAL_SUPERVISORY_AUTHORITY}</small>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="legal_odr_url">EU ODR Platform URL</label>
                        <input type="url" id="legal_odr_url" name="legal_odr_url" class="form-control" value="<?php echo e($legalOdrUrl); ?>" placeholder="https://ec.europa.eu/consumers/odr">
                        <small class="text-muted">Placeholders: {LEGAL_ODR_URL}, {LEGAL_ODR_LINK}</small>
                    </div>
                </div>
                <div class="form-group">
                    <label for="legal_dispute_participation">Dispute Participation Statement</label>
                    <textarea id="legal_dispute_participation" name="legal_dispute_participation" class="form-control" rows="2" placeholder="We are not willing or obliged ..."><?php echo e($legalDisputeParticipation); ?></textarea>
                    <small class="text-muted">Placeholder: {LEGAL_DISPUTE_PARTICIPATION}</small>
                </div>

                <hr>
                <h4>Hosting Provider</h4>
                <div class="form-row">
                    <div class="form-group">
                        <label for="legal_hosting_provider_name">Hosting Provider Name</label>
                        <input type="text" id="legal_hosting_provider_name" name="legal_hosting_provider_name" class="form-control" value="<?php echo e($legalHostingProviderName); ?>" placeholder="Hosting Provider Inc.">
                        <small class="text-muted">Placeholder: {LEGAL_HOSTING_PROVIDER_NAME}</small>
                    </div>
                </div>
                <div class="form-group">
                    <label for="legal_hosting_provider_address">Hosting Provider Address</label>
                    <textarea id="legal_hosting_provider_address" name="legal_hosting_provider_address" class="form-control" rows="3" placeholder="Provider Street 1&#10;City, Country"><?php echo e($legalHostingProviderAddress); ?></textarea>
                    <small class="text-muted">Placeholder: {LEGAL_HOSTING_PROVIDER_ADDRESS}</small>
                </div>

                <div class="text-muted">Also available: {LEGAL_FULL_ADDRESS}, {CONTACT_EMAIL}, {CONTACT_PHONE}.</div>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <h3>Contact Form Settings</h3>
            </div>
            <div class="card-body">
                <div class="form-row">
                    <div class="form-group">
                        <label for="contact_email">Contact Email <span class="info-tooltip" data-tooltip="Email address that receives contact form submissions.">ⓘ</span></label>
                        <input type="email" id="contact_email" name="contact_email" class="form-control" value="<?php echo e($contactEmail); ?>" placeholder="support@example.com">
                        <small class="text-muted">Placeholders: {CONTACT_EMAIL}, {LEGAL_CONTACT_EMAIL}</small>
                    </div>
                    <div class="form-group">
                        <label for="contact_phone">Contact Phone <span class="info-tooltip" data-tooltip="Used on legal pages and contact templates.">ⓘ</span></label>
                        <input type="text" id="contact_phone" name="contact_phone" class="form-control" value="<?php echo e($contactPhone); ?>" placeholder="+1 (555) 123-4567">
                        <small class="text-muted">Placeholders: {CONTACT_PHONE}, {LEGAL_CONTACT_PHONE}</small>
                    </div>
                </div>
                <div class="form-group">
                    <label for="contact_subject">Contact Email Subject <span class="info-tooltip" data-tooltip="Fallback subject used when a visitor leaves the subject empty.">ⓘ</span></label>
                    <input type="text" id="contact_subject" name="contact_subject" class="form-control" value="<?php echo e($contactSubject); ?>" placeholder="Contact Form Message">
                    <small class="text-muted">Default subject used when user leaves subject empty.</small>
                </div>
                <div class="form-group">
                    <button type="submit" class="btn btn-danger" name="clear_contact_uploads" value="1">Clear Contact Uploads</button>
                    <div class="text-muted mt-1">Deletes all files in data/contact_uploads.</div>
                </div>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <h3>GDPR & Privacy</h3>
            </div>
            <div class="card-body">
                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="gdpr_ip_anonymization" <?php echo $gdprIpAnonymization ? 'checked' : ''; ?>>
                        <span>Anonymize Stored IP Addresses</span>
                        <span class="info-tooltip" data-tooltip="Masks IPs for stored logs, visitors, and messages.">ⓘ</span>
                    </label>
                </div>
                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="gdpr_auto_purge" <?php echo $gdprAutoPurge ? 'checked' : ''; ?>>
                        <span>Auto-Purge Old Data</span>
                        <span class="info-tooltip" data-tooltip="Runs daily cleanup based on the retention days below.">ⓘ</span>
                    </label>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="gdpr_retention_logs_days">Logs Retention (days)</label>
                        <input type="number" id="gdpr_retention_logs_days" name="gdpr_retention_logs_days" class="form-control" value="<?php echo (int)$gdprRetentionLogs; ?>" min="1">
                    </div>
                    <div class="form-group">
                        <label for="gdpr_retention_visitors_days">Visitors Retention (days)</label>
                        <input type="number" id="gdpr_retention_visitors_days" name="gdpr_retention_visitors_days" class="form-control" value="<?php echo (int)$gdprRetentionVisitors; ?>" min="1">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="gdpr_retention_messages_days">User Messages Retention (days)</label>
                        <input type="number" id="gdpr_retention_messages_days" name="gdpr_retention_messages_days" class="form-control" value="<?php echo (int)$gdprRetentionMessages; ?>" min="1">
                    </div>
                    <div class="form-group">
                        <label for="gdpr_retention_contact_days">Contact Messages Retention (days)</label>
                        <input type="number" id="gdpr_retention_contact_days" name="gdpr_retention_contact_days" class="form-control" value="<?php echo (int)$gdprRetentionContact; ?>" min="1">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="gdpr_retention_carts_days">Cart Sessions Retention (days)</label>
                        <input type="number" id="gdpr_retention_carts_days" name="gdpr_retention_carts_days" class="form-control" value="<?php echo (int)$gdprRetentionCarts; ?>" min="1">
                    </div>
                    <div class="form-group">
                        <label for="gdpr_retention_rate_limits_days">Rate Limits Retention (days)</label>
                        <input type="number" id="gdpr_retention_rate_limits_days" name="gdpr_retention_rate_limits_days" class="form-control" value="<?php echo (int)$gdprRetentionRateLimits; ?>" min="1">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="gdpr_retention_geo_cache_days">Geo Cache Retention (days)</label>
                        <input type="number" id="gdpr_retention_geo_cache_days" name="gdpr_retention_geo_cache_days" class="form-control" value="<?php echo (int)$gdprRetentionGeoCache; ?>" min="1">
                    </div>
                </div>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <h3>GDPR Requests</h3>
            </div>
            <div class="card-body">
                <p class="text-muted">Process user export or deletion requests by email or user ID.</p>
                <div class="form-row">
                    <div class="form-group">
                        <label for="gdpr_user_identifier">User Email or ID</label>
                        <input type="text" id="gdpr_user_identifier" name="gdpr_user_identifier" class="form-control" placeholder="user@example.com or 123">
                    </div>
                    <div class="form-group">
                        <label for="gdpr_request_action">Action</label>
                        <select id="gdpr_request_action" name="gdpr_request_action" class="form-control">
                            <option value="export">Export Data</option>
                            <option value="delete">Delete Data</option>
                        </select>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="gdpr_delete_confirm">Type DELETE to confirm deletion</label>
                        <input type="text" id="gdpr_delete_confirm" name="gdpr_delete_confirm" class="form-control" placeholder="DELETE">
                    </div>
                </div>
                <button type="submit" class="btn btn-secondary" name="action" value="gdpr_user_tools">Run GDPR Action</button>
                <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <h3>Data Breach Alerts</h3>
            </div>
            <div class="card-body">
                <p class="text-muted">Log and notify about security incidents. Alerts are stored in the logs.</p>
                <div class="form-group">
                    <label for="breach_title">Incident Title</label>
                    <input type="text" id="breach_title" name="breach_title" class="form-control" placeholder="Example: Unauthorized access detected">
                </div>
                <div class="form-group">
                    <label for="breach_summary">Summary</label>
                    <textarea id="breach_summary" name="breach_summary" class="form-control" rows="4" placeholder="Describe what happened and the immediate response."></textarea>
                </div>
                <div class="form-group">
                    <label for="breach_impact">Impact</label>
                    <input type="text" id="breach_impact" name="breach_impact" class="form-control" placeholder="Affected systems, data types, or user groups">
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="breach_severity">Severity</label>
                        <select id="breach_severity" name="breach_severity" class="form-control">
                            <option value="low">Low</option>
                            <option value="medium" selected>Medium</option>
                            <option value="high">High</option>
                            <option value="critical">Critical</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="breach_records">Estimated Records</label>
                        <input type="number" id="breach_records" name="breach_records" class="form-control" min="0" placeholder="0">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="breach_detected_at">Detected At</label>
                        <input type="datetime-local" id="breach_detected_at" name="breach_detected_at" class="form-control">
                    </div>
                    <div class="form-group">
                        <label for="breach_resolved_at">Resolved At</label>
                        <input type="datetime-local" id="breach_resolved_at" name="breach_resolved_at" class="form-control">
                    </div>
                </div>
                <label class="form-check">
                    <input type="checkbox" name="breach_notify" checked>
                    <span>Email alert to contact address</span>
                </label>
                <div class="form-group mt-2">
                    <button type="submit" class="btn btn-secondary" name="action" value="gdpr_breach_alert">Log Breach Alert</button>
                    <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                </div>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <h3>SMTP Environment</h3>
            </div>
            <div class="card-body">
                <p class="text-muted mb-2">Configure the SMTP server to allow the contact form to send emails.</p>
                <div class="form-row">
                    <div class="form-group">
                        <label for="smtp_host">SMTP Host <span class="info-tooltip" data-tooltip="Mail server hostname (example: smtp.yourdomain.com).">ⓘ</span></label>
                        <input type="text" id="smtp_host" name="smtp_host" class="form-control" value="<?php echo e($smtpHost); ?>" placeholder="smtp.example.com">
                    </div>
                    <div class="form-group">
                        <label for="smtp_port">SMTP Port <span class="info-tooltip" data-tooltip="Typical ports: 587 (TLS) or 465 (SSL).">ⓘ</span></label>
                        <input type="number" id="smtp_port" name="smtp_port" class="form-control" value="<?php echo e($smtpPort); ?>" placeholder="587">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="smtp_user">SMTP Username <span class="info-tooltip" data-tooltip="Mailbox username used to authenticate with SMTP.">ⓘ</span></label>
                        <input type="text" id="smtp_user" name="smtp_user" class="form-control" value="<?php echo e($smtpUser); ?>">
                    </div>
                    <div class="form-group">
                        <label for="smtp_pass">SMTP Password <span class="info-tooltip" data-tooltip="Password or app-specific password for the SMTP account.">ⓘ</span></label>
                        <input type="password" id="smtp_pass" name="smtp_pass" class="form-control" value="<?php echo e($smtpPass); ?>">
                    </div>
                </div>
                <div class="form-group">
                    <label for="smtp_secure">Security <span class="info-tooltip" data-tooltip="TLS is recommended for most providers.">ⓘ</span></label>
                    <select id="smtp_secure" name="smtp_secure" class="form-control">
                        <option value="none" <?php echo $smtpSecure === 'none' ? 'selected' : ''; ?>>None</option>
                        <option value="tls" <?php echo $smtpSecure === 'tls' ? 'selected' : ''; ?>>TLS</option>
                        <option value="ssl" <?php echo $smtpSecure === 'ssl' ? 'selected' : ''; ?>>SSL</option>
                    </select>
                </div>
                <div class="form-group">
                    <button type="submit" class="btn btn-secondary" name="send_test_email" value="1">Send Test Email</button>
                </div>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <div class="card-header-row">
                    <h3>Stripe Settings</h3>
                    <button type="button" class="info-icon-btn" data-modal="stripe-webhook-info" aria-label="Stripe webhook setup info">
                        <svg viewBox="0 0 24 24" aria-hidden="true">
                            <circle cx="12" cy="12" r="10"></circle>
                            <path d="M12 10v6"></path>
                            <circle cx="12" cy="7" r="1"></circle>
                        </svg>
                    </button>
                </div>
            </div>
            <div class="card-body">
                <div class="form-group">
                    <label for="stripe_publishable_key">Stripe Publishable Key <span class="info-tooltip" data-tooltip="Used by the frontend checkout to start a Stripe session.">ⓘ</span></label>
                    <input type="text" id="stripe_publishable_key" name="stripe_publishable_key" class="form-control" value="<?php echo e($stripePublishableKey); ?>" placeholder="pk_live_...">
                </div>
                <div class="form-group">
                    <label for="stripe_secret_key">Stripe Secret Key <span class="info-tooltip" data-tooltip="Server-side key for creating Stripe sessions.">ⓘ</span></label>
                    <input type="password" id="stripe_secret_key" name="stripe_secret_key" class="form-control" value="<?php echo e($stripeSecretKey); ?>" placeholder="sk_live_...">
                </div>
                <div class="form-group">
                    <label for="stripe_webhook_secret">Stripe Webhook Secret <span class="info-tooltip" data-tooltip="Verifies webhook events sent by Stripe to your endpoint.">ⓘ</span></label>
                    <input type="password" id="stripe_webhook_secret" name="stripe_webhook_secret" class="form-control" value="<?php echo e($stripeWebhookSecret); ?>" placeholder="whsec_...">
                </div>
                <small class="text-muted">Publishable key is used on the checkout page. Secret keys stay server-side.</small>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <h3>Google Login</h3>
            </div>
            <div class="card-body">
                <div class="form-group">
                    <label class="checkbox-label">
                        <input type="checkbox" name="google_oauth_enabled" value="1" <?php echo $googleOAuthEnabled ? 'checked' : ''; ?>>
                        Enable Google Sign-In
                        <span class="info-tooltip" data-tooltip="Show a 'Sign in with Google' button on the login page. Requires a Google Client ID and Secret.">ⓘ</span>
                    </label>
                </div>
                <div class="form-group">
                    <label for="google_oauth_client_id">Google Client ID <span class="info-tooltip" data-tooltip="From Google Cloud Console → APIs &amp; Services → Credentials → OAuth 2.0 Client ID.">ⓘ</span></label>
                    <input type="text" id="google_oauth_client_id" name="google_oauth_client_id" class="form-control" value="<?php echo e($googleOAuthClientId); ?>" placeholder="123456789.apps.googleusercontent.com">
                </div>
                <div class="form-group">
                    <label for="google_oauth_client_secret">Google Client Secret <span class="info-tooltip" data-tooltip="The secret key paired with your Client ID. Never shared with the browser.">ⓘ</span></label>
                    <input type="password" id="google_oauth_client_secret" name="google_oauth_client_secret" class="form-control" value="<?php echo e($googleOAuthClientSecret); ?>" placeholder="GOCSPX-...">
                </div>
                <div class="form-group">
                    <label class="checkbox-label">
                        <input type="checkbox" name="google_oauth_allow_registration" value="1" <?php echo $googleOAuthAllowRegistration ? 'checked' : ''; ?>>
                        Allow new account creation via Google
                        <span class="info-tooltip" data-tooltip="When disabled, only existing users whose email matches a Google account can sign in. New accounts will not be created.">ⓘ</span>
                    </label>
                </div>
                <small class="text-muted">Create credentials at <a href="https://console.cloud.google.com/apis/credentials" target="_blank" rel="noopener">Google Cloud Console</a>. Set the Authorized redirect URI to: <code><?php echo e(((!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http') . '://' . ($_SERVER['HTTP_HOST'] ?? 'yourdomain.com') . '/auth/google/callback'); ?></code></small>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <h3>Refund &amp; Returns</h3>
            </div>
            <div class="card-body">
                <div class="form-row">
                    <div class="form-group">
                        <label for="refund_window_digital_days">Digital Products &mdash; Refund Window (days) <span class="info-tooltip" data-tooltip="Maximum number of days after purchase during which a user can request a refund on a digital product. Set to 0 to disable refund requests for digital products.">ⓘ</span></label>
                        <input type="number" id="refund_window_digital_days" name="refund_window_digital_days" class="form-control" value="<?php echo (int)$refundWindowDigitalDays; ?>" min="0" max="365" step="1">
                    </div>
                    <div class="form-group">
                        <label for="refund_window_physical_days">Physical Products &mdash; Return &amp; Refund Window (days) <span class="info-tooltip" data-tooltip="Maximum number of days after purchase during which a user can request a return and refund on a physical product. Set to 0 to disable return requests for physical products.">ⓘ</span></label>
                        <input type="number" id="refund_window_physical_days" name="refund_window_physical_days" class="form-control" value="<?php echo (int)$refundWindowPhysicalDays; ?>" min="0" max="365" step="1">
                    </div>
                </div>
                <small class="text-muted">Set the number of days after purchase during which users can request a refund (digital) or return &amp; refund (physical). Set to 0 to disable.</small>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <h3>Social Links</h3>
            </div>
            <div class="card-body">
                <div class="form-row">
                    <div class="form-group">
                        <label for="social_facebook">Facebook URL <span class="info-tooltip" data-tooltip="Shown in the sidebar social menu when provided.">ⓘ</span></label>
                        <input type="url" id="social_facebook" name="social_facebook" class="form-control" value="<?php echo e($socialFacebook); ?>" placeholder="https://facebook.com/yourpage">
                    </div>
                    <div class="form-group">
                        <label for="social_instagram">Instagram URL <span class="info-tooltip" data-tooltip="Shown in the sidebar social menu when provided.">ⓘ</span></label>
                        <input type="url" id="social_instagram" name="social_instagram" class="form-control" value="<?php echo e($socialInstagram); ?>" placeholder="https://instagram.com/yourprofile">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="social_x">X (Twitter) URL <span class="info-tooltip" data-tooltip="Shown in the sidebar social menu when provided.">ⓘ</span></label>
                        <input type="url" id="social_x" name="social_x" class="form-control" value="<?php echo e($socialX); ?>" placeholder="https://x.com/yourhandle">
                    </div>
                    <div class="form-group">
                        <label for="social_youtube">YouTube URL <span class="info-tooltip" data-tooltip="Shown in the sidebar social menu when provided.">ⓘ</span></label>
                        <input type="url" id="social_youtube" name="social_youtube" class="form-control" value="<?php echo e($socialYouTube); ?>" placeholder="https://youtube.com/@yourchannel">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="social_discord">Discord Invite URL <span class="info-tooltip" data-tooltip="Shown in the sidebar social menu when provided.">ⓘ</span></label>
                        <input type="url" id="social_discord" name="social_discord" class="form-control" value="<?php echo e($socialDiscord); ?>" placeholder="https://discord.gg/yourinvite">
                    </div>
                    <div class="form-group">
                        <label for="social_tiktok">TikTok URL <span class="info-tooltip" data-tooltip="Shown in the sidebar social menu when provided.">ⓘ</span></label>
                        <input type="url" id="social_tiktok" name="social_tiktok" class="form-control" value="<?php echo e($socialTikTok); ?>" placeholder="https://tiktok.com/@yourhandle">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="social_linkedin">LinkedIn URL <span class="info-tooltip" data-tooltip="Shown in the sidebar social menu when provided.">ⓘ</span></label>
                        <input type="url" id="social_linkedin" name="social_linkedin" class="form-control" value="<?php echo e($socialLinkedIn); ?>" placeholder="https://linkedin.com/in/yourprofile">
                    </div>
                </div>
                <div class="text-muted">Only filled links will appear in the sidebar.</div>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <h3>Custom Fonts</h3>
            </div>
            <div class="card-body">
                <div class="form-row">
                    <div class="form-group">
                        <label for="custom_font_name">Font Name <span class="info-tooltip" data-tooltip="Name used inside the theme typography settings.">ⓘ</span></label>
                        <input type="text" id="custom_font_name" name="custom_font_name" class="form-control" placeholder="e.g. Nebula Sans">
                    </div>
                    <div class="form-group">
                        <label for="custom_font_file">Font File (WOFF2/WOFF/TTF/OTF) <span class="info-tooltip" data-tooltip="Upload a font file to use in Theme Editor → Typography.">ⓘ</span></label>
                        <input type="file" id="custom_font_file" name="custom_font_file" class="form-control" accept=".woff2,.woff,.ttf,.otf">
                    </div>
                </div>
                <div class="form-group">
                    <button type="submit" class="btn btn-secondary" name="add_custom_font" value="1">Add Custom Font</button>
                </div>

                <?php if (!empty($customFonts)): ?>
                <div class="form-group">
                    <div class="form-label">Uploaded Fonts</div>
                    <div class="table-responsive">
                        <table class="admin-table">
                            <thead>
                                <tr>
                                    <th>Name</th>
                                    <th>File</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($customFonts as $font): ?>
                                <tr>
                                    <td><?php echo e($font['name'] ?? ''); ?></td>
                                    <td><?php echo e($font['file'] ?? ''); ?></td>
                                    <td>
                                        <button type="submit" class="btn btn-danger btn-sm" name="remove_custom_font" value="<?php echo e($font['file'] ?? ''); ?>">Remove</button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
                <?php endif; ?>
                <div class="text-muted">Custom fonts appear in Theme Editor → Typography.</div>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <h3>Admin Account</h3>
            </div>
            <div class="card-body">
                <div class="form-row">
                    <div class="form-group">
                        <label for="admin_username">Admin Username <span class="info-tooltip" data-tooltip="Username used to log into the admin panel.">ⓘ</span></label>
                        <input type="text" id="admin_username" name="admin_username" class="form-control" value="<?php echo e($adminUsername); ?>" placeholder="Admin username">
                    </div>
                    <div class="form-group">
                        <label for="new_password">New Password <span class="info-tooltip" data-tooltip="Set a new password for the admin account.">ⓘ</span></label>
                        <input type="password" id="new_password" name="new_password" class="form-control" placeholder="Leave blank to keep current">
                    </div>

                    <div class="form-group">
                        <label for="confirm_password">Confirm Password <span class="info-tooltip" data-tooltip="Repeat the new password to confirm the change.">ⓘ</span></label>
                        <input type="password" id="confirm_password" name="confirm_password" class="form-control">
                    </div>
                </div>
            </div>
            <div class="card-footer card-footer-actions">
                <input type="hidden" name="action" value="account">
                <button type="submit" class="btn btn-primary">Save Settings</button>
            </div>
        </div>
    </form>

    <div class="modal-overlay" id="stripe-webhook-info">
        <div class="modal">
            <div class="modal-header">
                <h3>Stripe Webhook Events</h3>
                <button type="button" class="modal-close" aria-label="Close">
                    <svg viewBox="0 0 24 24" aria-hidden="true">
                        <path d="M6 6l12 12"></path>
                        <path d="M18 6l-12 12"></path>
                    </svg>
                </button>
            </div>
            <div class="modal-body">
                <p class="text-muted">Destination URL: <strong>https://YOURDOMAINNAME.com/stripe-webhook</strong></p>

                <div class="modal-section">
                    <h4>Checkout + payments</h4>
                    <ul class="modal-list">
                        <li>checkout.session.completed</li>
                        <li>checkout.session.expired</li>
                        <li>checkout.session.async_payment_succeeded</li>
                        <li>checkout.session.async_payment_failed</li>
                        <li>payment_intent.succeeded</li>
                        <li>payment_intent.processing</li>
                        <li>payment_intent.payment_failed</li>
                        <li>payment_intent.canceled</li>
                    </ul>
                </div>

                <div class="modal-section">
                    <h4>Refunds + disputes</h4>
                    <ul class="modal-list">
                        <li>charge.refunded</li>
                        <li>charge.refund.updated</li>
                        <li>charge.dispute.created</li>
                        <li>charge.dispute.updated</li>
                        <li>charge.dispute.closed</li>
                    </ul>
                </div>

                <div class="modal-section">
                    <h4>Customer + payment method (optional but helpful)</h4>
                    <ul class="modal-list">
                        <li>customer.created</li>
                        <li>customer.updated</li>
                        <li>customer.deleted</li>
                        <li>payment_method.attached</li>
                        <li>payment_method.detached</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<?php AdminLayout::renderFooter(); ?>
