<?php
/**
 * Admin Panel - Security Scanner
 *
 * Scans the codebase and database for signs of tampering, backdoors,
 * injected code, or unauthorized changes. This is NOT a real-time
 * monitor — run it periodically or whenever you suspect compromise.
 *
 * Security: Admin-only, rate-limited, CSRF-protected.
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Auth\AdminAuth;
use NewSite\Logging\LogService;
use NewSite\Admin\AdminLayout;
use NewSite\Database\DatabaseManager;

AdminAuth::requireLogin();

const SCRIPT_INJECTION_REGEX = '/<script|<iframe|javascript:|on(load|error|click|mouseover)\s*=/i';

$db = DatabaseManager::getWriteConnection();
$baseDir = realpath(__DIR__ . '/../..');  // NewSite root
$publicDir = realpath(__DIR__ . '/..');    // NewSite/public
$dataDir = realpath($baseDir . '/data');
$includesDir = realpath($baseDir . '/includes');

$scanResults = null;
$scanTime = 0;

// ─── Suspicious pattern definitions ───────────────────────────────────────────
// Each pattern has: regex, severity (critical/warning/info), description
$suspiciousPatterns = [
    // Critical — almost always malicious in this codebase
    [
        'regex' => '/\beval\s*\(\s*(?:base64_decode|gzinflate|gzuncompress|str_rot13|rawurldecode)\b/i',
        'severity' => 'critical',
        'label' => 'eval() with obfuscation',
        'description' => 'Obfuscated code execution — classic backdoor signature.'
    ],
    [
        'regex' => '/\beval\s*\(\s*\$_(GET|POST|REQUEST|COOKIE|SERVER)\b/i',
        'severity' => 'critical',
        'label' => 'eval() on user input',
        'description' => 'Direct code execution from untrusted input — almost certainly a web shell.'
    ],
    [
        'regex' => '/\bassert\s*\(\s*\$_(GET|POST|REQUEST|COOKIE)\b/i',
        'severity' => 'critical',
        'label' => 'assert() on user input',
        'description' => 'assert() can execute code — likely a backdoor.'
    ],
    [
        'regex' => '/\b(system|passthru|popen|proc_open|pcntl_exec)\s*\(/i',
        'severity' => 'critical',
        'label' => 'Shell execution function',
        'description' => 'Direct OS command execution — not expected in this codebase.'
    ],
    [
        'regex' => '/\bshell_exec\s*\(/i',
        'severity' => 'critical',
        'label' => 'shell_exec()',
        'description' => 'Executes shell commands — not expected in this codebase.',
        /* files.php uses shell_exec() to detect ffmpeg path — safe */
        'allowFiles' => ['public/admin/files.php'],
    ],
    [
        'regex' => '/`\s*\$_(GET|POST|REQUEST|COOKIE)/i',
        'severity' => 'critical',
        'label' => 'Backtick shell with user input',
        'description' => 'Backtick operator executes shell commands with user data.'
    ],
    [
        'regex' => '/\bpreg_replace\s*\(\s*["\'].*\/e["\'\s]/i',
        'severity' => 'critical',
        'label' => 'preg_replace /e modifier',
        'description' => 'Deprecated /e modifier executes replacement as PHP code.'
    ],
    [
        'regex' => '/\bcreate_function\s*\(/i',
        'severity' => 'critical',
        'label' => 'create_function()',
        'description' => 'Deprecated function that can execute arbitrary code.'
    ],
    [
        'regex' => '/\$\w+\s*\(\s*\$_(GET|POST|REQUEST|COOKIE)/i',
        'severity' => 'critical',
        'label' => 'Variable function call on user input',
        'description' => 'Calling a variable as a function with user data — common obfuscation.'
    ],

    // Warning — potentially legitimate but worth reviewing
    [
        'regex' => '/\beval\s*\(/i',
        'severity' => 'warning',
        'label' => 'eval()',
        'description' => 'eval() can execute arbitrary code. Verify this is intentional.'
    ],
    [
        /* Negative lookbehind: skip PDO $db->exec() / $pdo->exec() which are safe SQL calls */
        'regex' => '/(?<!->)\bexec\s*\(\s*[^)]/i',
        'severity' => 'warning',
        'label' => 'exec()',
        'description' => 'Executes external programs. Verify this is intentional.',
        /* files.php uses exec() for FFmpeg thumbnails with escapeshellarg() — safe */
        'allowFiles' => ['public/admin/files.php'],
    ],
    [
        'regex' => '/\bbase64_decode\s*\(\s*["\'][A-Za-z0-9+\/=]{40,}/i',
        'severity' => 'warning',
        'label' => 'Long base64 string decoded',
        'description' => 'Large base64-encoded payload being decoded — may be obfuscated code.'
    ],
    [
        'regex' => '/\bfile_put_contents\s*\(.*\$_(GET|POST|REQUEST|COOKIE)/i',
        'severity' => 'warning',
        'label' => 'Writing user input to file',
        'description' => 'Could allow arbitrary file creation if not properly validated.'
    ],
    [
        'regex' => '/\bmove_uploaded_file\s*\(.*\$_(GET|POST|REQUEST)/i',
        'severity' => 'warning',
        'label' => 'Upload path from user input',
        'description' => 'Upload destination controlled by user — path traversal risk.'
    ],
    [
        /* Require 8+ hex sequences to avoid false positives on short magic-byte checks (e.g. font signatures) */
        'regex' => '/(\\\\x[0-9a-fA-F]{2}){8,}/i',
        'severity' => 'warning',
        'label' => 'Hex-encoded strings',
        'description' => 'Many hex-encoded characters — common obfuscation technique.'
    ],
    [
        'regex' => '/\bchr\s*\(\s*\d+\s*\)\s*\.\s*chr\s*\(\s*\d+\s*\)\s*\.\s*chr\s*\(\s*\d+\s*\)/i',
        'severity' => 'warning',
        'label' => 'chr() string building',
        'description' => 'Building strings from character codes — obfuscation technique.'
    ],
    [
        'regex' => '/<\?php.*\?>.*<\?php/is',
        'severity' => 'info',
        'label' => 'Multiple PHP open/close tags',
        'description' => 'Unusual pattern — injected code is sometimes appended after a closing tag.'
    ],
    [
        'regex' => '/\bini_set\s*\(\s*["\']allow_url_include/i',
        'severity' => 'warning',
        'label' => 'allow_url_include override',
        'description' => 'Enabling remote file inclusion — significant security risk.'
    ],
    [
        'regex' => '/\bini_set\s*\(\s*["\']disable_functions/i',
        'severity' => 'warning',
        'label' => 'disable_functions override attempt',
        'description' => 'Attempting to re-enable disabled functions.'
    ],
    [
        'regex' => '/GIF89a.*<\?php/is',
        'severity' => 'critical',
        'label' => 'PHP code in fake image',
        'description' => 'PHP code hidden behind a GIF header — classic upload bypass.'
    ],
];

// ─── Scan functions ───────────────────────────────────────────────────────────

/**
 * Recursively collect all PHP files under a directory.
 * Skips vendor, node_modules, cache directories.
 */
function collectPhpFiles(string $dir, array &$files, int $maxFiles = 2000): void
{
    if (!is_dir($dir) || count($files) >= $maxFiles) {
        return;
    }

    $skipDirs = ['vendor', 'node_modules', '.git', 'cache', '__pycache__'];
    $items = @scandir($dir);
    if ($items === false) {
        return;
    }

    foreach ($items as $item) {
        if ($item === '.' || $item === '..') {
            continue;
        }
        $path = $dir . DIRECTORY_SEPARATOR . $item;

        // Handle directories: recurse unless it's a known skip-dir
        if (is_dir($path)) {
            if (!in_array($item, $skipDirs, true)) {
                collectPhpFiles($path, $files, $maxFiles);
            }
            continue;
        }

        // Handle files: collect PHP and related extensions
        $ext = strtolower(pathinfo($item, PATHINFO_EXTENSION));
        if (is_file($path) && in_array($ext, ['php', 'phtml', 'inc', 'php5', 'php7', 'phar'], true)) {
            $files[] = $path;
        }
    }
}

/**
 * Scan a single file for suspicious patterns.
 * Returns array of findings.
 */
function scanFileForPatterns(string $filePath, array $patterns, string $baseDir): array
{
    $findings = [];
    $content = @file_get_contents($filePath);
    if ($content === false) {
        return $findings;
    }

    $relativePath = str_replace($baseDir . DIRECTORY_SEPARATOR, '', $filePath);
    $relativePath = str_replace('\\', '/', $relativePath);

    // Check for null bytes (classic injection indicator)
    if (strpos($content, "\0") !== false) {
        $findings[] = [
            'file' => $relativePath,
            'line' => '?',
            'severity' => 'critical',
            'label' => 'Null byte in file',
            'description' => 'Binary null byte found in PHP file — strong indicator of injection.',
            'snippet' => '(binary content)'
        ];
    }

    $lines = explode("\n", $content);
    foreach ($patterns as $pattern) {
        // Skip this pattern entirely if the file is in the per-pattern allowlist
        if (!empty($pattern['allowFiles']) && in_array($relativePath, $pattern['allowFiles'], true)) {
            continue;
        }
        foreach ($lines as $lineNum => $line) {
            if (@preg_match($pattern['regex'], $line)) {
                $findings[] = [
                    'file' => $relativePath,
                    'line' => $lineNum + 1,
                    'severity' => $pattern['severity'],
                    'label' => $pattern['label'],
                    'description' => $pattern['description'],
                    'snippet' => mb_substr(trim($line), 0, 200)
                ];
            }
        }
    }

    return $findings;
}

/**
 * Check for unexpected PHP files in upload directories.
 * Upload directories should NEVER contain .php files.
 */
function scanUploadsForPhp(string $baseDir): array
{
    $findings = [];
    $uploadDirs = [
        $baseDir . '/data/admin_uploads',
        $baseDir . '/data/chat_images',
        $baseDir . '/data/contact_uploads',
        $baseDir . '/data/forum_images',
        $baseDir . '/data/profile_photos',
        $baseDir . '/data/push_queue',
        $baseDir . '/public/assets/uploads',
    ];

    foreach ($uploadDirs as $dir) {
        if (!is_dir($dir)) {
            continue;
        }
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );
        foreach ($iterator as $file) {
            if (!$file->isFile()) {
                continue;
            }
            $ext = strtolower($file->getExtension());
            $relativePath = str_replace($baseDir . DIRECTORY_SEPARATOR, '', $file->getPathname());
            $relativePath = str_replace('\\', '/', $relativePath);

            if (in_array($ext, ['php', 'phtml', 'inc', 'php5', 'php7', 'phar', 'shtml', 'cgi'], true)) {
                $findings[] = [
                    'file' => $relativePath,
                    'line' => '-',
                    'severity' => 'critical',
                    'label' => 'PHP file in upload directory',
                    'description' => 'Executable file found in an upload directory. This should never exist here — likely a web shell upload.',
                    'snippet' => 'File size: ' . $file->getSize() . ' bytes'
                ];
            }

            // Also check for double extensions like image.php.jpg
            $basename = $file->getBasename();
            if (preg_match('/\.php\d?\./i', $basename)) {
                $findings[] = [
                    'file' => $relativePath,
                    'line' => '-',
                    'severity' => 'warning',
                    'label' => 'Double extension (PHP hidden)',
                    'description' => 'File has .php inside its name with another extension — potential bypass attempt.',
                    'snippet' => $basename
                ];
            }
        }
    }

    return $findings;
}

/**
 * Check for recently modified PHP files (last 48 hours).
 */
function scanRecentlyModified(string $baseDir, int $hours = 48): array
{
    $findings = [];
    $cutoff = time() - ($hours * 3600);
    $phpFiles = [];
    collectPhpFiles($baseDir, $phpFiles);

    foreach ($phpFiles as $filePath) {
        $mtime = @filemtime($filePath);
        if ($mtime && $mtime > $cutoff) {
            $relativePath = str_replace($baseDir . DIRECTORY_SEPARATOR, '', $filePath);
            $relativePath = str_replace('\\', '/', $relativePath);
            $findings[] = [
                'file' => $relativePath,
                'line' => '-',
                'severity' => 'info',
                'label' => 'Recently modified',
                'description' => 'Modified ' . date('Y-m-d H:i:s', $mtime),
                'snippet' => ''
            ];
        }
    }

    // Sort newest first
    usort($findings, function ($a, $b) {
        return strcmp($b['description'], $a['description']);
    });

    return $findings;
}

/**
 * Scan a single .htaccess file for suspicious rules.
 *
 * @param string $relativePath Display path for findings
 * @param string $content      File contents
 * @param array  $rules        Map of regex => description
 * @return array Findings list
 */
function scanSingleHtaccessFile(string $relativePath, string $content, array $rules): array
{
    $findings = [];
    $lines = explode("\n", $content);

    foreach ($rules as $regex => $desc) {
        foreach ($lines as $lineNum => $line) {
            $trimmed = trim($line);
            if ($trimmed === '' || $trimmed[0] === '#') {
                continue;
            }
            if (@preg_match($regex, $line)) {
                $findings[] = [
                    'file' => $relativePath,
                    'line' => $lineNum + 1,
                    'severity' => 'warning',
                    'label' => 'Suspicious .htaccess rule',
                    'description' => $desc,
                    'snippet' => mb_substr(trim($line), 0, 200)
                ];
            }
        }
    }

    return $findings;
}

/**
 * Check .htaccess files for suspicious rules.
 */
function scanHtaccess(string $baseDir): array
{
    $findings = [];
    $htaccessLocations = [
        $baseDir . '/.htaccess',
        $baseDir . '/public/.htaccess',
        $baseDir . '/data/.htaccess',
        $baseDir . '/public/assets/.htaccess',
        $baseDir . '/public/admin/.htaccess',
    ];

    $suspiciousHtaccess = [
        '/\bRewriteRule\b.*\bhttp/i' => 'External redirect in .htaccess — could redirect visitors to a malicious site.',
        '/\bSetHandler\b.*php/i' => 'Forcing PHP handler — could make non-PHP files executable.',
        '/\bAddType\b.*php/i' => 'Adding PHP MIME type — could make non-PHP files executable.',
        '/\bauto_prepend_file\b/i' => 'auto_prepend_file — injects PHP code before every request.',
        '/\bauto_append_file\b/i' => 'auto_append_file — injects PHP code after every request.',
        '/\bphp_value\b.*allow_url_include/i' => 'Enabling remote file inclusion via .htaccess.',
    ];

    foreach ($htaccessLocations as $path) {
        if (!file_exists($path)) {
            continue;
        }
        $content = @file_get_contents($path);
        if ($content === false) {
            continue;
        }
        $relativePath = str_replace($baseDir . DIRECTORY_SEPARATOR, '', $path);
        $relativePath = str_replace('\\', '/', $relativePath);

        $fileFindings = scanSingleHtaccessFile($relativePath, $content, $suspiciousHtaccess);
        $findings = array_merge($findings, $fileFindings);
    }

    return $findings;
}

/**
 * List admin accounts as informational findings.
 */
function scanDbAdminAccounts(PDO $db): array
{
    $findings = [];
    try {
        $admins = $db->query(
            "SELECT id, email, nickname, display_name, role, created_at FROM site_users WHERE role = 'admin' ORDER BY id"
        )->fetchAll(PDO::FETCH_ASSOC);
        foreach ($admins as $admin) {
            $findings[] = [
                'file' => 'database',
                'line' => 'site_users #' . (int)$admin['id'],
                'severity' => 'info',
                'label' => 'Admin account exists',
                'description' => 'Email: ' . ($admin['email'] ?? '?') . ' | Created: ' . ($admin['created_at'] ?? '?'),
                'snippet' => 'Verify all admin accounts are legitimate.'
            ];
        }
    } catch (PDOException) {
        // Table might not exist
    }
    return $findings;
}

/**
 * Scan a key-value settings table for script / HTML injection.
 *
 * @param PDO    $db        Database connection
 * @param string $table     Table name identifier (used in finding paths)
 * @param string $query     Full SELECT query (hardcoded at call site — no user input)
 * @param string $label     Human-readable label for findings
 * @param string $descNoun  Noun used in the description (e.g. "Setting", "Theme setting")
 */
function scanDbSettingsTableForInjection(
    PDO $db,
    string $table,
    string $query,
    string $label,
    string $descNoun
): array {
    $findings = [];
    try {
        $rows = $db->query($query)->fetchAll(PDO::FETCH_ASSOC);
        foreach ($rows as $row) {
            $val = (string)($row['setting_value'] ?? '');
            if ($val === '') {
                continue;
            }
            if (preg_match(SCRIPT_INJECTION_REGEX, $val)) {
                $findings[] = [
                    'file' => 'database',
                    'line' => $table . '.' . ($row['setting_key'] ?? '?'),
                    'severity' => 'critical',
                    'label' => $label,
                    'description' => $descNoun . ' "' . ($row['setting_key'] ?? '?') . '" contains potentially malicious HTML/JS.',
                    'snippet' => mb_substr($val, 0, 200)
                ];
            }
        }
    } catch (PDOException) {
        // Table might not exist
    }
    return $findings;
}

/**
 * Scan page titles for injected scripts.
 */
function scanDbPageTitles(PDO $db): array
{
    $findings = [];
    try {
        $pages = $db->query("SELECT id, slug, title FROM pages")->fetchAll(PDO::FETCH_ASSOC);
        foreach ($pages as $page) {
            $title = (string)($page['title'] ?? '');
            if (preg_match(SCRIPT_INJECTION_REGEX, $title)) {
                $findings[] = [
                    'file' => 'database',
                    'line' => 'pages #' . (int)$page['id'],
                    'severity' => 'warning',
                    'label' => 'Script in page title',
                    'description' => 'Page "' . ($page['slug'] ?? '?') . '" title contains script-like content.',
                    'snippet' => mb_substr($title, 0, 200)
                ];
            }
        }
    } catch (PDOException) {
        // Table might not exist
    }
    return $findings;
}

/**
 * Flag suspiciously high admin-role counts.
 */
function scanDbRoleCounts(PDO $db): array
{
    $findings = [];
    try {
        $roleCount = $db->query(
            "SELECT role, COUNT(*) as cnt FROM site_users GROUP BY role ORDER BY cnt DESC"
        )->fetchAll(PDO::FETCH_ASSOC);
        foreach ($roleCount as $rc) {
            if ($rc['role'] === 'admin' && (int)$rc['cnt'] > 3) {
                $findings[] = [
                    'file' => 'database',
                    'line' => 'site_users',
                    'severity' => 'warning',
                    'label' => 'Many admin accounts (' . (int)$rc['cnt'] . ')',
                    'description' => 'There are ' . (int)$rc['cnt'] . ' admin accounts. Verify all are legitimate.',
                    'snippet' => ''
                ];
            }
        }
    } catch (PDOException) {
        // Table might not exist
    }
    return $findings;
}

/**
 * Check the database for suspicious content.
 * Delegates to focused helpers for each check category.
 */
function scanDatabase(PDO $db): array
{
    return array_merge(
        scanDbAdminAccounts($db),
        scanDbSettingsTableForInjection(
            $db,
            'settings',
            "SELECT setting_key, setting_value FROM settings",
            'Script/HTML injection in settings',
            'Setting'
        ),
        scanDbSettingsTableForInjection(
            $db,
            'theme_settings',
            "SELECT setting_key, setting_value FROM theme_settings",
            'Script/HTML injection in theme settings',
            'Theme setting'
        ),
        scanDbPageTitles($db),
        scanDbRoleCounts($db)
    );
}

/**
 * Check for unexpected PHP files in the public root directory.
 */
function scanUnexpectedPublicFiles(string $baseDir): array
{
    $findings = [];
    $knownPublicFiles = [
        'index.php', '.htaccess', 'robots.txt', 'sitemap.xml',
        'manifest.json', 'browserconfig.xml', 'humans.txt', 'offline.html',
        'favicon.ico', 'sw.js', 'service-worker.js',
    ];

    $publicDir = $baseDir . '/public';
    $items = @scandir($publicDir);
    if ($items === false) {
        return $findings;
    }

    foreach ($items as $item) {
        if ($item === '.' || $item === '..' || is_dir($publicDir . '/' . $item)) {
            continue;
        }
        $ext = strtolower(pathinfo($item, PATHINFO_EXTENSION));
        if ($ext === 'php' && !in_array($item, $knownPublicFiles, true)) {
            $findings[] = [
                'file' => 'public/' . $item,
                'line' => '-',
                'severity' => 'warning',
                'label' => 'Unexpected PHP in public root',
                'description' => 'PHP file in the public root that is not part of the known set. Could be a leftover or injected file.',
                'snippet' => 'Size: ' . @filesize($publicDir . '/' . $item) . ' bytes, modified: ' . date('Y-m-d H:i', @filemtime($publicDir . '/' . $item))
            ];
        }
    }

    return $findings;
}

/**
 * Check for unexpected PHP files in the data directory.
 */
function scanUnexpectedDataFiles(string $baseDir): array
{
    $findings = [];
    $knownDataFiles = ['config.json', 'config.example.json', 'setup.php', 'allowlist_tool.php', '.htaccess'];
    $dataDir = $baseDir . '/data';

    $items = @scandir($dataDir);
    if ($items === false) {
        return $findings;
    }

    foreach ($items as $item) {
        if ($item === '.' || $item === '..' || is_dir($dataDir . '/' . $item)) {
            continue;
        }
        if (in_array($item, $knownDataFiles, true)) {
            continue;
        }
        $ext = strtolower(pathinfo($item, PATHINFO_EXTENSION));
        if (in_array($ext, ['php', 'phtml', 'phar', 'inc'], true)) {
            $findings[] = [
                'file' => 'data/' . $item,
                'line' => '-',
                'severity' => 'warning',
                'label' => 'Unexpected PHP in data directory',
                'description' => 'PHP file in data/ that is not part of the known set.',
                'snippet' => 'Size: ' . @filesize($dataDir . '/' . $item) . ' bytes, modified: ' . date('Y-m-d H:i', @filemtime($dataDir . '/' . $item))
            ];
        }
    }

    return $findings;
}

/**
 * Check for unexpected files in the project root and public root.
 * Delegates to focused helpers for each directory.
 */
function scanUnexpectedFiles(string $baseDir): array
{
    return array_merge(
        scanUnexpectedPublicFiles($baseDir),
        scanUnexpectedDataFiles($baseDir)
    );
}

// ─── Run scan on POST ─────────────────────────────────────────────────────────

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verifyCSRF();

    $startTime = microtime(true);
    $scanResults = [
        'patterns' => [],
        'uploads' => [],
        'htaccess' => [],
        'database' => [],
        'unexpected' => [],
        'recent' => [],
    ];

    // 1. Collect all PHP files
    $phpFiles = [];
    collectPhpFiles($baseDir, $phpFiles);

    // 2. Scan for suspicious patterns
    foreach ($phpFiles as $filePath) {
        $fileFindings = scanFileForPatterns($filePath, $suspiciousPatterns, $baseDir);
        if (!empty($fileFindings)) {
            // Skip this scanner file itself to avoid false positives from its own pattern definitions
            $relativePath = str_replace($baseDir . DIRECTORY_SEPARATOR, '', $filePath);
            $relativePath = str_replace('\\', '/', $relativePath);
            if ($relativePath === 'public/admin/security-scan.php') {
                continue;
            }
            $scanResults['patterns'] = array_merge($scanResults['patterns'], $fileFindings);
        }
    }

    // 3. Scan uploads for PHP files
    $scanResults['uploads'] = scanUploadsForPhp($baseDir);

    // 4. Scan .htaccess files
    $scanResults['htaccess'] = scanHtaccess($baseDir);

    // 5. Scan database
    $scanResults['database'] = scanDatabase($db);

    // 6. Scan for unexpected files
    $scanResults['unexpected'] = scanUnexpectedFiles($baseDir);

    // 7. Recently modified files (48h)
    $scanResults['recent'] = scanRecentlyModified($baseDir, 48);

    $scanTime = round(microtime(true) - $startTime, 2);

    // Log the scan
    $criticalCount = 0;
    $warningCount = 0;
    foreach ($scanResults as $group) {
        foreach ($group as $f) {
            if ($f['severity'] === 'critical') {
                $criticalCount++;
            }
            if ($f['severity'] === 'warning') {
                $warningCount++;
            }
        }
    }
    LogService::add('info', 'Security scan completed', json_encode([
        'critical' => $criticalCount,
        'warnings' => $warningCount,
        'files_scanned' => count($phpFiles),
        'duration' => $scanTime . 's'
    ]));
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header content-header-actions">
        <div>
            <h1>Security Scanner</h1>
            <p>Scan the codebase and database for signs of tampering, backdoors, or injected code.</p>
        </div>
        <form method="POST" action="/admin/security-scan.php">
            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
            <button type="submit" class="btn btn-primary" data-confirm="Run a full security scan? This may take a few seconds.">
                Run Scan
            </button>
        </form>
    </div>

    <?php if ($scanResults === null): ?>
        <div class="card">
            <div class="card-body">
                <p>Click <strong>Run Scan</strong> to analyze your site for potential security issues.</p>
                <p class="text-muted scan-intro-text">The scanner checks for:</p>
                <ul class="text-muted scan-intro-list">
                    <li>Backdoors and web shells (eval, exec, system, obfuscated code)</li>
                    <li>PHP files in upload directories (should never exist)</li>
                    <li>Suspicious <code>.htaccess</code> rules (redirects, handler overrides)</li>
                    <li>XSS/script injection in database (settings, theme, pages)</li>
                    <li>Unexpected files in public/ and data/ directories</li>
                    <li>Recently modified PHP files (last 48 hours)</li>
                    <li>Admin account inventory</li>
                </ul>
                <div class="scan-tip">
                    <strong>&#9888; Tip:</strong> The scan inspects every PHP file and database record on your site, which can take up to a minute or more. Additionally, newly uploaded or modified files may not appear in scan results immediately due to server-side file caching (OPcache / realpath cache). If you just uploaded a file and the scan does not detect it, wait about a minute and run the scan again.
                </div>
            </div>
        </div>
    <?php else: ?>

        <?php
        // Count totals
        $totalCritical = 0;
        $totalWarning = 0;
        $totalInfo = 0;
        foreach ($scanResults as $group) {
            foreach ($group as $f) {
                if ($f['severity'] === 'critical') {
                    $totalCritical++;
                } elseif ($f['severity'] === 'warning') {
                    $totalWarning++;
                } else {
                    $totalInfo++;
                }
            }
        }
        ?>

        <div class="card">
            <div class="card-header">
                <h3>Scan Summary</h3>
            </div>
            <div class="card-body">
                <div class="scan-summary-grid">
                    <div class="scan-summary-item <?php echo $totalCritical > 0 ? 'scan-critical' : 'scan-ok'; ?>">
                        <span class="scan-summary-count"><?php echo (int)$totalCritical; ?></span>
                        <span class="scan-summary-label">Critical</span>
                    </div>
                    <div class="scan-summary-item <?php echo $totalWarning > 0 ? 'scan-warning' : 'scan-ok'; ?>">
                        <span class="scan-summary-count"><?php echo (int)$totalWarning; ?></span>
                        <span class="scan-summary-label">Warnings</span>
                    </div>
                    <div class="scan-summary-item scan-info-item">
                        <span class="scan-summary-count"><?php echo (int)$totalInfo; ?></span>
                        <span class="scan-summary-label">Info</span>
                    </div>
                    <div class="scan-summary-item scan-info-item">
                        <span class="scan-summary-count"><?php echo e($scanTime); ?>s</span>
                        <span class="scan-summary-label">Duration</span>
                    </div>
                </div>
                <?php if ($totalCritical === 0 && $totalWarning === 0): ?>
                    <p class="scan-all-clear">&#10003; No critical or warning-level issues found.</p>
                <?php endif; ?>
            </div>
        </div>

        <!-- Severity filter bar (pure CSS — no JS needed) -->
        <div class="scan-filter-bar">
            <span class="scan-filter-label">Filter:</span>
            <input type="radio" name="scan-filter" id="scan-f-all" class="scan-filter-radio" checked>
            <label for="scan-f-all" class="btn btn-sm scan-filter-lbl">All</label>
            <input type="radio" name="scan-filter" id="scan-f-critical" class="scan-filter-radio">
            <label for="scan-f-critical" class="btn btn-sm scan-filter-lbl scan-filter-critical">Critical (<?php echo (int)$totalCritical; ?>)</label>
            <input type="radio" name="scan-filter" id="scan-f-warning" class="scan-filter-radio">
            <label for="scan-f-warning" class="btn btn-sm scan-filter-lbl scan-filter-warning">Warnings (<?php echo (int)$totalWarning; ?>)</label>
            <input type="radio" name="scan-filter" id="scan-f-info" class="scan-filter-radio">
            <label for="scan-f-info" class="btn btn-sm scan-filter-lbl scan-filter-info">Info (<?php echo (int)$totalInfo; ?>)</label>
        </div>

        <?php
        // Render each section
        $sections = [
            'patterns' => ['title' => 'Code Pattern Analysis', 'desc' => 'Suspicious patterns found in PHP files.'],
            'uploads' => ['title' => 'Upload Directory Check', 'desc' => 'Executable files found inside upload directories.'],
            'htaccess' => ['title' => '.htaccess Analysis', 'desc' => 'Suspicious server configuration rules.'],
            'database' => ['title' => 'Database Integrity', 'desc' => 'Injected content or suspicious records in the database.'],
            'unexpected' => ['title' => 'Unexpected Files', 'desc' => 'Files that are not part of the known project structure.'],
            'recent' => ['title' => 'Recently Modified (48h)', 'desc' => 'PHP files modified in the last 48 hours.'],
        ];

foreach ($sections as $sectionKey => $sectionMeta):
    $items = $scanResults[$sectionKey];
    $hasCritical = false;
    $hasWarning = false;
    foreach ($items as $f) {
        if ($f['severity'] === 'critical') {
            $hasCritical = true;
        }
        if ($f['severity'] === 'warning') {
            $hasWarning = true;
        }
    }
    ?>
        <div class="card">
            <div class="card-header">
                <h3>
                    <?php echo e($sectionMeta['title']); ?>
                    <span class="scan-section-count">(<?php echo count($items); ?>)</span>
                    <?php if ($hasCritical): ?>
                        <span class="status-badge status-badge-critical">CRITICAL</span>
                    <?php elseif ($hasWarning): ?>
                        <span class="status-badge status-badge-warning">WARNING</span>
                    <?php elseif (count($items) === 0): ?>
                        <span class="status-badge status-active">CLEAN</span>
                    <?php else: ?>
                        <span class="status-badge status-active">OK</span>
                    <?php endif; ?>
                </h3>
            </div>
            <?php if (!empty($items)): ?>
            <div class="card-body">
                <div class="scan-results-list">
                    <?php foreach ($items as $finding): ?>
                    <details class="scan-row" data-severity="<?php echo e($finding['severity']); ?>">
                        <summary class="scan-row-header">
                            <div class="scan-row-severity">
                                <?php if ($finding['severity'] === 'critical'): ?>
                                    <span class="status-badge status-badge-critical">CRITICAL</span>
                                <?php elseif ($finding['severity'] === 'warning'): ?>
                                    <span class="status-badge status-badge-warning">WARNING</span>
                                <?php else: ?>
                                    <span class="status-badge status-active">INFO</span>
                                <?php endif; ?>
                            </div>
                            <div class="scan-row-summary">
                                <strong><?php echo e($finding['label']); ?></strong>
                                <span class="scan-row-file"><code><?php echo e($finding['file']); ?></code><?php if ($finding['line'] !== '-' && $finding['line'] !== '?'): ?> <small class="text-muted">line <?php echo e($finding['line']); ?></small><?php endif; ?></span>
                            </div>
                            <div class="scan-row-chevron">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="18" height="18"><polyline points="6 9 12 15 18 9"></polyline></svg>
                            </div>
                        </summary>
                        <div class="scan-row-details">
                            <p class="scan-row-desc"><?php echo e($finding['description']); ?></p>
                            <?php if (!empty($finding['snippet'])): ?>
                                <pre class="scan-row-snippet"><?php echo e($finding['snippet']); ?></pre>
                            <?php endif; ?>
                        </div>
                    </details>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php else: ?>
            <div class="card-body">
                <p class="text-muted">No issues found.</p>
            </div>
            <?php endif; ?>
        </div>
        <?php endforeach; ?>

        <div class="card">
            <div class="card-header">
                <h3>What To Do If You Find Issues</h3>
            </div>
            <div class="card-body">
                <ul class="scan-advice-list">
                    <li><strong>Critical findings:</strong> Investigate immediately. Compare the file against a known-clean backup. If you cannot identify the code, assume compromise.</li>
                    <li><strong>PHP files in uploads:</strong> Delete them immediately. They are almost always web shells.</li>
                    <li><strong>Suspicious .htaccess:</strong> Compare against the original. Look for redirect rules or PHP handler overrides you didn't add.</li>
                    <li><strong>Database injection:</strong> Clean the affected setting/record. Check your CSP and output escaping.</li>
                    <li><strong>Unexpected admin accounts:</strong> If you don't recognize an admin, remove or demote them immediately and rotate all credentials.</li>
                    <li><strong>Info findings:</strong> These are informational and generally safe, but worth reviewing if you suspect a breach.</li>
                </ul>
            </div>
        </div>

    <?php endif; ?>
</div>

<?php AdminLayout::renderFooter(); ?>
