<?php
/**
 * Admin Panel - Refund Requests
 *
 * Security: CSRF-protected POST actions, all output escaped with e(),
 * prepared statements for all queries, admin auth required.
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Auth\AdminAuth;
use NewSite\Shop\CurrencyService;
use NewSite\Database\DbHelper;
use NewSite\Admin\AdminLayout;
use NewSite\Database\DatabaseManager;
use NewSite\Logging\LogService;
use NewSite\Upload\UploadService;

AdminAuth::requireLogin();

$db = DatabaseManager::getWriteConnection();
$message = '';
$error = '';
$filterStatus = trim($_GET['status'] ?? '');
$allowedFilters = ['pending', 'accepted', 'cancelled', ''];

if (!in_array($filterStatus, $allowedFilters, true)) {
    $filterStatus = '';
}

// ── Handle admin actions (accept / cancel) ──────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verifyCSRF();
    $action = $_POST['action'] ?? '';
    $refundId = (int)($_POST['refund_id'] ?? 0);

    if ($refundId <= 0) {
        $error = 'Invalid refund request.';
    } elseif ($action === 'accept_refund') {
        // Check if this is a physical item refund — handle optional return instruction uploads
        $stmt = $db->prepare(
            "SELECT oi.is_digital FROM refund_requests rr JOIN order_items oi ON oi.id = rr.order_item_id WHERE rr.id = ? LIMIT 1"
        );
        $stmt->execute([$refundId]);
        $refundItem = $stmt->fetch();
        $isPhysical = $refundItem && empty($refundItem['is_digital']);

        $instructionPaths = [];
        if ($isPhysical && !empty($_FILES['return_labels']['name'][0])) {
            $fileCount = count($_FILES['return_labels']['name']);
            $maxFiles = 5;
            $uploadCount = min($fileCount, $maxFiles);
            for ($i = 0; $i < $uploadCount; $i++) {
                if ($_FILES['return_labels']['error'][$i] !== UPLOAD_ERR_OK) {
                    continue;
                }
                $singleFile = [
                    'name'     => $_FILES['return_labels']['name'][$i],
                    'tmp_name' => $_FILES['return_labels']['tmp_name'][$i],
                    'size'     => $_FILES['return_labels']['size'][$i],
                    'error'    => $_FILES['return_labels']['error'][$i],
                    'type'     => $_FILES['return_labels']['type'][$i],
                ];
                $result = UploadService::uploadRefundImage($singleFile);
                if (!empty($result['success']) && !empty($result['path'])) {
                    $instructionPaths[] = $result['path'];
                }
            }
        }

        $instructionsJson = !empty($instructionPaths) ? json_encode($instructionPaths) : '';
        $now = DbHelper::nowString();
        $stmt = $db->prepare(
            "UPDATE refund_requests SET status = 'accepted', admin_reason = '', return_instructions = ?, updated_at = ? WHERE id = ? AND status = 'pending'"
        );
        $stmt->execute([$instructionsJson, $now, $refundId]);
        if ($stmt->rowCount() > 0) {
            LogService::add('refund', 'Admin accepted refund request', json_encode([
                'refund_id' => $refundId,
                'return_labels' => count($instructionPaths),
            ]));
            $message = 'Refund request accepted.';
        } else {
            $error = 'Refund request not found or already processed.';
        }
    } elseif ($action === 'cancel_refund') {
        $reason = mb_substr(trim($_POST['cancel_reason'] ?? ''), 0, 1000);
        if ($reason === '') {
            $error = 'Please provide a reason for declining this refund.';
        } else {
            $now = DbHelper::nowString();
            $stmt = $db->prepare(
                "UPDATE refund_requests SET status = 'cancelled', admin_reason = ?, updated_at = ? WHERE id = ? AND status = 'pending'"
            );
            $stmt->execute([$reason, $now, $refundId]);
            if ($stmt->rowCount() > 0) {
                LogService::add('refund', 'Admin declined refund request', json_encode(['refund_id' => $refundId, 'reason' => $reason]));
                $message = 'Refund request declined.';
            } else {
                $error = 'Refund request not found or already processed.';
            }
        }
    } else {
        $error = 'Unknown action.';
    }
}

// ── Fetch refund requests ───────────────────────────────────────
$query = "SELECT rr.*,
                 o.order_number, o.currency, o.total_amount,
                 oi.product_name, oi.variant_label, oi.unit_price, oi.quantity, oi.is_digital,
                 su.email, su.display_name, su.nickname,
                 dd.download_count
          FROM refund_requests rr
          JOIN orders o ON o.id = rr.order_id
          JOIN order_items oi ON oi.id = rr.order_item_id
          LEFT JOIN site_users su ON su.id = rr.user_id
          LEFT JOIN digital_downloads dd ON dd.order_item_id = rr.order_item_id";

$params = [];
if ($filterStatus !== '') {
    $query .= " WHERE rr.status = ?";
    $params[] = $filterStatus;
}
$query .= " ORDER BY CASE rr.status WHEN 'pending' THEN 0 WHEN 'accepted' THEN 1 ELSE 2 END, rr.created_at DESC";

$stmt = $db->prepare($query);
$stmt->execute($params);
$refunds = $stmt->fetchAll();

// Count pending for the badge
$pendingCount = 0;
foreach ($refunds as $r) {
    if ($r['status'] === 'pending') {
        $pendingCount++;
    }
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header">
        <h1>Refund Requests</h1>
        <p>Review and manage refund and return requests from customers for digital and physical purchases.</p>
    </div>

    <div class="card mb-20">
        <div class="card-body">
            <div class="refund-filters">
                <a href="/admin/refunds.php" class="btn btn-sm <?php echo $filterStatus === '' ? 'btn-primary' : 'btn-outline'; ?>">All</a>
                <a href="/admin/refunds.php?status=pending" class="btn btn-sm <?php echo $filterStatus === 'pending' ? 'btn-primary' : 'btn-outline'; ?>">Pending<?php if ($pendingCount > 0 && $filterStatus !== 'pending'): ?> (<?php echo $pendingCount; ?>)<?php endif; ?></a>
                <a href="/admin/refunds.php?status=accepted" class="btn btn-sm <?php echo $filterStatus === 'accepted' ? 'btn-primary' : 'btn-outline'; ?>">Accepted</a>
                <a href="/admin/refunds.php?status=cancelled" class="btn btn-sm <?php echo $filterStatus === 'cancelled' ? 'btn-primary' : 'btn-outline'; ?>">Declined</a>
            </div>
        </div>
    </div>

    <?php if ($message): ?>
        <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>
    <?php if ($error): ?>
        <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <div class="card">
        <div class="card-body">
            <?php if (empty($refunds)): ?>
                <div class="empty-state">No refund requests found.</div>
            <?php else: ?>
                <div class="table-container">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Order</th>
                                <th>Customer</th>
                                <th>Product</th>
                                <th>Price</th>
                                <th>Type</th>
                                <th>Downloaded</th>
                                <th>Status</th>
                                <th>Requested</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($refunds as $refund): ?>
                                <?php
                                    $symbol = CurrencyService::getSymbolFor($refund['currency'] ?? 'USD');
                                    $price = number_format((float)($refund['unit_price'] ?? 0), 2);
                                    $dlCount = (int)($refund['download_count'] ?? 0);
                                    $requestDate = !empty($refund['created_at']) ? date('M j, Y', strtotime($refund['created_at'])) : '';
                                    $refundStatus = $refund['status'] ?? 'pending';
                                    $cancelReasonId = 'cancel-reason-' . (int)$refund['id'];
                                    $isPhysicalItem = empty($refund['is_digital']);
                                ?>
                                <tr>
                                    <td>
                                        <strong>#<?php echo e($refund['order_number']); ?></strong>
                                    </td>
                                    <td>
                                        <div><?php echo e($refund['display_name'] ?: $refund['nickname'] ?: 'Customer'); ?></div>
                                        <div class="text-muted"><?php echo e($refund['email'] ?? ''); ?></div>
                                    </td>
                                    <td>
                                        <div><?php echo e($refund['product_name'] ?? 'Product'); ?></div>
                                        <?php if (!empty($refund['variant_label'])): ?>
                                            <span class="text-muted">(<?php echo e($refund['variant_label']); ?>)</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo e($symbol . $price); ?></td>
                                    <td>
                                        <?php if ($isPhysicalItem): ?>
                                            <span class="status-badge">Physical</span>
                                        <?php else: ?>
                                            <span class="status-badge status-active">Digital</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($dlCount > 0): ?>
                                            <span class="status-badge status-badge-warning">Yes (<?php echo $dlCount; ?>x)</span>
                                        <?php else: ?>
                                            <span class="status-badge status-badge-refundable">No</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($refundStatus === 'pending'): ?>
                                            <span class="status-badge status-badge-warning">Pending</span>
                                        <?php elseif ($refundStatus === 'accepted'): ?>
                                            <span class="status-badge status-active">Accepted</span>
                                        <?php elseif ($refundStatus === 'cancelled'): ?>
                                            <span class="status-badge status-inactive">Declined</span>
                                            <?php if (!empty($refund['admin_reason'])): ?>
                                                <div class="text-muted refund-admin-reason"><?php echo e($refund['admin_reason']); ?></div>
                                            <?php endif; ?>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="text-muted"><?php echo e($requestDate); ?></span>
                                    </td>
                                    <td>
                                        <?php if ($refundStatus === 'pending'): ?>
                                            <div class="refund-action-group">
                                                <form method="POST" action="/admin/refunds.php<?php echo $filterStatus !== '' ? '?status=' . e($filterStatus) : ''; ?>" enctype="multipart/form-data">
                                                    <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                                    <input type="hidden" name="action" value="accept_refund">
                                                    <input type="hidden" name="refund_id" value="<?php echo (int)$refund['id']; ?>">
                                                    <?php if ($isPhysicalItem): ?>
                                                        <?php $uploadFieldId = 'return-labels-' . (int)$refund['id']; ?>
                                                        <div class="refund-upload-fields">
                                                            <label for="<?php echo e($uploadFieldId); ?>" class="form-label-sm">Return label / instructions (optional, images only)</label>
                                                            <input type="file" id="<?php echo e($uploadFieldId); ?>" name="return_labels[]" accept="image/*" multiple class="form-control form-control-sm">
                                                        </div>
                                                    <?php endif; ?>
                                                    <button type="submit" class="btn btn-primary btn-sm">Accept</button>
                                                </form>
                                                <form method="POST" action="/admin/refunds.php<?php echo $filterStatus !== '' ? '?status=' . e($filterStatus) : ''; ?>" class="refund-cancel-form">
                                                    <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                                    <input type="hidden" name="action" value="cancel_refund">
                                                    <input type="hidden" name="refund_id" value="<?php echo (int)$refund['id']; ?>">
                                                    <div class="refund-cancel-fields">
                                                        <label class="sr-only" for="<?php echo e($cancelReasonId); ?>">Reason for decline</label>
                                                        <textarea id="<?php echo e($cancelReasonId); ?>" name="cancel_reason" class="form-control refund-reason-input" placeholder="Reason for declining..." rows="2" required></textarea>
                                                        <button type="submit" class="btn btn-danger btn-sm">Decline</button>
                                                    </div>
                                                </form>
                                            </div>
                                        <?php else: ?>
                                            <span class="text-muted">&mdash;</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php AdminLayout::renderFooter(); ?>
