<?php
/**
 * Admin Panel - Purchases
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Auth\AdminAuth;
use NewSite\Shop\CurrencyService;
use NewSite\Database\DbHelper;
use NewSite\Admin\AdminLayout;
use NewSite\Database\DatabaseManager;

AdminAuth::requireLogin();

$db = DatabaseManager::getWriteConnection();
$message = '';
$error = '';
$searchQuery = trim($_GET['q'] ?? '');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verifyCSRF();
    $action = $_POST['action'] ?? '';
    if ($action === 'update_shipping') {
        $orderId = (int)($_POST['order_id'] ?? 0);
        $status = trim($_POST['shipping_status'] ?? 'pending');
        $trackingNumber = trim($_POST['tracking_number'] ?? '');
        $trackingUrl = trim($_POST['tracking_url'] ?? '');
        $allowedStatuses = ['pending', 'processing', 'shipped', 'delivered', 'cancelled'];
        if (!in_array($status, $allowedStatuses, true)) {
            $status = 'pending';
        }
        if ($orderId > 0) {
            $now = DbHelper::nowString();
            $stmt = $db->prepare("UPDATE orders SET shipping_status = ?, tracking_number = ?, tracking_url = ?, updated_at = ? WHERE id = ?");
            $stmt->execute([$status, $trackingNumber, $trackingUrl, $now, $orderId]);
            $message = 'Shipping details updated.';
        } else {
            $error = 'Invalid order.';
        }
    }
}

$orders = [];
if ($searchQuery !== '') {
    $stmt = $db->prepare("SELECT o.*, su.email, su.display_name, su.nickname FROM orders o LEFT JOIN site_users su ON su.id = o.user_id WHERE o.order_number LIKE ? ORDER BY o.created_at DESC");
    $stmt->execute(['%' . $searchQuery . '%']);
    $orders = $stmt->fetchAll();
} else {
    $stmt = $db->query("SELECT o.*, su.email, su.display_name, su.nickname FROM orders o LEFT JOIN site_users su ON su.id = o.user_id ORDER BY o.created_at DESC");
    $orders = $stmt->fetchAll();
}
$orderItems = [];
$orderDownloads = [];

if (!empty($orders)) {
    $orderIds = array_column($orders, 'id');
    $placeholders = implode(',', array_fill(0, count($orderIds), '?'));
    $stmt = $db->prepare("SELECT * FROM order_items WHERE order_id IN ($placeholders) ORDER BY id ASC");
    $stmt->execute($orderIds);
    $items = $stmt->fetchAll();
    foreach ($items as $item) {
        $orderItems[$item['order_id']][] = $item;
    }

    // Fetch download tracking data for digital items
    $stmt = $db->prepare(
        "SELECT dd.order_item_id, dd.download_count, dd.last_download_at
         FROM digital_downloads dd
         JOIN order_items oi ON oi.id = dd.order_item_id
         WHERE oi.order_id IN ($placeholders)"
    );
    $stmt->execute($orderIds);
    $dlRows = $stmt->fetchAll();
    foreach ($dlRows as $dlRow) {
        $orderDownloads[(int)$dlRow['order_item_id']] = $dlRow;
    }
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header">
        <h1>Purchases</h1>
        <p>Track completed orders, downloads, and shipping updates.</p>
    </div>

    <div class="card mb-20">
        <div class="card-body">
            <form method="GET" action="/admin/purchases.php" class="form-inline">
                <div class="form-group flex-1">
                    <label class="sr-only" for="order-search">Order number</label>
                    <input type="text" id="order-search" name="q" class="form-control" placeholder="Search by order number" value="<?php echo e($searchQuery); ?>">
                </div>
                <button type="submit" class="btn btn-primary btn-sm">Search</button>
                <?php if ($searchQuery !== ''): ?>
                    <a href="/admin/purchases.php" class="btn btn-outline btn-sm">Clear</a>
                <?php endif; ?>
            </form>
        </div>
    </div>

    <?php if ($message): ?>
        <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>
    <?php if ($error): ?>
        <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <div class="card">
        <div class="card-body">
            <?php if (empty($orders)): ?>
                <div class="empty-state">No purchases found.</div>
            <?php else: ?>
                <div class="table-container">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Order</th>
                                <th>User</th>
                                <th>Items</th>
                                <th>Total</th>
                                <th>Status</th>
                                <th>Download</th>
                                <th>Shipping</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($orders as $order): ?>
                                <?php
                                    $items = $orderItems[$order['id']] ?? [];
                                $symbol = CurrencyService::getSymbolFor($order['currency'] ?? 'USD');
                                $total = number_format((float)($order['total_amount'] ?? 0), 2);
                                $hasPhysical = false;
                                foreach ($items as $itemCheck) {
                                    if (empty($itemCheck['is_digital'])) {
                                        $hasPhysical = true;
                                        break;
                                    }
                                }
                                ?>
                                <tr>
                                    <td>
                                        <strong>#<?php echo e($order['order_number']); ?></strong><br>
                                        <span class="text-muted"><?php echo !empty($order['created_at']) ? date('M j, Y', strtotime($order['created_at'])) : ''; ?></span>
                                    </td>
                                    <td>
                                        <div><?php echo e($order['display_name'] ?: $order['nickname'] ?: 'Customer'); ?></div>
                                        <div class="text-muted"><?php echo e($order['email'] ?? ''); ?></div>
                                    </td>
                                    <td>
                                        <ul class="list-plain">
                                            <?php foreach ($items as $item): ?>
                                                <li>
                                                    <?php echo e($item['product_name'] ?? 'Product'); ?>
                                                    <?php if (!empty($item['variant_label'])): ?>
                                                        <span class="text-muted"><?php echo e('(' . $item['variant_label'] . ')'); ?></span>
                                                    <?php endif; ?>
                                                    <span class="text-muted">x<?php echo (int)($item['quantity'] ?? 1); ?></span>
                                                    <?php if (!empty($item['is_digital'])): ?>
                                                        <span class="status-badge status-info">Digital</span>
                                                        <?php
                                                            $dlInfo = $orderDownloads[(int)$item['id']] ?? null;
                                                            $dlCount = $dlInfo ? (int)$dlInfo['download_count'] : 0;
                                                        ?>
                                                        <?php if ($dlCount > 0): ?>
                                                            <span class="status-badge status-badge-warning">Downloaded (<?php echo $dlCount; ?>x)</span>
                                                        <?php else: ?>
                                                            <span class="status-badge status-badge-refundable">Refundable</span>
                                                        <?php endif; ?>
                                                    <?php endif; ?>
                                                </li>
                                            <?php endforeach; ?>
                                        </ul>
                                    </td>
                                    <td><?php echo e($symbol . $total); ?></td>
                                <td>
                                    <span class="status-badge status-active"><?php echo e(ucfirst($order['status'] ?? 'paid')); ?></span>
                                </td>
                                <td>
                                    <?php
                                        // Aggregate download status for all digital items in this order
                                        $orderHasDigital = false;
                                        $allDigitalDownloaded = true;
                                        foreach ($items as $dlItem) {
                                            if (!empty($dlItem['is_digital'])) {
                                                $orderHasDigital = true;
                                                $dlCheck = $orderDownloads[(int)$dlItem['id']] ?? null;
                                                if (!$dlCheck || (int)$dlCheck['download_count'] === 0) {
                                                    $allDigitalDownloaded = false;
                                                }
                                            }
                                        }
                                    ?>
                                    <?php if ($orderHasDigital): ?>
                                        <?php if ($allDigitalDownloaded): ?>
                                            <span class="status-badge status-badge-warning">Downloaded</span>
                                        <?php else: ?>
                                            <span class="status-badge status-badge-refundable">Refundable</span>
                                        <?php endif; ?>
                                    <?php else: ?>
                                        <span class="text-muted">&mdash;</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if (!$hasPhysical): ?>
                                        <span class="text-muted">Digital only</span>
                                    <?php else: ?>
                                        <?php
                                            $shippingStatusId = 'shipping-status-' . (int)$order['id'];
                                        $trackingNumberId = 'tracking-number-' . (int)$order['id'];
                                        $trackingUrlId = 'tracking-url-' . (int)$order['id'];
                                        ?>
                                        <form method="POST" action="/admin/purchases.php" class="form-inline">
                                            <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                                            <input type="hidden" name="action" value="update_shipping">
                                            <input type="hidden" name="order_id" value="<?php echo (int)$order['id']; ?>">
                                            <div class="form-group">
                                                <label class="sr-only" for="<?php echo e($shippingStatusId); ?>">Status</label>
                                                <select id="<?php echo e($shippingStatusId); ?>" name="shipping_status" class="form-control">
                                                    <?php
                                                        $currentStatus = $order['shipping_status'] ?? 'pending';
                                        $shippingOptions = ['pending', 'processing', 'shipped', 'delivered', 'cancelled'];
                                        foreach ($shippingOptions as $option):
                                            ?>
                                                        <option value="<?php echo e($option); ?>" <?php echo $currentStatus === $option ? 'selected' : ''; ?>><?php echo e(ucfirst($option)); ?></option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </div>
                                            <div class="form-group">
                                                <label class="sr-only" for="<?php echo e($trackingNumberId); ?>">Tracking</label>
                                                <input type="text" id="<?php echo e($trackingNumberId); ?>" name="tracking_number" class="form-control" placeholder="Tracking #" value="<?php echo e($order['tracking_number'] ?? ''); ?>">
                                            </div>
                                            <div class="form-group">
                                                <label class="sr-only" for="<?php echo e($trackingUrlId); ?>">Tracking URL</label>
                                                <input type="text" id="<?php echo e($trackingUrlId); ?>" name="tracking_url" class="form-control" placeholder="Tracking URL" value="<?php echo e($order['tracking_url'] ?? ''); ?>">
                                            </div>
                                            <button type="submit" class="btn btn-secondary btn-sm">Save</button>
                                        </form>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

</div>

<?php AdminLayout::renderFooter(); ?>
