<?php
/**
 * Admin Panel - Products
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Admin\AdminLayout;
use NewSite\Auth\AdminAuth;
use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Email\TransactionalEmail;
use NewSite\Logging\LogService;
use NewSite\Shop\CurrencyService;
use NewSite\Util\SlugGenerator;

AdminAuth::requireLogin();

$db = DatabaseManager::getWriteConnection();
$action = $_GET['action'] ?? 'list';
$message = '';
$error = '';

$collections = $db->query("SELECT id, title FROM collections ORDER BY title")->fetchAll();

// Ensure a default Product page exists with a Product section
$defaultProductPageId = null;
$defaultProductPageSlug = 'product';
$defaultProductPageTitle = 'Product';
$defaultProductPageMetaTitle = 'Product';
$defaultProductPageMetaDescription = 'Product details page';

$defaultPageStmt = $db->prepare("SELECT id FROM pages WHERE slug = ? LIMIT 1");
$defaultPageStmt->execute([$defaultProductPageSlug]);
$defaultPageRow = $defaultPageStmt->fetch();
if (!$defaultPageRow) {
    $insertStmt = $db->prepare("INSERT INTO pages (title, slug, content, meta_title, meta_description, is_published) VALUES (?, ?, ?, ?, ?, ?)");
    $insertStmt->execute([$defaultProductPageTitle, $defaultProductPageSlug, '', $defaultProductPageMetaTitle, $defaultProductPageMetaDescription, 1]);
    $defaultProductPageId = (int)DbHelper::lastInsertId($db, 'pages');
} else {
    $defaultProductPageId = (int)$defaultPageRow['id'];
}

if ($defaultProductPageId > 0) {
    $sectionCheck = $db->prepare("SELECT id FROM sections WHERE page_id = ? AND section_type = 'products' LIMIT 1");
    $sectionCheck->execute([$defaultProductPageId]);
    if (!$sectionCheck->fetch()) {
        $defaultSectionSettings = [
            'section_title' => 'Products',
            'title_align' => 'center',
            'product_id' => '',
            'accordion_style' => 'separated',
            'content_align' => 'left',
            'variant_style' => 'select',
            'show_media' => true,
            'show_description' => true,
            'show_price' => true
        ];
        $orderStmt = $db->prepare("SELECT MAX(sort_order) FROM sections WHERE page_id = ?");
        $orderStmt->execute([$defaultProductPageId]);
        $maxOrder = (int)$orderStmt->fetchColumn();
        $insertSection = $db->prepare("INSERT INTO sections (page_id, section_type, settings, sort_order) VALUES (?, ?, ?, ?)");
        $insertSection->execute([$defaultProductPageId, 'products', json_encode($defaultSectionSettings), $maxOrder + 1]);
    }
}

// Pages that include product section
$pageStmt = $db->prepare("SELECT DISTINCT p.id, p.title, p.slug FROM pages p JOIN sections s ON s.page_id = p.id WHERE s.section_type = 'products' ORDER BY p.title");
$pageStmt->execute();
$productPages = $pageStmt->fetchAll();
$requiresPage = !empty($productPages);

if (!function_exists('buildUniqueProductSlug')) {
    function buildUniqueProductSlug(PDO $db, string $slugInput, string $name, int $excludeId = 0): string
    {
        $baseSlug = SlugGenerator::generate($slugInput);
        if ($baseSlug === '') {
            $baseSlug = SlugGenerator::generate($name);
        }
        if ($baseSlug === '') {
            $baseSlug = 'product';
        }

        $candidate = $baseSlug;
        $suffix = 2;
        $existsStmt = $db->prepare("SELECT id FROM products WHERE product_slug = ? AND id <> ? LIMIT 1");

        while (true) {
            $existsStmt->execute([$candidate, $excludeId]);
            if (!$existsStmt->fetch()) {
                return $candidate;
            }
            $candidate = $baseSlug . '-' . $suffix;
            $suffix++;
        }
    }
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_GET['action'] ?? 'save';

    if ($action === 'delete') {
        verifyCSRF();
        $id = $_POST['id'] ?? null;
        if ($id) {
            $stmt = $db->prepare("DELETE FROM products WHERE id = ?");
            $stmt->execute([$id]);
            header('Location: /admin/products.php?message=' . urlencode('Product deleted.'));
            exit;
        }
    }

    verifyCSRF();
    $productId = $_POST['id'] ?? null;
    $name = trim($_POST['name'] ?? '');
    $productSlugInput = trim((string)($_POST['product_slug'] ?? ''));
    $description = trim($_POST['description'] ?? '');
    $metaTitle = trim($_POST['meta_title'] ?? '');
    $metaDescription = trim($_POST['meta_description'] ?? '');
    $price = trim($_POST['price'] ?? '');
    $mediaUrls = $_POST['media_urls'] ?? [];
    $mediaUrls = array_values(array_filter(array_map('trim', (array)$mediaUrls), function ($url) {
        return $url !== '';
    }));
    $mediaUrl = $mediaUrls[0] ?? '';
    $stockUnlimited = isset($_POST['stock_unlimited']);
    $stockInput = trim($_POST['stock'] ?? '');
    $stock = null;
    if (!$stockUnlimited) {
        $stock = $stockInput === '' ? 0 : max(0, (int)$stockInput);
    }
    $isFeatured = isset($_POST['is_featured']) ? 1 : 0;
    $salesCount = max(0, (int)($_POST['sales_count'] ?? 0));
    $actionType = strtolower(trim($_POST['action_type'] ?? 'cart'));
    if (!in_array($actionType, ['cart', 'download', 'external_link'], true)) {
        $actionType = 'cart';
    }
    $downloadUrl = trim($_POST['download_url'] ?? '');
    $externalLinkLabel = trim($_POST['external_link_label'] ?? '');
    $pageId = $_POST['page_id'] !== '' ? (int)$_POST['page_id'] : null;
    if (!$pageId && $defaultProductPageId) {
        $pageId = $defaultProductPageId;
    }
    $collectionId = $_POST['collection_id'] !== '' ? (int)$_POST['collection_id'] : null;
    $quantityLimit = $_POST['quantity_limit'] ?? 'single';
    $quantityMax = $quantityLimit === 'unlimited' ? null : 1;
    $isActive = isset($_POST['is_active']) ? 1 : 0;
    $allowComments = isset($_POST['allow_comments']) ? 1 : 0;
    $blocksInput = $_POST['blocks'] ?? [];
    $variantsInput = $_POST['variants'] ?? [];
    $attributesInput = $_POST['attributes'] ?? [];
    $defaultVariantIndex = isset($_POST['variant_default']) ? (int)$_POST['variant_default'] : -1;

    // Version fields for digital products
    $newVersion = trim($_POST['current_version'] ?? '1.0');
    $versionChangelog = trim($_POST['version_changelog'] ?? '');
    $notifyBuyers = isset($_POST['notify_buyers']);

    // Check if product type is locked (has purchases and is digital non-free)
    $productTypeLocked = false;
    $existingProduct = null;
    if ($productId) {
        $stmt = $db->prepare("SELECT p.*, (SELECT COUNT(*) FROM digital_downloads dd WHERE dd.product_id = p.id) AS download_count FROM products p WHERE p.id = ?");
        $stmt->execute([$productId]);
        $existingProduct = $stmt->fetch();
        if ($existingProduct) {
            $existingActionType = $existingProduct['action_type'] ?? 'cart';
            $existingPrice = CurrencyService::parsePrice($existingProduct['price'] ?? '0');
            $hasPurchases = (int)($existingProduct['download_count'] ?? 0) > 0;
            // Lock if: is digital, has price > 0, and has been purchased
            if ($existingActionType === 'download' && $existingPrice > 0 && $hasPurchases) {
                $productTypeLocked = true;
                // Force the action type to remain download
                $actionType = 'download';
            }
        }
    }

    // Check if any variant provides a download/external link (skip main product URL requirement)
    $hasVariantWithUrl = false;
    if (is_array($variantsInput)) {
        foreach ($variantsInput as $v) {
            $vLabel = trim($v['label'] ?? '');
            $vAction = strtolower(trim($v['action_type'] ?? ''));
            $vUrl = trim($v['download_url'] ?? '');
            if ($vLabel !== '' && in_array($vAction, ['download', 'external_link'], true) && $vUrl !== '') {
                $hasVariantWithUrl = true;
                break;
            }
        }
    }

    if ($name === '') {
        $error = 'Product name is required.';
    } elseif (in_array($actionType, ['download', 'external_link'], true) && $downloadUrl === '' && !$hasVariantWithUrl) {
        $error = $actionType === 'external_link' ? 'Add a link to open for the external button.' : 'Select a file for the download button.';
    } elseif ($requiresPage && !$pageId) {
        $error = 'Select a page that contains a Product section.';
    } else {
        try {
            $existingSlug = trim((string)($existingProduct['product_slug'] ?? ''));
            $slugSeed = $productSlugInput !== '' ? $productSlugInput : $existingSlug;
            $productSlug = buildUniqueProductSlug($db, $slugSeed, $name, (int)$productId);

            $versionChanged = false;
            $oldVersion = null;
            $notificationSent = 0;
            $notificationFailed = 0;
            $notificationRequested = false;

            if ($productId) {
                $now = DbHelper::nowString();

                // Check if version changed for digital products
                if ($existingProduct && $actionType === 'download') {
                    $oldVersion = trim((string)($existingProduct['current_version'] ?? '1.0'));
                    $versionChanged = strcasecmp($oldVersion, $newVersion) !== 0;
                }

                $stmt = $db->prepare("UPDATE products SET name = ?, product_slug = ?, description = ?, meta_title = ?, meta_description = ?, price = ?, media_url = ?, action_type = ?, download_url = ?, external_link_label = ?, page_id = ?, collection_id = ?, stock = ?, quantity_max = ?, is_featured = ?, sales_count = ?, is_active = ?, allow_comments = ?, current_version = ?, version_changelog = ?, updated_at = ? WHERE id = ?");
                $stmt->execute([$name, $productSlug, $description, $metaTitle, $metaDescription, $price, $mediaUrl, $actionType, $downloadUrl, $externalLinkLabel, $pageId, $collectionId, $stock, $quantityMax, $isFeatured, $salesCount, $isActive, $allowComments, $newVersion, $versionChangelog, $now, $productId]);

                // If version changed, create version history and optionally notify buyers
                if ($versionChanged && $actionType === 'download') {
                    $stmt = $db->prepare("INSERT INTO product_version_history (product_id, version, changelog, file_url, created_at) VALUES (?, ?, ?, ?, ?)");
                    $stmt->execute([$productId, $newVersion, $versionChangelog, $downloadUrl, $now]);

                    if ($notifyBuyers) {
                        $notificationRequested = true;
                        $emailResult = TransactionalEmail::sendProductUpdate($productId, $newVersion, $versionChangelog);
                        $notificationSent += (int)($emailResult['sent'] ?? 0);
                        $notificationFailed += (int)($emailResult['failed'] ?? 0);
                    }

                    $message = 'Product updated successfully. Version ' . e($newVersion) . ' published.';
                } else {
                    $message = 'Product updated successfully.';
                }
            } else {
                $stmt = $db->prepare("INSERT INTO products (name, product_slug, description, meta_title, meta_description, price, media_url, action_type, download_url, external_link_label, page_id, collection_id, stock, quantity_max, is_featured, sales_count, is_active, allow_comments, current_version, version_changelog) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$name, $productSlug, $description, $metaTitle, $metaDescription, $price, $mediaUrl, $actionType, $downloadUrl, $externalLinkLabel, $pageId, $collectionId, $stock, $quantityMax, $isFeatured, $salesCount, $isActive, $allowComments, $newVersion, $versionChangelog]);
                $productId = DbHelper::lastInsertId($db, 'products');

                // Create initial version history for digital products
                if ($actionType === 'download') {
                    $now = DbHelper::nowString();
                    $stmt = $db->prepare("INSERT INTO product_version_history (product_id, version, changelog, file_url, created_at) VALUES (?, ?, ?, ?, ?)");
                    $stmt->execute([$productId, $newVersion, $versionChangelog, $downloadUrl, $now]);
                }

                $message = 'Product created successfully.';
            }

            if ($productId) {
                $db->prepare("DELETE FROM product_media WHERE product_id = ?")->execute([$productId]);
                foreach ($mediaUrls as $index => $url) {
                    $stmt = $db->prepare("INSERT INTO product_media (product_id, media_url, sort_order) VALUES (?, ?, ?)");
                    $stmt->execute([$productId, $url, $index]);
                }

                $db->prepare("DELETE FROM product_blocks WHERE product_id = ?")->execute([$productId]);
                if (is_array($blocksInput)) {
                    $position = 0;
                    foreach ($blocksInput as $block) {
                        $title = trim($block['title'] ?? '');
                        $content = trim($block['content'] ?? '');
                        if ($title === '' && $content === '') {
                            continue;
                        }
                        $stmt = $db->prepare("INSERT INTO product_blocks (product_id, title, content, sort_order) VALUES (?, ?, ?, ?)");
                        $stmt->execute([$productId, $title, $content, $position]);
                        $position++;
                    }
                }

                // Get existing variants to preserve IDs (important for purchase tracking)
                $existingVariants = $db->prepare("SELECT id, action_type, current_version FROM product_variants WHERE product_id = ?");
                $existingVariants->execute([$productId]);
                $existingVariantRows = $existingVariants->fetchAll();
                $existingVariantIds = array_column($existingVariantRows, 'id');
                $existingVariantMap = [];
                foreach ($existingVariantRows as $variantRow) {
                    $existingVariantMap[(int)$variantRow['id']] = $variantRow;
                }

                // Get locked variants (have digital purchases)
                $lockedVariantIds = [];
                if (!empty($existingVariantIds)) {
                    $placeholders = implode(',', array_fill(0, count($existingVariantIds), '?'));
                    $lockCheck = $db->prepare("SELECT DISTINCT variant_id FROM order_items WHERE variant_id IN ($placeholders) AND is_digital = 1");
                    $lockCheck->execute($existingVariantIds);
                    $lockedVariantIds = array_column($lockCheck->fetchAll(), 'variant_id');
                }

                $keptVariantIds = [];
                if (is_array($variantsInput)) {
                    $position = 0;
                    $defaultAssigned = false;
                    $firstVariantId = null;
                    $variantError = '';
                    foreach ($variantsInput as $index => $variant) {
                        $label = trim($variant['label'] ?? '');
                        $variantPrice = trim($variant['price'] ?? '');
                        $variantUnlimited = !empty($variant['unlimited']);
                        $variantStockInput = trim($variant['stock'] ?? '');
                        $variantStock = null;
                        if (!$variantUnlimited) {
                            $variantStock = $variantStockInput === '' ? 0 : max(0, (int)$variantStockInput);
                        }
                        $variantAction = strtolower(trim($variant['action_type'] ?? 'cart'));
                        if (!in_array($variantAction, ['cart', 'download', 'external_link'], true)) {
                            $variantAction = 'cart';
                        }
                        $variantDownload = trim($variant['download_url'] ?? '');
                        $variantExternalLabel = trim($variant['external_link_label'] ?? '');
                        $variantVersion = trim($variant['current_version'] ?? '1.0');
                        $variantChangelog = trim($variant['version_changelog'] ?? '');
                        $existingId = isset($variant['id']) ? (int)$variant['id'] : 0;
                        $variantVersionChanged = false;

                        if ($label === '') {
                            continue;
                        }
                        if (in_array($variantAction, ['download', 'external_link'], true) && $variantDownload === '') {
                            $variantError = $variantAction === 'external_link'
                                ? 'Each external link version needs a URL.'
                                : 'Each download version needs a file selected.';
                            break;
                        }

                        // Enforce action_type lock for purchased digital variants
                        if ($existingId && in_array($existingId, $lockedVariantIds)) {
                            $variantAction = 'download'; // Force keep as download
                        }

                        $isDefault = ((string)$index === (string)$defaultVariantIndex) ? 1 : 0;

                        if ($existingId && in_array($existingId, $existingVariantIds)) {
                            $existingVariant = $existingVariantMap[$existingId] ?? null;
                            if ($existingVariant && $variantAction === 'download') {
                                $oldVariantAction = trim((string)($existingVariant['action_type'] ?? ''));
                                $oldVariantVersion = trim((string)($existingVariant['current_version'] ?? '1.0'));
                                $variantVersionChanged = ($oldVariantAction !== 'download') || (strcasecmp($oldVariantVersion, $variantVersion) !== 0);
                            }

                            // Update existing variant
                            $stmt = $db->prepare("UPDATE product_variants SET label = ?, price = ?, is_default = ?, stock = ?, action_type = ?, download_url = ?, external_link_label = ?, current_version = ?, version_changelog = ?, sort_order = ? WHERE id = ?");
                            $stmt->execute([$label, $variantPrice, $isDefault, $variantStock, $variantAction, $variantDownload, $variantExternalLabel, $variantVersion, $variantChangelog, $position, $existingId]);
                            $keptVariantIds[] = $existingId;
                            $variantId = $existingId;
                        } else {
                            // Insert new variant
                            $stmt = $db->prepare("INSERT INTO product_variants (product_id, label, price, is_default, stock, action_type, download_url, external_link_label, current_version, version_changelog, sort_order) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                            $stmt->execute([$productId, $label, $variantPrice, $isDefault, $variantStock, $variantAction, $variantDownload, $variantExternalLabel, $variantVersion, $variantChangelog, $position]);
                            $variantId = DbHelper::lastInsertId($db, 'product_variants');
                        }

                        if ($notifyBuyers && $variantVersionChanged && $variantAction === 'download') {
                            $notificationRequested = true;
                            $emailResult = TransactionalEmail::sendProductUpdate((int)$productId, $variantVersion, $variantChangelog, (int)$variantId);
                            $notificationSent += (int)($emailResult['sent'] ?? 0);
                            $notificationFailed += (int)($emailResult['failed'] ?? 0);
                        }

                        if (!$firstVariantId) {
                            $firstVariantId = $variantId;
                        }
                        if ($isDefault) {
                            $defaultAssigned = true;
                        }
                        $position++;
                    }

                    if ($variantError !== '') {
                        $error = $variantError;
                    }

                    if (!$defaultAssigned && $firstVariantId) {
                        $stmt = $db->prepare("UPDATE product_variants SET is_default = 1 WHERE id = ?");
                        $stmt->execute([$firstVariantId]);
                    }
                }

                // Delete only variants that were removed (and don't have purchases)
                $toDelete = array_diff($existingVariantIds, $keptVariantIds);
                foreach ($toDelete as $deleteId) {
                    if (!in_array($deleteId, $lockedVariantIds)) {
                        $db->prepare("DELETE FROM product_variants WHERE id = ?")->execute([$deleteId]);
                    }
                }

                $db->prepare("DELETE FROM product_attributes WHERE product_id = ?")->execute([$productId]);
                if (is_array($attributesInput)) {
                    foreach ($attributesInput as $attribute) {
                        $attrName = trim($attribute['name'] ?? '');
                        $attrValue = trim($attribute['value'] ?? '');
                        if ($attrName === '' || $attrValue === '') {
                            continue;
                        }
                        $stmt = $db->prepare("INSERT INTO product_attributes (product_id, name, value) VALUES (?, ?, ?)");
                        $stmt->execute([$productId, $attrName, $attrValue]);
                    }
                }
            }

            if (!$error) {
                if ($notificationRequested) {
                    $message .= ' Buyer notifications: ' . $notificationSent . ' sent';
                    if ($notificationFailed > 0) {
                        $message .= ', ' . $notificationFailed . ' failed';
                    }
                    $message .= '.';
                }
                header('Location: /admin/products.php?message=' . urlencode($message));
                exit;
            }
        } catch (PDOException $e) {
            $error = 'Database error occurred while saving the product.';
            LogService::add('error', 'Product save failed: ' . $e->getMessage(), json_encode(['product_id' => $productId, 'name' => $name]));
        }
    }
}

// Handle delete
if ($action === 'delete' && isset($_GET['id'])) {
    // Moved to POST for CSRF protection
    header('Location: /admin/products.php');
    exit;
}

// Get product for editing
$product = null;
if ($action === 'edit' && isset($_GET['id'])) {
    $stmt = $db->prepare("SELECT * FROM products WHERE id = ?");
    $stmt->execute([$_GET['id']]);
    $product = $stmt->fetch();
    if (!$product) {
        header('Location: /admin/products.php');
        exit;
    }
}

$productMedia = [];
$productBlocks = [];
$productVariants = [];
$productAttributes = [];
if ($product) {
    $stmt = $db->prepare("SELECT * FROM product_media WHERE product_id = ? ORDER BY sort_order ASC, id ASC");
    $stmt->execute([$product['id']]);
    $productMedia = $stmt->fetchAll();
    if (empty($productMedia) && !empty($product['media_url'])) {
        $productMedia = [['media_url' => $product['media_url']]];
    }

    $stmt = $db->prepare("SELECT * FROM product_blocks WHERE product_id = ? ORDER BY sort_order ASC, id ASC");
    $stmt->execute([$product['id']]);
    $productBlocks = $stmt->fetchAll();

    $stmt = $db->prepare("SELECT * FROM product_variants WHERE product_id = ? ORDER BY sort_order ASC, id ASC");
    $stmt->execute([$product['id']]);
    $productVariants = $stmt->fetchAll();

    $stmt = $db->prepare("SELECT * FROM product_attributes WHERE product_id = ? ORDER BY name ASC, value ASC");
    $stmt->execute([$product['id']]);
    $productAttributes = $stmt->fetchAll();
}

// Fetch predefined attribute names for the dropdown
$predefinedAttributeNames = $db->query(
    "SELECT name FROM product_attribute_names ORDER BY sort_order ASC, name ASC"
)->fetchAll(PDO::FETCH_COLUMN);

// Get all products for listing
$products = $db->query("SELECT p.*, pg.title AS page_title, pg.slug AS page_slug, c.title AS collection_title FROM products p LEFT JOIN pages pg ON pg.id = p.page_id LEFT JOIN collections c ON c.id = p.collection_id ORDER BY p.created_at DESC")->fetchAll();

// Check for message
if (isset($_GET['message'])) {
    $message = $_GET['message'];
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <?php if ($action === 'list'): ?>
    <div class="content-header content-header-actions">
        <div>
            <h1>Products</h1>
            <p>Manage products shown in the Product section</p>
        </div>
        <a href="/admin/products.php?action=new" class="btn btn-primary has-tooltip" data-tooltip="Create a new product.">+ New Product</a>
    </div>

    <?php if ($message): ?>
    <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>

    <div class="table-container">
        <table class="data-table">
            <thead>
                <tr>
                    <th>Name</th>
                    <th>Price</th>
                    <th>Collection</th>
                    <th>Page</th>
                    <th>Status</th>
                    <th>Comments</th>
                    <th>Updated</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($products as $p): ?>
                <tr>
                    <td><?php echo e($p['name']); ?></td>
                    <td><?php echo e($p['price']); ?></td>
                    <td>
                        <?php if (!empty($p['collection_title'])): ?>
                            <?php echo e($p['collection_title']); ?>
                        <?php else: ?>
                            <span class="text-muted">Not set</span>
                        <?php endif; ?>
                    </td>
                    <td>
                        <?php if (!empty($p['page_id'])): ?>
                            <code>/<?php echo e($p['page_slug']); ?></code>
                        <?php else: ?>
                            <span class="text-muted">Not set</span>
                        <?php endif; ?>
                    </td>
                    <td>
                        <?php if ($p['is_active']): ?>
                        <span class="status-badge status-active">Active</span>
                        <?php else: ?>
                        <span class="status-badge status-inactive">Hidden</span>
                        <?php endif; ?>
                    </td>
                    <td>
                        <?php if ($p['allow_comments'] ?? 1): ?>
                        <span class="status-badge status-active">Enabled</span>
                        <?php else: ?>
                        <span class="status-badge status-inactive">Disabled</span>
                        <?php endif; ?>
                    </td>
                    <td><?php echo date('M j, Y', strtotime($p['updated_at'] ?? $p['created_at'])); ?></td>
                    <td class="table-actions">
                        <a href="/admin/products.php?action=edit&id=<?php echo $p['id']; ?>" class="btn btn-secondary btn-sm has-tooltip" data-tooltip="Edit this product.">Edit</a>
                        <form method="POST" action="/admin/products.php?action=delete" class="inline-block" data-confirm="Delete this product?">
                            <input type="hidden" name="id" value="<?php echo $p['id']; ?>">
                            <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                            <button type="submit" class="btn btn-danger btn-sm has-tooltip" data-tooltip="Delete this product.">Delete</button>
                        </form>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>

    <?php else: ?>
    <div class="content-header">
        <h1><?php echo $product ? 'Edit Product' : 'New Product'; ?></h1>
        <p><a href="/admin/products.php">← Back to Products</a></p>
    </div>

    <?php if ($error): ?>
    <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <?php
    $productActionType = $product['action_type'] ?? 'cart';
$productActionType = in_array($productActionType, ['cart', 'download', 'external_link'], true) ? $productActionType : 'cart';
$classIsHidden = ' is-hidden';
$productDownloadUrl = $product['download_url'] ?? '';
$productExternalLabel = $product['external_link_label'] ?? '';
$productVersion = $product['current_version'] ?? '1.0';
$productChangelog = $product['version_changelog'] ?? '';
$productSlugValue = trim((string)($product['product_slug'] ?? ''));
if ($productSlugValue === '') {
    $productSlugValue = SlugGenerator::generate((string)($product['name'] ?? ''));
}
$selectedPageId = $product['page_id'] ?? ($defaultProductPageId ?: '');

// Check if product type is locked (digital, has price, has been purchased)
$productTypeLocked = false;
$versionHistory = [];
if ($product) {
    $stmt = $db->prepare("SELECT COUNT(*) FROM digital_downloads WHERE product_id = ?");
    $stmt->execute([$product['id']]);
    $hasPurchases = (int)$stmt->fetchColumn() > 0;
    $existingPrice = CurrencyService::parsePrice($product['price'] ?? '0');
    $productTypeLocked = $productActionType === 'download' && $existingPrice > 0 && $hasPurchases;

    // Fetch version history for digital products
    if ($productActionType === 'download') {
        $stmt = $db->prepare("SELECT version, changelog, file_url, created_at FROM product_version_history WHERE product_id = ? ORDER BY created_at DESC");
        $stmt->execute([$product['id']]);
        $versionHistory = $stmt->fetchAll();
    }
}
?>

    <form method="POST" action="/admin/products.php?action=<?php echo $product ? 'edit' : 'new'; ?><?php echo $product ? '&id=' . $product['id'] : ''; ?>">
        <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
        <?php if ($product): ?>
        <input type="hidden" name="id" value="<?php echo $product['id']; ?>">
        <?php endif; ?>

        <div class="card">
            <div class="card-body">
                <div class="form-row">
                    <div class="form-group">
                        <label for="name">Product Name</label>
                        <input type="text" id="name" name="name" class="form-control" value="<?php echo e($product['name'] ?? ''); ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="product_slug">Product URL Slug</label>
                        <input type="text" id="product_slug" name="product_slug" class="form-control" value="<?php echo e($productSlugValue); ?>" placeholder="community-imageboard-php-sqlite">
                        <small class="form-hint">Optional. Used for clean URLs: <code>/page-slug/product-slug</code>.</small>
                    </div>
                    <div class="form-group">
                        <label for="price">Price</label>
                        <input type="text" id="price" name="price" class="form-control" value="<?php echo e($product['price'] ?? ''); ?>" placeholder="$99">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <?php if ($productTypeLocked): ?>
                            <label for="action_type_locked">Product Type</label>
                            <input type="hidden" name="action_type" value="download">
                            <select id="action_type_locked" class="form-control" disabled>
                                <option value="download" selected>Digital Download</option>
                            </select>
                            <small class="form-hint">Product type is locked because this digital product has been purchased.</small>
                        <?php else: ?>
                            <label for="action_type">Product Type</label>
                            <select id="action_type" name="action_type" class="form-control" data-product-action>
                                <option value="cart" <?php echo $productActionType === 'cart' ? 'selected' : ''; ?>>Physical (Cart)</option>
                                <option value="download" <?php echo $productActionType === 'download' ? 'selected' : ''; ?>>Digital Download</option>
                                <option value="external_link" <?php echo $productActionType === 'external_link' ? 'selected' : ''; ?>>External Link</option>
                            </select>
                        <?php endif; ?>
                    </div>
                    <div class="form-group product-download-field<?php echo in_array($productActionType, ['download', 'external_link'], true) ? '' : $classIsHidden; ?>" id="product-download-field">
                        <label for="download_url" data-download-label="Download File" data-link-label="External Link URL">Download File</label>
                        <input type="text" id="download_url" name="download_url" class="form-control" value="<?php echo e($productDownloadUrl); ?>" placeholder="/admin-file/file.zip" data-download-placeholder="/admin-file/file.zip" data-link-placeholder="https://example.com" data-media-picker="file">
                    </div>
                    <div class="form-group product-link-label-field<?php echo $productActionType === 'external_link' ? '' : $classIsHidden; ?>" id="product-link-label-field">
                        <label for="external_link_label">External Link Button Text</label>
                        <input type="text" id="external_link_label" name="external_link_label" class="form-control" value="<?php echo e($productExternalLabel); ?>" placeholder="Play on Steam">
                        <small class="form-hint">Shown on the product page when the type is External Link.</small>
                    </div>
                </div>

                <!-- Version fields for digital products -->
                <div class="form-row product-version-fields<?php echo $productActionType === 'download' ? '' : $classIsHidden; ?>" id="product-version-fields">
                    <div class="form-group flex-fixed-120">
                        <label for="current_version">Version</label>
                        <input type="text" id="current_version" name="current_version" class="form-control" value="<?php echo e($productVersion); ?>" placeholder="1.0">
                    </div>
                    <div class="form-group flex-1">
                        <label for="version_changelog">What's New / Changelog</label>
                        <textarea id="version_changelog" name="version_changelog" class="form-control" rows="2" placeholder="Describe changes in this version..."><?php echo e($productChangelog); ?></textarea>
                    </div>
                </div>

                <?php if ($product && $productActionType === 'download'): ?>
                <div class="form-row" id="notify-buyers-row">
                    <div class="form-group">
                        <label class="form-check">
                            <input type="checkbox" name="notify_buyers" id="notify_buyers">
                            <span>Notify buyers via email when saving with a new version</span>
                        </label>
                        <small class="form-hint">Sends an email to all customers who purchased this product informing them about the update.</small>
                    </div>
                </div>

                <?php if (!empty($versionHistory)): ?>
                <div class="form-group" id="version-history-section">
                    <div class="form-label">Version History</div>
                    <div class="version-history-list">
                        <?php foreach ($versionHistory as $vh): ?>
                        <div class="version-history-item">
                            <div class="version-history-header">
                                <span class="version-history-number">v<?php echo e($vh['version']); ?></span>
                                <span class="version-history-date"><?php echo date('M j, Y', strtotime($vh['created_at'])); ?></span>
                            </div>
                            <?php if (!empty($vh['changelog'])): ?>
                            <div class="version-history-changelog"><?php echo nl2br(e($vh['changelog'])); ?></div>
                            <?php else: ?>
                            <div class="version-history-changelog text-muted">No changelog provided</div>
                            <?php endif; ?>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>
                <?php endif; ?>

                <div class="form-row">
                    <div class="form-group">
                        <label for="stock">Stock</label>
                        <input type="number" id="stock" name="stock" class="form-control" min="0" value="<?php echo isset($product['stock']) ? (int)$product['stock'] : ''; ?>" placeholder="0">
                    </div>
                    <div class="form-group">
                        <label class="form-check">
                            <input type="checkbox" name="stock_unlimited" <?php echo !isset($product['stock']) || $product['stock'] === null ? 'checked' : ''; ?>>
                            <span>Unlimited stock</span>
                        </label>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label class="form-check">
                            <input type="checkbox" name="is_featured" <?php echo !empty($product['is_featured']) ? 'checked' : ''; ?>>
                            <span>Featured Product</span>
                        </label>
                    </div>
                    <div class="form-group">
                        <label for="sales_count">Best Seller Score</label>
                        <input type="number" id="sales_count" name="sales_count" class="form-control" min="0" value="<?php echo isset($product['sales_count']) ? (int)$product['sales_count'] : 0; ?>" placeholder="0">
                        <small class="form-help">Higher values rank higher in Bestselling sort.</small>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <div class="form-label">Quantity Limit</div>
                        <?php
                    $quantityMax = array_key_exists('quantity_max', (array)$product)
                        ? $product['quantity_max']
                        : 1;
?>
                        <div class="form-check">
                            <input type="radio" name="quantity_limit" value="single" <?php echo $quantityMax === null ? '' : 'checked'; ?>>
                            <span>Max 1 per cart</span>
                        </div>
                        <div class="form-check">
                            <input type="radio" name="quantity_limit" value="unlimited" <?php echo $quantityMax === null ? 'checked' : ''; ?>>
                            <span>Unlimited quantity</span>
                        </div>
                    </div>
                </div>

                <div class="form-group">
                    <div class="form-label">Media Gallery (Slider)</div>
                    <small class="form-help">Supports images and videos. Use "Add From Library" to select multiple at once.</small>
                    <div id="product-media-list">
                        <?php
$mediaIndex = 0;
foreach ($productMedia as $mediaItem):
    $mediaValue = $mediaItem['media_url'] ?? '';
    ?>
                            <div class="card section-item-card mb-2 product-media-item" data-index="<?php echo $mediaIndex; ?>">
                                <div class="card-body section-item-body">
                                    <div class="form-row product-inline-row">
                                        <?php
                        $thumbUrl = '';
    $isYt = preg_match('/youtu\.?be/', $mediaValue);
    if (!$isYt && $mediaValue !== '') {
        $ext = strtolower(pathinfo($mediaValue, PATHINFO_EXTENSION));
        if (in_array($ext, ['jpg','jpeg','png','gif','webp','svg','avif'], true)) {
            $thumbUrl = str_replace('/admin-file/', '/site-file/', $mediaValue);
        }
    }
    ?>
                                        <div class="product-media-thumb">
                                            <img src="<?php echo $thumbUrl !== '' ? e($thumbUrl) : ''; ?>" alt="" class="product-media-thumb-img" <?php echo $thumbUrl === '' ? 'hidden' : ''; ?>>
                                        </div>
                                        <div class="form-group product-inline-grow">
                                            <div class="form-label">Media URL (Image or Video)</div>
                                            <input type="text" name="media_urls[<?php echo $mediaIndex; ?>]" class="form-control" value="<?php echo e($mediaValue); ?>" placeholder="/admin-file/product.jpg or /admin-file/product.mp4" data-media-picker="media">
                                        </div>
                                        <div class="form-group product-inline-actions">
                                            <button type="button" class="btn btn-secondary btn-sm" data-move-media="up" title="Move up"><svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M12 19V5"/><path d="M5 12l7-7 7 7"/></svg></button>
                                            <button type="button" class="btn btn-secondary btn-sm" data-move-media="down" title="Move down"><svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M12 5v14"/><path d="M19 12l-7 7-7-7"/></svg></button>
                                            <button type="button" class="btn btn-danger btn-sm" data-remove="media">Remove</button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php
                            $mediaIndex++;
endforeach;
if ($mediaIndex === 0) {
    ?>
                            <div class="card section-item-card mb-2 product-media-item" data-index="0">
                                <div class="card-body section-item-body">
                                    <div class="form-row product-inline-row">
                                        <div class="product-media-thumb">
                                            <img src="" alt="" class="product-media-thumb-img" hidden>
                                        </div>
                                        <div class="form-group product-inline-grow">
                                            <div class="form-label">Media URL (Image or Video)</div>
                                            <input type="text" name="media_urls[0]" class="form-control" value="" placeholder="/admin-file/product.jpg or /admin-file/product.mp4" data-media-picker="media">
                                        </div>
                                        <div class="form-group product-inline-actions">
                                            <button type="button" class="btn btn-secondary btn-sm" data-move-media="up" title="Move up"><svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M12 19V5"/><path d="M5 12l7-7 7 7"/></svg></button>
                                            <button type="button" class="btn btn-secondary btn-sm" data-move-media="down" title="Move down"><svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M12 5v14"/><path d="M19 12l-7 7-7-7"/></svg></button>
                                            <button type="button" class="btn btn-danger btn-sm" data-remove="media">Remove</button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php } ?>
                    </div>
                    <div class="form-row gap-10">
                        <button type="button" class="btn btn-primary btn-sm" id="add-product-media-library">Add From Library</button>
                        <button type="button" class="btn btn-secondary btn-sm" id="add-product-media">+ Add Media URL</button>
                    </div>
                </div>

                <div class="form-group">
                    <label for="description">Description</label>
                    <textarea id="description" name="description" class="form-control" rows="6"><?php echo e($product['description'] ?? ''); ?></textarea>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="meta_title">SEO Title</label>
                        <input type="text" id="meta_title" name="meta_title" class="form-control" value="<?php echo e($product['meta_title'] ?? ''); ?>" placeholder="Product name for search results">
                        <small class="form-hint">Leave blank to use the product name.</small>
                    </div>
                </div>
                <div class="form-group">
                    <label for="meta_description">SEO Description</label>
                    <textarea id="meta_description" name="meta_description" class="form-control" rows="3" placeholder="Short summary for search results"><?php echo e($product['meta_description'] ?? ''); ?></textarea>
                    <small class="form-hint">Leave blank to use the product description.</small>
                </div>

                <div class="form-group">
                    <div class="form-label">Additional Content Areas (Left Side)</div>
                    <div id="product-blocks-list">
                        <?php
    $blockIndex = 0;
foreach ($productBlocks as $block) {
    ?>
                            <div class="card section-item-card mb-2 product-block-item" data-index="<?php echo $blockIndex; ?>">
                                <div class="card-header section-item-header">
                                    <span>Block <?php echo $blockIndex + 1; ?></span>
                                    <button type="button" class="btn btn-sm btn-danger" data-remove="block">Remove</button>
                                </div>
                                <div class="card-body section-item-body">
                                    <div class="form-group">
                                        <div class="form-label">Block Title</div>
                                        <input type="text" name="blocks[<?php echo $blockIndex; ?>][title]" class="form-control" value="<?php echo e($block['title'] ?? ''); ?>">
                                    </div>
                                    <div class="form-group">
                                        <div class="form-label">Block Content</div>
                                        <textarea name="blocks[<?php echo $blockIndex; ?>][content]" class="form-control" rows="4"><?php echo e($block['content'] ?? ''); ?></textarea>
                                    </div>
                                </div>
                            </div>
                        <?php
        $blockIndex++;
}
?>
                    </div>
                    <button type="button" class="btn btn-secondary btn-sm" id="add-product-block">+ Add Content Block</button>
                </div>

                <div class="form-group">
                    <div class="form-label">Versions (Select Version)</div>
                    <div id="product-variants-list">
                        <?php
$variantIndex = 0;
$hasDefaultVariant = false;
foreach ($productVariants as $variant) {
    if (!empty($variant['is_default'])) {
        $hasDefaultVariant = true;
        break;
    }
}

// Get purchase counts for each variant to determine if type should be locked
$variantPurchaseCounts = [];
if ($product) {
    $variantIds = array_column($productVariants, 'id');
    if (!empty($variantIds)) {
        $placeholders = implode(',', array_fill(0, count($variantIds), '?'));
        $stmt = $db->prepare("SELECT variant_id, COUNT(*) as cnt FROM order_items WHERE variant_id IN ($placeholders) AND is_digital = 1 GROUP BY variant_id");
        $stmt->execute($variantIds);
        foreach ($stmt->fetchAll() as $row) {
            $variantPurchaseCounts[(int)$row['variant_id']] = (int)$row['cnt'];
        }
    }
}

foreach ($productVariants as $variant):
    $variantId = (int)($variant['id'] ?? 0);
    $variantAction = $variant['action_type'] ?? $productActionType;
    $variantAction = in_array($variantAction, ['cart', 'download', 'external_link'], true) ? $variantAction : 'cart';
    $variantDownload = $variant['download_url'] ?? '';
    $variantExternalLabel = $variant['external_link_label'] ?? '';
    $variantVersion = $variant['current_version'] ?? '1.0';
    $variantChangelog = $variant['version_changelog'] ?? '';

    // Check if this variant's type is locked
    $variantPrice = CurrencyService::parsePrice($variant['price'] ?? '');
    $variantHasPurchases = ($variantPurchaseCounts[$variantId] ?? 0) > 0;
    $variantTypeLocked = $variantAction === 'download' && $variantPrice > 0 && $variantHasPurchases;
    ?>
                            <div class="card section-item-card mb-2 product-variant-item" data-index="<?php echo $variantIndex; ?>">
                                <input type="hidden" name="variants[<?php echo $variantIndex; ?>][id]" value="<?php echo $variantId; ?>">
                                <div class="card-body section-item-body">
                                    <div class="form-row product-inline-row">
                                        <div class="form-group product-inline-grow">
                                            <div class="form-label">Version Label</div>
                                            <input type="text" name="variants[<?php echo $variantIndex; ?>][label]" class="form-control" value="<?php echo e($variant['label'] ?? ''); ?>" placeholder="Standard">
                                        </div>
                                        <div class="form-group product-inline-grow">
                                            <div class="form-label">Price (Optional)</div>
                                            <input type="text" name="variants[<?php echo $variantIndex; ?>][price]" class="form-control" value="<?php echo e($variant['price'] ?? ''); ?>" placeholder="$129">
                                        </div>
                                        <div class="form-group product-inline-grow">
                                            <div class="form-label">Stock</div>
                                            <input type="number" name="variants[<?php echo $variantIndex; ?>][stock]" class="form-control" min="0" value="<?php echo isset($variant['stock']) ? (int)$variant['stock'] : ''; ?>" placeholder="0">
                                        </div>
                                        <div class="form-group product-inline-grow">
                                            <div class="form-label">Product Type</div>
                                            <?php if ($variantTypeLocked): ?>
                                                <input type="hidden" name="variants[<?php echo $variantIndex; ?>][action_type]" value="download">
                                                <select class="form-control" disabled title="Locked: this version has been purchased">
                                                    <option value="download" selected>Digital Download 🔒</option>
                                                </select>
                                            <?php else: ?>
                                                <select name="variants[<?php echo $variantIndex; ?>][action_type]" class="form-control" data-variant-action>
                                                    <option value="cart" <?php echo $variantAction === 'cart' ? 'selected' : ''; ?>>Physical (Cart)</option>
                                                    <option value="download" <?php echo $variantAction === 'download' ? 'selected' : ''; ?>>Digital Download</option>
                                                    <option value="external_link" <?php echo $variantAction === 'external_link' ? 'selected' : ''; ?>>External Link</option>
                                                </select>
                                            <?php endif; ?>
                                        </div>
                                        <div class="form-group product-inline-grow variant-download-field<?php echo in_array($variantAction, ['download', 'external_link'], true) ? '' : $classIsHidden; ?>">
                                            <div class="form-label" data-download-label="Download File" data-link-label="External Link URL">Download File</div>
                                            <input type="text" name="variants[<?php echo $variantIndex; ?>][download_url]" class="form-control" value="<?php echo e($variantDownload); ?>" placeholder="/admin-file/file.zip" data-download-placeholder="/admin-file/file.zip" data-link-placeholder="https://example.com" data-media-picker="file">
                                        </div>
                                        <div class="form-group product-inline-grow variant-link-label-field<?php echo $variantAction === 'external_link' ? '' : $classIsHidden; ?>">
                                            <div class="form-label">External Link Button Text</div>
                                            <input type="text" name="variants[<?php echo $variantIndex; ?>][external_link_label]" class="form-control" value="<?php echo e($variantExternalLabel); ?>" placeholder="Play on Steam">
                                        </div>
                                        <div class="form-group product-inline-small variant-version-field<?php echo $variantAction === 'download' ? '' : $classIsHidden; ?>">
                                            <div class="form-label">Version</div>
                                            <input type="text" name="variants[<?php echo $variantIndex; ?>][current_version]" class="form-control" value="<?php echo e($variantVersion); ?>" placeholder="1.0">
                                        </div>
                                        <div class="form-group product-inline-default">
                                            <label class="form-check">
                                                <input type="checkbox" name="variants[<?php echo $variantIndex; ?>][unlimited]" <?php echo !isset($variant['stock']) || $variant['stock'] === null ? 'checked' : ''; ?>>
                                                <span>Unlimited</span>
                                            </label>
                                        </div>
                                        <div class="form-group product-inline-default">
                                            <label class="form-check">
                                                <input type="radio" name="variant_default" value="<?php echo $variantIndex; ?>" <?php echo (!empty($variant['is_default']) || (!$hasDefaultVariant && $variantIndex === 0)) ? 'checked' : ''; ?>>
                                                <span>Default</span>
                                            </label>
                                        </div>
                                        <div class="form-group product-inline-actions">
                                            <button type="button" class="btn btn-danger btn-sm" data-remove="variant">Remove</button>
                                        </div>
                                    </div>
                                    <div class="form-row variant-changelog-row<?php echo $variantAction === 'download' ? '' : $classIsHidden; ?>">
                                        <div class="form-group flex-1">
                                            <div class="form-label">What's New / Changelog</div>
                                            <textarea name="variants[<?php echo $variantIndex; ?>][version_changelog]" class="form-control" rows="2" placeholder="Describe changes in this version..."><?php echo e($variantChangelog); ?></textarea>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php
        $variantIndex++;
endforeach;
?>
                    </div>
                    <button type="button" class="btn btn-secondary btn-sm" id="add-product-variant">+ Add Version</button>
                </div>

                <div class="form-group">
                    <div class="form-label">Filter Attributes (e.g. Color, Size) — <a href="/admin/attributes.php">Manage attribute names</a></div>
                    <div id="product-attributes-list">
                        <?php
$attributeIndex = 0;
foreach ($productAttributes as $attribute) {
    $currentAttrName = trim((string)($attribute['name'] ?? ''));
    ?>
                            <div class="card section-item-card mb-2 product-attribute-item" data-index="<?php echo $attributeIndex; ?>">
                                <div class="card-body section-item-body">
                                    <div class="form-row product-inline-row">
                                        <div class="form-group product-inline-grow">
                                            <div class="form-label">Attribute Name</div>
                                            <select name="attributes[<?php echo $attributeIndex; ?>][name]" class="form-control">
                                                <option value="">-- Select --</option>
                                                <?php foreach ($predefinedAttributeNames as $pan): ?>
                                                    <option value="<?php echo e($pan); ?>"<?php echo $pan === $currentAttrName ? ' selected' : ''; ?>><?php echo e($pan); ?></option>
                                                <?php endforeach; ?>
                                                <?php if ($currentAttrName !== '' && !in_array($currentAttrName, $predefinedAttributeNames, true)): ?>
                                                    <option value="<?php echo e($currentAttrName); ?>" selected><?php echo e($currentAttrName); ?> (custom)</option>
                                                <?php endif; ?>
                                            </select>
                                        </div>
                                        <div class="form-group product-inline-grow">
                                            <div class="form-label">Attribute Value</div>
                                            <input type="text" name="attributes[<?php echo $attributeIndex; ?>][value]" class="form-control" value="<?php echo e($attribute['value'] ?? ''); ?>" placeholder="Purple">
                                        </div>
                                        <div class="form-group product-inline-actions">
                                            <button type="button" class="btn btn-danger btn-sm" data-remove="attribute">Remove</button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php
        $attributeIndex++;
}
if ($attributeIndex === 0) {
    ?>
                            <div class="card section-item-card mb-2 product-attribute-item" data-index="0">
                                <div class="card-body section-item-body">
                                    <div class="form-row product-inline-row">
                                        <div class="form-group product-inline-grow">
                                            <div class="form-label">Attribute Name</div>
                                            <select name="attributes[0][name]" class="form-control">
                                                <option value="">-- Select --</option>
                                                <?php foreach ($predefinedAttributeNames as $pan): ?>
                                                    <option value="<?php echo e($pan); ?>"><?php echo e($pan); ?></option>
                                                <?php endforeach; ?>
                                            </select>
                                        </div>
                                        <div class="form-group product-inline-grow">
                                            <div class="form-label">Attribute Value</div>
                                            <input type="text" name="attributes[0][value]" class="form-control" value="" placeholder="Purple">
                                        </div>
                                        <div class="form-group product-inline-actions">
                                            <button type="button" class="btn btn-danger btn-sm" data-remove="attribute">Remove</button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php } ?>
                    </div>
                    <button type="button" class="btn btn-secondary btn-sm" id="add-product-attribute">+ Add Attribute</button>
                </div>

                <template id="product-media-template">
                    <div class="card section-item-card mb-2 product-media-item" data-index="__INDEX__">
                        <div class="card-body section-item-body">
                            <div class="form-row product-inline-row">
                                <div class="product-media-thumb">
                                    <img src="" alt="" class="product-media-thumb-img" hidden>
                                </div>
                                <div class="form-group product-inline-grow">
                                    <div class="form-label">Media URL (Image or Video)</div>
                                    <input type="text" name="media_urls[__INDEX__]" class="form-control" value="" placeholder="/admin-file/product.jpg or /admin-file/product.mp4" data-media-picker="media">
                                </div>
                                <div class="form-group product-inline-actions">
                                    <button type="button" class="btn btn-secondary btn-sm" data-move-media="up" title="Move up"><svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M12 19V5"/><path d="M5 12l7-7 7 7"/></svg></button>
                                    <button type="button" class="btn btn-secondary btn-sm" data-move-media="down" title="Move down"><svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><path d="M12 5v14"/><path d="M19 12l-7 7-7-7"/></svg></button>
                                    <button type="button" class="btn btn-danger btn-sm" data-remove="media">Remove</button>
                                </div>
                            </div>
                        </div>
                    </div>
                </template>

                <template id="product-block-template">
                    <div class="card section-item-card mb-2 product-block-item" data-index="__INDEX__">
                        <div class="card-header section-item-header">
                            <span>Block __NUMBER__</span>
                            <button type="button" class="btn btn-sm btn-danger" data-remove="block">Remove</button>
                        </div>
                        <div class="card-body section-item-body">
                            <div class="form-group">
                                <div class="form-label">Block Title</div>
                                <input type="text" name="blocks[__INDEX__][title]" class="form-control" value="">
                            </div>
                            <div class="form-group">
                                <div class="form-label">Block Content</div>
                                <textarea name="blocks[__INDEX__][content]" class="form-control" rows="4"></textarea>
                            </div>
                        </div>
                    </div>
                </template>

                <template id="product-variant-template">
                    <div class="card section-item-card mb-2 product-variant-item" data-index="__INDEX__">
                        <div class="card-body section-item-body">
                            <div class="form-row product-inline-row">
                                <div class="form-group product-inline-grow">
                                    <div class="form-label">Version Label</div>
                                    <input type="text" name="variants[__INDEX__][label]" class="form-control" value="" placeholder="Standard">
                                </div>
                                <div class="form-group product-inline-grow">
                                    <div class="form-label">Price (Optional)</div>
                                    <input type="text" name="variants[__INDEX__][price]" class="form-control" value="" placeholder="$129">
                                </div>
                                <div class="form-group product-inline-grow">
                                    <div class="form-label">Stock</div>
                                    <input type="number" name="variants[__INDEX__][stock]" class="form-control" min="0" value="" placeholder="0">
                                </div>
                                <div class="form-group product-inline-grow">
                                    <div class="form-label">Product Type</div>
                                    <select name="variants[__INDEX__][action_type]" class="form-control" data-variant-action>
                                        <option value="cart" selected>Physical (Cart)</option>
                                        <option value="download">Digital Download</option>
                                        <option value="external_link">External Link</option>
                                    </select>
                                </div>
                                <div class="form-group product-inline-grow variant-download-field is-hidden">
                                    <div class="form-label" data-download-label="Download File" data-link-label="External Link URL">Download File</div>
                                    <input type="text" name="variants[__INDEX__][download_url]" class="form-control" value="" placeholder="/admin-file/file.zip" data-download-placeholder="/admin-file/file.zip" data-link-placeholder="https://example.com" data-media-picker="file">
                                </div>
                                <div class="form-group product-inline-grow variant-link-label-field is-hidden">
                                    <div class="form-label">External Link Button Text</div>
                                    <input type="text" name="variants[__INDEX__][external_link_label]" class="form-control" value="" placeholder="Play on Steam">
                                </div>
                                <div class="form-group product-inline-small variant-version-field is-hidden">
                                    <div class="form-label">Version</div>
                                    <input type="text" name="variants[__INDEX__][current_version]" class="form-control" value="1.0" placeholder="1.0">
                                </div>
                                <div class="form-group product-inline-default">
                                    <label class="form-check">
                                        <input type="checkbox" name="variants[__INDEX__][unlimited]" checked>
                                        <span>Unlimited</span>
                                    </label>
                                </div>
                                <div class="form-group product-inline-default">
                                    <label class="form-check">
                                        <input type="radio" name="variant_default" value="__INDEX__">
                                        <span>Default</span>
                                    </label>
                                </div>
                                <div class="form-group product-inline-actions">
                                    <button type="button" class="btn btn-danger btn-sm" data-remove="variant">Remove</button>
                                </div>
                            </div>
                            <div class="form-row variant-changelog-row is-hidden">
                                <div class="form-group flex-1">
                                    <div class="form-label">What's New / Changelog</div>
                                    <textarea name="variants[__INDEX__][version_changelog]" class="form-control" rows="2" placeholder="Describe changes in this version..."></textarea>
                                </div>
                            </div>
                        </div>
                    </div>
                </template>

                <template id="product-attribute-template">
                    <div class="card section-item-card mb-2 product-attribute-item" data-index="__INDEX__">
                        <div class="card-body section-item-body">
                            <div class="form-row product-inline-row">
                                <div class="form-group product-inline-grow">
                                    <div class="form-label">Attribute Name</div>
                                    <select name="attributes[__INDEX__][name]" class="form-control">
                                        <option value="">-- Select --</option>
                                        <?php foreach ($predefinedAttributeNames as $pan): ?>
                                            <option value="<?php echo e($pan); ?>"><?php echo e($pan); ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="form-group product-inline-grow">
                                    <div class="form-label">Attribute Value</div>
                                    <input type="text" name="attributes[__INDEX__][value]" class="form-control" value="" placeholder="Purple">
                                </div>
                                <div class="form-group product-inline-actions">
                                    <button type="button" class="btn btn-danger btn-sm" data-remove="attribute">Remove</button>
                                </div>
                            </div>
                        </div>
                    </div>
                </template>

                <div class="form-group">
                    <label for="collection_id">Collection</label>
                    <select id="collection_id" name="collection_id" class="form-control">
                        <option value="">-- Select a collection --</option>
                        <?php foreach ($collections as $collection): ?>
                        <option value="<?php echo $collection['id']; ?>" <?php echo ($product['collection_id'] ?? '') == $collection['id'] ? 'selected' : ''; ?>>
                            <?php echo e($collection['title']); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label for="page_id">Product Page</label>
                    <select id="page_id" name="page_id" class="form-control" <?php echo $requiresPage ? 'required' : ''; ?>>
                        <option value="">-- Select a page with Product section --</option>
                        <?php foreach ($productPages as $pg): ?>
                        <option value="<?php echo $pg['id']; ?>" <?php echo (string)$selectedPageId === (string)$pg['id'] ? 'selected' : ''; ?>>
                            <?php echo e($pg['title']); ?> (/<?php echo e($pg['slug']); ?>)
                        </option>
                        <?php endforeach; ?>
                    </select>
                    <small class="form-hint">To see more pages here, create a page and add the Product section in the Theme Editor.</small>
                    <?php if (empty($productPages)): ?>
                        <div class="text-muted mt-1">No pages with a Product section yet.</div>
                    <?php endif; ?>
                </div>

                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="allow_comments" <?php echo ($product['allow_comments'] ?? 1) ? 'checked' : ''; ?>>
                        <span>Enable product comments</span>
                    </label>
                    <small class="form-hint">Allow logged-in users to post comments below this product page.</small>
                </div>

                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="is_active" <?php echo ($product['is_active'] ?? 1) ? 'checked' : ''; ?>>
                        <span>Visible</span>
                    </label>
                </div>
            </div>

            <div class="card-footer">
                <a href="/admin/products.php" class="btn btn-secondary">Cancel</a>
                <button type="submit" class="btn btn-primary">Save Product</button>
            </div>
        </div>
    </form>
    <?php endif; ?>
</div>

<?php AdminLayout::renderFooter(); ?>
