<?php
/**
 * Admin Panel - Pages Management
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Admin\AdminLayout;
use NewSite\Auth\AdminAuth;
use NewSite\Config\SetupService;
use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Logging\LogService;
use NewSite\Util\SlugGenerator;

AdminAuth::requireLogin();

const SQL_INSERT_PAGE_BASIC = "INSERT INTO pages (title, slug, content, meta_title, meta_description, is_published) VALUES (?, ?, ?, ?, ?, ?)";
const SQL_SELECT_MAX_SECTION_ORDER = "SELECT MAX(sort_order) FROM sections WHERE page_id = ?";
const SQL_INSERT_SECTION = "INSERT INTO sections (page_id, section_type, settings, sort_order) VALUES (?, ?, ?, ?)";
const SQL_UPDATE_PAGE_CONTENT_META_BY_ID = "UPDATE pages SET content = ?, meta_title = ?, meta_description = ? WHERE id = ?";

const PAGE_TITLE_PRIVACY_POLICY = 'Privacy Policy';
const PAGE_TITLE_TERMS_OF_SERVICE = 'Terms of Service';
const PAGE_TITLE_REFUND_POLICY = 'Refund Policy';
const PAGE_TITLE_SHIPPING_POLICY = 'Shipping Policy';
const PAGE_TITLE_LEGAL_NOTICE = 'Legal Notice';

const REDIRECT_ADMIN_PAGES = 'Location: /admin/pages.php';
const REDIRECT_ADMIN_PAGES_MESSAGE = 'Location: /admin/pages.php?message=';
const REDIRECT_ADMIN_PAGES_ERROR = 'Location: /admin/pages.php?error=';

$db = DatabaseManager::getWriteConnection();
$action = $_GET['action'] ?? 'list';
$message = '';
$error = '';
$systemPageSlugs = SetupService::getSystemPageSlugs();

// -----------------------------------------------------------------------------
// Legal template migration snapshots
// -----------------------------------------------------------------------------
// The $legacy* and $previous* template variables below are historical defaults
// kept only for safe one-time auto-update checks.
// They are NOT shown to visitors unless that exact content is already stored in
// the database for a page. Current visible defaults come from $default* values.
// -----------------------------------------------------------------------------

// Ensure Contact page exists (system page like Home)
$contactStmt = $db->prepare("SELECT id FROM pages WHERE slug = 'contact' LIMIT 1");
$contactStmt->execute();
if (!$contactStmt->fetch()) {
    $insertStmt = $db->prepare(SQL_INSERT_PAGE_BASIC);
    $insertStmt->execute(['Contact', 'contact', '', 'Contact', 'Contact page', 1]);
}

// Ensure Cart page exists (system page like Home)
$cartStmt = $db->prepare("SELECT id FROM pages WHERE slug = 'cart' LIMIT 1");
$cartStmt->execute();
if (!$cartStmt->fetch()) {
    $insertStmt = $db->prepare(SQL_INSERT_PAGE_BASIC);
    $insertStmt->execute(['Cart', 'cart', '', 'Cart', 'Cart page', 1]);
}

// Ensure Checkout page exists (system page like Home)
$checkoutStmt = $db->prepare("SELECT id FROM pages WHERE slug = 'checkout' LIMIT 1");
$checkoutStmt->execute();
if (!$checkoutStmt->fetch()) {
    $insertStmt = $db->prepare(SQL_INSERT_PAGE_BASIC);
    $insertStmt->execute(['Checkout', 'checkout', '', 'Checkout', 'Checkout page', 1]);
}

// Ensure Games page exists (system page for mini-games hub routing)
$gamesStmt = $db->prepare("SELECT id FROM pages WHERE slug = 'games' LIMIT 1");
$gamesStmt->execute();
if (!$gamesStmt->fetch()) {
    $insertStmt = $db->prepare(SQL_INSERT_PAGE_BASIC);
    $insertStmt->execute(['Mini-Games', 'games', '', 'Mini-Games', 'Play free browser mini-games.', 1]);
}

// Ensure Product page exists (system page for product detail routing)
$productPageStmt = $db->prepare("SELECT id FROM pages WHERE slug = 'product' LIMIT 1");
$productPageStmt->execute();
$productPageRow = $productPageStmt->fetch();
if (!$productPageRow) {
    $insertStmt = $db->prepare(SQL_INSERT_PAGE_BASIC);
    $insertStmt->execute(['Product', 'product', '', 'Product', 'Product details page', 1]);
    $productPageId = (int)DbHelper::lastInsertId($db, 'pages');
} else {
    $productPageId = (int)$productPageRow['id'];
}

if (!empty($productPageId)) {
    $sectionStmt = $db->prepare("SELECT id FROM sections WHERE page_id = ? AND section_type = 'products' LIMIT 1");
    $sectionStmt->execute([$productPageId]);
    if (!$sectionStmt->fetch()) {
        $defaultSectionSettings = [
            'section_title' => 'Products',
            'title_align' => 'center',
            'product_id' => '',
            'accordion_style' => 'separated',
            'content_align' => 'left',
            'variant_style' => 'select',
            'show_media' => true,
            'show_description' => true,
            'show_price' => true
        ];
        $orderStmt = $db->prepare(SQL_SELECT_MAX_SECTION_ORDER);
        $orderStmt->execute([$productPageId]);
        $maxOrder = (int)$orderStmt->fetchColumn();
        $insertSection = $db->prepare(SQL_INSERT_SECTION);
        $insertSection->execute([$productPageId, 'products', json_encode($defaultSectionSettings), $maxOrder + 1]);
    }
}

// Ensure Products List page exists (system page for product listing)
$productsListStmt = $db->prepare("SELECT id FROM pages WHERE slug = 'products' LIMIT 1");
$productsListStmt->execute();
$productsListRow = $productsListStmt->fetch();
if (!$productsListRow) {
    $insertStmt = $db->prepare(SQL_INSERT_PAGE_BASIC);
    $insertStmt->execute(['Products', 'products', '', 'Products', 'Browse all products', 1]);
    $productsListPageId = (int)DbHelper::lastInsertId($db, 'pages');
} else {
    $productsListPageId = (int)$productsListRow['id'];
}

if (!empty($productsListPageId)) {
    $sectionStmt = $db->prepare("SELECT id FROM sections WHERE page_id = ? AND section_type = 'products_list' LIMIT 1");
    $sectionStmt->execute([$productsListPageId]);
    if (!$sectionStmt->fetch()) {
        $defaultSectionSettings = [
            'section_title' => 'Products',
            'title_align' => 'center',
            'columns' => '3',
            'show_price' => true
        ];
        $orderStmt = $db->prepare(SQL_SELECT_MAX_SECTION_ORDER);
        $orderStmt->execute([$productsListPageId]);
        $maxOrder = (int)$orderStmt->fetchColumn();
        $insertSection = $db->prepare(SQL_INSERT_SECTION);
        $insertSection->execute([$productsListPageId, 'products_list', json_encode($defaultSectionSettings), $maxOrder + 1]);
    }
}

// Ensure Collections List page exists (system page for collections listing)
$collectionsListStmt = $db->prepare("SELECT id FROM pages WHERE slug = 'collections' LIMIT 1");
$collectionsListStmt->execute();
$collectionsListRow = $collectionsListStmt->fetch();
if (!$collectionsListRow) {
    $insertStmt = $db->prepare(SQL_INSERT_PAGE_BASIC);
    $insertStmt->execute(['Collections', 'collections', '', 'Collections', 'Browse all collections', 1]);
    $collectionsListPageId = (int)DbHelper::lastInsertId($db, 'pages');
} else {
    $collectionsListPageId = (int)$collectionsListRow['id'];
}

if (!empty($collectionsListPageId)) {
    $sectionStmt = $db->prepare("SELECT id FROM sections WHERE page_id = ? AND section_type = 'collections_list' LIMIT 1");
    $sectionStmt->execute([$collectionsListPageId]);
    if (!$sectionStmt->fetch()) {
        $defaultSectionSettings = [
            'section_title' => 'Collections',
            'title_align' => 'center',
            'columns' => '3',
            'show_description' => true,
            'show_image' => true
        ];
        $orderStmt = $db->prepare(SQL_SELECT_MAX_SECTION_ORDER);
        $orderStmt->execute([$collectionsListPageId]);
        $maxOrder = (int)$orderStmt->fetchColumn();
        $insertSection = $db->prepare(SQL_INSERT_SECTION);
        $insertSection->execute([$collectionsListPageId, 'collections_list', json_encode($defaultSectionSettings), $maxOrder + 1]);
    }
}

// Ensure Privacy Policy page exists (system page - GDPR required)
// Snapshot set: legacy + previous + current default
$legacyPrivacyContent = <<<'HTML'
<h2>Privacy Policy</h2>
<p>Last Updated: February 5, 2026</p>
<p>{SITE_NAME} ("we", "us", "our") operates {SITE_URL}. This Privacy Policy explains how we collect, use, and share information when you use our website, create an account, or purchase products.</p>

<h3>Information We Collect</h3>
<ul>
    <li>Account and profile information: email, display name, nickname, and any optional profile details you provide.</li>
    <li>User-generated content: posts, comments, images, and messages you submit to the forum or public areas.</li>
    <li>Orders and customer support: purchase history, digital download access, and support requests.</li>
    <li>Payment processing: payments are handled by Stripe. We do not store full card numbers.</li>
    <li>Future physical orders: shipping address and delivery details if you buy physical products.</li>
    <li>Technical and usage data: IP address, device and browser details, and security logs.</li>
    <li>Cookies and local storage: session, security, and cookie consent preferences.</li>
</ul>

<h3>How We Use Information</h3>
<ul>
    <li>Provide and maintain the site, accounts, and digital downloads.</li>
    <li>Process orders and deliver products.</li>
    <li>Moderate content and enforce our Terms of Service.</li>
    <li>Protect the site against fraud and abuse.</li>
    <li>Respond to support and legal requests.</li>
    <li>Improve performance and user experience.</li>
</ul>

<h3>Legal Bases (GDPR)</h3>
<p>We process personal data based on consent, contract performance, legitimate interests, and legal obligations.</p>

<h3>Sharing and Disclosure</h3>
<ul>
    <li>Service providers (hosting, email, analytics, payment processing).</li>
    <li>Legal and safety requirements.</li>
    <li>Business transfers if we restructure the business.</li>
</ul>

<h3>International Transfers</h3>
<p>We may process data in countries other than yours. We use appropriate safeguards when required.</p>

<h3>User Content and Public Areas</h3>
<p>Forum posts, public profiles, and uploaded images may be visible to other visitors. Do not upload sensitive personal data. We may keep backups for a limited time after content is removed.</p>

<h3>Sensitive Information</h3>
<p>We do not request sensitive personal data (such as government IDs, health data, precise location, or biometrics). If you choose to post or upload sensitive information, you do so at your own risk and you consent to our processing of that information for the purpose of providing, moderating, and securing the service.</p>

<h3>Data Retention</h3>
<p>We keep data only as long as needed for the purposes described, and per our retention settings. Logs and rate-limit data are automatically purged on a schedule.</p>

<h3>Your Rights</h3>
<p>You may request access, correction, deletion, or export of your data, and object to certain processing. Contact us at {CONTACT_EMAIL}.</p>

<h3>Regional Privacy Rights</h3>
<ul>
    <li><strong>EEA/UK (GDPR):</strong> access, correction, deletion, portability, restriction, objection, and the right to lodge a complaint with a supervisory authority.</li>
    <li><strong>California and other US states:</strong> right to know, delete, and correct personal data, and to opt out of selling or sharing personal data for targeted advertising. We do not sell personal data.</li>
    <li><strong>Brazil (LGPD):</strong> confirmation of processing, access, correction, deletion, portability, and information about sharing.</li>
    <li><strong>Canada (PIPEDA):</strong> access, correction, and the right to challenge compliance with privacy obligations.</li>
    <li><strong>Other regions:</strong> you may have similar rights under local law. Contact us to exercise your rights.</li>
</ul>

<h3>Children</h3>
<p>The site is not intended for children under the age required by local law.</p>

<h3>Contact</h3>
<p>Questions about this policy can be sent to {CONTACT_EMAIL}.</p>
HTML;

$previousPrivacyContent = <<<'HTML'
<h2>Privacy Policy</h2>
<p>Last Updated: February 5, 2026</p>
<p>{SITE_NAME} ("we", "us", "our") operates {SITE_URL}. This Privacy Policy explains how we collect, use, and share information when you use our website, create an account, or purchase products.</p>

<h3>Information We Collect</h3>
<ul>
    <li>Account and profile information: email, display name, nickname, and any optional profile details you provide.</li>
    <li>User-generated content: posts, comments, images, and messages you submit to the forum or public areas.</li>
    <li>Orders and customer support: purchase history, digital download access, and support requests.</li>
    <li>Payment processing: payments are handled by Stripe. We do not store full card numbers.</li>
    <li>Future physical orders: shipping address and delivery details if you buy physical products.</li>
    <li>Technical and usage data: IP address, device and browser details, and security logs.</li>
    <li>Cookies and local storage: session, security, and cookie consent preferences.</li>
</ul>

<h3>How We Use Information</h3>
<ul>
    <li>Provide and maintain the site, accounts, and digital downloads.</li>
    <li>Process orders and deliver products.</li>
    <li>Moderate content and enforce our Terms of Service.</li>
    <li>Protect the site against fraud and abuse.</li>
    <li>Respond to support and legal requests.</li>
    <li>Improve performance and user experience.</li>
</ul>

<h3>Legal Bases (GDPR)</h3>
<p>We process personal data based on consent, contract performance, legitimate interests, and legal obligations.</p>

<h3>Sharing and Disclosure</h3>
<ul>
    <li>Service providers (hosting, email, analytics, payment processing).</li>
    <li>Legal and safety requirements.</li>
    <li>Business transfers if we restructure the business.</li>
</ul>

<h3>International Transfers</h3>
<p>We may process data in countries other than yours. We use appropriate safeguards when required.</p>

<h3>Data Retention</h3>
<p>We keep data only as long as needed for the purposes described, and per our retention settings. Logs and rate-limit data are automatically purged on a schedule.</p>

<h3>Your Rights</h3>
<p>You may request access, correction, deletion, or export of your data, and object to certain processing. Contact us at {CONTACT_EMAIL}.</p>

<h3>Children</h3>
<p>The site is not intended for children under the age required by local law.</p>

<h3>Contact</h3>
<ul>
    <li>Name: {CONTACT_NAME}</li>
    <li>Email: {CONTACT_EMAIL}</li>
    <li>Phone: {CONTACT_PHONE}</li>
    <li>Address: {CONTACT_ADDRESS}</li>
</ul>
HTML;

$defaultPrivacyContent = <<<'HTML'
<h2>Privacy Policy</h2>
<p>Last Updated: February 18, 2026</p>
<p>{SITE_NAME} ("we", "us", "our") operates {SITE_URL}. This Privacy Policy explains how we collect, use, and share information when you use our website, create an account, participate in community features, and buy products.</p>

<h3>Information We Collect</h3>
<ul>
    <li>Account and profile data (email, display name, nickname, optional profile fields).</li>
    <li>Authentication and security data (login-link requests, security logs, abuse-prevention metadata).</li>
    <li>Community/forum content (posts, comments, media uploads, moderation metadata).</li>
    <li>Orders and fulfillment data (order history, downloadable access, shipping details for physical goods).</li>
    <li>Technical data (IP-derived security controls, browser/device details, cookie preferences).</li>
    <li>Support and legal communications you send to us.</li>
</ul>

<h3>Product Types and Fulfillment</h3>
<ul>
    <li><strong>Cart/Checkout products:</strong> payment and order details are processed to complete purchases.</li>
    <li><strong>Digital download products:</strong> we store delivery/access information and download activity for fraud prevention and support.</li>
    <li><strong>External-link products:</strong> you may be redirected to a third-party destination with its own privacy terms.</li>
    <li><strong>Physical products:</strong> we process delivery details required for shipping and order support.</li>
</ul>

<h3>How We Use Data</h3>
<ul>
    <li>Provide and secure accounts, community features, purchases, and support.</li>
    <li>Moderate abuse and enforce Terms of Service.</li>
    <li>Process transactions and fulfill digital/physical products.</li>
    <li>Handle legal requests and regulatory obligations.</li>
    <li>Improve service performance, reliability, and user experience.</li>
</ul>

<h3>Legal Bases (GDPR)</h3>
<p>We process data based on consent, contract performance, legitimate interests, and legal obligations.</p>

<h3>Sharing and Disclosure</h3>
<ul>
    <li>Payment, hosting, email, delivery, and infrastructure providers.</li>
    <li>Authorities/law-enforcement when required by law.</li>
    <li>Professional advisers and successors in a lawful business transfer.</li>
</ul>

<h3>International Transfers</h3>
<p>Your information may be processed in countries outside your own. Where required, we apply appropriate safeguards.</p>

<h3>Your Rights</h3>
<p>Depending on your region, you may have rights to access, correct, delete, restrict, object, or port your data, and to lodge complaints with a supervisory authority.</p>

<h3>Regional Privacy Notes</h3>
<ul>
    <li><strong>EEA/UK:</strong> GDPR/UK-GDPR rights apply.</li>
    <li><strong>US States:</strong> rights may include access, correction, deletion, and appeal under local law.</li>
    <li><strong>Brazil/Canada/other regions:</strong> equivalent statutory rights apply where required.</li>
</ul>

<h3>Retention</h3>
<p>We retain data only as needed for service operation, fraud prevention, support, and legal compliance, then delete or anonymize it according to configured retention periods.</p>

<h3>Children</h3>
<p>The service is not intended for children under the minimum age required by local law.</p>

<h3>Contact</h3>
<ul>
    <li>Name: {CONTACT_NAME}</li>
    <li>Email: {CONTACT_EMAIL}</li>
    <li>Phone: {CONTACT_PHONE}</li>
    <li>Address: {CONTACT_ADDRESS}</li>
</ul>
HTML;

$previousPrivacyContentV2 = $defaultPrivacyContent;
// Apply updated wording while preserving the immediately previous default
// snapshot above for migration matching.
$defaultPrivacyContent = str_replace(
    '<li>Professional advisers and successors in a lawful business transfer.</li>',
    '<li>We do not sell personal data and do not share personal data with advertising networks, data brokers, or professional advertisers.</li>',
    $defaultPrivacyContent
);
$defaultPrivacyContent = str_replace(
    '<li><strong>US States:</strong> rights may include access, correction, deletion, and appeal under local law.</li>',
    '<li><strong>US States:</strong> rights may include access, correction, deletion, and appeal under local law. We do not sell personal data or share it for targeted advertising.</li>',
    $defaultPrivacyContent
);

$privacyStmt = $db->prepare("SELECT id, content FROM pages WHERE slug = 'privacy-policy' LIMIT 1");
$privacyStmt->execute();
$privacyRow = $privacyStmt->fetch();
if (!$privacyRow) {
    $insertStmt = $db->prepare(SQL_INSERT_PAGE_BASIC);
    $insertStmt->execute([PAGE_TITLE_PRIVACY_POLICY, 'privacy-policy', $defaultPrivacyContent, PAGE_TITLE_PRIVACY_POLICY, 'Our privacy policy explains how we handle your data', 1]);
} else {
    $privacyContent = (string)($privacyRow['content'] ?? '');
    $shouldUpdate = trim(strip_tags($privacyContent)) === ''
        || trim($privacyContent) === trim($legacyPrivacyContent)
        || trim($privacyContent) === trim($previousPrivacyContent)
        || trim($privacyContent) === trim($previousPrivacyContentV2)
        || stripos($privacyContent, 'Last Updated: January 18, 2026') !== false;
    if ($shouldUpdate) {
        $stmt = $db->prepare(SQL_UPDATE_PAGE_CONTENT_META_BY_ID);
        $stmt->execute([$defaultPrivacyContent, PAGE_TITLE_PRIVACY_POLICY, 'Our privacy policy explains how we handle your data', $privacyRow['id']]);
    }
}

// Ensure Terms of Service page exists (system page - legal requirement)
// Snapshot set: legacy + previous + current default
$legacyTermsContent = <<<'HTML'
<h2>Terms of Service</h2>
<p>Last Updated: February 5, 2026</p>
<p>Welcome to {SITE_NAME}. By using {SITE_URL}, you agree to these Terms of Service.</p>

<h3>1. Accounts and Eligibility</h3>
<p>You are responsible for your account and any activity that occurs under it. You must provide accurate information and keep your login link secure.</p>

<h3>2. User Content and License</h3>
<p>You retain ownership of the content you post. By submitting content (including images) to forums or public areas, you grant us a worldwide, non-exclusive, royalty-free license to host, store, display, reproduce, and distribute your content solely for operating and promoting the service. You represent that you have the rights to share the content.</p>

<h3>3. Prohibited Conduct</h3>
<ul>
    <li>Do not post content that is illegal, infringing, abusive, or harmful.</li>
    <li>Do not upload malware or attempt to compromise site security.</li>
    <li>Do not misuse the platform or harass other users.</li>
</ul>

<h3>4. Digital Products</h3>
<p>Digital products are licensed, not sold. You may use them for personal, non-commercial purposes unless otherwise stated. Redistribution or resale is prohibited.</p>

<h3>5. Physical Products (Future)</h3>
<p>If we offer physical products, shipping and returns will be governed by our Shipping Policy and Refund Policy.</p>

<h3>6. Payments and Refunds</h3>
<p>Prices and taxes are shown at checkout. Refunds and returns follow our Refund Policy and applicable law.</p>

<h3>7. Intellectual Property and DMCA</h3>
<p>All site content and branding are protected by intellectual property laws. Our DMCA policy is available at /dmca.</p>

<h3>8. Third-Party Links</h3>
<p>We may link to third-party platforms (e.g., Steam, Epic Games). We are not responsible for their content or policies.</p>

<h3>9. Disclaimers</h3>
<p>The service is provided "as is" without warranties of any kind.</p>

<h3>10. Limitation of Liability</h3>
<p>To the maximum extent permitted by law, we are not liable for indirect or consequential damages arising from your use of the site.</p>

<h3>11. Changes</h3>
<p>We may update these terms from time to time. Continued use of the site means you accept the updated terms.</p>

<h3>12. Contact</h3>
<p>If you have questions about these Terms, contact us at {CONTACT_EMAIL}.</p>
HTML;

$previousTermsContent = <<<'HTML'
<h2>Terms of Service</h2>
<p>Last Updated: February 5, 2026</p>
<p>Welcome to {SITE_NAME}. By using {SITE_URL}, you agree to these Terms of Service.</p>

<h3>1. Accounts and Eligibility</h3>
<p>You are responsible for your account and any activity that occurs under it. You must provide accurate information and keep your login link secure.</p>

<h3>2. User Content and License</h3>
<p>You retain ownership of the content you post. By submitting content (including images) to forums or public areas, you grant us a worldwide, non-exclusive, royalty-free license to host, store, display, reproduce, and distribute your content solely for operating and promoting the service. You represent that you have the rights to share the content.</p>

<h3>3. Prohibited Conduct</h3>
<ul>
    <li>Do not post content that is illegal, infringing, abusive, or harmful.</li>
    <li>Do not upload malware or attempt to compromise site security.</li>
    <li>Do not misuse the platform or harass other users.</li>
</ul>

<h3>4. Digital Products</h3>
<p>Digital products are licensed, not sold. You may use them for personal, non-commercial purposes unless otherwise stated. Redistribution or resale is prohibited.</p>

<h3>5. Physical Products (Future)</h3>
<p>If we offer physical products, shipping and returns will be governed by our Shipping Policy and Refund Policy.</p>

<h3>6. Payments and Refunds</h3>
<p>Prices and taxes are shown at checkout. Refunds and returns follow our Refund Policy and applicable law.</p>

<h3>7. Intellectual Property and DMCA</h3>
<p>All site content and branding are protected by intellectual property laws. Our DMCA policy is available at /dmca.</p>

<h3>8. Third-Party Links</h3>
<p>We may link to third-party platforms (e.g., Steam, Epic Games). We are not responsible for their content or policies.</p>

<h3>9. Disclaimers</h3>
<p>The service is provided "as is" without warranties of any kind.</p>

<h3>10. Limitation of Liability</h3>
<p>To the maximum extent permitted by law, we are not liable for indirect or consequential damages arising from your use of the site.</p>

<h3>11. Changes</h3>
<p>We may update these terms from time to time. Continued use of the site means you accept the updated terms.</p>

<h3>12. Contact</h3>
<ul>
    <li>Name: {CONTACT_NAME}</li>
    <li>Email: {CONTACT_EMAIL}</li>
    <li>Phone: {CONTACT_PHONE}</li>
    <li>Address: {CONTACT_ADDRESS}</li>
</ul>
HTML;

$defaultTermsContent = <<<'HTML'
<h2>Terms of Service</h2>
<p>Last Updated: February 18, 2026</p>
<p>Welcome to {SITE_NAME}. By using {SITE_URL}, you agree to these Terms of Service.</p>

<h3>1. Accounts and Access</h3>
<p>You are responsible for activity on your account and for keeping account access secure. You must provide accurate information and comply with applicable law.</p>

<h3>2. Community Features (Forum, Profiles, Messages)</h3>
<p>You retain ownership of content you submit. By posting content, you grant us a non-exclusive license to host, display, and process it for service operation, moderation, security, and lawful enforcement.</p>

<h3>3. Prohibited Use</h3>
<ul>
    <li>No illegal, infringing, abusive, deceptive, or malicious activity.</li>
    <li>No malware, scraping abuse, credential abuse, or attempts to bypass security controls.</li>
    <li>No harassment, impersonation, or rights violations.</li>
</ul>

<h3>4. Product Types and Fulfillment</h3>
<ul>
    <li><strong>Cart/Checkout products:</strong> purchased directly on this site.</li>
    <li><strong>Digital downloads:</strong> licensed (not sold), subject to license restrictions and anti-abuse controls.</li>
    <li><strong>External-link products:</strong> may redirect you to third-party stores/platforms with their own terms.</li>
    <li><strong>Physical products:</strong> shipping and return conditions apply where offered.</li>
</ul>

<h3>5. Pricing, Taxes, and Payment</h3>
<p>Prices, taxes, and applicable fees are displayed at checkout where required. Payment processing providers handle payment authorization and settlement.</p>

<h3>6. Refunds, Returns, and Withdrawal Rights</h3>
<p>Refunds and returns follow our Refund Policy and mandatory local consumer law. Nothing in these terms limits statutory consumer rights.</p>

<h3>7. Third-Party Services and External Links</h3>
<p>Third-party destinations, stores, and services are outside our control. Their terms and privacy policies govern your use of those services.</p>

<h3>8. Intellectual Property and Takedowns</h3>
<p>Site content and branding are protected by intellectual property laws. Copyright claims may be submitted under our DMCA policy.</p>

<h3>9. Service Changes and Availability</h3>
<p>We may modify, suspend, or discontinue parts of the service for maintenance, legal compliance, abuse prevention, or product updates.</p>

<h3>10. Suspension and Termination</h3>
<p>We may suspend or terminate access for violations of these terms, abuse, fraud, or legal requirements.</p>

<h3>11. Disclaimer</h3>
<p>The service is provided "as is" and "as available" to the extent permitted by law.</p>

<h3>12. Limitation of Liability</h3>
<p>To the maximum extent permitted by law, we are not liable for indirect, incidental, special, or consequential damages.</p>

<h3>13. Local Law and Consumer Protection</h3>
<p>Mandatory rights under your local law remain unaffected.</p>

<h3>14. Contact</h3>
<ul>
    <li>Name: {CONTACT_NAME}</li>
    <li>Email: {CONTACT_EMAIL}</li>
    <li>Phone: {CONTACT_PHONE}</li>
    <li>Address: {CONTACT_ADDRESS}</li>
</ul>
HTML;

$termsStmt = $db->prepare("SELECT id, content FROM pages WHERE slug = 'terms-of-service' LIMIT 1");
$termsStmt->execute();
$termsRow = $termsStmt->fetch();
if (!$termsRow) {
    $insertStmt = $db->prepare(SQL_INSERT_PAGE_BASIC);
    $insertStmt->execute([PAGE_TITLE_TERMS_OF_SERVICE, 'terms-of-service', $defaultTermsContent, PAGE_TITLE_TERMS_OF_SERVICE, 'Terms and conditions for using our website', 1]);
} else {
    $termsContent = (string)($termsRow['content'] ?? '');
    $shouldUpdate = trim(strip_tags($termsContent)) === ''
        || trim($termsContent) === trim($legacyTermsContent)
        || trim($termsContent) === trim($previousTermsContent)
        || stripos($termsContent, 'Please read these terms carefully before downloading') !== false
        || stripos($termsContent, 'License to Play') !== false;
    if ($shouldUpdate) {
        $stmt = $db->prepare(SQL_UPDATE_PAGE_CONTENT_META_BY_ID);
        $stmt->execute([$defaultTermsContent, PAGE_TITLE_TERMS_OF_SERVICE, 'Terms and conditions for using our website', $termsRow['id']]);
    }
}

// Ensure DMCA page exists (system page - legal requirement)
$defaultDmcaContent = '<h2>DMCA Takedown Policy</h2><p>This page explains how to submit a DMCA takedown notice or counter-notice.</p><h3>How to Submit a Takedown Notice</h3><p>Please provide the following:</p><ul><li>Your full legal name, address, phone number, and email address.</li><li>Identification of the copyrighted work you claim has been infringed.</li><li>Exact URL(s) or location(s) of the allegedly infringing content.</li><li>A statement that you have a good-faith belief the use is not authorized.</li><li>A statement made under penalty of perjury that your notice is accurate and you are the rights holder or authorized to act on their behalf.</li><li>Your physical or electronic signature.</li></ul><h3>Where to Send Notices</h3><p>Email: {CONTACT_EMAIL}</p><h3>Counter-Notice</h3><p>If you believe content was removed in error, you may submit a counter-notice with your contact details, the removed content location, a statement under penalty of perjury that removal was a mistake, and consent to jurisdiction.</p>';
$dmcaStmt = $db->prepare("SELECT id FROM pages WHERE slug = 'dmca' LIMIT 1");
$dmcaStmt->execute();
if (!$dmcaStmt->fetch()) {
    $insertStmt = $db->prepare(SQL_INSERT_PAGE_BASIC);
    $insertStmt->execute(['DMCA', 'dmca', $defaultDmcaContent, 'DMCA', 'Digital Millennium Copyright Act (DMCA) policy', 1]);
}

// Ensure Refund Policy page exists (system page)
// Snapshot set: legacy + previous + previous V2 + current default
$legacyRefundContent = <<<'HTML'
<h2>Refund and Return Policy</h2>
<p>Last Updated: February 5, 2026</p>
<p>This policy applies to purchases made through {SITE_NAME}.</p>

<h3>Digital Products</h3>
<p>Digital products are delivered electronically. All sales are final once access is provided, except where required by law or if the product is defective or not as described.</p>

<h3>Digital Right of Withdrawal (Where Applicable)</h3>
<p>In some regions (e.g., EU/UK), you may have a statutory right to withdraw from a digital purchase unless you consent to immediate delivery and acknowledge that your right to withdraw is waived once access begins. Where required, we will request this consent at checkout.</p>

<h3>Free Downloads</h3>
<p>Free demos and downloads are provided as-is and do not include refunds.</p>

<h3>Physical Products (Future)</h3>
<p>If we offer physical products, return windows, eligibility, and shipping costs will be described on the product page or order confirmation.</p>

<h3>Return Windows (Where Required)</h3>
<p>Some jurisdictions provide a minimum return or withdrawal period for physical goods (often 14 or 30 days). We honor applicable local law.</p>

<h3>Statutory Rights</h3>
<p>Nothing in this policy limits your statutory rights. If your local law provides additional rights, those rights apply.</p>

<h3>How to Request a Refund</h3>
<p>Contact us at {CONTACT_EMAIL} with your order number and details.</p>
HTML;

$previousRefundContent = <<<'HTML'
<h2>Refund and Return Policy</h2>
<p>Last Updated: February 5, 2026</p>
<p>This policy applies to purchases made through {SITE_NAME}.</p>

<h3>Digital Products</h3>
<p>Digital products are delivered electronically. All sales are final once access is provided, except where required by law or if the product is defective.</p>

<h3>Physical Products (Future)</h3>
<p>If we offer physical products, return windows, eligibility, and shipping costs will be described on the product page or order confirmation.</p>

<h3>How to Request a Refund</h3>
<p>Contact us with your order number and details.</p>
<ul>
    <li>Name: {CONTACT_NAME}</li>
    <li>Email: {CONTACT_EMAIL}</li>
    <li>Phone: {CONTACT_PHONE}</li>
    <li>Address: {CONTACT_ADDRESS}</li>
</ul>

<h3>Chargebacks</h3>
<p>Initiating a chargeback without contacting support may result in account restrictions. Please contact us first so we can help resolve the issue.</p>
HTML;

$previousRefundContentV2 = <<<'HTML'
<h2>Refund and Return Policy</h2>
<p>Last Updated: February 18, 2026</p>
<p>This policy applies to purchases made through {SITE_NAME}.</p>

<h3>Digital Products</h3>
<p>For paid digital products, refunds are generally not available after delivery/access unless required by law, the product is defective, or it materially differs from its description.</p>

<h3>Free Digital Downloads</h3>
<p>Free downloads are provided as-is without warranties or refund obligations.</p>

<h3>External-Link Products</h3>
<p>For products fulfilled by a third-party external platform/store, refund and cancellation terms are governed by that third party unless mandatory law states otherwise.</p>

<h3>Physical Products</h3>
<p>Where physical goods are offered, return windows, conditions, and costs are shown at checkout or in your order confirmation.</p>

<h3>Statutory Withdrawal / Consumer Rights</h3>
<p>If your local law grants withdrawal or return rights (for example for certain consumer purchases), those rights apply and are not limited by this policy.</p>

<h3>How to Request a Refund or Return</h3>
<p>Contact us with your order number, purchase email, and reason for the request.</p>
<ul>
    <li>Name: {CONTACT_NAME}</li>
    <li>Email: {CONTACT_EMAIL}</li>
    <li>Phone: {CONTACT_PHONE}</li>
    <li>Address: {CONTACT_ADDRESS}</li>
</ul>

<h3>Chargebacks</h3>
<p>Please contact support before initiating a chargeback. Unresolved chargebacks may lead to account or purchase restrictions where legally permitted.</p>
HTML;

$defaultRefundContent = <<<'HTML'
<h2>Refund and Return Policy</h2>
<p>Last Updated: February 18, 2026</p>
<p>This policy applies to purchases made through {SITE_NAME}.</p>

<h3>Digital Products</h3>
<p>For paid digital products, refunds are generally not available after delivery/access unless required by law, the product is defective, or it materially differs from its description.</p>

<h3>Free Digital Downloads</h3>
<p>Free downloads are provided as-is without warranties or refund obligations.</p>

<h3>External-Link Products</h3>
<p>For products fulfilled by a third-party external platform/store, refund and cancellation terms are governed by that third party unless mandatory law states otherwise.</p>

<h3>Physical Products</h3>
<p>Where physical goods are offered, our standard voluntary return window is <strong>30 days from delivery</strong>, unless a longer period is required by your local law.</p>

<h3>Physical Return Eligibility</h3>
<ul>
    <li>Item must be returned in original condition (unused, clean, and complete).</li>
    <li>Proof of purchase (order number / receipt email) is required.</li>
    <li>Original packaging is recommended and may be required where legally permitted.</li>
</ul>

<h3>Non-Returnable or Restricted Items</h3>
<ul>
    <li>Personalized or custom-made items (unless defective or required by law).</li>
    <li>Sealed hygiene/health items after opening, where local law permits exclusion.</li>
    <li>Digital goods already delivered/accessed, except where mandatory law provides otherwise.</li>
</ul>

<h3>Refund Method and Timing</h3>
<p>Approved refunds are issued to the original payment method. Processing time depends on payment providers/banks and may vary by region.</p>

<h3>Return Shipping Costs</h3>
<p>Unless local law requires otherwise, return shipping costs are the buyer's responsibility unless the item is defective, damaged, or incorrect.</p>

<h3>Statutory Withdrawal / Consumer Rights</h3>
<p>If your local law grants withdrawal or return rights (for example for certain consumer purchases), those rights apply and are not limited by this policy.</p>

<h3>How to Request a Refund or Return</h3>
<p>Contact us with your order number, purchase email, and reason for the request.</p>
<ul>
    <li>Name: {CONTACT_NAME}</li>
    <li>Email: {CONTACT_EMAIL}</li>
    <li>Phone: {CONTACT_PHONE}</li>
    <li>Address: {CONTACT_ADDRESS}</li>
</ul>

<h3>Chargebacks</h3>
<p>Please contact support before initiating a chargeback. Unresolved chargebacks may lead to account or purchase restrictions where legally permitted.</p>
HTML;

$refundStmt = $db->prepare("SELECT id, content FROM pages WHERE slug = 'refund-policy' LIMIT 1");
$refundStmt->execute();
$refundRow = $refundStmt->fetch();
if (!$refundRow) {
    $insertStmt = $db->prepare(SQL_INSERT_PAGE_BASIC);
    $insertStmt->execute([PAGE_TITLE_REFUND_POLICY, 'refund-policy', $defaultRefundContent, PAGE_TITLE_REFUND_POLICY, 'Refunds and returns policy', 0]);
} else {
    $refundContent = (string)($refundRow['content'] ?? '');
    $shouldUpdate = trim(strip_tags($refundContent)) === ''
        || trim($refundContent) === trim($legacyRefundContent)
        || trim($refundContent) === trim($previousRefundContent)
        || trim($refundContent) === trim($previousRefundContentV2);
    if ($shouldUpdate) {
        $stmt = $db->prepare(SQL_UPDATE_PAGE_CONTENT_META_BY_ID);
        $stmt->execute([$defaultRefundContent, PAGE_TITLE_REFUND_POLICY, 'Refunds and returns policy', $refundRow['id']]);
    }
}

// Ensure Shipping Policy page exists (system page)
// Snapshot set: legacy + previous + previous V2 + current default
$legacyShippingContent = <<<'HTML'
<h2>Shipping Policy</h2>
<p>Last Updated: February 5, 2026</p>
<p>Digital products are available immediately after purchase.</p>
<p>If physical products are offered, shipping options, costs, and estimated delivery times will be shown at checkout.</p>
<p>Questions about shipping can be sent to {CONTACT_EMAIL}.</p>
HTML;

$previousShippingContent = <<<'HTML'
<h2>Shipping Policy</h2>
<p>Last Updated: February 5, 2026</p>
<p>Digital products are available immediately after purchase.</p>
<p>If physical products are offered, shipping options, costs, and estimated delivery times will be shown at checkout. Delivery times vary by destination and carrier.</p>
<p>Questions about shipping can be sent to {CONTACT_EMAIL}.</p>
<ul>
    <li>Name: {CONTACT_NAME}</li>
    <li>Email: {CONTACT_EMAIL}</li>
    <li>Phone: {CONTACT_PHONE}</li>
    <li>Address: {CONTACT_ADDRESS}</li>
</ul>

<h3>Order Processing</h3>
<p>If physical products are offered, orders are typically processed within 1 to 2 business days unless otherwise stated on the product page.</p>

<h3>International Shipping</h3>
<p>International availability, customs duties, and import taxes (if any) will be shown at checkout or in the order confirmation. Customs fees are the responsibility of the recipient unless otherwise stated.</p>

<h3>Address Accuracy</h3>
<p>Please ensure your shipping details are correct. We are not responsible for delays or losses caused by incorrect addresses.</p>

<h3>Shipping Issues</h3>
<p>If you experience a delivery issue, contact us with your order number and details.</p>

<h3>Local Consumer Rights</h3>
<p>If local law provides additional rights relating to delivery or cancellation, those rights apply.</p>
HTML;

$previousShippingContentV2 = <<<'HTML'
<h2>Shipping Policy</h2>
<p>Last Updated: February 18, 2026</p>

<h3>Scope</h3>
<p>This Shipping Policy applies only to physical goods sold through {SITE_NAME}. Digital downloads are delivered electronically, and external-link products are fulfilled by third parties.</p>

<h3>Processing Times</h3>
<p>Physical orders are generally processed within 1 to 2 business days unless a different timeframe is shown at checkout.</p>

<h3>Delivery Times and Carriers</h3>
<p>Estimated delivery times depend on destination, customs, and carrier performance. Estimates are not guaranteed unless local law requires otherwise.</p>

<h3>Shipping Fees, Customs, and Import Taxes</h3>
<p>Shipping fees are shown at checkout. International customs duties and import taxes are the recipient’s responsibility unless expressly stated otherwise.</p>

<h3>Address Accuracy</h3>
<p>You are responsible for providing accurate shipping information. We are not responsible for delays or failed delivery caused by incorrect addresses provided at checkout.</p>

<h3>Delivery Issues</h3>
<p>If a shipment is delayed, lost, or damaged, contact us with your order number so we can investigate.</p>

<h3>Contact</h3>
<ul>
    <li>Name: {CONTACT_NAME}</li>
    <li>Email: {CONTACT_EMAIL}</li>
    <li>Phone: {CONTACT_PHONE}</li>
    <li>Address: {CONTACT_ADDRESS}</li>
</ul>

<h3>Local Consumer Rights</h3>
<p>Mandatory consumer delivery and cancellation rights under your local law remain unaffected.</p>
HTML;

$defaultShippingContent = <<<'HTML'
<h2>Shipping Policy</h2>
<p>Last Updated: February 18, 2026</p>

<h3>Scope</h3>
<p>This Shipping Policy applies only to physical goods sold through {SITE_NAME}. Digital downloads are delivered electronically, and external-link products are fulfilled by third parties.</p>

<h3>Processing Times</h3>
<p>Physical orders are generally processed within 1 to 2 business days unless a different timeframe is explicitly stated in product information or order confirmation.</p>

<h3>Delivery Times and Carriers</h3>
<p>Estimated delivery times depend on destination, customs, and carrier performance. Estimates are not guaranteed unless local law requires otherwise.</p>

<h3>Shipping Fees, Customs, and Import Taxes</h3>
<p>International customs duties and import taxes are the recipient’s responsibility unless expressly stated otherwise. Where fees are shown during order flow, those values apply to that order.</p>

<h3>Address Accuracy</h3>
<p>You are responsible for providing accurate shipping information. We are not responsible for delays or failed delivery caused by incorrect addresses provided at checkout.</p>

<h3>Delivery Issues</h3>
<p>If a shipment is delayed, lost, or damaged, contact us with your order number so we can investigate.</p>

<h3>Contact</h3>
<ul>
    <li>Name: {CONTACT_NAME}</li>
    <li>Email: {CONTACT_EMAIL}</li>
    <li>Phone: {CONTACT_PHONE}</li>
    <li>Address: {CONTACT_ADDRESS}</li>
</ul>

<h3>Local Consumer Rights</h3>
<p>Mandatory consumer delivery and cancellation rights under your local law remain unaffected.</p>
HTML;

$shippingStmt = $db->prepare("SELECT id, content FROM pages WHERE slug = 'shipping-policy' LIMIT 1");
$shippingStmt->execute();
$shippingRow = $shippingStmt->fetch();
if (!$shippingRow) {
    $insertStmt = $db->prepare(SQL_INSERT_PAGE_BASIC);
    $insertStmt->execute([PAGE_TITLE_SHIPPING_POLICY, 'shipping-policy', $defaultShippingContent, PAGE_TITLE_SHIPPING_POLICY, 'Shipping and delivery information', 0]);
} else {
    $shippingContent = (string)($shippingRow['content'] ?? '');
    $shouldUpdate = trim(strip_tags($shippingContent)) === ''
        || trim($shippingContent) === trim($legacyShippingContent)
        || trim($shippingContent) === trim($previousShippingContent)
        || trim($shippingContent) === trim($previousShippingContentV2);
    if ($shouldUpdate) {
        $stmt = $db->prepare(SQL_UPDATE_PAGE_CONTENT_META_BY_ID);
        $stmt->execute([$defaultShippingContent, PAGE_TITLE_SHIPPING_POLICY, 'Shipping and delivery information', $shippingRow['id']]);
    }
}

// Ensure Legal Notice page exists (system page)
// Snapshot set: legacy + previous + current default
$legacyLegalNoticeContent = <<<'HTML'
<h2>Legal Notice</h2>
<p>Last Updated: February 5, 2026</p>
<p>{SITE_NAME} operates {SITE_URL}.</p>
<p>For legal inquiries, contact {CONTACT_EMAIL}.</p>
HTML;

$previousLegalNoticeContent = <<<'HTML'
<h2>Legal Notice</h2>
<p>Last Updated: February 5, 2026</p>
<p>{SITE_NAME} operates {SITE_URL}.</p>
<h3>Operator</h3>
<ul>
    <li>Name: {CONTACT_NAME}</li>
    <li>Email: {CONTACT_EMAIL}</li>
    <li>Phone: {CONTACT_PHONE}</li>
    <li>Address: {CONTACT_ADDRESS}</li>
</ul>
HTML;

$defaultLegalNoticeContent = <<<'HTML'
<h2>Legal Notice</h2>
<p>Last Updated: February 5, 2026</p>
<p>{SITE_NAME} operates {SITE_URL}.</p>

<h3>Provider Information</h3>
<ul>
    <li>Legal Entity: {LEGAL_ENTITY_NAME}</li>
    <li>Legal Form: {LEGAL_LEGAL_FORM}</li>
    <li>Authorized Representative: {LEGAL_REPRESENTATIVE}</li>
    <li>Address: {LEGAL_FULL_ADDRESS}</li>
    <li>Email: {CONTACT_EMAIL}</li>
    <li>Phone: {CONTACT_PHONE}</li>
</ul>

<h3>Register Entry</h3>
<ul>
    <li>Register Court/Authority: {LEGAL_REGISTER_COURT}</li>
    <li>Register Number: {LEGAL_REGISTER_NUMBER}</li>
    <li>VAT ID: {LEGAL_VAT_ID}</li>
    <li>Business ID: {LEGAL_BUSINESS_ID}</li>
</ul>

<h3>Editorial Responsibility (§18 MStV)</h3>
<ul>
    <li>Responsible for content: {LEGAL_EDITOR_NAME}</li>
    <li>Address: {LEGAL_EDITOR_ADDRESS}</li>
</ul>

<h3>Regulated Profession (if applicable)</h3>
<ul>
    <li>Profession: {LEGAL_PROFESSION_NAME}</li>
    <li>Professional Title: {LEGAL_PROFESSIONAL_TITLE}</li>
    <li>Title awarded in: {LEGAL_PROFESSIONAL_TITLE_COUNTRY}</li>
    <li>Chamber / Professional body: {LEGAL_CHAMBER}</li>
    <li>Professional rules: {LEGAL_PROFESSIONAL_RULES}</li>
    <li>Rules URL: {LEGAL_PROFESSIONAL_RULES_LINK}</li>
</ul>

<h3>Supervisory Authority</h3>
<p>{LEGAL_SUPERVISORY_AUTHORITY}</p>

<h3>Online Dispute Resolution</h3>
<p>EU ODR platform: {LEGAL_ODR_LINK}</p>
<p>{LEGAL_DISPUTE_PARTICIPATION}</p>

<h3>Hosting Provider</h3>
<ul>
    <li>Name: {LEGAL_HOSTING_PROVIDER_NAME}</li>
    <li>Address: {LEGAL_HOSTING_PROVIDER_ADDRESS}</li>
</ul>
HTML;

$noticeStmt = $db->prepare("SELECT id, content FROM pages WHERE slug = 'legal-notice' LIMIT 1");
$noticeStmt->execute();
$noticeRow = $noticeStmt->fetch();
if (!$noticeRow) {
    $insertStmt = $db->prepare(SQL_INSERT_PAGE_BASIC);
    $insertStmt->execute([PAGE_TITLE_LEGAL_NOTICE, 'legal-notice', $defaultLegalNoticeContent, PAGE_TITLE_LEGAL_NOTICE, 'Legal notice and contact information', 0]);
} else {
    $noticeContent = (string)($noticeRow['content'] ?? '');
    $shouldUpdate = trim(strip_tags($noticeContent)) === ''
        || trim($noticeContent) === trim($legacyLegalNoticeContent)
        || trim($noticeContent) === trim($previousLegalNoticeContent);
    if ($shouldUpdate) {
        $stmt = $db->prepare(SQL_UPDATE_PAGE_CONTENT_META_BY_ID);
        $stmt->execute([$defaultLegalNoticeContent, PAGE_TITLE_LEGAL_NOTICE, 'Legal notice and contact information', $noticeRow['id']]);
    }
}

$legalTemplateDefaults = [
    'privacy-policy' => $defaultPrivacyContent,
    'terms-of-service' => $defaultTermsContent,
    'dmca' => $defaultDmcaContent,
    'refund-policy' => $defaultRefundContent,
    'shipping-policy' => $defaultShippingContent,
    'legal-notice' => $defaultLegalNoticeContent,
];

// Ensure Easy Media AI page exists (system page)
$emStmt = $db->prepare("SELECT id, content FROM pages WHERE slug = 'easy-media-ai' LIMIT 1");
$emStmt->execute();
$emRow = $emStmt->fetch();
if (!$emRow) {
    $defaultEasyMediaContent = <<<'HTML'
<div id="easyMediaApp" hidden>
    <div class="toolbar">
        <button class="btn hide-on-detail" data-em-action="select-folder" title="Open local folders to scan">📁 Select Folders</button>
        <button class="btn btn-reconnect hide-on-detail hidden" id="reconnectBtn" data-em-action="reconnect-folders" title="Reconnect to previously used folders">🔄 Reconnect</button>
        <button class="btn hide-on-detail" data-em-action="clear-all" title="Clear all loaded files and folders">🗑️ Clear</button>
        <input type="text" class="search-box" placeholder="🔍 Search files or tags..." data-em-input="search" title="Filter by filename or tag">
        <select class="sort-select hide-on-detail" data-em-change="type-filter" title="Filter by media type">
            <option value="all">All</option>
            <option value="images">Images only</option>
            <option value="videos">Videos only</option>
            <option value="untagged">Untagged only</option>
        </select>
        <button class="btn" data-em-action="manual-refresh" title="Rescan folders for new files">🔄 Refresh</button>
        <button class="btn em-btn-live" data-em-action="start-live" title="Start random slideshow">🔥 Live</button>
        <button class="btn em-btn-settings" data-em-action="toggle-settings" title="Settings">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <circle cx="12" cy="12" r="3"></circle>
                <path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1 0 2.83 2 2 0 0 1-2.83 0l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-2 2 2 2 0 0 1-2-2v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83 0 2 2 0 0 1 0-2.83l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1-2-2 2 2 0 0 1 2-2h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 0-2.83 2 2 0 0 1 2.83 0l.06.06a1.65 1.65 0 0 0 1.82.33H9a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 2-2 2 2 0 0 1 2 2v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 0 2 2 0 0 1 0 2.83l-.06.06a1.65 1.65 0 0 0-.33 1.82V9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 2 2 2 2 0 0 1-2 2h-.09a1.65 1.65 0 0 0-1.51 1z"></path>
            </svg>
        </button>
        <button class="btn em-btn-info" data-em-action="toggle-about" title="About Easy Media" aria-label="About Easy Media">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <circle cx="12" cy="12" r="10"></circle>
                <line x1="12" y1="16" x2="12" y2="12"></line>
                <line x1="12" y1="8" x2="12.01" y2="8"></line>
            </svg>
        </button>
        <span class="item-count" id="itemCount" title="Total visible items">0 items</span>
    </div>

    <div class="tags-bar hide-on-detail" id="tagsBar">
        <span class="tags-bar-label">Tags:</span>
        <span class="tag-pill active" data-tag="">All</span>
    </div>

    <div id="content">
        <div class="media-grid" id="mediaGrid"></div>
    </div>

    <div id="detailView" class="detail-view hidden">
        <div class="detail-header">
            <button class="btn" data-em-action="close-detail" title="Return to grid view">← Back</button>
            <button class="btn" data-em-action="nav-detail-prev" title="Previous item (Left Arrow)">❮ Prev</button>
            <button class="btn" data-em-action="nav-detail-next" title="Next item (Right Arrow)">Next ❯</button>
            <div class="detail-title" id="detailTitle" title="Current filename"></div>
            <button class="btn" data-em-action="open-viewer" title="Open in fullscreen lightbox">🔍 Viewer</button>
            <button class="btn" data-em-action="open-original" title="Download or Open File">⬇️ Download</button>
            <button class="btn" id="btnEditMedia" data-em-action="open-editor" title="Crop or Edit Image">🎨 Edit</button>
            <button class="btn btn-ai" id="btnAutoTag" data-em-action="auto-tag" title="Auto-tag using AI">🤖 AI Auto Tag</button>
            <button class="btn" data-em-action="rename-file" title="Rename this file">✏️ Rename</button>
            <button class="btn" data-em-action="open-tag-editor" title="Modify tags">🏷️ Edit Tags</button>
            <button class="btn" data-em-action="generate-id" title="Randomize filename ID">🔑 Generate ID</button>
            <button class="btn btn-danger" data-em-action="delete-file" title="Delete file permanently">🗑️ Delete</button>
        </div>
        <div class="detail-tags" id="detailTags"></div>
        <div class="detail-media" id="detailMedia"></div>
        <div class="detail-similar">
            <div class="similar-title">Similar (by tags)</div>
            <div class="media-grid" id="detailSimilarGrid"></div>
            <div class="detail-actions">
                <button class="btn" id="detailLoadMore" data-em-action="load-more-similar" title="Load more similar items">Show More</button>
            </div>
        </div>
    </div>

    <div class="lightbox" id="lightbox">
        <button class="lightbox-close" data-em-action="close-lightbox" title="Close viewer">&times;</button>
        <div class="lightbox-content" id="lightboxContent">
            <div id="lightboxMedia"></div>
        </div>
    </div>

    <div class="editor-modal" id="editorModal">
        <div class="editor-toolbar">
            <button class="btn" data-em-action="apply-crop">✂️ Crop</button>
            <button class="btn" data-em-action="save-editor-overwrite">💾 Save Overwrite</button>
            <button class="btn" data-em-action="save-editor-copy">💾 Save Copy</button>
            <button class="btn em-editor-cancel" data-em-action="close-editor">❌ Cancel</button>
        </div>
        <div class="editor-workspace" id="editorWorkspace">
            <canvas id="editorCanvas"></canvas>
            <div id="cropOverlay"></div>
        </div>
    </div>

    <div class="modal" id="settingsModal">
        <div class="modal-content em-settings-content">
            <div class="modal-title em-modal-title-cyan">Settings</div>

            <div class="em-settings-section">
                <label class="em-strict-label" title="If checked, files must have ONLY the selected tags and no others.">
                    <input type="checkbox" id="settingStrictTags" data-em-change="strict-tags">
                    <span><strong>Strict Tag Matching</strong><br><span class="em-strict-help">Show only files that have exactly the selected tags (no extra tags).</span></span>
                </label>

                <div class="settings-slider-group">
                    <div class="settings-slider">
                        <div class="settings-slider-header">
                            <span><strong>Card Size</strong></span>
                            <span id="cardSizeValue">280px</span>
                        </div>
                        <input type="range" id="settingCardSize" min="220" max="420" step="10" data-em-input="card-size">
                    </div>
                    <div class="settings-slider">
                        <div class="settings-slider-header">
                            <span><strong>Tag Size</strong></span>
                            <span id="tagSizeValue">13px</span>
                        </div>
                        <input type="range" id="settingTagSize" min="10" max="20" step="1" data-em-input="tag-size">
                    </div>
                </div>
            </div>

            <div class="modal-title em-modal-title-purple">Utilities</div>
            <div class="em-utilities-row">
                <button class="btn em-flex-btn" data-em-action="toggle-logs-settings">🐞 Debug Logs</button>
                <button class="btn em-flex-btn" id="failedBtnSettings" data-em-action="toggle-failed-settings">✓ Check Errors</button>
            </div>
            <div class="em-utilities-row em-utilities-row-download">
                <a class="btn em-flex-btn em-ai-zip-btn" data-em-ai-zip-download href="{EASY_MEDIA_AI_ZIP_URL}" target="_blank" rel="noopener" download>⬇️ Download Media AI ZIP</a>
            </div>

            <div class="toggle-row">
                <span class="toggle-label">Enable AI Auto Tag</span>
                <label class="toggle-switch">
                    <input type="checkbox" id="settingAiEnabled" data-em-change="ai-enabled">
                    <span class="toggle-slider"></span>
                </label>
            </div>
            <div class="settings-help" id="aiStatusText">AI status: Off</div>

            <div id="aiTrainingSection" class="hidden">
                <div class="settings-section-title">AI Training</div>
                <div class="settings-field">
                    <label class="settings-label" for="settingAiMaxSamples">Max Training Samples</label>
                    <input type="number" id="settingAiMaxSamples" class="modal-input" value="100" min="1" max="10000" data-em-change="ai-max-samples">
                    <div class="settings-help">Maximum files to use when training the AI</div>
                </div>
                <div class="settings-field">
                    <label class="settings-label">Server Batch Size</label>
                    <div class="settings-help" id="aiBatchSizeText">Batch size: Loading...</div>
                </div>
                <div class="training-controls">
                    <button class="btn btn-ai" id="btnTrainAi" data-em-action="train-ai">🧠 Train AI from Tagged Files</button>
                    <button class="btn btn-danger hidden" id="btnCancelTraining" data-em-action="cancel-training">⏹ Cancel Training</button>
                    <button class="btn" id="btnUpdateTagDb" data-em-action="update-tag-db">🔄 Update tag_db.json</button>
                </div>
                <div class="settings-help" id="tagDbStatusText">Tag DB status: Idle</div>
                <div class="training-status" id="aiTrainingStatus">Training status: Idle</div>
                <div class="training-meta">
                    <span id="aiTrainingProgressText">0/0</span>
                    <span id="aiTrainingLossEmbed" title="Tag Embedding Network">Embed: -</span>
                    <span id="aiTrainingLossClassifier" title="Classifier Network">Class: -</span>
                    <span id="aiTrainingLossClip" title="CLIP Fine-tune">CLIP: -</span>
                    <span id="aiTrainingLossVlm" title="VLM LoRA Fine-tune" class="hidden">VLM: -</span>
                </div>
                <div class="training-bar">
                    <div class="training-bar-fill" id="aiTrainingBar"></div>
                </div>
            </div>

            <div class="modal-buttons em-modal-footer">
                <button class="btn" data-em-action="toggle-settings">Close</button>
            </div>
        </div>
    </div>

    <div class="modal" id="aboutModal">
        <div class="modal-content em-about-modal-content">
            <div class="modal-title em-modal-title-cyan">About</div>
            <div class="em-about-content">
                <p><strong>Easy Media</strong> is your private, local media manager.</p>
                <h4 class="em-about-heading em-about-heading-cyan">🚀 How to Use</h4>
                <ul class="em-about-list">
                    <li><strong>Step 1:</strong> Click <span class="em-pill">📁 Select Folders</span>.</li>
                    <li><strong>Step 2:</strong> Use the <strong>Tag Filters</strong>. You can now select multiple tags! Use Settings to toggle "Strict" mode.</li>
                    <li><strong>Step 3:</strong> Click any item to enter <strong>Detail View</strong> for more actions.</li>
                </ul>

                <h4 class="em-about-heading em-about-heading-cyan">🤖 AI Server Quick Start</h4>
                <ul class="em-about-list">
                    <li><strong>Start here:</strong> Open <code>ai_tag_server\windows</code> and double-click <code>launcher.bat</code>.</li>
                    <li><strong>If Python is missing:</strong> Install Python (Microsoft Store or python.org), then run <code>launcher.bat</code> again.</li>
                    <li><strong>Tip:</strong> Keep the launcher window open while using AI features in Easy Media.</li>
                </ul>

                <h4 class="em-about-heading em-about-heading-purple">✨ Features</h4>
                <ul class="em-about-list">
                    <li>🏷️ <strong>Auto-Tagging:</strong> Rename files like <code>vacation_beach.jpg</code> to automatically tag them.</li>
                    <li>✂️ <strong>Editing:</strong> Be in Detail View and click <span class="em-pill em-pill-purple">🎨 Edit</span> to crop images.</li>
                    <li>🔥 <strong>Live Mode:</strong> Starts an infinite customized slideshow of your visible items.</li>
                    <li>⬇️ <strong>Download/Open:</strong> Saves a copy or opens the file in your default system viewer.</li>
                    <li>🤖 <strong>AI Auto Tag:</strong> Applies AI-generated tags to images in Detail View.</li>
                </ul>
                <div class="em-about-footer">&copy; 2026 {SITE_NAME}. All rights reserved.</div>
            </div>

            <div class="modal-buttons em-modal-footer">
                <button class="btn" data-em-action="toggle-about">Close</button>
            </div>
        </div>
    </div>

    <div class="toast" id="toast"></div>

    <div class="log-panel" id="logPanel">
        <div class="log-header">
            <span>Debug Logs</span>
            <div class="em-log-actions">
                <button class="btn em-log-btn" data-em-action="clear-logs">Clear</button>
                <button class="btn em-log-btn" data-em-action="toggle-logs">Close</button>
            </div>
        </div>
        <div class="log-content" id="logContent"></div>
    </div>

    <div class="modal" id="tagModal">
        <div class="modal-content">
            <div class="modal-title">Edit Tags</div>
            <input type="text" class="modal-input" id="tagInput" placeholder="Type a tag and press Enter...">
            <div class="modal-tags" id="modalTags"></div>
            <div class="modal-suggestions" id="modalSuggestions"></div>
            <div class="modal-buttons">
                <button class="btn btn-danger" data-em-action="close-tag-editor">Cancel</button>
                <button class="btn" data-em-action="save-tags">Save &amp; Rename</button>
            </div>
        </div>
    </div>

    <div class="modal" id="aiTagModal">
        <div class="modal-content">
            <div class="modal-title">AI Tag Suggestions</div>
            <div class="settings-help">Click tags to select/deselect before applying.</div>
            <div class="ai-caption hidden" id="aiTagCaption"></div>
            <div class="ai-tags-preview" id="aiTagPreview"></div>
            <div class="modal-buttons">
                <button class="btn btn-danger" data-em-action="close-ai-preview">Cancel</button>
                <button class="btn" data-em-action="apply-ai-preview">Apply Tags</button>
            </div>
        </div>
    </div>

    <div id="liveOverlay" class="live-overlay hidden">
        <div class="live-media-container" id="liveContainer"></div>
        <div class="live-controls">
            <button class="btn" data-em-action="live-fullscreen">⛶ Full</button>
            <div class="live-control-group">
                <label>Img Sec</label>
                <input type="number" step="0.1" min="0.1" value="0.3" class="live-input" id="liveImgTime" data-em-change="live-settings">
            </div>
            <div class="live-control-group">
                <label>Vid Sec</label>
                <input type="number" step="0.1" min="0.1" value="0.9" class="live-input" id="liveVidTime" data-em-change="live-settings">
            </div>
            <div class="live-control-group em-live-control-center">
                <label>Zoom</label>
                <input type="checkbox" id="liveZoomEnabled" checked data-em-change="live-settings">
            </div>
            <div class="live-control-group">
                <label>Intensity</label>
                <input type="range" min="1.1" max="2.0" step="0.1" value="1.5" id="liveZoomAmt" data-em-change="live-settings">
            </div>
            <button class="btn btn-danger" data-em-action="stop-live">✖ Stop</button>
        </div>
    </div>
</div>
HTML;
    $insertStmt = $db->prepare(SQL_INSERT_PAGE_BASIC);
    $insertStmt->execute(['Easy Media AI', 'easy-media-ai', $defaultEasyMediaContent, 'Easy Media AI', 'Private local media manager with AI tagging', 1]);
} else {
    $emContent = (string)($emRow['content'] ?? '');
    $emUpdated = false;
    $needsCaption = strpos($emContent, 'id="aiTagCaption"') === false
        && strpos($emContent, 'id="aiTagPreview"') !== false;
    if ($needsCaption) {
        $needle = '<div class="ai-tags-preview" id="aiTagPreview"></div>';
        $captionMarkup = '<div class="ai-caption hidden" id="aiTagCaption"></div>';
        if (strpos($emContent, $needle) !== false) {
            $emContent = str_replace($needle, $captionMarkup . "\n            " . $needle, $emContent);
            $emUpdated = true;
        }
    }

    if (strpos($emContent, 'em-compat-note') !== false) {
        $compatBlock = "\n\n    <div class=\"em-compat-note hide-on-detail\" id=\"emCompatNote\">\n        <span class=\"em-compat-title\">Firefox notice:</span>\n        <span>Folder selection requires a Chromium-based browser (Chrome, Edge, Opera). Firefox does not support this API yet.</span>\n    </div>";
        $emContent = str_replace($compatBlock, '', $emContent);
        $emContent = str_replace("<div class=\"em-compat-note hide-on-detail\" id=\"emCompatNote\">", '', $emContent);
        $emContent = str_replace("<span class=\"em-compat-title\">Firefox notice:</span>", '', $emContent);
        $emContent = str_replace("<span>Folder selection requires a Chromium-based browser (Chrome, Edge, Opera). Firefox does not support this API yet.</span>", '', $emContent);
        $emContent = str_replace("</div>", "</div>", $emContent);
        $emUpdated = true;
    }

    if (strpos($emContent, 'em-compat-step') !== false) {
        $emContent = str_replace("<li class=\"em-compat-step\"><strong>Firefox note:</strong> Folder selection isn’t supported in Firefox yet.</li>", '', $emContent);
        $emUpdated = true;
    }

    $compatStepNeedle = '<li><strong>Step 1:</strong> Click <span class="em-pill">📁 Select Folders</span> (Chrome/Edge/Opera).</li>';
    if (strpos($emContent, $compatStepNeedle) !== false) {
        $emContent = str_replace($compatStepNeedle, '<li><strong>Step 1:</strong> Click <span class="em-pill">📁 Select Folders</span>.</li>', $emContent);
        $emUpdated = true;
    }

    if (strpos($emContent, 'em-compat-btn') !== false) {
        $emContent = preg_replace('/<button[^>]*class="[^"]*em-compat-btn[^"]*"[^>]*>.*?<\/button>\s*/s', '', $emContent);
        $emUpdated = true;
    }

    if (strpos($emContent, 'title="Settings &amp; About"') !== false) {
        $emContent = str_replace('title="Settings &amp; About"', 'title="Settings"', $emContent);
        $emUpdated = true;
    }

    if (strpos($emContent, 'data-em-action="toggle-about"') === false) {
        $settingsBtnStart = strpos($emContent, '<button class="btn em-btn-settings"');
        if ($settingsBtnStart !== false) {
            $settingsBtnEnd = strpos($emContent, '</button>', $settingsBtnStart);
            if ($settingsBtnEnd !== false) {
                $settingsBtnEnd += strlen('</button>');
                $infoBtnMarkup = "\n        <button class=\"btn em-btn-info\" data-em-action=\"toggle-about\" title=\"About Easy Media\" aria-label=\"About Easy Media\">\n            <svg width=\"20\" height=\"20\" viewBox=\"0 0 24 24\" fill=\"none\" stroke=\"currentColor\" stroke-width=\"2\">\n                <circle cx=\"12\" cy=\"12\" r=\"10\"></circle>\n                <line x1=\"12\" y1=\"16\" x2=\"12\" y2=\"12\"></line>\n                <line x1=\"12\" y1=\"8\" x2=\"12.01\" y2=\"8\"></line>\n            </svg>\n        </button>";
                $emContent = substr_replace($emContent, $infoBtnMarkup, $settingsBtnEnd, 0);
                $emUpdated = true;
            }
        }
    }

    $downloadHrefReplaceCount = 0;
    $downloadHrefUpdated = preg_replace(
        '/(<a\b[^>]*data-em-ai-zip-download[^>]*\bhref=")([^"]*)("[^>]*>)/i',
        '$1{EASY_MEDIA_AI_ZIP_URL}$3',
        $emContent,
        1,
        $downloadHrefReplaceCount
    );
    if ($downloadHrefReplaceCount > 0 && $downloadHrefUpdated !== null) {
        $emContent = $downloadHrefUpdated;
        $emUpdated = true;
    }

    if (strpos($emContent, 'data-em-ai-zip-download') === false) {
        $downloadRowMarkup = "\n            <div class=\"em-utilities-row em-utilities-row-download\">\n                <a class=\"btn em-flex-btn em-ai-zip-btn\" data-em-ai-zip-download href=\"{EASY_MEDIA_AI_ZIP_URL}\" target=\"_blank\" rel=\"noopener\" download>⬇️ Download Media AI ZIP</a>\n            </div>";
        $downloadInserted = false;
        $failedBtnPos = strpos($emContent, 'id="failedBtnSettings"');
        if ($failedBtnPos !== false) {
            $utilitiesRowEnd = strpos($emContent, '</div>', $failedBtnPos);
            if ($utilitiesRowEnd !== false) {
                $utilitiesRowEnd += strlen('</div>');
                $emContent = substr_replace($emContent, $downloadRowMarkup, $utilitiesRowEnd, 0);
                $downloadInserted = true;
            }
        }

        if (!$downloadInserted) {
            $toggleRowPos = strpos($emContent, '<div class="toggle-row">');
            if ($toggleRowPos !== false) {
                $emContent = substr_replace($emContent, $downloadRowMarkup . "\n\n            ", $toggleRowPos, 0);
                $downloadInserted = true;
            }
        }

        if ($downloadInserted) {
            $emUpdated = true;
        }
    }

    $aiServerQuickStartHeading = '<h4 class="em-about-heading em-about-heading-cyan">🤖 AI Server Quick Start</h4>';
    $featuresHeadingNeedle = '<h4 class="em-about-heading em-about-heading-purple">✨ Features</h4>';
    if (strpos($emContent, $aiServerQuickStartHeading) === false) {
        $featuresHeadingPos = strpos($emContent, $featuresHeadingNeedle);
        if ($featuresHeadingPos !== false) {
            $aiServerQuickStartMarkup = <<<'HTML'
                <h4 class="em-about-heading em-about-heading-cyan">🤖 AI Server Quick Start</h4>
                <ul class="em-about-list">
                    <li><strong>Start here:</strong> Open <code>ai_tag_server\windows</code> and double-click <code>launcher.bat</code>.</li>
                    <li><strong>If Python is missing:</strong> Install Python (Microsoft Store or python.org), then run <code>launcher.bat</code> again.</li>
                    <li><strong>Tip:</strong> Keep the launcher window open while using AI features in Easy Media.</li>
                </ul>
HTML;
            $emContent = substr_replace($emContent, $aiServerQuickStartMarkup . "\n\n                ", $featuresHeadingPos, 0);
            $emUpdated = true;
        }
    }

    $aboutModalExists = strpos($emContent, 'id="aboutModal"') !== false;
    $aboutTitleNeedle = '<div class="modal-title em-modal-title-cyan">About</div>';
    $aboutSectionMarkup = '';
    if (!$aboutModalExists) {
        $aboutTitlePos = strpos($emContent, $aboutTitleNeedle);
        if ($aboutTitlePos !== false) {
            $aboutFooterPos = strpos($emContent, '<div class="modal-buttons em-modal-footer">', $aboutTitlePos);
            if ($aboutFooterPos !== false) {
                $aboutSectionMarkup = trim(substr($emContent, $aboutTitlePos, $aboutFooterPos - $aboutTitlePos));
                $emContent = substr_replace($emContent, '', $aboutTitlePos, $aboutFooterPos - $aboutTitlePos);
                $emUpdated = true;
            }
        }
    }

    if (!$aboutModalExists) {
        if ($aboutSectionMarkup === '') {
            $aboutSectionMarkup = '<div class="modal-title em-modal-title-cyan">About</div>'
                . "\n            <div class=\"em-about-content\">"
                . "\n                <p><strong>Easy Media</strong> is your private, local media manager.</p>"
                . "\n                <h4 class=\"em-about-heading em-about-heading-cyan\">🚀 How to Use</h4>"
                . "\n                <ul class=\"em-about-list\">"
                . "\n                    <li><strong>Step 1:</strong> Click <span class=\"em-pill\">📁 Select Folders</span>.</li>"
                . "\n                    <li><strong>Step 2:</strong> Use the <strong>Tag Filters</strong>. You can now select multiple tags! Use Settings to toggle \"Strict\" mode.</li>"
                . "\n                    <li><strong>Step 3:</strong> Click any item to enter <strong>Detail View</strong> for more actions.</li>"
                . "\n                </ul>"
                . "\n"
                . "\n                <h4 class=\"em-about-heading em-about-heading-cyan\">🤖 AI Server Quick Start</h4>"
                . "\n                <ul class=\"em-about-list\">"
                . "\n                    <li><strong>Start here:</strong> Open <code>ai_tag_server\\windows</code> and double-click <code>launcher.bat</code>.</li>"
                . "\n                    <li><strong>If Python is missing:</strong> Install Python (Microsoft Store or python.org), then run <code>launcher.bat</code> again.</li>"
                . "\n                    <li><strong>Tip:</strong> Keep the launcher window open while using AI features in Easy Media.</li>"
                . "\n                </ul>"
                . "\n"
                . "\n                <h4 class=\"em-about-heading em-about-heading-purple\">✨ Features</h4>"
                . "\n                <ul class=\"em-about-list\">"
                . "\n                    <li>🏷️ <strong>Auto-Tagging:</strong> Rename files like <code>vacation_beach.jpg</code> to automatically tag them.</li>"
                . "\n                    <li>✂️ <strong>Editing:</strong> Be in Detail View and click <span class=\"em-pill em-pill-purple\">🎨 Edit</span> to crop images.</li>"
                . "\n                    <li>🔥 <strong>Live Mode:</strong> Starts an infinite customized slideshow of your visible items.</li>"
                . "\n                    <li>⬇️ <strong>Download/Open:</strong> Saves a copy or opens the file in your default system viewer.</li>"
                . "\n                    <li>🤖 <strong>AI Auto Tag:</strong> Applies AI-generated tags to images in Detail View.</li>"
                . "\n                </ul>"
                . "\n                <div class=\"em-about-footer\">&copy; 2026 {SITE_NAME}. All rights reserved.</div>"
                . "\n            </div>";
        }

        $aboutModalMarkup = "\n\n    <div class=\"modal\" id=\"aboutModal\">\n        <div class=\"modal-content em-about-modal-content\">\n            "
            . $aboutSectionMarkup
            . "\n\n            <div class=\"modal-buttons em-modal-footer\">\n                <button class=\"btn\" data-em-action=\"toggle-about\">Close</button>\n            </div>\n        </div>\n    </div>";

        $toastNeedle = "\n\n    <div class=\"toast\" id=\"toast\"></div>";
        if (strpos($emContent, $toastNeedle) !== false) {
            $emContent = str_replace($toastNeedle, $aboutModalMarkup . $toastNeedle, $emContent);
        } else {
            $emContent .= $aboutModalMarkup;
        }
        $emUpdated = true;
    }

    if ($emUpdated) {
        $stmt = $db->prepare("UPDATE pages SET content = ? WHERE id = ?");
        $stmt->execute([$emContent, $emRow['id']]);
    }
}

// Ensure Text Finder & Replacer page exists (system page)
$textFinderStmt = $db->prepare("SELECT id FROM pages WHERE slug = 'text-finder-replacer' LIMIT 1");
$textFinderStmt->execute();
if (!$textFinderStmt->fetch()) {
    $defaultTextFinderContent = '<h2>Text Finder &amp; Replacer</h2><p>Find and replace text in pasted content, text files, or zip archives containing text files.</p><p>Open the tool at <a href="/text-finder-replacer">/text-finder-replacer</a>.</p>';
    $insertStmt = $db->prepare(SQL_INSERT_PAGE_BASIC);
    $insertStmt->execute(['Text Finder & Replacer', 'text-finder-replacer', $defaultTextFinderContent, 'Text Finder & Replacer', 'Find and replace text in pasted content and uploaded text files.', 1]);
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_GET['action'] ?? 'save';

    if ($action === 'restore-template') {
        verifyCSRF();
        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) {
            header(REDIRECT_ADMIN_PAGES_ERROR . urlencode('Invalid page selection for template restore.'));
            exit;
        }

        $pageStmt = $db->prepare("SELECT slug FROM pages WHERE id = ? LIMIT 1");
        $pageStmt->execute([$id]);
        $pageRow = $pageStmt->fetch(PDO::FETCH_ASSOC);
        $slug = (string)($pageRow['slug'] ?? '');

        if ($slug === '' || !isset($legalTemplateDefaults[$slug])) {
            header('Location: /admin/pages.php?action=edit&id=' . $id . '&error=' . urlencode('This page does not have a legal template to restore.'));
            exit;
        }

        $updateStmt = $db->prepare("UPDATE pages SET content = ?, updated_at = ? WHERE id = ?");
        $updateStmt->execute([$legalTemplateDefaults[$slug], DbHelper::nowString(), $id]);

        header('Location: /admin/pages.php?action=edit&id=' . $id . '&message=' . urlencode('Default legal template restored.'));
        exit;
    }

    if ($action === 'delete') {
        verifyCSRF();
        $id = $_POST['id'] ?? null;
        if ($id) {
            $stmt = $db->prepare("SELECT slug FROM pages WHERE id = ?");
            $stmt->execute([$id]);
            $row = $stmt->fetch();
            if ($row && in_array((string)$row['slug'], $systemPageSlugs, true)) {
                header(REDIRECT_ADMIN_PAGES_MESSAGE . urlencode('System pages cannot be deleted.'));
                exit;
            }
            $stmt = $db->prepare("DELETE FROM pages WHERE id = ?");
            $stmt->execute([$id]);
            header(REDIRECT_ADMIN_PAGES_MESSAGE . urlencode('Page deleted.'));
            exit;
        }
    }

    verifyCSRF();
    $pageId = $_POST['id'] ?? null;
    $title = trim($_POST['title'] ?? '');
    $slug = trim($_POST['slug'] ?? '');
    $content = $_POST['content'] ?? '';
    $metaTitle = trim($_POST['meta_title'] ?? '');
    $metaDescription = trim($_POST['meta_description'] ?? '');
    $isPublished = isset($_POST['is_published']) ? 1 : 0;
    $useLayoutOverride = isset($_POST['use_layout_override']) ? 1 : 0;
    $contentMaxWidth = $_POST['content_max_width'] !== '' ? (int)$_POST['content_max_width'] : null;
    $sectionPadding = $_POST['section_padding'] !== '' ? (int)$_POST['section_padding'] : null;
    $sectionHeight = $_POST['section_height'] !== '' ? (int)$_POST['section_height'] : null;

    if (empty($title)) {
        $error = 'Title is required.';
    } else {
        try {
            $systemSlugLocked = false;
            $editingSystemPage = false;

            // Generate slug if empty
            if (empty($slug)) {
                $slug = SlugGenerator::generate($title);
            } else {
                $slug = SlugGenerator::generate($slug);
            }

            if ($pageId) {
                $existingPageStmt = $db->prepare("SELECT slug FROM pages WHERE id = ? LIMIT 1");
                $existingPageStmt->execute([$pageId]);
                $existingPage = $existingPageStmt->fetch();
                if ($existingPage) {
                    $existingSlug = (string)($existingPage['slug'] ?? '');
                    $editingSystemPage = in_array($existingSlug, $systemPageSlugs, true);
                    if ($editingSystemPage) {
                        if ($slug !== $existingSlug) {
                            $systemSlugLocked = true;
                        }
                        $slug = $existingSlug;
                    }
                }
            }

            if (!$editingSystemPage) {
                // Check for duplicate slug
                $checkStmt = $db->prepare("SELECT id FROM pages WHERE slug = ? AND id != ?");
                $checkStmt->execute([$slug, $pageId ?? 0]);

                if ($checkStmt->fetch()) {
                    $slug .= '-' . time();
                }
            }

            if (!$error) {
                if ($pageId) {
                    // Update
                    $now = DbHelper::nowString();
                    $stmt = $db->prepare("UPDATE pages SET title = ?, slug = ?, content = ?, meta_title = ?, meta_description = ?, is_published = ?, use_layout_override = ?, content_max_width = ?, section_padding = ?, section_height = ?, updated_at = ? WHERE id = ?");
                    $stmt->execute([$title, $slug, $content, $metaTitle, $metaDescription, $isPublished, $useLayoutOverride, $contentMaxWidth, $sectionPadding, $sectionHeight, $now, $pageId]);
                    $message = $systemSlugLocked
                        ? 'System page slug is locked. Other changes were saved.'
                        : 'Page updated successfully.';
                } else {
                    // Create
                    $stmt = $db->prepare("INSERT INTO pages (title, slug, content, meta_title, meta_description, is_published, use_layout_override, content_max_width, section_padding, section_height) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                    $stmt->execute([$title, $slug, $content, $metaTitle, $metaDescription, $isPublished, $useLayoutOverride, $contentMaxWidth, $sectionPadding, $sectionHeight]);
                    $pageId = DbHelper::lastInsertId($db, 'pages');
                    $message = 'Page created successfully.';
                }
            }

            if (!$error) {
                header(REDIRECT_ADMIN_PAGES_MESSAGE . urlencode($message));
                exit;
            }
        } catch (PDOException $e) {
            $error = 'Database error occurred while saving the page.';
            LogService::add('error', 'Page save failed: ' . $e->getMessage(), json_encode(['page_id' => $pageId, 'title' => $title]));
        }
    }
}

// Handle delete
if ($action === 'delete' && isset($_GET['id'])) {
    // Moved to POST for CSRF protection
    header(REDIRECT_ADMIN_PAGES);
    exit;
}

// Get page for editing
$page = null;
if ($action === 'edit' && isset($_GET['id'])) {
    $stmt = $db->prepare("SELECT * FROM pages WHERE id = ?");
    $stmt->execute([$_GET['id']]);
    $page = $stmt->fetch();

    if (!$page) {
        header(REDIRECT_ADMIN_PAGES);
        exit;
    }
}

$isEditingSystemPage = $page && in_array((string)($page['slug'] ?? ''), $systemPageSlugs, true);
$isEditingLegalTemplatePage = $page && isset($legalTemplateDefaults[(string)($page['slug'] ?? '')]);

// Get all pages for listing
$pages = $db->query("SELECT * FROM pages ORDER BY title")->fetchAll();

// Check for message
if (isset($_GET['message'])) {
    $message = $_GET['message'];
}
if (isset($_GET['error'])) {
    $error = $_GET['error'];
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <?php if ($action === 'list'): ?>
    <!-- Pages List -->
    <div class="content-header content-header-actions">
        <div>
            <h1>Pages</h1>
            <p>Manage your website pages</p>
        </div>
        <a href="/admin/pages.php?action=new" class="btn btn-primary">+ New Page</a>
    </div>

    <?php if ($message): ?>
    <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
    <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <div class="table-container">
        <table class="data-table">
            <thead>
                <tr>
                    <th>Title</th>
                    <th>Slug</th>
                    <th>Status</th>
                    <th>Updated</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php
                foreach ($pages as $p):
                    $isSystemPage = in_array((string)$p['slug'], $systemPageSlugs, true);
                    ?>
                <tr>
                    <td>
                        <?php echo e($p['title']); ?>
                        <?php if ($isSystemPage): ?>
                        <span class="page-system-badge">System</span>
                        <?php endif; ?>
                    </td>
                    <td><code>/<?php echo e($p['slug']); ?></code></td>
                    <td>
                        <?php if ($p['is_published']): ?>
                        <span class="status-published">Published</span>
                        <?php else: ?>
                        <span class="status-draft">Draft</span>
                        <?php endif; ?>
                    </td>
                    <td><?php echo date('M j, Y', strtotime($p['updated_at'])); ?></td>
                    <td class="table-actions">
                        <a href="/admin/theme-editor.php?page=<?php echo $p['id']; ?>" class="btn btn-secondary btn-sm">Edit Layout</a>
                        <a href="/admin/pages.php?action=edit&id=<?php echo $p['id']; ?>" class="btn btn-secondary btn-sm">Edit</a>
                        <a href="/<?php echo e($p['slug']); ?>" target="_blank" class="btn btn-secondary btn-sm">View</a>
                        <?php if (!$isSystemPage): ?>
                        <form method="POST" action="/admin/pages.php?action=delete" class="inline-block" data-confirm="Delete this page?">
                            <input type="hidden" name="id" value="<?php echo $p['id']; ?>">
                            <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                            <button type="submit" class="btn btn-danger btn-sm">Delete</button>
                        </form>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>

    <?php else: ?>
    <!-- New/Edit Page Form -->
    <div class="content-header">
        <h1><?php echo $page ? 'Edit Page' : 'New Page'; ?></h1>
        <p><a href="/admin/pages.php">← Back to Pages</a></p>
    </div>

    <?php if ($error): ?>
    <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <form method="POST" action="/admin/pages.php?action=<?php echo $page ? 'edit' : 'new'; ?><?php echo $page ? '&id=' . $page['id'] : ''; ?>">
        <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
        <?php if ($page): ?>
        <input type="hidden" name="id" value="<?php echo $page['id']; ?>">
        <?php endif; ?>

        <div class="card">
            <div class="card-body">
                <div class="form-row">
                    <div class="form-group">
                        <label for="title">Page Title</label>
                        <input type="text" id="title" name="title" class="form-control" value="<?php echo e($page['title'] ?? ''); ?>" required>
                    </div>

                    <div class="form-group">
                        <label for="slug">URL Slug</label>
                        <input type="text" id="slug" name="slug" class="form-control" value="<?php echo e($page['slug'] ?? ''); ?>" placeholder="Leave empty to auto-generate" <?php echo $isEditingSystemPage ? 'readonly aria-readonly="true"' : ''; ?>>
                        <?php if ($isEditingSystemPage): ?>
                        <small class="text-muted">System page slug is locked to protect built-in routing.</small>
                        <?php endif; ?>
                    </div>
                </div>

                <div class="form-group">
                    <div class="content-header-actions">
                        <label for="content">Content (HTML)</label>
                        <?php if ($isEditingLegalTemplatePage): ?>
                        <button
                            type="submit"
                            class="btn btn-secondary btn-sm"
                            formaction="/admin/pages.php?action=restore-template&id=<?php echo (int)$page['id']; ?>"
                            formmethod="POST"
                            name="restore_template"
                            value="1"
                        >
                            Restore Default Template
                        </button>
                        <?php endif; ?>
                    </div>
                    <textarea id="content" name="content" class="form-control" rows="10"><?php echo e($page['content'] ?? ''); ?></textarea>
                    <small class="text-muted">Placeholders: {SITE_NAME}, {SITE_URL}, {CONTACT_EMAIL}, {CONTACT_NAME}, {CONTACT_PHONE}, {CONTACT_ADDRESS}, {LEGAL_ENTITY_NAME}, {LEGAL_FULL_ADDRESS}, {LEGAL_REGISTER_COURT}, {LEGAL_REGISTER_NUMBER}, {LEGAL_VAT_ID}, {LEGAL_EDITOR_NAME}, {LEGAL_EDITOR_ADDRESS}, {LEGAL_SUPERVISORY_AUTHORITY}, {LEGAL_ODR_LINK}, {LEGAL_HOSTING_PROVIDER_NAME}, {LEGAL_HOSTING_PROVIDER_ADDRESS}, {EASY_MEDIA_AI_ZIP_URL}</small>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="meta_title">Meta Title</label>
                        <input type="text" id="meta_title" name="meta_title" class="form-control" value="<?php echo e($page['meta_title'] ?? ''); ?>">
                    </div>

                    <div class="form-group">
                        <label for="meta_description">Meta Description</label>
                        <input type="text" id="meta_description" name="meta_description" class="form-control" value="<?php echo e($page['meta_description'] ?? ''); ?>">
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="is_published" <?php echo ($page['is_published'] ?? 1) ? 'checked' : ''; ?>>
                        <span>Published</span>
                    </label>
                </div>

                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="use_layout_override" <?php echo ($page['use_layout_override'] ?? 0) ? 'checked' : ''; ?>>
                        <span>Override Global Layout Settings</span>
                    </label>
                    <small class="text-muted">Enable to use the page-specific width, padding, and height values below.</small>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="content_max_width">Content Max Width (px)</label>
                        <input type="number" id="content_max_width" name="content_max_width" class="form-control" min="900" max="2200" step="50" value="<?php echo e($page['content_max_width'] ?? ''); ?>" placeholder="Use global">
                    </div>
                    <div class="form-group">
                        <label for="section_padding">Section Vertical Padding (px)</label>
                        <input type="number" id="section_padding" name="section_padding" class="form-control" min="20" max="160" step="5" value="<?php echo e($page['section_padding'] ?? ''); ?>" placeholder="Use global">
                    </div>
                    <div class="form-group">
                        <label for="section_height">Section Height (vh)</label>
                        <input type="number" id="section_height" name="section_height" class="form-control" min="40" max="100" step="5" value="<?php echo e($page['section_height'] ?? ''); ?>" placeholder="Use global">
                    </div>
                </div>
            </div>

            <div class="card-footer">
                <a href="/admin/pages.php" class="btn btn-secondary">Cancel</a>
                <button type="submit" class="btn btn-primary">Save Page</button>
            </div>
        </div>
    </form>
    <?php endif; ?>
</div>

<?php AdminLayout::renderFooter(); ?>
