<?php
/**
 * Admin Panel - Mini-Games
 *
 * Allows enabling/disabling mini-games (hides from /games and 404s direct access).
 */

declare(strict_types=1);

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Auth\AdminAuth;
use NewSite\Admin\AdminLayout;
use NewSite\Database\DatabaseManager;

AdminAuth::requireLogin();

$message = '';
$error = '';

$catalog = \NewSite\Minigames\MinigameCatalog::getCatalog();

// Current DB states (if table exists, otherwise defaults apply).
try {
    $readDb = DatabaseManager::getReadConnection();
} catch (Throwable $e) {
    $readDb = null;
}

$states = [];
$gameplaySettingsBySlug = [];
if ($readDb instanceof PDO) {
    $states = \NewSite\Minigames\MinigameRepository::fetchDbStates($readDb);
    foreach ($catalog as $slug => $game) {
        if (!is_array($game)) {
            continue;
        }
        $s = \NewSite\Minigames\MinigameCatalog::normalizeSlug((string)($game['slug'] ?? $slug));
        if ($s === '') {
            continue;
        }
        $defaults = \NewSite\Minigames\MinigameCatalog::getGameplayDefaults($s);
        if (!empty($defaults)) {
            $gameplaySettingsBySlug[$s] = \NewSite\Minigames\MinigameRepository::getGameplaySettings($readDb, $s);
        }
    }
} else {
    foreach ($catalog as $slug => $game) {
        if (!is_array($game)) {
            continue;
        }
        $s = \NewSite\Minigames\MinigameCatalog::normalizeSlug((string)($game['slug'] ?? $slug));
        if ($s === '') {
            continue;
        }
        $defaults = \NewSite\Minigames\MinigameCatalog::getGameplayDefaults($s);
        if (!empty($defaults)) {
            $gameplaySettingsBySlug[$s] = $defaults;
        }
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && (string)($_POST['action'] ?? '') === 'save') {
    verifyCSRF();

    $enabled = $_POST['enabled'] ?? [];
    if (!is_array($enabled)) {
        $enabled = [];
    }

    $enabledSet = [];
    foreach ($enabled as $slug) {
        $s = \NewSite\Minigames\MinigameCatalog::normalizeSlug((string)$slug);
        if ($s !== '') {
            $enabledSet[$s] = true;
        }
    }

    $submittedSettings = $_POST['settings'] ?? [];
    if (!is_array($submittedSettings)) {
        $submittedSettings = [];
    }

    try {
        $writeDb = DatabaseManager::getWriteConnection();
        if ($writeDb instanceof PDO) {
            foreach ($catalog as $slug => $game) {
                $s = \NewSite\Minigames\MinigameCatalog::normalizeSlug((string)($game['slug'] ?? $slug));
                if ($s === '') {
                    continue;
                }
                $isEnabled = isset($enabledSet[$s]);
                \NewSite\Minigames\MinigameRepository::setEnabled($writeDb, $s, $isEnabled);

                $defaults = \NewSite\Minigames\MinigameCatalog::getGameplayDefaults($s);
                if (!empty($defaults)) {
                    $oneGameInput = $submittedSettings[$s] ?? [];
                    if (!is_array($oneGameInput)) {
                        $oneGameInput = [];
                    }
                    \NewSite\Minigames\MinigameRepository::setGameplaySettings($writeDb, $s, $oneGameInput);
                }
            }
            $message = 'Mini-games and gameplay settings updated.';
        } else {
            $error = 'Database connection is not available.';
        }
    } catch (Throwable $e) {
        $error = 'Could not update mini-games.';
    }

    // Re-read states (avoid cached view).
    if ($readDb instanceof PDO) {
        $states = \NewSite\Minigames\MinigameRepository::fetchDbStates($readDb);
        foreach ($catalog as $slug => $game) {
            if (!is_array($game)) {
                continue;
            }
            $s = \NewSite\Minigames\MinigameCatalog::normalizeSlug((string)($game['slug'] ?? $slug));
            if ($s === '') {
                continue;
            }
            $defaults = \NewSite\Minigames\MinigameCatalog::getGameplayDefaults($s);
            if (!empty($defaults)) {
                $gameplaySettingsBySlug[$s] = \NewSite\Minigames\MinigameRepository::getGameplaySettings($readDb, $s);
            }
        }
    }
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header">
        <h1>Mini-Games</h1>
        <p>Enable or disable mini-games. Disabled games are hidden from <code>/games</code> and return a 404 if accessed directly.</p>
    </div>

    <?php if ($message): ?>
        <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <form method="POST" action="">
        <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
        <input type="hidden" name="action" value="save">

        <div class="card">
            <div class="card-header">
                <h3>Games</h3>
            </div>
            <div class="card-body">
                <?php if (empty($catalog)): ?>
                    <p>No games are registered yet.</p>
                <?php else: ?>
                    <div class="table-container">
                        <table class="admin-table mini-games-table">
                            <thead>
                                <tr>
                                    <th class="mini-games-col-enabled">Enabled</th>
                                    <th>Game</th>
                                    <th class="mini-games-col-url">URL</th>
                                    <th>Gameplay Tuning</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($catalog as $slug => $game): ?>
                                    <?php
                                    if (!is_array($game)) {
                                        continue;
                                    }
                                    $s = \NewSite\Minigames\MinigameCatalog::normalizeSlug((string)($game['slug'] ?? $slug));
                                    if ($s === '') {
                                        continue;
                                    }
                                    $title = (string)($game['title'] ?? $s);
                                    $desc = (string)($game['description'] ?? '');
                                    $isEnabled = $states[$s] ?? true;
                                    $checkboxId = 'mini-game-enabled-' . $s;
                                    $defaults = \NewSite\Minigames\MinigameCatalog::getGameplayDefaults($s);
                                    $settings = $gameplaySettingsBySlug[$s] ?? $defaults;
                                    ?>
                                    <tr>
                                        <td>
                                            <div class="form-check">
                                                <input type="checkbox" id="<?php echo e($checkboxId); ?>" name="enabled[]" value="<?php echo e($s); ?>" <?php echo $isEnabled ? 'checked' : ''; ?>>
                                                <label for="<?php echo e($checkboxId); ?>" class="sr-only">Enabled</label>
                                            </div>
                                        </td>
                                        <td class="mini-games-col-game">
                                            <div class="mini-games-row">
                                                <div class="mini-games-icon" aria-hidden="true"><?php echo e((string)($game['icon'] ?? '🎮')); ?></div>
                                                <div class="mini-games-info">
                                                    <div class="mini-games-title"><?php echo e($title); ?></div>
                                                    <?php if ($desc !== ''): ?>
                                                        <div class="mini-games-desc"><?php echo e($desc); ?></div>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </td>
                                        <td>
                                            <code class="mini-games-url">/games/<?php echo e($s); ?></code>
                                        </td>
                                        <td>
                                            <?php if (empty($defaults)): ?>
                                                <span class="text-muted">No tunable gameplay settings.</span>
                                            <?php else: ?>
                                                <div class="mini-games-settings-grid">
                                                    <div class="form-group">
                                                        <label for="mg-<?php echo e($s); ?>-multiplier">Multiplier step</label>
                                                        <input
                                                            type="number"
                                                            class="form-control"
                                                            id="mg-<?php echo e($s); ?>-multiplier"
                                                            name="settings[<?php echo e($s); ?>][multiplier_step]"
                                                            min="1.01"
                                                            max="3"
                                                            step="0.01"
                                                            value="<?php echo e((string)($settings['multiplier_step'] ?? $defaults['multiplier_step'])); ?>"
                                                        >
                                                    </div>

                                                    <div class="form-group">
                                                        <label for="mg-<?php echo e($s); ?>-start-time">Start time (s)</label>
                                                        <input
                                                            type="number"
                                                            class="form-control"
                                                            id="mg-<?php echo e($s); ?>-start-time"
                                                            name="settings[<?php echo e($s); ?>][start_time_seconds]"
                                                            min="5"
                                                            max="240"
                                                            step="1"
                                                            value="<?php echo e((string)($settings['start_time_seconds'] ?? $defaults['start_time_seconds'])); ?>"
                                                        >
                                                    </div>

                                                    <div class="form-group">
                                                        <label for="mg-<?php echo e($s); ?>-bonus">Time bonus on make (s)</label>
                                                        <input
                                                            type="number"
                                                            class="form-control"
                                                            id="mg-<?php echo e($s); ?>-bonus"
                                                            name="settings[<?php echo e($s); ?>][time_bonus_on_make]"
                                                            min="0"
                                                            max="60"
                                                            step="1"
                                                            value="<?php echo e((string)($settings['time_bonus_on_make'] ?? $defaults['time_bonus_on_make'])); ?>"
                                                        >
                                                    </div>

                                                    <div class="form-group">
                                                        <label for="mg-<?php echo e($s); ?>-penalty">Time penalty on miss (s)</label>
                                                        <input
                                                            type="number"
                                                            class="form-control"
                                                            id="mg-<?php echo e($s); ?>-penalty"
                                                            name="settings[<?php echo e($s); ?>][time_penalty_on_miss]"
                                                            min="0"
                                                            max="120"
                                                            step="1"
                                                            value="<?php echo e((string)($settings['time_penalty_on_miss'] ?? $defaults['time_penalty_on_miss'])); ?>"
                                                        >
                                                    </div>

                                                    <div class="form-group">
                                                        <label for="mg-<?php echo e($s); ?>-max-time">Max time cap (s)</label>
                                                        <input
                                                            type="number"
                                                            class="form-control"
                                                            id="mg-<?php echo e($s); ?>-max-time"
                                                            name="settings[<?php echo e($s); ?>][max_time_seconds]"
                                                            min="10"
                                                            max="360"
                                                            step="1"
                                                            value="<?php echo e((string)($settings['max_time_seconds'] ?? $defaults['max_time_seconds'])); ?>"
                                                        >
                                                    </div>
                                                </div>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <div class="form-actions">
            <button class="btn btn-primary" type="submit">Save changes</button>
            <a class="btn btn-outline" href="/games" target="_blank" rel="noopener">View Mini-Games</a>
        </div>
    </form>
</div>

<?php AdminLayout::renderFooter(); ?>
