<?php
/**
 * Admin Panel - User Messages
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Auth\AdminAuth;
use NewSite\Logging\LogService;
use NewSite\Admin\AdminLayout;
use NewSite\Database\DatabaseManager;

AdminAuth::requireLogin();

$db = DatabaseManager::getWriteConnection();
$message = '';
$error = '';
$adminUserId = (int)($_SESSION['admin_user_id'] ?? 0);
$adminUsername = 'Admin';
if ($adminUserId > 0) {
    $stmt = $db->prepare("SELECT username FROM users WHERE id = ? LIMIT 1");
    $stmt->execute([$adminUserId]);
    $adminUsername = (string)$stmt->fetchColumn() ?: 'Admin';
}

$flash = $_SESSION['admin_flash_messages'] ?? null;
if (is_array($flash)) {
    $message = $flash['message'] ?? '';
    $error = $flash['error'] ?? '';
    unset($_SESSION['admin_flash_messages']);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verifyCSRF();
    $action = $_POST['action'] ?? '';
    try {
        if ($action === 'broadcast') {
            $subject = trim($_POST['subject'] ?? '');
            $body = trim($_POST['body'] ?? '');
            if ($subject === '' && $body === '') {
                $error = 'Subject or message body is required.';
            } else {
                $users = $db->query("SELECT id FROM site_users")->fetchAll();
                $stmt = $db->prepare("INSERT INTO user_messages (user_id, subject, body, sender_type, sender_label, message_type, is_read, created_at) VALUES (?, ?, ?, 'admin', ?, 'notice', 0, ?)");
                foreach ($users as $user) {
                    $stmt->execute([
                        (int)$user['id'],
                        $subject,
                        $body,
                        $adminUsername,
                        time()
                    ]);
                    // Push notification to user
                    \NewSite\Push\PushNotification::newMessage((int)$user['id'], $subject, $body);
                }
                $message = 'Message sent to all users.';
                LogService::add('info', 'Broadcast message sent', json_encode(['count' => count($users)]));
            }
        } elseif ($action === 'add_welcome') {
            $subject = trim($_POST['welcome_subject'] ?? '');
            $body = trim($_POST['welcome_body'] ?? '');
            if ($subject === '' && $body === '') {
                $error = 'Subject or message body is required.';
            } else {
                $stmt = $db->prepare("INSERT INTO welcome_messages (subject, body, is_active, created_at) VALUES (?, ?, 1, ?)");
                $stmt->execute([$subject, $body, time()]);
                $message = 'Welcome message added.';
                LogService::add('info', 'Welcome message added');
            }
        } elseif ($action === 'delete_welcome') {
            $id = (int)($_POST['welcome_id'] ?? 0);
            if ($id > 0) {
                $stmt = $db->prepare("DELETE FROM welcome_messages WHERE id = ?");
                $stmt->execute([$id]);
                $message = 'Welcome message removed.';
                LogService::add('info', 'Welcome message removed', json_encode(['id' => $id]));
            }
        }
    } catch (PDOException $e) {
        $error = 'Failed to save changes.';
        LogService::add('error', 'User messages admin error: ' . $e->getMessage());
    }

    if ($error === '') {
        $_SESSION['admin_flash_messages'] = ['message' => $message, 'error' => $error];
        header('Location: /admin/messages.php');
        exit;
    }
}

$welcomeMessages = $db->query("SELECT * FROM welcome_messages ORDER BY created_at DESC")->fetchAll();

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header">
        <h1>User Messages</h1>
        <p>Send broadcasts and configure welcome messages.</p>
    </div>

    <?php if ($message): ?>
        <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <div class="card">
        <div class="card-header">
            <h3>Broadcast Message</h3>
        </div>
        <div class="card-body">
            <form method="POST" action="">
                <input type="hidden" name="action" value="broadcast">
                <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                <div class="form-group">
                    <label for="subject">Subject</label>
                    <input type="text" id="subject" name="subject" class="form-control" placeholder="System update">
                </div>
                <div class="form-group">
                    <label for="body">Message</label>
                    <textarea id="body" name="body" class="form-control" rows="4" placeholder="Write your message..."></textarea>
                </div>
                <button type="submit" class="btn btn-primary has-tooltip" data-tooltip="Send this message to all site users.">Send to All Users</button>
            </form>
        </div>
    </div>

    <div class="card">
        <div class="card-header">
            <h3>Welcome Messages (First Login)</h3>
        </div>
        <div class="card-body">
            <form method="POST" action="">
                <input type="hidden" name="action" value="add_welcome">
                <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                <div class="form-group">
                    <label for="welcome_subject">Subject</label>
                    <input type="text" id="welcome_subject" name="welcome_subject" class="form-control" placeholder="Welcome">
                </div>
                <div class="form-group">
                    <label for="welcome_body">Message</label>
                    <textarea id="welcome_body" name="welcome_body" class="form-control" rows="3" placeholder="Thanks for joining!"></textarea>
                </div>
                <button type="submit" class="btn btn-secondary has-tooltip" data-tooltip="Add a welcome message shown on first login.">Add Welcome Message</button>
            </form>

            <?php if (empty($welcomeMessages)): ?>
                <p class="text-muted mt-16">No welcome messages configured.</p>
            <?php else: ?>
                <div class="table-responsive mt-16">
                    <table class="admin-table">
                        <thead>
                            <tr>
                                <th>Subject</th>
                                <th>Message</th>
                                <th>Created</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($welcomeMessages as $row): ?>
                                <tr>
                                    <td><?php echo e($row['subject'] ?? ''); ?></td>
                                    <td><?php echo e($row['body'] ?? ''); ?></td>
                                    <td><?php echo !empty($row['created_at']) ? date('Y-m-d H:i', (int)$row['created_at']) : '-'; ?></td>
                                    <td>
                                        <form method="POST" action="" data-confirm="Delete this welcome message?">
                                            <input type="hidden" name="action" value="delete_welcome">
                                            <input type="hidden" name="welcome_id" value="<?php echo (int)$row['id']; ?>">
                                            <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                                            <button type="submit" class="btn btn-danger btn-sm has-tooltip" data-tooltip="Remove this welcome message.">Delete</button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php AdminLayout::renderFooter(); ?>
