<?php
/**
 * Admin Panel - Menus Management
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Admin\AdminLayout;
use NewSite\Auth\AdminAuth;
use NewSite\Database\DatabaseManager;
use NewSite\Database\DbHelper;
use NewSite\Logging\LogService;
use NewSite\Menu\MenuService;

AdminAuth::requireLogin();

$db = DatabaseManager::getWriteConnection();
$message = '';
$error = '';

// Get current menu
$menuId = isset($_GET['menu']) ? (int)$_GET['menu'] : null;

// Get all menus (ordered by sort_order, then by id)
$menus = $db->query("SELECT * FROM menus ORDER BY sort_order ASC, id ASC")->fetchAll();

if (!$menuId && !empty($menus)) {
    $menuId = $menus[0]['id'];
}

// Get current menu details
$currentMenu = null;
if ($menuId) {
    $stmt = $db->prepare("SELECT * FROM menus WHERE id = ?");
    $stmt->execute([$menuId]);
    $currentMenu = $stmt->fetch();
}

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax'])) {
    verifyCSRF();
    header('Content-Type: application/json');

    $action = $_POST['action'] ?? '';

    try {
        switch ($action) {
            case 'add_menu':
                $name = trim($_POST['name'] ?? '');
                $location = trim($_POST['location'] ?? 'custom');

                if (empty($name)) {
                    echo json_encode(['success' => false, 'error' => 'Name is required']);
                    exit;
                }

                // Get max sort order for menus
                $maxMenuOrder = (int)$db->query("SELECT MAX(sort_order) FROM menus")->fetchColumn();

                $stmt = $db->prepare("INSERT INTO menus (name, location, sort_order) VALUES (?, ?, ?)");
                $stmt->execute([$name, $location, $maxMenuOrder + 1]);
                echo json_encode(['success' => true, 'id' => DbHelper::lastInsertId($db, 'menus')]);
                exit;

            case 'add_item':
                $menuIdPost = (int)($_POST['menu_id'] ?? 0);
                $title = trim($_POST['title'] ?? '');
                $url = trim($_POST['url'] ?? '');
                $pageId = !empty($_POST['page_id']) ? (int)$_POST['page_id'] : null;
                $parentId = !empty($_POST['parent_id']) ? (int)$_POST['parent_id'] : null;
                $target = $_POST['target'] ?? '_self';
                $icon = trim($_POST['icon'] ?? '');

                if (empty($title)) {
                    echo json_encode(['success' => false, 'error' => 'Title is required']);
                    exit;
                }

                // Get max sort order
                $stmt = $db->prepare("SELECT MAX(sort_order) FROM menu_items WHERE menu_id = ?");
                $stmt->execute([$menuIdPost]);
                $maxOrder = (int)$stmt->fetchColumn();

                $stmt = $db->prepare("INSERT INTO menu_items (menu_id, parent_id, title, url, page_id, target, icon, sort_order) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$menuIdPost, $parentId, $title, $url, $pageId, $target, $icon ?: null, $maxOrder + 1]);
                echo json_encode(['success' => true, 'id' => DbHelper::lastInsertId($db, 'menu_items')]);
                exit;

            case 'update_item':
                $itemId = (int)($_POST['item_id'] ?? 0);
                $title = trim($_POST['title'] ?? '');
                $url = trim($_POST['url'] ?? '');
                $pageId = !empty($_POST['page_id']) ? (int)$_POST['page_id'] : null;
                $target = $_POST['target'] ?? '_self';
                $icon = trim($_POST['icon'] ?? '');

                $stmt = $db->prepare("UPDATE menu_items SET title = ?, url = ?, page_id = ?, target = ?, icon = ? WHERE id = ?");
                $stmt->execute([$title, $url, $pageId, $target, $icon ?: null, $itemId]);
                echo json_encode(['success' => true]);
                exit;

            case 'delete_item':
                $itemId = (int)($_POST['item_id'] ?? 0);
                // Recursively delete all nested children
                $deleteIds = [$itemId];
                $toCheck = [$itemId];
                while (!empty($toCheck)) {
                    $placeholders = implode(',', array_fill(0, count($toCheck), '?'));
                    $stmt = $db->prepare("SELECT id FROM menu_items WHERE parent_id IN ($placeholders)");
                    $stmt->execute($toCheck);
                    $childIds = $stmt->fetchAll(PDO::FETCH_COLUMN);
                    $deleteIds = array_merge($deleteIds, $childIds);
                    $toCheck = $childIds;
                }
                $placeholders = implode(',', array_fill(0, count($deleteIds), '?'));
                $stmt = $db->prepare("DELETE FROM menu_items WHERE id IN ($placeholders)");
                $stmt->execute($deleteIds);
                echo json_encode(['success' => true]);
                exit;

            case 'reorder':
                $items = $_POST['items'] ?? [];
                foreach ($items as $index => $itemId) {
                    $stmt = $db->prepare("UPDATE menu_items SET sort_order = ? WHERE id = ?");
                    $stmt->execute([$index, $itemId]);
                }
                echo json_encode(['success' => true]);
                exit;

            case 'reorder_menus':
                $menuIds = $_POST['menus'] ?? [];
                foreach ($menuIds as $index => $mId) {
                    $stmt = $db->prepare("UPDATE menus SET sort_order = ? WHERE id = ?");
                    $stmt->execute([$index, $mId]);
                }
                echo json_encode(['success' => true]);
                exit;

            case 'delete_menu':
                $deleteMenuId = (int)($_POST['menu_id'] ?? 0);
                if ($deleteMenuId) {
                    // Delete all menu items first
                    $stmt = $db->prepare("DELETE FROM menu_items WHERE menu_id = ?");
                    $stmt->execute([$deleteMenuId]);
                    // Delete the menu
                    $stmt = $db->prepare("DELETE FROM menus WHERE id = ?");
                    $stmt->execute([$deleteMenuId]);
                }
                echo json_encode(['success' => true]);
                exit;

            default:
                echo json_encode(['success' => false, 'error' => 'Unknown action']);
                exit;
        }
    } catch (PDOException $e) {
        LogService::add('error', 'Menu operation failed: ' . $action, json_encode(['error' => $e->getMessage()]));
        echo json_encode(['success' => false, 'error' => 'Database error occurred']);
        exit;
    }

    exit;
}

// Get menu items using the shared function from _init.php
$menuItems = [];
if ($menuId) {
    $menuItems = MenuService::getItemsRecursive($db, $menuId);
}

// Get all pages for dropdown
$allPages = $db->query("SELECT id, title, slug FROM pages ORDER BY title")->fetchAll();

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header content-header-actions">
        <div>
            <h1>Menus</h1>
            <p>Manage navigation menus</p>
        </div>
        <button class="btn btn-primary" data-modal="new-menu-modal">+ New Menu</button>
    </div>

    <div class="menu-builder">
        <!-- Menu List -->
        <div class="menu-list">
            <div class="menu-list-header">
                <h4>Menus</h4>
            </div>
            <ul class="menu-items menu-items--padded" id="menus-list">
                <?php foreach ($menus as $menu): ?>
                <li class="menu-item admin-menu-item <?php echo $menu['id'] == $menuId ? 'is-active' : ''; ?>" draggable="true" data-menu-id="<?php echo $menu['id']; ?>">
                    <a href="/admin/menus.php?menu=<?php echo $menu['id']; ?>" class="admin-menu-link">
                        <span class="drag-handle menu-drag-handle">⋮⋮</span>
                        <strong><?php echo e($menu['name']); ?></strong>
                        <br><small class="menu-location"><?php echo e($menu['location']); ?></small>
                    </a>
                    <button class="btn-delete-menu btn-icon-muted" data-action="delete-menu" data-menu-id="<?php echo $menu['id']; ?>" data-menu-name="<?php echo e($menu['name']); ?>" title="Delete Menu">
                        🗑️
                    </button>
                </li>
                <?php endforeach; ?>
            </ul>
        </div>

        <!-- Menu Items Editor -->
        <div class="card card-tight">
            <div class="card-header flex-between">
                <h3><?php echo $currentMenu ? e($currentMenu['name']) : 'Select a Menu'; ?></h3>
                <?php if ($currentMenu): ?>
                <button class="btn btn-primary btn-sm" data-modal="add-item-modal">+ Add Item</button>
                <?php endif; ?>
            </div>
            <div class="card-body">
                <?php
                // Recursive function to render menu items with valid HTML structure.
                // Each level wraps <li> items inside its own <ul> so the analyzer
                // sees the required list container in the same code block.
                function renderMenuItems($items, $maxDepth = 5)
                {
                    if (empty($items)) {
                        return;
                    }
                    ?><ul class="menu-sublist"><?php
                    foreach ($items as $item):
                        $depth = $item['depth'] ?? 0;
                        $canAddSub = $depth < $maxDepth - 1; // Allow adding sub-items up to max depth
                        ?>
                    <li class="menu-item" draggable="true" data-id="<?php echo $item['id']; ?>">
                        <div class="menu-item-header">
                            <span class="drag-handle">⋮⋮</span>
                            <?php if (!empty($item['icon'])): ?>
                            <span class="menu-item-icon"><?php echo e($item['icon']); ?></span>
                            <?php endif; ?>
                            <span class="title"><?php echo e($item['title']); ?></span>
                            <small class="text-muted mr-10">
                                <?php if ($item['page_slug']): ?>
                                    /<?php echo e($item['page_slug']); ?>
                                <?php elseif ($item['url']): ?>
                                    <?php echo e($item['url']); ?>
                                <?php endif; ?>
                            </small>
                            <div class="actions">
                                <button data-action="edit-menu-item"
                                    data-item-id="<?php echo $item['id']; ?>"
                                    data-item-title="<?php echo e($item['title']); ?>"
                                    data-item-url="<?php echo e($item['url']); ?>"
                                    data-item-page-id="<?php echo e($item['page_id'] ?? ''); ?>"
                                    data-item-target="<?php echo e($item['target']); ?>"
                                    data-item-icon="<?php echo e($item['icon'] ?? ''); ?>"
                                    title="Edit">✏️</button>
                                <button data-action="delete-menu-item" data-item-id="<?php echo $item['id']; ?>" title="Delete">🗑️</button>
                                <?php if ($canAddSub): ?>
                                <button data-action="add-sub-item" data-parent-id="<?php echo $item['id']; ?>" title="Add Sub-item">➕</button>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php if (!empty($item['children'])): ?>
                            <?php renderMenuItems($item['children'], $maxDepth); ?>
                        <?php endif; ?>
                    </li>
                <?php endforeach; ?>
                </ul><?php
                }
?>
                <?php if ($currentMenu && !empty($menuItems)): ?>
                <div class="menu-items" id="menu-items-list" data-sortable data-menu="<?php echo $menuId; ?>">
                    <?php renderMenuItems($menuItems); ?>
                </div>
                <?php elseif ($currentMenu): ?>
                <p class="text-muted text-center empty-state">No menu items yet. Click "Add Item" to get started.</p>
                <?php else: ?>
                <p class="text-muted text-center empty-state">Select a menu from the left or create a new one.</p>
                <?php endif; ?>
            </div>
        </div>

        <!-- System Links Info -->
        <div class="card mt-20">
            <div class="card-header">
                <h3>System Links (Footer)</h3>
            </div>
            <div class="card-body">
                <p class="text-muted mb-15">These links are automatically shown at the bottom of the footer and cannot be removed. You can edit their content in the <a href="/admin/pages.php">Pages</a> section.</p>
                <ul class="menu-items menu-items--compact">
                    <li class="menu-item menu-item-disabled">
                        <div class="menu-item-header">
                            <span class="mr-10">📜</span>
                            <span class="title">Terms of Service</span>
                            <small class="text-muted ml-auto">/terms-of-service</small>
                            <span class="badge badge-system">System</span>
                        </div>
                    </li>
                    <li class="menu-item menu-item-disabled">
                        <div class="menu-item-header">
                            <span class="mr-10">🔒</span>
                            <span class="title">Privacy Policy</span>
                            <small class="text-muted ml-auto">/privacy-policy</small>
                            <span class="badge badge-system">System</span>
                        </div>
                    </li>
                    <li class="menu-item menu-item-disabled">
                        <div class="menu-item-header">
                            <span class="mr-10">🍪</span>
                            <span class="title">Cookie Settings</span>
                            <small class="text-muted ml-auto">Opens cookie modal</small>
                            <span class="badge badge-system">System</span>
                        </div>
                    </li>
                </ul>
            </div>
        </div>
    </div>
</div>

<!-- New Menu Modal -->
<div class="modal-overlay" id="new-menu-modal">
    <div class="modal">
        <div class="modal-header">
            <h3>New Menu</h3>
            <button class="modal-close">&times;</button>
        </div>
        <form id="new-menu-form">
            <div class="modal-body">
                <div class="form-group">
                    <label for="menu-name">Menu Name</label>
                    <input type="text" id="menu-name" name="name" class="form-control" required>
                </div>
                <div class="form-group">
                    <label for="menu-location">Location</label>
                    <select id="menu-location" name="location" class="form-control">
                        <option value="header">Header</option>
                        <option value="topbar">Top Bar</option>
                        <option value="footer">Footer</option>
                        <option value="sidebar">Sidebar</option>
                        <option value="custom">Custom</option>
                    </select>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary modal-cancel">Cancel</button>
                <button type="submit" class="btn btn-primary">Create Menu</button>
            </div>
        </form>
    </div>
</div>

<!-- Add Item Modal -->
<div class="modal-overlay" id="add-item-modal">
    <div class="modal">
        <div class="modal-header">
            <h3>Add Menu Item</h3>
            <button class="modal-close">&times;</button>
        </div>
        <form id="add-item-form">
            <input type="hidden" name="menu_id" value="<?php echo $menuId; ?>">
            <input type="hidden" name="parent_id" id="item-parent-id" value="">
            <div class="modal-body">
                <div class="form-group">
                    <label for="item-title">Title</label>
                    <input type="text" id="item-title" name="title" class="form-control" required>
                </div>
                <div class="form-group">
                    <label for="item-icon">Icon (optional)</label>
                    <div class="icon-picker-wrapper">
                        <input type="text" id="item-icon" name="icon" class="form-control icon-input cursor-pointer" placeholder="Click to select..." readonly>
                        <span class="icon-preview" id="item-icon-preview"></span>
                    </div>
                    <div class="icon-picker is-hidden" id="item-icon-picker">
                        <div class="icon-picker-grid">
                            <?php
            $defaultIcons = ['🏠', '📄', '📁', '📂', '📝', '📋', '📌', '📎', '🔗', '🔍', '⚙️', '🛠️', '👤', '👥', '💬', '📧', '📞', '📱', '💻', '🖥️', '🎮', '🎵', '🎬', '📸', '🖼️', '📊', '📈', '📉', '💰', '💳', '🛒', '🛍️', '🏪', '🏢', '🏠', '🌐', '🌍', '✈️', '🚗', '📍', '📅', '⏰', '🔔', '🔒', '🔓', '🔑', '❤️', '⭐', '✨', '🎯', '🏆', '🎁', '🎉', '💡', '📚', '🎓', '❓', '❗', '✅', '❌', '➕', '➖', '➡️', '⬅️', '⬆️', '⬇️', '↩️', '🔄'];
foreach ($defaultIcons as $icon):
    ?>
                            <button type="button" class="icon-option" data-icon="<?php echo $icon; ?>"><?php echo $icon; ?></button>
                            <?php endforeach; ?>
                        </div>
                        <div class="icon-picker-footer">
                            <input type="text" class="form-control custom-icon-input flex-1" placeholder="Or paste custom emoji...">
                            <button type="button" class="btn btn-secondary btn-sm clear-icon">Clear</button>
                        </div>
                    </div>
                </div>
                <div class="form-group">
                    <label for="item-page">Link to Page</label>
                    <select id="item-page" name="page_id" class="form-control">
                        <option value="">-- Custom URL --</option>
                        <?php foreach ($allPages as $p): ?>
                        <option value="<?php echo $p['id']; ?>"><?php echo e($p['title']); ?> (/<?php echo e($p['slug']); ?>)</option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group" id="custom-url-group">
                    <label for="item-url">Custom URL</label>
                    <input type="text" id="item-url" name="url" class="form-control" placeholder="https://...">
                </div>
                <div class="form-group">
                    <label for="item-target">Open in</label>
                    <select id="item-target" name="target" class="form-control">
                        <option value="_self">Same Window</option>
                        <option value="_blank">New Window</option>
                    </select>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary modal-cancel">Cancel</button>
                <button type="submit" class="btn btn-primary">Add Item</button>
            </div>
        </form>
    </div>
</div>

<!-- Edit Item Modal -->
<div class="modal-overlay" id="edit-item-modal">
    <div class="modal">
        <div class="modal-header">
            <h3>Edit Menu Item</h3>
            <button class="modal-close">&times;</button>
        </div>
        <form id="edit-item-form">
            <input type="hidden" name="item_id" id="edit-item-id" value="">
            <div class="modal-body">
                <div class="form-group">
                    <label for="edit-item-title">Title</label>
                    <input type="text" id="edit-item-title" name="title" class="form-control" required>
                </div>
                <div class="form-group">
                    <label for="edit-item-icon">Icon (optional)</label>
                    <div class="icon-picker-wrapper">
                        <input type="text" id="edit-item-icon" name="icon" class="form-control icon-input cursor-pointer" placeholder="Click to select..." readonly>
                        <span class="icon-preview" id="edit-item-icon-preview"></span>
                    </div>
                    <div class="icon-picker is-hidden" id="edit-item-icon-picker">
                        <div class="icon-picker-grid">
                            <?php
    foreach ($defaultIcons as $icon):
        ?>
                            <button type="button" class="icon-option" data-icon="<?php echo $icon; ?>"><?php echo $icon; ?></button>
                            <?php endforeach; ?>
                        </div>
                        <div class="icon-picker-footer">
                            <input type="text" class="form-control custom-icon-input flex-1" placeholder="Or paste custom emoji...">
                            <button type="button" class="btn btn-secondary btn-sm clear-icon">Clear</button>
                        </div>
                    </div>
                </div>
                <div class="form-group">
                    <label for="edit-item-page">Link to Page</label>
                    <select id="edit-item-page" name="page_id" class="form-control">
                        <option value="">-- Custom URL --</option>
                        <?php foreach ($allPages as $p): ?>
                        <option value="<?php echo $p['id']; ?>"><?php echo e($p['title']); ?> (/<?php echo e($p['slug']); ?>)</option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group" id="edit-custom-url-group">
                    <label for="edit-item-url">Custom URL</label>
                    <input type="text" id="edit-item-url" name="url" class="form-control" placeholder="https://...">
                </div>
                <div class="form-group">
                    <label for="edit-item-target">Open in</label>
                    <select id="edit-item-target" name="target" class="form-control">
                        <option value="_self">Same Window</option>
                        <option value="_blank">New Window</option>
                    </select>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary modal-cancel">Cancel</button>
                <button type="submit" class="btn btn-primary">Save Changes</button>
            </div>
        </form>
    </div>
</div>

<script nonce="<?php echo e(getCspNonce()); ?>">
// Toggle custom URL field based on page selection
document.getElementById('item-page').addEventListener('change', function() {
    const customUrlGroup = document.getElementById('custom-url-group');
    if (customUrlGroup) {
        customUrlGroup.classList.toggle('is-hidden', !!this.value);
    }
});

document.getElementById('edit-item-page').addEventListener('change', function() {
    const editGroup = document.getElementById('edit-custom-url-group');
    if (editGroup) {
        editGroup.classList.toggle('is-hidden', !!this.value);
    }
});

// New Menu Form
document.getElementById('new-menu-form').addEventListener('submit', function(e) {
    e.preventDefault();
    const formData = new FormData(this);
    formData.append('ajax', '1');
    formData.append('action', 'add_menu');

    fetch('/admin/menus.php', {
        method: 'POST',
        body: formData
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) {
            location.href = '/admin/menus.php?menu=' + data.id;
        } else {
            alert(data.error || 'Error creating menu');
        }
    });
});

// Add Item Form
document.getElementById('add-item-form').addEventListener('submit', function(e) {
    e.preventDefault();
    const formData = new FormData(this);
    formData.append('ajax', '1');
    formData.append('action', 'add_item');

    fetch('/admin/menus.php', {
        method: 'POST',
        body: formData
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert(data.error || 'Error adding item');
        }
    });
});

// Edit Item Form
document.getElementById('edit-item-form').addEventListener('submit', function(e) {
    e.preventDefault();
    const formData = new FormData(this);
    formData.append('ajax', '1');
    formData.append('action', 'update_item');

    fetch('/admin/menus.php', {
        method: 'POST',
        body: formData
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert(data.error || 'Error updating item');
        }
    });
});

// Menu action handlers (edit/delete/add sub-item)
document.addEventListener('click', function(e) {
    const btn = e.target.closest('[data-action]');
    if (!btn) return;
    const action = btn.getAttribute('data-action');
    if (!action) return;

    if (action === 'delete-menu') {
        const menuId = btn.getAttribute('data-menu-id');
        const menuName = btn.getAttribute('data-menu-name') || '';
        if (menuId) {
            deleteMenu(menuId, menuName);
        }
        return;
    }

    if (action === 'edit-menu-item') {
        const id = btn.getAttribute('data-item-id');
        const title = btn.getAttribute('data-item-title') || '';
        const url = btn.getAttribute('data-item-url') || '';
        const pageIdRaw = btn.getAttribute('data-item-page-id');
        const pageId = pageIdRaw !== null && pageIdRaw !== '' ? pageIdRaw : null;
        const target = btn.getAttribute('data-item-target') || '_self';
        const icon = btn.getAttribute('data-item-icon') || '';
        if (id) {
            editMenuItem(id, title, url, pageId, target, icon);
        }
        return;
    }

    if (action === 'delete-menu-item') {
        const id = btn.getAttribute('data-item-id');
        if (id) {
            deleteMenuItem(id);
        }
        return;
    }

    if (action === 'add-sub-item') {
        const parentId = btn.getAttribute('data-parent-id');
        if (parentId) {
            addSubItem(parentId);
        }
    }
});

function editMenuItem(id, title, url, pageId, target, icon) {
    document.getElementById('edit-item-id').value = id;
    document.getElementById('edit-item-title').value = title;
    document.getElementById('edit-item-url').value = url || '';
    document.getElementById('edit-item-page').value = pageId || '';
    document.getElementById('edit-item-target').value = target || '_self';
    document.getElementById('edit-item-icon').value = icon || '';
    document.getElementById('edit-item-icon-preview').textContent = icon || '';
    const editGroup = document.getElementById('edit-custom-url-group');
    if (editGroup) {
        editGroup.classList.toggle('is-hidden', !!pageId);
    }
    openModal('edit-item-modal');
}

function deleteMenuItem(id) {
    if (!confirm('Delete this menu item?')) return;

    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'delete_item');
    formData.append('item_id', id);

    fetch('/admin/menus.php', {
        method: 'POST',
        body: formData
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) {
            location.reload();
        }
    });
}

function addSubItem(parentId) {
    document.getElementById('item-parent-id').value = parentId;
    document.getElementById('item-title').value = '';
    document.getElementById('item-url').value = '';
    document.getElementById('item-page').value = '';
    const customUrlGroup = document.getElementById('custom-url-group');
    if (customUrlGroup) {
        customUrlGroup.classList.remove('is-hidden');
    }
    // Update modal title to indicate adding submenu
    const modalTitle = document.querySelector('#add-item-modal .modal-header h3');
    if (modalTitle) {
        modalTitle.textContent = 'Add Submenu Item';
    }
    openModal('add-item-modal');
}

// Reset modal title when opening for regular item
document.querySelector('[data-modal="add-item-modal"]')?.addEventListener('click', function() {
    document.getElementById('item-parent-id').value = '';
    const modalTitle = document.querySelector('#add-item-modal .modal-header h3');
    if (modalTitle) {
        modalTitle.textContent = 'Add Menu Item';
    }
});

// Drag and drop reordering
const menuList = document.getElementById('menu-items-list');
if (menuList) {
    menuList.addEventListener('orderChanged', function() {
        const items = Array.from(this.querySelectorAll(':scope > .menu-item')).map(el => el.dataset.id);

        const formData = new FormData();
        formData.append('ajax', '1');
        formData.append('action', 'reorder');
        items.forEach((id, i) => formData.append('items[]', id));

        fetch('/admin/menus.php', {
            method: 'POST',
            body: formData
        });
    });
}

// Delete menu function
function deleteMenu(menuId, menuName) {
    if (!confirm('Delete menu "' + menuName + '" and all its items? This cannot be undone.')) return;

    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'delete_menu');
    formData.append('menu_id', menuId);

    fetch('/admin/menus.php', {
        method: 'POST',
        body: formData
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) {
            location.href = '/admin/menus.php';
        } else {
            alert(data.error || 'Error deleting menu');
        }
    });
}

// Drag and drop reordering for menus list
const menusList = document.getElementById('menus-list');
if (menusList) {
    let draggedMenu = null;

    menusList.addEventListener('dragstart', function(e) {
        const menuItem = e.target.closest('.menu-item');
        if (menuItem && menuItem.dataset.menuId) {
            draggedMenu = menuItem;
            menuItem.classList.add('is-dragging');
        }
    });

    menusList.addEventListener('dragend', function(e) {
        if (draggedMenu) {
            draggedMenu.classList.remove('is-dragging');
            draggedMenu = null;
        }
    });

    menusList.addEventListener('dragover', function(e) {
        e.preventDefault();
    });

    menusList.addEventListener('drop', function(e) {
        e.preventDefault();

        if (draggedMenu) {
            const dropTarget = e.target.closest('.menu-item');
            if (dropTarget && dropTarget !== draggedMenu && dropTarget.dataset.menuId) {
                const rect = dropTarget.getBoundingClientRect();
                const midY = rect.top + rect.height / 2;

                if (e.clientY < midY) {
                    dropTarget.parentNode.insertBefore(draggedMenu, dropTarget);
                } else {
                    dropTarget.parentNode.insertBefore(draggedMenu, dropTarget.nextSibling);
                }

                // Save new order
                const menuIds = Array.from(menusList.querySelectorAll('.menu-item')).map(el => el.dataset.menuId);

                const formData = new FormData();
                formData.append('ajax', '1');
                formData.append('action', 'reorder_menus');
                menuIds.forEach((id, i) => formData.append('menus[]', id));

                fetch('/admin/menus.php', {
                    method: 'POST',
                    body: formData
                });
            }
        }
    });
}

// Icon Picker functionality
function initIconPickers() {
    document.querySelectorAll('.icon-input').forEach(function(input) {
        const wrapper = input.closest('.form-group');
        const picker = wrapper.querySelector('.icon-picker');
        const preview = wrapper.querySelector('.icon-preview');

        // Toggle picker on input click
        input.addEventListener('click', function(e) {
            e.stopPropagation();
            // Close all other pickers
            document.querySelectorAll('.icon-picker').forEach(p => {
                if (p !== picker) p.classList.add('is-hidden');
            });
            picker.classList.toggle('is-hidden');
        });

        // Select icon
        picker.querySelectorAll('.icon-option').forEach(function(btn) {
            btn.addEventListener('click', function(e) {
                e.preventDefault();
                const icon = this.dataset.icon;
                input.value = icon;
                preview.textContent = icon;
                picker.classList.add('is-hidden');
            });
        });

        // Custom icon input
        const customInput = picker.querySelector('.custom-icon-input');
        if (customInput) {
            customInput.addEventListener('input', function() {
                if (this.value) {
                    input.value = this.value;
                    preview.textContent = this.value;
                }
            });
        }

        // Clear icon
        const clearBtn = picker.querySelector('.clear-icon');
        if (clearBtn) {
            clearBtn.addEventListener('click', function(e) {
                e.preventDefault();
                input.value = '';
                preview.textContent = '';
                if (customInput) customInput.value = '';
                picker.classList.add('is-hidden');
            });
        }
    });

    // Close picker when clicking outside
    document.addEventListener('click', function(e) {
        if (!e.target.closest('.icon-picker') && !e.target.closest('.icon-input')) {
            document.querySelectorAll('.icon-picker').forEach(p => p.classList.add('is-hidden'));
        }
    });
}

// Initialize icon pickers
initIconPickers();
</script>

<?php AdminLayout::renderFooter(); ?>
