<?php
/**
 * Admin Panel - Website Logs
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Admin\AdminLayout;
use NewSite\Auth\AdminAuth;
use NewSite\Database\DatabaseManager;
use NewSite\Logging\LogService;

AdminAuth::requireLogin();

$db = DatabaseManager::getWriteConnection();

const LOGS_REDIRECT_PREFIX = 'Location: /admin/logs.php?message=';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_GET['action'])) {
    verifyCSRF();
    switch ($_GET['action']) {
        case 'test_logs':
            // Add test warning and error logs
            LogService::add('warning', 'Test Warning Log', json_encode(['type' => 'test', 'timestamp' => time(), 'message' => 'This is a test warning log entry']));
            LogService::add('error', 'Test Error Log', json_encode(['type' => 'test', 'timestamp' => time(), 'message' => 'This is a test error log entry', 'stack' => 'Test stack trace']));
            LogService::add('info', 'Test Info Log', json_encode(['type' => 'test', 'note' => 'This info log should NOT appear in the filtered view']));
            header(LOGS_REDIRECT_PREFIX . urlencode('Test logs created: 1 warning, 1 error, 1 info (info should be hidden)'));
            exit;

        case 'clear':
            $days = (int)($_POST['days'] ?? 30);
            $deleted = LogService::clearOld((int)$days);
            LogService::add('system', "Cleared logs older than $days days", "Deleted $deleted entries");
            header(LOGS_REDIRECT_PREFIX . urlencode("Cleared $deleted old log entries"));
            exit;

        case 'clear_all':
            $db->exec("DELETE FROM logs");
            header(LOGS_REDIRECT_PREFIX . urlencode('All logs cleared'));
            exit;

        default:
            break;
    }
}

if (isset($_GET['action']) && $_GET['action'] === 'export') {
    $logs = LogService::getAll(null, 10000);
    header('Content-Type: application/json');
    header('Content-Disposition: attachment; filename="logs-' . date('Y-m-d-His') . '.json"');
    echo json_encode($logs, JSON_PRETTY_PRINT);
    exit;
}

// Get filter parameters
$filterType = $_GET['type'] ?? '';
$page = max(1, (int)($_GET['page'] ?? 1));
$perPage = 50;
$offset = ($page - 1) * $perPage;

// Show all logs (filter by type if selected)
$whereSql = "WHERE 1=1";
$params = [];
if ($filterType) {
    $whereSql .= " AND log_type = ?";
    $params[] = $filterType;
}

// Get logs
$logsStmt = $db->prepare("SELECT * FROM logs $whereSql ORDER BY created_at DESC LIMIT ? OFFSET ?");
foreach ($params as $i => $value) {
    $logsStmt->bindValue($i + 1, $value);
}
$logsStmt->bindValue(count($params) + 1, $perPage, PDO::PARAM_INT);
$logsStmt->bindValue(count($params) + 2, $offset, PDO::PARAM_INT);
$logsStmt->execute();
$logs = $logsStmt->fetchAll(PDO::FETCH_ASSOC);

$countStmt = $db->prepare("SELECT COUNT(*) FROM logs $whereSql");
foreach ($params as $i => $value) {
    $countStmt->bindValue($i + 1, $value);
}
$countStmt->execute();
$totalLogs = (int)$countStmt->fetchColumn();
$totalPages = ceil($totalLogs / $perPage);

// Get unique log types for filter
$typesStmt = $db->prepare("SELECT DISTINCT log_type FROM logs $whereSql ORDER BY log_type");
foreach ($params as $i => $value) {
    $typesStmt->bindValue($i + 1, $value);
}
$typesStmt->execute();
$logTypes = $typesStmt->fetchAll(PDO::FETCH_COLUMN);

// Check for message
$message = $_GET['message'] ?? '';

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header content-header-actions">
        <div>
            <h1>Website Logs</h1>
            <p>View system logs, errors, warnings, and security events</p>
        </div>
        <div class="d-flex gap-1">
            <form method="POST" action="/admin/logs.php?action=test_logs" class="inline-block">
                <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                <button type="submit" class="btn btn-primary has-tooltip" data-tooltip="Create test warning and error logs to verify logging is working.">Test Logging</button>
            </form>
            <a href="/admin/logs.php?action=export" class="btn btn-secondary has-tooltip" data-tooltip="Download logs as JSON.">Export JSON</a>
            <form method="POST" action="/admin/logs.php?action=clear" class="inline-block" data-confirm="Clear logs older than 30 days?">
                <input type="hidden" name="days" value="30">
                <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                <button type="submit" class="btn btn-secondary has-tooltip" data-tooltip="Remove logs older than 30 days.">Clear Old</button>
            </form>
            <form method="POST" action="/admin/logs.php?action=clear_all" class="inline-block" data-confirm="Delete ALL logs? This cannot be undone.">
                <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                <button type="submit" class="btn btn-danger has-tooltip" data-tooltip="Delete all logs permanently.">Clear All</button>
            </form>
        </div>
    </div>

    <?php if ($message): ?>
    <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>

    <!-- Filter -->
    <div class="card logs-card">
        <div class="card-body logs-card-body">
            <form method="GET" action="" class="d-flex gap-1 align-center">
                <label class="mr-10" for="log-type-filter">Filter by type:</label>
                <select name="type" id="log-type-filter" class="form-control width-200">
                    <option value="">All Types</option>
                    <?php foreach ($logTypes as $type): ?>
                    <option value="<?php echo e($type); ?>" <?php echo $filterType === $type ? 'selected' : ''; ?>>
                        <?php echo e($type); ?>
                    </option>
                    <?php endforeach; ?>
                </select>
                <button type="submit" class="btn btn-secondary btn-sm has-tooltip" data-tooltip="Apply the selected log type filter.">Filter</button>
                <?php if ($filterType): ?>
                <a href="/admin/logs.php" class="btn btn-secondary btn-sm has-tooltip" data-tooltip="Clear the log type filter.">Clear Filter</a>
                <?php endif; ?>
            </form>
        </div>
    </div>

    <!-- Stats -->
    <div class="stats-grid stats-grid-spaced">
        <div class="stat-card">
            <div class="stat-icon">📊</div>
            <div class="stat-info">
                <h3><?php echo number_format($totalLogs); ?></h3>
                <p>Total Logs</p>
            </div>
        </div>
        <?php
        $todayStmt = $db->prepare("SELECT COUNT(*) FROM logs $whereSql AND created_at::date = CURRENT_DATE");
foreach ($params as $i => $value) {
    $todayStmt->bindValue($i + 1, $value);
}
$todayStmt->execute();
$todayLogs = $todayStmt->fetchColumn();
$errorLogs = $db->query("SELECT COUNT(*) FROM logs WHERE log_type ILIKE '%error%' OR log_type ILIKE '%warning%'")->fetchColumn();
?>
        <div class="stat-card">
            <div class="stat-icon">📅</div>
            <div class="stat-info">
                <h3><?php echo number_format($todayLogs); ?></h3>
                <p>Today</p>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon">⚠️</div>
            <div class="stat-info">
                <h3><?php echo number_format($errorLogs); ?></h3>
                <p>Errors</p>
            </div>
        </div>
    </div>

    <!-- Logs Table -->
    <div class="table-container">
        <table class="data-table">
            <thead>
                <tr>
                    <th class="log-th-time">Time</th>
                    <th class="log-th-type">Type</th>
                    <th>Message</th>
                    <th class="log-th-ip">IP</th>
                    <th class="log-th-actions">Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($logs)): ?>
                <tr>
                    <td colspan="5" class="text-center text-muted log-empty">No logs found.</td>
                </tr>
                <?php else: ?>
                <?php foreach ($logs as $log): ?>
                <tr>
                    <td>
                        <span class="log-subtext">
                            <?php echo date('M j, H:i:s', strtotime($log['created_at'])); ?>
                        </span>
                    </td>
                    <td>
                        <span class="log-type log-type-<?php echo e(strtolower($log['log_type'])); ?>">
                            <?php echo e($log['log_type']); ?>
                        </span>
                    </td>
                    <td>
                        <?php echo e($log['message']); ?>
                        <?php if ($log['details']): ?>
                        <button class="btn btn-sm has-tooltip log-details-btn" data-tooltip="Show full log details." data-log-details-toggle="<?php echo $log['id']; ?>">Details</button>
                        <div id="details-<?php echo $log['id']; ?>" class="log-details is-hidden">
<?php
$details = $log['details'];
                            $decoded = json_decode($details, true);
                            if ($decoded !== null) {
                                echo e(json_encode($decoded, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
                            } else {
                                echo e($details);
                            }
                    ?>
                        </div>
                        <?php endif; ?>
                    </td>
                    <td>
                        <span class="log-ip">
                            <?php echo e($log['ip_address'] ?? '-'); ?>
                        </span>
                    </td>
                    <td>
                        <a href="/admin/logs.php?type=<?php echo urlencode($log['log_type']); ?>" class="btn btn-sm btn-secondary" title="Filter by type">🔍</a>
                    </td>
                </tr>
                <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <!-- Pagination -->
    <?php if ($totalPages > 1): ?>
    <div class="pagination logs-pagination">
        <?php if ($page > 1): ?>
        <a href="/admin/logs.php?page=<?php echo $page - 1; ?><?php echo $filterType ? '&type=' . urlencode($filterType) : ''; ?>" class="btn btn-secondary btn-sm">← Prev</a>
        <?php endif; ?>

        <span class="p-10 text-muted">
            Page <?php echo $page; ?> of <?php echo $totalPages; ?>
        </span>

        <?php if ($page < $totalPages): ?>
        <a href="/admin/logs.php?page=<?php echo $page + 1; ?><?php echo $filterType ? '&type=' . urlencode($filterType) : ''; ?>" class="btn btn-secondary btn-sm">Next →</a>
        <?php endif; ?>
    </div>
    <?php endif; ?>
</div>

<script nonce="<?php echo e(getCspNonce()); ?>">
function toggleDetails(id) {
    const el = document.getElementById('details-' + id);
    if (el) {
        el.classList.toggle('is-hidden');
    }
}

document.querySelectorAll('[data-log-details-toggle]').forEach(function(btn) {
    btn.addEventListener('click', function() {
        const id = this.getAttribute('data-log-details-toggle');
        toggleDetails(id);
    });
});
</script>

<?php AdminLayout::renderFooter(); ?>
