<?php
/**
 * Admin Panel - Info & Documentation
 *
 * Displays useful information and documentation for administrators.
 */

if (!defined('ROOT_PATH')) {
    http_response_code(403);
    exit;
}

use NewSite\Auth\AdminAuth;
use NewSite\Settings\SettingsService;
use NewSite\Email\SmtpMailer;
use NewSite\Admin\AdminLayout;
use NewSite\Database\DatabaseManager;
use NewSite\Database\SchemaManager;

AdminAuth::requireLogin();

$db = DatabaseManager::getWriteConnection();

// Get current schema version from database
$currentSchemaVersion = 0;
try {
    $stmt = $db->query("SELECT setting_value FROM settings WHERE setting_key = 'db_schema_version'");
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($row) {
        $currentSchemaVersion = (int)$row['setting_value'];
    }
} catch (Exception $e) {
    // Table might not exist yet
}

// Read schema target directly from the PSR-4 SchemaManager constant
$schemaTarget = SchemaManager::SCHEMA_TARGET;
$schemaTargetInt = $schemaTarget;

// Get PHP version
$phpVersion = phpversion();

// Get PostgreSQL version
$pgVersion = 'Unknown';
try {
    $stmt = $db->query("SELECT version()");
    $row = $stmt->fetch(PDO::FETCH_NUM);
    if ($row) {
        // Extract just the version number
        if (preg_match('/PostgreSQL\s+([\d.]+)/', $row[0], $matches)) {
            $pgVersion = $matches[1];
        } else {
            $pgVersion = substr($row[0], 0, 50) . '...';
        }
    }
} catch (Exception $e) {
    $pgVersion = 'Error: ' . $e->getMessage();
}

// Get disk space (if available)
// Format bytes to human readable
function formatBytes($bytes, $precision = 2)
{
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    return round($bytes / pow(1024, $pow), $precision) . ' ' . $units[$pow];
}

// Get upload folder size
$uploadDirs = [
    __DIR__ . '/../../data/admin_uploads',
    __DIR__ . '/../../data/chat_images',
    __DIR__ . '/../../data/contact_uploads',
    __DIR__ . '/../../data/forum_images',
    __DIR__ . '/../../data/profile_photos',
];
$uploadSize = 0;
$uploadCount = 0;
foreach ($uploadDirs as $uploadPath) {
    if (is_dir($uploadPath)) {
        $iterator = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($uploadPath));
        foreach ($iterator as $file) {
            if ($file->isFile()) {
                $uploadSize += $file->getSize();
                $uploadCount++;
            }
        }
    }
}

// Get total site disk usage (entire NewSite directory)
$siteRootDir = realpath(__DIR__ . '/../..');
$siteTotalSize = 0;
$siteTotalFiles = 0;
$siteScanError = false;
if (is_dir($siteRootDir)) {
    try {
        $siteIter = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($siteRootDir, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::LEAVES_ONLY,
            RecursiveIteratorIterator::CATCH_GET_CHILD
        );
        foreach ($siteIter as $siteFile) {
            if ($siteFile->isFile()) {
                $siteTotalSize += $siteFile->getSize();
                $siteTotalFiles++;
            }
        }
    } catch (Exception $e) {
        $siteScanError = true;
    }
}

// Helper to check if a table exists
function tableExists($db, $tableName)
{
    try {
        $stmt = $db->prepare("SELECT EXISTS (SELECT FROM information_schema.tables WHERE table_schema = 'public' AND table_name = ?)");
        $stmt->execute([$tableName]);
        return (bool)$stmt->fetchColumn();
    } catch (Exception $e) {
        return false;
    }
}

// Helper to check if a column exists
function columnExists($db, $tableName, $columnName)
{
    try {
        $stmt = $db->prepare("SELECT EXISTS (SELECT FROM information_schema.columns WHERE table_schema = 'public' AND table_name = ? AND column_name = ?)");
        $stmt->execute([$tableName, $columnName]);
        return (bool)$stmt->fetchColumn();
    } catch (Exception $e) {
        return false;
    }
}

// Get database table counts (only for tables that exist)
$dbStats = [];
$tableChecks = [
    'Admin Accounts' => 'users',
    'Site Users' => 'site_users',
    'Products' => 'products',
    'Collections' => 'collections',
    'Orders' => 'orders',
    'Mini-Games' => 'mini_games',
    'Forum Posts' => 'forum_posts',
    'Log Entries' => 'logs',
];
foreach ($tableChecks as $label => $tableName) {
    if (tableExists($db, $tableName)) {
        try {
            $stmt = $db->query("SELECT COUNT(*) FROM " . $tableName);
            $dbStats[$label] = (int)$stmt->fetchColumn();
        } catch (Exception $e) {
            $dbStats[$label] = 'N/A';
        }
    } else {
        $dbStats[$label] = 'N/A';
    }
}

// Get recent error logs count (last 24h) - only if logs table has level column
$recentErrors = 0;
$recentWarnings = 0;
$activityLastHour = 0;
if (tableExists($db, 'logs') && columnExists($db, 'logs', 'level')) {
    try {
        $stmt = $db->prepare("SELECT level, COUNT(*) as cnt FROM logs WHERE created_at > NOW() - INTERVAL '24 hours' AND level IN ('error', 'warning') GROUP BY level");
        $stmt->execute();
        foreach ($stmt->fetchAll() as $row) {
            if ($row['level'] === 'error') {
                $recentErrors = (int)$row['cnt'];
            }
            if ($row['level'] === 'warning') {
                $recentWarnings = (int)$row['cnt'];
            }
        }

        $stmt = $db->query("SELECT COUNT(*) FROM logs WHERE created_at > NOW() - INTERVAL '1 hour'");
        $activityLastHour = (int)$stmt->fetchColumn();
    } catch (Exception $e) {
        // Logs query failed
    }
}

// Get active admin sessions - only available if sessions table with admin_user_id exists
$activeAdminSessions = null;
if (tableExists($db, 'sessions') && columnExists($db, 'sessions', 'admin_user_id')) {
    try {
        $stmt = $db->query("SELECT COUNT(*) FROM sessions WHERE admin_user_id IS NOT NULL AND last_active > NOW() - INTERVAL '30 minutes'");
        $activeAdminSessions = (int)$stmt->fetchColumn();
    } catch (Exception $e) {
        // Sessions query failed
        $activeAdminSessions = null;
    }
}

// Check SSL status
$isHttps = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ||
           (!empty($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https');

// Get Cache Status
$cacheStatus = \NewSite\Cache\CacheService::getStatus();

// Get Database Status (Replicas)
$dbStatus = \NewSite\Database\DatabaseManager::getStatus();

// Check critical PHP extensions
$requiredExtensions = ['pdo', 'pdo_pgsql', 'openssl', 'mbstring', 'json', 'curl', 'gd'];
$missingExtensions = [];
foreach ($requiredExtensions as $ext) {
    if (!extension_loaded($ext)) {
        $missingExtensions[] = $ext;
    }
}

// Server time info
$serverTime = date('Y-m-d H:i:s');
$serverTimezone = date_default_timezone_get();

// SMTP Test
$smtpTestResult = null;
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['smtp_test'])) {
    verifyCSRF();
    $testEmail = SettingsService::get('contact_email');
    if ($testEmail) {
        $smtp = [
            'host' => SettingsService::get('smtp_host', ''),
            'port' => SettingsService::get('smtp_port', '587'),
            'user' => SettingsService::get('smtp_user', ''),
            'pass' => SettingsService::get('smtp_pass', ''),
            'secure' => SettingsService::get('smtp_secure', 'tls')
        ];
        $sent = SmtpMailer::send(
            $testEmail,
            'SMTP Test - ' . date('Y-m-d H:i:s'),
            'This is a test email from your admin panel to verify SMTP settings are working correctly.',
            '',
            '',
            $smtp
        );
        $smtpTestResult = $sent ? 'success' : 'error';
    } else {
        $smtpTestResult = 'no_email';
    }
}

// Calculate overall health score
$healthIssues = [];
if (!$isHttps) {
    $healthIssues[] = 'No HTTPS';
}
if (!empty($missingExtensions)) {
    $healthIssues[] = 'Missing PHP extensions';
}
if ($schemaTargetInt !== null && $currentSchemaVersion < $schemaTargetInt) {
    $healthIssues[] = 'Pending migrations';
}
if ($recentErrors > 10) {
    $healthIssues[] = 'Many recent errors';
}
$healthScore = max(0, 100 - (count($healthIssues) * 25));

AdminLayout::renderHeader();
?>

<div class="admin-content">

<div class="content-header">
    <h1>Info & Documentation</h1>
    <p>System status, documentation, and helpful information for administrators.</p>
</div>

<?php if (!empty($healthIssues)): ?>
<div class="health-banner health-warning">
    <strong>⚠ Health Issues Detected:</strong> <?php echo e(implode(', ', $healthIssues)); ?>
</div>
<?php else: ?>
<div class="health-banner health-good">
    <strong>✓ System Health: Good</strong> — All checks passed
</div>
<?php endif; ?>

<div class="info-grid">
    <!-- System Status Card -->
    <div class="card">
        <div class="card-header">
            <h3>📊 System Status</h3>
        </div>
        <div class="card-body">
            <table class="info-table">
                <thead>
                    <tr>
                        <th scope="col">Setting</th>
                        <th scope="col">Value</th>
                    </tr>
                </thead>
                <tr>
                    <td><strong>PHP Version</strong></td>
                    <td><?php echo e($phpVersion); ?></td>
                </tr>
                <tr>
                    <td><strong>PostgreSQL Version</strong></td>
                    <td><?php echo e($pgVersion); ?></td>
                </tr>
                <tr>
                    <td><strong>Schema Version (Current)</strong></td>
                    <td><?php echo e($currentSchemaVersion); ?></td>
                </tr>
                <tr>
                    <td><strong>Schema Version (Target)</strong></td>
                    <td><?php echo e($schemaTarget); ?></td>
                </tr>
                <tr>
                    <td><strong>Site Disk Usage</strong></td>
                    <td>
                        <?php echo formatBytes($siteTotalSize); ?> across <?php echo number_format($siteTotalFiles); ?> files
                        <?php if ($siteScanError): ?>
                            <span class="text-warning"> (some directories unreadable)</span>
                        <?php endif; ?>
                        <br><small class="text-muted">Scanned: <?php echo e($siteRootDir); ?></small>
                    </td>
                </tr>
                <tr>
                    <td><strong>Upload Folder</strong></td>
                    <td><?php echo $uploadCount; ?> files (<?php echo formatBytes($uploadSize); ?>)</td>
                </tr>
                <tr>
                    <td><strong>Server Time</strong></td>
                    <td><?php echo e($serverTime); ?> (<?php echo e($serverTimezone); ?>)</td>
                </tr>
                <tr>
                    <td><strong>HTTPS</strong></td>
                    <td><?php echo $isHttps ? '<span class="text-success">✓ Secure</span>' : '<span class="text-warning">⚠ Not HTTPS</span>'; ?></td>
                </tr>
                <tr>
                    <td><strong>Active Admin Sessions</strong></td>
                    <td>
                        <?php if ($activeAdminSessions === null): ?>
                            <span class="text-muted">N/A</span> <small class="text-muted">(DB session tracking disabled)</small>
                        <?php else: ?>
                            <?php echo (int)$activeAdminSessions; ?> (last 30 min)
                        <?php endif; ?>
                    </td>
                </tr>
            </table>
        </div>
    </div>

    <!-- Database Statistics Card -->
    <div class="card">
        <div class="card-header">
            <h3>🗄️ Database Statistics</h3>
        </div>
        <div class="card-body">
            <table class="info-table">
                <thead>
                    <tr>
                        <th scope="col">Metric</th>
                        <th scope="col">Value</th>
                    </tr>
                </thead>
                <?php foreach ($dbStats as $label => $count): ?>
                <tr>
                    <td><strong><?php echo e($label); ?></strong></td>
                    <td><?php echo is_numeric($count) ? number_format($count) : $count; ?></td>
                </tr>
                <?php endforeach; ?>
                <?php if ($activityLastHour > 0): ?>
                <tr>
                    <td><strong>Logs (Last Hour)</strong></td>
                    <td><span class="text-info"><?php echo number_format($activityLastHour); ?></span> activity entries</td>
                </tr>
                <?php endif; ?>
            </table>
            <?php if ($recentErrors > 0 || $recentWarnings > 0): ?>
            <div class="admin-box mt-15">
                <strong>Last 24 Hours:</strong>
                <?php if ($recentErrors > 0): ?>
                    <span class="text-danger ml-10">🔴 <?php echo $recentErrors; ?> errors</span>
                <?php endif; ?>
                <?php if ($recentWarnings > 0): ?>
                    <span class="text-warning ml-10">🟡 <?php echo $recentWarnings; ?> warnings</span>
                <?php endif; ?>
                <a href="/admin/logs.php" class="ml-15">View Logs →</a>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- PHP Extensions Card -->
    <div class="card">
        <div class="card-header">
            <h3>🔧 PHP Extensions</h3>
        </div>
        <div class="card-body">
            <table class="info-table">
                <thead>
                    <tr>
                        <th scope="col">Extension</th>
                        <th scope="col">Status</th>
                    </tr>
                </thead>
                <?php foreach ($requiredExtensions as $ext): ?>
                <tr>
                    <td><strong><?php echo e($ext); ?></strong></td>
                    <td>
                        <?php if (extension_loaded($ext)): ?>
                            <span class="text-success">✓ Loaded</span>
                        <?php else: ?>
                            <span class="text-danger">✗ Missing</span>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </table>
            <?php if (!empty($missingExtensions)): ?>
            <div class="admin-alert-danger mt-10">
                <strong>⚠ Missing extensions may cause issues!</strong>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Database Migrations Card -->
    <div class="card">
        <div class="card-header">
            <h3>🗃️ Database Migrations</h3>
        </div>
        <div class="card-body">
            <div class="info-block">
                <p>The <code>SCHEMA_TARGET</code> constant in <code>src/Database/SchemaManager.php</code> controls database migrations.</p>
                <ul class="info-list">
                    <li><strong>How it works:</strong> When <code>SCHEMA_TARGET</code> is greater than the stored <code>db_schema_version</code>, migration blocks run automatically on first request.</li>
                    <li><strong>Adding migrations:</strong> Increase <code>SCHEMA_TARGET</code> (e.g., from <?php echo e($currentSchemaVersion); ?> to <?php echo e(max($currentSchemaVersion, $schemaTargetInt) + 1); ?>) and add a new <code>if ($v &lt; X)</code> block inside <code>runMigrations()</code>.</li>
                    <li><strong>One-time execution:</strong> Each migration block runs once, then <code>db_schema_version</code> is updated via <code>SettingsService::set()</code> to prevent re-running.</li>
                    <li><strong>Current status:</strong>
                        <?php if ($currentSchemaVersion >= $schemaTargetInt): ?>
                            <span class="text-success">✓ Up to date (v<?php echo e($currentSchemaVersion); ?>)</span>
                        <?php else: ?>
                            <span class="text-warning">⚠ Pending migrations will run on next page load</span>
                        <?php endif; ?>
                    </li>
                </ul>
            </div>
        </div>
    </div>

    <!-- Recent Schema Highlights Card -->
    <div class="card">
        <div class="card-header">
            <h3>🆕 Recent Schema Highlights</h3>
        </div>
        <div class="card-body">
            <div class="info-block">
                <ul class="info-list">
                    <li><strong>v33:</strong> Collection URL slugs (<code>collections.collection_slug</code>) with unique index, auto-backfilled from title.</li>
                    <li><strong>v32:</strong> Product URL slugs (<code>products.product_slug</code>) with unique index, auto-backfilled from name.</li>
                    <li><strong>v31:</strong> Product comments feature (<code>product_comments</code> table, <code>products.allow_comments</code> flag).</li>
                    <li><strong>v30:</strong> Free digital download statistics table (<code>free_digital_download_stats</code>).</li>
                    <li><strong>v29:</strong> Collection SEO fields (<code>collections.meta_title</code>, <code>collections.meta_description</code>).</li>
                    <li><strong>v28:</strong> Mini-game gameplay tuning fields added to <code>mini_games</code>.</li>
                    <li><strong>v27:</strong> Mini-game leaderboard table added.</li>
                    <li><strong>v26:</strong> Mini-games registry table with enable/disable controls.</li>
                </ul>
            </div>
        </div>
    </div>

    <!-- Folder Structure Card -->
    <div class="card">
        <div class="card-header">
            <h3>📁 Folder Structure</h3>
        </div>
        <div class="card-body">
            <div class="info-block">
                <pre class="folder-tree">
/composer.json           ← PSR-4 autoloader config (NewSite\ → src/)
/vendor/                 ← Composer autoloader (generated)

/src/                    ← PSR-4 classes under NewSite\ namespace
  ├── Admin/             ← AdminLayout
  ├── Auth/              ← AdminAuth, SiteAuth
  ├── Cache/             ← CacheService, CachedQueries
  ├── Config/            ← ConfigLoader, SetupService
  ├── Database/          ← DatabaseManager, DbHelper, SchemaManager
  ├── Http/              ← Response, StripeClient
  ├── Logging/           ← LogService, RequestLogger
  ├── Security/          ← CsrfService, IpBanService, RateLimiter
  ├── Settings/          ← SettingsService
  ├── Shop/              ← CartService, OrderService, DownloadService
  ├── Template/          ← TemplateRenderer, ContentRenderer
  ├── User/              ← UserService, FriendService, etc.
  └── ...                ← + Email, Forum, Gdpr, Geo, Push, Upload, Util, Visitor

/data/                   ← Private data (not web accessible)
  ├── config.json        ← Database credentials (JSON format)
  ├── config.example.json ← Template for new installations
  ├── setup.php          ← Setup wizard (move to /public to use)
  ├── allowlist_tool.php ← Emergency IP management
  ├── admin_uploads/     ← Admin-uploaded files
  ├── chat_images/       ← Chat image uploads
  ├── contact_uploads/   ← Contact form attachments
  ├── forum_images/      ← Forum post images
  ├── profile_photos/    ← User profile photos
  ├── push_queue/        ← Push notification queue
  └── cache/             ← File-based cache

/includes/               ← Bootstrap only (not web accessible)
  └── _init.php          ← Session, CSRF, CSP, error handling, e() helper

/public/                 ← Web root (only folder accessible via browser)
  ├── index.php          ← Front controller (requires vendor/autoload.php)
  ├── admin/             ← Admin panel
  ├── assets/            ← CSS, images
  └── templates/         ← Page templates</pre>
            </div>
        </div>
    </div>

    <!-- Security Notes Card -->
    <div class="card">
        <div class="card-header">
            <h3>🔐 Security Notes</h3>
        </div>
        <div class="card-body">
            <div class="info-block">
                <ul class="info-list">
                    <li><strong>IP Allowlist:</strong> Only IPs in the <code>admin_allowed_ips</code> table can access <code>/admin</code>. If locked out and SSH is unavailable, use hosting support or temporarily place <code>/data/allowlist_tool.php</code> into <code>/public</code> with a random filename, run once, then remove immediately.</li>
                    <li><strong>Content Security Policy:</strong> This site uses strict CSP headers. All styles must be in external CSS files (no inline styles or attributes). JavaScript is not allowed; all interactivity is handled server-side. If JS is ever re-enabled, all scripts must be in external files only.</li>
                    <li><strong>CSRF Protection:</strong> All POST requests require a valid CSRF token. Forms include a hidden <code>csrf_token</code>. All destructive admin actions (Delete, Ban, etc.) use POST + CSRF.</li>
                    <li><strong>XSS Prevention:</strong> The <code>e()</code> function is used for all dynamic output. Critical numeric IDs in admin tools are cast to integers before rendering.</li>
                    <li><strong>GDPR Compliance:</strong> Visitor data is automatically cleaned up based on retention settings. Check Settings → GDPR Retention.</li>
                    <li><strong>Rate Limiting:</strong> Login attempts, contact forms, and API endpoints have rate limiting to prevent abuse.</li>
                    <li><strong>Banned IPs:</strong> Malicious IPs are automatically banned and see a fake "connection error" page.</li>
                </ul>
            </div>
        </div>
    </div>

    <!-- SMTP Configuration Card -->
    <div class="card">
        <div class="card-header">
            <h3>📧 Email Configuration</h3>
        </div>
        <div class="card-body">
            <div class="info-block">
                <p>Configure SMTP settings in <strong>Settings → SMTP Environment</strong> for email functionality:</p>
                <ul class="info-list">
                    <li><strong>Contact Form:</strong> Requires a valid contact email and SMTP settings.</li>
                    <li><strong>Password Reset:</strong> Requires SMTP to send reset links.</li>
                    <li><strong>Order Notifications:</strong> Requires SMTP for purchase confirmations.</li>
                    <li><strong>Data Breach Alerts:</strong> Optional email notifications for security events.</li>
                </ul>

                <div class="admin-box-lg mt-15">
                    <strong>Quick SMTP Test:</strong>
                    <form method="post" class="inline-block ml-10">
                        <input type="hidden" name="csrf_token" value="<?php echo e(getCsrfToken()); ?>">
                        <button type="submit" name="smtp_test" value="1" class="btn btn-secondary btn-sm">Send Test Email</button>
                    </form>
                    <?php if ($smtpTestResult === 'success'): ?>
                        <span class="text-success ml-10">✓ Test email sent!</span>
                    <?php elseif ($smtpTestResult === 'error'): ?>
                        <span class="text-danger ml-10">✗ Failed to send. Check SMTP settings.</span>
                    <?php elseif ($smtpTestResult === 'no_email'): ?>
                        <span class="text-warning ml-10">⚠ No contact email configured.</span>
                    <?php endif; ?>
                    <p class="mt-8 text-muted text-sm">
                        Sends a test email to: <?php echo e(SettingsService::get('contact_email') ?: '(not configured)'); ?>
                    </p>
                </div>
            </div>
        </div>
    </div>

    <!-- Helpful Commands Card -->
    <div class="card">
        <div class="card-header">
            <h3>💻 Helpful Commands</h3>
        </div>
        <div class="card-body">
            <div class="info-block">
                <p><strong>Regenerate Composer autoloader after adding new classes:</strong></p>
                <pre class="code-block">composer dump-autoload --optimize</pre>
                <p class="mt-8 text-muted text-sm">Run locally, then upload the updated <code>vendor/</code> directory.</p>

                <p class="mt-15"><strong>Emergency IP Allowlist (command line if available):</strong></p>
                <pre class="code-block">php /path/to/data/allowlist_tool.php</pre>
                <p class="mt-8 text-muted text-sm">If SSH is unavailable, use hosting support or a temporary, randomly named copy placed in <code>/public</code> and delete it immediately after use.</p>

                <p class="mt-15"><strong>Clear all sessions (PostgreSQL):</strong></p>
                <pre class="code-block">TRUNCATE TABLE sessions;</pre>

                <p class="mt-15"><strong>Reset admin password (PostgreSQL):</strong></p>
                <pre class="code-block">UPDATE admin_users SET password_hash = '$argon2id$...' WHERE username = 'admin';</pre>
                <p><em>Generate hash with: <code>php -r "echo password_hash('newpassword', PASSWORD_ARGON2ID);"</code></em></p>
            </div>
        </div>
    </div>

    <!-- File Upload Limits Card -->
    <div class="card">
        <div class="card-header">
            <h3>📤 Upload Limits</h3>
        </div>
        <div class="card-body">
            <div class="info-block">
                <table class="info-table">
                    <thead>
                        <tr>
                            <th scope="col">Limit</th>
                            <th scope="col">Value</th>
                        </tr>
                    </thead>
                    <tr>
                        <td><strong>PHP upload_max_filesize</strong></td>
                        <td><?php echo ini_get('upload_max_filesize'); ?></td>
                    </tr>
                    <tr>
                        <td><strong>PHP post_max_size</strong></td>
                        <td><?php echo ini_get('post_max_size'); ?></td>
                    </tr>
                    <tr>
                        <td><strong>PHP max_execution_time</strong></td>
                        <td><?php echo ini_get('max_execution_time'); ?> seconds</td>
                    </tr>
                    <tr>
                        <td><strong>PHP memory_limit</strong></td>
                        <td><?php echo ini_get('memory_limit'); ?></td>
                    </tr>
                </table>
                <p class="mt-10"><em>To increase limits, edit <code>php.ini</code> or use <code>.htaccess</code> / <code>.user.ini</code></em></p>
            </div>
        </div>
    </div>

    <!-- Cache & Performance Card -->
    <div class="card">
        <div class="card-header">
            <h3>⚡ Cache & Performance</h3>
        </div>
        <div class="card-body">
            <table class="info-table">
                <thead>
                    <tr>
                        <th scope="col">Component</th>
                        <th scope="col">Status</th>
                    </tr>
                </thead>
                <tr>
                    <td><strong>Cache Method</strong></td>
                    <td>
                        <span class="text-info"><?php echo ucfirst($cacheStatus['type'] ?? 'None'); ?></span>
                        <?php if (!empty($cacheStatus['connected'])): ?>
                            <span class="text-success ml-10">✓ Active</span>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php if ($cacheStatus['type'] === 'redis' && !empty($cacheStatus['memory'])): ?>
                <tr>
                    <td><strong>Redis Memory</strong></td>
                    <td><?php echo e($cacheStatus['memory']); ?></td>
                </tr>
                <?php elseif ($cacheStatus['type'] === 'file' && isset($cacheStatus['files'])): ?>
                <tr>
                    <td><strong>Cache Files</strong></td>
                    <td><?php echo number_format($cacheStatus['files']); ?> items in <code>/data/cache</code></td>
                </tr>
                <?php endif; ?>
                <tr>
                    <td><strong>DB Replicas</strong></td>
                    <td>
                        <?php
                        $replicas = $dbStatus['replicas'] ?? [];
$replicaCount = 0;
foreach ($replicas as $r) {
    if (!isset($r['note'])) {
        $replicaCount++;
    }
}
echo $replicaCount > 0 ? $replicaCount . ' Active' : 'None (Using Primary)';
?>
                    </td>
                </tr>
            </table>
            <p class="mt-10 text-sm text-muted">Caching reduces database load by ~95% for settings and user sessions.</p>
        </div>
    </div>
</div>

</div><!-- /.admin-content -->

<?php AdminLayout::renderFooter(); ?>
