<?php
/**
 * Admin Panel – Front Controller & Dashboard
 *
 * All /admin/*.php requests are rewritten here via .htaccess.
 * This file bootstraps the application once, then routes to
 * the requested admin page or renders the dashboard.
 */

require_once __DIR__ . '/../../vendor/autoload.php'; // Composer PSR-4 + bootstrap

use NewSite\Admin\AdminLayout;
use NewSite\Auth\AdminAuth;
use NewSite\Database\DatabaseManager;
use NewSite\Logging\LogService;
use NewSite\Settings\SettingsService;
use NewSite\Shop\CurrencyService;
use NewSite\Template\TemplateRenderer;

// ── Front Controller: route to requested admin page ────────────
// Uses 'admin_route' from .htaccess RewriteRule to avoid colliding with
// individual admin pages' own '?page=' query parameters (e.g. theme-editor.php?page=16).
$adminRoutedPage = $_GET['admin_route'] ?? null;

if ($adminRoutedPage !== null) {
    // Sanitize: only lowercase alphanumeric, hyphens, and forward slash (for api/ sub-routes)
    $adminRoutedPage = preg_replace('/[^a-z0-9\-\/]/', '', strtolower((string) $adminRoutedPage));

    // Explicit allowlist of routable admin pages
    $adminAllowedPages = [
        'backup', 'collections', 'currency-rates', 'dmca',
        'email-templates', 'files', 'forum', 'info', 'legal-pages',
        'login', 'logout', 'logs', 'menus', 'messages', 'mini-games',
        'pages', 'products', 'purchases', 'refunds', 'security-scan', 'settings',
        'site-files', 'statistics', 'text-finder', 'theme-editor',
        'tickets', 'tickets-api', 'translations', 'vendor-libs',
        'attributes',
    ];

    // Admin API sub-routes (api/sections → api/sections.php)
    $adminAllowedApiPages = ['sections'];

    if (str_starts_with($adminRoutedPage, 'api/')) {
        $apiPage = substr($adminRoutedPage, 4);
        // Sanitize the API page name
        $apiPage = preg_replace('/[^a-z0-9-]/', '', strtolower($apiPage));
        if (in_array($apiPage, $adminAllowedApiPages, true)) {
            TemplateRenderer::renderOnce(__DIR__ . '/api/' . $apiPage . '.php');
            exit;
        }
    }

    if (in_array($adminRoutedPage, $adminAllowedPages, true)) {
        TemplateRenderer::renderOnce(__DIR__ . '/' . $adminRoutedPage . '.php');
        exit;
    }

    http_response_code(404);
    exit;
}

// ── Dashboard ──────────────────────────────────────────────────
AdminAuth::requireLogin();

const ADMIN_DATE_TIME_FORMAT = 'Y-m-d H:i:s';

$db = DatabaseManager::getWriteConnection();
$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verifyCSRF();
    $action = $_POST['action'] ?? '';
    if ($action === 'cleanup_data') {
        try {
            /* Use GDPR retention settings instead of wiping everything */
            $logDays = max(1, (int)SettingsService::get('gdpr_retention_logs_days', '30'));
            $visitorDays = max(1, (int)SettingsService::get('gdpr_retention_visitors_days', '30'));
            $contactDays = max(1, (int)SettingsService::get('gdpr_retention_contact_days', '365'));
            $rateDays = max(1, (int)SettingsService::get('gdpr_retention_rate_limits_days', '7'));
            $geoDays = max(1, (int)SettingsService::get('gdpr_retention_geo_cache_days', '7'));

            $now = time();
            $logCutoff = date(ADMIN_DATE_TIME_FORMAT, $now - ($logDays * 86400));
            $visitorCutoff = date(ADMIN_DATE_TIME_FORMAT, $now - ($visitorDays * 86400));
            $contactCutoff = date(ADMIN_DATE_TIME_FORMAT, $now - ($contactDays * 86400));
            $rateCutoff = $now - ($rateDays * 86400);
            $geoCutoff = $now - ($geoDays * 86400);

            $db->beginTransaction();
            $db->prepare("DELETE FROM logs WHERE created_at < ?")->execute([$logCutoff]);
            $db->prepare("DELETE FROM visitor_logs WHERE created_at < ?")->execute([$visitorCutoff]);
            $db->prepare("DELETE FROM visitors WHERE last_activity < ?")->execute([$visitorCutoff]);
            $db->prepare("DELETE FROM contact_messages WHERE created_at < ?")->execute([$contactCutoff]);
            $db->prepare("DELETE FROM contact_rate_limits WHERE last_time < ?")->execute([$rateCutoff]);
            $db->prepare("DELETE FROM geo_cache WHERE cached_at < ?")->execute([$geoCutoff]);
            $db->commit();

            /* Clean contact_uploads: files are already emailed, delete old ones */
            $uploadDir = __DIR__ . '/../../data/contact_uploads/';
            $deletedFiles = 0;
            if (is_dir($uploadDir)) {
                $contactFileCutoff = $now - ($contactDays * 86400);
                foreach (new DirectoryIterator($uploadDir) as $f) {
                    if ($f->isDot() || !$f->isFile() || $f->getFilename() === '.htaccess') {
                        continue;
                    }
                    if ($f->getMTime() < $contactFileCutoff) {
                        @unlink($f->getPathname());
                        $deletedFiles++;
                    }
                }
            }

            $message = "Cleanup completed using GDPR retention settings (logs: {$logDays}d, visitors: {$visitorDays}d, contact: {$contactDays}d). Deleted {$deletedFiles} old contact upload files.";
            LogService::add('info', 'Admin cleanup executed (GDPR-aware)', json_encode([
                'logs_days' => $logDays,
                'visitors_days' => $visitorDays,
                'contact_days' => $contactDays,
                'deleted_upload_files' => $deletedFiles
            ]));
        } catch (PDOException $e) {
            $db->rollBack();
            $error = 'Cleanup failed.';
            LogService::add('error', 'Admin cleanup failed: ' . $e->getMessage());
        }
    }
}

// Get stats
$totalPages = $db->query("SELECT COUNT(*) FROM pages")->fetchColumn();
$totalMenus = $db->query("SELECT COUNT(*) FROM menus")->fetchColumn();
$totalFiles = $db->query("SELECT COUNT(*) FROM files")->fetchColumn();
$totalSections = $db->query("SELECT COUNT(*) FROM sections")->fetchColumn();

// Business KPIs (same logic as statistics page)
$totalRevenue = (float)$db->query("SELECT COALESCE(SUM(total_amount), 0) FROM orders WHERE status = 'paid'")->fetchColumn();
$storeCurrency = SettingsService::get('store_currency', 'USD');
$currentCurrency = CurrencyService::getCurrent();
$totalRevenueConverted = CurrencyService::convert($totalRevenue, $storeCurrency, $currentCurrency);
$currencySymbol = CurrencyService::getSymbol();

$totalFreeDownloadCount = 0;
try {
    $totalFreeDownloadCount = (int)$db->query("SELECT COALESCE(SUM(download_count), 0) FROM free_digital_download_stats")->fetchColumn();
    $totalFreeDownloadCount += max(0, (int)SettingsService::get('easy_media_ai_zip_download_count', '0'));
} catch (PDOException $e) {
    // Table may not exist yet on older installs; keep dashboard available.
    $totalFreeDownloadCount = max(0, (int)SettingsService::get('easy_media_ai_zip_download_count', '0'));
}

AdminLayout::renderHeader();
?>

<div class="admin-content">
    <div class="content-header">
        <h1>Dashboard</h1>
        <p>Welcome back, <?php echo e($_SESSION['admin_username']); ?>!</p>
    </div>

    <?php if ($message): ?>
        <div class="alert alert-success"><?php echo e($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-error"><?php echo e($error); ?></div>
    <?php endif; ?>

    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-icon">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
                    <polyline points="14 2 14 8 20 8"></polyline>
                </svg>
            </div>
            <div class="stat-info">
                <h3><?php echo $totalPages; ?></h3>
                <p>Pages</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <line x1="3" y1="12" x2="21" y2="12"></line>
                    <line x1="3" y1="6" x2="21" y2="6"></line>
                    <line x1="3" y1="18" x2="21" y2="18"></line>
                </svg>
            </div>
            <div class="stat-info">
                <h3><?php echo $totalMenus; ?></h3>
                <p>Menus</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                    <polyline points="17 8 12 3 7 8"></polyline>
                    <line x1="12" y1="3" x2="12" y2="15"></line>
                </svg>
            </div>
            <div class="stat-info">
                <h3><?php echo $totalFiles; ?></h3>
                <p>Files</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <rect x="3" y="3" width="18" height="18" rx="2" ry="2"></rect>
                    <line x1="3" y1="9" x2="21" y2="9"></line>
                    <line x1="9" y1="21" x2="9" y2="9"></line>
                </svg>
            </div>
            <div class="stat-info">
                <h3><?php echo $totalSections; ?></h3>
                <p>Sections</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <line x1="12" y1="1" x2="12" y2="23"></line>
                    <path d="M17 5H9.5a3.5 3.5 0 0 0 0 7H14.5a3.5 3.5 0 0 1 0 7H6"></path>
                </svg>
            </div>
            <div class="stat-info">
                <h3><?php echo e($currencySymbol . number_format((float)$totalRevenueConverted, 2)); ?></h3>
                <p>Total Brutto Earned</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                    <polyline points="7 10 12 15 17 10"></polyline>
                    <line x1="12" y1="15" x2="12" y2="3"></line>
                </svg>
            </div>
            <div class="stat-info">
                <h3><?php echo number_format($totalFreeDownloadCount); ?></h3>
                <p>Free Downloads</p>
            </div>
        </div>
    </div>

    <div class="dashboard-grid">
        <div class="dashboard-card">
            <h3>Quick Actions</h3>
            <div class="quick-actions">
                <a href="/admin/pages.php?action=new" class="quick-action">
                    <span class="icon">+</span>
                    <span>New Page</span>
                </a>
                <a href="/admin/menus.php" class="quick-action">
                    <span class="icon">☰</span>
                    <span>Manage Menus</span>
                </a>
                <a href="/admin/theme-editor.php" class="quick-action">
                    <span class="icon">🎨</span>
                    <span>Theme Editor</span>
                </a>
                <a href="/admin/files.php" class="quick-action">
                    <span class="icon">📁</span>
                    <span>File Manager</span>
                </a>
                <a href="/admin/statistics.php" class="quick-action">
                    <span class="icon">📊</span>
                    <span>Statistics</span>
                </a>
            </div>
        </div>

        <div class="dashboard-card">
            <h3>System Cleanup</h3>
            <p class="text-muted mb-16">Clear logs, visitor tracking, and stored contact messages.</p>
            <form method="POST" action="/admin/">
                <input type="hidden" name="action" value="cleanup_data">
                <input type="hidden" name="csrf_token" value="<?php echo getCSRFToken(); ?>">
                <button type="submit" class="btn btn-outline" data-confirm="Run cleanup using GDPR retention settings? Data older than the configured retention periods will be removed.">Run Cleanup</button>
            </form>
        </div>

        <div class="dashboard-card">
            <h3>Recent Pages</h3>
            <?php
            $recentPages = $db->query("SELECT * FROM pages ORDER BY updated_at DESC LIMIT 5")->fetchAll();
?>
            <ul class="recent-list">
                <?php foreach ($recentPages as $page): ?>
                <li>
                    <a href="/admin/pages.php?action=edit&id=<?php echo $page['id']; ?>">
                        <?php echo e($page['title']); ?>
                    </a>
                    <span class="date"><?php echo date('M j', strtotime($page['updated_at'])); ?></span>
                </li>
                <?php endforeach; ?>
            </ul>
        </div>
    </div>
</div>

<?php AdminLayout::renderFooter(); ?>
